(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 20.6.99 by Hendrik
 *
 * Time-stamp: <Tuesday 10 August 10 22:17:58 tews@gromit.tews.net>
 *
 * dynamic name spaces with unique identifier creation
 *
 * $Id: name_space.mli,v 1.7 2010-10-19 12:51:07 tews Exp $
 *
 *)

open Top_variant_types;;

(***********************************************************************
 ***********************************************************************
 *
 *   the type of name spaces;
 * 
 *   name spaces manage ('cl, 'mem) top_pre_variant_types, therefore 
 *   the names spaces are polymorphic in the classes;

 *   Further, the implementation uses method get_name on the classes 
     and adt's in top_pre_variant_types, this is captured by the 
     (nearly vacuous) type get_name_type, and the abbreviation 'cl t;
     
     These layers of definitions and abbreviations are due to various 
     bugs of ocaml. 
 *)


type 'cl get_name_type = 'cl
constraint 'cl = < get_name : string; .. > 


type ('cl, 'mem) abstract


type ('cl, 'mem) t = ('cl get_name_type, 'mem) abstract

(***********************************************************************
 ***********************************************************************
 *
 * Interface
 *
 *)

  (* create a fresh top level name space, 
   * takes equality on types as argument 
   *)
val create : ( ('cl get_name_type, 'mem) top_pre_types -> 
		('cl get_name_type, 'mem) top_pre_types -> bool) 
  -> ('cl, 'mem) t


(* 
 * Declare the name space to be flat. This means that sub name space 
 * creation is switched off behind the scene. This way all identifiers 
 * created subsequently in this name space are unique. (This is not the 
 * case for non-flat name spaces. There, two sub name spaces can share 
 * the same identifier)
 *)
val make_flat : ('cl, 'mem) t -> unit

  (* create a sub name space,
   * this has no effect, if the name space was made flat 
   * via make_flat before
   *)
val sub_space : ('cl, 'mem) t -> ('cl, 'mem) t

  (* test, if an identifier is already bound in an name space *)
val test : ('cl, 'mem) t -> string -> bool

  (* bind names in name space *)
val reserve : ('cl, 'mem) t -> string list -> unit

  (* create a list of unique ids for types *)
val create_ids : ('cl, 'mem) t -> ('cl, 'mem) top_pre_types list -> 
  (string * ('cl, 'mem) top_pre_types) list 

  (* create just one identifier *)
val create_one_id : ('cl, 'mem) t -> ('cl, 'mem) top_pre_types -> string

  (* create two identifiers for each element in the type list,
   * one in the first result list and one in the second
   *)
val create_id_pairs : ('cl, 'mem) t -> ('cl, 'mem) top_pre_types list ->
  (string * ('cl, 'mem) top_pre_types) list 
     * (string * ('cl, 'mem) top_pre_types) list 


val create_ids_with_variance : 
  ('cl, 'mem) t -> 
    (('cl, 'mem) top_pre_types * variance_type) list ->
      ((string * ('cl, 'mem) top_pre_types) option *
       (string * ('cl, 'mem) top_pre_types) option) list


    (* 
     * create_id_with_preference ns name stem
     * reserves name and returns it, if possible;
     * otherwise use stem to create a name with appending digits
     *)
val create_id_with_preference :
  ('cl,'mem) t -> string -> string -> string


val create_ids_with_preference :
  ('cl,'mem) t -> (string * ('cl, 'mem) top_pre_types) list ->
    (string * ('cl, 'mem) top_pre_types) list 



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

