(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 26.2.99 by Bart
 *
 * Time-stamp: <Wednesday 30 June 10 11:24:35 tews@blau.inf.tu-dresden.de>
 *
 * PVS pretty printer
 *
 * $Id: pvs_pretty.ml,v 1.29 2010-06-30 09:38:02 tews Exp $
 *
 *)




open Util;;
open Formatter;;
open Top_names
open Top_variant_types;;
open Top_variant_types_util;;
open Pretty_util;;


(***********************************************************************
 ***********************************************************************
 *
 * Auxiliary functions
 *
 *)

let rec pvs_pp_top_type 
  = function
    | Groundtype(id,args) ->
	begin
	  print_string id.id_token.token_name;
	  open_box 0;
	  pvs_pp_arg_list args;
	  close_box();
	end
    | TypeConstant (name, flag, args) -> 
	begin
	  print_string name;
	  if flag = Always
	  then
	    begin
	      open_box 0;
	      pvs_pp_arg_list args;
	      close_box();
	    end
	end
    | BoundTypeVariable id -> 
	print_string id.id_token.token_name
    | Self -> 
	print_string (name_of_self ())
    | Bool ->
	print_string "bool"
    | Carrier ->
	(* To print a Carrier in PVS, you need the name of the adt.
	 * Use substitution, to substitute all Carriers with the 
	 * type of the Adt.
	 *)
	assert(false)
    | Function(dom, codom) ->
	begin
	  open_box 0;
	  print_string "[";
	  open_box 0;
	  pvs_pp_top_type dom; 
	  print_space(); 
	  print_string "-> ";
	  pvs_pp_top_type codom;
	  close_box(); 
	  print_string "]";
	  close_box()
	end	
    | SmartFunction(doml,codom) ->
	(match doml with
	   | [] -> assert(false)
	   | [t] -> pvs_pp_top_type (Function(t,codom))
	   | tl -> pvs_pp_top_type (Function(Product(tl),codom))
	)
    | Product typelist ->
	begin
	  open_box 0;
	  print_string "["; 
	  pp_list pvs_pp_top_type "" "" "" "," typelist;
	  print_string "]";
	  close_box()
	end
    | Record string_type_list -> 
	(match string_type_list with
	     [] -> 
	       assert false;
	   | l ->  
	       begin
		 open_box 0;
		 print_string "[#";
		 open_box 0;
		 pp_list
		   (fun (str, typ)
		      ->
			begin
			  force_newline();
			  print_string (str ^ " :");
			  print_space();
			  pvs_pp_top_type typ
			end) "" "" "" "," l;
		 close_box();
		 force_newline();
		 print_string "#]";
		 close_box()
	       end)      	
    | Class(cl, arg_list) ->
	let tname = 
	  if cl#has_feature FinalSemanticsFeature
	  then name_of_final_type cl
	  else name_of_loose_type cl
	in
	   begin
	     open_box 0;	    
	     print_string tname;
	     pvs_pp_arg_list arg_list; 
	     close_box()
	   end
    | Adt(adt, flag, arg_list) ->
	   begin
	     open_box 0;
	     print_string (name_of_type_of_adt adt);
	     if flag = Always
	     then
		 pvs_pp_arg_list arg_list;
	     close_box()
	   end

    | IFace(ifa, flag, arg_list) ->
  	begin
	  open_box 0;
	  print_string (name_of_method_functor ifa);
	  if flag = Always
	  then
            pvs_pp_arg_list arg_list;
	  close_box()
  	end
    | Array (_, typ, dim) ->
	begin
	  open_box 0;
	  print_string (match typ with
			  | TypeConstant( s, _, _) ->
			      name_of_array_of_primitive
				s (string_of_int dim)
			  | Class(c, _) ->
			      name_of_array_of_class
				c (string_of_int dim)
			  | _ ->
			      assert false
		       );
	  close_box()
	end

    | Predtype form -> 
	begin
	  open_box 0;
	  print_string "("; 
	  pvs_pp_top_formula form; 
	  print_string ")";
	  close_box()
	end
					(* not allowed *)
    | FreeTypeVariable _ -> assert(false)

and pvs_pp_top_basic_expression = function
    | TermVar idc ->
	print_string (resolution_of idc).id_token.token_name
    | Member(_,_)  ->
			(* Methods are treated in the prepritting pass *)
	assert false


and pvs_pp_top_expression (* : top_expressions -> unit *)
  = function
    | ExprLoc(ex,_) -> pvs_pp_top_expression ex
    | BasicExpr bexp -> 
	pvs_pp_top_basic_expression bexp
    | Term (name, flag, args) ->
      	begin
	  open_box 0;
	  print_string name;
	  if flag = Always
	  then
	      pvs_pp_arg_list args;
	  close_box();
      	end
    | TypedTerm(ex, typ) ->
	begin
	  open_box 0;
	  print_string "(";
	  pvs_pp_top_expression ex;
	  print_string " :: ";
	  print_space();
	  pvs_pp_top_type typ;
	  print_string ")";
	  close_box();
	end
    | TypeAnnotation(ex, _) ->
	  pvs_pp_top_expression ex;
    | QualifiedTerm (theory, flag, args, name) ->
	begin
	  open_box 0;
	  print_string theory;
	  if flag = Always
	  then
	      pvs_pp_arg_list args;
	  print_string ".";
	  print_string name;
	  close_box()
	end

    | Tuple expr_list ->
	(match expr_list with
	   | [] ->
	       ()
	   | l ->
	       begin
		 open_box 0;
		 print_string "(";
		 open_box 0;
		 (pp_list pvs_pp_top_expression "" "" "" "," l);
		 close_box();
		 print_string ")";
		 close_box()
	       end)
    | RecordTuple str_expr_list ->
	(match str_expr_list with
	   | [] -> 
	       assert false
	   | l ->  
	       begin
		 open_box 0;
		 print_string "(#";
		 open_box 0;
		 pp_list
		   (fun (str, expr) ->
		      begin
			force_newline();
			print_string (str ^ " :=");
			print_space();
			pvs_pp_top_expression expr;
		      end) "" "" "" "," l;
		 close_box();
		 force_newline();
		 print_string "#)";
		 close_box()
	       end)
    | RecordSelection (str, expr) ->
	begin
	  open_box 0;
	  print_string str; 
	  print_break 0 2; 
	  print_string "(";
	  open_box 0;
	  pvs_pp_top_expression expr; 
	  print_string ")";
	  close_box();
	  close_box()
	end
    | RecordUpdate (record, str_expr_list) ->
	begin
	  open_box 0;
	  pvs_pp_top_expression record; 
	  if
	    str_expr_list <> []
	  then
	    begin
	      print_space();
	      open_box 0;
	      print_string "WITH ["; 
	      print_break 1 1;
	      pp_list (fun (str, expr) ->
			 begin
			   open_box 0;
			   print_string( "(" ^ str ^ ") :=");
			   print_space();
			   pvs_pp_top_expression expr;
			   close_box();
			 end) "" "" "" "," str_expr_list;
	      print_string "]";
	      close_box(); 
	    end;
	    close_box();
	end
    | List (expr_list) ->
	begin
	  open_box 0;
	  print_string "(: ";
	  open_box 0;
	  (pp_list pvs_pp_top_expression "" "" "" "," expr_list);
	  close_box();
	  print_string " :)";
	  close_box()
	end

    | 
      ( SmartAbstraction _
        | Abstraction _ ) 
	as arg ->
	let (str_typ_list, expr) = 
	  (match arg with
	     | SmartAbstraction (str_typ_list, expr) -> (str_typ_list, expr)
	     | Abstraction (str_typ_list, expr) -> (str_typ_list, expr)
	     | _ -> assert(false)
	  )
	in
	  (match str_typ_list with
	     | [] ->
		 pvs_pp_top_expression expr
	     | l ->
		 begin
		   open_box 0;				(* 1 *)
		   print_string "(";
		   print_string "LAMBDA (";
		   pp_list pvs_pp_var "" "" "" "," str_typ_list;
		   print_string ") :";
		   print_break 1 2;
		   (* open_box 0; *)
		   pvs_pp_top_expression expr;
		   (* close_box (); *)
		   print_string ")";
		   close_box ()
		 end)
	(* the normal way to print a method *)
    | Application(  Projection (ith,_), expr) ->
	begin
	  open_box 0;
 	  print_string ("PROJ_" ^ (string_of_int ith) ^ "(");
	  print_break 0 2;
	  open_box 0;
	  pvs_pp_top_expression expr;
	  close_box ();
	  print_break 0 1;
	  print_string ")";
 	  close_box ()
    	end
    | Application (func, arg) ->
	(match arg with
	   | Tuple _ as arg ->
	       begin
		 open_box 0;
		 pvs_pp_top_expression func; 
		 print_break 0 2;
		 pvs_pp_top_expression arg;
		 close_box ()
	       end
	   | _ ->
	       begin
		 open_box 0;
		 pvs_pp_top_expression func; 
		 print_break 0 2; 
		 print_string "(";
		 open_box 0;
		 pvs_pp_top_expression arg; 
		 print_string ")";
		 close_box();
		 close_box()
	       end)
	
    | InfixApplication (expr1,instiface,mem,expr2) ->
	let mem_name = (resolution_of mem)#get_name 
	in
	  begin
	    open_box 0;			(* 1 *)
	    print_string "(";
	    pvs_pp_top_expression expr1; 
	    print_break 1 2;
	    print_string mem_name;
	    print_break 1 2;
	    pvs_pp_top_expression expr2; 
	    print_string ")";
	    close_box();		(* 0 *)
	  end

    | SmartApplication(func, arg_list) ->
	(match arg_list with
	   | [] -> assert(false)
	   | [t] -> pvs_pp_top_expression (Application(func, t))
	   | tl -> pvs_pp_top_expression (Application(func, Tuple(tl))))
    | FunUpdate (func, expr_expr_list) ->
	begin
	  open_box 0;
	  pvs_pp_top_expression func; 
	  print_space();
	  open_box 0;
	  print_string "WITH ["; 
	  print_break 1 1;
	  pp_list (fun (place, expr) ->
		     begin
		       open_box 0;
		       print_string "(";
		       pvs_pp_top_expression place;
		       print_string ") :=";
		       print_space();
		       pvs_pp_top_expression expr;
		       close_box();
		     end) "" "" "" "," expr_expr_list;
	  print_string "]";
	  close_box(); 
	  close_box();
	end
    | Let (id_typ_expr_list, expr) ->
	begin
	  open_box 0;					(* 1 *)
	  print_string "(";
	  if 
	    id_typ_expr_list <> [] 
	  then
	    begin
	      print_string "LET"; 
	      print_break 1 2;
	      pp_list
		(fun (id, typ_opt, expr) ->
		   begin
		     print_string id.id_token.token_name;
		     (match typ_opt with
			| None ->
			    if id.id_origin = CCSL_Var 
			    then
			      begin
				print_string " : ";
				pvs_pp_top_type id.id_type
			      end
			| Some typ ->
			    begin
			      print_string " : ";
			      pvs_pp_top_type typ
			    end);
		     print_string " =";
		     print_space();
		     print_break 1 2;
		     pvs_pp_top_expression expr;
		   end) "" "" "" "," id_typ_expr_list;
	      print_space(); 
	      print_string "IN";
	      print_break 1 2;
	    end;
	  open_box 0;					(* 2 *)
	  pvs_pp_top_expression expr;
	  close_box();					(* 2 *)
	  print_string ")"; 
	  close_box()					(* 1 *)
	end
    | If (form_expr_list, expr) ->
	begin
	  if 
	    form_expr_list <> [] 
	  then
	    begin
	      open_box 0;
	      print_string " IF"; 
	      print_space();
	      pp_list (fun (cond, expr) ->
			 print_if_newline(); 
			 print_space();
			 pvs_pp_top_formula cond; 
			 print_space();
			 open_box 0;
			 print_string " THEN"; 
			 print_break 1 2;
			 pvs_pp_top_expression expr;
			 close_box(); 
			 print_space();
			 open_box 0;
		      ) "" "" "" "ELSIF" form_expr_list;
	      print_string " ELSE"; 
	      print_break 1 2;
	    end;
	  pvs_pp_top_expression expr; 
	  print_space();
	  if 
	    form_expr_list <> [] 
	  then 
	    begin
	      for i = 1 to List.length form_expr_list do
		close_box()
	      done; 
	      print_string " ENDIF"; 
	      print_space();
	    end;
	  close_box();
	end
    | Case (expr, expr_expr_list) ->
	(match expr_expr_list with
	   | [] -> 
	       assert false
	   | (tag1, expr1) :: cases ->
 	       begin
		 open_box 0;				   (* 1 *)
		 print_string "CASES"; 
		 print_break 1 2;
		 open_box 0;				   (* 2 *)
		 pvs_pp_top_expression expr;
		 close_box ();				   (* 2 *)
		 print_space (); 
		 print_string "OF"; 
		 print_break 1 2;
		 open_box 0;				   (* 2a *)
		 pvs_pp_top_expression tag1;
		 print_string ":"; 
		 print_break 1 2;
		 open_box 0;				   (* 3 *)
		 pvs_pp_top_expression expr1; 
		 close_box ();				   (* 3 *)
		 List.iter
		   (fun (tag', expr') ->
		      begin
		   	print_string ","; 
			print_space ();
			force_newline(); (* added by BJ *)
		   	pvs_pp_top_expression tag';
			print_string ":";
		   	print_break 1 2;
		   	open_box 0;			   (* 3a *)
			pvs_pp_top_expression expr';
			close_box ();			   (* 3a *)
		      end) cases;
		 close_box ();				   (* 2a *)
		 print_space ();
		 print_string "ENDCASES"; 
		 print_space();
		 close_box()				   (* 1 *)
	       end)
    | CCSL_Case (expr, match_list) ->
	(match match_list with
	   | [] -> assert false
	   | (mem1, var_list1, expr1) :: cases ->
 	       begin
		 open_box 0;		                   (* 1 *)
		 print_string "CASES"; 
		 print_break 1 2;
		 open_box 0;				   (* 2 *)
		 pvs_pp_top_expression expr;
		 close_box ();				   (* 2 *)
		 print_space (); 
		 print_string "OF"; 
		 print_break 1 2;
		 open_box 0;				   (* 2a *)
		 print_string (resolution_of mem1)#get_name;
		 if var_list1 <> [] then
		   begin
		     print_string "(";
		     pp_list 
		       (fun id -> print_string id.id_token.token_name ) 
		       "" "" "" ", " var_list1;
		     print_string ")";
		   end;
		 print_string ":"; 
		 print_break 1 2;
		 open_box 0;				   (* 3 *)
		 pvs_pp_top_expression expr1; 
		 close_box ();				   (* 3 *)
		 List.iter
		   (fun (mem',var_list', expr') ->
		      begin
		   	print_string ","; 
			print_space ();
			force_newline(); (* added by BJ *)
		 	print_string (resolution_of mem')#get_name;
		 	if var_list' <> [] then
			  begin
			    print_string "(";
			    pp_list 
			      (fun id -> print_string id.id_token.token_name ) 
			      "" "" "" ", " var_list';
			    print_string ")";
			  end;
			print_string ":";
		   	print_break 1 2;
		   	open_box 0;			   (* 3a *)
			pvs_pp_top_expression expr';
			close_box ();			   (* 3a *)
		      end) cases;
		 close_box ();				   (* 2a *)
		 print_space ();
		 print_string "ENDCASES"; 
		 print_space();
		 close_box()				   (* 1 *)
	       end)

    | Every (typ, expr_list) ->
	let every_term = 
			(* every is only defined for three kinds of types *)
	  (match typ with
	     | Class( cl,_ ) -> 
		 let cl_name = 
		   if cl#has_feature FinalSemanticsFeature 
		   then name_of_final_coalgebra cl
		   else name_of_loose_coalgebra cl
		 in
		   Application(
		     Term(name_of_class_every cl, Always, []),
		     Term(cl_name,Always,[]))
	     | Adt( adt,_,_ ) -> 
		 Term(name_of_adt_every adt,Always,[])
	     | Groundtype(id,_) -> 
		 Term(name_of_const_every id.id_token.token_name,
		      Always,[])
	     | _ -> assert(false)
	  )
	in
	  pvs_pp_top_expression
	    (if expr_list = [] 
	     then every_term
	     else 
	       SmartApplication(
		 every_term, 
		 expr_list))

    | RelEvery (typ, expr_list) ->
	let relevery_term = 
			(* every is only defined for three kinds of types *)
	  (match typ with
	     | Class( cl,_ ) -> 
		 let cl_name = 
		   if cl#has_feature FinalSemanticsFeature 
		   then name_of_final_coalgebra cl
		   else name_of_loose_coalgebra cl
		 in
		   SmartApplication(
		     Term(name_of_class_rel_every cl, Always, []),
		     [Term(cl_name,Always,[]);Term(cl_name,Always,[])]
		   )
	     | Adt( adt,_,_ ) -> 
		 Term(name_of_adt_rel_every adt,Always,[])
	     | Groundtype(id,_) -> 
		 Term(name_of_const_rel_every id.id_token.token_name,
		      Always,[])
	     | _ -> assert(false)
	  )
	in
	  pvs_pp_top_expression
	    (if expr_list = [] 
	     then relevery_term
	     else 
	       SmartApplication(
		 relevery_term, 
		 expr_list))

    | Map (typ, expr_list) ->
	let map_term = 
			(* map is only defined for three kinds of types *)
	  (match typ with
	     | Class( cl,_ ) -> 
		 Term(name_of_class_map cl, Always, [])
	     | Adt( adt,_,_ ) -> 
		 Term(name_of_adt_map adt,Always,[])
	     | Groundtype(id,_) -> 
		 Term(name_of_const_map id.id_token.token_name,
		      Always,[])
	     | _ -> assert(false)
	  )
	in
	  pvs_pp_top_expression
	    (if expr_list = [] 
	     then map_term
	     else 
	       SmartApplication(
		 map_term, 
		 expr_list))

    | Expression form ->
	pvs_pp_top_formula form

    | Comprehension (str, typ, form) ->
	begin
	  open_box 0;
	  print_string "{ ";
	  pvs_pp_var (str, typ);
	  print_string " |";
	  print_break 1 2;
	  open_box 0;
	  pvs_pp_top_formula form;
	  close_box ();
	  print_string "}";
	  close_box ()
	end  
					(* washed away in a pre-pass! *)
    | Projection _ 
    | MethodSelection _
    | Modality _
	-> assert(false)

	
and pvs_pp_top_formula (* : top_formulas -> unit *)
  = function
    | FormLoc( f, _ ) -> pvs_pp_top_formula f
    | True ->
      	print_string "TRUE"
    | False ->
	print_string "FALSE"
    | Not form ->
	begin
	  open_box 0;
	  print_string "NOT(";
	  pvs_pp_top_formula form;
	  print_cut();
	  print_string ")";
	  close_box()
	end
    | And form_list ->
	begin
	  open_box 0; 
	  print_string "(";
	  pp_list pvs_pp_top_formula "(" ")" "TRUE" "AND" form_list; 
	  print_cut();
	  print_string ")";
	  close_box ();
	end
    | Or form_list ->
	begin
	  open_box 0; 
	  print_string "(";
	  pp_list pvs_pp_top_formula "(" ")" "FALSE" "OR" form_list; 
	  print_cut();
	  print_string ")";
	  close_box ();
	end
    | Implies (prem_expr, concl_expr) ->
	begin
	  open_box 0;
	  print_string "("; 
	  pvs_pp_top_formula prem_expr;
	  print_string ")";
	  print_space ();
	  open_box 0;
	  print_string "IMPLIES";
	  print_space();
	  print_string "("; 
	  pvs_pp_top_formula concl_expr;
	  print_cut();
	  print_string ")";
	  close_box(); 
	  close_box()
	end
    | Iff (prem_expr, concl_expr) ->
	begin
	  open_box 0;
	  print_string "("; 
	  pvs_pp_top_formula prem_expr;
	  print_string ")";
	  print_space ();
	  open_box 0;
	  print_string "IFF";
	  print_space();
	  print_string "("; 
	  pvs_pp_top_formula concl_expr;
	  print_cut();
	  print_string ")";
	  close_box(); 
	  close_box()
	end
    | Equal (left_expr, right_expr) ->
	begin
	  open_box 0;
	  pvs_pp_top_expression left_expr;
	  print_break 1 2; 
	  print_string "=";
	  print_space(); 
	  pvs_pp_top_expression right_expr;
	  close_box ()
	end
    | Forall (str_typ_list, form) ->
	(match str_typ_list with
	   | [] ->
	       pvs_pp_top_formula form
	   | l ->
	       begin
		 open_box 0;				(* 1 *)
	    	 print_string "FORALL (";
	    	 (* open_box 0;  *)
		 pp_list pvs_pp_var "" "" "" "," str_typ_list; 
		 (* close_box(); *)
	    	 print_string ") :";
	    	 print_break 1 2;
		 pvs_pp_top_formula form;
		 close_box ()
	       end)
    | Exists (str_typ_list, form) ->
	(match str_typ_list with
	   | [] ->
	       pvs_pp_top_formula form
	   | l ->
	       begin
		 open_box 0;
	    	 print_string "EXISTS (";
	    	 open_box 0; 
		 pp_list pvs_pp_var "" "" "" "," str_typ_list; 
		 close_box();
	    	 print_string ") :";
	    	 print_break 1 2;
		 pvs_pp_top_formula form;
		 close_box ()
	       end)
    | ConstantPredicate str ->
	print_string str
    | Formula expr ->
	pvs_pp_top_expression expr
    | Bisim (typ, ex1, ex2) ->		
	let bisim_term = 
			(* every is only defined for three kinds of types *)
	  (match typ with
	     | Class( cl,_ ) -> 
		 let cl_name = 
		   if cl#has_feature FinalSemanticsFeature 
		   then name_of_final_coalgebra cl
		   else name_of_loose_coalgebra cl
		 in
		   SmartApplication(
		     Term(name_of_greatest_bisim cl, Always, []),
		     [Term(cl_name,Always,[]);Term(cl_name,Always,[])]
		   )
	     | _ -> assert(false)
	  )
	in
	  pvs_pp_top_expression
	    (Application(
	       bisim_term, 
	       Tuple([ex1;ex2])))

    | MetaImplies (prem_expr, concl_expr) ->
	pvs_pp_top_formula (
	  Implies (prem_expr, concl_expr))
    | Obseq (Some(nam,_),ex1,ex2) ->
	begin
	  open_box 0;					    (* 1 *)
	  print_string nam; 
	  print_cut();
	  print_string "(";
	  open_box 0;					    (* 2 *)
	  pvs_pp_top_expression ex1;
	  print_string ",";
	  print_space();
	  pvs_pp_top_expression ex2;
	  close_box();					    (* 2 *)
	  print_string ")";
	  close_box()					    (* 1 *)
	end
    | Obseq (None,_,_) ->
	assert false

and pvs_pp_arg_list al = 
  let pp_arg = function
    | TypeArgument(t) -> pvs_pp_top_type t 
  in
    if al <> [] 
    then
      begin
	print_string "["; 
	pp_list pp_arg "" "" "" "," al;
	print_string "]"
      end

and pvs_pp_var (name, typ) =
      begin
	open_box 2;
	print_string (name ^ ":");
	print_space(); 
	pvs_pp_top_type typ;
	close_box ()
      end

and pvs_pp_var_decl  = function
  | Undeclared(name, typ) ->
      begin
	open_box 2;
	print_string (name ^ ":");
	print_space(); 
	pvs_pp_top_type typ;
	close_box ()
      end
  | Declared(name, typ) ->
      begin
	open_box 2;
	print_string (name);
	close_box ()
      end


and pvs_pp_top_theory_declaration 
  = function
    | Lemma (name, form) -> 
	begin
	  open_box 2;
	  print_string (name ^ " : LEMMA");
	  print_space (); 
	  pvs_pp_top_formula form;
	  close_box()
	end
    | Axiom (name, form) ->
	begin
	  open_box 2;
	  print_string (name ^ " : AXIOM");
	  print_space (); 
	  pvs_pp_top_formula form;
	  close_box()
	end
    | Vardecl (name, typ) ->
	begin
	  open_box 2;
	  print_string (name ^ " : VAR");
	  print_space (); 
	  pvs_pp_top_type typ;
	  close_box()
	end
				(* typeparameters are ignored for Pvs *)
    | Typevardecl(typeparameters, name) ->
	begin
	  open_box 0;
	  print_string (name ^ " : TYPE");
	  close_box()
	end
    | Typedecl (args, name, typ) ->
	begin
	  open_box 2;
	  print_string (name ^ " : TYPE = ");
	  print_space (); 
	  pvs_pp_top_type typ;
	  close_box()
	end
    | Enumdecl (name, con_list) ->
	begin
		open_box 2;
		print_string (name ^ " : TYPE = ");
		print_space();
		print_string "{";
		pp_list (print_string) "" "" "" "," con_list;
		print_string "}";
		close_box();
	end		
    | Defn (name, var_listlist, typ, expr) ->
	begin
	  open_box 2;
	  print_string name;
	  if 
	    var_listlist <> [[]] 
	  then
	      pp_list 
		(pp_list pvs_pp_var_decl "" "" "" ",")
		"(" 
		")" 
		"" 
		"" 
		var_listlist;
	  print_string " :";
	  print_space (); 
	  pvs_pp_top_type typ;
	  print_string " ="; 
	  print_space ();
	  pvs_pp_top_expression expr; 
	  print_space (); 
	  print_string ";";
	  close_box()
	end
    | Defrecfn (name, var_listlist, typ, measure, expr) ->
	begin
	  open_box 2;
	  print_string name;
	  if 
	    var_listlist <> [[]] 
	  then
	      pp_list 
		(pp_list pvs_pp_var_decl "" "" "" ",")
		"(" 
		")" 
		"" 
		"" 
		var_listlist;
	  print_string " : RECURSIVE";
	  print_space (); 
	  pvs_pp_top_type typ;
	  print_string " ="; 
	  print_space ();
	  pvs_pp_top_expression expr; 
	  print_space (); 
	  print_string " MEASURE ";
	  pvs_pp_top_expression measure;
	  print_space ();
	  print_string ";";
	  close_box()
	end
    | Primrecdefn _ -> 
	assert(false)
    | Defnuninterpret (name, var_listlist, typ) ->
	begin
	  open_box 2;
	  print_string name;
	  if 
	    var_listlist <> [[]] 
	  then
	    pp_list 
	      (pp_list pvs_pp_var_decl "" "" "" ",")
	      "(" 
	      ")" 
	      "" 
	      "" 
	      var_listlist;
 	  print_string " :";
	  print_space (); 
	  pvs_pp_top_type typ;
	  close_box()
	end

	(* for PVS some fields are ignored, see top_variant_types.ml *)
    | Datatypedecl(_,_,constructor_list) ->
	List.iter
	  (fun(name, accessor_list, recognizer) ->
	     begin
	       open_box 2;
	       print_string name;
	       if 
		 accessor_list <> [] 
	       then
		 begin
		   print_string "(";
		   (pp_list 
		      (fun (s,t) ->
			 (pvs_pp_top_theory_declaration 
			    (Defnuninterpret(s,[],t))))
		      "" "" "" "," 
		      accessor_list
		   );
		   print_string ")"
		 end;
	       print_string " :";
	       print_space (); 
	       print_string recognizer;
	       close_box();
	       force_newline();
	     end
	  ) constructor_list
    | Comment str ->
	begin
	  open_box 0;
	  List.iter
	    (fun s ->
	      if (String.length s = 0) || (s.[0] <> '%') then
		print_string "% ";
	      print_string s;
	      force_newline();
	    )
	    (split_on_newline str);
	  print_string "% ";
	  close_box()
	end
    | Comment_decl_expr expr ->
	pvs_pp_top_theory_declaration 
	  (Comment 
	     (stringwrapper pvs_pp_top_expression expr))
    | Comment_decl decl ->
	  pvs_pp_top_theory_declaration 
	    (Comment 
	       (stringwrapper pvs_pp_top_theory_declaration decl))
    | Import ilist ->
	if 
	  ilist <> [] 
	then
	  begin
	    open_box 0;
	    print_string "IMPORTING ";
	    open_box 0;
	    pp_list
	      (fun (name, arglist) ->
		 begin
		   print_string name;
		   pvs_pp_arg_list arglist;
		 end) 
	      "" 
	      "" 
	      "" 
	      "," 
	      ilist;
	    close_box ();
	    close_box()
	  end
    | LibImport(lib,name,args) -> 
	begin
	  print_string "IMPORTING ";
	  open_box 0;
	  print_string lib;
	  print_string "@";
	  print_string name;
	  pvs_pp_arg_list args;
	  close_box ()
	end;
	
    | Library(name,path) ->
	begin
	  open_box 0;
	  print_string (name ^ " : Library = ") ;
	  print_break 1 2;
	  print_string ("\"" ^ path ^ "\"") ;
	  close_box()
	end
    | Conversion (name, typ) ->
	begin
	  open_box 0;
	  print_string ("CONVERSION " ^ name ^ " :") ;
	  print_break 1 2;
	  pvs_pp_top_type typ;
	  close_box()
	end
    | Proved (decl, proof) -> (* proofs are treated somewhere else *)
	pvs_pp_top_theory_declaration decl

	  (* ignore Isar ML code *)
    | IsarML _ -> ()


and pvs_pp_param_list param_list =
  let pp_param = function
    | TypeParameter id ->
	print_string (id.id_token.token_name ^ " : TYPE")
  in
    (pp_list pp_param "" "" "" "," param_list)


let pvs_pp_theory th =
  match th#kind with
    | Theory ->
	begin
	  open_box 0;                                   (* 1 *)
	  print_string th#get_name;
	  if 
	    th#get_parameters <> [] 
	  then
	    begin
	      print_break 0 2;
	      print_string "[";
	      pvs_pp_param_list th#get_parameters;
	      print_string "]"
	    end;
	  print_break 1 4;
	  print_string ": THEORY";
	  force_newline();
	  print_string "BEGIN";
	  force_newline();
	  print_break 2 2;
	  open_box 0;					(* 2 *)
	  List.iter
	    (fun decl ->
	       begin
		 pvs_pp_top_theory_declaration decl;
		 force_newline();
		 force_newline()
	       end) 
	    th#get_body;
	  close_box();					(* 2 *)
	  force_newline();
	  print_string ("END " ^ th#get_name);
	  force_newline();
	  force_newline();
          close_box()					(* 1 *)
	end
    | Datatype ->
	begin
	  open_box 0;					(* 1 *)
	  print_string th#get_name;
	  if 
	    th#get_parameters <> [] 
	  then
	    begin
	      print_string "[";
	      pvs_pp_param_list th#get_parameters;
	      print_string "]"
	    end;
	  print_string " : DATATYPE";
	  force_newline();
	  print_string "BEGIN";
	  force_newline();
	  print_break 2 2;
	  open_box 0;					(* 2 *)
	  List.iter
	    (fun decl ->
	       begin
		 pvs_pp_top_theory_declaration decl;
		 force_newline()
	       end) 
	    th#get_body;
	  close_box();					(* 2 *)
	  force_newline();
	  print_string ("END " ^ th#get_name);
	  force_newline();
          close_box()					(* 1 *)
	end
					(* ignore Isabelle delimiters *)
    | IsabelleStartFile
    | IsabelleCloseFile -> ()

(******************************************************************
 ******************************************************************
 *
 * Pretty Lisp
 *)

let rec pvs_pp_lisp_expr = function
  | LAtom s -> print_string s;
  | LList( [] ) -> print_string "()"
  | LList([ex]) -> begin
	  print_space();
	  print_string "(";
	  pvs_pp_lisp_expr ex;
	  print_cut();
	  print_string ")";
	  print_break 1 0;
	end
  | LList( li ) -> begin
	  print_space();
	  print_string "(";
	  pvs_pp_lisp_expr (List.hd li);
	  print_space();
	  open_box 0;
	  ignore(List.fold_left 
		   (fun i ex -> begin
		      pvs_pp_lisp_expr ex; 
		      if i > 1 then print_space();
		      i-1
		    end
		   ) (List.length (List.tl li))  (List.tl li));
	  print_cut();
	  print_string ")";
	  print_break 1 0;
	  close_box();
	end

(******************************************************************
 ******************************************************************
 *
 * Pretty Proofs for prf
 *)

let close_paren i =
  print_string( String.make i ')');;

let rec pvs_pp_proof_commands = function
  | [] -> 0
  | [c] -> pvs_pp_lisp_expr c; 0
  | c :: cl -> begin
	  pvs_pp_lisp_expr c;
	  print_string "(( \"1\"";
	  print_break 1 1;
	  (pvs_pp_proof_commands cl) + 2
	end

let rec pvs_pp_proof_tree = function 
  | PTree( command_list, []) ->
	  open_box 0;
	  let cp = pvs_pp_proof_commands command_list 
	  in
	    close_paren cp;
	    close_box()
  | PTree( cl1, [PTree(cl2, branches)] ) ->
	  pvs_pp_proof_tree (PTree( cl1 @ cl2, branches ))
  | PTree( command_list, branches ) ->
	  open_box 0;
	  let cp1 = pvs_pp_proof_commands command_list 
	  in 
	    open_box 0;
	    print_string "(";
	    print_break 1 0;
	    ignore(List.fold_left
		     (fun i pt -> 
		 	print_string( "(\"" ^ (string_of_int i) ^ "\"" );
		 	pvs_pp_proof_tree pt;
		 	print_string(")");
		 	print_break 1 0;
		 	i+1
		     ) 1 branches);
	    close_paren (cp1 +1);
	    close_box();
	    close_box()


let pvs_pp_proof = function
  | Named_proof(name, pt ) -> 
	  print_string( "(|" ^ name ^ "| \"1\"" );
	  print_break 1 4;
	  open_box 0;
	  pvs_pp_proof_tree pt;
	  print_string( ")");
	  close_box();
	  force_newline()
  | Anon_proof _ -> assert(false)

let pvs_pp_th_proofs thname proofs =
  open_box 0;
  print_string( "(|" ^ thname ^ "|");
  print_break 1 2;
  open_box 0;
  List.iter pvs_pp_proof proofs;
  close_box();
  print_string( ")" );
  close_box()

(******************************************************************
 ******************************************************************
 *
 * Pretty Proofs for edit-proof
 *)

let rec pvs_pp_edit_proof_commands cl =
  List.iter (fun ex -> 
		   force_newline();
		   pvs_pp_lisp_expr ex
		   )
	cl

let rec pvs_pp_edit_proof_tree = function 
  | PTree( command_list, branches ) ->
	  open_box 0;
	  pvs_pp_edit_proof_commands command_list;
	  if branches <> [] then begin
	    print_string "(";
	    print_break 0 1;
	    open_box 0;
	    ignore(List.fold_left
		     (fun i pt -> 
		 	if i <> 1 then force_newline();
		 	print_string( "(\"" ^ (string_of_int i) ^ "\"" );
		 	force_newline();
		 	print_space();
		 	pvs_pp_edit_proof_tree pt;
		 	print_string(")");
		 	i+1
		     ) 1 branches);
	    close_box();
	    print_string ")";
	  end;
	  close_box()


let pvs_pp_edit_proof name tree = 
  open_box 0;
  print_string( "Proof script for " ^ name );
  force_newline();
  print_string "(\"\"";
  pvs_pp_edit_proof_tree tree;
  print_break 0 1;
  print_string ")";
  close_box()
	

(******************************************************************
 ******************************************************************
 *
 * string wrapper section
 *)

let string_of_pvs_top_expression ex = stringwrapper pvs_pp_top_expression ex

let string_of_pvs_argument_list al = stringwrapper pvs_pp_arg_list al


(******************************************************************
 ******************************************************************
 *
 * other utilities
 *)


let pvs_identifier th args decl =
  if th = ""
  then
    decl ^ (string_of_pvs_argument_list args)
  else
    th
    ^ (string_of_pvs_argument_list args)
    ^ "."
    ^ decl


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

