(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 22.4.99 by Hendrik
 *
 * Time-stamp: <Monday 8 October 01 17:58:00 tews@ithif51>
 *
 * type hierarchy for CCSL
 *
 * $Id: classtypes.ml,v 1.14 2002/01/23 16:00:14 tews Exp $
 *
 *)

open Top_variant_types
open Top_classtypes
;;

(***********************************************************************
 ***********************************************************************
 *
 * Ccsl class types
 * 
 * In this file I specialize top_iface_type, top_member_type and
 * top_theory_body_type to the needs of ccsl. For the theories a
 * whole hierarchy is built:
 * 
 * 			  top_pre_theory_body_type
 * 				       |
 * 				       |
 * 			  ccsl_pre_theory_body_type
 * 				/               \
 * 			       /                 \
 * ccsl_pre_virtual_adt_theory_type         ccsl_pre_virtual_class_theory_type
 * 		 | 		 	       	  |
 * 		 |                                |
 * ccsl_pre_adt_theory_type                 ccsl_pre_class_theory_type
 * 
 * 
 * At the bottom ccsl type abbreviations are defined.
 * 
 *)


class type ['member_type, 'theory_body_type, 'super_class] ccsl_pre_iface_type
  =
object
    inherit ['member_type, 'theory_body_type, 'super_class] top_pre_iface_type


    (**********************************************
     * make theory generation real
     *)
    method generate_all_theories
      : 'theory_body_type  list

					(* true, if from the prelude *)
    method belongs_to_prelude : bool

      (* convenience method, tells if self#get_constructors <> [] *)
    method has_constructors : bool

    (**********************************************
     * variances & functor type
     *)

    method get_self_variance : variance_type

    method set_self_variance : variance_type -> unit

    method get_functor_type : functor_type

    method set_functor_type : functor_type -> unit

      (*
       * substitution is implemented by copying,
       * but to minimize the overhead only the absolute necessary
       * fields of an iface are copied
       * That is members, ancestores and components are copied, nothing else.
       * The only methods which work on an instanciated class are
       *
       *  - get_full_ancestors  - add_ancestor         - get_methods
       *  - become_class        - get_ancestors        - get_actions
       *  - become_adt          - get_all_ancestors    - get_constructors
       *  - is_class            - string_of_ancestors  - find_member
       *  - is_adt              - add_member           - find_local_member
       *  - get_token           - get_members          - elaborate_inheritance
       *  - get_name            - get_all_members      - dump_iface
       *  - get_local           - get_attributes       - marshal
       *  - dependencies        - add_component        - get_components
       *)

    method become_instanciated : unit

    method is_instanciated : bool

      (* for instantiation *)
    method set_arguments: 
      ('super_class, 'member_type) top_pre_argument_type list -> unit

    method get_arguments:
      ('super_class, 'member_type) top_pre_argument_type list


    (*******************************************************************
     * final or loose model coalgebra
     *)

    method get_model_coalgebra :
      ('super_class, 'member_type) top_pre_expressions

    method get_model_algebra :
      ('super_class, 'member_type) top_pre_expressions

    method get_model_type_argument :
      ('super_class, 'member_type) top_pre_argument_type list -> 
	('super_class, 'member_type) top_pre_argument_type

    (*******************************************************************
     * inheritance
     *)
	(* During inheritance ancestor classes get 
	 * first instanciated, then in the instanciated copy
	 * the heir is set to the non-instanciated class that inherits
	 *)
    method inherit_to : 'super_class -> unit

    method get_heir : 'super_class option

      (* parameters and arguments *)
    method add_parameter : 
      ('super_class, 'member_type) top_pre_parameter_type -> unit

    method get_parameters : 
      ('super_class, 'member_type) top_pre_parameter_type list

	(* check if argument list is a valid instanciation for this iface *)
    method check_parameters : 
      ('super_class, 'member_type) top_pre_argument_type list -> bool

	(* imports *)
    method add_iface_import :  
      ('super_class, 'member_type) top_pre_importing_type -> unit
    method add_assertion_import : 
      ('super_class, 'member_type) top_pre_importing_type -> unit
    method get_iface_imports : 
      ('super_class, 'member_type) top_pre_importing_type list
    method get_assertion_imports : 
      ('super_class, 'member_type) top_pre_importing_type list

      (*
       * all components with all instanciations must be added !!
       * it is the responsibility of the caller, to traverse the
       * argument list and add components in there first
       * see iter_component_arg and iter_components
       *)
    method add_component : 
      ('super_class, 'member_type) top_pre_component_type -> unit

    method get_components :
      ('super_class, 'member_type) top_pre_component_type list

	(***************************************** 
	 * Assertions
	 * 
	 *)

    method add_assertion : 
      ('super_class, 'member_type) top_pre_assertion_record -> unit
    method add_creation : 
      ('super_class, 'member_type) top_pre_assertion_record  -> unit
    method set_assertions:
      ('super_class, 'member_type) top_pre_assertion_record list -> unit
    method get_assertions :
      ('super_class, 'member_type) top_pre_assertion_record list
    method get_creations :
      ('super_class, 'member_type) top_pre_assertion_record list
    method set_creations :
       ('super_class, 'member_type) top_pre_assertion_record list -> unit     

	(*****************************************
	 *
	 * ancestors
	 *)
      
      (* override *)
    method add_ancestor : 
      ('super_class, 'member_type) top_pre_ancestor_type -> unit


      (* override

	 access through the ancestor type to return 
	 the _resolved_ ancestors;
	 this requires inheritance_ready
      *)
    method get_resolved_ancestors :
      'super_class list  


	(*****************************************
	 *
	 * members
	 *)

      (* override *)
    method add_member : 'member_type -> unit     


    (*******************************************************************
     *
     * local symbol tables
     *)

       (* filled in inheritance pass, retained in instanciations *)
    val mutable inherited_locals :  
      ('super_class, 'member_type) top_pre_scope_type list

    method inherit_locals : 
      ('super_class, 'member_type) top_pre_scope_type list -> unit

    method get_inherited_locals : 
      ('super_class, 'member_type) top_pre_scope_type list

    method find_local :
      string -> ('super_class, 'member_type) top_pre_symbol_type

       (*
 	* a call is only valid, if the inheritance pass is done
 	* maybe raise Member_not_found 
 	*)
    method find_member : 
 	 string -> 'member_type
 
       (* find a member in this class
 	* maybe raise Member_not_found 
 	*)
    method find_local_member : 
 	 string -> 'member_type


    (*******************************************************************
     *
     * definitional extensions
     *)

    val mutable definitions : 
      ('super_class,'member_type) top_pre_definition_record list

    method add_definition : 
      ('super_class,'member_type) top_pre_definition_record -> unit

    method get_definitions : 
      ('super_class,'member_type) top_pre_definition_record list


    (*******************************************************************
     *
     * lifting requests
     *)

      (* extra liftings, liftings for the full class and for the
       * public part are generated by default. Here extra types can be
       * registered, the corresponding liftings appear in the
       * PredicateLifting theory
       *)
    method add_lifting :
      string -> ('super_class, 'member_type) top_pre_types -> unit
    method get_lifting_requests : 
      (string * ('super_class, 'member_type) top_pre_types) list
    method add_rel_lifting :
      string -> ('super_class, 'member_type) top_pre_types -> unit
    method get_rel_lifting_requests : 
      (('super_class, 'member_type) top_pre_types * string) list
    method find_rel_lifting_for_type :
      ('super_class, 'member_type) top_pre_types -> string

	(*****************************************
	 *
	 * Signature stuff
	 *)

    method add_ground_type : 
      ('super_class, 'member_type) top_pre_identifier_record_type -> unit

    method get_all_ground_types : 
      ('super_class, 'member_type) top_pre_identifier_record_type list

  end (* ccsl_pre_iface_type *)


(***********************************************************************
 ***********************************************************************
 * 
 *        Members
 * 
 ***********************************************************************
 ***********************************************************************
 *)

class type ['class_type, 'member_type] ccsl_pre_member_type
  =
  object 
    inherit ['class_type, 'member_type] top_pre_member_type
    constraint 'class_type =
	       < dump_iface : string; get_name : string; .. >

	(* returns the true type of the method
	 *)
    method get_full_type : ('class_type, 'member_type) top_pre_types

	(* returns the type of the method with arguments in
	 * in curried form
	 *)
    method get_curried_type : ('class_type, 'member_type) top_pre_types

        (* returns the class, this method is in
	 *)
    method hosting_class : 'class_type

    (*******************************************************************
     *
     * Renaming
     *)

      (* override to treat symboltable entry *)
    method rename_member : token_type -> unit  

    (*******************************************************************
     *
     * Class stuff
     *)

    method needs_lifting : bool

    method register_update_method : 'member_type -> unit

    method get_update_method : 'member_type

    (*******************************************************************
     *
     * Adt stuff
     *)
	       (* register accessor names, if this is an adt constructor,
		* raise an assertion, if this is not an adt constructor
		*)
    method register_accessors : 'member_type list -> unit

    method get_ith_accessor : int -> 'member_type 

    method get_accessors : 'member_type list

    method register_recognizer : 'member_type -> unit
    method get_recognizer : 'member_type

  end (* ccsl_pre_member_type *)



(***********************************************************************
 ***********************************************************************
 * 
 *        Theories top
 * 
 ***********************************************************************
 ***********************************************************************
 *)


class type virtual ['class_type, 'member_type] ccsl_pre_theory_body_type =
  object
    inherit ['class_type, 'member_type] top_pre_theory_body_type

      (* add ccsl specific stuff *)

      (* the various versions of Self : Self1 Self2 *)
    val self1 : ('class_type, 'member_type) top_pre_identifier_record_type

    val self2 : ('class_type, 'member_type) top_pre_identifier_record_type

    val self_parameter : ('class_type, 'member_type) top_pre_parameter_type

    method self1_parameter :
      ('class_type, 'member_type) top_pre_parameter_type

    method self2_parameter : 
      ('class_type, 'member_type) top_pre_parameter_type

    val self_argument :
      ('class_type, 'member_type) top_pre_argument_type

    method self1_argument : 
      ('class_type, 'member_type) top_pre_argument_type

    method self2_argument :
      ('class_type, 'member_type) top_pre_argument_type

      (* original parameters *)
    val orig_parameters :
      ('class_type, 'member_type) top_pre_parameter_type list

      (* argument list corresponding to orig_parameters *)
    val orig_arguments :
      ('class_type, 'member_type) top_pre_argument_type list

      (* compute simple theory parameter list,
       * containing self and the origional parameters
       *)
    method simple_parameters :
      ('class_type, 'member_type) top_pre_parameter_type list

    method simple_arguments :
      ('class_type, 'member_type) top_pre_argument_type list

      (* compute parameter list with two self's *)
    method double_self_parameters :
      ('class_type, 'member_type) top_pre_parameter_type list

    method double_self_arguments :
      ('class_type, 'member_type) top_pre_argument_type list

    method double_parameter_ids : 
      (('class_type, 'member_type) top_pre_identifier_record_type *
       ('class_type, 'member_type) top_pre_identifier_record_type) list

      (* double parameters without self, 
       * generated from double_parameter_ids 
       *)
    method double_parameters :
      (('class_type, 'member_type) top_pre_parameter_type list) *
      (('class_type, 'member_type) top_pre_parameter_type list)

      (* double arguments without self *)
    method double_arguments :
      (('class_type, 'member_type) top_pre_argument_type list) *
      (('class_type, 'member_type) top_pre_argument_type list)


    val ns : ('class_type, 'member_type) Name_space.t

  
  end (* ccsl_pre_theory_body_type *)



(***********************************************************************
 ***********************************************************************
 * 
 *        Theories for classes (virtual)
 * 
 ***********************************************************************
 ***********************************************************************
 *)


class type virtual 
  ['class_type, 'member_type] ccsl_pre_virtual_class_theory_type =
  object

    inherit ['class_type, 'member_type] ccsl_pre_theory_body_type

    method get_method_functor_type : ('class_type, 'member_type) top_pre_types

    method get_constructor_functor_type : 
      ('class_type, 'member_type) top_pre_types

      (* the type of the coalgebra *)
    method coalgebra_type : ('class_type, 'member_type) top_pre_types

    method coalgebra1_type : ('class_type, 'member_type) top_pre_types

    method coalgebra2_type : ('class_type, 'member_type) top_pre_types

    method get_member_fun : 
      'member_type -> ('class_type, 'member_type) top_pre_expressions

    method get_iface_fun : 
      ('class_type, 'member_type) top_pre_types -> 
	('class_type, 'member_type) top_pre_expressions

    method assert_coalgebra_type : ('class_type, 'member_type) top_pre_types

    method assert_coalgebra1_type : ('class_type, 'member_type) top_pre_types

    method assert_coalgebra2_type : ('class_type, 'member_type) top_pre_types

      (* a complete variable declaration for the coalgebra *)
    method coalgebra_decl :
      ('class_type, 'member_type) top_pre_theory_declaration

    method coalgebra1_decl : 
      ('class_type, 'member_type) top_pre_theory_declaration

    method coalgebra2_decl :
      ('class_type, 'member_type) top_pre_theory_declaration

    method assert_coalgebra_decl :
      ('class_type, 'member_type) top_pre_theory_declaration

    method assert_coalgebra1_decl :
      ('class_type, 'member_type) top_pre_theory_declaration

    method assert_coalgebra2_decl :
      ('class_type, 'member_type) top_pre_theory_declaration

    val coalgebra_term : ('class_type, 'member_type) top_pre_expressions

    val coalgebra1_term : ('class_type, 'member_type) top_pre_expressions

    val coalgebra2_term : ('class_type, 'member_type) top_pre_expressions

      (* This is a bit strange! Since we model the algebra functor
       * as Record of functions (instead of a sum of the domains) the
       * type of the algebra is then just this record
       * (instead of a function going to Self)
       *)
    method algebra_type : ('class_type, 'member_type) top_pre_types

    method assert_algebra_type : ('class_type, 'member_type) top_pre_types

    method assert_coalgebra_hypothesis : 
      ('class_type, 'member_type) top_pre_formulas

    method creation_algebra_hypothesis : 
      ('class_type, 'member_type) top_pre_formulas

      (* a complete variable declaration for the constructor algebra *)
    method algebra_decl :
      ('class_type, 'member_type) top_pre_theory_declaration

    val algebra_term : ('class_type, 'member_type) top_pre_expressions

  end (* ccsl_pre_virtual_class_theory_type *)



(***********************************************************************
 ***********************************************************************
 * 
 *        Theories for classes
 * 
 ***********************************************************************
 ***********************************************************************
 *)


class type ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object
    inherit ['class_type, 'member_type] ccsl_pre_virtual_class_theory_type

      (* make the inherited virtual method real *)
    method get_name :
      string

    method get_parameters :
      ('class_type, 'member_type) top_pre_parameter_type list

    method get_file_name :
      string

    method make_body :
      ('class_type, 'member_type) top_pre_theory_declaration list

    method get_proofs :
      ('class_type, 'member_type) top_pre_proof_file_entry list

  end (* ccsl_pre_class_theory_type *)


(***********************************************************************
 ***********************************************************************
 * 
 *        Theories for adts (virtual)
 * 
 ***********************************************************************
 ***********************************************************************
 *)


class type virtual ['class_type, 'member_type] 
  ccsl_pre_virtual_adt_theory_type =
  object
    inherit ['class_type, 'member_type] ccsl_pre_theory_body_type

    method this_adt_type : ('class_type, 'member_type) top_pre_types
end



(***********************************************************************
 ***********************************************************************
 * 
 *        Theories for adts 
 * 
 ***********************************************************************
 ***********************************************************************
 *)


class type ['class_type, 'member_type] ccsl_pre_adt_theory_type =
  object
    inherit ['class_type, 'member_type] ccsl_pre_virtual_adt_theory_type

      (* make the inherited virtual method real *)
    method get_name :
      string

    method get_parameters :
      ('class_type, 'member_type) top_pre_parameter_type list

    method get_file_name :
      string

    method make_body :
      ('class_type, 'member_type) top_pre_theory_declaration list

    method get_proofs :
      ('class_type, 'member_type) top_pre_proof_file_entry list

  end (* ccsl_pre_adt_theory_type *)



(***********************************************************************
 ***********************************************************************
 *
 * Instantiated types
 *
 *
 *)


class type ccsl_iface_type
  =  [ccsl_member_type,
      (ccsl_iface_type, ccsl_member_type) ccsl_pre_theory_body_type,
      ccsl_iface_type] ccsl_pre_iface_type

and  ccsl_member_type =
    [ccsl_iface_type, ccsl_member_type] ccsl_pre_member_type


type ccsl_top_types = (ccsl_iface_type, ccsl_member_type) top_pre_types


type ccsl_inst_iface_type = 
    (ccsl_iface_type, ccsl_member_type) top_pre_inst_iface_type

type ccsl_theory_body_type =
    (ccsl_iface_type, ccsl_member_type) ccsl_pre_theory_body_type


class type ccsl_adt_theory_type = 
	  [ccsl_iface_type, ccsl_member_type] ccsl_pre_adt_theory_type

class type ccsl_class_theory_type = 
	  [ccsl_iface_type, ccsl_member_type] ccsl_pre_class_theory_type


(* From top_variant_types *)

type ccsl_input_types =
    ccsl_top_types
(********************** allowed
 *   | Groundtype of 
 * 	('cl, 'mem) top_pre_identifier_record_type * 
 * 	('cl, 'mem) top_pre_argument_type list
 *   | BoundTypeVariable of string
 *   | Self
 *   | Carrier
 *   | Bool
 *   | Function of ('cl top_pre_types * 'cl top_pre_types)
 *   | Product of 'cl top_pre_types list
 *   | Class of 'cl * 'cl top_pre_argument_type list
 *   | Adt of 'cl * argument_flag * 'cl top_pre_argument_type list
 ********************** not allowed
 *   | TypeConstant of string * argument_flag * 'cl top_pre_argument_type list
 *   | Predtype of 'cl top_pre_formulas
 *   | Record of (string * 'cl top_pre_types) list
 *   | Array of 'cl top_pre_types * int
 *   | StatResult of 'cl top_pre_types
 *   | ExprResult of 'cl top_pre_types * 'cl top_pre_types
 *   | UnresolvedType of token_type container list
 *   | IFace of 'cl * argument_flag * 'cl top_pre_argument_type list
 *)

type ccsl_output_types =
    ccsl_top_types
(********************** allowed
 *   | Groundtype of string
 *   | TypeConstant of string * argument_flag * 'cl top_pre_argument_type list
 *   | BoundTypeVariable of 'cl top_pre_identifier_record_type
 *   | Self
 *   | Carrier
 *   | Function of ('cl top_pre_types * 'cl top_pre_types)
 *   | Product of 'cl top_pre_types list
 *   | Class of 'cl * 'cl top_pre_argument_type list
 *   | Adt of 'cl * argument_flag * 'cl top_pre_argument_type list
 *   | Predtype of 'cl top_pre_formulas
 *   | Record of (string * 'cl top_pre_types) list
 *   | IFace of 'cl * arbument_flag * 'cl top_pre_argument_type list
 ********************** not allowed
 *   | Array of 'cl top_pre_types * int
 *   | StatResult of 'cl top_pre_types
 *   | ExprResult of 'cl top_pre_types * 'cl top_pre_types
 *   | UnresolvedType of token_type container list
 *)


type ccsl_assertion =
    (ccsl_iface_type, ccsl_member_type) top_pre_assertion

type ccsl_assertion_record =
    (ccsl_iface_type, ccsl_member_type) top_pre_assertion_record

type ccsl_basic_expressions = 
    (ccsl_iface_type, ccsl_member_type) top_pre_basic_expressions
(********************** allowed
 *    | TermVar of ('cl, 'mem) top_pre_identifier_record_type
 *    | Member of  ('cl, 'mem) top_pre_types option * 'mem container 
 *  
 ********************** not allowed
 *)


type ccsl_expressions =
    (ccsl_iface_type, ccsl_member_type) top_pre_expressions
(********************** allowed
 *    | BasicExpr of ('cl,'mem) top_pre_basic_expressions
 *    | TypedTerm of ('cl, 'mem) top_pre_expressions * 
 *                     ('cl, 'mem) top_pre_types
 *    | MethodSelection of
 *  	  ('cl, 'mem) top_pre_expressions * 
 *  	  ('cl, 'mem) top_pre_types option * 'mem container
 *    | Tuple of ('cl, 'mem) top_pre_expressions list
 *    | Projection of int * int
 *    | Abstraction of 
 *  	  (string * ('cl, 'mem) top_pre_types) list * 
 *  	  ('cl, 'mem) top_pre_expressions
 *    | Application of
 *  	  ('cl, 'mem) top_pre_expressions *
 *  	  ('cl, 'mem) top_pre_expressions
 *  	    (* Sidecondition in SmartApplication: the list must be non-empty *)
 *    | Let of
 *  	  (string * ('cl, 'mem) top_pre_types option * 
 *  	     ('cl, 'mem) top_pre_expressions) list * 
 *  	  ('cl, 'mem) top_pre_expressions
 *  	      (* Sidecondition: empty 'a list raises 
 *  	       * exception *)
 *    | If of 
 *  	  (('cl, 'mem) top_pre_formulas * 
 *             ('cl, 'mem) top_pre_expressions) list 
 *  	  * ('cl, 'mem) top_pre_expressions
 *    | CCSL_Case of
 *  	  ('cl, 'mem) top_pre_expressions *
 *  	  ('mem container * ('cl, 'mem) top_pre_identifier_record_type list * 
 *  	   ('cl, 'mem) top_pre_expressions) list
 *    | Box of
 *  	   ('cl, 'mem) top_pre_types *
 *  	   ('cl, 'mem) top_pre_expressions * token_type list
 *    | Diamond of
 *  	   ('cl, 'mem) top_pre_types *
 *  	   ('cl, 'mem) top_pre_expressions * token_type list
 *  
 *    | Expression of ('cl, 'mem) top_pre_formulas
 *  
 *
 ************************ not allowed
 *    | Term of string * argument_flag * ('cl, 'mem) top_pre_argument_type list
 *    | QualifiedTerm of 
 *  	  string * argument_flag *
 *  	  (('cl, 'mem) top_pre_argument_type) list * 
 *  	  string
 *    | RecordTuple of (string * ('cl, 'mem) top_pre_expressions) list
 *    | RecordSelection of string * ('cl, 'mem) top_pre_expressions
 *    | RecordUpdate of 
 *  	  ('cl, 'mem) top_pre_expressions * 
 *  	  (string * ('cl, 'mem) top_pre_expressions) list
 *    | List of ('cl, 'mem) top_pre_expressions list
 *    | SmartApplication of
 *  	  ('cl, 'mem) top_pre_expressions *
 *  	  ('cl, 'mem) top_pre_expressions list
 *    | FunUpdate of
 *  	  ('cl, 'mem) top_pre_expressions * 
 *  	  (('cl, 'mem) top_pre_expressions * 
 *  	      ('cl, 'mem) top_pre_expressions) list
 *    | Case of
 *  	  ('cl, 'mem) top_pre_expressions * 
 *  	  (('cl, 'mem) top_pre_expressions * 
 *         ('cl, 'mem) top_pre_expressions) list
 *    | Reduce of
 *  	  'cl * argument_flag * ('cl, 'mem) top_pre_argument_type list 
 *    | Every of ('cl, 'mem) top_pre_types * 
 *                ('cl, 'mem) top_pre_expressions list 
 *  
 *    | RelEvery of
 *  	  ('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_expressions list 
 *    | Map of
 *  	  ('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_expressions list
 *    | Infix of string * ('cl, 'mem) top_pre_expressions list
 *    | Comment_str of string * ('cl, 'mem) top_pre_expressions
 *    | Comment_expr of 
 *  	  ('cl, 'mem) top_pre_expressions * 
 *  	  ('cl, 'mem) top_pre_types * 
 *  	  ('cl, 'mem) top_pre_expressions
 *    | Comprehension of 
 *  	  string * ('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_formulas
 *   *)


type ccsl_formulas =
    (ccsl_iface_type, ccsl_member_type) top_pre_formulas
(********************** allowed
 *    | True
 *    | False
 *    | Not of ('cl, 'mem) top_pre_formulas
 *    | And of ('cl, 'mem) top_pre_formulas list
 *    | Or of ('cl, 'mem) top_pre_formulas list
 *    | Implies of 
 *  	  ('cl, 'mem) top_pre_formulas * ('cl, 'mem) top_pre_formulas
 *    | Iff of
 *  	  ('cl, 'mem) top_pre_formulas * ('cl, 'mem) top_pre_formulas
 *    | Equal of
 *  	  ('cl, 'mem) top_pre_expressions * ('cl, 'mem) top_pre_expressions
 *    | Forall of
 *  	  (string * ('cl, 'mem) top_pre_types) list * 
 *  	  ('cl, 'mem) top_pre_formulas
 *    | Exists of
 *  	  (string * ('cl, 'mem) top_pre_types) list * 
 *  	  ('cl, 'mem) top_pre_formulas
 *    | Formula of ('cl, 'mem) top_pre_expressions
 *    | Obseq of
 *  	  (string * ('cl, 'mem) top_pre_types) option *
 *  	  ('cl, 'mem) top_pre_expressions * 
 *  	  ('cl, 'mem) top_pre_expressions
 *  
 *  
 *********************** not allowed
 *    | ConstantPredicate of string
 *    | MetaImplies of
 *  	  ('cl, 'mem) top_pre_formulas * ('cl, 'mem) top_pre_formulas
 *    | MetaForall of
 *  	  (string * ('cl, 'mem) top_pre_types) list * 
 *  	  ('cl, 'mem) top_pre_formulas
 *  	  (* bisimilarity for a class: the first argument must be a class *)
 *    | Bisim of
 *  	  'cl * ('cl, 'mem) top_pre_expressions * 
 *  	  ('cl, 'mem) top_pre_expressions
 *)


type ccsl_visibility = top_visibility
 (**************** allowed constructors
  * | NoVisibility
  * | Public 
  * | Private
  **************** not allowed
  * | Protected
  * | Friendly
  * | Abstract
  * | Final
  * | Static
  * | Synchronized
  * | Volatile
  * | Native
  * | Transient
  *)

type ccsl_importing_type = 
    (ccsl_iface_type, ccsl_member_type) top_pre_importing_type

type ccsl_theory_declaration =
    (ccsl_iface_type, ccsl_member_type) top_pre_theory_declaration

type ccsl_parameter_type =
    (ccsl_iface_type, ccsl_member_type) top_pre_parameter_type

type ccsl_argument_type =
    (ccsl_iface_type, ccsl_member_type) top_pre_argument_type

type ccsl_origin_type =
   (ccsl_iface_type, ccsl_member_type) Top_variant_types.top_pre_origin_type

type ccsl_identifier_record_type =
    (ccsl_iface_type, ccsl_member_type) top_pre_identifier_record_type

(*********** restrictions in use
 * id_origin is one of CCSL_TypeParameter, CCSL_ValueParameter
 * relevance of id_type depends on id_origin
 * id_position equals 0 and is not used
 *)

type ccsl_renaming =
    (ccsl_iface_type, ccsl_member_type) top_pre_renaming

type ccsl_ancestor_type =
    (ccsl_iface_type,
     ccsl_member_type) top_pre_ancestor_type

(************ allowed
 *   | Unresolved_renaming of
 * 	 'cl *
 * 	 'cl top_pre_argument_type list *
 * 	 'cl top_pre_renaming list
 *   | Resolved_renaming of
 * 	 'cl *
 * 	 'cl top_pre_argument_type list *
 * 	 'mem list
 *********** not allowed
 *   | Unresolved_ancestor of token_type
 *   | Resolved_ancestor of 'cl
 *)

type ccsl_shape_type =
    (ccsl_input_types, ccsl_argument_type list) top_pre_shape_type
(************** allowed
 *   | No_overloading
 *   | Instantiated of 'b 
 ************** not allowed
 *   | Overloading of 'a
 *)

type ccsl_symbol_type =
    (ccsl_iface_type,
     ccsl_member_type) top_pre_symbol_type


type ccsl_symbol_table_type =
    (top_name_space, ccsl_shape_type, ccsl_symbol_type) Table.global
    

type ccsl_scope_type =
    (top_name_space, ccsl_shape_type, ccsl_symbol_type) Table.local

(*************** allowed
 *   | CCSL_GroundTypeSymb of token_type
 *   | TypevarSymb of token_type
 *   | AttributeSymb of 'member
 *   | MemberSymb of 'member
 *   | ClassSymb of 'cl
 *   | InstClassSymb of 'cl
 *   | AdtSymb of 'cl
 *   | InstAdtSymb of 'cl
 ********** not allowed
 *   | VarSymb of token_type * 'cl top_pre_types * 'cl top_pre_identifier_type
 *   | IfaceSymb of 'cl
 *)


type ccsl_name_space_type = (ccsl_iface_type, ccsl_member_type) Name_space.t

type ccsl_ast =
    (ccsl_iface_type, ccsl_member_type) top_pre_ast
(************** allowed
 *   | CCSL_TypeAbbrev_dec of token_type * 'cl top_pre_identifier_record_type
 *   | CCSL_GroundType_dec of token_type
 *   | Spec_class_dec of 'cl
 *   | CCSL_adt_dec of 'cl
 ************** not allowed
 *   | Java_iface_dec of 'cl
 *)


(***********************************************************************
 *
 * In the next class, we define some extra methods that are required
 * in all classes that deal with methodwise invariants and boxes.
 * 
 *
 **********************************************************************)
       
class type ccsl_method_inv_helper_type 
  = object

(* 
 * 	 This is only for method invariances
 * 
 *)

    val mutable name_of_list : string

    val mutable name_of_step : string

    val mutable name_of_path1 : string

    val mutable name_of_path2 : string

    method private list_adt : ccsl_iface_type

    method private list_type :  ccsl_top_types

    method private list_decl :  ccsl_theory_declaration

    method private list_term :  ccsl_expressions

    method private method_enum_type : ccsl_top_types

    method private method_decl : ccsl_theory_declaration

    method private method_term : ccsl_expressions
  end       



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** End: ***)
