(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Monday 8 October 01 17:58:00 tews@ithif51>
 *
 * The inheritance pass
 * 
 *    - copy methods from superclasses in the local symboltable 
 *    - apply renaming if requested
 *    - try automatic renaming otherwise
 * 
 * $Id: inherit.ml,v 1.10 2002/01/23 16:00:15 tews Exp $
 *
 *)


exception Inheritance_error

open Util
open Error
open Global
open Top_variant_types
open Top_variant_types_util
open Types_util
open Substitution
open Member_class
;;

(*********************************************************
 *
 * Error messages
 *
 *)

let error_msg_old cl old_name = 
    "Invalid renaming clause: Method " ^
    old_name.token_name ^ " " ^
    " not defined in class " ^
    cl#get_name ^
    "\n"


let member_decl_message mem =
  mem#get_name ^ " " ^
  "(declared in class " ^
  mem#hosting_class#get_name ^
  (if mem#is_renamed 
   then ""
   else (" as " ^ mem#original_name)
  ) ^ ")" 


let error_msg_new cl new_name new_clash = 
  "Invalid renaming clause: Method " ^
  (member_decl_message new_clash) ^
  " already defined in class " ^
  cl#get_name ^
  "\n"


let name_clash_message (cl : Classtypes.ccsl_iface_type)  mem = 
  "Name clash in class " ^
  cl#get_name ^
  "\n" ^
  (member_decl_message mem) ^ "\n" ^
  (member_decl_message (cl#find_member (mem#get_name))) ^ "\n"


let automatic_super_warning cl mem new_name =
  "Method " ^   
  (member_decl_message mem) ^ 
  " automatically renamed in " ^ new_name ^ 
  " to avoid name clashes"

let d s = 
  if debug_level _DEBUG_INHERITANCE then
    print_verbose s

(*********************************************************
 *
 * Utility functions
 *
 *)


let find_method cl name = 
  let m = cl#find_member name in
    if m#is_action then m
    else raise Member_not_found


let method_absent cl name = 
  try 
    ignore(find_method cl name); false 
  with 
    | Member_not_found -> true


let automatic_super_rename (heir : Classtypes.ccsl_iface_type) cl mem = 
  let new_name = mem#hosting_class#get_name ^ "_" ^ mem#get_name in
  let new_tok = {token_name = new_name; loc = None }
  in
    if method_absent cl new_name then
      begin
	warning_message (remove_option heir#get_token.loc)
	  (automatic_super_warning cl mem new_name);
	mem#rename_member new_tok;
	true
      end
    else
      false
      
      

(*********************************************************
 *
 * Renaming
 *
 *)


let process_renaming cl renaming = 
   (* method names are unique in a class structure *)
  let do_clause (old_name,_, new_name) = 
    try
      let mem = find_method cl old_name.token_name in
	if method_absent cl new_name.token_name then
	  begin
	    mem#rename_member new_name;
	    mem
	  end
	else
	  let new_clash = find_method cl new_name.token_name 
	  in
	    begin
	      error_message (remove_option new_name.loc)
		(error_msg_new cl new_name new_clash);
	      raise Inheritance_error
	    end
    with
      | Member_not_found -> 
	  begin
	    error_message (remove_option old_name.loc)
	      (error_msg_old cl old_name);
	    raise Inheritance_error
	  end
  in
    List.map do_clause renaming


let do_super_class heir child args renaming = 
  let _ = assert(not heir#is_instanciated) in 
  let _ = assert(not child#is_instanciated) in
  let _ = assert(not heir#inheritance_ready) in 
  let _ = assert(child#inheritance_ready) in
  let inst_child = instanciate child args in
  let renamed_members = process_renaming inst_child renaming in
    Resolved_renaming(child, args, renamed_members, inst_child)


(*********************************************************
 *
 * Inheritance
 *
 *)

let inherit_class (heir : Classtypes.ccsl_iface_type)  = function
  | Resolved_renaming(ancestor, _, renamed_members, inst_anc) ->
(* 	 (* debugging support *)
 * 	 let heir_name = heir#get_name in
 * 	 let anc_name = inst_anc#get_name in
 * 	 (* end debugging support *)
 *)
      let was_renamed mem = 
	List.exists (fun m -> m = mem) renamed_members
      in
      let check_member mem = 
	begin
	  if not (method_absent heir mem#get_name) then
	    if not (was_renamed mem) then
	      begin
		if not (automatic_super_rename heir inst_anc mem) then
		  begin
		    error_message (remove_option heir#get_token.loc)
		      (name_clash_message heir mem);
		    raise Inheritance_error
		  end
	      end
	    else
	      begin
		error_message (remove_option heir#get_token.loc)
		  (name_clash_message heir mem);
		raise Inheritance_error
	      end;
	  mem#inherited
	end
      in
	begin
	  List.iter check_member (inst_anc#get_all_members);
	  heir#inherit_locals (inst_anc#get_inherited_locals);
	  inst_anc#inherit_to heir
	end
  | _ -> assert(false); raise Internal_error


let do_class heir  = 
  let _ = d ("Inheritance in " ^ heir#get_name ) in
  let nanc = List.map
    (function
					(* do instanciation & renamings *)
       | Unresolved_renaming(cl, al, rename) -> 
	   do_super_class heir cl al rename
					(* no renaming done before *)
       | Resolved_renaming _ -> (assert(false); raise Internal_error)
					(* no Java stuff *)
       | Unresolved_ancestor _
       | Resolved_ancestor _ -> (assert(false); raise Internal_error)
    ) 
    heir#get_ancestors
  in
  let _ = heir#set_ancestors nanc in
  let _ = heir#inheritance_done in		(* enable find_member *)
  let _ =
    List.iter (inherit_class heir) nanc;
    (match heir#get_all_actions with
       | [] -> (error_message (remove_option heir#get_token.loc)
		  ("A class specification must contain at least one method " ^
		   "or attribute.");
		raise Inheritance_error
	       )
       | _ -> ()
    )
  in
    if heir#has_feature FinalSemanticsFeature then
					(* define coreduce *)
      let tv = FreeTypeVariable(Type_variable.fresh()) in 
      let subst = [ Self, tv  ] in
      let do_type mem = ccsl_substitute_types subst mem#get_full_type in
      let coreduce_arg_typ = match heir#get_all_sig_actions with
	| [ c ] -> do_type c
	| cl -> Product(List.map do_type cl)
      in
      let coreduce_type = Function( coreduce_arg_typ, Function(tv, Self))
      in
      let coreduce_tok = {token_name = (Names.name_of_class_coreduce heir);
			  loc = None};
      in
      let coreduce_mem = new ccsl_pre_member_class heir coreduce_tok []
			   (Product[]) coreduce_type
			   NoVisibility Class_Coreduce
      in
	heir#add_member coreduce_mem
    else
      ()


let do_adt adt = 
  begin
    d ("Inheritance in " ^ adt#get_name );
    adt#inheritance_done
  end

let do_sig si = 
  begin
    d ("Inheritance in " ^ si#get_name );
    si#inheritance_done
  end


let inherit_ast = function
  | CCSL_class_dec cl -> do_class cl
  | CCSL_adt_dec adt -> do_adt adt
  | CCSL_sig_dec si -> do_sig si
  | CCSL_TypeAbbrev_dec _ -> ()


let ccsl_inheritance_pass (ast: Classtypes.ccsl_ast list) = 
    List.iter inherit_ast ast;;


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** End: ***)

