(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 1.10.97 by Hendrik
 *
 * Time-stamp: <Monday 8 October 01 17:58:00 tews@ithif51>
 *
 * Utility functions for ccsl parser
 *
 * $Id: parser.mli,v 1.12 2002/01/24 14:44:01 tews Exp $
 *
 *)

open Top_variant_types
open Classtypes;;

    (* keeps the last lexed location, for parse errors *)

val last_loc : location_type ref


    (* At some places we need inherited attributes in the grammar. 
     * For instance the reduction rules for methods and constructors 
     * must have access to the currently created iface. 
     * 
     * We use a global variable, holding the currently parsed iface, 
     * whenever available. The function cif, ccl and cadt 
     * provides some shortcut for 
     * access
     *)

type parsing_type = 
  | Parse_Nothing
  | Parse_IFace of ccsl_iface_type
  | Parse_Class of  ccsl_iface_type 
  | Parse_Adt of ccsl_iface_type
  | Parse_Sig of ccsl_iface_type

val current_iface : parsing_type ref

val ciface     : unit -> ccsl_iface_type
val ccl        : unit -> ccsl_iface_type
val cadt       : unit -> ccsl_iface_type
val csig       : unit -> ccsl_iface_type


    (* same applies to current_visibility. Visibility modifiers
     * occur at the beginning of a method section.
     * Their value is stored here
     *)
val current_visibility : top_visibility ref

  
    (* the internal representation of nat *) 
val ground_type_nat : location_type -> ccsl_input_types

    (* make a new location spanning the two arguments *)
val new_loc : location_type -> location_type -> location_type


(**********************************************************************
 **********************************************************************
 * 
 * context checks, type checks
 *
 *)

    (* checks if the name at the end of a data specification is matched *)
val check_closing_id : string -> token_type -> unit


(**********************************************************************
 **********************************************************************
 * 
 * make approbriate entries in the symboltable
 *
 *)


    (* action for paramdeclaration 1*)
val add_type_parameter : variance_type -> token_type -> ccsl_parameter_type

    (* check for variance <> (?,?) *)
val do_exact_variance : location_type -> int -> int -> variance_type

    (* action for declaration 1 *)
val add_class : ccsl_iface_type -> ccsl_ast option

    (* action for declaration 1.5 *)
val add_adt : ccsl_iface_type -> ccsl_ast option

    (* action for ground types *)
val add_groundtype : token_type -> ccsl_parameter_type list -> ccsl_ast option

    (* action for ground terms *)
val add_groundterm : token_type -> ccsl_input_types -> ccsl_ast option

    (* action for infix ground terms *)
val add_infix_groundterm : token_type -> ccsl_input_types -> ccsl_ast option

    (* action for typedef typeabbrev  *)
val add_typeabbrev : token_type -> ccsl_input_types -> ccsl_ast option

    (* action for signatures *)
val add_sig : ccsl_iface_type -> ccsl_ast option

(***********************************************************************
 * 
 * Semantic action section
 *
 *)

   (* action for BEGIN ID *)
val do_iface_start : token_type -> unit

   (* action for end of ADT *)
val do_adt_end: location_type -> token_type -> ccsl_iface_type

   (* action for end of Classes *)
val do_class_end: 
  location_type -> ccsl_importing_type list -> token_type -> ccsl_iface_type

   (* action for end of ground signatures *)
val do_sig_end: 
  location_type -> ccsl_importing_type list -> token_type -> ccsl_iface_type

    (* action for pvs_type Product *)
val do_type_product : 
  ccsl_input_types list -> location_type -> ccsl_input_types

    (* action for type application *)
val do_type_appl : token_type -> ccsl_argument_type list -> ccsl_input_types

    (* action for the optional class in Always and eventually *)
val do_instclass : token_type -> ccsl_argument_type list -> ccsl_input_types

    (* action for inherit: find a class or print an error *)
val do_inherit : token_type -> ccsl_argument_type list -> 
  ccsl_renaming list -> ccsl_ancestor_type

    (* action for identifiers in argument lists *)
val do_argument_id : token_type -> ccsl_argument_type

val do_qualified_type_id : 
  token_type option * ccsl_argument_type list option * token_type -> 
      ccsl_input_types

    (* do a constant class constructor *)
val do_const_class_constructor : token_type ->  ccsl_input_types -> unit 

    (* do a parametrized class constructor *)
val do_var_class_constructor : token_type -> ccsl_input_types -> ccsl_input_types -> unit 

    (* do a attribute *)
val do_attribute : token_type -> ccsl_input_types -> ccsl_input_types -> unit 

    (* do a method *)
val do_method : token_type ->  ccsl_input_types -> ccsl_input_types -> unit 

    (* do class imports *)
val do_class_imports : ccsl_importing_type list -> unit

    (* do a definitional extension *)
val do_definition : 
  token_type -> ccsl_input_types -> ccsl_input_types -> ccsl_assertion -> unit

    (* do assertion imports, var declarations *)
val do_assertion_start : 
  ccsl_importing_type list -> token_type * ccsl_input_types ->
      (token_type * ccsl_input_types) list -> 
      ccsl_identifier_record_type * ccsl_identifier_record_type list

    (* do an assertion *)
val do_assertion : 
  token_type -> 
    ccsl_identifier_record_type * ccsl_identifier_record_type list -> 
      ccsl_assertion -> unit

    (* do assertion imports, var declarations *)
val do_creation_start : 
  location_type -> 
    ccsl_importing_type list -> 
      (token_type * ccsl_input_types) list -> 
        ccsl_identifier_record_type list

    (* do a creation condition *)
val do_creation : token_type -> ccsl_identifier_record_type list -> 
  ccsl_assertion -> unit

    (* add a type to a ground signature *)
val do_sig_type_decl : token_type -> unit

    (* add a constant symbol to a ground signature *)
val do_sig_const : token_type -> ccsl_input_types -> unit

    (* add a infix constant symbol to a ground signature *)
val do_sig_infix_const : token_type -> ccsl_input_types -> unit


   (* Create an identifier record for a CCSL_Var
    *)
val do_id_declaration :
  token_type * ccsl_input_types -> ccsl_identifier_record_type


   (* creates an optional selfvar "x", if nothing was declared *)
val do_selfvar_default : unit -> token_type * ccsl_input_types


    (* doing the constant adt constructor, name, accessor list, 
     * codomain type (must be `Carrier)
     *)
val do_const_adt_constructor : token_type -> token_type list -> 
  ccsl_input_types -> unit

    (* doing the general adt constructor, name, accessor list, 
     * domain type, codomain type (must be `Carrier)
     *)
val do_var_adt_constructor : token_type -> token_type list -> 
  ccsl_input_types -> ccsl_input_types -> unit

    (* give `Self back, if we are parsing a class *)
val self_action : location_type -> ccsl_input_types

    (* give `Carrier back, if we are parsing an adt *)
val carrier_action : location_type -> ccsl_input_types

    (* Arguments: 
     *   ADT_token          contains the ADT name
     *   ADT_arguments      contains the instanciation (ADT arguments)
     *   id_token           contains the member name
     *
     * gives the internal representation of the (suitably instantiated) 
     * constructor of the ADT/Class/Sig
     * action for method selection
     * within hol_formula
     *)
val do_method_selection : ccsl_expressions ->
  token_type option * ccsl_argument_type list option * token_type -> 
      ccsl_formulas

    (* action for qualified id's in term position *)
val do_qualified_term_id : 
  token_type option * ccsl_argument_type list option * token_type -> 
      ccsl_expressions

    (* action for case patterns *)
val do_case_pattern : token_type * token_type list ->
      ccsl_expressions ->
      ccsl_member_type container * ccsl_identifier_record_type list *
      ccsl_expressions

    (* actions for and and or *)

val do_and : ccsl_formulas -> ccsl_formulas -> ccsl_formulas

val do_or : ccsl_formulas -> ccsl_formulas -> ccsl_formulas



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** End: ***)



