(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Monday 8 October 01 17:58:00 tews@ithif51>
 *
 * generic class for all ccsl class and adt theories
 *
 * $Id: theory_class.ml,v 1.12 2002/01/23 16:00:19 tews Exp $
 *
 *)

(***********************************************************************
 ***********************************************************************
 *
 * The inheritance tree looks like this:
 * 
 * 				top_theory_body
 * 				       |
 * 				       |
 * 				      \/
 * 				ccsl_pre_theory_body
 * 			       /          	 \
 * 			      / 		  \
 * 			     / 			   \
 * 	      ccsl_adt_theory                  ccsl_pre_virtual_class_theory
 * 		    |              	       	   |
 * 		    |                              |
 * 		    |      			   |
 * 	     all adt theories		     ccsl_pre_class_theory
 *                                           (all class theories)
 * 
 *
 * some of those classes are currently empty (ie. do not add any features)
 * but who knows, whats happening ...
 *
 *)

open Util
open Top_names
open Top_variant_types
open Top_classtypes
open Types_util
open Top_classes
open Name_space
open Names
open Classtypes;;


class virtual ['class_type, 'member_type ] ccsl_pre_theory_body_class 
  cl 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_theory_body_type
  = object (self : 'self)
    inherit ['class_type, 'member_type] top_pre_theory_body_class

    val self1 = 
		 { id_token = { token_name = (name_of_self^"1");
				loc = None };
		   (* the other fields are to ignored *)
		   id_type = Self;
		   id_origin = CCSL_TypeParameter;
		   id_parameters = [];
		   id_variance = Unset;
		 }

    val self2 = 
		 { id_token = { token_name = (name_of_self^"2");
				loc = None};
		   (* the other fields are to ignored *)
		   id_type = Self;
		   id_origin = CCSL_TypeParameter;
		   id_parameters = [];
		   id_variance = Unset;
		 }

    method self1_parameter = TypeParameter self1

    method self2_parameter = TypeParameter self2

    val self_argument = 
      (TypeArgument Self : ('class_type, 'member_type) top_pre_argument_type)

    method self1_argument = TypeArgument(BoundTypeVariable self1)

    method self2_argument = TypeArgument(BoundTypeVariable self2)

    val orig_parameters = cl#get_parameters
			    
    val orig_arguments = arguments_from_parameters(cl#get_parameters)

    method simple_parameters = self_parameter :: orig_parameters

    method simple_arguments = self_argument :: orig_arguments 

    method double_self_parameters = 
      self#self1_parameter :: self#self2_parameter :: orig_parameters

    method double_self_arguments =
      self#self1_argument :: self#self2_argument :: orig_arguments


    method private make_double_parameter_ids = 
      let change_id id s =
	{id with id_token = { token_name = (id.id_token.token_name ^ s);
			      loc = None}}
      in
	List.map 
	  (function
	     | TypeParameter id -> (change_id id "1", change_id id "2")
	  )
	  orig_parameters

    val mutable store_double_ids =
        (None : 
	   (('class_type, 'member_type) top_pre_identifier_record_type *
	    ('class_type, 'member_type) top_pre_identifier_record_type )
	   list option)

    method double_parameter_ids = match store_double_ids with
      | Some l -> l
      | None -> (let l = self#make_double_parameter_ids
		 in
		   store_double_ids <- Some l;
		   l)

    method double_parameters = 
	List.split
	  (List.map 
	     (fun (id1,id2) -> (TypeParameter id1, TypeParameter id2))
	     self#double_parameter_ids)

    method double_arguments = 
	List.split
	  (List.map 
	     (fun (id1,id2) -> 
		(TypeArgument (BoundTypeVariable id1), 
		 TypeArgument (BoundTypeVariable id2)))
	     self#double_parameter_ids)

    val ns = Name_space.create eq_types

       (* reserve all method/constructor/ancestor names *)
    initializer 
      begin
        reserve ns 
          (List.map (fun m -> m#get_name) cl#get_all_members);
        reserve ns
          (List.map 
             (function
                | Resolved_renaming (ifa,_,_,_) -> 
                    super_label ifa
                | Unresolved_renaming _      (* no other stuff *)
                | Resolved_ancestor _ 
                | Unresolved_ancestor _ -> 
                    (assert(false); raise Internal_error)
             )
             cl#get_ancestors)

      end

  end


class virtual ['class_type, 'member_type ] 
  ccsl_pre_class_theory_class 
  (cl : 'class_type)
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_virtual_class_theory_type
  = object (self : 'self)
    inherit ['class_type, 'member_type] ccsl_pre_theory_body_class cl eq_types


	  (* the same is already hardcoded in assert_coalgebra_type 
	   * 
	   *)
    val coalgebra_term = (Term(name_of_coalgebra,Always,[]) : 
			  ('class_type, 'member_type) top_pre_expressions)

    val coalgebra1_term = (Term(name_of_coalgebra1,Always,[]) : 
			  ('class_type, 'member_type) top_pre_expressions)

    val coalgebra2_term = (Term(name_of_coalgebra2,Always,[]) : 
			  ('class_type, 'member_type) top_pre_expressions)

    val algebra_term = (Term(name_of_algebra,Always,[]) : 
			  ('class_type, 'member_type) top_pre_expressions)
	

    (*******************************************************************
     *
     * class specific things to be computed only once
     *)

					(* old version below *)
    method get_method_functor_type = 
      Record(
	(List.map
	   (function
	      | Resolved_renaming (_,args,_,ianc) -> 
		  (super_label ianc,
		   IFace(ianc, Always, self_argument :: args))
	      | Unresolved_renaming _ 
		  (* no java stuff *)
	      | Resolved_ancestor _ 
	      | Unresolved_ancestor _ -> 
		  (assert(false); raise Internal_error)
	   )
	   cl#get_ancestors)
	@
	(List.map
	   (fun m ->
	      (method_label m,
	       m#get_full_type
	      ))
	   cl#get_sig_actions))

       (* because PVS does not have primitive Coproducts, we formalize
	* the constructors as a record of functions
	*)
    method get_constructor_functor_type =
      Record(List.map
	       (fun m -> (if m#get_sort = Const_Constructor then
			    m#get_name, m#get_codomain
			  else
			    m#get_name, Function(m#get_domain, 
						 m#get_codomain)))
	       cl#get_constructors)


       (* the coalgebra type *)
    method coalgebra_type =
      IFace( cl, Always,self#simple_arguments)

    method coalgebra1_type = 
      IFace( cl, Always, self#self1_argument :: orig_arguments)

    method coalgebra2_type = 
      IFace( cl, Always, self#self2_argument :: orig_arguments)

    method get_member_fun (mem : 'member_type) =
      let top_class = 
	match mem#hosting_class#get_heir with
	  | None -> mem#hosting_class
	  | Some ocl -> ocl
      in let in_this_class = top_class#get_name = cl#get_name
      in let needs_coalgebra = 
	  mem#is_action || (mem#get_sort = Defined_Method) ||
	  (mem#get_sort = Class_Coreduce) 
      in
	match needs_coalgebra, in_this_class with
	  | true,true   -> coalgebra_term
	  | true,false  -> top_class#get_model_coalgebra
	  | false,true  -> algebra_term
	  | false,false -> top_class#get_model_algebra

    method get_iface_fun (typ : ('class_type, 'member_type) top_pre_types) =
      match typ with
	| Class(ocl,args) -> 
	    if ocl#get_name = cl#get_name 
					(* Change for -final switch *)
	    then coalgebra_term
	    else ocl#get_model_coalgebra
	| Self -> coalgebra_term
	| _ -> (assert(false); raise Internal_error)


    method assert_coalgebra_hypothesis =
      Formula(
	Application(
	  Expression(ConstantPredicate(name_of_assert cl)),
	  Term(name_of_coalgebra,Always,[])))

    method creation_algebra_hypothesis =
      Formula(
	Application(
	  Application(
	    Expression(ConstantPredicate(name_of_creation cl)),
	    Term(name_of_coalgebra,Always,[])),
	  Term(name_of_algebra,Always,[])))

    method assert_coalgebra_type = 
      Predtype(Formula(Term(name_of_assert cl,
			    Always,self#simple_arguments)))

    method assert_coalgebra1_type = 
      Predtype(Formula(Term(name_of_assert cl,
			    Always,self#self1_argument :: orig_arguments)))

    method assert_coalgebra2_type = 
      Predtype(Formula(Term(name_of_assert cl,
			    Always,self#self2_argument :: orig_arguments)))

      (* This is a bit strange! Since we model the algebra functor
       * as Record of functions (instead of a sum of the domains) the 
       * type of the algebra is then just this record 
       * (instead of a function going to Self)
       *)
    method algebra_type =
      TypeConstant(name_of_constructor_functor cl,Always,self#simple_arguments)

    method assert_algebra_type = 
      Predtype(Formula(
		 Application(Term(name_of_creation cl,
				  Always,self#simple_arguments),
			     Term(name_of_coalgebra,Always,[]))))

      (* the coalgebra declaration *)
    method coalgebra_decl = 
      	Vardecl(name_of_coalgebra, self#coalgebra_type)

    method coalgebra1_decl = 
      	Vardecl(name_of_coalgebra1, self#coalgebra1_type)

    method coalgebra2_decl = 
      Vardecl(name_of_coalgebra2, self#coalgebra2_type)

    method assert_coalgebra_decl =
      Vardecl(name_of_coalgebra, self#assert_coalgebra_type)

    method assert_coalgebra1_decl =
      Vardecl(name_of_coalgebra1, self#assert_coalgebra1_type)

    method assert_coalgebra2_decl =
      Vardecl(name_of_coalgebra2, self#assert_coalgebra2_type)

      (* the algebra declaration *)
    method algebra_decl = 
      	Vardecl(name_of_algebra, self#algebra_type)

  end

(*******************************************************************
 *******************************************************************
 *
 * special class only for the method lifting related classes
 *
 *)
class ccsl_method_inv_helper_class (cl: ccsl_iface_type)
  : ccsl_method_inv_helper_type
  = object (self: 'self)

    val pre_list_adt =             
      (match Symbol.find  "list" with 
               | AdtSymb( adt ) -> adt
               | _ -> (assert(false); raise Internal_error)
      )

(* for method invariance theories only: *)
      
    val mutable name_of_list = "" (* needs to be set!! *)

    val mutable name_of_step = ""

    val mutable name_of_path1 = ""
    
    val mutable name_of_path2 = ""

    method private list_adt = pre_list_adt
	      
    method private method_enum_type =
      TypeConstant(name_of_method_enum_type cl, Always, [])

    method private list_type =
      Adt(pre_list_adt, Always, [TypeArgument(self#method_enum_type)])

    method private list_term =
      ( Term(name_of_list, Always, []) : ccsl_expressions)

    method private method_term =
      ( Term(name_of_method_id, Always, []) : 
	ccsl_expressions)

    method private list_decl =
      Vardecl(name_of_list,self#list_type)
     
    method private method_decl =
      Vardecl(name_of_method_id, self#method_enum_type)

  end

(*******************************************************************
 *******************************************************************
 *
 * Adt's
 *
 *)

class virtual ['class_type, 'member_type ] ccsl_pre_adt_theory_class 
  adt
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_virtual_adt_theory_type
  = object
    inherit ['class_type, 'member_type] ccsl_pre_theory_body_class adt eq_types

    method this_adt_type = Adt(adt, Isabelle_only, orig_arguments)
  end


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** End: ***)

