(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * * Created 1.10.97 by Hendrik
 *
 * Time-stamp: <Monday 8 October 01 17:58:00 tews@ithif51>
 *
 * Utility functions for variant types
 *
 * $Id: top_variant_types_util.ml,v 1.20 2002/01/24 14:44:03 tews Exp $
 *
 *)


open Util
open Top_variant_types
open Type_variable
;;


(*******************************************************************
 *******************************************************************
 *
 * Resolution utility
 * extended for ancestor containers
 * it is not clear to me, if delivering the class is really the 
 * intention of this function, but for the moment ...
 *)


let resolution_of = function
  | Resolved r -> r
  | Unresolved _ ->
      assert false;
      raise Internal_error;;


let ancestor_resolution_of = function
  | Resolved_ancestor r -> r
  | Resolved_renaming (_, _, _, icl) -> icl
  | Unresolved_ancestor _ ->
      assert false;
      raise Internal_error
  | Unresolved_renaming _ ->
	 assert false;
	 raise Internal_error;;



(***********************************************************************
 ***********************************************************************
 *
 * dump section
 *
 *)

let string_of_loc = function
  | None -> "(no location known)"
  | Some loc -> 
      loc.file_name ^ ", line " ^
      (string_of_int loc.start_line) ^ "(Char " ^
      (string_of_int loc.start_char) ^ ") - line " ^
      (string_of_int loc.end_line) ^ "(Char " ^
      (string_of_int loc.end_char) ^ ")" 

let string_of_token token =
  token.token_name ^ (string_of_loc token.loc)


let dump_token = string_of_token;;


let dump_origin = function
  | CCSL_TypeParameter -> "CCSL Type Param"
  | CCSL_Var -> "CCSL Var"
  | CCSL_TypeAbbrev -> "CCSL Type Abbrev"
  | CCSL_GroundType _ -> "CCSL Ground Type"
  | CCSL_Output     -> "CCSL back end"


(* function to map a visibility to string *)
let string_of_visibility visibility (* : top_visibility -> string *) =
  match visibility with
    | NoVisibility  -> "None"
    | Public  -> "public"
    | Private -> "private"


let dump_argument_flag = function
  | Always -> "Always"
  | Isabelle_only -> "IsaOnly"
  | Never -> "Never"

(* function for dumping an 'a list, for debugging purposes
   input: dump_function : a -> string
   separator : string
   list : 'a list)
   output: string
*)
let rec dump_list : ('a -> string) -> string -> 'a list -> string
  = fun f sep l ->
    match l with
      | [] ->
	  ""
      | hd::[] ->
	  (f hd)
      | hd::tl ->
	  (f hd) ^ sep ^ (dump_list f sep tl);;

(* function for dumping a 'a container 
 * input : dump_function : 'a -> string
 *)

let dump_container dump_function = function
  | Unresolved tok -> "{U: " ^ (dump_token tok) ^ "}"
  | Resolved x -> dump_function x


(* functions for dumping a token_type list, for debugging purposes
   input: token_type list
   output: string
*)
let rec dump_token_type_list token_types =
  (* the explicit type ": token_type list -> string"
   * should only be added for debugging purposes
   *)
  dump_list dump_token "," token_types;;


(* functions for stringyfying a top_type
   input: top_type
   output: string
*)
let rec string_of_top_type typ =
  dump_top_type typ;


(* functions for dumping a top_type, for debugging purposes
   input: top_type
   output: string
*)
and dump_top_type = function
  | Groundtype (id, args) -> "Groundtype " ^ id.id_token.token_name ^
      (match args with
	 | [] -> ""
	 | _ ->
	     "{ " ^ (dump_list dump_argument " , " args) ^
	     "}") 
  | TypeConstant (str, flag, arg_list) ->
      str ^ (dump_argument_flag flag) ^
      "(" ^
      (dump_list dump_argument " , " arg_list) ^
      ")"
  | BoundTypeVariable (id) ->
      dump_var_record id
  | FreeTypeVariable tv ->
      "`a" ^ (string_of_tv tv)
  | Bool -> 
      "bool"
  | Self ->
      "Self"
  | Carrier  ->
      "Carrier"
  | Function (t1, t2) ->
      (dump_top_type t1) ^
      " -> " ^
      (dump_top_type t2)
  | SmartFunction(tl, t) ->
      "[" ^
      (dump_list dump_top_type ", " tl) ^ 
      "](" ^
      (dump_top_type t) ^
      ")"
  | Product (t_list) ->
      "(" ^
      (dump_list dump_top_type " * " t_list) ^
      ")"
  | Record (t_record_list) ->
      "[" ^
      (dump_list dump_type_record " , " t_record_list) ^
      "]"
  | Array (_, t, size) ->
      (dump_top_type t) ^
      "[" ^
      (string_of_int size) ^
      "]"
  | Class (top_iface, arg_list) ->
	 "Class " ^
	 (dump_top_iface top_iface) ^
	 (match arg_list with
	    | [] ->
		(* For Java I don't like these braces, etc. Joachim *)
		""
	    | _ ->
		"{\n" ^
		"  " ^
		(dump_list dump_argument " , " arg_list) ^
		"}\n") 

  | IFace (top_iface, flag, arg_list) ->
      "Functor " ^ (dump_argument_flag flag) ^
      (dump_top_iface top_iface) ^
      "{\n" ^
      "  " ^
      (dump_list dump_argument " , " arg_list) ^
      "}\n"  
  | Adt (adt_iface, flag, arg_list) ->
	 "Adt " ^
	 (dump_top_iface adt_iface) ^
	 "{\n" ^
	 "  " ^
	 (dump_list dump_argument " , " arg_list) ^
	 "}\n"

  | Predtype (f) ->
      "Predtype " ^
      "(" ^
      (dump_top_formula f) ^
      ")"


and dump_inst_iface = function
  | NoIface -> ""
  | CurrentIface -> "Self::"
  | InstIface(iface,arg_list) ->
      iface#get_name ^ "[" ^
      (dump_list dump_argument " , " arg_list) ^
      "]::"

and dump_parameter = function
  | TypeParameter id -> dump_var_record id


and dump_argument = function
  | TypeArgument typ ->
      dump_top_type typ


and dump_type_record (str, t) =
  "(" ^
  str ^
  " : " ^
  (dump_top_type t) ^
  ")";

  
and dump_top_iface x =
  x#get_name;


and dump_any_iface x =
  x#dump_iface;


and dump_top_formula = function
  | FormLoc(f,loc) -> dump_top_formula f
  | True ->
      "True"
  | False ->
      "False"
  | Not (f) ->
      "Not " ^
      "(" ^
      (dump_top_formula f) ^
      ")"
  | And (f_list) ->
      "(" ^
      (dump_list dump_top_formula " And " f_list) ^
      ")"
  | Or (f_list) ->
      "(" ^
      (dump_list dump_top_formula " Or " f_list) ^
      ")"
  | Implies (f1, f2) ->
      (dump_top_formula f1) ^
      " => " ^
      (dump_top_formula f2)
  | Iff (f1, f2) ->
      (dump_top_formula f1) ^
      " <=> " ^
      (dump_top_formula f2)
  | Equal (e1, e2) ->
      (dump_top_expression e1) ^
      " == " ^
      (dump_top_expression e2)
(*   | LessOrEqual (e1, e2) ->
 * 	 (dump_top_expression e1) ^
 * 	 " == " ^
 * 	 (dump_top_expression e2)
 *)
  | Forall (t_record_list, f) ->
      "Forall " ^
      (dump_list dump_type_record " , " t_record_list) ^
      "(" ^
      (dump_top_formula f) ^
      ")"
  | Exists (t_record_list, f) ->
      "Exists " ^
      (dump_list dump_type_record " , " t_record_list) ^
      "(" ^
      (dump_top_formula f) ^
      ")"
  | ConstantPredicate (str) ->
      str
  | Formula (e) ->
      "Formula( " ^ (dump_top_expression e) ^")"
  | Bisim (typ, e1, e2) ->
      "Bisim " ^
      (dump_top_type typ) ^
      "(" ^
      (dump_top_expression e1) ^ ", " ^
      (dump_top_expression e2) ^
      ")"

  | MetaImplies (f1, f2) ->
      (dump_top_formula f1) ^
      " ==> " ^
      (dump_top_formula f2)
  | MetaForall (t_record_list, f) ->
      "MetaForall " ^
      (dump_list dump_type_record " , " t_record_list) ^
      "(" ^
      (dump_top_formula f) ^
      ")"
  | Obseq (typ, ex1, ex2) ->
      (dump_top_expression ex1) ^
      "~"^
      (match typ with
	   None -> ""
	 | Some (n,t) -> (n ^ (dump_top_type t))
      ) ^
      (dump_top_expression ex2)

and dump_top_basic_expression = function
  | Member(instiface, member) ->
      (dump_inst_iface instiface) ^ 
      (dump_container (fun m -> m#get_name) member) 
  | TermVar (idcontainer) ->
      (dump_container (fun id -> dump_var_record id) idcontainer)
      


and dump_top_expression = function
  | ExprLoc(e,l) -> dump_top_expression e
  | BasicExpr bexp -> dump_top_basic_expression bexp
  | Term (str, flag, arg_list) ->
      "Term " ^ (dump_argument_flag flag) ^
      str ^
      "(" ^
      (dump_list dump_argument " , " arg_list) ^
      ")"
  | TypedTerm(ex, typ) -> 
      "(" ^ (dump_top_expression ex) ^ " : " ^
      (dump_top_type typ) ^ ")"
  | TypeAnnotation(ex, typ) -> 
      "(" ^ (dump_top_expression ex) ^ " :: " ^
      (dump_top_type typ) ^ ")"
  | QualifiedTerm (str1, flag, arg_list, str2) ->
      "QualifiedTerm " ^ (dump_argument_flag flag) ^
      str1 ^
      "(" ^
      (dump_list dump_argument " , " arg_list) ^
      ")" ^
      " " ^
      str2

  | MethodSelection(ex, instiface, membercontainer) -> 
      "(" ^ (dump_top_expression ex) ^ ")." ^
      (dump_inst_iface instiface) ^ 
      (dump_container (fun m -> m#get_name) membercontainer)

  | Tuple (e_list) ->
      "Tuple " ^
      "(" ^
      (dump_list dump_top_expression " , " e_list) ^
      ")"
  | Projection (i,n) ->
      "Projection(" ^
      (string_of_int i) ^ "," ^
      (string_of_int n) ^ ")"

  | RecordTuple (e_record_list) ->
      "RecordTuple " ^
      "[" ^
      (dump_list dump_e_record " , " e_record_list) ^
      "]"
  | RecordSelection (str, e) ->
      "RecordSelection " ^
      str ^
      "[" ^
      (dump_top_expression e) ^
      "]"
  | RecordUpdate (e, e_record_list) ->
      "RecordUpdate " ^
      (dump_top_expression e) ^
      "[" ^
      (dump_list dump_e_record " , " e_record_list) ^
      "]"
  | List (e_list) ->
      "List " ^
      "(" ^
      (dump_list dump_top_expression " , " e_list) ^
      ")"
  | Abstraction (t_record_list, e) ->
      "Abstraction " ^
      "[" ^
      (dump_list dump_type_record " , " t_record_list) ^
      "]" ^
      (dump_top_expression e)
  | SmartAbstraction (t_record_list, e) ->
      "SmartAbstraction " ^
      "[" ^
      (dump_list dump_type_record " , " t_record_list) ^
      "]" ^
      (dump_top_expression e)
  | Application (e1, e2) ->
      "Application " ^
      "(" ^
      (dump_top_expression e1) ^
      "." ^
      (dump_top_expression e2) ^
      ")"
  | InfixApplication (exp1, instiface, memcontainer, exp2) ->
      "Infix " ^
      "(" ^
      (dump_top_expression exp1) ^
      (dump_inst_iface instiface) ^
      (dump_container (fun m -> m#get_name) memcontainer) ^
      (dump_top_expression exp2) ^
      ")"
  | SmartApplication (e1, tl) ->
      "Application " ^
      "(" ^
      (dump_top_expression e1) ^
      "[" ^
      (dump_list dump_top_expression " , " tl) ^
      "]"
  | FunUpdate (e, e_e_list) ->
      "FunUpdate " ^
      (dump_top_expression e) ^
      "(" ^
      (dump_list dump_e_e " , " e_e_list) ^
      ")"
  | Let (str_t_e_list, e) ->
      "Let " ^
      (dump_list dump_str_t_e " , " str_t_e_list) ^
      "=" ^
      "(" ^
      (dump_top_expression e) ^
      ")"
  | If (f_e_list, e) ->
      "If " ^
      (dump_list dump_f_e " , " f_e_list) ^
      "=" ^
      "(" ^
      (dump_top_expression e) ^
      ")"
  | Case (e, e_e_list) ->
      "Case " ^
      (dump_top_expression e) ^
      "(" ^
      (dump_list dump_e_e " , " e_e_list) ^
      ")"
  | CCSL_Case (e, m_v_e_list) ->
      "Case " ^
      (dump_top_expression e) ^
      "(" ^
      (dump_list 
	 (fun (member,varlist,ex) -> 
	    (dump_container (fun m -> m#get_name) member) ^
	    (if varlist <> [] 
	     then
	       "(" ^ 
	       (dump_list (fun id -> id.id_token.token_name) ", " varlist) ^
	       ")"
	     else
	       "") ^
	    " : " ^
	    dump_top_expression ex
	 )
	 " , " m_v_e_list) ^
      ")"
(*   | Reduce (iface, flag, arg_list ) ->
 * 	 "Reduce[" ^
 * 	 (dump_any_iface iface) ^ (dump_argument_flag flag) ^
 * 	 "[" ^
 * 	 (dump_list dump_argument " , " arg_list) ^
 * 	 "]]" 
 *)
  | Box(typ,pred,tlist) ->
      "Always " ^ dump_top_expression pred ^ " for " ^ 
      (dump_top_type typ) ^
      ".{" ^ (dump_list dump_token " , " tlist) ^
      "}"
  | Diamond(typ,pred,tlist) ->
      "Eventually " ^ dump_top_expression pred ^ " for " ^ 
      (dump_top_type typ) ^
      ".{" ^ (dump_list dump_token " , " tlist) ^
      "}"
  | Every (typ, f_list) ->
      "Every " ^
      "(" ^
      (dump_top_type typ) ^
      ", " ^
      (dump_list dump_top_expression "\n" f_list) ^
      ")" 
  | RelEvery (typ, f_list) ->
      "RelEvery " ^
      "(" ^
      (dump_top_type typ) ^
      ", " ^
      (dump_list dump_top_expression "\n" f_list) ^
      ")" 
  | Map (t, e_list) ->
      "Map " ^ (dump_top_type t) ^ "(" ^
      (dump_list dump_top_expression ", " e_list)
  | Expression (f) ->
      "Expression " ^
      "(" ^
      (dump_top_formula f) ^
      ")"
  | Comment_str (str, e) ->
      "Comment_str" ^
      str ^
      "(" ^
      (dump_top_expression e) ^
      ")"
  | Comment_expr (e1, t, e2) ->
      "Comment_expr " ^
      "(" ^
      (dump_top_expression e1) ^
      " " ^
      (dump_top_type t) ^
      " " ^
      (dump_top_expression e2) ^
      ")"
  | Comprehension (str, t, f) ->
      "Comprehention " ^
      str ^
      "(" ^
      (dump_top_type t) ^
      " " ^
      (dump_top_formula f) ^
      ")"


and dump_e_record = function
  | (str, t) ->
      "(" ^
      str ^
      " = " ^
      (dump_top_expression t) ^
      ")";


and dump_e_e = function
  | (e1, e2) ->
      "(" ^
      (dump_top_expression e1) ^
      " , " ^
      (dump_top_expression e2) ^
      ")";


and dump_str_t_e = function
  | (id_rec, t, e) ->
      (dump_var_record id_rec) ^
      "(" ^
      (match t with
	| None -> 
	    ""
	| Some s ->
	    (dump_top_type s)) ^
      " : " ^
      (dump_top_expression e) ^
      ")";


and dump_f_e = function
  | (f, e) ->
      "(" ^
      (dump_top_formula f) ^
      " , " ^
      (dump_top_expression e) ^
      ")"

(***********************************************************************
 *
 * dump a variable record
 *)

and dump_var_record id =
  "Id( " ^ id.id_token.token_name ^ ", " ^
  (dump_origin id.id_origin) ^ 
  (match id.id_origin with
      (* CCSL section: id_position is irrelevant *)
			(* CCSL Type Parameter: id_type is irrelevant *)
     | CCSL_TypeParameter -> ""
			   (* logical Variable in CCSL Assertions *)
     | CCSL_Var -> ", " ^ dump_top_type id.id_type
			(* ccsl type abbreviation *)
     | CCSL_TypeAbbrev -> ", " ^ dump_top_type id.id_type
			(* ccsl ground type *)
     | CCSL_GroundType _ -> 
	 "[" ^ (dump_list dump_parameter ", " id.id_parameters) ^ "]"
			(* ccsl ground term *)
     | CCSL_Output -> ""

  ) ^ ")"
;;


(***********************************************************************
 ***********************************************************************
 *
 * dump a top_symbol
 *
 *)


let dump_symbol (* : top_symbol_type -> string *) = function
  | CCSL_GroundTypeSymb id -> "Groundtype " ^ (dump_var_record id)
  | CCSL_TypeAbbrevSymb id -> "Abbrev " ^ (dump_var_record id)
  | TypevarSymb id -> "Typevar " ^ (dump_var_record id)
  | VarSymb id -> "Var " ^ (dump_var_record id)
  | MemberSymb m -> "Member " ^ m#dump_member
					   (* a ccsl class *)
  | ClassSymb cl -> "Class " ^ cl#dump_iface
					(* an instanciated ccsl class *)
  | InstClassSymb cl -> "IClass " ^ cl#dump_iface
					(* a ccsl adt *)
  | AdtSymb adt -> "Adt " ^ adt#dump_iface
					(* an instanciated ccsl adt *)
  | InstAdtSymb adt -> "IAdt " ^ adt#dump_iface
					(* a ground signature *)
  | SigSymb si -> "Sig " ^ si#dump_iface


(*
 * same as above, but do not dump entries inside the symbols
 *)

let dump_symbol_brief (* : top_symbol_type -> string *) = function
  | CCSL_GroundTypeSymb id -> "Groundtype " ^ id.id_token.token_name
  | CCSL_TypeAbbrevSymb id -> "Abbrev " ^ id.id_token.token_name
  | TypevarSymb id -> "Typevar " ^ id.id_token.token_name
  | VarSymb id -> "Var " ^ id.id_token.token_name
  | MemberSymb m -> "Member " ^ m#get_name
					   (* a ccsl class *)
  | ClassSymb cl -> "Class " ^ cl#get_name
					(* an instanciated ccsl class *)
  | InstClassSymb cl -> "IClass " ^ cl#get_name
					(* a ccsl adt *)
  | AdtSymb adt -> "Adt " ^ adt#get_name
					(* an instanciated ccsl adt *)
  | InstAdtSymb adt -> "IAdt " ^ adt#get_name
					(* a ground signature *)
  | SigSymb si -> "Sig " ^ si#get_name



(* FIXME Why are there 2 functions dump_ancestor? *)
(* Because this module is a big mess ;-) *)
(* Even more puzzling: why is nobody using either of these two ? *)
let dump_ancestor (* : top_ancestor_type -> string *) = function
  | Unresolved_ancestor token_list ->
      "Unresolved " ^ 
      (List.fold_left
	 (fun res token ->
	    res ^ "." ^ dump_token token)
	 ""
	 token_list)
  | Resolved_ancestor ifa ->
      "Resolved (" ^ ifa#dump_iface ^ ")"
  | Unresolved_renaming (ifa, al, renaming ) ->
      "Unresolved (" ^ ifa#dump_iface ^ "[" ^ 
      (dump_list dump_argument ", " al) ^
      " renaming ...]"
  | Resolved_renaming (_, al, member_list, iifa) ->
      "Resolved (" ^ iifa#dump_iface ^ "[" ^ 
      (dump_list dump_argument ", " al) ^
      " renaming " ^
      (dump_list (fun m -> m#dump_member) " | " member_list);;


(***********************************************************************
 ***********************************************************************
 *
 * dump abstract syntax tree
 *
 *)


let dump_top_ast ast : string = 
  let dump = function 
    | CCSL_TypeAbbrev_dec id -> 
	  id.id_token.token_name ^ " = " ^ (dump_top_type id.id_type)
    | CCSL_class_dec cl ->  cl#dump_iface
    | CCSL_adt_dec adt  -> adt#dump_iface
    | CCSL_sig_dec si -> si#dump_iface
  in
  List.fold_left 
    (fun s k -> s ^ "\n" ^ (dump k) ^ "\n" ) 
    ""
    ast;;


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** End: ***)
