(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 16.03.00 by Jan
 *
 * Time-stamp: <Wednesday 15 May 02 19:27:35 tews@ithif51>
 *
 * Coalgebra morphism theories
 *
 * $Id: morphism_theory.ml,v 1.10 2002/05/22 13:42:42 tews Exp $
 *
 *)

open Util
open Global
open Top_variant_types
open Name_space
open Top_names
open Names
open Classtypes
open Types_util
open Theory_class
open Logic_util
open Morphism;;

(***********************************************************************
 ***********************************************************************
 *
 * The definitions of coalgebra morphisms
 *
 *)


class ['class_type, 'member_type] ccsl_pre_morphism_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
       [ name_of_coalgebra1;
	 name_of_coalgebra2
       ]

    method get_name = ccsl_morphism_theory_name cl

    method get_parameters = self#double_self_parameters 

    initializer top_theory#override_file_name (ccsl_class_file_name cl)

    method private imports =
      let do_anc anc args =
	(ccsl_morphism_theory_name anc,
	 self#self1_argument :: self#self2_argument :: args) in
      let anc_imports =
	List.fold_right
	(fun parent accu -> match parent with
	   | Resolved_renaming (anc,args,_,_,_) -> 
	       if (count_self_args args) = 0 
	       then (do_anc anc args) :: accu
	       else accu
					(* no other stuff *)
	   | Unresolved_renaming _
	     -> assert(false)
	)
	  cl#get_ancestors 
	  []
      in
      let component_imports =
	let self1_typ = BoundTypeVariable self1 in
	let self2_typ = BoundTypeVariable self2 in
	let left_domain_subst = [ (Self, (self1_typ,self1_typ)) ] in
	let right_domain_subst = [ (Self, (self2_typ, self2_typ)) ] in
	let codomain_pos_subst = [ (Self, (self1_typ, self2_typ)) ] in
	let codomain_neg_subst = [ (Self, (self2_typ, self1_typ)) ] 
	in
	  List.fold_right
	    (fun (v,comp,args) accu ->
	       let left_args = 
		 (substitute_variance_argument_list
		    left_domain_subst args comp#get_parameters)
	       in let right_args = 
		   (substitute_variance_argument_list
		      right_domain_subst args comp#get_parameters)	
	       in let res_pos_args = 
		   (substitute_variance_argument_list
		    codomain_pos_subst args comp#get_parameters)
	       in let res_neg_args = 
		   (substitute_variance_argument_list
		    codomain_neg_subst args comp#get_parameters)
	       in let rec do_v accu th_name = function
		 | Pos -> ((th_name, left_args @ res_pos_args) :: 
			     (th_name, right_args @ res_pos_args)
			   :: accu)
		 | Neg -> ((th_name, res_neg_args @ left_args) :: 
			     (th_name, res_neg_args @ right_args)
			   :: accu)
		 | Mixed -> (do_v (do_v accu th_name Pos)
			       th_name Neg)
		 | _ -> assert(false)
	       in
	         match comp#get_kind with
		 | Spec_class -> 
		     if comp#has_feature HasMapFeature
		     then
		       do_v accu (ccsl_map_theory_name comp) v
		     else
		       accu
		 | Spec_adt -> 
		     if comp#has_feature NeedsMapFeature
		     then 
		       do_v accu (ccsl_adt_map_theory_name comp) v
		     else accu
		 | Spec_sig -> accu
		 | Spec_Spec -> assert(false)
	    )
	    (List.flatten
	       (cl#get_components ::
		  (List.map (fun cl -> cl#get_components) 
		     cl#get_all_ancestors)))
	    []
      in
       	Import(
	  [((ccsl_interface_theory_name cl),
	    self#self1_argument :: orig_arguments);
	   ((ccsl_interface_theory_name cl),
	    self#self2_argument :: orig_arguments)
	  ] 
	  @
	  anc_imports
	  @
	  component_imports
	)

    method private morphism_decl class_morph method_filter = 
      let sub_ns = sub_space ns in
      let self1_type = BoundTypeVariable(self1) in
      let type_of_morphism = 	  
	Function(
	  self1_type,
	  BoundTypeVariable(self2)
      	)
      in
      let f = create_one_id sub_ns type_of_morphism in
      let x = create_one_id sub_ns self1_type in
      let type_of_pred =
	Function(
	  type_of_morphism,
	  Bool
	)
      in
      let method_list_list, const_ancestor_list = 
	List.fold_right 
	  (fun parent (accu_m, accu_anc) -> match parent with 
	     | Resolved_renaming(_, args,_,ianc,_) -> 
		 if (count_self_args args) = 0
		 then (accu_m, (ianc,args) :: accu_anc)
		 else ((List.filter method_filter ianc#get_all_actions)
		       :: accu_m, accu_anc)

					(* no other stuff here *)
	     | Unresolved_renaming _
	     | _ -> assert(false)
	  ) 
	  cl#get_ancestors
	  ([List.filter method_filter cl#get_actions], [])
      in
	Defn(class_morph cl,
	     [[Declared(name_of_coalgebra1, self#coalgebra1_type);
	       Declared(name_of_coalgebra2, self#coalgebra2_type)]],
	     type_of_pred,
	     Abstraction(
	       [f, type_of_morphism],
	       Expression
		 (coalgebra_morphism
		    (Term(f, Never, []))                    (* morphism *)
		    sub_ns				   (* name_space *)
		    (Term(name_of_coalgebra1,Never,[]))	   (* coalgebra1 *)
		    (Term(name_of_coalgebra2,Never,[]))	   (* coalgebra2 *)
		    const_ancestor_list			   (* ancestor_list *)
		    class_morph				   (* anc_morphism *)
		    (List.flatten method_list_list)
		    (Term(x, Never, []))		   (* self_var *)
		    [self#self1_argument;
		     self#self2_argument]	      (* self_type_argument *)
		    (BoundTypeVariable self1)		   (* self1_typ *)
		    (BoundTypeVariable self2)		   (* self2_typ *)
	     )
	   )
	  )
		     
    method private full_morphism_decl =
      self#morphism_decl name_of_full_morphism_pred 
	(fun m -> m#is_sig_action)

    method private public_morphism_decl =
      self#morphism_decl name_of_public_morphism_pred
	(fun m -> m#is_sig_action && (m#get_visibility = Public))

    method private struct_morphism =
      let sub_ns = sub_space ns in 
      let (decl1_list, decl2_list, c1_tuple, c2_tuple ) = 
	self#two_coalgebras_as_tuple sub_ns in
      let type_of_morphism = 	  
	Function(
	  BoundTypeVariable(self1),
	  BoundTypeVariable(self2)
      	)
      in
      let f = create_one_id sub_ns type_of_morphism 
      in
	Defn(name_of_full_morphism_struct_pred cl,
	     [decl1_list; decl2_list; [ Undeclared(f, type_of_morphism) ]],
	     Bool,
	     Application(
	       Application(
		 Term(name_of_full_morphism_pred cl, Always, []),
		 Tuple([c1_tuple; c2_tuple])
	       ),
	       Term(f,Always,[])
	     ))



    method make_body = 
      [
      	self#imports;
	self#coalgebra1_decl;
	self#coalgebra2_decl;
	self#full_morphism_decl;
	self#struct_morphism;
	self#public_morphism_decl
      ]

    method get_proofs = []

end

class ccsl_morphism_theory cl =
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_morphism_theory cl eq_ccsl_types


(***********************************************************************
 ***********************************************************************
 *
 * Rewrite Lemmas for morphisms
 *
 *)

class ['class_type, 'member_type] ccsl_pre_morphism_rewrite_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
       [ name_of_coalgebra1;
	 name_of_coalgebra2
       ]

    method get_name = ccsl_morphism_rewrite_theory_name cl

    method get_parameters = self#double_self_parameters 

    initializer top_theory#override_file_name (ccsl_class_file_name cl)

    method private imports =
      Import(
	[((ccsl_morphism_theory_name cl),
	  self#double_self_arguments);
	 ((ccsl_bibisim_theory_name cl),
	  self#double_self_arguments)
	]
      )

	  (* print the next lemma only, if the interface functor is 
	   * at most extended polynomial 
	   *)
    method private morph_bisim =
      let sub_ns = sub_space ns in
      let type_of_morphism = 	  
	Function(
	  BoundTypeVariable(self1),
	  BoundTypeVariable(self2)
      	)
      in
      let f = create_one_id sub_ns type_of_morphism in
      Lemma(
	name_of_morphism_bisim_lemma cl,
	Forall(
	  [f,type_of_morphism],
	  Iff(
	    Formula(
	      Application(
	    	Application(
		  Term(name_of_full_morphism_pred cl, 
		       Isabelle_only, 
		       self#double_self_arguments),
		  Tuple(
		    [coalgebra1_term;coalgebra2_term]
		  )
	    	),
	    	Term(f,Never,[])
	      )
	    ),
	    Formula(
	      Application(
		Application(
		  Term(name_of_private_bibisimulation cl,
		       Isabelle_only,
		       self#double_self_arguments),
		  Tuple(
		    [coalgebra1_term;coalgebra2_term]
		  )
		),
		Application(
		  Term(name_of_graph,Never,[]),
		  Term(f,Never,[])
		)
	      )
	    )
	  )
	)
      )

    method private do_method m =
      let sub_ns = sub_space ns in
      let self1_type = BoundTypeVariable(self1) in
      let self2_type = BoundTypeVariable(self2) in
      let type_of_morphism = 	  
	Function(
	  self1_type,
	  BoundTypeVariable(self2)
      	) in
      let f = create_one_id sub_ns type_of_morphism in
      let x = create_one_id sub_ns self1_type in
      	Lemma(name_of_method_morphism_lemma m,
	  Forall(
	    [f,type_of_morphism],
	    Implies(
	      Formula(
		Application(
		  Application(
		    Term(name_of_full_morphism_pred cl, 
			 Isabelle_only, 
			 self#double_self_arguments),
		    Tuple(
		      [coalgebra1_term;coalgebra2_term]
		    )
	    	  ),
	    	  Term(f,Never,[])
	      	)
	      ),
	      (coalgebra_morphism 
		 (Term(f,Never,[]))
		 sub_ns
		 coalgebra1_term
		 coalgebra2_term
		 []
		 (fun _ -> "")
		 [m]
		 (Term(x,Never,[]))
		 [self#self1_argument;
		  self#self2_argument]
		 self1_type
		 self2_type
	      )
	    )
	  )
	)       

    method make_body = 
      [
      	self#imports;
	self#coalgebra1_decl;
	self#coalgebra2_decl;
	self#morph_bisim
      ] @
      (List.map
	 self#do_method
	 cl#get_all_actions)

    method get_proofs = []

end


class ccsl_morphism_rewrite_theory cl =
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_morphism_rewrite_theory cl eq_ccsl_types


(***********************************************************************
 ***********************************************************************
 *
 * Definition of Finality
 *
 *)


class ['class_type, 'member_type] ccsl_pre_finality_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
       [ name_of_coalgebra1;
	 name_of_coalgebra2
       ]

    method get_name = ccsl_finality_theory_name cl

    method get_parameters = self#double_self_parameters 

    initializer top_theory#override_file_name (ccsl_class_file_name cl)

    method private imports =
      Import(
	[((ccsl_morphism_rewrite_theory_name cl),
	  self#double_self_arguments);
	 ((ccsl_basic_theory_name cl),
	  self#self1_argument :: orig_arguments);
	 ((ccsl_basic_theory_name cl),
	  self#self2_argument :: orig_arguments);
	]
      )

    method private finality_existence_decl =
      let sub_ns = sub_space ns in
      let type_of_morphism = 	  
	Function(
	  BoundTypeVariable(self1),
	  BoundTypeVariable(self2)
      	)
      in
      let f = create_one_id sub_ns type_of_morphism in
      	Defn( name_of_finality_existence cl,
	      [[Declared(name_of_coalgebra2,
			 self#assert_coalgebra2_type)]],
	      Bool,
	      Expression(
	       	Forall(
		  [name_of_coalgebra1,self#assert_coalgebra1_type],
		  Exists(
		    [f,type_of_morphism],
		    Formula(
		      Application(
		       	Application(
			  Term(name_of_full_morphism_pred cl,
			       Isabelle_only,
			       self#double_self_arguments),
			  Tuple([coalgebra1_term;
				 coalgebra2_term])
		       	),
		       	Term(f,Never,[])
		      )
		    )
		  )
	       	)
	      )
	    )
	  
    method private finality_uniqueness_decl = 
      let sub_ns = sub_space ns in
      let type_of_morphism = 	  
	Function(
	  BoundTypeVariable(self1),
	  BoundTypeVariable(self2)
      	)
      in
      let f = create_one_id sub_ns type_of_morphism in
      let g = create_one_id sub_ns type_of_morphism in
      let is_morph fn = 
	Formula(
	  Application(
	    Application(
	      Term(name_of_full_morphism_pred cl,
		   Isabelle_only,
		   self#double_self_arguments),
	      Tuple([coalgebra1_term;
		     coalgebra2_term])),
	    Term(fn,Never,[]))) 
      in
      	Defn( name_of_finality_uniqueness cl,
	      [[Declared(name_of_coalgebra2,
			 self#assert_coalgebra2_type)]],
	      Bool,
	      Expression(
	       	Forall(
		  [name_of_coalgebra1,
		   self#assert_coalgebra1_type],
		  Forall(
		    [f, type_of_morphism;
		     g, type_of_morphism],
		    Implies(
		      And([ (is_morph f); (is_morph g) ]),
		      Equal(
			Term(f,Never,[]),
			Term(g,Never,[])
		      ))))))
	
    method private finality_decl =
      Defn( name_of_finality cl,
	    [[Declared(name_of_coalgebra2, self#assert_coalgebra2_type)]],
	    Bool,
	    Expression(
	      And(
		[
		  Formula(
		    Application(
		      Term(name_of_assert cl, Always,
			   self#self2_argument :: orig_arguments),
		      coalgebra2_term
		    ));			    
		  Formula(
		    Application(
		      Term(name_of_finality_existence cl, Never, []),
		      coalgebra2_term
		    )
		  );
		  Formula(
		    Application(
		      Term(name_of_finality_uniqueness cl, Never, []),
		      coalgebra2_term
		    ))]
	      )))


    method private struct_coreduce_def = 
      let sub_ns = sub_space ns in
      let type_of_morphism = 	  
	Function(
	  BoundTypeVariable(self1),
	  BoundTypeVariable(self2)
      	)
      in let f = create_one_id sub_ns type_of_morphism 
      in
      Defn(
	name_of_class_struct_coreduce cl,
	[[Undeclared(name_of_coalgebra2,
		     Predtype(ConstantPredicate(name_of_finality cl)))];
	 [Undeclared(name_of_coalgebra1,self#assert_coalgebra1_type)]],
	type_of_morphism,
	Application(
	  Term(name_of_choose(), Always,[]),
	  Abstraction(
	    [f, type_of_morphism],
	    Application(
	      Application(
		Term(name_of_full_morphism_pred cl,
		     Isabelle_only,
		     self#double_self_arguments),
		Tuple([coalgebra1_term;
		       coalgebra2_term])),
	      Term(f,Never,[])))
	))		     
		     

    method private coreduce_def = 
      let sub_ns = sub_space ns in
      let type_of_morphism = 	  
	Function(
	  BoundTypeVariable(self1),
	  BoundTypeVariable(self2)
      	)
      in let f = create_one_id sub_ns type_of_morphism in
      let mt = create_id_with_preference sub_ns "m_tuple" "p" 
      in
      Defn(
	name_of_class_coreduce cl,
	[[Undeclared(name_of_coalgebra2,
		     Predtype(ConstantPredicate(name_of_finality cl)))];
	 [Undeclared(mt,
		     Predtype(Formula(
				Term(name_of_method_assert cl,Always,
				     self#self1_argument :: orig_arguments)))
		    )]],
	type_of_morphism,
	Application(
	  Application(
	    Term(name_of_class_struct_coreduce cl,Always,[]),
	    coalgebra2_term),
	  Application(
	    Term(name_of_struct_of cl, Always,[]),
	    Term(mt,Always,[]))
	))


    method private finality_bisim_eq_lemma =
      let sub_ns = sub_space ns in
      let self2_type = BoundTypeVariable(self2) in
      let (x1,x2) = match create_ids sub_ns [self2_type;self2_type] with
	  [a1,t1;a2,t2] -> (a1,a2)
	| _ -> assert false;
      in
      let x1_term = Term(x1,Never,[]) in
      let x2_term = Term(x2,Never,[]) in
	Lemma(
	  name_of_finality_bisim_eq_lemma cl,
	  Implies(
	    Formula(
	      Application(
	      	Term(name_of_finality cl, Never, []),
		coalgebra2_term
	      )
	    ),
	    Forall(
	      [x1,self2_type;
	       x2,self2_type],
	      Implies(
		Formula(
		  Application(
		    Application(
		      Term(name_of_private_bibisim cl,
			   Isabelle_only,
			   (self#self2_argument :: orig_arguments)
			  ),
		      coalgebra2_term
		    ),
		    Tuple([x1_term;x2_term])
		  )
		),
		Equal(x1_term,x2_term)
	      )
	    )
	  )
	)

    method make_body = 
      [
      	self#imports;
	self#coalgebra2_decl;
	self#finality_existence_decl;
	self#finality_uniqueness_decl;
	self#finality_decl;
	self#struct_coreduce_def;
	self#coreduce_def
      ]
      @ ( 
	if cl#has_feature HasGreatestBisimFeature 
	then
	  [self#finality_bisim_eq_lemma]
	else
	  []
      )

    method get_proofs = []

end

class ccsl_finality_theory cl =
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_finality_theory cl eq_ccsl_types



(***********************************************************************
 ***********************************************************************
 *
 * Definition of MAP: first the structure conversion
 *
 *)

class ['class_type, 'member_type] ccsl_pre_map_struct_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

(* leave type constraint out, because the next theory inherits from this one
 *   : ['class_type, 'member_type] ccsl_pre_class_theory_type 
 * 
 *)
  =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
       [ name_of_class_map_struct cl;
	 name_of_coalgebra
       ]

    method get_name = ccsl_map_struct_theory_name cl

    method get_parameters = 
      let left,right = self#double_parameters
      in
        self_parameter :: left@right

    initializer top_theory#override_file_name (ccsl_class_file_name cl)

    val mutable from_subst_field = None

    val mutable to_subst_field = None

    method private make_subst = 
      let left,right = self#double_parameters 
      in
	from_subst_field <- 
	  Some(make_substitution_param_param orig_parameters left);
	to_subst_field <- 
	  Some(make_substitution_param_param orig_parameters right)
	
    method private from_subst = match from_subst_field with
      | None -> self#make_subst; self#from_subst
      | Some s -> s

    method private to_subst = match to_subst_field with
      | None -> self#make_subst; self#to_subst
      | Some s -> s

    method private do_import =
      let left,right = self#double_arguments in
      let do_ancestor anc args =
	ccsl_map_struct_theory_name anc, 
	self_argument ::
	  ((substitute_arguments_types_only eq_types self#from_subst args)
	   @ (substitute_arguments_types_only eq_types self#to_subst args))
      in
      let component_imports =
	List.fold_right
	  (fun (v,comp,args) accu ->
	     let from_args = substitute_arguments_types_only 
			       eq_types self#from_subst args
	     in let to_args = substitute_arguments_types_only 
				eq_types self#to_subst args
	     in let rec do_v th_name accu = function
	       | Pos -> (th_name, from_args @ to_args) :: accu
	       | Neg -> (th_name, to_args @ from_args) :: accu
	       | Mixed -> do_v th_name (do_v th_name accu Pos) Neg
	       | _ -> assert(false)
	     in
	       match comp#get_kind with
		 | Spec_class -> do_v (ccsl_map_theory_name comp) accu v
		 | Spec_adt -> 
		     if comp#has_feature NeedsMapFeature
		     then do_v (ccsl_adt_map_theory_name comp) accu v
		     else accu
		 | Spec_sig -> accu
		 | Spec_Spec -> assert(false)
	    )
	    cl#get_components []
      in
	Import(
	  (ccsl_basic_theory_name cl, self_argument :: left) ::
	  (ccsl_basic_theory_name cl, self_argument :: right) ::
	  (List.map (function
		       | Resolved_renaming(oanc, args,_,_,_) -> 
			   do_ancestor oanc args
						(* no other stuff here *)
		       | _ -> assert(false)
		    ) 
	     cl#get_ancestors)
	  @
	  component_imports
	)

    method private coalgebra_decl =
      let (args_from,_) = self#double_arguments 
      in
	Vardecl(name_of_coalgebra, 
		IFace( cl, Always, self_argument :: args_from))


    val mutable fun_ids_var_field = None

    method private fun_ids_var = 
      match fun_ids_var_field with
	| Some l -> l
	| None ->
	    let (param_from,param_to) = self#double_parameters in
	    let cl_param_ids = 
	      List.map (function TypeParameter id -> id) 
		cl#get_parameters 
	    in
	    let fun_types = 
	      Util.map3 (fun t_from t_to o_id ->
			   (Function(t_from, t_to), 
			    o_id.id_variance))
		(types_from_parameters param_from)
		(types_from_parameters param_to)
		cl_param_ids
	    in
	    let pre_fun_ids_var = 
	      create_ids_with_variance ns fun_types
	    in
	    let fun_ids_var =		(* toggle direction for negatives *)
	      List.map
		(function
		   | (None, _) as x -> x
		   | (Some(neg_name, Function(dom,codom)), pos_id) ->
		       (Some(neg_name, Function(codom, dom)), pos_id)
		   | _ -> assert(false)
		) 
		pre_fun_ids_var
	    in
	      fun_ids_var_field <- Some fun_ids_var;
	      fun_ids_var


    method private fun_decls =
      List.map
	(fun (id,t) -> Vardecl(id,t))
	(variance_flatten self#fun_ids_var)
	

    method private class_map_struct =
  (* debugging support 
      let adt_name = cl#get_name in
  *)
      let sub_ns = sub_space ns in
      let (param_from,param_to) = self#double_parameters in
      let args_from = arguments_from_parameters param_from in
      let args_to = arguments_from_parameters param_to in
      let map_action = 
	var_terms_from_var_ids 
	  (types_from_parameters cl#get_parameters)
	  self#fun_ids_var
      in
      let fun_ids = variance_flatten self#fun_ids_var in
      let fun_exprs = List.map (fun (id,t) -> Term(id,Always,[])) fun_ids
      in
      let cl_from = IFace(cl, Always, self_argument :: args_from) in
      let cl_to = IFace(cl, Always, self_argument :: args_to) in

      let var_from_subst = 
	List.map 
	  (fun (t,tsubst) -> (t, (tsubst,tsubst)))
	  self#from_subst
      in let var_to_subst = 
	List.map 
	  (fun (t,tsubst) -> (t, (tsubst,tsubst)))
	  self#to_subst
      in
      let ancestor_funs =
	List.map 
	  (function
	     | Resolved_renaming(oanc, args,_,_,_) -> 
		 let arg_funs =
		   functor_action_argument_list
		     sub_ns				(* name_space *)
		     false				(* id_crea_flag *)
		     map_action				(* actions *)
		     true				(* variance *)
		     var_from_subst			(* domain_subst *)
		     var_to_subst			(* codomain_subst *)
		     args				(* argument_list *)
		     oanc#get_parameters		(* parameter_list *)
		 in
		 let body =
		   if args = [] 
		   then
		     Term(name_of_class_map_struct oanc,Always,[])
		   else
		     SmartApplication(
		       Term(name_of_class_map_struct oanc,Always,[]),
		       arg_funs)
		 in let term = 
		     if !Global.optimize_expressions && (args = []) 
		     then
		       Term(super_label oanc,Always,[])
		     else
		       Application(
			 body,
			 Application(
			   Term(super_label oanc,Always,[]),
			   coalgebra_term))
		 in
		     super_label oanc, term
						(* no other stuff here *)
	     | _ -> assert(false)
	  ) 
	  cl#get_ancestors
      in
      let method_funs =
	List.map
	  (fun meth ->
	     let ssub_ns = sub_space sub_ns in		(* do currying here *)
	     let argtypes = member_arg_list meth in
	     let substargtypes = 
	       List.map (substitute_types_only eq_types self#to_subst) 
		 argtypes in
	     let argids = 
	       create_ids ssub_ns substargtypes in 
	     let pre_args =			(* convert method arguments *)
		     List.map2
		       (fun (id,substtyp) typ ->
			  functor_action 
    			    ssub_ns			(* name_space *)
			    false			(* id_crea_flag *)
			    map_action			(* actions *)
			    false			(* variance *)
			    var_to_subst		(* domain_subst *)
			    var_from_subst		(* codomain subst *)
			    (Term(id,Always,[]))	(* expression *)
			    typ				(* typ *)
		       )
		       argids
		       argtypes
	     in let args = match meth#get_domain with
	       | Product _ -> Tuple(pre_args)
	       | _ -> List.hd pre_args
	     in
	     let mexpr = 
	       Application(
		 Application(
		   Term(meth#get_name, Always, []),
		   coalgebra_term),
		 args)
	     in
	     let cfun = 
		 Abstraction(
		   argids,
		   functor_action 
    		     ssub_ns				(* name_space *)
		     false				(* id_crea_flag *)
		     map_action				(* actions *)
		     true				(* variance *)
		     var_from_subst			(* domain_subst *)
		     var_to_subst			(* codomain subst *)
		     mexpr				(* expression *)
		     meth#get_codomain			(* typ *)
		   )
	     in
	       meth#get_name, opt_expression cfun
	  )
	  cl#get_sig_actions
      in
      	Defn(
	  name_of_class_map_struct cl,
	  (if fun_ids = [] 
	   then
	     []
	   else
	     [(List.map (fun (id,t) -> Declared(id,t)) fun_ids)]
	  ),
					(* the type *)
	  Function(cl_from,cl_to),
					(* the definition *)
	  if !Global.optimize_expressions && (fun_ids = []) 
	  then
	    Term(name_of_identity_function(),Always,[])
	  else
	    Abstraction(
	      [name_of_coalgebra, cl_from],
	      RecordTuple(ancestor_funs @ method_funs)
	    )
	)


	  (* this override changes also assert_coalgebra_decl ! *)
    method assert_coalgebra_type = 
      let from_args,_ = self#double_arguments 
      in
      Predtype(Formula(Term(name_of_assert cl,
			    Always,
			    self_argument :: from_args)))


    method private do_admissable =
      let from_args,to_args = self#double_arguments in
      let fun_ids = variance_flatten self#fun_ids_var in
      let fun_exprs = List.map (fun (id,t) -> Term(id,Always,[])) fun_ids in
      let struct_body =
	if fun_ids = []
	then Term(name_of_class_map_struct cl, Isabelle_only,
		  self_argument :: (from_args @ to_args))
	else
	  SmartApplication(
	    Term(name_of_class_map_struct cl, Isabelle_only,
		 self_argument :: (from_args @ to_args)),
	    fun_exprs)
      in
	Defn(
	  name_of_admissable_map_fun_pred cl,
	  ([Declared(name_of_coalgebra, self#assert_coalgebra_type)] ::
	     if fun_ids = [] 
	     then []
	     else
	       [(List.map (fun (id,t) -> Declared(id,t)) fun_ids)]),
	  Bool,
	  Application(
	    Term(name_of_assert cl, Always, self_argument :: to_args),
	    Application(
	      struct_body,
	      coalgebra_term)))


    method make_body = 
      [self#do_import]
      @ self#fun_decls
      @
      [
	self#class_map_struct;
	self#assert_coalgebra_decl;
	self#do_admissable;
      ]

    method get_proofs = []

end

class ccsl_map_struct_theory cl =
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_map_struct_theory 
  cl eq_ccsl_types


(***********************************************************************
 ***********************************************************************
 *
 * Definition of MAP: now the real map
 *
 *)

class ['class_type, 'member_type] ccsl_pre_map_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_map_struct_theory cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
       [ name_of_class_map cl;
       ]

    method get_name = ccsl_map_theory_name cl

    method get_parameters = 
      let left,right = self#double_parameters
      in
        left@right

    method private do_import =
      let left,right = self#double_arguments
      in
	Import(
	  [(ccsl_final_theory_name cl, left);
	   (ccsl_map_struct_theory_name cl, 
	    (TypeArgument(TypeConstant(
			    name_of_final_type cl, Always, left)))
	    :: ( left @ right));
	   (ccsl_final_props_theory_name cl, 
	    (TypeArgument(TypeConstant(
			    name_of_final_type cl, Always, left))) :: right)
	  ]
	)


    method private class_map =
      let (param_from,param_to) = self#double_parameters in
      let args_from = arguments_from_parameters param_from in
      let args_to = arguments_from_parameters param_to in
      let cl_from = 
	TypeConstant(name_of_final_type cl, Always, args_from ) in
      let cl_to = 
	TypeConstant(name_of_final_type cl, Always, args_to) in
      let funs_type = 
	Predtype(
	  Formula(
	    Application(
	      Term(name_of_admissable_map_fun_pred cl, Always,
		   TypeArgument(cl_from) :: args_from @ args_to),
	      Term(name_of_final_coalgebra cl, Always, args_from))))
      in
      let funs_name = create_id_with_preference ns "funs" "f"
      in
      let pre_body = 
	if args_from = []		(* no type parameters *)
	then
	  Term(name_of_class_map_struct cl, Always,[])
	else
	  Application(
	    Term(name_of_class_map_struct cl, Always,[]),
	    Term(funs_name,Always,[]))
      in let body =
	  Application(
	    pre_body,
	    Term(name_of_final_coalgebra cl, Always, args_from)
	  )
      in
      	Defn(
	  name_of_class_map cl,
	  (if args_from = []		(* no type parameters *)
	   then []
	   else
	     [[Undeclared(funs_name, funs_type)]]
	  ),
					(* the type *)
	  Function(cl_from, cl_to),
					(* the definition *)
	  (if !Global.optimize_expressions && (args_from = []) 
	   then
	     Term(name_of_identity_function(), Always, [])
	   else
	     Application(
	       Application(
		 Term(name_of_class_struct_coreduce cl,Always,[]),
		 Term(name_of_final_coalgebra cl, Always, args_to)
	       ),
	       body))
	)


    method make_body = 
      [
	self#do_import;
	self#class_map
      ]

    method get_proofs = []

end

class ccsl_map_theory cl =
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_map_theory cl eq_ccsl_types




(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
