(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 11.11.99 by Hendrik
 *
 * Time-stamp: <Wednesday 15 May 02 18:22:20 tews@ithif51>
 *
 * substitution and optimization for top_types
 *
 * $Id: logic_util.mli,v 1.10 2002/05/22 13:42:45 tews Exp $
 *
 *)

open Top_variant_types


(***********************************************************************
 ***********************************************************************
 * substitution
 *
 * the substitution is given as two assoc lists, 
 * the first gives a substitution on types, 
 * with elements top_types * top_types 
 * the second gives a substitution on values.
 *
 * (Until now substitution on formulas was not needed. But if it is, 
 *  this type will be changed into a triple and the functions below
 *  will be adjusted.)
 *
 * Substitution on types works straightforward, since there are no 
 * binders for type variables. 
 * 
 * Types that occur inside formulas and expressions are not correctly 
 * substituted. Currently type substitution occurs under the constructor
 * TypedTerm (internal representation of type annotations), but not under 
 * TermVar, Forall, and the other binders. This bug is related with 
 * another one in comparing identifiers (they are always compared by 
 * name). It doesn't really make sense to solve one without the other.
 *
 * Substitution on expressions is more difficult because of 
 * binders (Lambda, Forall, Exists): If the substitution 
 * contains a pair (Term(v,_,_), t), where t is an arbitrary 
 * top_expression, then t is NOT substituted for Term(v,_,_) 
 * below all binders, which declare a variable v. 
 * If the first element is not of the form Term(_,_,_), 
 * then substitution takes place EVERYWHERE.
 * 
 * Currently the following binders are recognized:
 * 
 * Abstraction(decl_list,_ ) 	   all strings in decl_list
 * Let(decl_list,_)		   following *PVS Semantics*
 * 				   bindings accumulate as the
 * 				   decl_list is processed from left 
 * 				   to right
 *
 * Case(_,variant_list)		   For pairs in variant_list of the 
 * 				   form 
 * 	   (Application(_,Term(v,_,_)),_)
 * 	   (Application(_,Tuple([Term(v1,_,_); Term(v2,_,_); ...]),_)
 * 	   (SmartApplication(_,[Term(v1,_,_); ... ]),_)
 * 				   all strings v, v1, v2 ... 
 * 				   (ocurring in a term are recognized
 *
 * Comprehension(s,_,_)            s is recognized
 * Forall/Exists/MetaForall        all strings in the declaration lists
 *
 * 
 * Note that the substitution functions below can not only be used 
 * to substitute variables and TypeParameters. You can substitute 
 * comples types (or expressions)!
 * 
 * Other caveats: 
 * 
 * - You can't substitute methods and constructors directly, instead 
 *   you have to substitute the Method( ... ) and Constructor( ... ) 
 *   expressions.
 * 
 *)


type ('cl, 'mem) substitution_type = 
  (('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_types) list  *
  (('cl, 'mem) top_pre_expressions * ('cl, 'mem) top_pre_expressions) list



(****************
 * this is the general substitution function:
 * in substitute (eq_typ,eq_ex) (sub_typ,sub_ex) typ
 * there are two equality functions eq_typ and eq_ex, used to find 
 * types and expressions in the two assoc lists sub_typ and sub_ex.
 * 
 * For substituting types only, there is an abbreviation below
 *)

val substitute_types : 
  ((('cl, 'mem) top_pre_types -> 
      ('cl, 'mem) top_pre_types -> bool) *
   (('cl, 'mem) top_pre_expressions -> 
      ('cl, 'mem) top_pre_expressions -> bool)) ->
    ('cl, 'mem) substitution_type -> 
      ('cl, 'mem) top_pre_types -> 
	('cl, 'mem) top_pre_types 


(*
 * general substitution on argument lists
 *)
val substitute_arguments :
  ((('cl, 'mem) top_pre_types -> 
      ('cl, 'mem) top_pre_types -> bool) *
   (('cl, 'mem) top_pre_expressions -> 
      ('cl, 'mem) top_pre_expressions -> bool)) ->
    ('cl, 'mem) substitution_type -> 
      ('cl, 'mem) top_pre_argument_type list  -> 
	('cl, 'mem) top_pre_argument_type list 

(****************
 * specialize the general function to just substitute the types 
 * given in the assoc list
 *)
val substitute_types_only :
  (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types -> bool) ->
    (('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_types) list ->
      ('cl, 'mem) top_pre_types -> 
	('cl, 'mem) top_pre_types 


val substitute_arguments_types_only :
  (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types -> bool) ->
    (('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_types) list ->
      ('cl, 'mem) top_pre_argument_type list -> 
	('cl, 'mem) top_pre_argument_type list
  

(***********************************************************************
 *
 * optimization of formulas and expressions 
 *
 *)

val opt_formula : 
  ('cl, 'mem) top_pre_formulas -> 
    ('cl, 'mem) top_pre_formulas
     
val opt_expression :
  ('cl, 'mem) top_pre_expressions ->
    ('cl, 'mem) top_pre_expressions


(*******************************************************************
 *
 * utility functions for variances 
 *
 *)

    (* replace Pair with Pos/Neg/Mixed *)

val make_simple : variance_type -> variance_type

val valid_variance : variance_type -> bool

val variance_subst : variance_type -> variance_type -> variance_type

val variance_join : variance_type -> variance_type -> variance_type


    (* flatten this option list in the obvious way *)

val variance_flatten : 
  ('a option * 'a option) list -> 'a list


    (* convert a list of identifiers with variance information, 
     * as procuced by create_ids_with_variance, having elements like
     *   (Some( neg_name, typ ), Some(pos_name, typ))
     * into a list of terms with variance information, haveing elements
     *   (xx, (Some pos_term, Some neg_term))
     * where the xx comes from the first argument list.
     * This result type matches the type param_lifting_type in 
     * lifting.ml and ccsl_variance_separation_type in morphism.ml
     *)
val var_terms_from_var_ids : 
  'a list ->
    ((string * 'b) option * (string * 'c) option) list ->
      ('a
       * (('cl, 'mem) top_pre_expressions option 
	  * ('cl, 'mem) top_pre_expressions option )) list


(* HH
 *     (* flatten a list of identifiers with variance information 
 * 	* and convert them into an expression list, ready 
 * 	* for application
 * 	*)
 * val arg_terms_from_var_ids :
 *   ((string * 'a) option * (string * 'a) option) list ->
 *     ('cl, 'mem) top_pre_expressions list
 *   
 *)

(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)



