(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 9.3.02 by Hendrik
 *
 * Time-stamp: <Friday 10 May 02 12:42:56 tews@ithif51>
 *
 * The class method pass : coreduce and friends
 * requires that feature pass has been done
 * 
 * $Id: class_methods.ml,v 1.3 2002/05/22 13:42:38 tews Exp $
 *
 *)

open Util
open Error
open Top_variant_types
open Types_util
open Member_class
;;


(*********************************************************
 *
 * Utility functions
 *
 *)


let check_for_clash loc cl name = 
  try 
    ignore(cl#find_member name); 
    warning_message loc
      (name ^ " not defined because of name clash.");
    false
  with 
    | Member_not_found -> true


let make_class_suit_type cl self_subst =
  let pre_class_suit_type = match cl#get_all_sig_actions with
    | [ m ] -> m#get_full_type
    | ml -> Product(List.map (fun m -> m#get_full_type) ml)
  in
  let subst = [ Self, self_subst  ] 
  in
    ccsl_substitute_types subst pre_class_suit_type 


let add_class_method cl name typ sort =
  let m_tok = {token_name = name; loc = None};
  in
  let mem = new ccsl_pre_member_class cl m_tok []
	      (Product[]) typ NoVisibility sort
  in
    cl#add_member mem



(*********************************************************
 *
 * invariance
 *
 *)

let add_invariance cl inv_name = 
  let tv = FreeTypeVariable(Type_variable.fresh()) in
  let class_suit_type = make_class_suit_type cl tv in			  
  let inv_type = Function( class_suit_type, 
			   Function( Function(tv, Bool), Bool))
  in
    add_class_method cl inv_name inv_type Class_Special


(*********************************************************
 *
 * bisimulation
 *
 *)

let add_bisimulation cl bisim_name =
  let tv1 = FreeTypeVariable(Type_variable.fresh()) in
  let tv2 = FreeTypeVariable(Type_variable.fresh()) in
  let suit_1 = make_class_suit_type cl tv1 in
  let suit_2 = make_class_suit_type cl tv2 in
  let bisim_type =
    Function
      (suit_1,
       Function(suit_2,
		Function( 
		  Function(Product[tv1;tv2], Bool), Bool)))
  in
    add_class_method cl bisim_name bisim_type Class_Special


(*********************************************************
 *
 * morphism
 *
 *)

let add_morphism cl morph_name = 
  let tv1 = FreeTypeVariable(Type_variable.fresh()) in
  let tv2 = FreeTypeVariable(Type_variable.fresh()) in
  let suit_1 = make_class_suit_type cl tv1 in
  let suit_2 = make_class_suit_type cl tv2 in
  let morph_type =
    Function
      (suit_1,
       Function(suit_2,
		Function( 
		  Function(tv1,tv2), Bool)))
  in
    add_class_method cl morph_name morph_type Class_Special


(*********************************************************
 *
 * coreduce
 *
 *)

(* add coreduce to cl *)
let add_coreduce cl coreduce_name =
  let tv = FreeTypeVariable(Type_variable.fresh()) in 
  let class_suit_type = make_class_suit_type cl tv in
  let coreduce_type = Function( class_suit_type, Function(tv, Self))
  in
    add_class_method cl coreduce_name
      coreduce_type Class_Coreduce


(*********************************************************
 *
 * the pass
 *
 *)


let do_class cl  =
  let loc = remove_option (cl#get_token).loc 
  in
    (let coreduce_name = Names.name_of_class_coreduce cl in
       if (cl#has_feature FinalSemanticsFeature) &&
	 (check_for_clash loc cl coreduce_name)
       then
	 add_coreduce cl coreduce_name
    );

    (let inv_name = Names.name_of_private_struct_invariance cl 
     in
       if check_for_clash loc cl inv_name
       then
	 add_invariance cl inv_name
    );

    (let morph_name = Names.name_of_full_morphism_struct_pred cl in
       if (cl#has_feature HasMorphismFeature) &&
	 (check_for_clash loc cl morph_name)
       then 
	 add_morphism cl morph_name
    );

    (let bisim_name = Names.name_of_private_struct_bibisimulation cl in
       if (cl#has_feature HasBisimulationFeature) &&
	 (check_for_clash loc cl bisim_name)
       then
	 add_bisimulation cl bisim_name
    )


let do_ast = function
  | CCSL_class_dec cl -> do_class cl
  | CCSL_adt_dec adt -> ()
  | CCSL_sig_dec si -> ()


let ccsl_class_methods_pass (ast: Classtypes.ccsl_ast list) = 
    List.iter do_ast ast;;


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

