(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 1.10.97 by Hendrik
 *
 * Time-stamp: <Sunday 23 June 02 14:38:41 tews@ithif56.inf.tu-dresden.de>
 *
 * Utility functions for ccsl parser
 *
 * $Id: parser.ml,v 1.25 2002/07/03 12:01:16 tews Exp $
 *
 *)


open Util
open Error
open Global
open Top_variant_types
open Top_variant_types_util
open Classtypes
open Ccsl_pretty
open Types_util
open Iface_class
open Member_class
open Substitution
open Parser_util

open Symbol;;

let d s =
  if debug_level _DEBUG_PARSER
  then begin
    prerr_string ("evaluation: " ^ s ^ "\n");
    flush stderr
  end;;


    (* keeps the last lexed location, for parse errors *)
let last_loc = ref { file_name = "";
		     start_line = -1;
		     start_char = -1;
		     end_line = -1;
		     end_char = -1
		   }

 (* documentation -> mli *)
type parsing_type = 
  | Parse_Nothing (* that is, parse outside of a class and outside
		   * of an adt. It might well be, that we parse a 
		   * groundtype declaration!
		   *)
  | Parse_IFace of ccsl_iface_type
  | Parse_Class of  ccsl_iface_type 
  | Parse_Adt of ccsl_iface_type
  | Parse_Sig of ccsl_iface_type
  | Parse_Anon of ccsl_iface_type

let current_iface = ref (Parse_Nothing : parsing_type )

 (* documentation -> mli *)
let ciface () = match !current_iface with
  | Parse_IFace( ifa ) -> ifa
  | Parse_Class( cl ) -> cl
  | Parse_Adt( adt ) -> adt
  | Parse_Sig si -> si
  | Parse_Anon an -> an
  | Parse_Nothing -> 
      begin assert(false) end

 (* documentation -> mli *)
let ccl () = match !current_iface with
  | Parse_Class( cl ) -> cl 
  | Parse_IFace _
  | Parse_Adt _
  | Parse_Sig _
  | Parse_Anon _
  | Parse_Nothing -> 
      begin assert(false) end

 (* documentation -> mli *)
let cadt () = match !current_iface with
  | Parse_Adt( adt ) -> adt 
  | Parse_IFace _
  | Parse_Class _ 
  | Parse_Sig _
  | Parse_Anon _
  | Parse_Nothing ->
      begin assert(false) end

 (* documentation -> mli *)
let csig () = match !current_iface with
  | Parse_Sig( si ) -> si
  | Parse_Anon an -> an
  | Parse_IFace _
  | Parse_Class _ 
  | Parse_Adt _
  | Parse_Nothing ->
      begin assert(false) end


 (* documentation -> mli *)
let current_visibility = ref (Public : ccsl_visibility)

let cvis() = match !current_visibility with
  | Public -> Public
  | Private -> Private
  | NoVisibility ->			(* called in wrong context *)
	 assert(false)


let ground_type_nat loc =
  let nat_error () =
    begin
      error_message loc
	("Groundtype " ^ (get_nat_type()) ^ " not defined.");
      raise Parsing.Parse_error
    end
  in
    try 
      match find (get_nat_type()) with
	| CCSL_GroundTypeSymb(id) ->
 	    (if check_parameters (get_ground_type_parameters id) []
	     then Groundtype(id,[])
	     else nat_error()
	    )
	    
	| ClassSymb _ 
	| AdtSymb _ 
	| TypevarSymb _
	| VarSymb _
	| SigSymb _
	| MemberSymb _ ->
	    nat_error()
					(* this is an internal error *)
	| InstClassSymb _
	| InstAdtSymb _ -> assert(false)
    with
	Table.Not_defined -> nat_error()



    (* make a new location spanning the two arguments 
     * 
     * val new_loc : location_type -> location_type -> location_type
     *)

let new_loc firstloc lastloc =
  {file_name = firstloc.file_name;
   start_line = firstloc.start_line;
   start_char = firstloc.start_char;
   end_line = lastloc.end_line;
   end_char = lastloc.end_char
  }


    (*****************************************************************
     * 
     * anonymous ground signatures, Part I
     *
     *)

let anon_ground_sig_count = ref 0

let name_anon_ground_sig () =
  incr anon_ground_sig_count;
  Names.anon_sig_prefix ^ (string_of_int !anon_ground_sig_count) ^ "_"

let close_anon_ground_sig () =
  match !current_iface with
    | Parse_Nothing -> ()
    | Parse_Anon ansig ->
	begin
	  ignore(Symbol.close_block());
	  Symbol.create_sig_content ansig;
	  current_iface := Parse_Nothing
	end
    | Parse_IFace _
    | Parse_Class _ 
    | Parse_Adt _
    | Parse_Sig _
      -> assert(false)


(**********************************************************************
 **********************************************************************
 * 
 * context checks, type checks, various error messages
 *
 *)
    (* val check_closing_id : string -> token_type -> unit
     *
     * checks if the name at the end of a data specification is matched
     *)
let check_closing_id name token =
  if name <> token.token_name then 
    warning_message (remove_option token.loc)
      ("Expected matching Id "^ name ^ " here, but found " ^
       token.token_name ^ ".")

let symbol_error_message token = begin
  error_message (remove_option token.loc)
    ("Undefined type symbol " ^ token.token_name ^ ".");
  if Global.debug_level _DEBUG_DUMP_SYMBOL_TABLE then begin
    prerr_string( "Symboltable: " );
    print_verbose (symbol_table_dump Top_variant_types_util.dump_symbol_brief);
  end;
  raise Parsing.Parse_error
end;;

let qualified_symbol_error_message token iface_name = begin
  error_message (remove_option token.loc)
    ("Type symbol " ^ token.token_name ^ " is not defined in " ^ 
     iface_name ^ ".");
  if Global.debug_level _DEBUG_DUMP_SYMBOL_TABLE then begin
    prerr_string( "Symboltable: " );
    print_verbose (symbol_table_dump Top_variant_types_util.dump_symbol_brief);
  end;
  raise Parsing.Parse_error
end;;

let instantiation_error loc name parameters actuals =
  begin
    error_message loc
      (name ^ " takes " ^
       (string_of_int (List.length parameters)) ^
       " type parameter(s); " ^
       "but you supplied " ^
       (if actuals = [] then "nothing" 
	else 
	  (string_of_ccsl_arguments actuals)
       )
       ^ "."
      );
  end

let instantiation_error_token token parameters actuals =
  begin
    instantiation_error (remove_option token.loc) 
      token.token_name parameters actuals;
    raise Parsing.Parse_error
  end


let overloading_error ifa token member = 
  begin
    error_message (remove_option token.loc)
      ("Member " ^ token.token_name ^ 
       " already declared in this signature");
    raise Parsing.Parse_error
  end

let wrong_id_type token expect found =
  begin
    error_message (remove_option token.loc)
      ("Expected a " ^ expect ^ ", but found " ^ 
       found ^ 
       (if found = "" then "" else " ") ^ 
       token.token_name ^ ".");
    raise Parsing.Parse_error
  end

let double_type_param token =
  begin
    error_message (remove_option token.loc)
      ("Typeparameter " ^ token.token_name ^
       " already defined\n"
      );
    raise Parsing.Parse_error
  end

(**********************************************************************
 **********************************************************************
 * 
 * make approbriate entries in the symboltable
 *
 *)


(* add_type_parameter is called either in the declaration of classes/adts 
 * or in the declaration of groundtypes. In the former case, identifiers 
 * must be inserted in the (newly created) symboltable, in the latter 
 * case not. I use the variable current_iface to distinguish the two 
 * cases. It is either Parse_Iface (for classes/adts) or Parse_Nothing 
 * for groundtypes.
 *)

let add_type_parameter variance token =		
  let var = Symbol.identifier_record token CCSL_TypeParameter in
  let _ = var.id_variance <- variance in
    match !current_iface with
      | Parse_Nothing			(* in a groundtype decl *)
      | Parse_Anon _ 
      | Parse_Sig _
	-> (TypeParameter var)
      | Parse_IFace _ ->			(* in a class/adt decl *)
	  begin
					(* no overloading for types *)
	    Symbol.create_type_parameter var;
	    (ciface())# add_parameter (TypeParameter var);
	    (TypeParameter var)
	  end

      | Parse_Class _ 
      | Parse_Adt _ 
	  -> assert(false)


let unique_type_parameter_check parameters =
  ignore(
    List.fold_left 
	   (fun present_list ->
	      function TypeParameter id -> 
		if List.mem id.id_token.token_name present_list
		then
		  double_type_param id.id_token;
		id.id_token.token_name :: present_list
	   )
	   [] parameters
  )
	   

let do_exact_variance loc vn vp =
  if (* ((vn = -1) && (vp = -1)) or *)
    ((vn <> -1) && (vn mod 2 = 0)) or
    ((vp <> -1) && (vp mod 2 = 1)) 
 then
    begin
      error_message loc "Invalid variance annotation.";
      raise Parsing.Parse_error
    end
  else
    Pair(vn,vp)
  


let add_class cl =			
  Symbol.create_class cl;
  Some( CCSL_class_dec cl )


let add_adt adt =			
  Symbol.create_adt adt;
  Some( CCSL_adt_dec adt )



let add_sig si =			
  List.iter Symbol.create_ground_type (si#get_all_ground_types);
  Symbol.create_sig si;
  Some( CCSL_sig_dec si )


(**********************************************************************
 **********************************************************************
 * 
 * semantic actions which are to long to get included in the grammar 
 *
 * Types, Parameters, Arguments
 *)

    (* 
     * common last action of the following do_{constructor,method,attribute} 
     * functions
     * adds the member to the current iface
     * now that I decided to treat components and the symbol table entry
     * in the iface, this 
     * add_member_to_iface looks a bit empty ;-(
     *)
let add_member_to_iface m = (ciface())#add_member m



   (* setting up an iface *)
let do_all_iface_start token =
  let _ = close_anon_ground_sig() in
					(* check for uniqueness *)
  let _ = 
    try match (find token.token_name) with
      | ClassSymb _ 
      | AdtSymb _
      | SigSymb _
	-> (begin
	      error_message (remove_option token.loc)
		("Signatures, Adt's, and classes must " ^
		 "have uniqe names.");
	      raise Parsing.Parse_error
	    end)
      | CCSL_GroundTypeSymb _

		(* these things have not yet added to the symboltable *)
      | TypevarSymb _	
      | VarSymb _
      | MemberSymb _ 
      | InstClassSymb _
      | InstAdtSymb _
      | MemberSymb _
	-> assert(false)
    with Table.Not_defined -> ()
  in let scope = Symbol.start_block () in
  let ifa = new ccsl_pre_iface_class token scope !parsing_prelude
  in
    (if !parsing_builtin_prelude then
       ifa#put_feature BuildinFeature);
    current_iface := Parse_IFace(ifa)


    (********************
     * anonymous ground signatures, Part II
     *)

let do_local_type_params parameterlist =
         (* block for local type parameters *)
  ignore(Symbol.start_block ());
         (* add them to the symboltable *)
  List.iter 
    (function (TypeParameter sym as tp) ->
       Symbol.create_type_parameter sym
    )
    parameterlist


    (* start an anon ground sig if
     * - we are not in a regular ground sig and 
     * - there is no anon ground sig
     * 
     * val start_anon_ground_sig : 
     *   token_type -> ccsl_parameter_type list -> ccsl_ast option
     *)
let start_anon_ground_sig token parameterlist =
  match !current_iface with
    | Parse_Nothing ->			(* outside everything *)
	let gsig_token = 
	  { token_name = name_anon_ground_sig();
	    loc = token.loc
	  } 
	in	
	          (* close the current anon ground sig, if there is one  *)
	let _ = do_all_iface_start gsig_token in
	let gsig = ciface() 
	in
	  begin
	    gsig#become_sig;
	    current_iface := Parse_Anon gsig;
	    do_local_type_params parameterlist;
	    (* return *)
	    Some (CCSL_sig_dec gsig)
	  end
			(* have an anon ground sig without type parameters 
			 * => keep it
			 *)
    | Parse_Anon gsig -> 
	do_local_type_params parameterlist;
	None
			(* inside an groundsig => just add type parameters *)
    | Parse_Sig _ -> 
	do_local_type_params parameterlist;
	None

					(* everything else is wrong *)
    | Parse_Class _
    | Parse_IFace _
    | Parse_Adt _
	-> assert false

   (* action for BEGIN ID *)
   (* anonymous ground signatures use the entry point 
    * do_all_iface_start
    *)
let do_iface_start token =
					(* check for clash with anon sig *)
  let anon = Names.anon_sig_prefix in
  let _ = 
    if String.length token.token_name >= String.length anon then
      if String.sub token.token_name 0 (String.length anon) = anon then
	begin
	  error_message (remove_option token.loc)
	    ("Identifiers starting with \"" ^ anon ^ "\"\n" ^
	     "are reserved for anonymous ground signatures.");
	      raise Parsing.Parse_error
	end
  in
    do_all_iface_start token

  (**********************
   * add importings to the class;
   * Note: the import list comes reversed !
   *
   * val do_class_imports : ccsl_importing_type list -> unit
   *
   *)
let do_class_imports il = 
    List.fold_right (fun i () -> (ccl())#add_iface_import i) il ()


let do_class_end startloc imports end_id =	
  let cl = ccl() in
  let name = cl # get_name in
    (* coreduce defined after inheritance in inheritance_pass *)
    do_class_imports imports; 
    ignore(Symbol.close_block ());
    current_iface := Parse_Nothing;
    check_closing_id name end_id;
    cl


let do_adt_end startloc end_id =			
  let adt = cadt() in
  let _ = match adt#get_adt_constructors with
    | [] -> (error_message (new_loc startloc (remove_option end_id.loc))
	       ("An adt must contain at least one constructor." );
	     raise Parsing.Parse_error
	    )
    | _ -> ()
  in
  let name = adt#get_name in
  let tv = FreeTypeVariable(Type_variable.fresh()) in 
  let subst = [ Carrier, tv  ] in
  let do_const_type const = ccsl_substitute_types subst const#get_full_type in
  let reduce_arg_typ = match adt#get_adt_constructors with
    | [ c ] -> do_const_type c
    | cl -> Product(List.map do_const_type cl)
  in
  let reduce_type = Function( reduce_arg_typ, Function(Carrier, tv))
  in
  let reduce_tok = {token_name = (Names.name_of_adt_reduce adt);
		    loc = None};
  in
  let reduce_mem = new ccsl_pre_member_class (ciface()) reduce_tok []
		     (Product[]) reduce_type
		     NoVisibility Adt_Reduce
  in
    add_member_to_iface reduce_mem;
    check_closing_id name end_id;
    current_iface := Parse_Nothing;
    ignore(Symbol.close_block ());
    adt


  (**********************
   * 
   * Note: the import list comes reversed !
   *
   * val do_sig_end: location_type -> ccsl_importing_type list -> 
   *                       token_type -> ccsl_iface_type
   *)

let do_sig_end startloc imports end_id =			
  let si = csig() in
  let name = si#get_name 
  in
    check_closing_id name end_id;
    current_iface := Parse_Nothing;
    ignore(Symbol.close_block ());
    List.fold_right (fun i () -> si#add_iface_import i) imports ();
    si


(* let start_groundtype token parameterlist =
 *   let gsig,new_gsig =
 *     match !anon_ground_sig, parameterlist with
 * 	 | None, _
 * 	 | Some _, _::_  -> 
 * 	     ((start_anon_ground_sig token.loc parameterlist),
 * 	      true)
 * 	 | Some gsig, [] -> gsig, false
 *   in
 *   let idrec = Symbol.identifier_record token (CCSL_GroundTypeDecl gsig)
 *   in begin
 * 	 ((if new_gsig 
 * 	   then Some (CCSL_sig_dec gsig)
 * 	   else None),
 * 	  idrec)
 *     end
 *)


    (* add a groundtype definition or declaration to the current signature
     * (which might be anonymous)
     * 
     * val add_groundtype : 
     *   token_type -> ccsl_parameter_type list -> 
     * 	 ccsl_input_types option -> unit
     *)
let add_groundtype token parameters typeopt =
			(* close local type parameters *)
  let _ = Symbol.close_block () in
  let gsig = csig() in
  let _ = unique_type_parameter_check (gsig#get_parameters @ parameters) in
  let idrec = Symbol.identifier_record token (CCSL_GroundTypeDecl gsig) in
  let _ = idrec.id_parameters <- parameters
  in
    begin
      (match typeopt with
	 | None -> ()
	 | Some typ -> 
	     begin
	       idrec.id_origin <- CCSL_GroundTypeDef gsig;
	       idrec.id_type <- typ;
	     end
      );
      gsig#add_ground_type idrec
    end


let infix_test = function
  | Function( Product( [_;_] ), _ ) -> true
  | Function( _, Function _ ) -> true
  | _ -> false


(* Binary operators in PVS 2.4
 * 
 * BinOp ::= 'o' | 'IFF' | '<=>' | 'IMPLIES' | '=>' | 'WHEN' | 'OR'
 * 	   | '\/' | 'AND' | '/\' | '&' | 'XOR' | 'ANDTHEN' | 'ORELSE'
 * 	   | '^' | '+' | '-' | '*' | '/' | '++' | '~' | '**' | '//' | '^^'
 * 	   | '|-' | '|=' | '<|' | '|>' | '=' | '/=' | '==' | '<' | '<='
 * 	   | '>' | '>=' | '<<' | '>>' | '<<=' | '>>=' | '#' | '@@' | '##'
 * 
 * These work in CCSL
 *
 *   # ## & * ** + ++ - / // /= /\ < << 
 *   <<= <= <=> <| == => > >= >> >>= @@ \/ ^ ^^ |- |= |> 
 *
 * excluding = and ~
 *
 *)

let valid_infix_string infixtoken =
  if !output_mode = Pvs_mode 
  then match infixtoken.token_name with
					(* let these pass *)
    | "#" | "##" | "&" | "*" | "**" | "+" | "++" | "-"
    | "/" | "//" | "/=" | "/\\" | "<" | "<<" | "<<=" 
    | "<=" | "<=>" | "<|" | "==" | "=>" | ">" | ">=" | ">>"
    | ">>=" | "@@" | "\\/" | "^" | "^^" | "|-" | "|=" 
    | "|>" 
      -> ()
					(* barf on others *)
    | _ -> 
	begin error_message (remove_option infixtoken.loc)
	    ("This infix operator is not allowed in PVS");
	  raise Parsing.Parse_error
	end 
  else
(* HENDRIK : add checks for Isabelle *)
    ()

    (* check for infix ground terms
     * 
     * val check_infix : 
     * 	 token_type -> ccsl_input_types -> unit
     *)
let check_infix infixtoken typ = 
  begin
    if not (infix_test typ) then
      begin error_message (remove_option infixtoken.loc)
	  ("Infix constants must be declared as functions " ^
	   "taking two arguments.");
	raise Parsing.Parse_error
      end;
    valid_infix_string infixtoken;
  end


    (* add a constant symbol to a ground signature
     *
     * val add_groundterm : 
     * 	 token_type -> ccsl_member_sort -> ccsl_parameter_type list ->
     * 	     ccsl_input_types -> ccsl_assertion option -> unit
     *)

let add_groundterm token membersort parameters typ formulaoption = 
			(* close local type parameters *)
  let _ = Symbol.close_block () in
  let _ = match membersort with
    | GroundTerm -> ()
    | InfixGroundTerm -> check_infix token typ
    | _ -> assert false
  in
  let gsig = csig() in
  let _ = unique_type_parameter_check (gsig#get_parameters @ parameters) in
  let dom = Product [] in
  let _ = 
    try 
      let second = gsig#find_local_member token.token_name in
      overloading_error (ciface()) token second
    with Member_not_found -> () 
  in
  let m = new ccsl_pre_member_class gsig token [] dom typ NoVisibility
    membersort
  in begin
      m#set_local_parameters parameters;
      add_member_to_iface m;
      (match formulaoption with
	 | None -> ()
	 | Some f -> 
	     (let def = { defined_method = m;
			  variables = [];
			  definition = f;
			  def_sequence = -1;
			}
	      in
		if debug_level _DEBUG_PARSERCHECK then
		  (match f with
		     | Symbolic f -> print_verbose
			 ("Def " ^ token.token_name ^ " : " ^
			  string_of_ccsl_formula f)
		     | _ -> ()
		  );
		gsig#add_definition def
	     )
      );
    end


(**********************************************************************
 **********************************************************************
 * 
 * semantic actions on class members
 *
 *)


(* val do_inherit : token_type -> ccsl_argument_type list -> 
 *   (ccsl_renaming list * location_type option) -> ccsl_ancestor_type
 *
 * tries to find the symbol in the symboltable, 
 * if not produce an errormessage and an error token
*)

let do_inherit token al (rename, endloc) =	
  let s = 
    try find token.token_name
    with Table.Not_defined -> symbol_error_message token in 
  let cl = match s with
					(* a ccsl class is ok*)
    | ClassSymb(cl) -> cl
					(* a ground type is not ok *)
    | CCSL_GroundTypeSymb _
					(* neither a type variable *)
    | TypevarSymb(_)
					(* neither variable *)
    | VarSymb(_)
					(* neither ccsl adt are all not ok*)
    | AdtSymb(_) 
					(* neither a signature *)
    | SigSymb _
					(* neiter a member *)
    | MemberSymb(_) -> 
	   wrong_id_type token "class" ""
					(* cant happen *)
    | InstClassSymb _
    | InstAdtSymb _
    | MemberSymb(_) -> assert(false)

  in
  let _ = if not (check_parameters cl#get_parameters al) then
    instantiation_error_token token cl#get_parameters al
  in let inherit_loc = (match endloc with
			  | None -> remove_option token.loc
			  | Some endl -> 
			      new_loc (remove_option token.loc) endl
		       )
  in
    Unresolved_renaming(cl, al, rename, inherit_loc)


    (* test if this is a valid type for a class constructor *)
let classconstructor_test dom codom =
  (match codom with
     | Self -> true
     | _ -> false)
  &
  (count_self dom = 0)


    (* test if this is a valid type for a proper attribute *)
let attribute_test dom codom =
  (match dom with
     | Product( Self :: typlist ) ->
	 (List.fold_left (fun n typ -> n + count_self typ) 0 typlist) = 0
     | Self -> true
     | _ -> false)
  &
  (count_self codom = 0)


    (* test if this is a valid type for a method *)
let method_test dom codom =
  (match dom with
     | Product( Self :: typlist ) -> true
     | Self -> true
     | _ -> false)


    (* action for class constructors 
     * val do_var_class_constructor : token_type -> 
     *        pvs_type -> pvs_type -> unit 
     *)
let do_var_class_constructor token dom codom =
  let _ =
    try 
      let second = (ccl())#find_local_member token.token_name in
      overloading_error (ciface()) token second
    with Member_not_found -> () in
  let _ = if not (classconstructor_test dom codom) then
    begin
      error_message (remove_option token.loc)
	("The Type " ^ 
	 (string_of_ccsl_type( Function(dom,codom) )) ^ 
	 " is not allowed for class constructors.");
      raise Parsing.Parse_error
    end in
  let m : ccsl_member_type = 
    new ccsl_pre_member_class (ciface()) token [] dom codom
      NoVisibility Var_Constructor 
  in
    add_member_to_iface m


    (* action for class constructors 
     * val do_const_class_constructor : token_type -> 
     *        pvs_type -> unit 
     *)
let do_const_class_constructor token codom =
			      (* Sidecondition CONST_CONSTRUCTOR applies *)
  let dom = Product [] in
  let _ =
    try 
      let second = (ccl())#find_local_member token.token_name in
      overloading_error (ciface()) token second
    with Member_not_found -> () in
  let _ = if not (classconstructor_test dom codom) then
    begin
      error_message (remove_option token.loc)
	("The Type " ^ 
	 (string_of_ccsl_type( Function(dom,codom) )) ^ 
	 " is not allowed for constant class constructors.");
      raise Parsing.Parse_error
    end in
  let m = new ccsl_pre_member_class (ciface()) token [] dom codom 
	    NoVisibility Const_Constructor 
  in
    add_member_to_iface m


    (* action for proper class attributes 
     * val do_attribute : token_type -> visibility -> 
     *       pvs_type -> pvs_type -> unit
     *)
let do_attribute token dom codom =	
  let _ =
    try 
      let second = (ccl())#find_local_member token.token_name in
      overloading_error (ciface()) token second
    with Member_not_found -> () in
  let _ = if not (attribute_test dom codom) then
    begin
      error_message (remove_option token.loc)
	("The Type " ^ 
	 (string_of_ccsl_type( Function(dom,codom) )) ^ 
	 " is not allowed for proper attributes.");
      raise Parsing.Parse_error
    end in
  let m = new ccsl_pre_member_class (ciface()) 
	    token [] dom codom (cvis()) (Proper_Attribute None)
  in
    add_member_to_iface m



    (* action for methods and improper class attributes
     * val do_method : token_type 
     *                 -> pvs_type -> pvs_type -> unit 
     *)
let do_method token dom codom =		
  let _ =
    try 
      let second = (ccl())#find_local_member token.token_name in
      overloading_error (ciface()) token second
    with Member_not_found -> () in
  let _ = if not (method_test dom codom) then
    begin error_message (remove_option token.loc)
	("The Type " ^ 
	 (string_of_ccsl_type( Function(dom,codom) )) ^ 
	 " is not allowed for methods.");
      raise Parsing.Parse_error
    end in
  let m = new ccsl_pre_member_class (ciface()) token [] dom codom (cvis()) 
	    Normal_Method 
  in    
    add_member_to_iface m


    (* do a definitional extension 
     * 
     * val do_definition : 
     *   token_type -> ccsl_input_types -> ccsl_input_types -> 
     * 	  ccsl_assertion -> unit
     * 
     *)
let do_definition token dom codom equation =
  let _ =
    try 
      let second = (ccl())#find_local_member token.token_name in
      overloading_error (ciface()) token second
    with Member_not_found -> () in
  let _ = if not (method_test dom codom) then
    begin error_message (remove_option token.loc)
	("The Type " ^ 
	 (string_of_ccsl_type( Function(dom,codom) )) ^ 
	 " is not allowed for methods.");
      raise Parsing.Parse_error
    end in
  let m = new ccsl_pre_member_class (ciface()) token [] dom codom 
	    NoVisibility Defined_Method
  in let def =
      { defined_method = m;
	variables = [];
	definition = equation;
	def_sequence = -1;
      }
  in
    if debug_level _DEBUG_PARSERCHECK then
      (match equation with
	 | Symbolic f -> print_verbose
	     ("Def " ^ token.token_name ^ " : " ^
	      string_of_ccsl_formula f)
	 | _ -> ()
      );
    (ciface())#add_definition def
  


   (* Create an identifier for a variable in an assertion/creation
    * 
    *)
let do_id_declaration (id, typ) =		
  let symb = { id_token = id;
	       id_type = typ;
	       id_parameters = [];
	       id_origin = CCSL_Var;
	       id_variance = Unset;
	       id_sequence = -1;
	     }
  in
    symb


  (**********************
   * add asserion importings to the class;
   * Note: the import list comes reversed !
   *
   * val do_assertion_start : 
   *     ccsl_importing_type list -> token_type * ccsl_input_types ->
   *       (token_type * ccsl_input_types) list -> 
   *         ccsl_identifier_record_type * ccsl_identifier_record_type list
   *   
   *
   *)
let do_assertion_start il selfvar varlist =
  begin
    List.fold_right (fun i () -> (ccl())#add_assertion_import i) il ();
    (do_id_declaration selfvar, List.map do_id_declaration varlist)
  end


  (**********************
   * built an assertion and add it to the current class
   *
   * val do_assertion : 
   *     token_type -> 
   *	    ccsl_identifier_record_type * ccsl_identifier_record_type list -> 
   *	      ccsl_assertion -> unit
   *)
let do_assertion token (selfvar,varlist) formula =
  let ass = { assertion_name = token;
	      self_variable = Some selfvar;
	      free_variables = varlist;
	      is_generated = false;
	      assertion_formula = formula
	    } in
    if debug_level _DEBUG_PARSERCHECK then
      (match formula with
	 | Symbolic f -> print_verbose
	     ("Ass " ^ token.token_name ^ " : " ^
	      string_of_ccsl_formula f)
	 | _ -> ()
      );
    (ccl())#add_assertion ass


  (**********************
   * built an creation condition and add it to the current class
   *
   * val do_creation : token_type -> ccsl_identifier_record_type list -> 
   *      ccsl_assertion -> unit
   *)
let do_creation token varlist formula =	
  let ass = { assertion_name = token;
	      self_variable = None;
	      free_variables = varlist;
	      is_generated = false;
	      assertion_formula = formula
	    } in
    if debug_level _DEBUG_PARSERCHECK then
      (match formula with
	 | Symbolic f -> print_verbose
	     ("Crea " ^ token.token_name ^ " : " ^
	      string_of_ccsl_formula f)
	 | _ -> ()
      );
    (ccl())#add_creation ass


  (**********************
   * add creation importings to the class;
   * Note: the import list comes reversed !
   *
   * val do_creation_start : 
   *   	 location_type -> 
   *   	   ccsl_importing_type list -> 
   *   	     (token_type * ccsl_input_types) list -> 
   *   	       ccsl_identifier_record_type list
   *
   *)
let do_creation_start crealoc il varlist =
  if (ccl())#has_constructors 
  then
    begin
      List.fold_right (fun i () -> (ccl())#add_assertion_import i) il ();
      List.map do_id_declaration varlist
    end
  else
    begin 
      error_message (crealoc)
	"Creation conditions are only allowed if constructors are present.";
      raise Parsing.Parse_error
    end
    



let do_selfvar_default () =		
  ( { token_name = "x";
      loc = None;
    },
    Self)


  (**********************
   * add theorem importings to the class;
   * Note: the import list comes reversed !
   *
   * val do_theorem_start : 
   *   	   ccsl_importing_type list -> 
   *   	     (token_type * ccsl_input_types) list -> 
   *   	       ccsl_identifier_record_type list
   *
   *)
let do_theorem_start il varlist =
  begin
    List.fold_right (fun i () -> (ccl())#add_theorem_import i) il ();
    List.map do_id_declaration varlist
  end


  (**********************
   * built an theorem and add it to the current class
   *
   * val do_theorem : token_type -> ccsl_identifier_record_type list -> 
   *      ccsl_formula -> unit
   *)
let do_theorem token varlist formula =
  let ass = { assertion_name = token;
	      self_variable = None;
	      free_variables = varlist;
	      is_generated = false;
	      assertion_formula = formula
	    } in
    if debug_level _DEBUG_PARSERCHECK then
      (match formula with
	 | Symbolic f -> print_verbose
	     ("Theorem " ^ token.token_name ^ " : " ^
	      string_of_ccsl_formula f)
	 | _ -> ()
      );
    (ccl())#add_theorem ass




(**********************************************************************
 **********************************************************************
 * 
 * semantic actions for adt members
 *
 *)

    (* check if the elements of the accessor list are unique *)
let rec unique_accessors = 
			(* don't have a better name for this function *)
  let rec spec_mem token = function
    | [] -> false
    | t :: tl -> (if t.token_name = token.token_name 
		  then begin
		    error_message (remove_option token.loc)
		      ("accessor names must be unique " ^
		       "within one constructor.");
		    raise Parsing.Parse_error
		  end
		  else spec_mem token tl) in
  function
    | [] -> true
    | [t] -> true
    | t :: tl -> (unique_accessors tl) & not (spec_mem t tl)


    (* check if number of accessors matches arity of domain *)
let adt_check_accessors token accessors dom =
  let arity = match dom with 
    | Product l -> List.length l
    | _ -> 1
  in
    if arity <> List.length accessors then
      begin
	error_message (remove_option token.loc)
	  ("Number of accessors does not match " ^
	   "arity of " ^ token.token_name ^ ".");
	raise Parsing.Parse_error
      end


    (* action for constant adt constructors
     *
     * takes name accessorlist type
     * because it is a constructor with no arguments, accessorlist 
     * should contain at most one id and type should be Carrier
     *
     * val do_const_adt_constructor : token_type -> token_type list -> 
     * 		pvs_type -> unit
     *)
let do_const_adt_constructor token accessors codom =
  let dom = Product [] in
       (* no overloading any more *)
  let _ = 
    try 
      let second = (cadt())#find_local_member token.token_name in
      overloading_error (ciface()) token second
    with Member_not_found -> () in
  let _ = if accessors <> [] then 
    begin 
      error_message (remove_option token.loc)
	("Constant constructors cannot have accessors.");
      raise Parsing.Parse_error
    end in
  let _ = match codom with
    | Carrier -> ()
    | _ -> begin error_message (remove_option token.loc)
	  ("An ADT constructor must have a " ^ 
	   "codomain type of Carrier.");
	         raise Parsing.Parse_error
      end in
  let m = new ccsl_pre_member_class (ciface()) token [] dom codom NoVisibility
    Adt_Const_Constructor
  in
  let rec_token = {token_name = (Names.constructor_recognizer m);
		   loc = None;
		  }
  in
  let recognizer = new ccsl_pre_member_class (ciface())
		     rec_token []
		     (Product[]) (Function(Carrier,Bool))
		     NoVisibility Adt_Recognizer
  in
    m#register_recognizer recognizer;
    add_member_to_iface m;
    add_member_to_iface recognizer
  

    (* generate accessor names c_acc1 ... c_acc5 *)
let generate_accessor_names c_token number = 
  let acc = (Names.constructor_accessor_name c_token.token_name) in 
  let rec doit = function
    | 0 -> []
    | i ->  (acc ^ (string_of_int i)):: doit (i - 1)
  in
    List.rev_map
      (fun name -> {token_name = name; loc = None }
      )
    (if number = 1 then [acc]
     else doit number)
    

    (* action for non constant adt constructors 
     *
     * takes name accessorlist domain codomain
     * because it is a constructor, codomain should be Carrier
     *
     * val do_var_adt_constructor : token_type -> token_type list -> 
     *   pvs_type -> pvs_type -> unit
     *)
let do_var_adt_constructor token accessors dom codom =
  let _ =
    try 
      let second = (cadt())#find_local_member token.token_name in
      overloading_error (ciface()) token second
    with Member_not_found -> () in
  let _ = unique_accessors accessors in
(* Hendrik check for adt wide unique accessors, recognizers *)
				(* check arity of domain and accessors *)
  let _ = if accessors <> [] then 
    adt_check_accessors token accessors dom in
					(* check docomain *)
  let _ = match codom with
    | Carrier -> ()
    | _ -> begin error_message (remove_option token.loc)
	  ("An ADT constructor must have a " ^ 
	   "codomain type of Carrier.");
	         raise Parsing.Parse_error
      end in
		(* accessors are optional, generate something if necessary *)
  let accessor_names = 
    if accessors <> [] then 
      accessors
    else 
      let arity = match dom with 
    	| Product l -> List.length l
    	| _ -> 1
      in generate_accessor_names token arity
			(* for constructors the visibility flag is ignored *)
  in let m = new ccsl_pre_member_class (ciface()) token [] 
	       dom codom NoVisibility
	       Adt_Var_Constructor  
  in let domtyplist = match dom with
    | Product tl -> tl
    | t -> [t]
  in let accessor_mems =
      List.map2 (fun acc domtype -> 
		   let acc = new ccsl_pre_member_class (ciface())
			       acc []
			       (Product[]) (Function(Carrier,domtype))
			       NoVisibility Adt_Accessor
		   in
		     acc
		) 
	accessor_names domtyplist
  in
  let rec_token = {token_name = (Names.constructor_recognizer m);
		   loc = None }
  in
  let recognizer = new ccsl_pre_member_class (ciface())
		     rec_token []
		     (Product[]) (Function(Carrier,Bool))
		     NoVisibility Adt_Recognizer
  in
    m#register_accessors accessor_mems;
    m#register_recognizer recognizer;
    add_member_to_iface m;
    add_member_to_iface recognizer;
    List.iter add_member_to_iface accessor_mems


(**********************************************************************
 **********************************************************************
 * 
 * semantic actions for signature members
 *
 *)

let do_sig_type_decl token =
  let si = csig() in
  let var = Symbol.identifier_record token (CCSL_GroundTypeDecl si) in
  let _ = var.id_parameters <- []
  in 
    si#add_ground_type var
    


    (*******************************************************************
     *******************************************************************
     *
     * Type action section
     *
     *)


    (* check what token is and return a Value or a Type *)
let do_argument_id token  : ccsl_argument_type =
  let s = 
    try find token.token_name
    with Table.Not_defined -> symbol_error_message token 
  in
    match s with
					(* a basetype might be ok *)
      | CCSL_GroundTypeSymb id -> 
	  (if check_parameters (get_ground_type_parameters id) [] 
	   then TypeArgument(Groundtype(id,[]))
	   else instantiation_error_token token [] []
	  )
					(* a type variable is ok *)
      | TypevarSymb(id) -> TypeArgument(BoundTypeVariable(id))
					(* a ccsl class might be ok*)
      | ClassSymb(cl) -> (if (check_parameters cl#get_parameters [])  then
		       	    TypeArgument(Class(cl, []))
			  else
		       	    instantiation_error_token token 
			      cl#get_parameters []
			 )
					(* a ccsl adt might be ok*)
      | AdtSymb(adt) -> (if (check_parameters adt#get_parameters []) then
			   TypeArgument(Adt(adt, Always, []))
			 else
			   instantiation_error_token token 
			     adt#get_parameters [] 
			)

					(* a signature is wrong *)
      | SigSymb _ ->
	  wrong_id_type token "type" "signature identifier"
					(* a term is wrong *)
      | VarSymb _ 
	-> wrong_id_type token "type" "term"
					(* members too *)
      | MemberSymb(m) -> wrong_id_type token "type" 
	    (if m#hosting_class#is_class 
	     then "class member"
	     else "adt constructor")
	                    (* no instanciations are stored in CCSL_ID *)
      | InstClassSymb _ 
      | InstAdtSymb _ -> assert(false)



		       (* give Self back, if we are parsing a class *)
let self_action loc =			
  match !current_iface with
    | Parse_Class( _ ) -> Self
    | _ ->  (begin 
	       error_message loc
		 "Keyword Self is not allowed in this context.";
	       raise Parsing.Parse_error
	     end);;


		    (* give Carrier back, if we are parsing an adt *)
let carrier_action loc =
  match !current_iface with
    | Parse_Adt(_) -> Carrier
    | _ -> (begin 
	      error_message loc
		"Keyword Carrier is not allowed in this context.";
	      raise Parsing.Parse_error
	    end)

let do_type_product tl loc =
  begin
    if List.length tl < 2 then 
      let emergency_token = emergency_token() in 
	begin
    	  error_message loc
	    ( "Typeconstructor Product needs at least two arguments.");
    	  raise Parsing.Parse_error
	end 
    else 
      Product tl
  end


let do_type_appl token arglist =	
  try
    match find token.token_name with
					(* a ccsl class is ok*)
    | ClassSymb(cl) -> (if (check_parameters cl#get_parameters arglist) then
			    Class(cl,arglist)
		     	else
			  instantiation_error_token 
			    token cl#get_parameters arglist
		       )
					(* an adt is ok*)
    | AdtSymb(adt) -> (if (check_parameters adt#get_parameters arglist) then
			 Adt(adt,Always,arglist)
		       else
			 instantiation_error_token 
			   token adt#get_parameters arglist
		      )
					(* a ground type might be ok *)
    | CCSL_GroundTypeSymb(id) ->
	let ground_params = get_ground_type_parameters id
	in
 	  if check_parameters ground_params arglist 
	  then Groundtype(id,arglist)
	  else instantiation_error_token token ground_params arglist
	  
					(* the rest is not ok *)
    | TypevarSymb(_) 
    | VarSymb(_)
    | SigSymb _
    | MemberSymb(_) ->
	wrong_id_type token "parametric type" ""

					(* this is an internal error *)
    | InstClassSymb _
    | InstAdtSymb _ -> assert(false)
  with
      Table.Not_defined -> symbol_error_message token;;


    (*************
     * action for the optional class in Always and eventually
     *)
let do_instclass token arglist =	
  let s = 
    try find token.token_name
    with Table.Not_defined -> symbol_error_message token in 
  let cl = match s with
					(* a ccsl class is ok*)
    | ClassSymb(cl) -> cl
					(* a ground type is not ok *)
    | CCSL_GroundTypeSymb _
					(* neither a type variable *)
    | TypevarSymb(_)
					(* neither variable *)
    | VarSymb(_)
					(* neither ccsl adt are all not ok*)
    | AdtSymb(_) 
					(* neither a signature *)
    | SigSymb _
					(* neiter a member *)
    | MemberSymb(_) -> 
	   wrong_id_type token "class" ""
					(* cant happen *)
    | InstClassSymb _
    | InstAdtSymb _
    | MemberSymb(_) -> assert(false)

  in if (check_parameters cl#get_parameters arglist) then
      Class(cl, arglist)
    else 
      instantiation_error_token token cl#get_parameters arglist


let treat_q_id_info = function
  | (None,None,None,id) -> ( NoIface, id )
  | (Some ifacetoken, argsopt, Some instloc, id) ->
      (try 
	 let iface = 
	   match find ifacetoken.token_name with (* Classes and ADTs are ok *)
	     | ClassSymb(cl) ->  cl
	     | AdtSymb(adt) -> adt
	     | SigSymb(si) -> si
		 
					(* a ground type is not ok *)
	     | CCSL_GroundTypeSymb(_)
					(* nor a type variable *)
	     | TypevarSymb(_) 
	     | VarSymb(_)
	     | MemberSymb(_) ->
		 wrong_id_type ifacetoken "signature" ""
	    
	     (* this is an internal error *)
	     | InstClassSymb _
	     | InstAdtSymb _  -> assert(false)
	 in
	 let arglist = match argsopt with 
	   | None -> []
	   | Some arglist -> arglist
					(* check type parameters in client *)
(* 	    in
 * 	    let _ = if not (iface#check_parameters arglist) then
 * 	      instantiation_error_token ifacetoken iface#get_parameters arglist 
 *)
	 in
	   (InstIface(iface, arglist, Some instloc), id)
       with
	   Table.Not_defined -> symbol_error_message ifacetoken
      )
  | _ -> assert(false)



		    (* check if symbol denotes a type if so, return the type *)
let do_type_symbol token symbol =
    match symbol with
					(* a basetype might be ok *)
      | CCSL_GroundTypeSymb(id) -> 
	  let ground_params = get_ground_type_parameters id
	  in
	    if check_parameters ground_params [] 
	    then Groundtype(id, [])
	    else instantiation_error_token token ground_params []
					(* a type variable is ok *)
      | TypevarSymb(id) -> BoundTypeVariable(id)
					(* a ccsl class might be ok*)
      | ClassSymb(cl) -> (if (check_parameters cl#get_parameters []) then
			    Class(cl, [])
			  else
			    instantiation_error_token token 
			      cl#get_parameters [] 
			 )
					(* a ccsl adt might be ok*)
      | AdtSymb(adt) -> (if (check_parameters adt#get_parameters []) then
			   Adt(adt, Always, [])
			 else
			   instantiation_error_token token 
			     adt#get_parameters [] 
			)

					(* a signature is wrong *)
      | SigSymb _ ->
	  wrong_id_type token "type" "signature identifier"
					(* a variable is not a type *)
      | VarSymb(id) -> 
	  wrong_id_type id.id_token "type" "variable"

      | MemberSymb(m) -> wrong_id_type token "type" 
	    (if m#hosting_class#is_class 
	     then "class member"
	     else "adt constructor")
			    (* no Instanciations are stored in Classnames *)
      | InstClassSymb _
      | InstAdtSymb _ -> assert(false)
	       
	    
let do_qualified_type_id q_id_info = 
  let instiface, id = treat_q_id_info q_id_info 
  in match instiface with
    | NoIface -> 
	let s = 
	  try find id.token_name
	  with Table.Not_defined -> symbol_error_message id 
	in
	  do_type_symbol id s

    | InstIface(iface,arglist,instloc) -> 
	let _ = (if not iface#is_sig 
		 then wrong_id_type id "type" ""
		) 
	in let gtypid = 
	    (try 
	       match iface#find_local id.token_name with
					(* a ground type might be ok *)
		 | CCSL_GroundTypeSymb(id) -> id

		 | TypevarSymb(_) 
		 | MemberSymb(_) ->
		     wrong_id_type id "type" ""
					(* this is an internal error *)
		 | ClassSymb(_) 
		 | AdtSymb(_) 
		 | VarSymb(_)
		 | SigSymb _

		 | InstClassSymb _
		 | InstAdtSymb _ -> 
		     assert(false)
	     with 
		 Table.Not_defined -> 
		   qualified_symbol_error_message id iface#get_name
	    )
	in let typeparams = get_ground_type_parameters gtypid
	in let _ =
	    if not (check_parameters typeparams arglist) then
	      instantiation_error_token id typeparams arglist 
	in
	  Groundtype(gtypid, arglist)

    | CurrentIface -> 
	assert(false)




    (*******************************************************************
     *******************************************************************
     *
     * functions for the logic parser
     *
     *)


let do_method_selection sub_expr q_id_info =
  let instiface, id = treat_q_id_info q_id_info in
  let _ = match instiface with
    | InstIface(iface,arglist,instloc) -> 
	begin
	  (if not iface#is_class then
	     let cltoken = match q_id_info with
	       | Some id,_,_,_ -> id 
	       | _ -> assert(false)
	     in
	       wrong_id_type cltoken "class" id.token_name);
	  (if not (check_parameters iface#get_parameters arglist) 
	   then
	     let cltoken = match q_id_info with
	       | Some id,_,_,_ -> id 
	       | _ -> assert(false)
	     in
	       instantiation_error_token cltoken iface#get_parameters arglist)
	end
    | _ -> ()
  in
    Formula(MethodSelection( sub_expr, instiface, Unresolved id))


let do_qualified_term_id q_id_info = 
  let instiface, id = treat_q_id_info q_id_info 
  in match instiface with
    | NoIface -> BasicExpr(TermVar(Unresolved id))
    | InstIface _ -> BasicExpr(Member( instiface, Unresolved id))
    | CurrentIface -> assert(false)



let do_create_var_scope tv_list =	
  let get_var_name tvar =
    tvar.id_token.token_name
  in
  let get_var_type tvar =
    tvar.id_type
  in
    begin
      List.map (fun tvar -> (get_var_name tvar, get_var_type tvar)) tv_list
    end;;


let do_close_var_scope () = ignore(close_block());;

       (*************************
	*
	* case patterns
	* 
	*)

let do_case_pattern (mem,args) formula =
  let ids = List.map
	      (fun id -> 
		 {id_token = id;
		  id_type = Self;
		  id_origin = CCSL_Var;
		  id_parameters = [];
		  id_variance = Unset;
		  id_sequence = -1;
		 })
	      args
  in
    (Unresolved mem, ids, formula)


    (* actions for and and or
     * 
     * val do_and : ccsl_formulas -> ccsl_formulas -> ccsl_formulas
     * 
     * val do_or : ccsl_formulas -> ccsl_formulas -> ccsl_formulas
     *)

let do_and f1 f2 = 
  let f1',f1loc = match f1 with
    | FormLoc( f, l ) -> f,l
    | _ -> assert(false)
  in 
  let f2',f2loc = match f2 with
    | FormLoc( f, l ) -> f,l
    | _ -> assert(false)
  in 
  let newloc = new_loc f1loc f2loc 
  in
    match f1',f2' with
      | (And fl1), (And fl2) -> FormLoc(And( fl1 @ fl2 ), newloc)
      | (And fl1), ex2 -> FormLoc(And (fl1 @ [f2] ), newloc)
      | ex1, (And fl2) -> FormLoc(And( f1 :: fl2 ), newloc)
      | ex1, ex2 -> FormLoc(And( [f1;f2] ), newloc )


let do_or f1 f2 = 
  let f1',f1loc = match f1 with
    | FormLoc( f, l ) -> f,l
    | _ -> assert(false)
  in 
  let f2',f2loc = match f2 with
    | FormLoc( f, l ) -> f,l
    | _ -> assert(false)
  in 
  let newloc = new_loc f1loc f2loc 
  in
    match f1',f2' with
      | (Or fl1), (Or fl2) -> FormLoc(Or( fl1 @ fl2 ), newloc)
      | (Or fl1), ex2 -> FormLoc(Or (fl1 @ [f2] ), newloc)
      | ex1, (Or fl2) -> FormLoc(Or( f1 :: fl2 ), newloc)
      | ex1, ex2 -> FormLoc(Or( [f1;f2] ), newloc )


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

