(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Tuesday 25 June 02 14:55:52 tews@ithif51>
 *
 * Resolution pass for CCSL
 *
 * $Id: resolution.ml,v 1.15 2002/07/03 12:01:16 tews Exp $
 *
 *)

open Util
open Error
open Global
open Top_variant_types
open Classtypes
open Types_util
open Ccsl_pretty
;;

(***********************************************************************
 *
 * Errors
 *
 *)

exception Resolution_error

let undefined_symbol token = 
    begin
      error_message (remove_option token.loc)
	("Undefined Symbol " ^ token.token_name ^ ".");
      if Global.debug_level _DEBUG_DUMP_SYMBOL_TABLE 
      then 
	begin
	  print_verbose( "Symboltable: " );
	  print_verbose (Symbol.symbol_table_dump 
			   Top_variant_types_util.dump_symbol_brief);
	end;
      raise Resolution_error
    end;;


let undefined_method cl token loc = begin
  error_message loc ("Method " ^ token.token_name ^
		       " not defined in class " ^
		       cl#get_name ^ ".");
  if Global.debug_level _DEBUG_DUMP_SYMBOL_TABLE then begin
    print_verbose( "Symboltable: " );
    print_verbose (Symbol.symbol_table_dump 
		     Top_variant_types_util.dump_symbol_brief);
  end;
  raise Resolution_error
end;;


let undefined_member iface token loc = begin
  error_message loc ("Member " ^ token.token_name ^
		       " not defined in " ^
		       iface#get_name ^ ".");
  if Global.debug_level _DEBUG_DUMP_SYMBOL_TABLE then begin
    print_verbose( "Symboltable: " );
    print_verbose (Symbol.symbol_table_dump 
		     Top_variant_types_util.dump_symbol_brief);
  end;
  raise Resolution_error
end;;


let wrong_id_type token expect found =
    begin
      error_message (remove_option token.loc)
	("Expected a " ^ expect ^ ", but found " ^ 
	 found ^ 
	 (if found = "" then "" else " ") ^ 
	 token.token_name ^ ".");
      raise Resolution_error
  end


let d s = 
  if debug_level _DEBUG_RESOLUTION then
    print_verbose s

let dr s = d( "   res " ^ s)


(***********************************************************************
 ***********************************************************************
 *
 * assertion and creation conditions
 *
 *)

    (* check a method label in a modal operator *)
let check_method_label cl token =
  let mem = (try cl#find_member (token.token_name)
	     with Member_not_found -> 
	       undefined_method cl token (remove_option token.loc)) 
  in
    if mem#is_method then
      Some token
(* Hendrik clean up after variance pass
 * 	 if (mem#get_sort = Nary_Method) || (mem#get_sort = Unary_Method) then
 * 	   Some token
 * 	 else
 * 	   begin
 * 	     warning_message token 
 * 	       ("Ignoring method " ^ token.token_name ^
 * 		" in method list because it " ^
 * 		"does not yield any successor states");
 * 	     None
 * 	   end
 *)
    else
      undefined_method cl token (remove_option token.loc)


let rec do_basic_expr ccl loc = function  
  | TermVar(Resolved _ ) -> assert false
  | TermVar(Unresolved token) ->
      let symb = 
	try Symbol.find token.token_name 
	with Table.Not_defined -> undefined_symbol token
      in (match symb with
					(* a variable is ok *)
	    | VarSymb id_rec -> 
		begin
		  dr ("TermVar " ^ 
		     id_rec.id_token.token_name ^ " : " ^
		     (string_of_ccsl_type id_rec.id_type)
		    );
		  BasicExpr(TermVar(Resolved id_rec))
		end
					   (* members are ok *)
	    | MemberSymb(m) -> 
		let topiface = 
		  if m#is_inherited 
		  then remove_option(m#hosting_class#get_heir)
		  else m#hosting_class 
		in let mfulltype =
			(* make type variables unique, before exporting
			 * this is only neccessary for (co)reduce, 
			 * but does not harm in general
			 *)
		    uniquify_type_variables m#get_full_type
		in let instiface, memtyp = 
		    if ccl#get_name = topiface#get_name 
                    then
		      if ccl#is_sig 
		      then		(* export local typeparameters *)
			export_member_local topiface m mfulltype
		      else		(* no export *)
			(CurrentIface, mfulltype)
                    else		(* export everything *)
		      export_member topiface m mfulltype
		in let _ = dr ("member " ^ 
			       (string_of_ccsl_inst_iface instiface) ^
			       token.token_name ^ " : " ^
			       (string_of_ccsl_type memtyp)
			      )
		in		    
		  TypeAnnotation(
		    BasicExpr(Member(instiface, Resolved m)),
		    memtyp)

	    | CCSL_GroundTypeSymb _
	    | TypevarSymb _
	    | ClassSymb _ 
	    | AdtSymb _ 
	    | SigSymb _ -> 
		wrong_id_type token "expression" "type"

	                    (* no instanciations are stored in CCSL_ID *)
	    | InstClassSymb _ 
	    | InstAdtSymb _ -> assert(false)
	 )

  | Member(InstIface(iface,args,Some instloc) as instiface, Unresolved id) -> 
	(* iface must be different from ccl, because otherwise 
	 * it could not have been resolved in the parser
	 *)
      let m = 
	try iface#find_member id.token_name
	with Member_not_found -> undefined_member iface id 
	    (remove_option id.loc)
      in 
      let typeparams = get_member_parameters m in
      let _ = (if not (check_parameters typeparams args) 
	       then
		 Parser.instantiation_error instloc 
		   (iface#get_name ^ "::" ^ id.token_name)
		   typeparams args
	      ) in
      let _, memtype =
			(* make type variables unique, before exporting
			 * this is only neccessary for (co)reduce, 
			 * but does not harm in general
			 *)
	  export_member_with_args iface m args 
	    (uniquify_type_variables m#get_full_type)
      in let _ = dr ("quali member " ^ 
		    m#get_name ^ " : " ^
		    (string_of_ccsl_type memtype)
		   )
      in
	TypeAnnotation(
	  BasicExpr(Member(instiface, Resolved m)),
	  memtype)

	  (* other Members are not generated in the parser *)
  | Member _
    -> assert(false)

	  (* do one branch of a CCSL_Case *)
and do_variant ccl loc adt tvars (const, args, ex) =
  let recurse_exp = do_expression ccl loc in
  let token = match const with 
    | Unresolved t -> t
    | _ -> assert(false) in
  let member =
    try
      adt#find_local_member token.token_name
    with 
      | Member_not_found ->
	  begin
	    error_message loc
	      ("This is a case expression for values of " ^ 
	       adt#get_name ^ "\n" ^
	       "There is no constructor " ^
	       token.token_name ^
	       " in " ^ adt#get_name ^ ".");
	    raise Resolution_error
	  end
  in let _ = (if not member#is_adt_constructor then 
		wrong_id_type token "adt constructor" "")
  in let _, member_type = export_member_with_args adt member
			    tvars member#get_full_type
  in let arg_types = match member_type with
    | Function(Product tl, _ ) -> tl
    | Function(t , _) -> [t]
    | _ when member#get_sort = Adt_Const_Constructor -> []
    | _ -> assert(false) 
  in let _ = (if (List.length arg_types) <> (List.length args) 
		 then begin
		   error_message (remove_option token.loc)
		     ("Constructor " ^ 
		      member#get_name ^ " expects " ^
		      (string_of_int (List.length arg_types)) ^
		      " arguments.");
		     raise Resolution_error
		 end)
  in 
  let _ = dr ("case const " ^ 
	      member#get_name ^ " : " ^
	      (string_of_ccsl_type member_type)
	     )
  in let local_block = Symbol.start_block() in
  let do_one_arg (id_rec,typ) =
    let _ =				(* check for doubble occurences *)
	   try
	     ignore(Symbol.find_local local_block id_rec.id_token.token_name);
	     error_message (remove_option id_rec.id_token.loc)
	       ("Variables in patterns must be unique.");
	     raise Resolution_error
	   with 
	     | Table.Not_defined -> ()
    in begin
	Symbol.create_var id_rec;
	id_rec.id_type <- typ
      end
  in let _ = List.iter do_one_arg (List.combine args arg_types) in
  let nex = recurse_exp ex 
  in
    ignore(Symbol.close_block());
    (Resolved member, args, nex)

and do_expression ccl loc expr = 
  let recurse_exp = do_expression ccl loc in
  let recurse_form = do_formula ccl loc in
    match expr with
      | ExprLoc(ex,loc) -> ExprLoc(do_expression ccl loc ex, loc)
      | BasicExpr bexpr -> do_basic_expr ccl loc bexpr
      | TypedTerm(Term _, typ) ->
	  dr "typed groundterm";
	  expr	  
      | TypedTerm(ex, typ) -> 
	  dr "TypedTerm";
	  TypedTerm(recurse_exp ex, typ)

	  (* method selection are resolved during type checking *)
      | MethodSelection(ex, typeopt, m) ->
	  dr ("MethodSelection " ^ 
	      (match m with
		 | Unresolved name -> name.token_name
		 | Resolved mem -> mem#get_name
	      ));
	  MethodSelection(recurse_exp ex, typeopt, m)
      | Tuple(ex_list) -> 
	  dr "Tuple";
	  Tuple(List.map recurse_exp ex_list)
      | Projection(i,_) -> 
	  dr "Proj";
	  expr
      | Abstraction(decl_list,ex) ->
	  begin
	    dr "Abstraction";
	    ignore(Symbol.start_block ());
	    Symbol.create_var_list decl_list;
	    let new_ex = recurse_exp ex
	    in
	      ignore(Symbol.close_block ());
	      Abstraction(decl_list, new_ex)
	  end
      | Application(ex1,ex2) -> 
	  dr "Application";
	  Application(recurse_exp ex1, recurse_exp ex2)
      | InfixApplication(ex1, NoIface, Unresolved token, ex2) ->
	  let _ = dr "InfixApplication" in
	  let nex1 = recurse_exp ex1 in
	  let nex2 = recurse_exp ex2 in
	  let symb = 
	    try Symbol.find token.token_name 
	    with Table.Not_defined -> undefined_symbol token
	  in (match symb with
					(* members are ok *)
		| MemberSymb(m) -> 
		    let _ = assert(not m#is_inherited) in
					(* dont need trace down the heir *)
		    let topiface = m#hosting_class in
		    let _ = assert(m#get_sort = InfixGroundTerm) in
		    let _ = assert(topiface#is_sig) in
			(* in the symboltable only special symbols like 
			 * reduce/coreduce have free type variables
			 *)
		    let mfulltype = m#get_full_type in
		    let _ = assert(not (type_is_nonground mfulltype)) in
		    let instiface, memtyp = 
		      if ccl#get_name = topiface#get_name 
                      then	   (* may happen in top level definition *)
					(* export local type parameters *)
			export_member_local topiface m mfulltype
                      else		(* export everything *)
			export_member topiface m mfulltype
		    in let _ = dr ("infix " ^ 
				   (string_of_ccsl_inst_iface instiface)
				  )
		    in
		      TypeAnnotation(
			InfixApplication(nex1, instiface, Resolved m, nex2),
			memtyp)

	         (* infixes are only stored as members of ground signatures *)
		| VarSymb _ 
		| CCSL_GroundTypeSymb _
		| TypevarSymb _
		| ClassSymb _ 
		| AdtSymb _ 
		| SigSymb _ 
		| InstClassSymb _ 
		| InstAdtSymb _ 
		  -> assert(false)
	     )
					(* no other InfixApplication *)
      | InfixApplication _ -> 
	  assert(false)

      | FunUpdate(fex, changes) -> 
	  dr "FunUpdate";
	  FunUpdate( recurse_exp fex, 
		     List.map (fun (ex1,ex2) -> (recurse_exp ex1, 
						 recurse_exp ex2))
		       changes)
      | Let(decl_list, ex) ->
	  let _ = dr "Let" in
					(* Let binds sequentially *)
	  let ndecl_list =
	    List.map (fun (id_rec, typopt, ex) ->
			let nex = recurse_exp ex in
			let typ = match typopt with
			  | Some typ -> typ
			  | None -> 
			      FreeTypeVariable(Type_variable.fresh())
			in let _ = id_rec.id_type <- typ
			in begin
			    ignore(Symbol.start_block());
			    Symbol.create_var id_rec;
			    (id_rec, typopt, nex)
			  end)
	      decl_list
	  in
	  let nex = recurse_exp ex in
	    List.iter (fun _ -> ignore(Symbol.close_block())) decl_list;
	    Let(ndecl_list, nex)
      | If(conds,ex) -> 
	  dr "If";
	  If( List.map (fun (f,ex) -> (recurse_form f, recurse_exp ex)) conds,
	      recurse_exp ex)
      | CCSL_Case(ex,variants) -> 
	  let _ = dr "Case" in
	  let first_id  = match variants with
	    | (Unresolved id,_,_) :: _ -> id
	    | _ -> assert(false) in
	  let symb = 
	    try Symbol.find first_id.token_name 
	    with Table.Not_defined -> undefined_symbol first_id
	  in let adt = match symb with
					(* members might be ok *)
	    | MemberSymb(m) -> 
		if m#is_adt_constructor 
		then m#hosting_class
		else
		  wrong_id_type first_id "adt constructor" ""
					(* other expressions are not ok *)
	    | VarSymb id_rec -> 
		wrong_id_type first_id "adt constructor" "variable"

					   (* types are not ok *)
	    | CCSL_GroundTypeSymb _
	    | TypevarSymb _
	    | ClassSymb _ 
	    | AdtSymb _ 
	    | SigSymb _ -> wrong_id_type first_id "expression" "type"
	                    (* no instanciations are stored in CCSL_ID *)
	    | InstClassSymb _ 
	    | InstAdtSymb _ -> assert(false)
	  in let _ = assert(adt#is_adt) in
	  let tvars = (List.map (fun _ -> TypeArgument(
				   FreeTypeVariable(Type_variable.fresh())))
			 adt#get_parameters)
	  in 
	  let nvariants = List.map (do_variant ccl loc adt tvars) variants in
	  let nex = TypeAnnotation(recurse_exp ex,
				   Adt(adt, Always, tvars))
	  in 
	    CCSL_Case(nex, nvariants)

      | Box(typ,pred,tlist) ->
	  let _ = dr "Box" in
	  let cl = match typ with
	    | Self -> ccl
	    | Class(cl,_) -> cl
	    | _ -> assert(false) 
	  in let ntlist = 
	      List.fold_right 
		(fun t tl -> match check_method_label cl t with 
		   | Some t' -> t' :: tl
		   | None -> tl) 
		tlist [];
	  in
	    Box(typ, recurse_exp pred, ntlist)

      | Diamond(typ,pred,tlist) ->
	  let _ = dr "Diamond" in
	  let cl = match typ with
	    | Self -> ccl
	    | Class(cl,_) -> cl
	    | _ -> assert(false) 
	  in let ntlist = 
	      List.fold_right 
		(fun t tl -> match check_method_label cl t with 
		   | Some t' -> t' :: tl
		   | None -> tl) 
		tlist [];
	  in
	    Diamond(typ, recurse_exp pred, ntlist)

      | Expression form -> 
	  dr "Expression";
	  Expression(recurse_form form)

					(* not allowed in ccsl_input_types *)
      | TypeAnnotation _
      | Term _
      | QualifiedTerm _
      | RecordTuple _
      | RecordSelection _
      | RecordUpdate _
      | Case _
      | List _
(* 	 | Reduce _
 *)
      | Every _
      | RelEvery _
      | Map _
      | SmartAbstraction _
      | SmartApplication _
      | Comment_str _
      | Comment_expr _
      | Comprehension _ ->
	  assert(false)
	
and do_formula ccl loc formula =
  let recurse_exp = do_expression ccl loc in
  let recurse_form = do_formula ccl loc in
    match formula with
      | FormLoc(f,loc) -> FormLoc(do_formula ccl loc f, loc)
      | True -> formula
      | False -> formula
      | Not f -> 
	  dr "Not";
	  Not( recurse_form f)
      | And f_list -> 
	  dr "And";
	  And(List.map recurse_form f_list)
      | Or f_list -> 
	  dr "Or";
	  Or(List.map recurse_form f_list)
      | Implies(assum,concl) -> 
	  dr "Implies";
	  Implies(recurse_form assum, recurse_form concl)
      | Iff(assum,concl) -> 
	  dr "Iff";
	  Iff(recurse_form assum, recurse_form concl)
      | Equal(ex_a,ex_b) -> 
	  dr "Equal";
	  Equal(recurse_exp ex_a, recurse_exp ex_b)
      | Forall(quant_list, f) -> 
	  begin
	    dr "Forall";
	    ignore(Symbol.start_block ());
	    Symbol.create_var_list quant_list;
	    let new_f = recurse_form f 
	    in
	      ignore(Symbol.close_block ());
	      Forall(quant_list, new_f)
	  end
      | Exists(quant_list, f) -> 
	  begin
	    dr "Exists";
	    ignore(Symbol.start_block ());
	    Symbol.create_var_list quant_list;
	    let new_f = recurse_form f 
	    in
	      ignore(Symbol.close_block ());
	      Exists(quant_list, new_f)
	  end
      | Formula ex -> 
	  dr "Formula";
	  Formula( recurse_exp ex )
      | Obseq(t,ex1,ex2) -> 
	  dr "Obseq";
	  Obseq(t, recurse_exp ex1, recurse_exp ex2)
	  
					(* not in ccsl_input_formulas *)
      | MetaImplies _
      | MetaForall _
      | Bisim _
      | ConstantPredicate _
(* 	 | LessOrEqual _
 *)
	-> assert(false)



let do_assertion ccl assertion = 
  let _ = d ("  * Resolve assertion " ^ assertion.assertion_name.token_name) 
  in
    if not assertion.is_generated
    then 				(* user assertion *)
      match assertion.assertion_formula with
	| Symbolic (FormLoc(formula,loc)) -> 
	    begin
	      ignore(Symbol.start_block());
	      (match assertion.self_variable with
		 | None -> ()
		 | Some id_rec -> Symbol.create_var id_rec
	      );
	      List.iter Symbol.create_var assertion.free_variables;
	      let new_form = do_formula ccl loc formula 
	      in
		ignore(Symbol.close_block());
		assertion.assertion_formula <- Symbolic(FormLoc(new_form,loc))
	    end
	  
	| Symbolic _ -> assert(false)
	| Pvs_String _
	| Isa_String _ -> ()


(* Process a definition in a ground signature or in a class
 * 
 * From the parser the definition comes in as a formula, which 
 * should better be an equation.
 * From the left hand side of the equation I extract variable declarations, 
 * by matching these variables against the declared type.
 * After that the left hand side is thrown away, the extracted variables 
 * are stored in the definition record and the right hand side is wrapped
 * in a TypeAnnotation to remember its supposed type
 *)

let do_symbolic_definition iface def =
  let def_eq = match def.definition with
    | Symbolic eq -> eq
    | _ -> assert(false)
  in let mem = def.defined_method in
				(* create the block for this equation *)
  let local_block = Symbol.start_block() in
  let _ = d ("  * Resolve definition " ^ def.defined_method#get_name) in

					(* create local type parameters *)
  let _ = List.iter (function TypeParameter id_rec ->
		       Symbol.create_type_parameter id_rec) 
	    mem#get_local_parameters
  in					(* extract left and right hand sides *)
  let left, right, right_loc = 
    match def_eq with
      | FormLoc(Equal(left, ExprLoc(right,right_loc)), _)
	  -> left,right,right_loc
      | _ -> 
	  begin
	    error_message (get_form_loc def_eq)
	      "Definition not in form of an equation.";
	    raise Resolution_error
	  end
		(* recursively extract variables from the left hand side 
		 * take care to handle cases like
		 *   f = ...
		 *   f x y = ...
		 *   f (x,y) (a,b) = ...
		 *   (x + y) (a,b) = ...
		 *)
  in let rec get_args accu = function
					(* base case normal application *)
    | ExprLoc(Application(
		ExprLoc(BasicExpr(TermVar(Unresolved defname)),_),
		args),_) 
	when defname.token_name = mem#get_name ->
	args :: accu
					(* base case def =  *)
    | ExprLoc(BasicExpr(TermVar(Unresolved defname)),_)
	when defname.token_name = mem#get_name 
	  ->
	accu
					(* base case with infixes *)
    | ExprLoc(InfixApplication(leftex, _, Unresolved op, rightex), loc) 
	when op.token_name = mem#get_name ->
	(match mem#get_full_type with
	   | Function(Product [_;_], _) -> 
	       ExprLoc(Tuple([leftex;rightex]), loc) :: accu
	   | Function(_, Function(_)) -> leftex :: rightex :: accu
	   | _ -> assert false
	)
					(* non base case *)
    | ExprLoc(Application(innerterm, args),_) ->
	get_args (args :: accu) innerterm
    | _ ->				(* error no application *)
	begin
	  error_message (get_ex_loc left)
	    ("Wrong form of definition. Expected \"" ^ 
	     (if mem#get_sort = InfixGroundTerm 
	      then
		"x " ^ mem#get_name ^ " y\""
	      else
		mem#get_name ^ " ... =\"")
	    );
	  raise Resolution_error
	end
  in let arg_list_list = get_args [] left in
			(* pair one argument and one type in a id record *)
  let make_arg_id typ = function
    | ExprLoc(BasicExpr(TermVar(Unresolved tok)), vloc) ->
	let _ =				(* check for doubble occurences *)
	  try
	    ignore(Symbol.find_local local_block tok.token_name);
	    error_message vloc
	      ("Variables on the left hand side of a defining " ^
	       "equation must be unique.");
	    raise Resolution_error
	  with 
	    | Table.Not_defined -> ()
	in 
	let id_rec = Symbol.identifier_record tok CCSL_Var
	in
	  id_rec.id_type <- typ;
	  Symbol.create_var id_rec;
	  id_rec
    | v ->
	begin
	  error_message (get_ex_loc v) "Variable expected.";
	  raise Resolution_error
	end
  in
				(* recursively match arguments and types *)
  let rec match_args argslist funtype = 
    let recurse restargs codomain ids =
      let rest_ids, typ = match_args restargs codomain 
      in
	((ids :: rest_ids), typ)
    in
      match argslist,funtype with
	| ExprLoc(Tuple arggroup, loc ) :: restargs, 
	  Function(Product grouptyp, codomain)
	  -> 
	    if List.length grouptyp <> List.length arggroup 
	    then
	      begin
		error_message loc
		  "Number of arguments does not match type declaration.";
		raise Resolution_error
	      end
	    else
	      recurse restargs codomain
		(List.map2 make_arg_id grouptyp arggroup)
	| (ExprLoc _ as singlearg) :: restargs, 
	  Function(typ, codomain) ->
	    recurse restargs codomain [make_arg_id typ singlearg]
	| _ , Groundtype(idrec, args) when is_type_def idrec ->
	    (* do not use recurse here, 
	     * because we ded not produce any binding
	     *)
	    match_args argslist (expand_type_def eq_ccsl_types funtype)

	| [], resttyp -> [], resttyp
	| ExprLoc(_, loc) :: _, _ ->
	    error_message loc "Superfluous variable.";
	    raise Resolution_error
	| _ -> let x = 2 + 2 in
	    assert false

  in let id_list_list, righttyp = match_args arg_list_list mem#get_full_type

  in let nright = do_expression iface right_loc right
  in 
    def.definition <- 
    Symbolic(FormLoc(Formula( TypeAnnotation(nright, righttyp)),
		     right_loc));
    def.variables <- id_list_list;
    ignore(Symbol.close_block())
      
	     
let do_definition iface def =
  match def.definition with
    | Symbolic _ -> do_symbolic_definition iface def
    | _ -> ()






(***********************************************************************
 ***********************************************************************
 *
 * top level units
 *
 *)

let do_class ccl = 
  let nest = Symbol.nesting_size() 
  in begin
      d (" ** Resolve class " ^ ccl#get_name);
      (* d (ccl#dump_iface); *)
      ignore(Symbol.start_block());
      (* print_verbose( "Symboltable: " );
       * print_verbose (Symbol.symbol_table_dump 
       * 			  Top_variant_types_util.dump_symbol_brief);
       *)

      List.iter (function TypeParameter id_rec ->
		   Symbol.create_type_parameter id_rec) ccl#get_parameters;
      List.iter (fun acl -> List.iter Symbol.create_member
		     acl#get_all_actions)
	ccl#get_resolved_ancestors;
      List.iter Symbol.create_member ccl#get_sig_actions;

      List.iter (fun def -> 
		   do_definition ccl def;
		   Symbol.create_member def.defined_method
		)
	ccl#get_definitions;
      assert(Symbol.nesting_size() = nest + 1);

         (* make class specials available in assertions *)
      List.iter (fun mem -> (if mem#get_sort = Class_Special
			     then
			       Symbol.create_member mem)
		)
	ccl#get_members;

      (* print_verbose( "Symboltable: " );
       * print_verbose (Symbol.symbol_table_dump 
       * 			  Top_variant_types_util.dump_symbol_brief);
       *)

      List.iter (fun ass -> do_assertion ccl ass) ccl#get_assertions;
      assert(Symbol.nesting_size() = nest + 1);
      List.iter (fun mem -> Symbol.create_member mem)
	ccl#get_constructors;
      List.iter (fun crea -> do_assertion ccl crea) 
	ccl#get_creations; 
      List.iter (fun thm -> do_assertion ccl thm) 
	ccl#get_theorems; 

      (* print_verbose( "Symboltable: " );
       * print_verbose (Symbol.symbol_table_dump 
       * 			  Top_variant_types_util.dump_symbol_brief);
       *)

      assert(Symbol.nesting_size() = nest + 1);
      ignore(Symbol.close_block());
      Symbol.create_class_content ccl
    end


let do_sig si =
  let nest = Symbol.nesting_size() in
			(* pair the definitions with the declarations *)
  let rec pair_consts members definitions =
    match members,definitions with
      | [],[] -> []
      | m::ms,[] ->  (m,None) :: (pair_consts ms definitions)
      | [], d::_ -> assert(false)
      | m::ms, d::ds ->
	  if m == d.defined_method 
	  then
	    (m, Some d) :: (pair_consts ms ds)
	  else
	    (m, None) :: (pair_consts ms definitions)
  in
  let const_list = pair_consts si#get_members si#get_definitions
  in
    begin
      print_verbose ("Resolve signature in " ^ si#get_name );
      ignore(Symbol.start_block());
      List.iter (function TypeParameter id_rec ->
		   Symbol.create_type_parameter id_rec) si#get_parameters;

      List.iter (fun (m,defopt) -> 
		   (match defopt with
		      | None -> ()
		      | Some def -> do_definition si def
		   );

		   Symbol.create_member m
		)
	const_list;

      assert(Symbol.nesting_size() = nest + 1);
      ignore(Symbol.close_block());
      Symbol.create_sig_content si
    end


let do_adt adt =
  begin
    print_verbose ("Resolve adt in " 
		   ^ adt#get_name );
    Symbol.create_adt_content adt
  end


let resolve_ast = function
  | CCSL_class_dec cl -> do_class cl
  | CCSL_adt_dec adt -> do_adt adt
  | CCSL_sig_dec si -> do_sig si


let ccsl_resolution_pass (ast: Classtypes.ccsl_ast list) = 
  let nest = Symbol.nesting_size() in
    List.iter resolve_ast ast;
    assert(Symbol.nesting_size() = nest)



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

