(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 26.2.99 by Joachim
 *
 * Time-stamp: <Monday 11 March 02 11:47:20 tews@ithif51>
 *
 * Isabelle pretty printer
 *
 * $Id: isabelle_pretty.ml,v 1.19 2002/05/03 15:01:17 tews Exp $
 *
 *)



open Util;;
open Formatter;;
open Top_names;;
open Top_variant_types;;
open Top_variant_types_util;;
open Pretty_util;;


let comment_prefix = "(* ";;
let comment_infix = " * ";;
let comment_postfix = " *)";;


(***********************************************************************
 ***********************************************************************
 *
 * Signature implementation
 *
 *)



    (*******************************************************************
     *
     * section comment
     *)

let rec isa_pp_top_type 
  = function
    | Groundtype(id,args) ->
	begin
          open_box 0;
	  if args <> [] then
	    begin
              isa_pp_arg_list args;
              print_space()
	    end;
          print_string id.id_token.token_name;
          close_box()
        end
    | TypeConstant (name, flag, args) -> 
	begin
          open_box 0;
	  if (flag <> Never) && args <> [] then
	    begin
              isa_pp_arg_list args;
              print_space()
	    end;
	  if name = "char" (* fixme: should be done with java_type_wrapper *)
	  then print_string "char_java"
          else print_string name;
          close_box()
        end
    | BoundTypeVariable id -> 
	print_string ("'" ^ id.id_token.token_name)
    | Self -> 
	print_string ("'" ^ name_of_self ())
    | Bool ->
	print_string "bool"
    | Carrier ->
      (* if you want to print a Carrier in Isabelle, you need the 
       * instanciation of the class, for this you have to do 
       * get_parameters on adt, for this the java classes have to 
       * support get_parameters, otherwise, you cannot use them with
       * this pretty printer. 
       * The solution is to use substitution before and change the Carrier 
       * into some Adt(adt, flag, args).
       *)

	assert(false)
    | Function(dom, codom) ->
	begin
	  open_box 0;
	  open_box 0;
          print_string "(";
	  isa_pp_top_type dom; 
	  print_space(); 
	  print_string "=> ";
	  isa_pp_top_type codom;
          print_string ")";
	  close_box(); 
	  close_box()
	end	
    | SmartFunction(doml,codom) ->
	(match doml with
	   | [] -> assert(false)
	   | tl -> isa_pp_top_type
		 (List.fold_right (fun t res -> Function(t, res))
		    tl codom))
    | Product typelist ->  
	begin
	  open_box 0;
	  print_string "("; 
	  pp_list isa_pp_top_type "" "" "" "*" typelist;
	  print_string ")";
	  close_box()
	end
    | Record string_type_list -> 
	(match string_type_list with    
	     [] -> 
	       assert false;
	   | l ->  
	       begin
		 open_box 0;
                 print_string "(| ";
		 open_box 0;
		 pp_list
		   (fun (str, typ)
		      ->
			begin
			  force_newline();
			  print_string (str ^ " ::");
			  print_space();
			  isa_pp_top_type typ
			end) "" "" "" "," l;
		 close_box();
		 force_newline();
                 print_string "|)";
		 close_box()
	       end)      	
    | Class(cl, arg_list) ->
	let tname = 
	  if cl#has_feature FinalSemanticsFeature
	  then name_of_final_type cl
	  else name_of_loose_type cl
	in
	   begin                
	     open_box 0;	    
	     if                    (* this probably is not correct *)
	       arg_list <> [] 
	     then isa_pp_arg_list arg_list; 
	     close_box();
	     print_string tname;
	   end
    | Adt (adt, flag, arg_list) ->
	   begin                
	     open_box 0;	    
	     if
	       (arg_list <> []) && (flag <> Never)
	     then (isa_pp_arg_list arg_list; 
		   print_space()
		  );
	     close_box();
	     print_string (name_of_type_of_adt adt);
	   end

    | IFace(cl, flag, arg_list) ->    (* print the fuctor *)
   	begin
	  open_box 0;
	  if (arg_list <> [] ) && (flag <> Never) 
	  then isa_pp_arg_list arg_list;   
	  print_space();
	  print_string (name_of_method_functor cl);
	  close_box()
   	end
    | Array (_, typ, dim) ->
	begin
	  open_box 0;          
	  print_string (match typ with
			  | TypeConstant( s, _, _) ->
			      name_of_array_of_primitive s (string_of_int dim)
			  | Class(c, _) ->
			      name_of_array_of_class c (string_of_int dim)
			  | _ -> assert false
		       );	
	  close_box()
	end

    | Predtype form -> (* does not exist in Isabelle *)
	begin
	  open_box 0;
	  print_string "("; 
	  isa_pp_top_formula form; 
	  print_string ")";
	  close_box()
	end			
					(* not allowed *)
    | FreeTypeVariable _ -> assert(false)

and isa_pp_top_basic_expression = function
    | TermVar idc ->
	print_string (resolution_of idc).id_token.token_name
    | Member(ex, m) ->
			(* Members are treted in the prepretty printer *)
	assert false


and isa_pp_top_expression 
  = function
    | ExprLoc(ex, _ ) -> isa_pp_top_expression ex
    | BasicExpr bexp -> isa_pp_top_basic_expression bexp
    | Term (name, flag, args) ->  (* To do *)
      	begin
	  open_box 0;
	  (* remember to print args only, if flag <> Never *)
	  print_string name;
(*           print_space(); *)
	  close_box();
      	end
    | TypedTerm(ex, typ) ->  
	begin
	  open_box 0;
	  print_string "(";
	  isa_pp_top_expression ex;
	  print_string " ::";
	  print_space();
	  isa_pp_top_type typ;
	  print_string ")";
	  close_box();
	end
    | TypeAnnotation(ex, _) ->  
	  isa_pp_top_expression ex;

    | QualifiedTerm (theory, flag, args, name) ->  (* To do *)
	begin
	  open_box 0;
	  print_string theory;
	  print_string ".";
	  (* remember to print arguments only of flag <> Never *)
	  print_string name;
(*           print_space(); *)
	  close_box()
	end

    | Tuple expr_list ->
	(match expr_list with
	   | [] ->
	       ()
	   | l ->
	       begin
		 open_box 0;
		 print_string "(";
		 open_box 0;
		 (pp_list isa_pp_top_expression "" "" "" "," l);
		 close_box();
		 print_string ")";
		 close_box()
	       end)

    | RecordTuple str_expr_list ->
	(match str_expr_list with
	   | [] -> 
	       assert false
	   | l ->  
	       begin
		 open_box 0;
		 print_string "(|";
		 open_box 0;
		 pp_list
		   (fun (str, expr) ->
		      begin
			force_newline();
			print_string (str ^ " =");
			print_space();
			isa_pp_top_expression expr;
		      end) "" "" "" "," l;
		 close_box();
		 force_newline();
		 print_string "|)";
		 close_box()
	       end)
    | RecordSelection (str, expr) ->
	begin
	  open_box 0;
	  print_string str; 
	  print_break 0 2; 
	  open_box 0;
	  print_string "(";
	  isa_pp_top_expression expr; 
	  print_string ")";
	  close_box();
	  close_box()
	end
    | RecordUpdate (record, str_expr_list) ->
	begin
	  open_box 0;
	  isa_pp_top_expression record; 
	  if
	    str_expr_list <> []
	  then
	    begin
	      print_space();
	      open_box 0;
	      print_string "(| "; 
	      print_break 1 1;
	      pp_list (fun (str, expr) ->
			 begin
			   open_box 0;
			   print_string( str ^ " :=");
			   print_space();
			   isa_pp_top_expression expr;
			   close_box();
			 end) "" "" "" "," str_expr_list;
	      print_string " |)";
	      close_box(); 
	    end;
	    close_box();
	end
    | List (expr_list) ->
	begin
	  open_box 0;
	  print_string "[ ";
	  open_box 0;
	  (pp_list isa_pp_top_expression "" "" "" "," expr_list);
	  close_box();
	  print_string " ]";
	  close_box()
	end
    | Abstraction (str_typ_list, expr) ->
	(match str_typ_list with
	   | [] ->
	       isa_pp_top_expression expr
	   | l ->
	       begin
		 open_box 0;
		 print_string "(";
		 print_string ("% (");
		 pp_list isa_pp_var "" "" "" "," str_typ_list;
		 print_string ") . ";
		 print_break 1 2;
		 open_box 0;
		 isa_pp_top_expression expr;
		 close_box ();
		 print_string ")";
		 close_box ()
	       end)
    | SmartAbstraction (str_typ_list, expr) ->
	(match str_typ_list with
	   | [] ->
	       isa_pp_top_expression expr
	   | l ->
	       begin
		 open_box 0;
		 print_string "(";
		 print_string ("% ");
		 pp_list isa_pp_var "" "" "" "" str_typ_list;
		 print_string " . ";
		 print_break 1 2;
		 open_box 0;
		 isa_pp_top_expression expr;
		 close_box ();
		 print_string ")";
		 close_box ()
	       end)

	(* projections are always applied (because of stupid PVS) *)
    | Application(Projection (ith,n), expr) ->
	begin
	  assert(0 < ith && ith <= n);
          open_box 0;			(* 1 *)
          if ith < n then
	    begin
	      print_string "fst( ";
              print_break 0 2; 
	      open_box 0		(* opt *)
	    end;
          for j = 2 to ith do
            begin
	      open_box 0;		(* ith *)
              print_string "snd( ";
              print_break 0 2
            end
	  done; 
          open_box 0;			(* ith+1 *)
          isa_pp_top_expression expr;
          close_box ();			(* >ith *)
          for j = 2 to ith do
            begin
              print_break 0 1;
              print_string ")";
              close_box ()		(* >1 *)
            end
          done;
	  if ith < n then
	    begin
              print_break 0 1;
              print_string ")";
              close_box ()		(* >opt *)
	    end;
	  close_box ();			(* >0 *)
    	end

    | Application (func, arg) ->
         begin
	   open_box 0;
	   isa_pp_top_expression func; 
	   print_break 0 2;
           print_space(); 
	   open_box 0;
           print_string "(";
	   isa_pp_top_expression arg; 
	   print_string ")";
	   close_box();
	   close_box()
	 end

    | InfixApplication (expr1,instiface,mem,expr2) ->
	let mem_name = (resolution_of mem)#get_name 
	in
	  begin
	    open_box 0;			(* 1 *)
	    print_string "(";
	    isa_pp_top_expression expr1; 
	    print_break 1 2;
	    print_string mem_name;
	    print_break 1 2;
	    isa_pp_top_expression expr2; 
	    print_string ")";
	    close_box();		(* 0 *)
	  end

    | SmartApplication(func, arg_list) ->
	(match arg_list with
	   | [] -> assert(false)
	   | tl -> isa_pp_top_expression 
		 (List.fold_left (fun res t -> Application(res, t))
		    func tl))
    | FunUpdate (func, expr_expr_list) ->
	begin
	  open_box 0;
	  isa_pp_top_expression func; 
	  print_space();
	  open_box 0;
	  print_string "("; 
	  print_break 1 1;
	  pp_list (fun (place, expr) ->
		     begin
		       open_box 0;
		       isa_pp_top_expression place;
		       print_string " :=";
		       print_space();
		       isa_pp_top_expression expr;
		       close_box();
		     end) "" "" "" "," expr_expr_list;
	  print_string ")";
	  close_box(); 
	  close_box();
	end
    | Let (id_typ_expr_list, expr) ->
	begin
	  open_box 0; 
	  print_string "(";
	  if 
	    id_typ_expr_list <> [] 
	  then
	    begin
	      print_string "let"; 
	      print_break 1 2;
	      pp_list
		(fun (id, typ_opt, expr) ->
		   begin
		     print_string id.id_token.token_name;
		     (match typ_opt with (* Necessary for Isabelle?? *)
			| None ->
			    ()
			| Some typ ->
			    begin
			      print_string " :: ";
			      isa_pp_top_type typ
			    end);
		     print_string " =";
		     print_space();
		     print_break 1 2;
		     isa_pp_top_expression expr;
		   end) "" "" "" ";" id_typ_expr_list;
	      print_space(); 
	      print_string "in";
	      print_break 1 2;
	    end;
	  open_box 0;
	  isa_pp_top_expression expr;
	  close_box();
	  print_string ")"; 
	  close_box()
	end
    | If (form_expr_list, expr) ->
	begin
	  if 
	    form_expr_list <> [] 
	  then
	    begin
	      open_box 0;
	      print_string " (if"; 
	      print_space();
	      pp_list (fun (cond, expr) ->
			 print_if_newline(); 
			 print_space();
			 isa_pp_top_formula cond; 
			 print_space();
			 open_box 0;
			 print_string " then"; 
			 print_break 1 2;
			 isa_pp_top_expression expr;
			 close_box(); 
			 print_space();
			 open_box 0;
		      ) "" "" "" "else if" form_expr_list;
	      print_string " else"; 
	      print_break 1 2;
	    end;
	  isa_pp_top_expression expr; 
	  print_space();
	  if 
	    form_expr_list <> [] 
	  then 
	    begin
	      for i = 1 to List.length form_expr_list do
		close_box()
	      done; 
	      print_string " )"; 
	      print_space();
	    end;
	  close_box();
	end
    | Case (expr, expr_expr_list) ->
	(match expr_expr_list with
	   | [] -> 
	       assert false
	   | (tag1, expr1) :: cases ->
 	       begin
		 open_box 0;
		 print_string "(case "; 
		 print_break 1 2;
		 open_box 0; 
		 isa_pp_top_expression expr;
		 close_box ();
		 print_space (); 
		 print_string "of"; 
		 print_break 1 2;
		 open_box 0;
		 isa_pp_top_expression tag1;
		 print_string " =>"; 
		 print_break 1 2;
		 open_box 0; 
                 print_space();
		 isa_pp_top_expression expr1; 
		 close_box ();
                 force_newline();
		 List.iter
		   (fun (tag', expr') ->
		      begin
		   	print_string "|"; 
			print_space ();
		   	isa_pp_top_expression tag';
			print_string "=>";
		   	print_break 1 2;
		   	open_box 0; 
			isa_pp_top_expression expr';
			close_box ();
                        force_newline();
		      end) cases;
		 close_box ();
		 print_space ();
		 print_string ")"; 
		 print_space();
		 close_box()
	       end)
    | CCSL_Case (expr, match_list) ->
	(match match_list with
	   | [] -> 
	       assert false
	   | (mem1,var_list1, expr1) :: cases ->
 	       begin
		 open_box 0;				    (* 1 *)
		 print_string "(case "; 
		 print_break 1 2;
		 open_box 0;				    (* 2 *)
		 isa_pp_top_expression expr;
		 close_box ();				    (* 2 *)
		 print_space (); 
		 print_string "of"; 
		 print_break 1 2;
		 open_box 0;				    (* 2a *)
		 print_string (resolution_of mem1)#get_name;
		 if var_list1 <> [] then
		   begin
		     print_space (); 
		     pp_list 
		       (fun id -> print_string id.id_token.token_name ) 
		       "" "" "" " " var_list1;
		       print_space ()
		   end;
		 print_string " =>"; 
		 print_break 1 2;
		 open_box 0;				    (* 3 *)
                 print_space();
		 isa_pp_top_expression expr1; 
		 close_box ();				    (* 3 *)
                 force_newline();
		 List.iter
		   (fun (mem',var_list', expr') ->
		      begin
		   	print_string "|"; 
			print_space ();
		 	print_string (resolution_of mem')#get_name;
		 	if var_list' <> [] then
			  begin
			    print_space (); 
			    pp_list 
			      (fun id -> print_string id.id_token.token_name ) 
			      "" "" "" " " var_list';
			    print_space ()
			  end;
			print_string "=>";
		   	print_break 1 2;
		   	open_box 0;			    (* 3a *)
			isa_pp_top_expression expr';
			close_box ();			    (* 3a *)
                        force_newline();
		      end) cases;
		 close_box ();				    (* 2a *)
		 print_space ();
		 print_string ")"; 
		 print_space();
		 close_box()				    (* 1 *)
	       end)
(*     | Reduce(adt, flag, arg_list) -> (* To do *)
 * 	      begin
 * 		open_box 0;				    (* 1 *)
 * 		print_string (name_of_adt_reduce adt);
 * 		if (arg_list <> []) && (flag <> Never)
 * 		then
 * 		  begin
 * 		    print_string "[";
 * 		    isa_pp_arg_list arg_list;
 * 		    print_string "]";
 * 		    print_break 0 2
 * 		  end;
 * 		close_box()				    (* 1 *)
 * 	      end
 *)
    | Every (typ, expr_list) ->
	let every_term = 
			(* every is only defined for three kinds of types *)
	  (match typ with
	     | Class( cl,_ ) -> 
		 let cl_name = 
		   if cl#has_feature FinalSemanticsFeature 
		   then name_of_final_coalgebra cl
		   else name_of_loose_coalgebra cl
		 in
		   Application(
		     Term(name_of_class_every cl, Always, []),
		     Term(cl_name,Always,[]))
	     | Adt( adt,_,_ ) -> 
		 Term(name_of_adt_every adt,Always,[])
	     | Groundtype(id,_) -> 
		 Term(name_of_const_every id.id_token.token_name,
		      Always,[])
	     | _ -> assert(false)
	  )
	in
	  isa_pp_top_expression
	    (if expr_list = [] 
	     then every_term
	     else 
	       SmartApplication(
		 every_term, 
		 expr_list))

    | RelEvery (typ, expr_list) ->
	let relevery_term = 
			(* every is only defined for three kinds of types *)
	  (match typ with
	     | Class( cl,_ ) -> 
		 let cl_name = 
		   if cl#has_feature FinalSemanticsFeature 
		   then name_of_final_coalgebra cl
		   else name_of_loose_coalgebra cl
		 in
		   SmartApplication(
		     Term(name_of_class_rel_every cl, Always, []),
		     [Term(cl_name,Always,[]);Term(cl_name,Always,[])]
		   )
	     | Adt( adt,_,_ ) -> 
		 Term(name_of_adt_rel_every adt,Always,[])
	     | Groundtype(id,_) -> 
		 Term(name_of_const_rel_every id.id_token.token_name,
		      Always,[])
	     | _ -> assert(false)
	  )
	in
	  isa_pp_top_expression
	    (if expr_list = [] 
	     then relevery_term
	     else 
	       SmartApplication(
		 relevery_term, 
		 expr_list))

    | Map (typ, expr_list) ->
	let map_term = 
			(* every is only defined for three kinds of types *)
	  (match typ with
	     | Class( cl,_ ) -> 
		 Term(name_of_class_map cl, Always, [])
	     | Adt( adt,_,_ ) -> 
		 Term(name_of_adt_map adt,Always,[])
	     | Groundtype(id,_) -> 
		 Term(name_of_const_map id.id_token.token_name,
		      Always,[])
	     | _ -> assert(false)
	  )
	in
	  isa_pp_top_expression
	    (if expr_list = [] 
	     then map_term
	     else 
	       SmartApplication(
		 map_term, 
		 expr_list))

    | Expression form ->
	isa_pp_top_formula form
    | Comment_str (str, expr) ->
	begin
	  force_newline ();
	  List.iter
	    (fun s ->
	       begin
		 print_string "(* ";
		 print_string s;
                 print_string " *)"; 
		 force_newline()
	       end)
	    (split_on_newline str);
	  isa_pp_top_expression expr;
	  force_newline()
	end
    | Comment_expr (expr1, typ, expr2) ->
	isa_pp_top_expression
	  (Comment_str(
	     (stringwrapper isa_pp_top_expression expr1)^
	     " : " ^ 
	     (stringwrapper isa_pp_top_type typ),
	     expr2))
    | Comprehension (str, typ, form) -> (* To do *)
	begin
	  open_box 0;
	  print_string "{ ";
	  isa_pp_var (str, typ);
	  print_string " |";
	  print_break 1 2;
	  open_box 0;
	  isa_pp_top_formula form;
	  close_box ();
	  print_string "}";
	  close_box ()
	end  
					(* washed away in a pre-pass! *)
    | Projection _ 
    | MethodSelection _
    | Box _
    | Diamond _ ->
      	assert(false)
	

	
and isa_pp_top_formula (* : top_formulas -> unit *)
  = function
    | FormLoc(f, _) -> isa_pp_top_formula f
    | True ->
      	print_string "True"
    | False ->
	print_string "False"
    | Not form ->
	begin
	  open_box 0;
	  print_string "~ (";
	  isa_pp_top_formula form;
	  print_string ")";
	  close_box()
	end
    | And form_list ->
	begin
	  print_string "(";
	  open_box 0; 
	  pp_list isa_pp_top_formula "(" ")" "True" "&" form_list; 
	  close_box ();
	  print_string ")";
	end
    | Or form_list ->
	begin
	  print_string "(";
	  open_box 0; 
	  pp_list isa_pp_top_formula "(" ")" "False" "|" form_list; 
	  close_box ();
	  print_string ")";
	end
    | Implies (prem_expr, concl_expr) ->
	begin
	  open_box 0;
	  print_string "("; 
	  isa_pp_top_formula prem_expr;
	  print_string ")";
	  print_space ();
	  open_box 0;
	  print_string "-->";
	  print_space();
	  print_string "("; 
	  isa_pp_top_formula concl_expr;
	  print_string ")";
	  close_box(); 
	  close_box()
	end
    | Iff (p1, p2) ->
	begin
	  open_box 0;
	  print_string "(";
	  isa_pp_top_formula p1;
	  print_string ")";
	  print_space ();
	  open_box 0;
	  print_string "=";
	  print_space();
	  print_string "("; 
	  isa_pp_top_formula p2;
	  print_string ")";
	  close_box(); 
	  close_box()
	end	  
    | Equal (left_expr, right_expr) ->
	begin
	  open_box 0;
	  isa_pp_top_expression left_expr;
	  print_break 1 2; 
	  print_string "=";
	  print_space(); 
	  isa_pp_top_expression right_expr;
	  close_box ()
	end
(*     | LessOrEqual (left_expr, right_expr) ->
 * 	   begin
 * 	     open_box 0;
 * 	     isa_pp_top_expression left_expr;
 * 	     print_break 1 2; 
 * 	     print_string "<=";
 * 	     print_space(); 
 * 	     isa_pp_top_expression right_expr;
 * 	     close_box ()
 * 	   end
 *)
    | Forall (str_typ_list, form) ->
	(match str_typ_list with
	   | [] ->
	       isa_pp_top_formula form
	   | l ->
	       begin
		 open_box 0;
	    	 print_string "ALL ";
	    	 open_box 0; 
		 pp_list isa_pp_var "" "" "" "" str_typ_list; 
		 close_box();
	    	 print_string " . ";
	    	 print_break 1 2;
		 isa_pp_top_formula form;
		 close_box ()
	       end)
    | Exists (str_typ_list, form) ->
	(match str_typ_list with
	   | [] ->
	       isa_pp_top_formula form
	   | l ->
	       begin
		 open_box 0;
	    	 print_string "EX ";
	    	 open_box 0; 
		 pp_list isa_pp_var "" "" "" "" str_typ_list; 
		 close_box();
	    	 print_string " . ";
	    	 print_break 1 2;
		 isa_pp_top_formula form;
		 close_box ()
	       end)
    | ConstantPredicate str ->
	print_string str
    | Formula expr ->
	isa_pp_top_expression expr
    | Bisim (typ, ex1, ex2) ->		
	let bisim_term = 
			(* every is only defined for three kinds of types *)
	  (match typ with
	     | Class( cl,_ ) -> 
		 let cl_name = 
		   if cl#has_feature FinalSemanticsFeature 
		   then name_of_final_coalgebra cl
		   else name_of_loose_coalgebra cl
		 in
		   SmartApplication(
		     Term(name_of_greatest_bisim cl, Always, []),
		     [Term(cl_name,Always,[]);Term(cl_name,Always,[])]
		   )
	     | _ -> assert(false)
	  )
	in
	  isa_pp_top_expression
	    (Application(
	       bisim_term, 
	       Tuple([ex1;ex2])))	      

    | MetaImplies (prem_expr, concl_expr) ->
	begin
	  open_box 0;
	  print_string "("; 
	  isa_pp_top_formula prem_expr;
	  print_string ")";
	  print_space ();
	  open_box 0;
	  print_string "==>";
	  print_space();
	  print_string "("; 
	  isa_pp_top_formula concl_expr;
	  print_string ")";
	  close_box(); 
	  close_box()
	end
    | MetaForall (str_typ_list, form) ->
	(match str_typ_list with
	   | [] ->
	       isa_pp_top_formula form
	   | l ->
	       begin
		 isa_pp_top_formula form;
	       end) 
    | Obseq _ ->
	assert false



and isa_pp_top_theory_declaration (* : top_theory_declaration -> unit *)
  = function
    | Lemma (name, form) ->   (* hack: lemma's tijdelijk als axioms *)
	begin                  (* als lemma moet ook bewijs er direct bij *)
	  open_box 2;          (* commentaar: provable *)
          print_string "rules  (* provable! *)";
          force_newline ();
	  print_string (name);
	  print_space (); 
          print_string "\"";
	  isa_pp_top_formula form;
          print_string "\"";
	  close_box()
	end
    | Axiom (name, form) ->
	begin
	  open_box 2;          
          print_string "rules";
          force_newline ();
	  print_string (name);
	  print_space (); 
          print_string "\"";
	  isa_pp_top_formula form;
          print_string "\"";
	  close_box()
	end
    | Vardecl (name, typ) -> print_string "" (* do nothing, ignored 
                                                 in Isabelle *)
    | Typevardecl name ->    
        begin
          open_box 2;
          print_string "types";
          force_newline();
          print_string name;
          close_box()
        end
    | Enumdecl (typname, id_list) -> 
	(* In Isabelle we use degenerated Adt's for enumeration types.
	 *)
	isa_pp_top_theory_declaration(
	  Datatypedecl( typname, [],
		      	List.map 
			  (* leave the accessor and recognizer fields empty
			   * they are not used for Isabelle anyway 
			   *)
			  (fun id -> (id,[],""))
			  id_list))

    | Typedecl (params, name, typ) ->  (* Type parameters missing !! *)
	begin                   (* various forms in Isabelle *)
          (match typ with
             Record string_type_list ->
	        (match string_type_list with
	           | [] -> 
	               assert false;
	           | l -> 
                       begin
                         open_box 2;
			 (* print_space(); *)
                         print_string "record ";
			 isa_pp_param_list params;
                         print_space();
                         print_string (name ^ " = ");
                         force_newline();
                         List.iter
                           (fun (str, typ)
                              ->
                                begin
                                  print_string (str ^ " ::");
                                  print_space();
				  print_string "\"";
                                  isa_pp_top_type typ;
				  print_string "\"";
                                  force_newline();
                                end) string_type_list;
                         close_box()
                       end)
            | _ ->
                begin
                  open_box 2;
                  print_string "types";
                  force_newline();
		  isa_pp_param_list params;
                  print_space ();
		  print_string (name ^ " =");
                  print_space ();
                  isa_pp_top_type typ;
                  close_box()
                end)
	end
    | Defn (name, var_listlist, typ, expr) ->
	begin
	  open_box 2;
          print_string "constdefs";
          force_newline ();
          open_box 0;
	  print_string name;
          print_string " :: ";
          print_string "\"";
          open_box 0;
	  if 
	    var_listlist <> [[]] & var_listlist <> []
	  then   
            begin
              print_string "[";                
	      pp_list 
		(pp_list isa_pp_var_type "" "" "" ", ")
		"" 
		"" 
		"" 
		", " 
		var_listlist;
   	      print_string "] => "
            end;
	  print_space ();  
	  isa_pp_top_type typ;  
          print_string "\"";  
          close_box();  
          force_newline ();
          print_string "\"";
          open_box 0;
          print_string name;
          print_space ();
	  if 
	    var_listlist <> [[]]
	  then                  
	      pp_list 
		(pp_list isa_pp_var_name "" "" "" "")
		"" 
		"" 
		"" 
		"" 
		var_listlist;
	  print_string " =="; 
          print_space ();
	  isa_pp_top_expression expr; 
	  print_space (); 
	  print_string "\"";
          close_box();
	  close_box();
          close_box();
	end
    | Defrecfn _ ->
	isa_pp_top_theory_declaration 
	  (Comment ("Measure Recursive functions not" ^
                    " defined in isabelle - pretty-printer (JR)"))
    | Primrecdefn (name, typ, eq_list) ->
	begin
	  open_box 2;			(* 1 *)
          print_string "consts";
          force_newline ();
          open_box 0;			(* 2 *)
	  print_string name;
          print_string " :: ";
          print_string "\"";
          open_box 0;			(* 3 *)
	  isa_pp_top_type typ;  
          print_string "\"";  
          close_box();			(* 3 *)
          close_box();			(* 2 *)
          close_box();			(* 1 *)
          force_newline ();
	  open_box 2;			(* 1 *)
	  print_string "primrec";
	  force_newline();
	  List.iter (fun (l,r) ->
		       begin
			 open_box 2;			(* 2 *)
			 print_string "\"";
			 isa_pp_top_expression l; 
			 print_space (); 
			 print_string "= ";
			 isa_pp_top_expression r; 
			 print_string "\"";
			 close_box();			(* 2 *)
			 force_newline();
		       end
		    ) eq_list;
          close_box();			(* 1 *)
	end
    | Defnuninterpret (name, var_listlist, typ) ->
	begin
	  open_box 2;
          print_string "consts";
          force_newline ();
          open_box 2;
	  print_string name;
          print_string " :: ";
          print_string "\"";
          open_box 0;
	  if 
	    var_listlist <> [[]] & var_listlist <> []
	  then   
            begin
              print_string "[";                
	      pp_list 
		(pp_list isa_pp_var_type "" "" "" ", ")
		"" 
		"" 
		"" 
		", " 
		var_listlist;
   	      print_string "] => "
            end;
	  print_space ();  
	  isa_pp_top_type typ;  
          print_string "\"";  
          close_box();  
          close_box();
	end
    | Datatypedecl(name, param_list, constructor_list) ->
	begin
	  open_box 2;			                    (* 1 *)
          print_string "datatype";
	  print_space();
	  if param_list <> [] then
	    isa_pp_param_list param_list;
	  print_space();
	  print_string name;
	  print_string " =";
          force_newline ();
          open_box 0;					    (* 2 *)
	  (* print_string "    "; *)
	  pp_list
		(* the accsessor list contains the type of the constructor *)
	    (fun(name, accessor_list, typ) ->  
	       begin
		 open_box 2;				    (* 3 *)
		 print_string name;
		 print_space();
		 if 
		   accessor_list <> [] 
		 then
		   begin
		     (pp_list 
		 	(fun (_,t) -> 
			   print_string "\"(";
			   (isa_pp_top_type t);
			   print_string ")\""
			)
		 	"" 
		 	"" 
		 	"" 
		 	"" 
		 	accessor_list);
		   end;
		 close_box();				    (* 3 *)
		 force_newline();
	       end;
	    ) 
	    "" "" "" "|"
	    constructor_list;
	  close_box();					    (* 2 *)
	  close_box();					    (* 1 *)
	  force_newline()
	end

    | Comment str ->
	begin
	  open_box 0;
          print_string "(* ";
          force_newline();
	  List.iter
	    (fun s ->
               print_string (" * " ^ s);
	       force_newline())
	    (split_on_newline str);
	  print_string " *)";
	  close_box()
	end
    | Comment_decl_expr expr ->
	isa_pp_top_theory_declaration 
	  (Comment 
	     (stringwrapper isa_pp_top_expression expr))
    | Comment_decl decl ->
	  isa_pp_top_theory_declaration 
	    (Comment 
	       (stringwrapper isa_pp_top_theory_declaration decl))
    | Import ilist ->
	begin
	  open_box 0;
          print_space(); 
          print_space();
	  if 
	    ilist <> [] 
	  then
	    begin
              open_box 0;
              List.iter
                (fun (th_name,_) ->
                   begin
                     print_string th_name;
                     print_string " + "
                   end) ilist;
              close_box ()
	    end;
	  close_box()
	end
    | LibImport(lib,name,args) -> 
	isa_pp_top_theory_declaration (Import([name,args]))
    | Library(name,path) -> ()		(* no libraries in Isabelle *)
    | Conversion (name, typ) -> (* Do we use this? What is this in Isabelle?*)
	begin
	  open_box 0;
	  print_string ("CONVERSION " ^ name ^ " :") ;
	  print_break 1 2;
	  isa_pp_top_type typ;
	  close_box()
	end
    | Proved (decl, proof) -> (* proofs are treated somewhere else *)
	isa_pp_top_theory_declaration decl


and isa_pp_param_list param_list =
  let pp_param = function
    | TypeParameter id ->
	print_string ("'" ^ id.id_token.token_name)
  in 
    if param_list <> [] then
      begin
      	print_string "(";
      	(pp_list pp_param "" "" "" "," param_list);
      	print_string ")"
      end

and isa_pp_arg_list al = 
  let pp_arg = function
    | TypeArgument(t) -> 
	begin
	  print_string "(";
	  isa_pp_top_type t;
	  print_string ")"
	end
  in
    begin
      print_string "(";
      (pp_list pp_arg "" "" "" "," al);
      print_string ")";
    end

and isa_pp_var =
    function (name, typ) ->
      begin
	open_box 2;
	print_string ("(" ^ name ^ " :: ");
        isa_pp_top_type typ;
        print_string ")";
	close_box ()
      end

and isa_pp_var_name  vardecl = 
  let (name,typ) = match vardecl with
    | Declared(name,typ) -> (name,typ)
    | Undeclared(name,typ) -> (name,typ)
  in
      begin
	open_box 0;
	print_string (name);
	close_box ()
      end

and isa_pp_var_type vardecl = 
  let (name,typ) = match vardecl with
    | Declared(name,typ) -> (name,typ)
    | Undeclared(name,typ) -> (name,typ)
  in
      begin
	open_box 2;
	isa_pp_top_type typ;
	close_box ()
      end

let isa_pp_theory theory =
    match theory#kind with
      | Theory ->
          open_box 0;
          (* Print the theory header *)
	  print_string theory#get_name;
          print_string " =";
	  force_newline();
          (* print_space(); *)
          (* The theory declarations *)
          open_box 0;
	  List.iter
	    (fun decl ->
	       isa_pp_top_theory_declaration decl;
	       force_newline();
	       force_newline())
	    theory#get_body;
          close_box();
          (* The theory footer *)
	  print_string "end";
          force_newline ();
          close_box()
      | Datatype ->
				(* datatypes are in ordinary theories *)
	  assert(false)
;;


let isa_pp_proof th = 
  match th#kind with 
    | Theory ->
      	List.iter
    	  (function
	   | Proof (name, goal, script) ->
	       open_box 0;
               print_string "Goal \"";
               isa_pp_top_formula goal;
               print_string "\"";
               force_newline();
               pp_list print_string "" "" "" ";\n" script;
               force_newline();
               print_string "qed \"";
               print_string name;
               print_string "\"";
	       close_box()
	   | Rewrite_set (name, rules) ->
               open_box 0;
               print_string "val ";
	       print_string name;
               print_string " = ";
	       (match rules with
		  |[] -> print_string "[];"
		  |(hd :: tl) -> 
		      begin
		       	pp_list print_string "" "" "" "@" rules;
		       	print_string ";"
		      end);
               force_newline();
               force_newline();
               close_box();
		 (* include the following line, 
                  * if the top_pre_proof_file_entry variant is
		  * extended to generate PVS proof files.
		    | _ -> assert false
		  *)
    )
    th#get_proofs
  | Datatype -> 
      (* What to do with Datatypes ? Nothing ? *)
      ();;

(***********************************************************************
 ***********************************************************************
 *
 * module implementation
 *
 *)



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

