(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Adopted for Revision on 2.2.99 by Hendrik
 *
 * Time-stamp: <Monday 8 October 01 17:58:00 tews@ithif51>
 *
 * Interface for the polymorphic symboltable
 *
 * $Id: table.mli,v 1.5 2002/05/03 15:01:19 tews Exp $
 *
 *)


(***********************************************************************
 * Signatur der Symboltabelle
 *)

 
       (* This symboltable is polymorph in three types:
        * 'space : an enumeration of the different namespaces
        * 'shape : a type information used for overloading
        * 'symbol: the internal representation this table manages
        *)
type ('space, 'shape, 'symbol) global

       (* Type of the returned tables *)
type ('space, 'shape, 'symbol) local


(*******************************************************************
 *
 * Exception section
 *)
       (* exception if a string is not found in a table *)
exception Not_defined

    (* exception to be raised if a symbol is overloaded,
       but see documentation for find 
     *)
exception Overloaded


(*******************************************************************
 *
 * Creating a new table
 * 
 * the argument is suposed to be an equality predicate on
 * the type 'shape used for overloading
 *)

val new_table : ('shape -> 'shape -> bool) ->
                        ('space, 'shape, 'symbol) global

(*************************************************************************
 *
 * Creating a new local (= block) table, without changing the global 
 * symbol table. This is neccessary in special cases, ccsl class 
 * instanciation for instance.
 * 
 * For manageing scopes you should use start_block/close_block, see below
 *)

val new_local : ('space, 'shape, 'symbol) global -> 
                              ('space, 'shape, 'symbol) local

(*******************************************************************
 *
 * Creating entries in the global Table
 *)
    (* (create name x) creates an new entry for name in 
       table, possibly hiding other entries
     *)
val create : ('space, 'shape, 'symbol) global ->
  'space -> string -> 'shape -> 'symbol -> unit

    (* overload an entry with new symbol, 
       create it, if not already done in this block
    *)
val overload : ('space, 'shape, 'symbol) global ->
  'space -> string -> 'shape -> 'symbol -> unit

(*******************************************************************
 *
 * adding entries to a local table
 *)

val local_overload : ('space, 'shape, 'symbol) local -> 
  'space -> string -> 'shape -> 'symbol -> unit


(*******************************************************************
 *
 * finding entries in the global table
 *)
    (* look for a symbol in the hope it is not overloaded
       raise Overloaded if it is overloaded
     *)
val find : ('space, 'shape, 'symbol) global ->
  'space -> string -> 'symbol

    (* find a symbol with type information *)
val find_overloaded : ('space, 'shape, 'symbol) global ->
  'space -> string -> 'shape -> 'symbol

    (* find all symbols of a given name *)
val find_all : ('space, 'shape, 'symbol) global ->
  'space -> string -> ('shape*'symbol) list

    (* look for a non overloaded symbol, 
     * if found, return both, the local block and the symbol
     *)
val find_with_block : ('space, 'shape, 'symbol) global ->
  'space -> string -> 'symbol * ('space, 'shape, 'symbol) local

    (* find all symbols of a given name, return them 
     * with the block in which they are defined
     *)
val find_all_with_block : 
  ('space, 'shape, 'symbol) global ->
    'space -> string -> 
      ('shape * 'symbol * ('space, 'shape, 'symbol) local) list

(*******************************************************************
 *
 * finding entries in local tables
 *)

    (* do lookup in a local table *)
val find_local : ('space, 'shape, 'symbol) local -> 
  'space -> string -> 'symbol

    (* look for an overloaded symbol in a local table *)
val find_local_overloaded : ('space, 'shape, 'symbol) local ->
  'space -> string -> 'shape -> 'symbol

    (* find all symbols of a given name in local table *)
val find_all_local : ('space, 'shape, 'symbol) local -> 
  'space -> string -> ('shape * 'symbol) list


(*******************************************************************
 *
 * deleting entries from a local tables
 *)
    (* delete an overloaded symbol from a local table *)
val delete_local_overloaded : ('space, 'shape, 'symbol) local ->
  'space -> string -> 'shape -> unit


(*******************************************************************
 *
 * managing scopes
 *)
       (* start a new block -> make entries into a local table 
        * this function actually _creates_ a new local symbol
        * table! Use open_block nesting of scopes in a second 
        * pass
        *)
val start_block : ('space, 'shape, 'symbol) global -> 
  ('space, 'shape, 'symbol) local

       (* end the definition section of the current block
        * make entries into the same block 
        * like before start_block was called
        *
        * return the definitions
        *)
val end_of_defs : ('space, 'shape, 'symbol) global -> 
  ('space, 'shape, 'symbol) local

       (* close the current block *)
val close_block : ('space, 'shape, 'symbol) global -> 
  ('space, 'shape, 'symbol) local

       (* reset the symbol table. In case of errors all blocks 
	* upto the number of the second argument are closed.
        *)
val reset_gst : ('space, 'shape, 'symbol) global -> int -> unit

       (* nesting scopes: add this local name space to the global 
        * symboltable, use close_block to remove it
        *)
val open_block : ('space, 'shape, 'symbol) global -> 
  ('space, 'shape, 'symbol) local -> unit


       (* return the number of open blocks
        *)
val nesting_size : ('space, 'shape, 'symbol) global -> int


(*******************************************************************
 *
 * miscellany
 *)

       (* iterate over all entries *)
val iter : ('space,'shape,'symbol) global -> 
  ('space -> string -> 'shape -> 'symbol -> unit) -> unit

       (* print the whole table *)
val dump  : ('space, 'shape, 'symbol) global ->
(* Hendrik: should probably be 'shape -> 'symbol -> string *)
  ('symbol -> string) -> string

       (* print the whole table 
          (same as output_string channel (dump ) ) *)
val print  : ('space, 'shape, 'symbol) global ->
  ('symbol -> string) -> out_channel -> unit

       (* print the whole table *)
val dump_local  : ('space, 'shape, 'symbol) local ->
(* Hendrik: should probably be 'shape -> 'symbol -> string *)
  ('symbol -> string) -> string




(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

