(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Tuesday 16 July 02 10:42:21 tews@ithif51>
 *
 * Utility functions for ccsl input types
 *
 * $Id: types_util.mli,v 1.13 2002/07/18 13:43:25 tews Exp $
 *
 *)

open Top_variant_types
open Classtypes;;

(***********************************************************************
 *
 * counting selfs and checking for the various functor classes
 *
 *)

    (****************************
     * check_for_params walks through the type 
     * and returns true, if in typ, there is no 
     * typeparameter from param_p, if one parameter 
     * from param_p is found, it returns false.
     *
     *)
val check_for_params : 
  (('cl,'mem) top_pre_identifier_record_type -> bool) ->
    ('cl,'mem) top_pre_types -> bool

val check_for_params_args : 
  (('cl,'mem) top_pre_identifier_record_type -> bool) ->
    ('cl,'mem) top_pre_argument_type list -> bool

    (****************
     * checks if type corresponds to a constant functor, 
     * in which certain parameters (given by param_p) do not occur
     * needed in predicate lifting
     * 
     * val constant_type : ccsl_input_types -> bool
     *)
val constant_type : 
  (('cl, 'mem) top_pre_identifier_record_type -> bool) ->
  ('cl, 'mem) top_pre_types -> bool

val constant_arg_list : 
  (('cl, 'mem) top_pre_identifier_record_type -> bool) ->
  ('cl, 'mem) top_pre_argument_type list -> bool


    (* count all selfs *)
val count_self : ('cl, 'mem) top_pre_types -> int

    (* count all selfs in a list of arguments *)
val count_self_args :
  (<get_name:string;..> as 'cl, 'mem) top_pre_argument_type list -> int

    (* count all carriers *)
val count_carrier : ('cl, 'mem) top_pre_types -> int

    (* count all carriers in a list of arguments *)
val count_carrier_args :
  (<get_name:string;..> as 'cl, 'mem) top_pre_argument_type list -> int


    (****************
     * returns true if argument contains a free type variable
     *)

val type_is_nonground : ('cl, 'mem) top_pre_types -> bool

val args_are_nonground : ('cl, 'mem) top_pre_argument_type list -> bool

    (****************
     * returns true if the type variable occurs in the type
     *)

val occur_check : Type_variable.t -> ('cl, 'mem) top_pre_types  -> bool

    (******************************************************
     *
     * check if this type yields a successor state in SELF
     * needed for enumeration type of successor states
     *
     * type should be
     *     ccsl_input_types without Carrier -> bool
     *)

val has_successor_state : ('cl, 'mem) top_pre_types -> bool



    (********
     * curry an Adt constructor type to convert it for Isabelle 
     *)
val curry_constructor_type : 
  ('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types 


    (***************
     *
     * replace SmartFunction with iterated Function's or Products,
     * depending on Global.output_mode
     * 
     * needed for normalizing types during typechecking
     *)

val unsmart_type : ('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types 


(***********************************************************************
 *
 * iter_components walks through the type and applies action 
 * to all components
 * the intention is, that action is applied exactly in the order in 
 * which the importings must be generated
 *
 *)

val iter_components : 
  variance_type -> 
    (variance_type -> 
       (<get_parameters : ('cl, 'mem) top_pre_parameter_type list; 
	..> as 'cl, 'mem) 
       top_pre_types -> unit) ->
      ('cl, 'mem) top_pre_types -> unit

val iter_component_arglist : 
  variance_type -> 
    (variance_type -> 
       (<get_parameters : ('cl, 'mem) top_pre_parameter_type list; 
	..> as 'cl, 'mem) 
       top_pre_types -> unit) ->
      ('cl, 'mem) top_pre_argument_type list -> 
	('cl, 'mem) top_pre_parameter_type list -> unit

    (* this is equivalent to 
     * iter_component Pos (.... -> cl#add_component ... ) typ
     *)
val iter_components_add_component : 
  (<get_name : string;
    get_parameters : ('cl, 'mem) top_pre_parameter_type list; 
    add_component : ('cl, 'mem) top_pre_component_type -> unit;
	..> as 'cl) ->
    ('cl, 'mem) top_pre_types -> unit

    (* this is equivalent to 
     * iter_component_arglist Pos (.... -> cl#add_component ... ) args params
     *)
val iter_component_arglist_add_component : 
  (<get_name : string;
    get_parameters : ('cl, 'mem) top_pre_parameter_type list; 
    add_component : ('cl, 'mem) top_pre_component_type -> unit;
	..> as 'cl) ->
    ('cl, 'mem) top_pre_argument_type list -> 
      ('cl, 'mem) top_pre_parameter_type list -> unit

(* equality on ccsl_output_types *)
val eq_ccsl_types : ccsl_output_types -> ccsl_output_types -> bool

val eq_ccsl_args : ccsl_argument_type -> ccsl_argument_type -> bool

val eq_ccsl_expressions : ccsl_expressions -> ccsl_expressions -> bool

(* equality on textual representation of typeparameters *)
val eq_ccsl_params : 
  ccsl_parameter_type list -> ccsl_parameter_type list -> bool

  (* computes the list of argument types from a method (without leading self)
   *)
val method_arg_list : 
  <get_domain : ('cl,'mem) top_pre_types; ..> -> ('cl,'mem) top_pre_types list

  (* the argument list for methods/constructors *)
val member_arg_list : 
  <get_domain : ('cl,'mem) top_pre_types; ..> -> ('cl,'mem) top_pre_types list


(***********************************************************************
 *
 * quantifier utilities
 * 
 * Attention!! These functions do not check for the free variable 
 * condition!. They do no alpha conversion. It is the responsibility
 * of the user, to ensure that no free variables are bound and 
 * no variable is bound twice (using flat name spaces for instance)
 *
 * pull_forall pulls universal quantification to the front.
 * In principle it performs the following rewrites:
 * 
 *    Not Exists(y) : F   ===> Forall(y) : Not F
 * 
 *    (Forall(x) : F) AND (Forall(y) : G) 
 * 			   ===> Forall(x @ y) : (F AND G)
 * 
 *    (Forall(x) : F) OR (Forall(y) : G) 
 * 			   ===> Forall(x @ y) : (F OR G)
 * 
 *    Forall( y ): F  IMPLIES Forall( z ) : G
 * 			   ===> Forall(z) : (Forall(y) : F IMPLIES G)
 *    
 * pull_exits does the same for existential quantification.
 * 
 *)

val pull_forall : ('cl, 'mem) top_pre_formulas -> ('cl, 'mem) top_pre_formulas

val pull_exits : ('cl, 'mem) top_pre_formulas -> ('cl, 'mem) top_pre_formulas


(***********************************************************************
 *
 * make_rewrite_form takes a formula and tries to transform it into 
 * a PVS rewrite formula or a isabelle resolution formula
 *
 * !!! for proper functioning formulas must not contain any !!!
 * !!! FormLoc, ExprLoc                                     !!!
 *)

val make_pvs_rewrite_form : ('cl, 'mem) top_pre_formulas -> ('cl, 'mem) top_pre_formulas

val make_isa_rewrite_form : ('cl, 'mem) top_pre_formulas -> ('cl, 'mem) top_pre_formulas



(************************************************************************
 *
 * specialize substitution from logic_util
 * for eq_ccsl_types as equality on types
 *)

val ccsl_substitute_types :
  (ccsl_output_types * ccsl_output_types) list ->
    ccsl_output_types -> 
      ccsl_output_types

val ccsl_substitute_arguments : 
  (ccsl_output_types * ccsl_output_types) list ->
    ccsl_argument_type list ->
      ccsl_argument_type list


(***********************************************************************
 *
 * export method / constructor types
 * 
 * export_member class/adt member 
 * substitutes each BoundType with a uniqe free type variable,
 * self/carrier is substituted with the adt/class type
 *
 * returns first the type which is substituted for Self
 * and second the exported type
 *)

val export_member : 
  ccsl_iface_type -> ccsl_member_type -> ccsl_input_types -> 
    ccsl_inst_iface_type * ccsl_input_types

    (* same as export member, but uses the given arguments for the 
     * type parameters
     *)
val export_member_with_args : 
  ccsl_iface_type -> ccsl_member_type -> 
    ccsl_argument_type list -> ccsl_input_types -> 
      ccsl_inst_iface_type * ccsl_input_types

    (* export the local type parameters only *)
val export_member_local : 
  ccsl_iface_type -> ccsl_member_type -> ccsl_input_types -> 
    ccsl_inst_iface_type * ccsl_input_types

(***********************************************************************
 *
 * substitute all type variables with fresh ones
 * 
 * top_type must not contain Predtype (this restriction could
 *  be lifted, but I am too lazy now)
 *)

val uniquify_type_variables :
  ('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types


(********************************************************************
 * other stuff
 *)


  (***********
   * arguments_from_parameters rewrites
   *   TypeParameter -> TypeArgument
   *   ValueParameter -> ValueArgument
   *)
val arguments_from_parameters : 
  ('cl, 'mem) top_pre_parameter_type list -> 
    ('cl, 'mem) top_pre_argument_type list

  (**********
   * types_from_parameters rewrites
   *   TypeParameter -> TypeVariable
   *)
val types_from_parameters :
  ('cl, 'mem) top_pre_parameter_type list -> ('cl, 'mem) top_pre_types list

val types_from_arguments : 
  ('cl, 'mem) top_pre_argument_type list ->  ('cl, 'mem) top_pre_types list


val make_substitution :
  ('cl, 'mem) top_pre_parameter_type list ->  
    ('cl, 'mem) top_pre_argument_type list ->
      ( ('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_types) list

val make_substitution_param_param :
  ('cl, 'mem) top_pre_parameter_type list ->  
    ('cl, 'mem) top_pre_parameter_type list ->  
      ( ('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_types) list


  (**************
   * check if the arguments suit the parameters in number, kind and type
   *)
val check_parameters : ('cl, 'mem) top_pre_parameter_type list -> 
  ('cl, 'mem) top_pre_argument_type list -> bool


val id_record_from_string : 
  string -> ('cl, 'mem) top_pre_identifier_record_type


(*******************************************************************
 *
 * utility functions for ground types
 *
 *)

val get_ground_type_parameters :
  (<get_parameters : ('cl, 'mem) top_pre_parameter_type list; ..> as 'cl,
   'mem) top_pre_identifier_record_type ->
    ('cl, 'mem) top_pre_parameter_type list

val get_definition_parameters :
  (<get_parameters : 
     ('cl, <hosting_class : 'cl; 
            get_local_parameters : ('cl,'mem) top_pre_parameter_type list; .. 
           > as 'mem) 
     top_pre_parameter_type list; ..> as 'cl,
     'mem) 
  top_pre_definition_record -> 
    ('cl, 'mem) top_pre_parameter_type list

val is_type_def : ('cl,'mem) top_pre_identifier_record_type -> bool

  (* extracts the signature out of a type definition *)
val sig_of_typedef : ccsl_identifier_record_type -> ccsl_iface_type

    (* expand_type_def eq_types typ
     * expands typ as long as the top level constructor is a 
     * type definition
     *)
val expand_type_def : 
  (
    ( <get_parameters : ('cl, 'mem) top_pre_parameter_type list; ..> as 'cl,
      'mem) 
    top_pre_types -> ('cl, 'mem) top_pre_types  -> bool) ->
    ('cl, 'mem) top_pre_types -> 
      ('cl, 'mem) top_pre_types


  (************************************************
   *
   * extract type parameters of a member, taking local parameters 
   * into account
   *)
val get_member_parameters : 
  (< get_local_parameters : 
     (<get_parameters : ('cl, 'mem) top_pre_parameter_type list;..> as 'cl, 
      'mem) top_pre_parameter_type list;
     get_sort : 'mem top_pre_member_sort;
     hosting_class : 'cl; .. 
   > as 'mem) ->
      ('cl, 'mem) top_pre_parameter_type list


  (**************
   * utility functions for features
   * at the moment groundtypes have a fixed feature set
   *
   * supplied type must be one of Class / Adt / Groundtype
   *)

val type_has_feature : 
  feature_type ->
    (< has_feature : feature_type -> bool; .. > as 'cl,
     'mem) top_pre_types ->
      bool


  (**************
   * the next two functions are for extracting the location field
   * !! they fail if the top constructor is not a location !!
   *)

val get_ex_loc : ('cl, 'mem) top_pre_expressions -> location_type

val get_form_loc : ('cl, 'mem) top_pre_formulas -> location_type


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

