(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 12/3/99 by Joachim
 *
 * Time-stamp: <Tuesday 16 July 02 11:49:09 tews@ithif51>
 *
 * Output theories
 *
 * $Id: syntax.ml,v 1.9 2002/07/18 13:43:28 tews Exp $
 *
 *)


open Unix
open Global
open Util
open Top_variant_types;;


(* put this in front of every generated file *)
let file_header() = 
					(* isar mode emacs comment *)
  (if !output_mode = Isa_mode 
   then " -*- isar -*- \n\n"
   else ""
  ) ^
					(* compiler version *)
  "This file was automatically generated by " ^
  compiler_version ^ "\n\n" ^
				(* source file for this theory, if present *)
  (match !current_top_level_input with
     | Some f -> "source file  : " ^ f ^ "\n"
     | None -> "")  ^
					(* date *)
  "date         : " ^ 
  (let t = localtime(time()) in
     Printf.sprintf "%d.%d.%d at %d:%02d:%02d"
       t.tm_mday (t.tm_mon+1)
       (t.tm_year + 1900) t.tm_hour t.tm_min
       t.tm_sec
  ) ^"\n\n" ^
					(* command line *)
  "command line :" ^ 
  (Array.fold_left (fun s arg -> s ^" "^ arg)
     "" Sys.argv) ^ "\n" 


    (* a list of theories may demand several filenames, possibly unsorted
     * we keep a list of files, which have been already reset here
     *)
let reset_files = ref ([] : string list);;



let output_pvs_theory th =
  match th#kind with
    | Datatype
    | Theory -> 
	begin
	  Pvs_pretty.pvs_pp_theory th;
	  Formatter.force_newline();
	  Formatter.print_newline()
	end
    | IsabelleStartFile
    | IsabelleCloseFile -> ()

let output_pvs_proof th =
  match th#kind with
    | Datatype
    | Theory -> 
	begin
	  Pvs_pretty.pvs_pp_th_proofs th#get_name th#extract_pvs_proofs;
	  Formatter.force_newline();
	  Formatter.print_newline()
	end
    | IsabelleStartFile
    | IsabelleCloseFile -> ()

(* OLD ISABELLE
 *
 * let output_isa_theory th = 
 *   Isabelle_pretty.isa_pp_theory th;
 *   Formatter.force_newline();
 *   Formatter.print_newline();;
 *)

let output_isar_theory th = 
  Isar_pretty.isar_pp_theory th;
  Formatter.force_newline();
  Formatter.print_newline();;

(* OLD ISABELLE
 * 
 * let output_isa_proof th =
 *   (* no support for Isabelle proofs *)
 *    Isabelle_pretty.isa_pp_proof th;
 *    Formatter.force_newline();
 *    Formatter.print_newline();;
 *)
	   

(* output function for generating theories *)
let pvs_output thl = 
  let rec doit last th = 
    let file_name = th#get_file_name in
      match last with
	| Some(last_name,last_pvs,last_prf) -> 
	    if last_name = file_name or !filter_style
	    then begin
	      if !theory_generation then begin
		Formatter.set_formatter_out_channel last_pvs;
		output_pvs_theory th
	      end;
	      if th#do_proofs then begin
		Formatter.set_formatter_out_channel last_prf;
		output_pvs_proof th;
	      end;
	      last
	    end else begin
	      reset_files := last_name :: !reset_files;
	      if last_pvs != Pervasives.stdout then
		close_out( last_pvs );
	      if last_prf != Pervasives.stdout then
		close_out( last_prf );
	      doit None th
	    end
	| None ->
	    let pvs_file_name = 
	      (get_target_directory() ^ file_name ^ ".pvs") in
	    let prf_file_name =
	      (get_target_directory() ^ file_name ^ ".prf") in
	    let new_file = 
	      not ((Sys.file_exists pvs_file_name) 
		   && (List.mem file_name !reset_files)
		  ) in
	    let flags = [Open_wronly; Open_text] @
		  	if new_file
			then [Open_creat; Open_trunc]
		  	else [Open_append] in
	    let _ = if new_file then 
	      if !theory_generation then
		begin
		  make_backup pvs_file_name;
		  if th#do_proofs then
		    make_backup prf_file_name
		end
	    in
  	    let theory_file = 
	      if !theory_generation then
		open_out_gen flags 0o666 pvs_file_name 
	      else Pervasives.stdout in
	    let proof_file = 
	      if th#do_proofs then
		open_out_gen flags 0o666 prf_file_name
	      else
		Pervasives.stdout
	    in
	      (if new_file		(* initializing new file *)
	       then
		 if !theory_generation then 
		   begin
		     Formatter.set_formatter_out_channel theory_file;
		     Pvs_pretty.pvs_pp_top_theory_declaration
		       (Comment (file_header()));
		     Formatter.force_newline();
		     Formatter.print_newline()
		   end;
					(* initialize prf file *)
		    (* nothing to do *)
	       doit (Some(file_name, theory_file, proof_file)) th)
  in 
    try
      let last = (if !filter_style 
		  then 
		    List.fold_left 
		      doit (Some("",Pervasives.stdout,Pervasives.stdout)) thl
		  else List.fold_left doit None thl)
      in 
	(match last with
	   | Some(last_name,last_pvs,last_prf) -> 
	       begin
		 reset_files := last_name :: !reset_files;
		 if last_pvs != Pervasives.stdout then
		   close_out( last_pvs );
		 if last_prf != Pervasives.stdout then
		   close_out( last_prf );
		 Formatter.set_formatter_out_channel Pervasives.stdout
	       end
	   | None ->			(* no files opened *)
		 Formatter.set_formatter_out_channel Pervasives.stdout

	)
    with
      | exc ->
	  begin
(*
	    Formatter.print_newline();
	    Formatter.open_box 0;
	    Formatter.print_string 
	      ("!!! Output truncated by exception\n!!! " ^
	       Printexc.to_string exc);
	    Formatter.print_newline();
	    Formatter.print_newline();
(* Hendrik : close files *)
*)
	    raise exc
	  end

(* OLD ISABELLE
 * 
 * let isabelle_output thl =
 *   let rec doit last th = 
 *     let file_name = th#get_file_name in
 * 	 match last with
 * 	   | Some(last_name,last_theory_file,last_proof_file) -> 
 * 	       if last_name = file_name or !filter_style
 * 	       then begin
 * 		 if !theory_generation then 
 * 		   begin
 * 		     (* generate the theory files for Isabelle *)
 * 		     Formatter.set_formatter_out_channel last_theory_file;
 * 		     output_isa_theory th
 * 		   end;
 * 
 * 		 (* generate the proof files for Isabelle *)
 * 		 if th#do_proofs then begin
 * 		   Formatter.set_formatter_out_channel last_proof_file;
 * 		   output_isa_proof th;
 * 		 end;
 * 		 last
 * 	       end else begin
 * 		 reset_files := last_name :: !reset_files;
 * 		 if last_theory_file != Pervasives.stdout then
 * 		   close_out( last_theory_file );
 * 		 if last_proof_file != Pervasives.stdout then
 * 		   close_out( last_proof_file );
 * 		 doit None th
 * 	       end
 * 	   | None ->
 * 	       let new_file = (not (Sys.file_exists file_name)) or 
 * 			      (not (List.mem file_name !reset_files)) in
 * 	       let flags = [Open_wronly; Open_text] @
 * 			   if new_file
 * 			   then [Open_creat; Open_trunc]
 * 			   else [Open_append] in
 * 	       let _ = if new_file then 
 * 		 if !theory_generation then
 * 		   make_backup (get_target_directory() ^ file_name ^ ".thy");
 * 		 if th#do_proofs then
 * 		   make_backup (get_target_directory() ^ file_name ^ ".ML")
 * 	       in
 * 	       let theory_file = 
 * 		 if !theory_generation then
 * 		   open_out_gen flags 0o666
 * 		     (get_target_directory() ^ file_name ^ ".thy") 
 * 		 else Pervasives.stdout in
 * 	       let proof_file = 
 * 		 if th#do_proofs then
 * 		   open_out_gen flags 0o666 
 * 		     (get_target_directory() ^ file_name ^ ".ML") 
 * 		 else Pervasives.stdout 
 * 	       in
 * 		 (if new_file		(* initializing new file *)
 * 		  then
 * 		    if !theory_generation then 
 * 		      begin
 * 			Formatter.set_formatter_out_channel theory_file;
 * 			Isabelle_pretty.isa_pp_top_theory_declaration 
 * 			  (Comment (file_header()));
 * 			Formatter.force_newline();
 * 			Formatter.print_newline()
 * 		      end;
 * 					   (* initialize prf file *)
 * 		       (* nothing to do *)
 * 		  doit (Some(file_name, theory_file, proof_file)) th)
 *   in
 *     try
 * 	 let last = (if !filter_style 
 * 		     then 
 * 		       List.fold_left 
 * 			 doit (Some("",Pervasives.stdout,Pervasives.stdout)) thl
 * 		     else List.fold_left doit None thl)
 * 	 in 
 * 	   (match last with
 * 	      | Some(last_name,last_pvs,last_prf) -> 
 * 		  begin
 * 		    reset_files := last_name :: !reset_files;
 * 		    if last_pvs != Pervasives.stdout then
 * 		      close_out( last_pvs );
 * 		    if last_prf != Pervasives.stdout then
 * 		      close_out( last_prf );
 * 		    Formatter.set_formatter_out_channel Pervasives.stdout
 * 		  end
 * 	      | None -> assert(false)
 * 	   )
 *     with
 * 	 | exc ->
 * 	     begin
 * 	       Formatter.print_newline();
 * 	       Formatter.open_box 0;
 * 	       Formatter.print_string 
 * 		 ("!!! Output truncated by exception " ^
 * 		  Printexc.to_string exc);
 * 	       Formatter.print_newline();
 * 	       Formatter.print_newline();
 * (* Hendrik : close files *)
 * 	       raise exc
 * 	     end
 * 
 *)

let isar_output thl =
  let rec doit last th = 
    let file_name = th#get_file_name in
      match last with
	| Some(last_name,last_theory_file,last_proof_file) -> 
	    if last_name = file_name or !filter_style
	    then begin
	      if !theory_generation then 
		begin
		  (* generate the theory files for Isabelle *)
		  Formatter.set_formatter_out_channel last_theory_file;
		  output_isar_theory th
		end;

	      last
	    end else begin
	      reset_files := last_name :: !reset_files;
	      if last_theory_file != Pervasives.stdout then
		begin
		  close_out( last_theory_file )
		end;
	      doit None th
	    end
	| None ->
	    let isar_file_name = get_target_directory() ^ file_name ^ ".thy"
	    in
	    let new_file = (not (Sys.file_exists isar_file_name)) or 
			   (not (List.mem file_name !reset_files)) in
	    let flags = [Open_wronly; Open_text] @
		  	if new_file
			then [Open_creat; Open_trunc]
		  	else [Open_append] in
	    let _ = if new_file then 
	      if !theory_generation then
		make_backup isar_file_name;
	    in
  	    let theory_file = 
	      if !theory_generation 
	      then
		open_out_gen flags 0o666 isar_file_name
	      else
		Pervasives.stdout 
	    in let proof_file = Pervasives.stdout (* no isar proof files *)
	    in
	      (if new_file		(* initializing new file *)
	       then
		 if !theory_generation then 
		   begin
		     Formatter.set_formatter_out_channel theory_file;
		     Isar_pretty.isar_pp_theory_declaration 
		       (Comment (file_header()));
		     Formatter.force_newline();
		     Formatter.print_newline();
		   end;
	       doit (Some(file_name, theory_file, proof_file)) th)
  in
    try
      let last = (if !filter_style 
		  then 
		    List.fold_left 
		      doit (Some("",Pervasives.stdout,Pervasives.stdout)) thl
		  else List.fold_left doit None thl)
      in 
	(match last with
	   | Some(last_name,last_thy,last_ml) -> 
	       begin
		 reset_files := last_name :: !reset_files;

		 if last_thy != Pervasives.stdout then
		   begin
		     close_out( last_thy )
		   end;
		 if last_ml != Pervasives.stdout then
		   close_out( last_ml );
		 Formatter.set_formatter_out_channel Pervasives.stdout
	       end
	   | None ->			(* no files opened *)
		 Formatter.set_formatter_out_channel Pervasives.stdout
	)
    with
      | exc ->
	  begin
(* 	       Formatter.print_newline();
 * 	       Formatter.open_box 0;
 * 	       Formatter.print_string 
 * 		 ("!!! Output truncated by exception " ^
 * 		  Printexc.to_string exc);
 * 	       Formatter.print_newline();
 * 	       Formatter.print_newline();
 *)
(* Hendrik : close files *)
	    raise exc
	  end




(*******************************************************************
 *
 * proof test section
 *)

let output_edit_proof name pr =
  begin
    Formatter.set_formatter_out_channel Pervasives.stdout;
    Pvs_pretty.pvs_pp_edit_proof name pr;
    Formatter.force_newline();
    Formatter.print_newline()
  end;;


let do_proof_test thl =
  let proofs = List.flatten
    (List.map (fun th -> th#extract_pvs_proofs)
       thl) in
  let doit name = function
    | Named_proof(name',pr) when name=name' -> output_edit_proof name pr
    | _ -> () 
  in
    List.iter (fun name -> List.iter (doit name) proofs) (!do_proofs)

(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

