(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 22.4.99 by Bart
 *
 * Time-stamp: <Thursday 4 July 02 18:05:25 tews@ithif51>
 *
 * top level class structure
 *
 * $Id: top_classtypes.ml,v 1.12 2002/07/18 13:43:28 tews Exp $
 *
 *)


(*
 * Polymorphic class types are defined for iface, member and 
 * theory_body used at top level, from which classes for all 
 * translations (Ccsl, Java, ...) inherit. The use of type variables
 * like 'member_type, 'theory_body and 'super_class is crucial, so
 * that appropriate instantiations for specific translations work
 * deep into the relevant structure. Note that these class types
 * are not mutually recursive (which is not needed through the use
 * of type variables).
 *
 * The following three class types are introduced.
 *   ['member_type, 'theory_body_type, 'super_class] top_pre_iface_type 
 *   ['class_type] top_pre_member_type 
 *   ['class_type] top_pre_theory_body_type
 *
 *)


open Util;;
open Top_variant_types;;


(***********************************************************************
 ***********************************************************************
 *
 * Top class types 
 *
 *)


class type virtual 
        ['member_type, 'theory_body_type, 'super_class] top_pre_iface_type 
  = object ('self)

    val name : 
      token_type

      (* file name is the name of the file the class declaration is in *)
    val file_name :
      string

    val mutable ancestors : 
      ('super_class, 'member_type) top_pre_ancestor_type list

    val mutable members : 
      'member_type list

    val mutable inheritance_ready : 
      bool

					(* public method section *)
    method get_token : 
      token_type

    method get_name : 
      string

    method get_file_name :
      string

    method set_file_name :
      string -> unit

    method get_local : ('super_class, 'member_type) top_pre_scope_type

					(* inheritance pass done ? *)
    method inheritance_done : unit

    method inheritance_ready : bool

    method set_ancestors : 
      ('super_class, 'member_type) top_pre_ancestor_type list -> unit

    method add_ancestor : 
      ('super_class, 'member_type) top_pre_ancestor_type -> unit

    method get_ancestors : 
      ('super_class, 'member_type) top_pre_ancestor_type list

    (* access through the ancestor type to return 
       the _resolved_ ancestors;
       for CCSL this requires inheritance_ready
    *)
    method get_resolved_ancestors :
      'super_class list  

    (* returns only the _resolved_ ancestors, recursively *)
    method get_all_ancestors : 
      'super_class list

    method string_of_ancestors : 
      string

    method set_members : 
      'member_type list -> unit

    method add_member : 
      'member_type -> unit

    method get_members : 
      'member_type list

(*     method get_all_members_count : int
 *)

	(* Certain things rely on the traversion order of 
         * get_all_members !!
         * for instance method coalgebra_as_tuple, 
	 * method struct_of in the interface class,
	 * and ??
	 *)
    method get_all_members : 
      'member_type list

      (* delivers only proper attributes *)
    method get_attributes : 
      'member_type list

      (* delivers pseudo attributes and methods *)
    method get_methods : 
      'member_type list

      (* returns all methods reachable from this class *)
    method get_all_methods :
      'member_type list

    method get_all_attributes :
      'member_type list

      (* delivers all attributes and methods but not defined ones *)
    method get_sig_actions : 'member_type list

    method get_all_sig_actions : 'member_type list

      (* delivers all attributes and methods and defined methods
	 (but no constructors)*)
    method get_actions : 'member_type list

    method get_all_actions : 'member_type list

      (* get class constructors *)
    method get_constructors : 'member_type list

    method get_adt_constructors : 'member_type list


    (**********************************************
     * Features
     *)

    method put_feature : feature_type -> unit

    method has_feature : feature_type -> bool


    (**********************************************
     * support for theory generation
     *)
    method virtual generate_all_theories : 
      'theory_body_type list

					(* debugging section *)
    method dump_iface : 
      string

    method dependencies : 
      'super_class list

    (**********************************************
     * different specification kinds 
     *)

      (* For ccsl we only see after parsing the parameters, if the thing
       * becomes a class specification, an adt or a ground signature. 
       * We make therefore a union
       * class for all of them, called iface. A new object starts
       * up in an indefinite state (compare type ccsl_iface_kind), 
       * later exactly
       * one of the three methods become_class, become_adt or
       * become_sig is called.
       *)
    method get_kind : top_spec_kind
    method become_class : unit
    method become_adt : unit
    method become_sig : unit
    method is_class : bool
    method is_adt : bool
    method is_sig : bool

  end


class type ['class_type, 'member_type] top_pre_member_type 
  = object 
    val mutable name : 
      token_type

    val mutable domain : 
      ('class_type, 'member_type) top_pre_types

    val mutable codomain : 
      ('class_type, 'member_type) top_pre_types

    val mutable visibility : 
      top_visibility

    val mutable sort : 
      'member_type top_pre_member_sort

    val mutable is_hidden : 
      bool

    method get_token : 
      token_type

    method get_name : 
      string

    method get_domain : 
      ('class_type, 'member_type) top_pre_types

    method set_domain :
      ('class_type, 'member_type) top_pre_types -> unit

    method get_codomain : 
      ('class_type, 'member_type) top_pre_types

    method set_codomain :
      ('class_type, 'member_type) top_pre_types -> unit

    method get_visibility : 
      top_visibility

    method get_sort : 'member_type top_pre_member_sort

    method is_attribute : bool

    method is_method : bool

    method is_defined : bool

    method is_sig_action : bool

    method is_action : bool

    method is_class_method : bool

    method is_constructor : bool

    method is_adt_constructor : bool

    (*******************************************************************
     *
     * Renaming
     *)

    val mutable old_names : 
      token_type list 

    method get_old_names : token_type list

    method last_name : string

    val mutable renamed : bool

    method clear_renamed_flag : unit

    method is_renamed : bool

    method rename_member : token_type -> unit  

    method add_old_name : 
      token_type -> unit

    method original_name : string

    (*******************************************************************
     *
     * Inheritance
     *)

    method inherited : 
      unit

    method is_inherited : 
      bool

    method hide :
      unit

    method is_hidden : 
      bool

			(* debugging section *)
    method dump_member : 
      string


	(* list of package name (denoted by a string list) and 
	 * the name of the class
	 *)
    method dependencies : 
      (string list * token_type) list

  end


class type virtual ['class_type, 'member_type] top_pre_theory_body_type = 
  object ('self)

    val self_parameter :
      ('class_type, 'member_type) top_pre_parameter_type

    method virtual get_name : 
      string 

    method kind : top_theory_type

      (* Override this method, if proofs should not be written *)
    method do_proofs : bool

    method virtual get_parameters : 
      ('class_type, 'member_type) top_pre_parameter_type list

    method get_file_name : string

    method override_file_name : string -> unit

    method virtual make_body : 
      ('class_type, 'member_type) top_pre_theory_declaration list 

    method get_body : 
      ('class_type, 'member_type) top_pre_theory_declaration list 

      (* dependency list (aka imports) for Isar theories *)
    method get_isar_imports : string list 

(* ISAR
 *     method virtual get_proofs :
 * 	 ('class_type, 'member_type) top_pre_proof_file_entry list
 *)

    (* extract the pvs proofs from a given theory
     * and give them names, if they don't have yet 
     *
     *)
    method extract_pvs_proofs : pvs_proof list

  end


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
