(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Thursday 21 August 03 15:06:33 tews@ithif51.inf.tu-dresden.de>
 *
 * The inheritance pass
 * 
 *    - copy methods from superclasses in the local symboltable 
 *    - apply renaming if requested
 *    - try automatic renaming otherwise
 * 
 * $Id: inherit.ml,v 1.14 2003/08/21 15:15:04 tews Exp $
 *
 *)


exception Inheritance_error

open Util
open Error
open Global
open Top_variant_types
open Top_variant_types_util
open Types_util
open Substitution
open Member_class
;;

(*********************************************************
 *
 * Error messages
 *
 *)

let error_msg_old cl old_name = 
    "Invalid renaming clause: Method " ^
    old_name.token_name ^ " " ^
    " not defined in class " ^
    cl#get_name ^
    "\n"


let member_decl_message mem =
  mem#get_name ^ " " ^
  "(declared in class " ^
  mem#hosting_class#get_name ^
  (if not mem#is_renamed 
   then ""
   else (" as " ^ mem#original_name)
  ) ^ ")" 


let error_msg_new cl new_name new_clash = 
  "Invalid renaming clause: Method " ^
  (member_decl_message new_clash) ^
  " already defined in class " ^
  cl#get_name ^
  "\n"


let name_clash_message (cl : Classtypes.ccsl_iface_type)  mem = 
  "Name clash in class " ^
  cl#get_name ^
  ": member\n" ^
  (member_decl_message mem) ^ 
  "\nclashes with member\n" ^
  (member_decl_message (cl#find_member (mem#get_name))) ^ "\n"


let automatic_super_warning cl mem new_name =
  "Method " ^   
  (member_decl_message mem) ^ 
  "\nautomatically renamed as " ^ new_name ^ 
  " to avoid name clashes."

let d s = 
  if debug_level _DEBUG_INHERITANCE then
    print_verbose s

(*********************************************************
 *
 * Utility functions
 *
 *)


(* let find_method cl name = 
 *   let m = cl#find_member name in
 *     if m#is_action then m
 *     else raise Member_not_found
 *)

let find_method cl name = cl#find_member name

let method_absent cl name = 
  try 
    ignore(find_method cl name); false 
  with 
    | Member_not_found -> true


let automatic_super_rename (heir : Classtypes.ccsl_iface_type) cl mem loc = 
  let new_name = mem#hosting_class#get_name ^ "_" ^ mem#get_name in
  let new_tok = {token_name = new_name; loc = None }
  in
    if (method_absent cl new_name) && 
       (method_absent heir new_name) then
      begin
	warning_message loc
	  (automatic_super_warning cl mem new_name);
	mem#rename_member new_tok;
	true
      end
    else
      false
      
      

(*********************************************************
 *
 * Renaming
 *
 *)


(* process renaming clauses in class cl (which is some ancestor class) *)
let process_renaming cl renaming = 
   (* method names are unique in a class structure *)
  let do_clause (old_name,_, new_name) = 
    try
      let mem = find_method cl old_name.token_name in
	if method_absent cl new_name.token_name then
	  begin
	    mem#rename_member new_name;
	    mem
	  end
	else
	  let new_clash = find_method cl new_name.token_name 
	  in
	    begin
	      error_message (remove_option new_name.loc)
		(error_msg_new cl new_name new_clash);
	      raise Inheritance_error
	    end
    with
      | Member_not_found -> 
	  begin
	    error_message (remove_option old_name.loc)
	      (error_msg_old cl old_name);
	    raise Inheritance_error
	  end
  in
    List.map do_clause renaming


(* instanciate child and process user renamings *)
let do_super_class heir child args renaming loc = 
  let _ = assert(not heir#is_instanciated) in 
  let _ = assert(not child#is_instanciated) in
  let _ = assert(not heir#inheritance_ready) in 
  let _ = assert(child#inheritance_ready) in
    (* the filter for instanciation: 
     * let only attributes/methods/definitions pass 
     *)
  let filter = fun m -> m#is_action in
  let inst_child = instanciate child args filter in
  let renamed_members = process_renaming inst_child renaming 
  in
    Resolved_renaming(child, args, renamed_members, inst_child, loc)


(*********************************************************
 *
 * Inheritance
 *
 *)


(* inherit from one (resolved) ancestor:
 *   - check/resolve name clashes
 *   - inherit methods (via inheriting the local symbol tables)
 *   - update heir in ancestor
 *)
let inherit_class (heir : Classtypes.ccsl_iface_type)  = function
  | Resolved_renaming(ancestor, _, renamed_members, inst_anc, loc) ->
(* 	 (* debugging support *)
 * 	 let heir_name = heir#get_name in
 * 	 let anc_name = inst_anc#get_name in
 * 	 (* end debugging support *)
 *)
      let was_renamed mem = 
	List.exists (fun m -> m = mem) renamed_members
      in
					(* check for name clashes with mem *)
      let check_member mem = 
	begin
	  if not (method_absent heir mem#get_name) then
	    if not (was_renamed mem) then
	      begin   (* name clash with mem and mem was not renamed 
                         -> try automatic renaming
                      *)
		if not (automatic_super_rename heir inst_anc mem loc) then
		  begin
					(* automatic renaming failed *)
		    error_message loc
		      (name_clash_message heir mem);
		    raise Inheritance_error
		  end
	      end
	    else
	        (* name clash with mem although user renamed it ! *)
	      begin
		error_message loc
		  (name_clash_message heir mem);
		raise Inheritance_error
	      end;
	    (* no name clash with mem or 
             * name clash could be resolved automatically 
             *)
	  mem#inherited
	end
      in
	begin
	  List.iter check_member (inst_anc#get_all_members);
	  heir#inherit_locals 
	    (inst_anc#get_local :: inst_anc#get_inherited_locals);
	  inst_anc#inherit_to heir
	end
  | _ -> assert(false)


(* inheritance pass for classes *)
let do_class heir  = 
  let _ = d ("Inheritance in " ^ heir#get_name ) in
		(* instanciate ancestors and process user renamings *)
  let nanc = List.map
    (function
       | Unresolved_renaming(cl, al, rename, loc) -> 
	   do_super_class heir cl al rename loc
					(* no renaming done before *)
       | Resolved_renaming _ -> assert(false)
    ) 
    heir#get_ancestors
  in
  let _ = heir#set_ancestors nanc in
  let _ = heir#inheritance_done 	       (* enable find_member in heir *)
  in
    List.iter (inherit_class heir) nanc;
					(* check for empty classes *)
    (match heir#get_all_actions with
       | [] -> (error_message (remove_option heir#get_token.loc)
		  ("A class specification must contain at least one method " ^
		   "or attribute.");
		raise Inheritance_error
	       )
       | _ -> ()
    )


let do_adt adt = 
  begin
    d ("Inheritance in " ^ adt#get_name );
    adt#inheritance_done
  end

let do_sig si = 
  begin
    d ("Inheritance in " ^ si#get_name );
    si#inheritance_done
  end


let inherit_ast = function
  | CCSL_class_dec cl -> do_class cl
  | CCSL_adt_dec adt -> do_adt adt
  | CCSL_sig_dec si -> do_sig si


let ccsl_inheritance_pass (ast: Classtypes.ccsl_ast list) = 
    List.iter inherit_ast ast;;


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

