(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * * Created 15.1.99 by Hendrik
 *
 * Time-stamp: <Wednesday 20 August 03 18:27:57 tews@debian>
 *
 * main compiler driver
 *
 * $Id: main.ml,v 1.21 2003/08/21 15:15:03 tews Exp $
 *
 *)


(* Modules *)
open Global;;
open Error;;
open Util;;

open Lexing;;
open Syntax;;

open Top_variant_types;;
open Top_variant_types_util;;
open Top_classes;;

open Classtypes
open Inherit
open Attribute_pass
open Variance
open Component_pass
open Feature
open Class_methods
open Resolution
open Derive
open Newtypecheck
open Behavioural
;;


(***********************************************************************
 ***********************************************************************
 *
 * declaring some constants
 *
 *)


let builtin_prelude = function
  | Pvs_mode -> 
  "
   Begin EmptySig : GroundSignature
      Importing EmptyTypeDef
      Type EmptyType
   End EmptySig
   
   Begin EmptyFunSig [A : Type]: GroundSignature
      Importing EmptyFun[A]
      Constant
         empty_fun : [EmptyType -> A];
   End EmptyFunSig
   
   Begin list[ X : Type ] : Adt
      Constructor
         null : Carrier;
         cons( car, cdr ) : [X, Carrier] -> Carrier
   End list"
      ^ 
      (if !old_lift 
       then ""
       else
  "

   Begin lift[ X : Type ] : Adt
      Constructor
         bottom : Carrier;
         up( down ) : X -> Carrier
   End lift"
      )
  | Isa_mode -> 
  "
   Begin list[ X : Type ] : Adt
      Constructor
         Nil : Carrier;
         Cons( car, cdr ) : [X, Carrier] -> Carrier
   End list"


let ccsl_prelude = function
  | Pvs_mode -> 
      (if !old_lift 
       then
  "
   Begin Lift[ X : Type ] : Adt
      Constructor
         bot : Carrier;
         up( down ) : X -> Carrier
   End Lift
  "
       else ""
      ) ^
  "
   Begin Coproduct[ X : Type, Y : Type ] : Adt
      Constructor
         in1(out1) : X -> Carrier;
         in2(out2) : Y -> Carrier;
   End Coproduct
   
   Begin Unit : Adt
      Constructor
         unit : Carrier;
   End Unit
  "
  | Isa_mode ->
      (if !old_lift 
       then
  "
   Begin Lift[ X : Type ] : Adt
      Constructor
         bot : Carrier;
         up( down ) : X -> Carrier
   End Lift
  "
       else
  "
   Begin lift[ X : Type ] : Adt
      Constructor
         bottom : Carrier;
         up( down ) : X -> Carrier
   End lift
  "
      ) ^
  "
   Begin Coproduct[ X : Type, Y : Type ] : Adt
      Constructor
         in1(out1) : X -> Carrier;
         in2(out2) : Y -> Carrier;
   End Coproduct
   
   Begin Unit : Adt
      Constructor
         unit : Carrier;
   End Unit
  ";;


let output_prelude () =
  print_endline "PVS prelude:";
  print_endline (builtin_prelude Pvs_mode);
  print_endline (ccsl_prelude Pvs_mode);
  print_endline "Isabelle/HOL prelude:";
  print_endline (builtin_prelude Isa_mode);
  print_endline (ccsl_prelude Isa_mode);
  exit 0


(***********************************************************************
 ***********************************************************************
 *
 * Parsing the commandline, have a look at the module Arg
 *
 *)


let option_not_supported opt =
  print_error ("Option '" ^ opt ^ "' not supported");
  exit 1;;


(* The most common options consist of one letter
 * other options should be written in full.
 *)
let options =
  [
    ("-pvs",
     Arg.Unit (fun () -> output_mode := Pvs_mode),
     "Set output mode to PVS");
    ("-isa",
     Arg.Unit (fun () -> output_mode := Isa_mode),
     "Set output mode to Isabelle");
    ("-old-lift",
     Arg.Set old_lift,
     "Use backward compatible prelude with Lift"
    );
    ("-output-prelude",
     Arg.Unit output_prelude,
     "Output prelude"
    );

    ("-class",
     Arg.String (fun s ->
		  selected_classes_only := true;
		  selected_classes := s :: !selected_classes),
     "Do selected classes only");
(* 
 *     ("-butclass",
 * 	Arg.String (fun s ->
 * 		      deselected_classes_not := true;
 * 		      deselected_classes := s :: !deselected_classes),
 * 	"Don't do selected classes");
 *)
    ("-d",
     Arg.String set_target_directory,
     "Set target directory for creating output");
(* 
 *     ("-o",
 * 	Arg.String set_target_file,
 * 	"Use one output file instead of multiple files (PVS only)");
 *)
    ("-v",
     Arg.Unit (fun () -> add_debug_level _VERBOSE),
     "Set verbose output");
    ("-nattype",
     Arg.String (fun name -> set_nat_type name),
     "Type used for integers");
    ("-D",
     Arg.Int add_debug_level,
     "Turn on debug output\
     \n          1 - Verbose\
     \n          2 - Lexer comments\
     \n          4 - Parser comments\
     \n          8 - Resolution comments\
     \n         16 - Inheritance comments\
     \n         32 - Typecheck comments\
     \n        128 - Print Symbol table on unknown identifier\
     \n        256 - Apply debug_level also to prelude\
     \n        512 - print information about unification\
     \n       1024 - print assertion/creation condition right after parsing\
     \n       2048 - Variance check comments\
     \n       4096 - Feature check comments\
     \n        Combine debug flag by addition,\
     \n        eg -D 5 means Verbose and Parser comments");
					(* help tuareg fontification "" *)
(* 
 *     ("-unparse",
 * 	Arg.Set (unparse),
 * 	"Unparse the AST");
 *)
    ("-c",
     Arg.Set (filter_style),
     "Print on stdout");
    ("-pedantic",
     Arg.Set (pedantic_mode),
     "Enable pedantic consistency checking");
    ("-expert",
     Arg.Set (expert_mode),
     "Expert mode");

    ("-no-prelude",
     Arg.Clear(parse_prelude),
     "do not parse CCSL prelude");

    ("-prf",
     Arg.Clear (theory_generation),
     "Do not generate any theory files (proof testing)");
    ("-prooftest",
     Arg.String (fun s ->
		  proof_test := true;
		  do_proofs := s :: !do_proofs),
     "Output proof on stdout");
    ("-batch",
     Arg.Set (generate_batch),
     "Generate batch file for theorem prover");
    ("-path",
     Arg.Set (ccsl_generate_paths),
     "Generate theories for inductive characterisation of invariants");
    ("-no-opt",
     Arg.Clear (optimize_expressions),
     "Do not optimize expressions and formulas in the generated output");
    ("-no-inline-lifting",
     Arg.Clear (inline_liftings),
     "Do not inline liftings for non-recursive classes and adt's");
    ("-dependent-assertions",
     Arg.Set(dependent_assertions),
     "Make each assertions dependent on the previous ones");
    ("-fixedpointlib",
     Arg.String (fun s -> ()),
     "No effect; present only for backward compatibility")
  ];;


let anonymous_function name =
  add_input_file name;;


let usage_string =
  "Usage: " ^ Sys.argv.(0) ^ " [option] file\n" ^ "where option is one of";;


(***********************************************************************
 ***********************************************************************
 *
 * common utilities
 *
 *)


let check_suffix_input file =
  if Filename.check_suffix file ".beh"
  then (* it's a CCSL file *)
    add_ccsl_input_file file
  else (* Unknown file format *)
    print_error ("For " ^ file ^ " a parsing is not possible.");;


(***********************************************************************
 ***********************************************************************
 *
 * ccsl utilities
 *
 *)


let ccsl_pass =
  Grammar.file (Abstract_lexer.token @@ Ccsl_hashkeys.hash);;


let ccsl_parse_prelude () =
  let parse_prelude_part s =
    print_verbose "    Reading prelude";
    let prelude_ast =
      ccsl_pass (Lexing.from_string s) 
    in
      print_verbose "    Parsing done ...";
      prelude_ast 
  in
  let _ = parsing_prelude := true in
  let _ = parsing_builtin_prelude := true in
  let _ = Abstract_lexer.initialize "Backend specific prelude" in
  let pvs_p_ast = parse_prelude_part (builtin_prelude !output_mode) in
  let _ = parsing_builtin_prelude := false in
  let _ = Abstract_lexer.initialize "CCSL prelude" in
  let ccsl_p_ast = parse_prelude_part (ccsl_prelude !output_mode) in
  let _ = parsing_prelude := false 
  in
    pvs_p_ast @ ccsl_p_ast;;


let ccsl_parse_file filename =
  let _ = print_verbose ("    Reading classes from " ^ filename) in
  let ic = 
    try
      open_in filename
    with
      | Sys_error msg -> 
	  begin
	    prerr_endline msg;
	    prerr_endline "Aborting";
	    exit 1
	  end
  in
  let _ = Abstract_lexer.initialize filename in
  let file_ast = ccsl_pass (Lexing.from_channel ic) 
  in begin
      print_verbose "    Parsing done ...";
      close_in ic;
      file_ast
    end


(* DELETE
 * let ccsl_dump ast =
 *   dump_top_ast (ast : ccsl_ast list);;
 *)


let ccsl_selected_classes_filter ast =
  List.filter
    (function
       | CCSL_class_dec cl -> List.mem cl#get_name !selected_classes
       | CCSL_adt_dec adt ->  List.mem adt#get_name !selected_classes
       | CCSL_sig_dec _        -> false
    )
    ast;;


(* let ccsl_theory_list ast =
 *   List.fold_left
 *     (fun res ast_el ->
 * 	  let prelude_flag = ref false in 
 * 	  let newths = match ast_el with
 * 	    | CCSL_class_dec cl -> 
 * 		prelude_flag := cl#belongs_to_prelude;
 * 		cl#generate_all_theories
 * 	    | CCSL_adt_dec adt -> 
 * 		prelude_flag := adt#belongs_to_prelude;
 * 		adt#generate_all_theories
 * 	    | CCSL_sig_dec si -> 
 * 		prelude_flag := si#belongs_to_prelude;
 * 		si#generate_all_theories
 * 	  in 
 * 	    if !prelude_flag then
 * 	      List.iter (fun th -> th#override_file_name Names.ccsl_prelude_name)
 * 		newths;
 * 	    res @ newths
 *     )
 *     []
 *     ast;;
 *)


let ccsl_theory_list ast =
  List.flatten 
    (List.map
       (function
	  | CCSL_class_dec cl -> cl#generate_all_theories
	  | CCSL_adt_dec adt -> adt#generate_all_theories
	  | CCSL_sig_dec si -> si#generate_all_theories
       )
       ast)
;;


let make_ccsl_isabelle_file_name th =
  th#override_file_name (th#get_name);;


let ccsl_output_theories thl = 
  match !output_mode with
    | Pvs_mode ->
	print_verbose "    PVS output mode";
      	if !generate_batch
      	then
	  Pvs_batch.register_theories thl;
      	if !proof_test
      	then
	  Syntax.do_proof_test thl
      	else
	  Syntax.pvs_output thl
    | Isa_mode ->
	print_verbose "    Isabelle output mode";
(* 	   List.iter
 * 	     make_ccsl_isabelle_file_name
 * 	     thl;
 *)
      	if !generate_batch
      	then
					(* Pvs_batch works also for isabelle *)
	  Pvs_batch.register_theories thl;
      	if !proof_test
      	then
	  Syntax.do_proof_test thl
      	else
	  Syntax.isar_output thl;;



let do_ccsl_ast ast =
  print_verbose "*** Ccsl update methods ***";
  ccsl_update_method_pass ast;


  print_verbose "*** Ccsl inheritance ***";
  ccsl_inheritance_pass ast; 

  print_verbose "*** Ccsl Update assertions ***";
  ccsl_attribute_assertion_pass ast;

  print_verbose "*** Ccsl Variance ***";
  ccsl_variance_pass (ast);
  
  print_verbose "*** Ccsl Features ***";
  ccsl_feature_pass ast;

  print_verbose "*** Ccsl Class Methods ***";
  ccsl_class_methods_pass ast;

  print_verbose( "*** Ccsl Resolution/Typecheck (" ^
		 (string_of_int (Type_variable.number_of_free_variables ())) ^
		 ")  ***");
  ccsl_resolution_typecheck_pass ast;

  (* DELETE 
   * print_verbose( "*** Ccsl Resolution (" ^
   * 		 (string_of_int (Type_variable.number_of_free_variables ())) ^
   * 		 ")  ***");
   * ccsl_resolution_pass (ast); 
   * 
   * print_verbose( "*** Ccsl_typecheck (" ^
   * 		 (string_of_int (Type_variable.number_of_free_variables ())) ^
   * 		 ")  ***");
   * ccsl_typecheck_class_pass (ast); 
   *)

  print_verbose( "*** Ccsl_typecheck done (" ^
		 (string_of_int (Type_variable.number_of_free_variables ())) ^
		 ")  ***");

  if !pedantic_mode then begin
    print_verbose( "*** Ccsl pedanticness pass ");
    ccsl_pedantic_pass (ast); 
  end;
    
  let p_ast =
    if !selected_classes_only
    then
      ccsl_selected_classes_filter ast
    else
      ast 
  in
  (* DELETE
   * let _ = (if debug_level _DEBUG_DUMP_SYNTAX_TREE
   * 	   then
   * 	     print_string (ccsl_dump (p_ast)))
   * in
   *)
  let _ = print_verbose "*** Ccsl_theory_list ***" in
  let _ = ccsl_output_theories (ccsl_theory_list p_ast) in
  let _ = print_verbose ("*** file  done ***")
  in ()
	       



let do_ccsl_prelude () =
  let dl = get_debug_level () in
  let start_prelude_ths = 
    if !output_mode = Pvs_mode 
    then
      [ new Emptytype_theory.ccsl_empty_type_theory;
	new Emptytype_theory.ccsl_empty_fun_theory;
      ]
    else
      [
	new Emptytype_theory.ccsl_isar_start_prelude;
	new Emptytype_theory.ccsl_isabelle_addon_theory;
      ]
  in let end_prelude_ths = 
    if !output_mode = Pvs_mode 
    then []
    else
      [
	new Emptytype_theory.ccsl_isar_close_prelude;
      ]
  in
    if not (debug_level _DEBUG_PRELUDE) then
      begin
	print_verbose "*** Doing Prelude";
	add_debug_level 0
      end;
					(* do prelude *)
    let all_p_ast = ccsl_parse_prelude () in
      ignore(Symbol.close_block());
      assert( Symbol.nesting_size() = 0 );
      ignore(Symbol.start_block());
      if (start_prelude_ths <> []) & (not !selected_classes_only) then
	ccsl_output_theories start_prelude_ths;
      do_ccsl_ast all_p_ast;
      if (end_prelude_ths <> []) & (not !selected_classes_only) then
	ccsl_output_theories end_prelude_ths;
      add_debug_level dl;
      assert( Symbol.nesting_size() = 1 )
      

let fatal_exception = function
  | Parsing.Parse_error 
  | Abstract_lexer.Include_error
  | Update_Method_Error
  | Inheritance_error
  | Variance_error
  | Feature_error
  | Resolution_error
  | Typecheck_error
  | PedanticViolation -> false
  | _ -> true


let do_ccsl_file file = 
  try
					(* open the file scope *)
    assert( Symbol.nesting_size() = 1 );
    ignore(Symbol.start_block());

    let all_ast = ccsl_parse_file file 
    in
      ignore(Symbol.close_block());
      assert( Symbol.nesting_size() = 1 );
      ignore(Symbol.start_block());
      do_ccsl_ast all_ast;
      assert( Symbol.nesting_size() = 2 ) ;
      ignore(Symbol.close_block())
  with
    | exc -> 
	if fatal_exception exc then
	  (match exc with
	     | Assert_failure (src_file, start_char, end_char) ->
		 print_error (
		   "Assertion in file: " ^ file ^
		   (match !output_mode with
		      | Pvs_mode -> " (pvs_mode)"
		      | Isa_mode -> " (isa_mode)") ^
		   "\nSee " ^ src_file ^ ", char " ^
		   (string_of_int start_char) ^ "-" ^
		   (string_of_int end_char) ^ "\n\n");
		 raise exc;
	     | Internal_error ->
		 print_error (
		   "Internal error in file: " ^ file ^
		   "\n\n");
		 raise exc;
					(* catch all *)
	     | exc -> 
		 print_error ("Exception \n" ^
			      (Printexc.to_string exc) ^
			      "\n thrown in " ^ 
			      file ^ "\n\n");
		 raise exc;
	  )


;;

(***********************************************************************
 ***********************************************************************
 *
 * Start the program
 *
 *)


if not !Sys.interactive
then
  begin

    (* parse command line options *)
    Arg.parse options anonymous_function usage_string;

    (* set on verbose if debug_level <> 0 *)
    if get_debug_level() <> 0 then
      add_debug_level 1;

    (* pedanticness implications *)
    if !pedantic_mode then begin
      if !expert_mode then begin
	print_error "Strict checking and expert mode are mutually exclusive";
	exit 1;
      end;
      if not !optimize_expressions then begin
	print_error "Optimizations are required in pedantic mode";
	exit 1;
      end;
    end;
    

    print_verbose compiler_version;

    if get_input_files () = []
    then
      begin
	print_error "Nothing to do!";
	exit 1
      end;

    (* The input files will be separated into several parts:
     *  ccsl\_input\_files, java\_input\_files, jml\_input\_files
     *)
    List.iter check_suffix_input (get_input_files ());

					(* open first scope *)
    ignore(Symbol.start_block());

    if !parse_prelude then
      do_ccsl_prelude();
    assert( Symbol.nesting_size() = 1 );

    List.iter
      do_ccsl_file
      (get_ccsl_input_files ());

					(* generate batch, if wanted *)
    if !generate_batch
    then
      if !output_mode = Pvs_mode 
      then Pvs_batch.output_pvs_batch Top_names.name_of_pvs_batch
      else Pvs_batch.output_isa_batch Top_names.name_of_isa_batch

  end;;


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
