(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 22.4.99 by Bart
 *
 * Time-stamp: <Thursday 21 August 03 15:01:13 tews@ithif51.inf.tu-dresden.de>
 *
 * top level class structure
 *
 * $Id: top_classes.ml,v 1.17 2003/08/21 15:14:58 tews Exp $
 *
 *)


(*
 * Polymorphic classes are introduced for iface, member and
 * theory_body used at top level, from which classes for all
 * translations (Ccsl, Java, ...) inherit. The type variables
 * like 'member_type, 'theory_body and 'super_class are still
 * used in these implementations.
 *
 * The following three classes are introduced.
 *   ['member_type, 'theory_body_type, 'super_class] top_pre_iface_class
 *   ['class_type] top_pre_member_class
 *   ['class_type] top_pre_theory_body_class
 *
 *)


open Global;;
open Util;;
open Top_variant_types_util;;
open Top_variant_types;;
open Top_classtypes;;


(* DELETE
 * let d level s =
 *   if debug_level level
 *   then begin
 *     prerr_string ("top_classes: " ^ s ^ "\n");
 *     flush stderr
 *   end;;
 * 
 * 
 * let dr s = d _DEBUG_RESOLUTION s;;
 * 
 * 
 * let di s = d _DEBUG_INHERITANCE s;;
 * 
 * 
 * let dt s = d _DEBUG_TYPECHECK s;;
 *)


(***********************************************************************
 ***********************************************************************
 *
 * The top classes themselves.
 *
 *)


class virtual 
  ['member_type, 'theory_body_type, 'super_class] top_pre_iface_class
  name					(* the token *)
  local :				(* the local table *)
  ['member_type, 'theory_body_type, 'super_class] top_pre_iface_type
  = object (s : 'self)

    val name =
	      (name : token_type)

    val mutable file_name =
	      ("" : string)

    val mutable local =
	      (local : ('super_class, 'member_type) top_pre_scope_type)

    val mutable ancestors =
	      ([] : ('super_class,
		     'member_type) top_pre_ancestor_type list)

    val mutable members =
	      ([] : 'member_type list)

					(* public method section *)
    method get_token =
      name

    method get_name =
      name.token_name

    method get_file_name =
      file_name

    method set_file_name file =
      file_name <- file

    method get_local = local

    method set_local new_local = local <- new_local

					(* inheritance pass done ? *)
    val mutable inheritance_ready = false

    method inheritance_done = 
      inheritance_ready <- true

    method inheritance_ready = inheritance_ready
		

    method set_ancestors al =
      ancestors <- al

    method add_ancestor a =
      (ancestors <- ancestors @ [a])

    method get_ancestors = ancestors

    (* return only the _resolved_ ancestor 
     * if applied to a ccsl class, return the instanciated ancestors
     *)
    method get_resolved_ancestors =
      List.map ancestor_resolution_of ancestors

    (* return only the _resolved_ ancestor, recursively *)
    method get_all_ancestors =
      List.fold_left
	(fun result cl -> result @ cl#get_all_ancestors)
	s#get_resolved_ancestors
	s#get_resolved_ancestors

    method string_of_ancestors =
      String.concat
	", "
	(List.map
	  (function
	    | Unresolved_renaming (ifa,_,_,_) -> ifa#get_name
	    | Resolved_renaming (ifa,_,_,_,_) -> ifa#get_name)
	  ancestors)

    method set_members ml =
      members <- ml

    method add_member m =
      (* prerr_endline ("## Add " ^ m#get_name ^ " to " ^ s#get_name); *)
      (members <- members @ [m])

    method get_members =
      members

	(* Certain things rely on the traversion order of 
         * get_all_members !!
         * for instance method coalgebra_as_tuple, 
	 * method struct_of in the interface class,
	 * and ??
	 *)
    method get_all_members =
      List.fold_right
	(fun cl result -> cl#get_all_members @ result)
	s#get_resolved_ancestors
	s#get_members

    method get_attributes =
      List.filter (fun m -> m#is_attribute) members

    method get_methods =
      List.filter (fun m -> m#is_method) members

    method get_all_methods =
      List.filter (fun m -> m#is_method) s#get_all_members

    method get_all_attributes =
      List.filter (fun m -> m#is_attribute) s#get_all_members

    method get_sig_actions = 
      List.filter (fun m -> m#is_sig_action) members

    method get_all_sig_actions =
      List.filter (fun m -> m#is_sig_action) s#get_all_members

    method get_actions = 
      List.filter (fun m -> m#is_action) members

    method get_all_actions =
      List.filter (fun m -> m#is_action) s#get_all_members

    method get_constructors =
      List.filter (fun m -> m#is_constructor) members

    method get_adt_constructors = 
      List.filter (fun m -> m#is_adt_constructor) members


    (**********************************************
     * Features
     *)

    val mutable features = ([] : feature_type list)

    method put_feature f = features <- f :: features

    method has_feature f = List.mem f features


    (**********************************************
     * support for theory generation
     *)
    method virtual generate_all_theories 
      : 'theory_body_type  list


    (**********************************************
     * different specification kinds 
     *)

    val mutable iface_kind = Spec_Spec

    method get_kind = iface_kind

    method become_class = 
      begin
	assert(iface_kind = Spec_Spec);
	iface_kind <- Spec_class
      end

    method become_adt =
      begin
	assert(iface_kind = Spec_Spec);
	iface_kind <- Spec_adt
      end

    method become_sig =
      begin
	assert(iface_kind = Spec_Spec);
	iface_kind <- Spec_sig
      end

    method is_class = iface_kind = Spec_class
    method is_adt = iface_kind = Spec_adt
    method is_sig = iface_kind = Spec_sig


end


class  ['class_type, 'member_type] top_pre_member_class 
  name old_names domain codomain visibility sort 
  : ['class_type, 'member_type] top_pre_member_type
  = object (s)
    val mutable name =
	      (name : token_type)

    val mutable domain =
	      (domain : ('class_type, 'member_type) top_pre_types)

    val mutable codomain =
	      (codomain : ('class_type, 'member_type) top_pre_types)

    val mutable visibility =
	      (visibility : top_visibility)

    val mutable sort =
	      (sort : 'member_type top_pre_member_sort)

	      (* set if this is an inherited method
	       * might become obsolate, if we introduce an orign pointer
	       *)
    val mutable is_inherited =
      false

    val mutable is_hidden =
      false

    method get_token =
      name

    method get_name =
      name.token_name

    method get_domain =
      domain

    method set_domain dom =
      domain <- dom

    method get_codomain =
      codomain

    method set_codomain codom =
      codomain <- codom

    method get_visibility =
      visibility

    method get_sort = sort

    method is_attribute =
      match sort with 
	| Proper_Attribute _ -> true
	| _ -> false

    method is_method = 
       ( sort = Update_Method) ||
       ( sort = Normal_Method)

    method is_defined = sort = Defined_Method
 
    method is_sig_action = s#is_attribute || s#is_method 

    method is_action = s#is_sig_action || s#is_defined

    method is_class_method = 
      ((sort = Class_Coreduce) ||
       (sort = Class_Sig_Special) ||
       (sort = Class_Map))

    method is_constructor =
      ((sort = Var_Constructor) ||
       (sort = Const_Constructor))

    method is_adt_constructor =
      (( sort = Adt_Var_Constructor) ||
       ( sort = Adt_Const_Constructor))

    (*******************************************************************
     *
     * Renaming
     *)

    val mutable old_names = (old_names : token_type list)

    method get_old_names = old_names

      (* true, if this method was renamed in the *last* inheritance
       * step. During class instanciation this flag is reset for all 
       * members of a class
       *)
    val mutable renamed = false

    method clear_renamed_flag =
      renamed <- false

    method add_old_name tok =
      (old_names <- tok :: old_names)

    method is_renamed = renamed

    method rename_member new_name = 
	s#add_old_name name;
	name <- new_name;
	renamed <- true

    method last_name = 
      if renamed then
	match old_names with
	  | [] -> s#get_name
	  | old ::_ -> old.token_name
      else
	s#get_name

    method original_name = 
      match old_names with
	| [] -> s#get_name
	| old ::_ -> (last old_names).token_name


    method is_inherited =
      is_inherited

    method inherited =
      is_inherited <- true

    method is_hidden =
      is_hidden

    method hide =
      is_hidden <- true

  end


class virtual ['class_type, 'member_type] top_pre_theory_body_class 
  : ['class_type, 'member_type] top_pre_theory_body_type
  = object (self : 'self )

    val self_parameter =
	      TypeParameter
		{ id_token = 
		    { token_name = (Top_names.name_of_self ());
		      loc = None };
		  (* the other fields are to ignored *)
		  id_type = Self;
		  id_origin = CCSL_TypeParameter;
		  id_parameters = [];
		  id_variance = Unset;
		  id_sequence = -1;
		}

    method virtual get_name : string

      (* Override this method, if you are really producing an Datatype *)
    method kind = Theory

      (* Override this method, if proofs should not be written *)
    method do_proofs = true

    method virtual get_parameters : 
      ('class_type, 'member_type) top_pre_parameter_type list

    val mutable file_name = ""

    method get_file_name = file_name

    method override_file_name new_name = file_name <- new_name

    method virtual make_body : 
      ('class_type, 'member_type) top_pre_theory_declaration list

    val mutable body = (None : ('class_type, 'member_type) 
		  top_pre_theory_declaration list option)

    method get_body = 
      match body with
      | Some b -> b
      | None -> (let b = self#make_body 
		 in 
		   body <- Some b;
		   b
		)

      (* dependency list (aka imports) for Isar theories *)
    method get_isar_imports = ([] : string list)


    (* extract the proofs from a given theory
     * and give them names, if they don't have yet 
     *
     *)
    method extract_pvs_proofs =
      let doit = function 
    	| Proved(lemma,PvsProof(proof)) -> 
	    (match proof with
	       | Named_proof _ -> Some proof
	       | Anon_proof anon -> 
		   (match lemma with 
		      | Lemma(name,_) -> Some(Named_proof(name,anon))
		  (* jump out if we find a unnamed proof somewhere else 
		   * -> improve this code or supply named proofs
		   *)
		      | _ -> assert(false))
	    )
	| _ -> None
      in
    	option_filter (List.map doit self#get_body)


  end


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
