(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 17.6.99 by Hendrik
 *
 * Time-stamp: <Wednesday 18 September 02 17:26:13 tews@ithif51>
 *
 * the adt theories
 *
 * $Id: adt_theory.ml,v 1.17 2002/10/28 16:33:17 tews Exp $
 *
 *)


open Util
open Global
open Top_variant_types
open Name_space
open Names
open Top_names
open Logic_util
open Classtypes 
open Types_util
open Morphism
open Lifting
open Theory_class


(***********************************************************************
 ***********************************************************************
 *
 * Isar start / close class
 * 
 *
 *)

class ccsl_isar_start_adt adt =
  let pre_imports = 
    List.fold_right
      (fun (v,c,args) accu -> match c#get_kind with
	 | Spec_adt 
	 | Spec_class ->
	     (isar_theory_name c) :: accu
	 | Spec_sig -> 
	     if c#has_own_output
	     then			(* does have an own theory *)
	       (isar_theory_name c) :: accu
	     else 			(* no own theory for c *)
	       (List.map fst c#get_iface_imports) @ accu
	 | Spec_Spec -> assert(false)
      )
      adt#get_components []
  in 
  let imports =
    match remove_duplicates (=) pre_imports with
      | [] -> [isabelle_top_theory]
      | l -> l
  in
    [ccsl_iface_type, ccsl_member_type] 
    ccsl_pre_isabelle_delimiter_theory_class
      adt
      eq_ccsl_types
      IsabelleStartFile
      (ccsl_adt_file_name adt)
      imports
      true				(* do proofs *)


class ccsl_isar_close_adt adt =
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_isabelle_delimiter_theory_class
    adt
    eq_ccsl_types
    IsabelleCloseFile
    (ccsl_adt_file_name adt)
    []
    true				(* do proofs *)

    

(***********************************************************************
 ***********************************************************************
 *
 * Adt declaration for PVS, this is the easy case, simply declare the
 * constructors and accessors
 *
 *)

  
class ['class_type, 'member_type] ccsl_pvs_pre_adt_theory 
  (adt : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_adt_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_adt_theory_class adt eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns []

      (* This is a datatype declaration ! *)
    method kind = Datatype

    method get_name = ccsl_adt_theory_name adt

    method get_parameters = orig_parameters 

    initializer top_theory#override_file_name (ccsl_adt_file_name adt)

(* ISAR
 *     method get_proofs = []
 *)

    method private component_imports =
      let subst_carrier = 
	ccsl_substitute_arguments [Carrier, self#this_adt_type] 
      in
      	Import(
	  List.fold_right
		 (fun (vari,c,args) accu -> match c#get_kind with
		    | Spec_adt -> 
			if c#has_feature BuildinFeature 
			then accu
			else 
			  (ccsl_adt_theory_name c, 
			   []
(* Hendrik: Use precise importings after PVS bug is fixed
		       subst_carrier args
*)
			) :: accu
		    | Spec_class ->
			(if c#has_feature FinalSemanticsFeature
			 then (ccsl_final_theory_name c, args)
			 else (ccsl_loose_theory_name c, args)
			) :: accu
		    | Spec_sig -> 
			if c#has_own_output
			then		(* does have an own theory *)
			  (ccsl_sig_def_theory_name c, args) :: accu
			else		(* no own theory for c *)
			  (if c#instanciate_importings && 
			     adt#instanciate_importings
			   then
			     let subst = make_substitution 
					   c#get_parameters args 
			     in
			       (List.map 
				  (fun (name,args) ->
				     (name, 
				      substitute_arguments_types_only
					eq_types subst args)
				  )
				  c#get_iface_imports
			       )
			   else
			     List.map (fun (name,_) -> (name, [])) 
			       c#get_iface_imports
			  )
			  @ accu
		    | Spec_Spec -> assert(false)
		 )
		 adt#get_components
		 []
      	)

    method private constructor_decl c = 
      let recognizer_name = c#get_recognizer#get_name in
      let accessor_names = List.map (fun m -> m#get_name) c#get_accessors in
      let dom_types = 
	List.map (fun t -> (ccsl_substitute_types 
			      [Carrier, self#this_adt_type] t))
	  (member_arg_list c)
      in
	assert((List.length dom_types) = (List.length c#get_accessors));
	(c#get_name, 
	 List.combine accessor_names dom_types, 
	 recognizer_name)

    method make_body =
      let _ = assert(!output_mode = Pvs_mode)
      in
					(* some fields are ignored in PVS *)
	[
	  self#component_imports;
	  Datatypedecl("", [], 
		       List.map (self#constructor_decl) 
			 adt#get_adt_constructors)
	]
end

class ccsl_pvs_adt_theory adt = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pvs_pre_adt_theory adt eq_ccsl_types



(***************************************************************************
 ***************************************************************************
 *
 *      Adt's for Isabelle, declare the datatype
 *
 *)

class ['class_type, 'member_type] ccsl_isa_pre_adt_theory 
  (adt : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_adt_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_adt_theory_class adt eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns []

    method get_name = ccsl_adt_theory_name adt

    method get_parameters = orig_parameters 

    initializer top_theory#override_file_name (ccsl_adt_file_name adt)

(* ISAR
 *     method get_proofs = []
 *)

    method private component_imports =
      let imp_list = 
	List.fold_right
	  (fun (v,c,args) accu -> match c#get_kind with
	     | Spec_adt -> (ccsl_adtutil_theory_name c, []) :: accu
	     | Spec_class ->
		 if c#has_feature FinalSemanticsFeature
		 then (ccsl_final_theory_name c, args) :: accu
		 else (ccsl_loose_theory_name c, args) :: accu
	     | Spec_sig -> 
		 if c#has_own_output
		 then			(* does have an own theory *)
		   (ccsl_sig_def_theory_name c, args) :: accu
		 else 			(* no own theory for c *)
		   (if c#instanciate_importings && 
		      adt#instanciate_importings
		    then
		      let subst = make_substitution c#get_parameters args 
		      in
			(List.map 
			   (fun (name,args) ->
			      (name, 
			       substitute_arguments_types_only
				 eq_types subst args)
			   )
			   c#get_iface_imports
			)
		    else
		      List.map (fun (name,_) -> (name, [])) 
			c#get_iface_imports
		   )
		   @ accu
	     | Spec_Spec -> assert(false)
	  )
	  adt#get_components []
      in
      	Import(
	  (isabelle_top_theory,[]) :: imp_list
      	)

					(* Isabelle declaration of datatype *)
    method private datatype_decl =
      let _ = assert(!output_mode = Isa_mode) in
      let do_constr c = 
      	let dom_types = 
	  List.map (fun t -> (ccsl_substitute_types 
			      	[Carrier, self#this_adt_type] t))
	    (member_arg_list c)
      	in
	  (c#get_name, 
	   List.map (fun x -> "",x) dom_types, 
	   "") 
      in
	Datatypedecl( adt#get_name, orig_parameters, 
		  List.map do_constr adt#get_adt_constructors)


    method make_body =
      [self#component_imports;
       self#datatype_decl
      ]
end

class ccsl_isa_adt_theory adt = 
  [ccsl_iface_type, ccsl_member_type] ccsl_isa_pre_adt_theory 
  adt eq_ccsl_types



(***************************************************************************
 ***************************************************************************
 *
 * Adt's for Isabelle, define accessors and 
 * recognizers later self.
 *
 *)

class ['class_type, 'member_type] ccsl_isa_pre_adtutil_theory 
  (adt : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_adt_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_adt_theory_class adt eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns []

    method get_name = ccsl_adtutil_theory_name adt

    method get_parameters = orig_parameters 

    initializer top_theory#override_file_name (ccsl_adt_file_name adt)

(* ISAR
 *     method get_proofs = []
 *)


					(* define accessors as primrec's *)
    method private accessors constr = 
      let i = ref ( -1) in
      let dom_types = 
	List.map (fun t -> (ccsl_substitute_types 
			      [Carrier, self#this_adt_type] t))
	  (member_arg_list constr) in
      let do_acc acc_name arg_type =
	Primrecdefn(
	  acc_name, 
	  Function(self#this_adt_type, arg_type),
	  List.map
	    (fun o_constr ->
	       let sub_ns = sub_space ns in
	       let o_dom_types = 
		 List.map (fun t -> (ccsl_substitute_types 
				       [Carrier, self#this_adt_type] t))
		   (member_arg_list o_constr) in
	       let ids = (create_ids sub_ns o_dom_types) in
	       let terms = List.map (fun (n,_) -> Term(n,Always,[])) ids in
					(* do smart application now *)
	       let arg = terms in
	       let left_ex =
		 Application(
		   Term(acc_name,Always,[]),
		   match terms with
		     | [] -> Term(o_constr#get_name,Always,[])
		     | _ -> 
			 SmartApplication(
			   Term(o_constr#get_name,Always,[]),
			   arg)) in
	       let right_ex =
		 if constr == o_constr then
		   List.nth terms !i
		 else Term("arbitrary",Always,[]) 
	       in
		 left_ex, right_ex
	    ) adt#get_adt_constructors
	) in
	if constr#get_accessors = [] 
	then []
	else
	  List.map2
	    (fun acc typ ->
	       incr i;
	       do_acc acc#get_name typ)
	    constr#get_accessors dom_types


					(* define accessors with cases *)
    method private accessors_case constr = 
      let i = ref ( -1) in
      let dom_types = 
	List.map (fun t -> (ccsl_substitute_types 
			      [Carrier, self#this_adt_type] t))
	  (member_arg_list constr) in
      let sub_ns = sub_space ns in
      let x = create_one_id sub_ns self#this_adt_type  in
      let do_acc acc_name arg_type =
	Defn(
	  acc_name, 
	  [[Undeclared(x, self#this_adt_type)]],
	  arg_type,
	  Case(
	    Term(x,Always,[]),
	    List.map
	      (fun o_constr ->
		 let ssub_ns = sub_space sub_ns in
		 let o_dom_types = 
		   List.map (fun t -> (ccsl_substitute_types 
					 [Carrier, self#this_adt_type] t))
		     (member_arg_list o_constr) in
		 let ids = (create_ids ssub_ns o_dom_types) in
		 let terms = List.map (fun (n,_) -> Term(n,Always,[])) ids in
					(* do smart application now *)
		 let left_ex =
		   if o_constr#get_sort = Adt_Const_Constructor
		   then Term(o_constr#get_name,Always,[])
		   else SmartApplication(Term(o_constr#get_name,Always,[]),
					 terms)
		 in
		 let right_ex =
		   if constr == o_constr 
		   then List.nth terms !i
		   else Term("arbitrary",Always,[]) 
		 in
		   left_ex, right_ex
	      ) adt#get_adt_constructors
	  ))
      in
	if constr#get_accessors = [] 
	then []
	else
	  List.map2
	    (fun acc typ ->
	       incr i;
	       do_acc acc#get_name typ)
	    constr#get_accessors dom_types


					(* define recognizers as primrecs *)
    method private recognizers constr =
      let rec_name = constr#get_recognizer#get_name in
	Primrecdefn(
	  rec_name,
	  Function(self#this_adt_type, Bool),
	  List.map 
	    (fun o_constr -> 
	       let sub_ns = sub_space ns in
	       let o_dom_types = 
		 List.map (fun t -> (ccsl_substitute_types 
				       [Carrier, self#this_adt_type] t))
		   (member_arg_list o_constr) in
	       let ids = (create_ids sub_ns o_dom_types) in
	       let terms = List.map (fun (n,_) -> Term(n,Always,[])) ids in
					(* do smart application now *)
	       let arg = terms in
	       let left_ex =
		 Application(
		   Term(rec_name,Always,[]),
		   match terms with
		     | [] -> Term(o_constr#get_name,Always,[])
		     | _ -> 
			 SmartApplication(
			   Term(o_constr#get_name,Always,[]),
			   arg)) in
	       let right_ex = 
		 if constr == o_constr then 
		   Expression(True)
		 else 
		   Expression(False)
	       in
		 left_ex, right_ex
	    ) 
	    adt#get_adt_constructors)


					(* define recognizers as reduce *)
    method private recognizers_red constr =
      let rec_name = constr#get_recognizer#get_name in
      let sub_ns = sub_space ns in
      let x = create_one_id sub_ns self#this_adt_type  in
      let constr_fun o_constr = 
	let ssub_ns = sub_space sub_ns in
	let o_dom_types = 
	  List.map (fun t -> (ccsl_substitute_types 
				[Carrier, Bool] t))
	    (member_arg_list o_constr) in
	let ids = (create_ids ssub_ns o_dom_types) in
	let t_f = 
	  if constr == o_constr 
	  then Expression(True)
	  else Expression(False)
	in	
	  if o_constr#get_sort = Adt_Const_Constructor
	  then t_f
	  else SmartAbstraction(ids, t_f)
      in
	Defn(
	  rec_name,
	  [],
	  Function(self#this_adt_type, Bool),
	  Abstraction
	    ([x, self#this_adt_type],
	     Application(
	       SmartApplication(
		 Term(name_of_adt_reduce adt, Always, []),
		 List.map 
		   constr_fun 
		   adt#get_adt_constructors),
	       Term(x,Always,[])
	     )))


    (* Define a *useable* reduce in isabelle 
     * 
     * The _rec combinator provided by isabelle from an expanded adt.
     * Expanded means here, that all components are expanded into one 
     * nested adt, see diploma theses of Stefan Berghofer for details.
     * 
     * While processing the constructors I register complex types via
     * extract_complex_types in a list. These complex types are component
     * adt's or products, that lead to an expansion in isabelle. These 
     * complex types give rise to further arguments for the _rec combinator. 
     * 
     * During the expansion I substitute all adt's that come along with a 
     * unique type variable. This way it is easy to keep track of which 
     * types need further processing.
     * 
     * Another problem is that the _rec combinator is a strong reduce. 
     * I filter the arguments for one constructor in create_split_args 
     * and double and select appropriately.
     *)

    method private reduce_rec =
      (* Debugging support *)
      let adt_name = adt#get_name in
      (* *)
      let sub_ns = sub_space ns in
      let res_name = create_id_with_preference ns "Result" "T" in
      let res_type = 
	BoundTypeVariable (id_record_from_string res_name) in
      let smart_type c =
	if c#get_sort = Adt_Const_Constructor 
	then c#get_codomain
	else SmartFunction(member_arg_list c,c#get_codomain)
      in
      let fun_args_ids =
	create_ids_with_preference sub_ns
	  (List.map 
	     (fun c -> 
		((c#get_name ^ "_fun"),
		 (substitute_types_only eq_types
		    [Carrier,res_type]
		    (smart_type c)))
	     )
	     adt#get_adt_constructors)
      in
      let fun_args_terms = 
	List.map (fun (n,t) -> Term(n,Always,[])) fun_args_ids in
      let partial_reduce = SmartApplication
			     (Term(name_of_adt_reduce adt,Always,[]),
			      fun_args_terms)
      in let x = create_one_id sub_ns self#this_adt_type 
      in
      let do_subst subst typ = 
	substitute_types_only eq_types subst typ 
      in
      let do_args_subst subst arg_list = 
	List.map 
	  (function TypeArgument t -> TypeArgument( do_subst subst t))
	  arg_list
      in let subst_to_adt typ =
	do_subst [Carrier, self#this_adt_type] typ
      in 
      let subst_to_res typ =
	do_subst [Carrier, res_type] typ
      in 
      let decl_subst subst_fun =
	List.map (fun (n,t) -> (n, subst_fun t))
      in
      let create_split_args ns csubst arglist = 
	let subst_args = 
	  List.map (do_subst csubst) arglist 
	in
	let orig_args, red_args = 
	  (create_id_pairs ns subst_args) in
	let r_params,args =
	  List.fold_right
	    (fun ((on,ot), (rn,rt)) (r_params,args) ->
	       if count_carrier ot <> 0		      (* not constant *)
	       then ((rn,rt)::r_params, (rn,rt) :: args)
	       else (r_params, (on,ot) :: args)
	    )
	    (List.combine orig_args red_args)
	    ([],[])
	in
	  (orig_args,r_params,args)

      in 

				(* function type is primitive in isabelle *)
      let rec remove_fun_type = function
	| Function(dom,codom) -> 
				(* carrier appears only stricly covariant *)
	    assert(not (type_is_nonground dom));
	    remove_fun_type codom
	| t -> t
      in

      let do_red_constr constr c_fun =
	let ssub_ns = sub_space sub_ns in
	let name : string = constr#get_name in
	let arglist = member_arg_list constr in
	let o_params,r_params,args = create_split_args ssub_ns [] arglist
	in let body =
	    if constr#get_sort = Adt_Const_Constructor 
	    then c_fun
	    else 
	      SmartApplication
		(c_fun,
		 (List.map (fun (n,t) -> Term(n,Always,[])) args))
	in let red_fun = 
	    SmartAbstraction
	      ((decl_subst (subst_to_adt) o_params) @ 
	       (decl_subst (subst_to_res) r_params) ,
	       body
	      )
	in
	  red_fun, arglist
      in
      let do_map_constr constr csubst =
	let ssub_ns = sub_space sub_ns in
	let name = constr#get_name in
	let arglist = member_arg_list constr in
	let o_params,r_params,args = 
	  create_split_args ssub_ns csubst arglist
	in let body = 
	    if constr#get_sort = Adt_Const_Constructor 
	    then Term(name,Always,[])
	    else 
	       SmartApplication
		 (Term(name,Always,[]),
		  (List.map (fun (n,t) -> Term(n,Always,[])) args)
		 )
	in let map_fun = 
	    SmartAbstraction
	      ((decl_subst (subst_to_adt) o_params) @ 
	       (decl_subst (subst_to_res) r_params),
	       body
	      )
	in
	  map_fun, arglist
      in

      let rec process_arg_types csubst arg_types =
	let fun_list_list, arg_types_list =
	  List.split
	    (List.map (process_arg_typ csubst) arg_types)
	in 
	  (List.flatten fun_list_list,
	   List.flatten arg_types_list)
      and process_arg_typ csubst typ =
	if count_carrier typ = 0
	then ([], [typ])
	else
	  map_type csubst (remove_fun_type typ)
	
      and map_type csubst  = function
	| Carrier -> [],[]
	| Adt(adt,flag,adt_args) as this_adt ->
	    let subst_args = do_args_subst csubst adt_args in
	    let args_subst = 
	      (make_substitution adt#get_parameters subst_args) in
	    let nsubst = (Carrier, Adt(adt,flag,subst_args)) :: args_subst in
	    let funs1, arg_types1 = 
	      List.split
		(List.map
		   (fun c  -> do_map_constr c nsubst)
		   adt#get_adt_constructors)
	    in let funs2, arg_types2 = 
		process_arg_types nsubst (List.flatten arg_types1) in
	    let arg_types3 = 
	      List.map (do_subst 
			  (make_substitution adt#get_parameters adt_args))
		arg_types2
	    in let funs3, arg_types4 = 
		process_arg_types csubst arg_types3
	    in
	      ((funs1 @ funs2 @ funs3), arg_types4)

	| Product([]) 
	| Product([_])
	    -> assert(false)
	| Product([t1;t2] as tl) ->
	    let ssub_ns = sub_space sub_ns in
	    let o_params,r_params,args = create_split_args ssub_ns csubst tl 
	    in let map_fun1 =
		SmartAbstraction
		  ((decl_subst subst_to_adt o_params) @ 
		   (decl_subst subst_to_res r_params),
		   Tuple
		     (List.map (fun (n,t) -> Term(n,Always,[])) args)
		  )
	    in let map_fun2, arg_types = process_arg_types csubst tl
	    in
	      (map_fun1 :: map_fun2), arg_types

		(* n-ary products are nested in isabelle *)
	| Product(t::tl) ->
	    map_type csubst 
	      (Product([t; Product(tl)]))

	| t -> 
	  (*  Debugging support *)
	    let n = string_of_int( 9 )
	    in let x = [t;t;t]
	    in
	    assert(false)
      in
      let red_funs,arg_types_list = 
	List.split
	  (List.map2
	     (fun c c_fun ->
		do_red_constr c c_fun
	     )
	     adt#get_adt_constructors
	     fun_args_terms
	  )
      in
      let map_funs,const_types = 
	process_arg_types [] (List.flatten arg_types_list)
      in
      let name_of_internal_isabelle_reduce =
	adt#get_name ^ "_rec" ^ (if map_funs = [] then "" else "_1")
      in
	Defn(
	  name_of_adt_reduce adt,
	  [(List.map (fun (n,t) -> Undeclared(n,t)) fun_args_ids)],
	  Function(
	    self#this_adt_type,
	    res_type),
	  Abstraction(
	    [x,self#this_adt_type],
	    Application
	      (SmartApplication
		 (Term(name_of_internal_isabelle_reduce,Always,[]),
		  red_funs @ map_funs),
		 Term(x,Always,[]))
	))
	  
      

(* 
 * 	     (* define reduce via primrec 
 * 	      * does not work in isabell for nested adt's 
 * 	      *)
 *     method private reduce_primrec =
 * 	 let sub_ns = sub_space ns in
 * 	 let res_name = create_id_with_preference ns "Result" "T" in
 * 	 let res_type = 
 * 	   BoundTypeVariable (id_record_from_string res_name) in
 * 	 let subst = [Carrier, res_type] in
 * 	 let smart_type c =
 * 	   if c#get_sort = Adt_Const_Constructor 
 * 	   then c#get_codomain
 * 	   else match c#get_domain with
 * 	     | Product tl -> SmartFunction(tl,c#get_codomain)
 * 	     | t -> SmartFunction([t],c#get_codomain)
 * 	 in
 * 	 let fun_args_ids =
 * 	   create_ids_with_preference sub_ns
 * 	     (List.map 
 * 		(fun c -> 
 * 		   ((c#get_name ^ "_fun"),
 * 		    substitute_types_only eq_types subst 
 * 		      (smart_type c))
 * 		)
 * 		adt#get_adt_constructors)
 * 	 in
 * 	 let type_of_reduce =
 * 	   SmartFunction(
 * 	     (List.map snd fun_args_ids),
 * 	     Function(
 * 	       self#this_adt_type,
 * 	       res_type))
 * 	 in 
 * 	 let fun_args_terms = 
 * 	   List.map (fun (n,t) -> Term(n,Always,[])) fun_args_ids in
 * 	 let partial_reduce = SmartApplication
 * 				(Term(name_of_adt_reduce adt,Always,[]),
 * 				 fun_args_terms)
 * 	 in
 * 	 let do_constructor constr c_fun = 
 * 	   let ssub_ns = sub_space sub_ns in
 * 	   let dom = constr#get_domain in
 * 	   let codom = constr#get_codomain in 
 * 	   let name = constr#get_name in
 * 	   let arglist = match dom with
 * 	     | Product tl -> tl 
 * 	     | t -> [t] in
 * 	   let typed_args = (create_ids ssub_ns arglist) in
 * 	   let args =  (* SmartApplicatin deals with one tuples*)
 * 	     List.map (fun (v,_) -> Term(v,Always,[])) typed_args
 * 	   in
 * 	   let pre_left = 
 * 	     if constr#get_sort = Adt_Const_Constructor 
 * 	     then Term(constr#get_name, Always, [])
 * 	     else SmartApplication(Term(constr#get_name, Always, []), args) in
 * 	   let left = 
 * 	     Application(partial_reduce, pre_left) in
 * 	   let right = 
 * 	     if constr#get_sort = Adt_Const_Constructor
 * 	     then c_fun
 * 	     else
 * 	       SmartApplication(
 * 		 c_fun,
 * 		 List.map2
 * 		   (fun t a -> 
 * 		      functor_action
 * 			ssub_ns			      (* name_space *)      
 * 			false			      (* id_crea_flag *)    
 * 			[Carrier,			      (* actions *)
 * 			 (None, Some partial_reduce)]
 * 			true			      (* variance *)        
 * 			[Carrier,			      (* domain_subst *)    
 * 			 (self#this_adt_type, self#this_adt_type)]
 * 			[Carrier,			      (* codomain subst *)  
 * 			 (res_type, res_type)]
 * 			a				      (* expression *)      
 * 			t				      (* typ *)
 * 		   )
 * 		   (List.map (fun (n,t) -> t) typed_args)
 * 		   args
 * 	       )
 * 	   in
 * 	     left, (opt_expression right)
 * 	 in
 * 	   Primrecdefn(
 * 	     name_of_adt_reduce adt,
 * 	     type_of_reduce,
 * 					   (* the equations *)
 * 	     List.map2
 * 	       do_constructor 
 * 	       adt#get_adt_constructors
 * 	       fun_args_terms
 * 	   )
 * 	   
 *)




    method make_body =
(* PRIMREC 
 * 	 @
 * 	 List.flatten(
 * 	   List.map self#accessors adt#get_adt_constructors
 * 	 )
 * 	 @
 * 	 List.map self#recognizers adt#get_adt_constructors
 *)
      [self#reduce_rec]
      @
      List.flatten(
        List.map self#accessors_case adt#get_adt_constructors
      )
      @
      (List.map self#recognizers_red adt#get_adt_constructors)
end

class ccsl_isa_adtutil_theory adt = 
  [ccsl_iface_type, ccsl_member_type] ccsl_isa_pre_adtutil_theory 
  adt eq_ccsl_types


(********************************************************************** 
 * predicate lifting for adt's,
 * 
 * Since there is no pattern matching in isabelles primrecs, it 
 * seems easier to define predicate lifting with a reduce. 
 * Then this theory can also used for PVS, when PVS fails to 
 * define predicate lifting.
 *)

class ['class_type, 'member_type] ccsl_pre_adt_every_theory 
  (adt : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_adt_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_adt_theory_class adt eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns []

    method get_name = ccsl_adt_every_theory_name adt

    method get_parameters = orig_parameters

    initializer top_theory#override_file_name (ccsl_adt_file_name adt)

(* ISAR
 *     method get_proofs = []
 *)

    method private import =
      Import(
	(match !output_mode with
	   | Isa_mode -> (ccsl_adtutil_theory_name adt,[])
	   | Pvs_mode -> (ccsl_adt_theory_name adt, orig_arguments)
	)::
	List.fold_right
	       (fun (v,comp,args) accu ->
		  match comp#get_kind with
		    | Spec_adt -> 
			if comp#has_feature NeedsMapFeature
			then (ccsl_adt_every_theory_name comp, args) :: accu
			else accu
		    | Spec_class ->
			(ccsl_greatest_invariance_theory_name comp, 
			 (comp#get_model_type_argument args) :: args) 
			:: accu
		    | Spec_sig -> accu
		    | Spec_Spec -> assert(false)
	       )
	       adt#get_components
	       []
      )


(* 	(* predicate lifting via primrec, does not work for nested adts *)
 *     method private every =
 * 	    let sub_ns = sub_space ns in
 * 	    let type_param_ids = 
 * 	      List.map (function TypeParameter id -> id)
 * 		adt#get_parameters 
 * 	    in
 * 				      (* compute the types of the predicates *)
 * 	    let pred_types = 
 * 	      List.map
 * 		(fun id -> id, Function (BoundTypeVariable(id), Bool))
 * 		type_param_ids
 * 	    in				(* create identifiers *)
 * 	    let pred_ids_var = 
 * 	      create_ids_with_variance sub_ns 
 * 		(List.map (fun (id,t) -> t,id.id_variance) pred_types)
 * 	    in
 * 	    let build_term name = Some(Term(name,Always,[])) in
 * 	    let param_lifting = 
 * 	      List.map2
 * 		(fun id -> function
 * 		   | Some(name, _), None -> (id, (build_term name, None))
 * 		   | None, Some(name, _) -> (id, (None, build_term name))
 * 		   | Some (name1,_), Some(name2,_) ->
 * 		       (id, (build_term name1, build_term name2))
 * 		   | _ -> assert(false)
 * 		)
 * 		type_param_ids pred_ids_var
 * 	    in
 * 	    let pred_ids = variance_flatten pred_ids_var in
 * 	    let pred_exprs = List.map (fun (id,t) -> Term(id,Always,[])) pred_ids
 * 	    in
 * 	    let lift_pred = Every(self#this_adt_type, pred_exprs) in
 * 	    let carrier_subst = 
 * 	      ccsl_substitute_types [Carrier, self#this_adt_type]
 * 
 * 	    in
 * 	      Primrecdefn(
 * 		name_of_adt_every adt,
 * 		(if type_param_ids = [] 
 * 					      (* adt without parameters *)
 * 		 then Function(self#this_adt_type, Bool)
 * 		 else				(* adt with parameters *)
 * 		   SmartFunction(
 * 		     (List.map (fun (n,t) -> t) pred_ids),
 * 		     Function(self#this_adt_type, Bool))),
 * 					      (* the equations *)
 * 		List.map 
 * 		  (fun c -> 
 * 		     let left,right = (adt_constr_predicate_lifting 
 * 					 carrier_subst
 * 					 param_lifting sub_ns
 * 					 (Formula(lift_pred)) c) in
 * 		       (Application(
 * 			  lift_pred,
 * 			  left),
 * 			right)
 * 				     
 * 		  )
 * 		  adt#get_adt_constructors)
 * 
 *)


	     (* every via reduce, as in my thesis *)
    method private every_red =
      let sub_ns = sub_space ns in
      let type_param_ids = 
	List.map (function TypeParameter id -> id)
	  adt#get_parameters 
      in
				   (* compute the types of the predicates *)
      let pred_types = 
	List.map
	  (fun id -> id, Function (BoundTypeVariable(id), Bool))
	  type_param_ids
      in				(* create identifiers *)
      let pred_ids_var = 
	create_ids_with_variance sub_ns 
	  (List.map (fun (id,t) -> t,id.id_variance) pred_types)
      in
      let param_lifting = var_terms_from_var_ids type_param_ids pred_ids_var
      in
      let pred_ids = variance_flatten pred_ids_var in
      let pred_exprs = List.map (fun (id,t) -> Term(id,Always,[])) pred_ids
      in
      let carrier_subst = ccsl_substitute_types [Carrier, Bool] in
      let ttexp = Abstraction
		    (["z",Bool], 
		     Expression (Equal(Term("z",Always,[]), 
				       Expression(True)))) in
      let self_lifting = (None, Some(ttexp))
      in
      let do_constr constr =
	let ssub_ns = sub_space sub_ns  in
          (* lifting for constructors *)
	let dom = constr#get_domain in
	let arglist = match dom with
	  | Product tl -> tl 
	  | t -> [t] in
	let typed_args = create_ids ssub_ns
			   (List.map carrier_subst arglist) in
	let c_lift = 
	  SmartAbstraction(
	    typed_args,
	    Expression(
	      And(
		List.map2 
		    (fun (v,_) typ -> 
		       fullpredlift
			 eq_types
			 carrier_subst		     (* self_subst *)
			 param_lifting	             (* param_lifting *)
			 self_lifting		     (* self_lifting *)
			 ssub_ns		     (* top_name_space *)
			 typ			     (* typ *)
			 (Term(v,Always,[]))	     (* expression *)
		    )
		    typed_args arglist)))
	in
	  opt_expression c_lift
      in let body = 
	  if !Global.inline_liftings &&
	    make_simple adt#get_self_variance = Unused
	  then				(* lifting as case expression *)
	    let x = create_one_id sub_ns self#this_adt_type 
	    in
	      Abstraction(
		[x,self#this_adt_type],
		inline_adt_pred_lifting 
		  eq_types
		  carrier_subst
		  param_lifting 
		  self_lifting 
		  sub_ns 
		  adt
		  (arguments_from_parameters adt#get_parameters)
		  (Term(x,Always,[]))
	      )
	  else				(* lifting as reduce *)
	    SmartApplication(
	      Term(name_of_adt_reduce adt,Always,[]),
	      List.map (do_constr)
		adt#get_adt_constructors
	    )
      in
	Defn(
	  name_of_adt_every adt,
	  [(List.map (fun (n,t) -> Undeclared(n,t)) pred_ids)],
	  Function(self#this_adt_type, Bool),
	  body
	)




    method make_body =
      [self#import;
       self#every_red]
end

class ccsl_adt_every_theory adt = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_adt_every_theory 
  adt eq_ccsl_types


  
(********************************************************************** 
 * relation lifting for adt's,
 * 
 * Since there is no pattern matching in isabelles primrecs, it 
 * seems easier to define relation lifting with a reduce. 
 *)

class ['class_type, 'member_type] ccsl_pre_adt_rellift_theory 
  (adt : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_adt_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_adt_theory_class adt eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      [
	name_of_adt_rel_every adt
      ]

    method get_name = ccsl_adt_rellift_theory_name adt

    method get_parameters = 
      let (p1,p2) = self#double_parameters in
	p1 @ p2

    initializer top_theory#override_file_name (ccsl_adt_file_name adt)

(* ISAR
 *     method get_proofs = []
 *)

    method private adt_import = 
      let (param_left,param_right) = self#double_parameters in
      let (self1_subst, self2_subst) = 
    	adt_rel_lifting_substitutions adt param_left param_right in
      let comps = 
	List.fold_right
	  (fun (v,c,args) accu ->
	     match c#get_kind with
	       | Spec_adt -> 
		   (ccsl_adt_rellift_theory_name c,
		    (ccsl_substitute_arguments self1_subst args)
		    @ (ccsl_substitute_arguments self2_subst args)) :: accu
	       | Spec_class ->
		   (ccsl_greatest_bibisim_theory_name c, []) :: accu
	       | Spec_sig -> accu
	       | Spec_Spec -> assert(false)
	  )
	  adt#get_components []
      in
      let this_adt_theory = match !output_mode with
	| Isa_mode -> ccsl_adtutil_theory_name adt
	| Pvs_mode -> ccsl_adt_theory_name adt
      in	
	Import(
	  (this_adt_theory, arguments_from_parameters param_left)
	  :: (this_adt_theory, 
	      arguments_from_parameters param_right)
	  :: comps
	)

    method private rellift =
  (* debugging support 
      let adt_name = adt#get_name in
   *)
      let sub_ns = sub_space ns in
      let type_param_ids param_list = 
	List.map (function TypeParameter id -> id) param_list in
      let (param_left,param_right) = self#double_parameters in
      let adt_param_ids = type_param_ids adt#get_parameters 
      in
      let rel_types = 
	Util.map3 (fun idl idr oid ->
		     (Function(
			Product([BoundTypeVariable idl; 
				 BoundTypeVariable idr]),
			Bool),
		      oid.id_variance
		     ))
	  (type_param_ids param_left)
	  (type_param_ids param_right)
	  adt_param_ids
      in
      let rel_ids_var = 
	create_ids_with_variance sub_ns rel_types
      in
      let param_rel_lifting = 
	var_terms_from_var_ids adt_param_ids rel_ids_var
      in
      let rel_ids = variance_flatten rel_ids_var in
      let rel_exprs = List.map (fun (id,t) -> Term(id,Always,[])) rel_ids
      in
      let adt_left = 
	Adt(adt, Always, arguments_from_parameters param_left) in 
      let adt_right = 
	Adt(adt, Always, arguments_from_parameters param_right) in
      let adt_args = create_ids sub_ns [adt_left; adt_right] in
      let x1term = Term(fst(List.nth adt_args 0), Always,[]) in 
      let x2term = Term(fst(List.nth adt_args 1), Always,[]) in
      let body = 
	if !Global.inline_liftings && 
	  make_simple adt#get_self_variance = Unused
	then				(* lifting as case expression *)
	  let carrier_subst1 = 
	      substitute_types_only eq_types 
		(make_substitution_param_param adt#get_parameters param_left)
	  in let carrier_subst2 = 
	      substitute_types_only eq_types 
		(make_substitution_param_param adt#get_parameters param_right)
	  in
	      inline_adt_rel_lifting 
		eq_types				(* eq_types *)
		carrier_subst1				(* self1_subst *)
		carrier_subst2				(* self2_subst *)
                param_rel_lifting			(* param_rel_lifting *)
                (None,None)				(* self_lifting *)
                sub_ns					(* top_name_space *)
                adt					(* adt  *)
		x1term					(* expr1 *)
                x2term					(* expr2  *)
	else
	  let constructor_funs = 
 	    adt_rel_lifting
	      param_left
	      param_right
	      param_rel_lifting
	      sub_ns
	      adt 
	  in
	    Application(
	      Application(
		SmartApplication(
	      	  Term(name_of_adt_reduce adt,Always,[]),
		  constructor_funs),
		x1term),
	      x2term)
      in
      	Defn(
	  name_of_adt_rel_every adt,
	  [List.map (fun (id,t) -> Undeclared(id, t)) rel_ids],
					(* the type *)
	  Function(
	    Product(
	      [adt_left; adt_right]),
	    Bool),
					(* the definition *)
	  Abstraction(
	    adt_args,
	    body
	  ))

    method make_body =
      [
	self#adt_import;
	self#rellift
      ]

end

class ccsl_adt_rellift_theory adt = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_adt_rellift_theory 
  adt eq_ccsl_types



(********************************************************************** 
 * 
 * Map combinator for adt's,
 * 
 *)

class ['class_type, 'member_type] ccsl_pre_adt_map_theory 
  (adt : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_adt_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_adt_theory_class adt eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      [
	name_of_adt_map adt
      ]

    method get_name = ccsl_adt_map_theory_name adt

    method get_parameters = 
      let (p1,p2) = self#double_parameters in
	p1 @ p2

    initializer top_theory#override_file_name (ccsl_adt_file_name adt)

(* ISAR
 *     method get_proofs = []
 *)

    method private adt_import = 
      let (param_from,param_to) = self#double_parameters in
      let adt_from = Adt(adt,Always, arguments_from_parameters param_from) in
      let adt_to = Adt(adt,Always, arguments_from_parameters param_to) in
      let from_subst = 
	(Carrier, adt_from) ::
	(make_substitution_param_param adt#get_parameters param_from)
      in let to_subst = 
	  (Carrier, adt_to) ::
	  (make_substitution_param_param adt#get_parameters param_to)
      in
      let comps = 
	List.fold_right
	  (fun (v,c,args) accu ->
	     let from_args = ccsl_substitute_arguments from_subst args in
	     let to_args = ccsl_substitute_arguments to_subst args
	     in
	     let rec do_v th_name accu = function
	       | Unused
	       | Pos -> (th_name, from_args @ to_args) :: accu
	       | Neg -> (th_name, to_args @ from_args) :: accu
	       | Mixed -> do_v th_name (do_v th_name accu Pos) Neg
	       | _ -> assert(false)
	     in
	       match c#get_kind with
		 | Spec_adt -> 
		     if c#has_feature NeedsMapFeature
		     then do_v (ccsl_adt_map_theory_name c) accu v
		     else accu
	       | Spec_class ->
		   do_v (ccsl_map_theory_name c) accu v
	       | Spec_sig -> accu
	       | Spec_Spec -> assert(false)
	  )
	  adt#get_components 
	  []
      in
      let this_adt_theory = match !output_mode with
	| Isa_mode -> ccsl_adtutil_theory_name adt
	| Pvs_mode -> ccsl_adt_theory_name adt
      in	
	Import(
	  (this_adt_theory, 
	   arguments_from_parameters param_from)
	  :: (this_adt_theory, 
	      arguments_from_parameters param_to)
	  :: comps
	)


    method private adt_map =
(* debugging support 
      let adt_name = adt#get_name in
*)
      let sub_ns = sub_space ns in
      let type_param_ids param_list = 
	List.map (function TypeParameter id -> id) param_list in
      let (param_from,param_to) = self#double_parameters in
      let adt_param_ids = type_param_ids adt#get_parameters 
      in
      let fun_types = 
	Util.map3 (fun t_from t_to o_id ->
		     (Function(t_from, t_to), 
		      o_id.id_variance))
	  (types_from_parameters param_from)
	  (types_from_parameters param_to)
	  adt_param_ids
      in
      let pre_fun_ids_var = 
	create_ids_with_variance sub_ns fun_types
      in
      let fun_ids_var =			(* toggle direction for negatives *)
	List.map
	  (function
	     | (None, _) as x -> x
	     | (Some(neg_name, Function(dom,codom)), pos_id) ->
		 (Some(neg_name, Function(codom, dom)), pos_id)
	     | _ -> assert(false)
	  ) 
	  pre_fun_ids_var
      in

      let map_action = 
	var_terms_from_var_ids 
	  (types_from_parameters adt#get_parameters)
	  fun_ids_var
      in
      let fun_ids = variance_flatten fun_ids_var in
      let fun_exprs = List.map (fun (id,t) -> Term(id,Always,[])) fun_ids
      in
      let adt_from = Adt(adt,Always, arguments_from_parameters param_from) in
      let adt_to = Adt(adt,Always, arguments_from_parameters param_to) in
      let from_subst = 
	(Carrier, adt_to) ::
	(make_substitution_param_param adt#get_parameters param_from)
      in
      let var_from_subst = 
	List.map 
	  (fun (t,tsubst) -> (t, (tsubst,tsubst)))
	  from_subst
      in let var_to_subst = 
	  (Carrier, (adt_to,adt_to)) ::
	  (List.map 
	     (fun (t,tsubst) -> (t, (tsubst,tsubst)))
	     (make_substitution_param_param adt#get_parameters param_to)
	  )
      in
      let adt_arg = create_one_id sub_ns adt_from in
      let constructor_funs =
	List.map
	  (fun c ->
	     let ssub_ns = sub_space sub_ns in
	     let argtypes = member_arg_list c in
	     let substargtypes = 
	       List.map (ccsl_substitute_types from_subst) 
		 argtypes in
	     let argids = create_ids ssub_ns substargtypes in 
	     let cfun = 
	       if c#get_sort = Adt_Const_Constructor 
	       then
		 Term(c#get_name, Always,[])
	       else
		 SmartAbstraction(
		   argids,
		   SmartApplication(
		     Term(c#get_name, Always, []),
		     List.map2
		       (fun (id,substtyp) typ ->
			  functor_action 
    			    ssub_ns			(* name_space *)
			    false			(* id_crea_flag *)
			    map_action			(* actions *)
			    true			(* variance *)
			    var_from_subst		(* domain_subst *)
			    var_to_subst		(* codomain subst *)
			    (Term(id,Always,[]))	(* expression *)
			    typ				(* typ *)
		       )
		       argids
		       argtypes
		   ))
	     in
	       opt_expression cfun
	  )
	  adt#get_adt_constructors
      in
      	Defn(
	  name_of_adt_map adt,
	  [List.map (fun (id,t) -> Undeclared(id,t)) fun_ids],
					(* the type *)
	  Function(adt_from, adt_to),
					(* the definition *)
	  Abstraction(
	    [adt_arg,adt_from],
	    Application(
	      SmartApplication(
	      	Term(name_of_adt_reduce adt,Always,[]),
		constructor_funs),
	      Term(adt_arg,Always,[]))))

    method make_body =
      [
	self#adt_import;
	self#adt_map
      ]

end

class ccsl_adt_map_theory adt = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_adt_map_theory 
  adt eq_ccsl_types



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

