(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Adopted 25.1.99 by Hendrik
 *
 * Time-stamp: <Monday 11 March 02 11:42:20 tews@ithif51>
 *
 * Main Module for keyword hash generators
 *
 * $Id: main.ml,v 1.5 2002/05/03 15:01:21 tews Exp $
 *
 *)

open Global;;


(* Parsing the commandline -- Have a look at the module Arg *)

let options =
  [
    ("-o",
     Arg.String (fun s -> outfile := Some s),
     "Override default filename");
    ("-v",
    Arg.Set verbose,
    "Turn on verbose output");
    ("-D",
    Arg.Set debug,
    "Turn on debug output");
(*     ("-exact",
 *     Arg.Unit (fun () -> case := Exact),
 *     "Turn on exactcase keywords (default)");
 *     ("-dont-care",
 *     Arg.Unit (fun () -> case := Dont_care),
 *     "Turn on anycase keywords");
 *     ("-uppercase",
 *     Arg.Unit (fun () -> case := Uppercase),
 *     "Turn on uppercase keywords");
 *     ("-lowercase",
 *     Arg.Unit (fun () -> case := Lowercase),
 *     "Turn on lowercase keywords");
 *)
    ("-size",
    Arg.Int (fun x -> size := x),
    "Set size of hash table (default " ^ (string_of_int !size) ^ ")");
  ];;

let anonymous_function file_name =
  match !file with
    | None ->
	file := Some (file_name)
    | Some file_name' ->
	raise (Arg.Bad file_name);;

let usage_string =
  "Usage: " ^ Sys.argv.(0) ^ " [option] file_name\n" ^
  "where option is one of";;


(* Now start the program *)

try
  begin
    Arg.parse options anonymous_function usage_string;
    if !file = None
    then
      raise (Arg.Bad "")
  end
with Arg.Bad (s) ->
  begin
    Arg.usage options usage_string;
    exit(1)
  end;;

let file_name =
  match !file with
    | None ->
	raise (Arg.Bad "No file provided")
    | Some file_name' ->
	file_name';;

(* Strip of directory path and extension *)

let base_name =
  Filename.basename (Filename.chop_extension file_name);;

(* Strip of "_grammar" *)

let short_base_name =
  let base = String.sub base_name 0 
	       (try
		  (String.rindex base_name '_')
		with
		  | Not_found -> 0)
  in 
    if base = "" then "ccsl" else base

(* Create the interface *)

let interface_string =
  "(* This file is automatically generated from: " ^ file_name ^ "\n" ^
  " * by keyword Version " ^ version ^ "\n" ^
  " *\n" ^
  " * Copyright (C) 2002\n" ^
  " *\n" ^
  " * This program is free software; you can redistribute it and/or\n" ^
  " * modify it under the terms of the GNU General Public License as\n" ^
  " * published by the Free Software Foundation; either version 2 of\n" ^
  " * the License, or (at your option) any later version.\n" ^
  " *\n" ^
  " * This program is distributed in the hope that it will be useful,\n" ^
  " * but WITHOUT ANY WARRANTY; without even the implied warranty of\n" ^
  " * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU\n" ^
  " * General Public License in file COPYING in this or one of the\n" ^
  " * parent directories for more details.\n" ^
  " *\n" ^
  " *)\n" ^
  "\n" ^
  "val hash : " ^
  (String.capitalize base_name) ^ ".token -> " ^
  (String.capitalize base_name) ^ ".token;;\n" ^
  "\n" ^
  "(* End of automatically generated file *)\n";;


(* Create the module *)

let module_header_string =
  "(* This file is automatically generated from: " ^ file_name ^ "\n" ^
  " * by keyword Version " ^ version ^ "\n" ^
  " *\n" ^
  " * Copyright (C) 2002\n" ^
  " *\n" ^
  " * This program is free software; you can redistribute it and/or\n" ^
  " * modify it under the terms of the GNU General Public License as\n" ^
  " * published by the Free Software Foundation; either version 2 of\n" ^
  " * the License, or (at your option) any later version.\n" ^
  " *\n" ^
  " * This program is distributed in the hope that it will be useful,\n" ^
  " * but WITHOUT ANY WARRANTY; without even the implied warranty of\n" ^
  " * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU\n" ^
  " * General Public License in file COPYING in this or one of the\n" ^
  " * parent directories for more details.\n" ^
  " *\n" ^
  " *)\n" ^
  "\n" ^
    (* global dependency *)
  "open Global\n" ^
  "open Top_variant_types\n" ^
  "open Top_classes\n" ^
  "open Util;;\n" ^
    (* local dependency *)
  "open " ^ (String.capitalize base_name) ^ ";;\n" ^
  "\n";;


let module_keyword_string =
    (* tokens and keywords *)
  let tokens =
    Yacc_grammar.tokens Yacc_lexer.yacc (Lexing.from_channel 
					   (open_in file_name)) in
  let keyword_string =
    List.fold_left
      (fun s (token, typeopt) -> 
	 token ^ (match typeopt with
		    | None -> ""
		    | Some t -> "(" ^ t ^ ")"
		 ) ^
	   ", " ^ s) "" tokens in
  let keyword_table =
    List.fold_left
      (fun s (token, typeopt) ->
	 "  (( fun x -> " ^ 
	 (match typeopt with 
	    | None -> token 
	    | Some t -> token ^ "( x : " ^ t ^ " )"
	 )^ 
	 "), " ^
	 "\"" ^ (String.uppercase token) ^ "\");\n" ^ s)
      "" tokens in
  if !verbose
  then begin
    print_string "Keywords: \n";
    print_string keyword_string;
    print_string "\n";
    flush stdout
  end;
    (* keyword string *)
  "(* Keywords:\n" ^
  keyword_string ^ "\n" ^
  "*)\n" ^
  "\n" ^
    (* keyword table *)
  "let " ^ short_base_name ^ "_keyword_table = [\n" ^
  keyword_table ^
  "];;\n" ^
  "\n";;

let module_trailer_string =
    (* hash table *)
  "let " ^ short_base_name ^ "_hash_table =\n" ^
  "  Hashtbl.create " ^ (string_of_int !size) ^ ";;\n" ^
  "\n" ^
  "List.iter\n" ^
  "  (fun (t, s) -> Hashtbl.add " ^ short_base_name ^ "_hash_table s t)\n" ^
  "  " ^ short_base_name ^ "_keyword_table;;\n" ^
  "\n" ^
    (* hash function *)
  "let hash = function\n" ^
  "  | ID (token) as id ->\n" ^
  "      let loc = (match token.loc with \n" ^
  "                 | None -> assert(false) \n" ^
  "                 | Some l -> l \n" ^
  "                ) \n" ^
  "      in \n" ^
  "        (try\n" ^
  "          (Hashtbl.find " ^ short_base_name ^ "_hash_table\n" ^
  "            (String.uppercase token.token_name)) loc \n" ^
  "         with Not_found -> id)\n" ^
  "  | x -> x;;\n" ^
  "\n" ^
  "(* End of automatically generated file *)\n";;

let module_string =
  module_header_string ^ module_keyword_string ^ module_trailer_string;;


(* Create the output file *)

(* create _hashkeys.ml file name *)
let module_file =
  (match !outfile with
    | None -> short_base_name
    | Some s -> s)
  ^ "_hashkeys.ml";;
    
(* create .mli file name, by appending the 'i'. *)
let interface_file =
  module_file ^ "i";;

if !verbose
then
  print_string ("Generate " ^ interface_file ^ ".\n");
if !debug
then
  print_string interface_string;

let interface_file = open_out (interface_file) in
output_string interface_file interface_string;
close_out interface_file;

if !verbose
then
  print_string ("Generate " ^ module_file ^ ".\n");
if !debug
then
  print_string module_string;

let module_file = open_out (module_file) in
output_string module_file module_string;
close_out module_file;


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
