(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 11.10.01 by Hendrik
 *
 * Time-stamp: <Friday 2 July 10 13:12:19 tews@blau.inf.tu-dresden.de>
 *
 * addon theories 
 *
 * $Id: emptytype_theory.ml,v 1.14 2010-07-02 11:16:13 tews Exp $
 *
 *)

(***********************************************************************
 ***********************************************************************
 *
 * In this file there are additional theories for
 *   - empty type in pvs
 *   - empty function in pvs
 *   - sets2pred : 'a set => 'a => bool in isabelle
 *
 *)



open Name_space
open Top_names
open Names
open Top_variant_types  
open Pvs_proof_util
open Theory_class
open Iface_class
open Classtypes
open Types_util
;;


(*******************************************************************
 *******************************************************************
 *
 * Empty class for technical reasons
 *
 *)


let emptytoken = { token_name = "";
		   loc = None 
		 }

let empty_iface = 
  new ccsl_pre_iface_class emptytoken (Symbol.new_local ()) true
;;   
empty_iface#inheritance_done
;;


(***********************************************************************
 ***********************************************************************
 *
 * empty type definition
 *
 *)

class ['class_type, 'member_type] ccsl_pre_empty_type_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_sig_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_sig_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      (
       [ name_of_empty_type;
       ]
      )

    method get_name = ccsl_empty_type_defn_theory_name

    method get_parameters = []

    initializer top_theory#override_file_name ccsl_prelude_name

    method private emptytype =
      let bool_var = "b" 
      in
	Typedecl(
	  [],
	  name_of_empty_type,
	  Predtype(
	    Formula(
	      Abstraction(
		[bool_var, Bool],
		Expression(False)))))
	      

    method make_body = 
      [      
	self#emptytype
      ]

end (* ccsl_pre_empty_type_theory *)

class ccsl_empty_type_theory = 
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_empty_type_theory
    empty_iface
    eq_ccsl_types


(***********************************************************************
 ***********************************************************************
 *
 * empty function definition
 *
 *)

class ['class_type, 'member_type] ccsl_pre_empty_fun_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_sig_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_sig_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      (
       [ name_of_empty_fun;
       ]
      )

    val other_type_id = 
		 { id_token = { token_name = "A";
				loc = None };
		   (* the other fields are to ignored *)
		   id_type = Self;
		   id_origin = CCSL_TypeParameter;
		   id_parameters = [];
		   id_variance = Unset;
		   id_sequence = -1;
		   id_components = [];
		 }

    method get_name = ccsl_empty_fun_theory_name

    method get_parameters = [TypeParameter other_type_id]

    initializer top_theory#override_file_name ccsl_prelude_name

    method private fundefn =
      let empty_type = TypeConstant(name_of_empty_type,Always,[]) in
      let other_type = BoundTypeVariable other_type_id in
      let empty_var = "z" in
      let other_var = "a" 
      in
	Defn(
	  name_of_empty_fun,
	  [],
	  (Function(empty_type, other_type)),
	  Abstraction(
	    [empty_var, empty_type],
	    Application(
	      Term("epsilon",Always,[]),
	      Abstraction(
		[other_var, other_type],
		Expression True))))
	

    method private singlefun =
      let proof = make_simple_proof [skosimp_star; extensionality] in
      let empty_type = TypeConstant(name_of_empty_type,Always,[]) in
      let other_type = BoundTypeVariable other_type_id in
      let other_f = "f" in
      let lem =
	Lemma(
	  name_of_unique_empty_fun_lemma,
	  Forall(
	    [other_f, Function(empty_type, other_type)],
	    Equal(
	      Term(other_f,Always,[]), 
		   Term(name_of_empty_fun, Always,[]))))
      in
	Proved( lem, PvsProof(Anon_proof proof))
	  

    method make_body = 
      [      
	Import( [ccsl_empty_type_defn_theory_name, []]);
	self#fundefn;
	self#singlefun
      ]

end (* ccsl_pre_empty_fun_theory *)

class ccsl_empty_fun_theory = 
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_empty_fun_theory
    empty_iface
    eq_ccsl_types



(***********************************************************************
 ***********************************************************************
 *
 * powerset definitions with 1 Type parameter
 *
 *)


(* TODO reclassify the infinite powerset so that greatest 
   bisimulations exists 
*)

(* 
 * The following theory class is generic for both the infinite and 
 * the finite powerset. There are a number of string valued 
 * virtual methods to abstract from the concreate names. These methods 
 * are declared virtual in power_parameters and later realized in
 * power_instantiation and fpower_instantiation.
 * 
 * The theories for the powerset and the finite powerset are later 
 * derived from this.
 *)


class virtual power_parameters =
object

    method virtual private power_type_name : string

    method virtual private power_pred_char_name : string

    method virtual private power_def_1_name : string

    method virtual private power_def_2_name : string

end (* power_parameters *)


class virtual ['class_type, 'member_type] ccsl_pre_virtual_power_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_sig_theory_class cl eq_types
	as top_theory

    inherit power_parameters

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      (
       [ 
       ]
      )

    method get_name = self#power_def_1_name

    val param_id = 
		 { id_token = { token_name = "X";
				loc = None };
		   (* the other fields are to ignored *)
		   id_type = Self;
		   id_origin = CCSL_TypeParameter;
		   id_parameters = [];
		   id_variance = Unset;
		   id_sequence = -1;
		   id_components = [];
		 }

    method private param_type = BoundTypeVariable(param_id)

    method private power_type = 
      TypeConstant(self#power_type_name, 
		   Isabelle_only, 
		   [TypeArgument(self#param_type)])


    method get_parameters = [TypeParameter param_id]

    initializer top_theory#override_file_name ccsl_prelude_name

    method private power =
      Typedecl(
	[],
	self#power_type_name,
	Function(self#param_type, Bool))

    method private emptyset =
      Comment "emptyset : Power is defined in the Pvs prelude"

    method private member =
      Comment "member : [X,Power -> bool] is defined in the Pvs prelude"

    method private pred =
      Defn(
	self#power_pred_char_name,
	[],
	Function(self#power_type, Function(self#param_type, Bool)),
	Abstraction(["s", self#power_type],
		    Term("s", Never, [])))

    method private every =
      Defn(
	name_of_const_every self#power_type_name,
	[],
	Function(
	  Function(self#param_type, Bool),
	  Function(self#power_type, Bool)),
	Abstraction(
	  ["P", Function(self#param_type, Bool)],
	  Abstraction(
	    ["s", self#power_type],
	    Expression(
	      Forall(
		["x", self#param_type],
		Implies(
		  Formula(
		    Application(
		      Term("s", Never,[]),
		      Term("x", Never,[]))),
		  Formula(
		    Application(
		      Term("P", Never,[]),
		      Term("x", Never,[])))))))))
		    

    method make_body = 
      [      
	self#power;
	self#emptyset;
	self#member;
	self#pred;
	self#every
      ]

end (* ccsl_pre_power_theory *)



class power_instantiation =
object

    method private power_type_name = name_of_power_type

    method private power_pred_char_name = name_of_power_pred_char

    method private power_def_1_name = ccsl_power_theory_1_name

    method private power_def_2_name = ccsl_power_theory_2_name

end (* power_instantiation *)


class ['class_type, 'member_type] ccsl_pre_power_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_sig_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_virtual_power_theory cl eq_types

    inherit power_instantiation

  end (* ccsl_pre_power_theory *)


class ccsl_power_def_1_theory = 
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_power_theory
    empty_iface
    eq_ccsl_types



(***********************************************************************
 ***********************************************************************
 *
 * powerset definitions with 2 Type parameters
 *
 *)


(* 
 * The game is the same as before: First a generic class that gets 
 * refined for finite and infinite powerset as before.
 *)

class virtual ['class_type, 'member_type] ccsl_pre_virtual_power_morph_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_sig_theory_class cl eq_types
      as top_theory

    inherit power_parameters

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      (
       [ 
       ]
      )

    method get_name = self#power_def_2_name

    val x_param_id = 
		 { id_token = { token_name = "X";
				loc = None };
		   (* the other fields are to ignored *)
		   id_type = Self;
		   id_origin = CCSL_TypeParameter;
		   id_parameters = [];
		   id_variance = Unset;
		   id_sequence = -1;
		   id_components = [];
		 }

    val y_param_id = 
		 { id_token = { token_name = "Y";
				loc = None };
		   (* the other fields are to ignored *)
		   id_type = Self;
		   id_origin = CCSL_TypeParameter;
		   id_parameters = [];
		   id_variance = Unset;
		   id_sequence = -1;
		   id_components = [];
		 }

    method private x_param_type = BoundTypeVariable(x_param_id)

    method private y_param_type = BoundTypeVariable(y_param_id)

    method private power_type param = 
      TypeConstant(self#power_type_name, 
		   Always, 
		   [TypeArgument(param)])


    method get_parameters = 
      [TypeParameter x_param_id; TypeParameter y_param_id]

    initializer top_theory#override_file_name ccsl_prelude_name

    method private importing = 
      Import([self#power_def_1_name, []])

    method private rel_every =
      Defn(
	name_of_const_rel_every self#power_type_name,
	[],
	Function(
	  Function(Product[self#x_param_type; self#y_param_type], Bool),
	  Function(
	    Product(
	      [self#power_type self#x_param_type; 
	       self#power_type self#y_param_type]), 
	    Bool)),
	Abstraction(
	  ["R", Function(Product[self#x_param_type; self#y_param_type], Bool)],
	  Abstraction(
	    [("s1", self#power_type self#x_param_type);
	     ("s2", self#power_type self#y_param_type)],
	    Expression(
	      And(
		[Forall(
		   ["x", self#x_param_type],
		   Implies(
		     Formula(
		       Application(
			 Term("s1", Never,[]),
			 Term("x", Never,[]))),
		     Exists(
		       [ "y", self#y_param_type],
		       And(
			 [Formula(
			    Application(
			      Term("s2", Never,[]),
			      Term("y",Never,[])));
			  Formula(
			    Application(
			      Term("R", Never,[]),
			      Tuple([Term("x", Never,[]); 
				     Term("y", Never,[])])))
		      ]))));
		 Forall(
		   ["y", self#y_param_type],
		   Implies(
		     Formula(
		       Application(
			 Term("s2", Never,[]),
			 Term("y", Never,[]))),
		     Exists(
		       [ "x", self#x_param_type],
		       And(
			 [Formula(
			    Application(
			      Term("s1", Never,[]),
			      Term("x",Never,[])));
			  Formula(
			    Application(
			      Term("R", Never,[]),
			      Tuple([Term("x", Never,[]); 
				     Term("y", Never,[])])))
		      ]))))
		])))))


    method private morph_def =
      Defn(
	name_of_const_map self#power_type_name, [],
	Function(
	  Function(self#x_param_type, self#y_param_type),
	  Function(self#power_type self#x_param_type,
		   self#power_type self#y_param_type)),
	Abstraction(
	  ["f", Function(self#x_param_type, self#y_param_type)],
	  Abstraction(
	    ["s1", self#power_type self#x_param_type],
	    Abstraction(
	      ["y", self#y_param_type],
	      Expression(
		Exists(
		  ["x", self#x_param_type],
		  And(
		    [Formula(
		       Application(
			 Term("s1", Never,[]),
			 Term("x", Never,[])));
		     Equal(
		       Term("y",Never,[]),
		       Application(
			 Term("f",Never,[]),
			 Term("x",Never,[])))
		    ])))))))


    method make_body = 
      [      
	self#importing;
	self#rel_every;
	self#morph_def
      ]

end (* ccsl_pre_power_morph_theory *)



class ['class_type, 'member_type] ccsl_pre_power_morph_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_sig_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_virtual_power_morph_theory 
	cl eq_types

    inherit power_instantiation

  end (* ccsl_pre_power_morph_theory *)


class ccsl_power_def_2_theory = 
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_power_morph_theory
    empty_iface
    eq_ccsl_types



(***********************************************************************
 ***********************************************************************
 *
 * finite powerset definitions with one parameter
 *
 *)


class fpower_instantiation =
object

    method private power_type_name = name_of_fpower_type

    method private power_pred_char_name = name_of_fpower_pred_char

    method private power_def_1_name = ccsl_fpower_theory_1_name

    method private power_def_2_name = ccsl_fpower_theory_2_name

end (* power_instantiation *)


class ['class_type, 'member_type] ccsl_pre_fpower_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_sig_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_virtual_power_theory cl eq_types

    inherit fpower_instantiation
	
    method private power =
      Typedecl(
	[],
	self#power_type_name,
	TypeConstant(name_of_finite_set_type, 
		     Always, [TypeArgument(self#param_type)])
      )

    method private emptyset =
      Defn(
	name_of_femptyset,
	[],
	self#power_type,
	Term(name_of_emptyset, Never, []))


    method private member =
      Defn(
	name_of_fmember,
	[],
	Function(
	  Product(
	    [self#param_type; self#power_type]),
	  Bool),
	Term(name_of_member, Never, []))


  end (* ccsl_pre_power_theory *)


class ccsl_fpower_def_1_theory = 
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_fpower_theory
    empty_iface
    eq_ccsl_types



(***********************************************************************
 ***********************************************************************
 *
 * finite powerset definitions with two parameters
 *
 *)


class ['class_type, 'member_type] ccsl_pre_fpower_morph_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_sig_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_virtual_power_morph_theory
	cl eq_types
	as super

    inherit fpower_instantiation
	

(* 
 * MapFPower_TCC1: OBLIGATION
 *   FORALL (f: [X -> Y], s1: FPower[X]):
 *     is_finite[Y]((LAMBDA (y: Y): EXISTS (x: X): ((s1(x)) AND (y = f(x)))));
 * 
 * (""
 *  (skosimp* :preds? t)
 *  (rewrite "is_finite_surj" :dir rl)
 *  (rewrite "is_finite_surj" :dir rl)
 *  (skosimp* :preds? t)
 *  (inst 1 "N!1" "f!1 o f!2")
 *  (("1" (expand* "surjective?" "o") (reduce))
 *   ("2" (expand "o") (reduce))))
 *)

    method private finite_image_lemma =
      let proof =
	PTree(
	  [skosimp_preds;
	   rewrite_left "is_finite_surj";
	   rewrite_left "is_finite_surj";
	   skosimp_star;
	   inst_val 1 ["N!1"; "f!1 o f!2"];
	  ],
	  [make_simple_proof [
	     expand_star ["surjective?"; "o"];
	     reduce
	   ];
	   make_simple_proof [
	     expand "o";
	     reduce
	   ]])
      in
      let lemma = 
	Lemma(
	  name_of_finite_image_lemma,
	  Forall(
	    [("f", Function(self#x_param_type, self#y_param_type));
	     ("s1", self#power_type self#x_param_type)],
	    Formula(
	      Application(
		Term(name_of_finite_pred, Never, []),
		Abstraction(
		  ["y", self#y_param_type],
		  Expression(
		    Exists(
		      ["x", self#x_param_type],
		      And(
			[Formula(
			   Application(
			     Term("s1", Never,[]),
			     Term("x", Never,[])));
			 Equal(
			   Term("y",Never,[]),
			   Application(
			     Term("f",Never,[]),
			     Term("x",Never,[])))
			]))))))))
      in 
	Proved( lemma, PvsProof(Anon_proof proof))


    method private morph_def =
      let tcc_proof =
	  (* 
	   * (skosimp* :preds? t) 
	   * (rewrite "finite_image_finite")	
	   *)
	make_simple_proof [
	  skosimp_star;
	  rewrite name_of_finite_image_lemma;
	]
      in
	Proved(
	  super#morph_def,
	  PvsProof(
	    Named_proof(
	      (name_of_const_map self#power_type_name) ^ "_TCC1",
	      tcc_proof)))
	      
	  

    method make_body = 
      [      
	self#importing;
	self#rel_every;
	self#finite_image_lemma;
	self#morph_def
      ]

  end (* ccsl_pre_fpower_morph_theory *)


class ccsl_fpower_def_2_theory = 
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_fpower_morph_theory
    empty_iface
    eq_ccsl_types



(***********************************************************************
 ***********************************************************************
 *
 * Isabelle part
 *
 * open close the prelude
 *)


class ccsl_isar_start_prelude =
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_isabelle_delimiter_theory_class
    empty_iface
    eq_ccsl_types
    IsabelleStartFile
    ccsl_prelude_name
    [isabelle_top_theory]
    true				(* do proofs *)

class ccsl_isar_close_prelude =
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_isabelle_delimiter_theory_class
    empty_iface
    eq_ccsl_types
    IsabelleCloseFile
    ccsl_prelude_name
    []
    true				(* do proofs *)

    
  
(***********************************************************************
 ***********************************************************************
 *
 * Isabelle part
 *
 * set2pred
 *)

class ['class_type, 'member_type] ccsl_pre_isabelle_addon_theory
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_sig_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_sig_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      (
       [ name_of_set2pred;
       ]
      )

    val other_type_id = 
		 { id_token = { token_name = "A";
				loc = None };
		   (* the other fields are to ignored *)
		   id_type = Self;
		   id_origin = CCSL_TypeParameter;
		   id_parameters = [];
		   id_variance = Unset;
		   id_sequence = -1;
		   id_components = [];
		 }
    method get_name = ccsl_isabelle_addon_theory_name

    method get_parameters = [TypeParameter other_type_id]

    initializer top_theory#override_file_name ccsl_prelude_name

    method private quick_and_dirty =
      [Comment("Enable quick_and_dirty for sorry proofs");
       IsarML "set quick_and_dirty";
      ]

    method private pred2set =
      let subns = sub_space ns in
      let a = BoundTypeVariable other_type_id in
      let styp = TypeConstant("set", Always, [TypeArgument a]) in
      let s = create_id_with_preference subns "s" "s" in
      let x = create_id_with_preference subns "a" "a" in
      Defn(
	name_of_set2pred,
	[[Undeclared(s, styp); Undeclared(x,a)]],
	Bool,
	Term("( " ^ x ^ " : " ^ s ^ " )", Always, [])
      )

    method make_body = 
      self#quick_and_dirty
      @
      [      
	self#pred2set;
      ]

end (* ccsl_pre_empty_fun_theory *)

class ccsl_isabelle_addon_theory =
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_isabelle_addon_theory
    empty_iface
    eq_ccsl_types





(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)


