(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Tuesday 22 June 04 23:50:53 tews@debian>
 *
 * Feature pass for CCSL
 * Determines the features for adt's, classes and groundsignatures
 *
 * $Id: feature.ml,v 1.10 2004-07-08 22:10:02 tews Exp $
 *
 *)

open Util
open Global
open Top_variant_types
open Error
open Component_pass

(***********************************************************************
 *
 * Errors & Warnings
 *
 *)

exception Feature_error


let d s = 
  if debug_level _DEBUG_FEATURES then
    print_verbose s

let feature_error loc s =
  if !expert_mode 
  then warning_message loc s
  else 
    begin
      error_message loc s;
      raise Feature_error
    end


(***********************************************************************
 ***********************************************************************
 *
 * Determine Features
 *
 *)

let has_final_model functor_type = 
  match (functor_type.grade, functor_type.functor_components) with
    | ((ConstantFunctor | PolynomialFunctor), 
       (PurelyIterated_components | FKripke_components)) -> true

    | _, Kripke_components
    | _, Arbitrary_components
    | ExtendedCartesianFunctor,_
    | ExtendedPolynomialFunctor,_
    | HigherOrderFunctor,_ -> false

    | UnknownFunctor,_ -> assert(false)


    (* be conservative here: do not allow powerset in 
     * extended cartesian functors
     *)
let has_greatest_bisim functor_type = 
    match (functor_type.grade, functor_type.functor_components) with
      | ((ConstantFunctor | PolynomialFunctor),
	 (PurelyIterated_components | FKripke_components | Kripke_components))
      | (ExtendedCartesianFunctor, PurelyIterated_components)
	-> true

      | _, Arbitrary_components
      | ExtendedCartesianFunctor, _
      | ExtendedPolynomialFunctor,_
      | HigherOrderFunctor,_ -> false

      | UnknownFunctor,_ -> assert(false)


let do_class cl =
  let loc = (remove_option cl#get_token.loc) in
  let hasmorphism = ref true in
  let hasmap = ref false
  in
    begin
      class_component_pass cl;

      d (" ** Determining features in " ^ cl#get_name );

			(* check for Self in mixed parameter position *)
      if cl#has_feature MixedSelfInstFeature 
      then
	if !expert_mode then
	  begin
	    warning_message loc 	
	      ("Definition of ``" ^ cl#get_name ^ 
	       "'' coalgebra morphisms is probably wrong\n" ^
	       "because Self appears in a parameter position with " ^
	       "mixed variance.");
	    cl#put_feature HasMorphismFeature;
	  end
	else
	  begin 
	    warning_message loc 	
	      ("No definition of ``" ^ cl#get_name ^ 
	       "'' coalgebra morphisms generated\n" ^
	       "because Self appears in a parameter position with " ^
	       "mixed variance.");
	    hasmorphism := false
	  end;

					(* check if all components have map *)
      if !hasmorphism then
	if cl#has_feature ComponentNoMapFeature
	then
	  if !expert_mode then
	    begin
	      warning_message loc 	
		("Definition of ``" ^ cl#get_name ^ 
		 "'' coalgebra morphisms is probably wrong\n" ^
		 "because there is no Map for some component.");
	      cl#put_feature HasMorphismFeature
	    end
	  else 
	    begin 
	      warning_message loc 	
		("No definition of ``" ^ cl#get_name ^ 
		 "'' coalgebra morphisms generated\n" ^
		 "because there is no Map for some component.")
	    end
	else 
	  cl#put_feature HasMorphismFeature;


			(* check for Bisimulation and simple rel lifting *)
      if cl#has_feature ComponentNoRelLiftFeature
      then
	if !expert_mode then
	  begin
	    warning_message loc 	
	      ("Definition of relation lifting for ``" ^ cl#get_name ^ 
	       "'' is probably wrong\n" ^
	       "because there is no relation lifting for some component.");
	    cl#put_feature HasBisimulationFeature
	  end
	else
	  begin 
	    warning_message loc 	
	      ("No definition of bisimulation for class ``" ^ cl#get_name ^ 
	       "'' generated\n" ^
	       "because there is no relation lifting for some component.")
	  end
      else
	cl#put_feature HasBisimulationFeature;


					(* check for full rel lifting *)
      if cl#has_feature HasBisimulationFeature then
	if cl#has_feature ComponentNoFullRelLiftFeature then
	  if !expert_mode then
	    begin
	      warning_message loc
		("Definition of full relation lifting for ``" ^ cl#get_name ^ 
		 "'' is probably wrong\n" ^
		 "because there is no full relation lifting for " ^ 
		 "some component.");
	      cl#put_feature HasFullRelLiftingFeature
	    end
	  else
	    begin 
	      warning_message loc 	
		("No full relation lifting for class ``" ^ cl#get_name ^ 
		 "'' generated\n" ^
		 "because there is no full relation lifting for " ^
		 "some component.")
	    end
	else
	  cl#put_feature HasFullRelLiftingFeature;


					(* check for greatest bisimulation *)
      if (cl#has_feature HasBisimulationFeature) then
	if has_greatest_bisim cl#get_functor_type
	then 
	  begin
	    cl#put_feature HasGreatestBisimFeature;
	  end
	else
	  begin
	    warning_message loc 
	      ("Class " ^ cl#get_name ^ " corresponds to a " ^
	       (Variance.string_of_functor cl#get_functor_type) ^
	       " functor. \nThere exists no greatest bisimulation.");
	    if !expert_mode then
	      cl#put_feature HasGreatestBisimFeature
	  end;
	      


					(* check for rel lift*)
      if (cl#has_feature HasFullRelLiftingFeature) &&
	(cl#has_feature HasGreatestBisimFeature)
      then 
	if has_greatest_bisim cl#get_functor_type
	then 
	  begin
	    cl#put_feature HasRelLiftFeature;
	  end
	else
	  begin
	    warning_message loc 
	      ("Class " ^ cl#get_name ^ " corresponds to a " ^
	       (Variance.string_of_functor cl#get_functor_type) ^
	       " functor. \nIt has no relation lifting.");
	    if !expert_mode then
	      cl#put_feature HasRelLiftFeature;
	  end;


					(* check for final models *)
      if cl#has_feature FinalSemanticsFeature && 
	(not (has_final_model cl#get_functor_type))
      then
	feature_error loc
	  ("Class " ^ cl#get_name ^ 
	   " corresponds not to an iterated polynomial functor.\n" ^
	   "It has no final models.");

					(* check for finality definition *)
      if cl#has_feature FinalSemanticsFeature && 
	(not (cl#has_feature HasMorphismFeature))
      then
	feature_error loc
	  ("Class " ^ cl#get_name ^ " has no final semantics\n" ^
	   "because ``" ^ cl#get_name ^ "'' morphisms are undefined.");


					(* check if this class has map *)
      if cl#has_feature FinalSemanticsFeature &&
	(cl#get_parameters <> [])
      then
	if cl#has_feature ComponentNoFullMapFeature then
	  if !expert_mode then
	    begin
	      warning_message loc
		("Definition of Map combinator for " ^ cl#get_name ^ 
		 " is probably wrong\n" ^
		 "because there is no map for some component.");
	      cl#put_feature HasMapFeature;
	    end
	  else
	    begin
	      warning_message loc
		("No map combinator generated for " ^ cl#get_name ^ 
		 "\nbecause there is no map for some component.")
	    end
	else
	  hasmap := true;
	  

      if !hasmap then
	if (cl#get_assertions = []) && (cl#get_creations = []) 
	then
	  cl#put_feature HasMapFeature
	else
	  if !expert_mode then
	    begin
	      warning_message loc
		("Map combinator for " ^ cl#get_name ^ 
		 " may produce unprovable TCC's\n" ^
		 "because there are assertions in class " 
		 ^ cl#get_name ^ ".");
	      cl#put_feature HasMapFeature;
	    end
	  else
	    begin
	      warning_message loc
		("No map combinator generated for " ^ cl#get_name ^ 
		 "\nbecause there are assertions in " ^ cl#get_name);
	    end;

	

    end


let do_adt adt =
  let loc = (remove_option adt#get_token.loc) in
    begin
      adt_component_pass adt;
      
      d (" ** Determining features in " ^ adt#get_name );

      if adt#get_parameters <> []
      then
	if adt#has_feature ComponentNoFullMapFeature
	then
	  if !expert_mode then
	    begin
	      warning_message loc
		("Definition of Map combinator for " ^ adt#get_name ^ 
		 " is probably wrong\n" ^
		 "because there is no map for some component.");
	      adt#put_feature HasMapFeature;
	    end
	  else
	    begin
	      warning_message loc
		("No map combinator generated for " ^ adt#get_name ^ 
		 "\nbecause there is no map for some component.")
	    end
	else
	  adt#put_feature HasMapFeature;
      
      if adt#has_feature ComponentNoFullRelLiftFeature
      then
	if !expert_mode then
	  begin
	    warning_message loc
	      ("Definition of relation lifting for ``" ^ adt#get_name ^ 
	       "'' is probably wrong\n" ^
	       "because there is no relation lifting for " ^ 
	       "some component.");
	    adt#put_feature HasFullRelLiftingFeature
	  end
	else
	  begin 
	    warning_message loc 	
	      ("No relation lifting for adt ``" ^ adt#get_name ^ 
	       "'' generated\n" ^
	       "because there is no relation lifting for " ^
	       "some component.")
	  end
      else
	adt#put_feature HasRelLiftFeature;
      
					(* adt needs map and every *)
      if adt#has_feature HasMapFeature then
	(match !output_mode with
	   | Isa_mode -> adt#put_feature NeedsMapFeature
	   | Pvs_mode -> 
	       if (List.length adt#get_parameters > 0) 
		 && (List.for_all
		       (function TypeParameter id ->
					(* is strictly positive *)
			  (id.id_variance = Pair(-1,-1)) ||
			  (id.id_variance = Pair(-1,0)))
		       adt#get_parameters
		    )
	       then
		 begin
		   if adt#has_feature ClassComponentFeature then
		     adt#put_feature NeedsMapFeature
		 end
	       else			(* doit ourselves *)
		 adt#put_feature NeedsMapFeature
	);
      
      if adt#has_feature CarrierClassComponentFeature then
(* HENDRIK: PVS type abbreviations work
 -> need to distinguish between ground types and class types
*)
	feature_error loc
	  ("Adt " ^ adt#get_name ^ 
	   " instanciates a class or a ground type with Carrier.\n" ^
	   "This is not allowed in PVS or Isabelle/Hol"
	  );      
      
    end


let do_sig si =
  sig_component_pass si


let feature_ast = function
  | CCSL_class_dec cl -> do_class cl
  | CCSL_adt_dec adt -> do_adt adt
  | CCSL_sig_dec si -> do_sig si


let ccsl_feature_pass (ast: Classtypes.ccsl_ast list) = 
  List.iter feature_ast ast;



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

