(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Tuesday 29 June 10 12:58:09 tews@blau.inf.tu-dresden.de>
 *
 * predicate and relation lifting
 *
 * $Id: lifting.mli,v 1.10 2010-06-30 08:46:13 tews Exp $
 *
 *)

open Top_variant_types
open Top_classtypes
open Classtypes

(***********************************************************************
 *
 * type param_lifting_type is an assoc list that
 * associates with every type parameter id two 
 * optional expressions. The first is the lifting for 
 * negative occurences, the second for positive occurences.
 * If None, then no lifting is performed for this occurence.
 * 
 * The constraint ensure that the function can access the 
 * variances of components and compute their lifing.
 *)

type ('cl, 'mem) param_lifting_type =
    (('cl, 'mem) top_pre_identifier_record_type
     * (('cl, 'mem) top_pre_expressions option 
	* ('cl, 'mem) top_pre_expressions option )
    ) list
constraint 'cl = <get_name : string; 
                  get_parameters : ('cl,'mem) top_pre_parameter_type list;
		  get_adt_constructors : 'mem list;
		  get_all_sig_actions : 'mem list;
		  get_self_variance : variance_type;
		  get_model_coalgebra : ('cl, 'mem) top_pre_expressions;
                  .. > 
constraint 'mem = <get_domain : ('cl,'mem) top_pre_types; 
		   get_codomain : ('cl,'mem) top_pre_types; 
		   get_name : string;
		   get_accessors : 'mem list;
		   get_recognizer : 'mem;
		   get_sort : 'mem top_pre_member_sort;
		   ..>

(***********************************************************************
 *
 * type self_lifting_type contains the negative (first) and the 
 * positive lifting for self. 
 *)


type ('cl, 'mem) self_lifting_type = 
    ('cl, 'mem) top_pre_expressions option 
    * ('cl, 'mem) top_pre_expressions option
constraint 'cl = <get_name : string; 
                  get_parameters : ('cl,'mem) top_pre_parameter_type list;
		  get_adt_constructors : 'mem list;
		  get_all_sig_actions : 'mem list;
		  get_self_variance : variance_type;
		  get_model_coalgebra : ('cl, 'mem) top_pre_expressions;
                  .. > 
constraint 'mem = <get_domain : ('cl,'mem) top_pre_types; 
		   get_codomain : ('cl,'mem) top_pre_types; 
		   get_name : string;
		   get_accessors : 'mem list;
		   get_recognizer : 'mem;
		   get_sort : 'mem top_pre_member_sort;
		   ..>


(**************************************************************************
 *
 * Predicate and relation lifting is designed to work on polymorphic types
 * and expressions. However, the concrete types creep in at two occurences:
 *  - type substitution needs an equality function
 *  - inlining of powerset liftings relies on reference cells (down in 
 *    the prelude module) that hold the powerset types. Comparison
 *    with these reference cells forces the concrete type.
 * 
 * Therefore we abstract away the concrete types with two operations
 * eq_types and is_any_power_type, bundeled in the following type.
 *)


type ('cl, 'mem) concrete_lifting_operations = {
  eq_types : ('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types -> bool;
  is_any_power_type : ('cl,'mem) top_pre_types -> bool
}



(***********************************************************************
 *
 * fullpredlift 
 *
 * This implements full predicate lifting for 
 * full higher order polynomial functors with optimization
 * 
 * for classes, adt's, groundtypes a symbolic Every is produced
 *
 * Arguments
 *     lifting_ops	concrete operations
 *     self_subst	substitution for Self
 *     param_lifting	lifting for typeparameters
 *     self_lifting	lifting for self
 *     top_name_space	name space for argument name creation
 *     typ		type to lift
 *     expression 	expression of type typ, to which the lifting 
 *         		is applied
 * 
 *)

val fullpredlift : 
  ('cl, 'mem) concrete_lifting_operations -> 
    (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types) -> 
      ('cl, 'mem) param_lifting_type ->
	('cl, 'mem) self_lifting_type ->
	  ('cl, 'mem) Name_space.t -> 
	    ('cl, 'mem) top_pre_types -> 
	      ('cl, 'mem) top_pre_expressions -> 
		('cl, 'mem) top_pre_formulas

(***********************************************************************
 *
 * predlift -- version of fullpredlift without parameter lifting 
 *             and substitution
 *
 * Arguments
 *     lifting_ops	concrete operations
 *     self_lifting	lifting for self
 *     top_name_space	name space for argument name creation
 *     typ		type to lift
 *     expression 	expression of type typ, to which the lifting 
 *         		is applied
 *)

val predlift : 
  ('cl, 'mem) concrete_lifting_operations -> 
    ('cl, 'mem) self_lifting_type ->
      ('cl, 'mem) Name_space.t -> 
	('cl, 'mem) top_pre_types -> 
	  ('cl, 'mem) top_pre_expressions -> 
	    ('cl, 'mem) top_pre_formulas



(***********************************************************************
 *
 * argument_list_fullpredlift -- predicate lifting for argument lists
 * 
 * Arguments
 *     lifting_ops	    concrete operations
 *     self_subst	    substitution for Self
 *     param_lifting	    lifting for typeparameters
 *     self_lifting	    lifting for self
 *     top_name_space	    name space for argument name creation
 *     parameter_list       original parameter list of the 
 *			    ancestor or component (needed to get variances)
 *     argument_list        argument type list that drives the lifting
 *)


val argument_list_fullpredlift : 
  ('cl, 'mem) concrete_lifting_operations -> 
    (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types) -> 
      ('cl, 'mem) param_lifting_type ->
	('cl, 'mem) self_lifting_type ->
	  ('cl, 'mem) Name_space.t -> 
	    ('cl, 'mem) top_pre_parameter_type list -> 
	      ('cl, 'mem) top_pre_argument_type list -> 
		('cl, 'mem) top_pre_expressions list


(*************************************************************************
 *
 * inline_adt_pred_lifting -- predicate lifting of adt as case expression
 * 
 * Arguments
 * 
 *     lifting_ops	concrete operations
 *     self_subst	substitution for carrier
 *     param_lifting	lifting for parameters
 *     self_lifting	lifting for self
 *     top_name_space	the name space
 *     adt			the adt
 *     args		its arguments 
 *     expr 		the expression to lift
 *)


val inline_adt_pred_lifting :
  ('cl, 'mem) concrete_lifting_operations -> 
    (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types) -> 
      ('cl, 'mem) param_lifting_type ->
	('cl, 'mem) self_lifting_type ->
	  ('cl, 'mem) Name_space.t -> 
	    'cl ->
	      ('cl,'mem) top_pre_argument_type list ->
		('cl,'mem) top_pre_expressions ->
		  ('cl,'mem) top_pre_expressions


(******************************
 * Same as predlift, with the following differences:
 * - the lifting wrt. to a one method instead of a type
 * - applications of this method occurs in standard form: m(c)( args )
 * 
 * In class_method_lifting anc_lifting name_space predicate coalgebra 
 * 	      ancestor_list method_list selfvar self_type_argument
 * the arguments are
 *     lifting_ops	  -> concrete operations
 *     param_lifting	  -> assoc list of param_lifting_type that gives 
 *			     the liftings form typeparameters
 *     self_lifting	  -> lifting for Self
 *     name_space         -> name space for name creation
 *     coalgebra          -> the coalgebra for getting the methods right
 *     meth		  -> the method
 *     selfvar            -> the variable of Self, to which we apply
 * 				the lifting
 *)


val class_pred_lifting :
  ('cl, 'mem) concrete_lifting_operations -> 
    ('cl, 'mem) param_lifting_type ->
      ('cl, 'mem) self_lifting_type ->
	('cl, 'mem) Name_space.t -> 
	  ('cl, 'mem) top_pre_expressions ->
	    <get_name : string; 
	     last_name : string;
	     original_name : string;
	     get_domain : ('cl, 'mem) top_pre_types; 
	     get_codomain: ('cl, 'mem) top_pre_types;.. 
            > ->
	      string -> 
		('cl, 'mem) top_pre_formulas



(************************************************************************
 * 
 * Ccsl specific predicate lifting
 * 
 *)

val ccsl_lifting_operations : 
  (ccsl_iface_type, ccsl_member_type) concrete_lifting_operations



(************************************************************************
 * lifting wrt sets of methods:
 *
 *)

					   
val class_method_pred_definition :
  ccsl_name_space_type -> 
    ccsl_output_types ->
      string ->
	ccsl_output_types ->
	  ccsl_member_type ->
	    ccsl_theory_declaration


val class_real_method_pred_lifting :
  ccsl_iface_type ->
    ccsl_argument_type list ->
      ccsl_name_space_type ->
	ccsl_expressions ->				  (* list *)
	  ccsl_formulas ->				  (* predicate *) 
	    ccsl_expressions ->				  (* coalgebra *)
	      string ->					  (* method_id *)
		ccsl_output_types ->
		  string -> 
		    ccsl_formulas


(***********************************************************************
 ***********************************************************************
 *
 * RELATION LIFTING
 *
 ***********************************************************************
 ***********************************************************************
 *
 * This implements relation lifting for 
 * full higher order polynomial functors
 * 
 * for classes and adt's a symbolic RelEvery is produced
 * 
 * Arguments
 *     lifting_ops	    concrete operations
 *     self1_subst          substitution function for Self for expr1
 *     self2_subst          substitution function for Self for expr2
 *     param_rel_lifting    the lifting for typeparameters
 *     self_lifting	    the lifting for Self/Carrier
 *     top_name_space       name space for argument name creation
 *     typ                  type to lift
 *     expr1, expr2         expressions of type typ, to which the lifting 
 *         		    is applied
 *)

val fullrellift : 
  ('cl, 'mem) concrete_lifting_operations -> 
    (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
      (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
	('cl, 'mem) param_lifting_type -> 
	  ('cl, 'mem) self_lifting_type -> 
	    ('cl, 'mem) Name_space.t ->
	      ('cl, 'mem) top_pre_types -> 
		('cl, 'mem) top_pre_expressions -> 
		  ('cl, 'mem) top_pre_expressions -> 
		    ('cl, 'mem) top_pre_formulas


(* same as fullrellift, but with param_lifting = [] *)

val rellift : 
  ('cl, 'mem) concrete_lifting_operations -> 
    (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
      (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
	('cl, 'mem) self_lifting_type -> 
	  ('cl, 'mem) Name_space.t ->
	    ('cl, 'mem) top_pre_types -> 
	      ('cl, 'mem) top_pre_expressions -> 
		('cl, 'mem) top_pre_expressions -> 
		  ('cl, 'mem) top_pre_formulas


(* version of fullrellift for argument lists of components and ancestors
 * this function performs separation of variances, so the resulting list 
 * might be longer than the argument list supplied
 *
 * Arguments
 *     lifting_ops	    concrete operations
 *     self1_subst          substitution function for Self for expr1
 *     self2_subst          substitution function for Self for expr2
 *     param_rel_lifting    the lifting for typeparameters
 *     self_lifting	    the lifting for Self/Carrier
 *     top_name_space       name space for argument name creation
 *     parameter_list       original parameter list of the 
 *			    ancestor or component (needed to get variances)
 *     argument_list        argument type list that drives the lifting
 *
 *)


val argument_list_fullrellift :
  ('cl, 'mem) concrete_lifting_operations -> 
    (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
      (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
	('cl, 'mem) param_lifting_type -> 
	  ('cl, 'mem) self_lifting_type -> 
	    ('cl, 'mem) Name_space.t ->
	      ('cl, 'mem) top_pre_parameter_type list -> 
		('cl, 'mem) top_pre_argument_type list -> 
		  ('cl, 'mem) top_pre_expressions list


(**************************************************************************
 *
 * exported version of inlined adt relation lifting 
 * 
 *     lifting_ops	  -> concrete operations
 *     self1_subst        -> substitution function for Self for selfvar1
 *     self2_subst        -> substitution function for Self for selfvar2
 *     param_lifting	  -> lifting for type parameters
 *     self_lifting	  -> lifting for self
 *     top_name_space     -> name space for name creation
 *     adt		  -> the adt to lift
 *     expr1, expr2       -> the expressions
 *)

val inline_adt_rel_lifting:
  ('cl, 'mem) concrete_lifting_operations -> 
    (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
      (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
	('cl, 'mem) param_lifting_type -> 
	  ('cl, 'mem) self_lifting_type -> 
	    ('cl, 'mem) Name_space.t ->
	      'cl ->
		('cl, 'mem) top_pre_expressions -> 
		  ('cl, 'mem) top_pre_expressions -> 
		    ('cl, 'mem) top_pre_expressions
		


(******************************
 * Same as fullrellift, with the following differences:
 * - the lifting wrt. to one method instead of a type
 * - applications of this method occur in standard form: m(c)( args )
 *
 * parameters
 *     lifting_ops	  -> concrete operations
 *     param_lifting	  -> lifting for type parameters
 *     self_lifting	  -> lifting for self
 *     name_space         -> name space for name creation
 *     self1_subst        -> substitution function for Self for selfvar1
 *     self2_subst        -> substitution function for Self for selfvar2
 *     coalgebra1
 *     coalgebra2         -> the coalgebras for getting the methods right
 *     meth		  -> the method
 *     selfvar1
 *     selfvar2           -> the variables of Self, to which we apply
 *           		     the lifting
 * 
 *)

val class_method_full_rel_lifting :
  ('cl, 'mem) concrete_lifting_operations -> 
    (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
      (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
	('cl,'mem) param_lifting_type ->
	  ('cl,'mem) self_lifting_type ->
	    ('cl, 'mem) Name_space.t -> 
	      ('cl, 'mem) top_pre_expressions ->
		('cl, 'mem) top_pre_expressions ->
		  'mem ->
		    string -> 
		      string ->
			('cl, 'mem) top_pre_formulas 


(***************************************************************** 
 * relation lifting for adt's, we do the whole adt here and 
 * generate the arguments for reduce
 *
 * Because of this higher order definition of the lifting, 
 * substitutions play a important role, and are quite difficult.
 * To avoid to much code duplication, this fuction computes the 
 * substitution for the adt relation lifting. It is exported, 
 * because the substitutions are needed, to generate precise 
 * importings.
 * 
 * arguments
 * 	   adt		the adt
 * 	   typed_args1  	parameter list 1
 * 	   typed_args2  	parameter list 2
 * result 
 * 	   a pair of substitution functions
 *)

val adt_rel_lifting_substitutions :
  ccsl_iface_type ->
    ccsl_parameter_type list -> 
	 ccsl_parameter_type list -> 
	   ((ccsl_output_types * ccsl_output_types) list *
	    (ccsl_output_types * ccsl_output_types) list)


(***********************************************************************
 * now the real adt lifting
 *
 * arguments
 *   ns 			name space
 *   type_param1		first set of type parameters
 *   type_param2		second set of type parameters
 *   adt			the adt for which the lifting is done
 *   param_rel_assoc	an assoc list for parameter relations,
 * 			   this should associate original parameters 
 * 			   from the adt with relations. Equal is taken
 * 			   for missing associations.
 * 
 * The result is a list of expressions, which can be applied to a 
 * reduce.
 * 
 *)


val adt_rel_lifting :
  ccsl_parameter_type list -> 
    ccsl_parameter_type list -> 
      (ccsl_iface_type, ccsl_member_type) param_lifting_type -> 
	ccsl_name_space_type ->
	   ccsl_iface_type -> 
	     ccsl_expressions list
  


(***********************************************************************
 ***********************************************************************
 *
 *  Here, we generate the constructor declarations
 *
 *  step_constructor_declarations ns l
 *  takes arguments
 *    ns  --> the namespace
 *    cl  --> the class description for wich the adt is generated
 *
 * delivers a list of constructor declarations
 *)


val step_constructor_declarations:
  (ccsl_iface_type, ccsl_member_type)
  Name_space.t ->
    ccsl_iface_type ->
	 (ccsl_iface_type, ccsl_member_type) 
	 Top_variant_types.top_pre_theory_declaration 

val admissible_cases_for_method:
  (ccsl_iface_type, ccsl_member_type)
  Name_space.t ->
    ccsl_member_type ->
	 (ccsl_iface_type, ccsl_member_type) top_pre_expressions ->
	   ((ccsl_iface_type, ccsl_member_type) top_pre_expressions * 
	    (ccsl_iface_type, ccsl_member_type) top_pre_expressions) list

val transition_cases_for_method:
  (ccsl_iface_type, ccsl_member_type)
  Name_space.t ->
    ccsl_member_type ->
	 (ccsl_iface_type, ccsl_member_type) top_pre_expressions ->
	   (ccsl_iface_type, ccsl_member_type) top_pre_expressions ->
	     ((ccsl_iface_type, ccsl_member_type) top_pre_expressions * 
	      (ccsl_iface_type, ccsl_member_type) top_pre_expressions) list



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

