(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: absorb.v,v 1.41 2013/04/10 12:06:16 tews Exp $
 *)


(** * Syntactic cut elimination, Section 5 *)

(** ** Absorbtion properties and non-atomic axioms, 5.1 - 5.6.1

      This module contains the absorbtion properties 5.1, 5.3 and 5.5.
      It also contains the proof for the admisibility of non-atomic
      axioms in GR_n (5.6.1). In this formalization, this proof does
      not depend on cut elimination or contraction.
*)

Require Export factor_subst propositional_properties backward_substitution
               cut_properties.

Section Absorbtion.

  Variable V : Type.
  Variable L : modal_operators.

  (** Need decidable equalities on operators and variables 
      for the support in absorbs_contraction.
   *)
  Variable op_eq : eq_type (operator L).
  Variable v_eq : eq_type V.


  (***************************************************************************)
  (** *** Two properties about one-step rules

         They are not in module 
         #<A HREF="rule_sets.html"><span class="inlinecode">rule_sets</span></A># 
         because they depend on module
         #<A HREF="renaming.html"><spanclass="inlinecode">renaming</span></A>#. 
    *)
  (***************************************************************************)

  Lemma one_step_rule_simple_modal_conclusion_subst_reorder :
    forall(rules : set (sequent_rule V L))(sigma : lambda_subst V L)
          (r : sequent_rule V L)(s : sequent V L),
      one_step_rule_set rules ->
      rules r -> 
      renaming sigma ->
      list_reorder s (subst_sequent sigma (conclusion r)) ->
        simple_modal_sequent s.
  Proof.
    intros rules sigma r s H H0 H1 H2.
    eapply simple_modal_sequent_list_reorder.
      apply list_reorder_symm.
      eexact H2.
    apply simple_modal_sequent_renaming.
      trivial.
    apply one_step_rule_simple_modal_conclusion.
    apply H.
    trivial.
  Qed.

  Lemma one_step_rule_propositional_subst_assumptions :
    forall(rules : set (sequent_rule V L))(sigma : lambda_subst V L)
          (r : sequent_rule V L),
      one_step_rule_set rules ->
      rules r -> 
      renaming sigma ->
        every_nth propositional_sequent 
          (map (subst_sequent sigma) (assumptions r)).
  Proof.
    intros rules sigma r H H0 H1.
    apply rank_subst_assumptions; auto.
    apply rank_renaming.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Towards absorbs congruence, Definition 5.1 *)
  (***************************************************************************)

  (** First define the assumptions used in absorbtion of congruence. *)
  Definition congruence_assumption_list(pv qv : list (lambda_formula V L))
                                          : list (list (lambda_formula V L)) :=
    let pqv := combine pv qv 
    in
      (map (fun pq => [lf_neg (fst pq); snd pq]) pqv) ++
      (map (fun pq => [fst pq; lf_neg (snd pq)]) pqv).

  Definition congruence_assumptions(n : nat)
                              (pv qv : counted_list (lambda_formula V L) n) 
                                                        : set (sequent V L) :=
    reordered_sequent_list_set
      (congruence_assumption_list (list_of_counted_list pv)
                                  (list_of_counted_list qv)).

  Lemma congruence_assumptions_char :
    forall(n : nat)(args : counted_list (lambda_formula V L) n)
          (s : sequent V L),
      every_nth prop_form (list_of_counted_list args) ->
      congruence_assumptions n args args s ->
        (exists(v : V), s = [lf_neg (lf_prop v); lf_prop v] \/
                        s = [lf_prop v; lf_neg (lf_prop v)]
        ) /\ rank_sequent 1 s.
  Proof.
    intros n args s H H0.
    assert (exists v : V, s = [lf_neg (lf_prop v); lf_prop v] \/ 
                          s = [lf_prop v; lf_neg (lf_prop v)]).
      unfold congruence_assumptions, congruence_assumption_list, 
             reordered_sequent_list_set in *.
      decompose [ex and] H0; clear H0.
      rename x into s'.
      apply in_app_or in H3.
      destruct H3.
        rewrite in_map_iff in H0.
        decompose [ex and] H0; clear H0.
        destruct x as [p p0].
        simpl in *.
        apply in_combine_same in H4.
        destruct H4.
        subst p0 s'.
        apply list_reorder_symm in H2.
        apply list_reorder_2_char in H2.
        apply every_nth_In with (a := p) in H.
          destruct p; try contradiction.
          exists v.
          auto.
        trivial.
      rewrite in_map_iff in H0.
      decompose [ex and] H0; clear H0.
      destruct x as [p p0].
      simpl in *.
      apply in_combine_same in H4.
      destruct H4.
      subst p0 s'.
      apply list_reorder_symm in H2.
      apply list_reorder_2_char in H2.
      apply every_nth_In with (a := p) in H.
        destruct p; try contradiction.
        exists v.
        destruct H2.
          auto.
        auto.
      trivial.
    split.
      trivial.
    destruct H1 as [v].
    destruct H1.
      subst s.
      apply rank_sequent_cons.
        apply rank_formula_lf_neg.
        apply rank_formula_prop_form.
        simpl.
        trivial.
      apply rank_sequent_cons.
        apply rank_formula_prop_form.
        simpl.
        trivial.
      apply rank_sequent_empty.
    subst s.
    apply rank_sequent_cons.
      apply rank_formula_prop_form.
      simpl.
      trivial.
    apply rank_sequent_cons.
      apply rank_formula_lf_neg.
      apply rank_formula_prop_form.
      simpl.
      trivial.
    apply rank_sequent_empty.
  Qed.

  Lemma rank_sequent_set_congruence_assumptions :
    forall(n : nat)(args : counted_list (lambda_formula V L) n),
      every_nth prop_form (list_of_counted_list args) ->
        rank_sequent_set 1 (congruence_assumptions n args args).
  Proof.
    unfold rank_sequent_set in *.
    intros n args H s H0.
    decompose [ex and or] (congruence_assumptions_char n args s H H0).
      trivial.
    trivial.
  Qed.

  Lemma congruence_assumptions_subset :
    forall(n r : nat)(args : counted_list (lambda_formula V L) n)
          (sigma : lambda_subst V L),
      rank_subst (S r) sigma ->
      every_nth prop_form (list_of_counted_list args) ->
        subset (subst_sequent_set sigma (congruence_assumptions n args args))
               (subst_Ax_n_set sigma (2 + r)).
  Proof.
    clear. 
    intros n r args sigma H H0 s H1.
    unfold subst_sequent_set, subst_Ax_n_set in *.
    decompose [ex and] H1; clear H1.
    subst s.
    rename x into s.
    decompose [ex and or] (congruence_assumptions_char n args s H0 H3).
      rename x into v.
      subst s.
      exists (subst_sequent sigma [lf_prop v; lf_neg (lf_prop v)]), [].
      simpl.
      repeat split.
          exists v.
          repeat rewrite subst_form_char.
          trivial.
        apply list_reorder_swap_head.
      apply rank_sequent_empty.
    rename x into v.
    subst s.
    exists (subst_sequent sigma [lf_prop v; lf_neg (lf_prop v)]), [].
    simpl.
    repeat split.
        exists v.
        repeat rewrite subst_form_char.
        trivial.
      apply list_reorder_refl.
    apply rank_sequent_empty.
  Qed.

  Lemma congruence_assumptions_provable_with_ax :
    forall(rules : set (sequent_rule V L))(n : nat)
          (args : counted_list (lambda_formula V L) n)
          (s : sequent V L),
      subset (rank_rules 1 is_ax_rule) rules ->
      every_nth prop_form (list_of_counted_list args) ->
        (provable rules (congruence_assumptions n args args) s <->
         provable rules (empty_sequent_set V L) s).
  Proof.
    clear. 
    intros rules n args s H H0.
    split.
    - intros H1.
      apply plug_empty_hypothesis_proof
            with (provable_hyp := congruence_assumptions n args args).
      + clear - H H0.
        intros s H1.
        decompose [ex and or] (congruence_assumptions_char n args s H0 H1).
        * rename x into v.
          subst s.
          apply provable_with_rule with (assum := []).
          {- apply H.
             split.
             + split.
               * trivial.
               * simpl.
                 exists 1, 0, v.
                 apply dep_conj with (a := lt_n_Sn _).
                 assert (0 < 2).
                 {- omega. }
                 {- apply dep_conj with (a := H2).
                    auto.
                 }
             + split.
               * apply every_nth_empty.
               * trivial.
          }
          {- apply every_nth_empty. }
        * rename x into v.
          subst s.
          apply provable_with_rule with (assum := []).
          {- apply H.
             split.
             + split.
               * trivial.
               * simpl.
                 exists 0, 1, v.
                 assert (0 < 2).
                 {- omega. }
                 {- apply dep_conj with (a := H2).
                    apply dep_conj with (a := lt_n_Sn _).
                    auto.
                 }
             + split.
               * apply every_nth_empty.
               * trivial.
          }
          {- apply every_nth_empty. }
      + trivial.
    - intros H1.
      eapply proof_mono_hyp.
      + apply subset_empty_set.
      + trivial.
  Qed.


  (** **** Definition 5.1, page 25 *)
  Definition absorbs_congruence(rules : set (sequent_rule V L)) : Prop :=
    forall(op : operator L)
          (pv qv : counted_list (lambda_formula V L) (arity L op)),
      every_nth prop_form (list_of_counted_list pv) ->
      every_nth prop_form (list_of_counted_list qv) ->
        exists(r : sequent_rule V L)(sigma : lambda_subst V L),
          rules r /\
          rank_subst 1 sigma /\
          multi_subset (subst_sequent sigma (conclusion r))
                       [lf_neg (lf_modal op pv); lf_modal op qv]
          /\
          every_nth
            (fun(s : sequent V L) =>
               provable (GC_n_set V L 1)
                        (congruence_assumptions (arity L op) pv qv)
                 (subst_sequent sigma s))
            (assumptions r).


  (***************************************************************************)
  (** *** Absorbs contraction, Definition 5.3  *)
  (***************************************************************************)

  Definition absorbs_contraction(rules : set (sequent_rule V L)) : Prop :=
    forall(or : sequent_rule V L)(sigma : lambda_subst V L),
      rules or ->
      renaming sigma ->
        exists(cr : sequent_rule V L)(rho : lambda_subst V L),
          rules cr /\ renaming rho /\ 
          multi_subset
            (subst_sequent rho (conclusion cr))
            (sequent_support op_eq v_eq (subst_sequent sigma (conclusion or)))
          /\
          every_nth
            (fun ca => 
               provable (GC_n_set V L 1) 
                        (reordered_sequent_list_set 
                                 (map (subst_sequent sigma) (assumptions or)))
                 (subst_sequent rho ca))
            (assumptions cr).

  Lemma absorbs_contraction_head :
    forall(rules : set (sequent_rule V L))(or : sequent_rule V L)
          (sigma : lambda_subst V L)(f : lambda_formula V L)(s : sequent V L),
      one_step_rule_set rules ->
      absorbs_contraction rules ->
      rules or ->
      renaming sigma ->
      list_reorder (f :: f :: s) (subst_sequent sigma (conclusion or)) ->
        exists(cr : sequent_rule V L)(rho : lambda_subst V L)
              (delta : sequent V L),
          rules cr /\ renaming rho /\ simple_modal_sequent delta /\
          list_reorder (f :: s) ((subst_sequent rho (conclusion cr)) ++ delta)
          /\
          every_nth
            (fun ca => 
               provable (GC_n_set V L 1) 
                        (reordered_sequent_list_set 
                                 (map (subst_sequent sigma) (assumptions or)))
                 (subst_sequent rho ca))
            (assumptions cr).
  Proof.
    intros rules or sigma f s H H0 H1 H2 H3.
    specialize (H0 or sigma H1 H2).
    decompose [ex and] H0; clear H0.
    rename x into cr, x0 into rho.
    assert (simple_modal_sequent (f :: s)).
      eapply simple_modal_sequent_tail.
      eapply one_step_rule_simple_modal_conclusion_subst_reorder 
             with (4 := H3); eauto.
    apply (sequent_support_reorder op_eq v_eq) in H3; trivial.
    rewrite sequent_support_head_contract in H3; trivial.
    assert (H9 := sequent_support_correct_subset op_eq v_eq (f :: s)).
    unfold multi_subset in *.
    destruct H6 as [cr_delta].
    destruct H9 as [fs_dups].
    assert (list_reorder (f :: s) 
                 (subst_sequent rho (conclusion cr) ++ cr_delta ++ fs_dups)).
      clear - H3 H6 H7.
      eapply list_reorder_trans.
        apply list_reorder_symm.
        eexact H7.
      rewrite app_assoc.
      apply list_reorder_append_right.
      eapply list_reorder_trans.
        eexact H3.
      apply list_reorder_symm.
      trivial.
    exists cr, rho, (cr_delta ++ fs_dups).
    split; trivial.
    split; trivial.
    split.
      eapply simple_modal_sequent_append_right.
      eapply simple_modal_sequent_list_reorder.
        eexact H9.
      trivial.
    split; trivial.
  Qed.


  (***************************************************************************)
  (** *** Absorbs cut, Definition 5.5 *)
  (***************************************************************************)

  Definition absorbs_cut(rules : set (sequent_rule V L)) : Prop :=
    forall(rl rr : sequent_rule V L)(sl sr : lambda_subst V L)(nl nr : nat)
          (nl_less : nl < length (subst_sequent sl (conclusion rl)))
          (nr_less : nr < length (subst_sequent sr (conclusion rr))),
      rules rl -> rules rr ->
      renaming sl -> renaming sr ->
      lf_neg (nth (subst_sequent sl (conclusion rl)) nl nl_less) =
      nth (subst_sequent sr (conclusion rr)) nr nr_less ->
        exists(rb : sequent_rule V L)(sb : lambda_subst V L),
          rules rb /\ renaming sb /\ 
          multi_subset
            (sequent_support op_eq v_eq (subst_sequent sb (conclusion rb)))
            ((cutout_nth (subst_sequent sl (conclusion rl)) nl) ++
             (cutout_nth (subst_sequent sr (conclusion rr)) nr)) /\
          every_nth
            (fun ba => 
               provable (GC_n_set V L 1)
                        (reordered_sequent_list_set 
                           ((map (subst_sequent sl) (assumptions rl)) ++
                            (map (subst_sequent sr) (assumptions rr))))
                 (subst_sequent sb ba))
            (assumptions rb).


  (***************************************************************************)
  (** *** Towards non-atomit axioms in GR_n, 5.6 (1)  *)
  (***************************************************************************)

  (** Induction step for the proposition *)
  Lemma GR_n_non_atomic_axiom_head :
    forall(rules : set (sequent_rule V L))(n : nat)
          (s : sequent V L)(f : lambda_formula V L),
      countably_infinite V ->
      one_step_rule_set rules ->
      absorbs_congruence rules ->
      rank_sequent (2 + n) s ->
      rank_formula (2 + n) f ->
      (forall (s : sequent V L) (f : lambda_formula V L),
         rank_sequent (S n) s ->
         rank_formula (S n) f ->
           provable (GR_n_set rules (S n)) (empty_sequent_set V L)
                    (f :: lf_neg f :: s)) ->
        provable (GR_n_set rules (2 + n)) (empty_sequent_set V L)
                 (f :: lf_neg f :: s).
  Proof.
    intros rules n s f H H0 H1 H2 H3 H4.
    apply countably_infinite_non_empty with (X := V); trivial.
    intro nonempty_v.
    eapply list_weakening_admissible_GR_n with (s1 := [f; lf_neg f]); eauto.
      apply lt_0_Sn.
    decompose [ex and] (prop_back_subst_form_prop v_eq f (S n) H H3).
    rename x into simple_f, x0 into sigma.
    subst f.
    apply plug_empty_hypothesis_proof
          with (provable_hyp := subst_Ax_n_set sigma (2 + n)).
      clear s H2 simple_f H6 H3.
      intros s H2.
      unfold subst_Ax_n_set in *.
      decompose [ex and] H2; clear H2.
      rename x into s_ax, x0 into s_weak.
      eapply provable_GR_n_list_reorder.
        eexact H5.
      eapply list_weakening_admissible_GR_n; eauto.
        apply lt_0_Sn.
      clear s s_weak H5 H8.
      unfold subst_Ax_set in *.
      destruct H3 as [v].
      subst s_ax.
      specialize (H9 v).
      destruct H9.
        eapply proof_mono_rules.
          eapply subset_trans.
            apply prop_G_subset_Gn with (n := 2 + n).
            omega.
          apply G_n_subset_GR_n.
        apply G_non_atomic_axiom_head.
            apply sequent_multiset_empty.
          apply prop_form_is_propositional.
          trivial.
        apply propositional_sequent_empty.
      specialize (H7 v).
      destruct (sigma v); try contradiction.
      clear sigma v H2.
      rename c into args.
      decompose [ex and] (mod_arg_back_subst_mod_args_prop v_eq 
                            (arity L op) args n H 
                              (rank_formula_modal_args_TCC _ _ _ H7)).
      rename x into simple_args, x0 into tau.
      apply plug_empty_hypothesis_proof
            with (provable_hyp := subst_Ax_n_set tau (2 + n)).
        clear op args H7 simple_args H2 H3.
        intros s H2.
        unfold subst_Ax_n_set in *.
        decompose [ex and] H2; clear H2.
        rename x into s_ax, x0 into s_weak.
        eapply provable_GR_n_list_reorder.
          eexact H5.
        eapply list_weakening_admissible_GR_n; eauto.
          apply lt_0_Sn.
        clear s s_weak H5 H8.
        eapply proof_mono_rules.
          apply rank_rules_subset_rank with (n1 := S n).
          omega.
        unfold subst_Ax_set in *.
        destruct H3 as [v].
        subst s_ax.
        apply H4.
          apply rank_sequent_empty.
        apply H6.
      assert ([lf_modal op args; lf_neg (lf_modal op args)] =
                  subst_sequent tau [lf_modal op simple_args;
                                lf_neg (lf_modal op simple_args)]).
        rewrite <- H2.
        simpl.
        rewrite subst_form_char.
        rewrite subst_form_char.
        rewrite subst_form_char.
        trivial.
      rewrite H5.
      clear H5.
      eapply proof_mono_hyp.
        apply set_equal_implies_subset.
        apply set_equal_union_subset_right.
        apply congruence_assumptions_subset.
          trivial.
        eexact H3.
      apply GR_n_substitution_lemma with (n := 2)(k := n); trivial.
        apply rank_sequent_set_mono with (n1 := 1).
          apply le_n_Sn.
        apply rank_sequent_set_congruence_assumptions.
        trivial.
      clear H4 args H7 tau H2 H6.
      decompose [ex and] (H1 op simple_args simple_args H3 H3).
      rename x into r, x0 into rho.
      apply provable_with_rule 
                    with (assum := map (subst_sequent rho) (assumptions r)).
        clear H7.
        apply R_n_subset_GR_n.
        assert (0 < 2).
          omega.
        apply stratified_one_step_rules 
                       with (1 := v_eq)(2 := nonempty_v)(npos := H6).
          trivial.
        destruct H5 as [delta].
        exists r, rho, delta.
        repeat split; auto.
          eapply rank_sequent_append_right.
          eapply rank_sequent_list_reorder.
            eexact H5.
          assert (rank_formula 2 (lf_modal op simple_args)).
            apply rank_formula_simple_modal_form.
            trivial.
          apply rank_sequent_cons.
            apply rank_formula_lf_neg.
            trivial.
          apply rank_sequent_cons.
            trivial.
          apply rank_sequent_empty.
        simpl.
        eapply list_reorder_trans.
          apply list_reorder_swap_head.
        apply list_reorder_symm.
        trivial.
      apply every_nth_map.
      clear - H0 H3 H2 H7.
      intros i i_less.
      specialize (H7 i i_less).
      simpl in *.
      eapply proof_mono_rules.
        eapply subset_trans.
          apply (G_n_set_mono _ _ 1 2).
          apply le_n_Sn.
        apply G_n_subset_GR_n.
      rewrite congruence_assumptions_provable_with_ax in H7; trivial.
        rewrite congruence_assumptions_provable_with_ax; trivial.
          apply provable_GC_1_G_1.
          trivial.
        apply ax_n_subset_G_n.
      eapply subset_trans.
        apply ax_n_subset_G_n.
      apply G_n_subset_GC_n.
    change [subst_form sigma simple_f; lf_neg (subst_form sigma simple_f)]
       with (subst_sequent sigma [simple_f; lf_neg simple_f]).
    apply GR_n_substitution_lemma_empty_hyp with (n := 1)(k := S n); trivial.
    eapply proof_mono_rules.
      apply G_n_subset_GR_n.
    apply G_non_atomic_axiom_head.
        apply sequent_multiset_empty.
      trivial.
    apply rank_sequent_empty.
  Qed.

  (** **** Propsition 5.6 (1), non atomic axioms are admissible in GR_n *)
  Lemma syntactic_GR_n_non_atomic_axioms : 
    forall(rules : set (sequent_rule V L))(n : nat)
          (s : sequent V L)(f : lambda_formula V L),
      countably_infinite V ->
      one_step_rule_set rules ->
      absorbs_congruence rules ->
      rank_sequent n s -> 
      rank_formula n f ->
        provable (GR_n_set rules n) (empty_sequent_set V L)
                 (f :: (lf_neg f) :: s).
  Proof.
    induction n.
      intros s f H H0 H1 H2 H3.
      eapply rank_formula_zero_TCC; eauto.
    intros s f H H0 H1 H2 H3.
    destruct n.
      clear IHn.
      eapply proof_mono_rules.
        apply G_n_subset_GR_n.
      apply G_non_atomic_axiom_head; trivial.
      apply sequent_multiset_empty.
    apply GR_n_non_atomic_axiom_head; trivial.
    intros s0 f0 H4 H5.
    apply IHn; trivial.
  Qed.

  Lemma syntactic_GR_n_provable_subst_Ax :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))
          (sigma : lambda_subst V L)(n : nat)(s : sequent V L),
      countably_infinite V ->
      one_step_rule_set rules ->
      absorbs_congruence rules ->
      rank_subst n sigma ->
      subst_Ax_n_set sigma n s ->
        provable (GR_n_set rules n) hyp s.
  Proof.
    intros rules hyp sigma n s H H0 H1 H2 H3.
    eapply proof_mono_hyp.
      apply subset_empty_set.
    unfold subst_Ax_n_set, subst_Ax_set in *.
    decompose [ex and] H3; clear H3.
    subst x.
    rename x0 into delta, x1 into v.
    change (empty_set (sequent V L)) with (empty_sequent_set V L).
    eapply provable_GR_n_list_reorder.
      eexact H4.
    clear H4.
    simpl.
    apply syntactic_GR_n_non_atomic_axioms; auto.
  Qed.

End Absorbtion.

Implicit Arguments absorbs_congruence [V L].
Implicit Arguments absorbs_contraction [V L].
Implicit Arguments absorbs_contraction_head [V L].
Implicit Arguments absorbs_cut [V L].
