(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: backward_substitution.v,v 1.11 2013/04/10 11:17:13 tews Exp $
 *)

(** ** Substitution construction

      This module defines two functions that replace subformulas with
      propositional variables, thus constructing a simpler formula and
      a substitution. The first function replaces the arguments in a
      modality. This is needed in the completeness proof 4.13. The
      second one replaces whole modal formulas. This is needed for
      non-atomic axioms in 5.6.1.
*)

Require Export propositional_formulas plain_prop_mod.

Section Back_subst.

  (***************************************************************************)
  (** *** Backward substitution for arguments of modalities
         Substitute unique variables for formulas, needed in 4.13 *)
  (***************************************************************************)

  Variable V : Type.
  Variable L : modal_operators.

  (** Need a decidable equality on propositional constants for 
      function update in substitutions.
   *)
  Variable v_eq : eq_type V.


  (***************************************************************************)
  (** **** Function definition  *)
  (***************************************************************************)

  Definition back_subst_result(form_type : Type) : Type :=
    ( form_type * (lambda_subst V L) * nat )%type.


  Fixpoint mod_arg_back_subst_mod_args(f : nat -> V)(next : nat)
               (subst : lambda_subst V L)
               (n : nat)(args : counted_list (lambda_formula V L) n) :
                    back_subst_result (counted_list (lambda_formula V L) n) :=
    match args with
      | counted_nil => (counted_nil, subst, next)
      | counted_cons n arg args =>
        let (res_subst, next) := 
            mod_arg_back_subst_mod_args f next subst n args in
        let (res, subst) := res_subst
        in
          (counted_cons (lf_prop (f next)) res,
           function_update v_eq subst (f next) arg,
           S next)
    end.

  Definition mod_arg_back_subst_modal_form(f : nat -> V)(next : nat)
               (subst : lambda_subst V L)
               (form : lambda_formula V L)(mod_form : modal_form form)
                                   : back_subst_result (lambda_formula V L) :=
    match form return modal_form form -> back_subst_result (lambda_formula V L)
    with
      | lf_modal op args => fun _ =>
        let (res_subst, next) := 
                mod_arg_back_subst_mod_args f next subst (arity L op) args in
        let (res, subst) := res_subst
        in
          (lf_modal op res, subst, next)
      | _ => fun(H : False) => False_rect _ H
    end mod_form.

  Definition mod_arg_back_subst_neg_mod_form(f : nat -> V)(next : nat)
               (subst : lambda_subst V L)
               (form : lambda_formula V L)
               (neg_mod_form : top_modal_form form)
                                   : back_subst_result (lambda_formula V L) :=
    match form 
        return top_modal_form form
                     -> back_subst_result (lambda_formula V L)
    with
      | lf_neg form => fun(H : modal_form form) =>
        let (res_subst, next) := 
            mod_arg_back_subst_modal_form f next subst form H in
        let (res, subst) := res_subst
        in (lf_neg res, subst, next)
      | lf_modal op args => fun(H : modal_form (lf_modal op args)) =>
          mod_arg_back_subst_modal_form f next subst (lf_modal op args) H
      | _ => fun(H : False) => False_rect _ H
    end neg_mod_form.

  Fixpoint mod_arg_back_subst_sequent_rec(f : nat -> V)(next : nat)
               (subst : lambda_subst V L)
               (s : sequent V L)(top_mod_s : top_modal_sequent s)
                                          : back_subst_result (sequent V L) :=
    match s return top_modal_sequent s -> back_subst_result (sequent V L) 
    with
      | [] => fun _ => ([], subst, next)
      | form :: seq => fun(H : top_modal_sequent (form :: seq)) =>
        let (res_subst, next) := 
              mod_arg_back_subst_sequent_rec f next subst seq 
                                                  (every_nth_tail _ _ _ H) in
        let (sres, subst) := res_subst in
        let (res_subst, next) := 
              mod_arg_back_subst_neg_mod_form f next subst form 
                                                  (every_nth_head _ _ _ H) in
        let (fres, subst) := res_subst
        in
          (fres :: sres, subst, next)
    end top_mod_s.

  Definition mod_arg_back_subst_sequent(f : nat -> V)
                  (s : sequent V L)(top_mod_s : top_modal_sequent s)
                                      : (sequent V L) * (lambda_subst V L) :=
    let (res, _) := mod_arg_back_subst_sequent_rec f 0 id_subst s top_mod_s
    in res.


  (***************************************************************************)
  (** **** Utility predicates  *)
  (***************************************************************************)

  Definition subst_equal_below(f : nat -> V)(n : nat)
                              (sigma1 sigma2 : lambda_subst V L) : Prop :=
    forall(i : nat),
      i < n ->
      sigma1 (f i) = sigma2 (f i).

  Definition subst_equal_below_refl :
    forall(f : nat -> V)(n : nat)(sigma : lambda_subst V L),
      subst_equal_below f n sigma sigma.
  Proof.
    unfold subst_equal_below in *.
    intros f n sigma i H.
    trivial.
  Qed.

  Definition subst_equal_below_trans :
    forall(f : nat -> V)(n : nat)(sigma1 sigma2 sigma3 : lambda_subst V L),
      subst_equal_below f n sigma1 sigma2 ->
      subst_equal_below f n sigma2 sigma3 ->
        subst_equal_below f n sigma1 sigma3.
  Proof.
    unfold subst_equal_below in *.
    intros f n sigma1 sigma2 sigma3 H H0 i H1.
    rewrite H.
      apply H0.
      trivial.
    trivial.
  Qed.

  Definition subst_equal_below_single_update :
    forall(f : nat -> V)(n1 n2 : nat)(sigma : lambda_subst V L)
          (form : lambda_formula V L),
      injective f ->
      n1 <= n2 ->
        subst_equal_below f n1 
          (function_update v_eq sigma (f n2) form)
          sigma.
  Proof.
    unfold subst_equal_below in *.
    intros f n1 n2 sigma form H H0 i H1.
    apply function_update_unequal.
      trivial.
    intros H2.
    apply H in H2.
    omega.
  Qed.

  Definition subst_equal_below_mono :
    forall(f : nat -> V)(n1 n2 : nat)(sigma1 sigma2 : lambda_subst V L),
      subst_equal_below f n1 sigma1 sigma2 ->
      n2 <= n1 ->
        subst_equal_below f n2 sigma1 sigma2.
  Proof.
    unfold subst_equal_below in *.
    intros f n1 n2 sigma1 sigma2 H H0 i H1.
    apply H.
    omega.
  Qed.

  Definition prop_var_list_below(f : nat -> V)(n : nat)
                                (prop_vars : list V) : Prop :=
    forall(v : V),
      In v prop_vars ->
      exists(i : nat),
        i < n /\ f i = v.

  Lemma prop_var_list_below_singleton :
    forall(f : nat -> V)(bound n : nat),
      n < bound ->
        prop_var_list_below f bound [f n].
  Proof.
    unfold prop_var_list_below in *.
    simpl.
    intros f bound n H v H0.
    destruct H0.
      exists n.
      auto.
    contradiction.
  Qed.

  Lemma prop_var_list_below_empty :
    forall(f : nat -> V)(n : nat),
      prop_var_list_below f n [].
  Proof.
    clear. 
    unfold prop_var_list_below in *.
    intros f n v H.
    contradiction.
  Qed.

  Lemma prop_var_list_below_append :
    forall(f : nat -> V)(n : nat)(pv1 pv2 : list V),
      prop_var_list_below f n pv1 ->
      prop_var_list_below f n pv2 ->
        prop_var_list_below f n (pv1 ++ pv2).
  Proof.
    clear. 
    unfold prop_var_list_below in *.
    intros f n pv1 pv2 H H0 v H1.
    apply in_app_or in H1.
    destruct H1.
      apply H.
      trivial.
    apply H0.
    trivial.
  Qed.

  Lemma subst_eq_on_vars_below :
    forall(f : nat -> V)(n : nat)
          (sigma1 sigma2 : lambda_subst V L)(prop_vars : list V),
      subst_equal_below f n sigma1 sigma2 ->
      prop_var_list_below f n prop_vars ->
        subst_eq_on_vars sigma1 sigma2 prop_vars.
  Proof.
    unfold subst_eq_on_vars.
    intros f n sigma1 sigma2 prop_vars H H0 v H1.
    apply H0 in H1.
    decompose [ex and] H1; clear H1.
    subst v.
    apply H.
    trivial.
  Qed.


  (***************************************************************************)
  (** **** Prove substitution property for mod arg back subst  *)
  (***************************************************************************)

  Lemma mod_arg_back_subst_mod_args_prop_ind :
    forall(f : nat -> V)(arity : nat)
          (args back_args : counted_list (lambda_formula V L) arity)
          (r n n_res : nat)(sigma sigma_res : lambda_subst V L),
      injective f ->
      every_nth (rank_formula (S r)) (list_of_counted_list args) ->
      rank_subst (S r) sigma ->
      mod_arg_back_subst_mod_args f n sigma arity args = 
                                (back_args, sigma_res, n_res) ->
        (forall(bound : nat),
           n_res <= bound ->
             prop_var_list_below f bound (prop_var_modal_args arity back_args))
        /\ n <= n_res /\
        subst_equal_below f n sigma_res sigma /\
        counted_map (subst_form sigma_res) back_args = args /\
        every_nth prop_form (list_of_counted_list back_args) /\
        rank_subst (S r) sigma_res.
  Proof.
    induction args.
      intros back_args r n n_res sigma sigma_res H H0 H1 H2.
      simpl in *.
      inversion H2.
      subst sigma_res.
      clear H4 H6 H2.
      repeat split.
            unfold prop_var_list_below in *.
            intros bound H2 v H3.
            contradiction.
          trivial.
        apply every_nth_empty.
      trivial.
    rename a into first_arg.
    intros back_args r n0 n_res sigma sigma_res H H0 H1 H2.
    simpl in *.
    destruct (mod_arg_back_subst_mod_args f n0 sigma n args) 
       as [rs n_res_pred] eqn:?.
    destruct rs as [back_args_tail subst_res].
    inversion H2.
    subst back_args sigma_res n_res.
    clear H2.
    simpl.
    specialize (IHargs _ _ _ _ _ _ H (every_nth_tail _ _ _ H0) H1 Heqb).
    decompose [and] IHargs; clear IHargs.
    repeat split.
              intros bound H7.
              rewrite prop_var_modal_args_cons.
              apply prop_var_list_below_append.
                rewrite prop_var_formula_char.
                apply prop_var_list_below_singleton.
                trivial.
              apply H2.
              apply le_Sn_le.
              trivial.
            apply le_S.
            trivial.
          eapply subst_equal_below_trans.
            apply subst_equal_below_single_update.
              trivial.
            trivial.
          trivial.
        f_equal.
          rewrite subst_form_char.
          rewrite function_update_eq; trivial.
        rewrite <- H5.
        eapply subst_modal_args_eq.
          apply incl_refl.
        apply (subst_eq_on_vars_below f n_res_pred).
          apply subst_equal_below_single_update.
            trivial.
          trivial.
        apply H2.
        trivial.
      apply every_nth_cons.
        simpl.
        trivial.
      trivial.
    apply rank_subst_update.
      eapply every_nth_head.
      eexact H0.
    trivial.
  Qed.

  Lemma mod_arg_back_subst_mod_args_prop :
    forall(arity : nat)(args : counted_list (lambda_formula V L) arity)
          (rank : nat),
      countably_infinite V ->
      every_nth (rank_formula (S rank)) (list_of_counted_list args) ->
      exists(simple_args : counted_list (lambda_formula V L) arity)
            (sigma : lambda_subst V L),
        counted_map (subst_form sigma) simple_args = args /\
        every_nth prop_form (list_of_counted_list simple_args) /\
        rank_subst (S rank) sigma.
  Proof.
    clear. 
    intros arity args rank H H0.
    unfold countably_infinite, enumerator in *.
    decompose [ex sig] H; clear H.
    clear H1.
    rename x0 into f, p into H.
    destruct (mod_arg_back_subst_mod_args f 0 id_subst arity args) 
             as [rs next] eqn:?.
    destruct rs as [simple_args sigma].
    exists simple_args, sigma.
    lapply (rank_subst_id_subst V L (S rank)).
      intros H1.
      repeat split.
          eapply mod_arg_back_subst_mod_args_prop_ind; eauto.
        eapply mod_arg_back_subst_mod_args_prop_ind; eauto.
      eapply mod_arg_back_subst_mod_args_prop_ind; eauto.
    apply lt_0_Sn.
  Qed.

  Lemma mod_arg_back_subst_modal_form_prop :
    forall(f : nat -> V)(form back_form : lambda_formula V L)
          (mod_form : modal_form form)
          (r n n_res : nat)(sigma sigma_res : lambda_subst V L) ,
      injective f ->
      rank_formula (2 + r) form ->
      rank_subst (S r) sigma ->
      mod_arg_back_subst_modal_form f n sigma form mod_form = 
                                       (back_form, sigma_res, n_res) ->
        (forall(bound : nat),
           n_res <= bound -> 
             prop_var_list_below f bound (prop_var_formula back_form))
        /\
        n <= n_res /\
        subst_equal_below f n sigma_res sigma /\
        subst_form sigma_res back_form = form /\
        simple_modal_form back_form /\
        rank_subst (S r) sigma_res.
  Proof.
    intros f form back_form mod_form r n n_res sigma sigma_res H H0 H1 H2.
    destruct form; try contradiction.
    rename c into args.
    simpl in *.
    destruct (mod_arg_back_subst_mod_args f n sigma (arity L op) args)
      as [rs n_res1] eqn:?.
    destruct rs as [args_res sigma_res1].
    inversion H2.
    subst back_form sigma_res1 n_res1.
    clear H2.
    assert (H2 := rank_formula_modal_args_TCC _ _ _ H0).
    simpl in H2.
    repeat split.
              rewrite prop_var_formula_char.
              eapply mod_arg_back_subst_mod_args_prop_ind; eauto.
            eapply mod_arg_back_subst_mod_args_prop_ind; eauto.
          eapply mod_arg_back_subst_mod_args_prop_ind; eauto.
        rewrite subst_form_char.
        f_equal.
        eapply mod_arg_back_subst_mod_args_prop_ind; eauto.
      eapply mod_arg_back_subst_mod_args_prop_ind; eauto.
    eapply mod_arg_back_subst_mod_args_prop_ind; eauto.
  Qed.

  Lemma mod_arg_back_subst_neg_form_prop :
    forall(f : nat -> V)(form back_form : lambda_formula V L)
          (neg_mod_form : top_modal_form form)
          (r n n_res : nat)(sigma sigma_res: lambda_subst V L),
      injective f ->
      rank_formula (2 + r) form ->
      rank_subst (S r) sigma ->
      mod_arg_back_subst_neg_mod_form f n sigma form neg_mod_form = 
                  (back_form, sigma_res, n_res) ->
        (forall(bound : nat),
           n_res <= bound -> 
             prop_var_list_below f bound (prop_var_formula back_form))
        /\
        n <= n_res /\
        subst_equal_below f n sigma_res sigma /\
        subst_form sigma_res back_form = form /\
        neg_form_maybe simple_modal_form back_form /\
        rank_subst (S r) sigma_res.
  Proof.
    intros f form back_form neg_mod_form r n n_res sigma sigma_res H H0 H1 H2.
    destruct form; try contradiction.
      simpl in *.
      destruct (mod_arg_back_subst_modal_form f n sigma form neg_mod_form)
        as [rs n_res1] eqn:?.
      destruct rs as [back_f_1 sigma_res1].
      inversion H2.
      subst back_form sigma_res1 n_res1.
      clear H2.
      rewrite rank_formula_lf_neg in H0.
      repeat split.
                rewrite prop_var_formula_char.
                eapply mod_arg_back_subst_modal_form_prop; eauto.
              eapply mod_arg_back_subst_modal_form_prop; eauto.
            eapply mod_arg_back_subst_modal_form_prop; eauto.
          rewrite subst_form_char.
          f_equal.
          eapply mod_arg_back_subst_modal_form_prop; eauto.
        eapply mod_arg_back_subst_modal_form_prop; eauto.
      eapply mod_arg_back_subst_modal_form_prop; eauto.
    unfold mod_arg_back_subst_neg_mod_form in *.
    decompose [and] (mod_arg_back_subst_modal_form_prop 
                            _ _ _ _ _ _ _ _ _ H H0 H1 H2).
    repeat split.
              apply H3.
            trivial.
          trivial.
        trivial.
      clear - H7.
      destruct back_form; try contradiction.
      trivial.
    trivial.
  Qed.

  Lemma mod_arg_back_subst_sequent_rec_prop :
    forall(f : nat -> V)(s back_s : sequent V L)
          (top_mod_s : top_modal_sequent s)(r n n_res : nat)
          (sigma sigma_res : lambda_subst V L),
      injective f ->
      rank_sequent (2 + r) s ->
      rank_subst (S r) sigma ->
      mod_arg_back_subst_sequent_rec f n sigma s top_mod_s
                             = (back_s, sigma_res, n_res) ->
        (forall(bound : nat),
           n_res <= bound ->
             prop_var_list_below f bound (prop_var_sequent back_s)) /\
        subst_sequent sigma_res back_s = s /\
        simple_modal_sequent back_s /\
        rank_subst (S r) sigma_res.
  Proof.
    induction s.
      intros back_s top_mod_s r n n_res sigma sigma_res H H0 H1 H2.
      simpl in *.
      inversion H2.
      subst sigma.
      clear H4 H6 H2.
      repeat split.
          intros bound H2.
          apply prop_var_list_below_empty.
        apply simple_modal_sequent_empty.
      trivial.
    rename a into form.
    intros back_s top_mod_s r n n_res sigma sigma_res H H0 H1 H2.
    simpl in *.
    destruct (mod_arg_back_subst_sequent_rec f n sigma s 
                   (every_nth_tail _ _ _ top_mod_s)) as [ss n1] eqn:?.
    destruct ss as [back_s_tail sigma_tail].
    destruct (mod_arg_back_subst_neg_mod_form f n1 sigma_tail form 
                   (every_nth_head _ _ _ top_mod_s)) as [ss n2] eqn:?.
    destruct ss as [back_f sigma2].
    inversion H2.
    subst sigma2 n2.
    clear H2 H4.
    simpl.
    specialize (IHs _ _ _ _ _ _ _ H (rank_sequent_tail _ _ _ H0) H1 Heqb).
    decompose [and] IHs; clear IHs.
    apply rank_sequent_head in H0.
    repeat split.
          intros bound H5.
          rewrite prop_var_sequent_cons.
          apply prop_var_list_below_append.
            eapply mod_arg_back_subst_neg_form_prop; eauto.
          apply H2.
          eapply le_trans.
            eapply mod_arg_back_subst_neg_form_prop; eauto.
          trivial.
        f_equal.
          eapply mod_arg_back_subst_neg_form_prop; eauto.
        rewrite <- H4.
        eapply subst_sequent_eq.
          apply incl_refl.
        apply (subst_eq_on_vars_below f n1).
          eapply mod_arg_back_subst_neg_form_prop; eauto.
        apply H2.
        trivial.
      apply simple_modal_sequent_cons.
        eapply mod_arg_back_subst_neg_form_prop; eauto.
      trivial.
    eapply mod_arg_back_subst_neg_form_prop; eauto.
  Qed.


  Lemma mod_arg_back_subst_sequent_prop :
    forall(s : sequent V L)(n : nat),
      countably_infinite V ->
      top_modal_sequent s ->
      rank_sequent (2 + n) s ->
      exists(simple_s : sequent V L)(sigma : lambda_subst V L),
        subst_sequent sigma simple_s = s /\
        simple_modal_sequent simple_s /\
        rank_subst (S n) sigma.
  Proof.
    clear. 
    intros s n H H0 H1.
    unfold countably_infinite, enumerator in *.
    decompose [ex sig] H; clear H.
    clear H2.
    rename x0 into f, p into H.
    destruct (mod_arg_back_subst_sequent f s H0) 
             as [simple_s sigma] eqn:?.
    exists simple_s, sigma.
    unfold mod_arg_back_subst_sequent in *.
    destruct (mod_arg_back_subst_sequent_rec f 0 id_subst s H0) 
       as [rs n_res] eqn:?.
    destruct rs as [simple_so sigmao].
    inversion Heqp.
    subst simple_so sigmao.
    clear Heqp.
    lapply (rank_subst_id_subst V L (S n)).
      intros H2.
      repeat split.
          eapply mod_arg_back_subst_sequent_rec_prop; eauto.
        eapply mod_arg_back_subst_sequent_rec_prop; eauto.
      eapply mod_arg_back_subst_sequent_rec_prop; eauto.
    apply lt_0_Sn.
  Qed.


  (***************************************************************************)
  (** *** Backward substitution for modal formulas
         Needed in 5.6.1  *)
  (***************************************************************************)

  (***************************************************************************)
  (** **** Function definition *)
  (***************************************************************************)

  Fixpoint prop_back_subst_form_rec(f : nat -> V)(next : nat)
               (subst : lambda_subst V L)(form : lambda_formula V L)
                                   : back_subst_result (lambda_formula V L) :=
    match form
    with
      | lf_prop v => 
        (lf_prop (f next), 
         function_update v_eq subst (f next) (lf_prop v),
         S next)
      | lf_neg form => 
        let (res_subst, next) := 
            prop_back_subst_form_rec f next subst form in
        let (res, subst) := res_subst
        in  (lf_neg res, subst, next)
      | lf_and form1 form2 =>
        let (res_subst, next) := 
            prop_back_subst_form_rec f next subst form1 in
        let (res_1, subst) := res_subst in
        let (res_subst, next) := 
            prop_back_subst_form_rec f next subst form2 in
        let (res_2, subst) := res_subst
        in (lf_and res_1 res_2, subst, next)
      | lf_modal op args =>
        (lf_prop (f next),
         function_update v_eq subst (f next) (lf_modal op args),
         S next)
    end.

  Definition prop_back_subst_form(f : nat -> V)(form : lambda_formula V L)
                               : (lambda_formula V L) * (lambda_subst V L) :=
    let (res, _) := prop_back_subst_form_rec f 0 id_subst form
    in res.


  (***************************************************************************)
  (** **** Prove substitution property for prop back subst  *)
  (***************************************************************************)

  Lemma prop_back_subst_form_prop_ind :
    forall(f : nat -> V)(form back_form : lambda_formula V L)
          (r n n_res : nat)(sigma sigma_res : lambda_subst V L) ,
      injective f ->
      rank_formula (S r) form ->
      rank_subst (S r) sigma ->
      plain_prop_mod_subst sigma ->
      prop_back_subst_form_rec f n sigma form = (back_form, sigma_res, n_res) ->
        (forall(bound : nat),
           n_res <= bound -> 
             prop_var_list_below f bound (prop_var_formula back_form))
        /\
        n <= n_res /\
        subst_equal_below f n sigma_res sigma /\
        subst_form sigma_res back_form = form /\
        propositional back_form /\
        rank_subst (S r) sigma_res /\
        plain_prop_mod_subst sigma_res.
  Proof.
    induction form.
          (* Case form = lf_prop _ *)
          intros back_form r n n_res sigma sigma_res H H0 H1 H2 H3.
          simpl in *.
          inversion_clear H3.
          repeat split; trivial.
                      rewrite prop_var_formula_char.
                      intros bound H3.
                      apply prop_var_list_below_singleton.
                      trivial.
                    apply le_S.
                    trivial.
                  apply subst_equal_below_single_update; trivial.
                rewrite subst_form_char.
                apply function_update_eq.
                trivial.
              apply propositional_lf_prop.
            apply rank_subst_update; trivial.
          apply plain_prop_mod_subst_update; trivial.
          left.
          simpl.
          trivial.
        (* Case form = lf_neg _ *)
        intros back_form_neg r n n_res sigma sigma_res H H0 H1 H2 H3.
        simpl in H3.
        destruct (prop_back_subst_form_rec f n sigma form)
                 as [rs n_reso] eqn:?.
        destruct rs as [back_form sigma_reso].
        inversion H3.
        subst n_reso sigma_reso.
        clear H5 H3.
        repeat split.
                    rewrite prop_var_formula_char.
                    eapply IHform; eauto.
                  eapply IHform; eauto.
                eapply IHform; eauto.
              rewrite subst_form_char.
              f_equal.
              eapply IHform; eauto.
            apply propositional_neg_inv.
            eapply IHform; eauto.
          eapply IHform; eauto.
        eapply IHform; eauto.
      (* Case form = lf_and _ *)
      intros back_form_and r n n_res sigma sigma_res H H0 H1 H2 H3.
      simpl in H3.
      destruct (prop_back_subst_form_rec f n sigma form1)
               as [rs n_res_1] eqn:?.
      destruct rs as [back_form_1 sigma_res_1].
      destruct (prop_back_subst_form_rec f n_res_1 sigma_res_1 form2)
               as [rs n_res_2] eqn:?.
      destruct rs as [back_form_2 sigma_res_2].
      inversion H3.
      subst sigma_res n_res.
      clear H5 H3.
      assert (H3 := rank_formula_and_left _ _ _ H0).
      assert (H4 := rank_formula_and_right _ _ _ H0).
      assert (rank_subst (S r) sigma_res_1).
        eapply IHform1; eauto.
      assert (plain_prop_mod_subst sigma_res_1).
        eapply IHform1 with (5 := Heqb); eauto.
      repeat split.
                  intros bound H7.
                  rewrite prop_var_formula_char.
                  apply prop_var_list_below_append.
                    eapply IHform1 with (5 := Heqb); eauto.
                    eapply le_trans.
                      eapply IHform2 with (5 := Heqb0); eauto.
                    trivial.
                  eapply IHform2 with (5 := Heqb0); eauto.
                eapply le_trans.
                  eapply IHform1 with (5 := Heqb); eauto.
                eapply IHform2 with (5 := Heqb0); eauto.
              eapply subst_equal_below_trans.
                eapply subst_equal_below_mono.
                  eapply IHform2 with (5 := Heqb0); eauto.
                eapply IHform1 with (5 := Heqb); eauto.
              eapply IHform1 with (5 := Heqb); eauto.
            rewrite subst_form_char.
            f_equal.
              erewrite subst_formula_eq.
                  eapply IHform1 with (5 := Heqb); eauto.
                apply incl_refl.
              eapply subst_eq_on_vars_below.
                eapply IHform2 with (5 := Heqb0); eauto.
              eapply IHform1 with (5 := Heqb); eauto.
            eapply IHform2 with (5 := Heqb0); eauto.
          apply propositional_lf_and.
            eapply IHform1 with (5 := Heqb); eauto.
          eapply IHform2 with (5 := Heqb0); eauto.
        eapply IHform2 with (5 := Heqb0); eauto.
      eapply IHform2 with (5 := Heqb0); eauto.
    (* Case form = lf_modal _ *)
    intros back_form r n n_res sigma sigma_res H0 H1 H2 H3 H4.
    clear H.
    simpl in *.
    inversion_clear H4.
    repeat split; trivial.
                intros bound H.
                rewrite prop_var_formula_char.
                apply prop_var_list_below_singleton.
                trivial.
              apply le_S.
              trivial.
            apply subst_equal_below_single_update; trivial.
          rewrite subst_form_char.
          apply function_update_eq.
          trivial.
        apply propositional_lf_prop.
      apply rank_subst_update; trivial.
    apply plain_prop_mod_subst_update.
      right.
      simpl.
      trivial.
    trivial.
  Qed.

  Lemma prop_back_subst_form_prop :
    forall(form : lambda_formula V L)(n : nat),
      countably_infinite V ->
      rank_formula (S n) form ->
      exists(simple_f : lambda_formula V L)(sigma : lambda_subst V L),
        subst_form sigma simple_f = form /\
        propositional simple_f /\
        rank_subst (S n) sigma /\
        plain_prop_mod_subst sigma.
  Proof.
    intros form n H H0.
    unfold countably_infinite, enumerator in *.
    decompose [ex sig] H; clear H.
    clear H1.
    rename x0 into f, p into H.
    destruct (prop_back_subst_form f form) as [simple_f sigma] eqn:?.
    exists simple_f, sigma.
    unfold prop_back_subst_form in *.
    destruct (prop_back_subst_form_rec f 0 id_subst form) as [rs n_res] eqn:?.
    destruct rs as [simple_so sigmao].
    inversion Heqp.
    subst simple_so sigmao.
    clear Heqp.
    assert (H1 := plain_prop_mod_id_subst V L).
    lapply (rank_subst_id_subst V L (S n)).
      intros H2.
      repeat split.
            eapply prop_back_subst_form_prop_ind; eauto.
          eapply prop_back_subst_form_prop_ind; eauto.
        eapply prop_back_subst_form_prop_ind; eauto.
      eapply prop_back_subst_form_prop_ind; eauto.
    apply lt_0_Sn.
  Qed.

End Back_subst.

Implicit Arguments mod_arg_back_subst_mod_args_prop [V L].
Implicit Arguments mod_arg_back_subst_sequent_prop [V L].
Implicit Arguments prop_back_subst_form_prop [V L].
