(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: build_prop_proof.v,v 1.12 2013/04/10 11:17:14 tews Exp $
 *)

(** ** Propositional proof search

      This module instantiates the generic proof search for the
      propositional fragment. It defines the oracle functions for the
      rule set G and proves the necessary properties.
 *)

Require Export build_proof rule_sets.

Section Build_prop_proof.

  Variable V : Type.
  Variable L : modal_operators.

  (* need a decidable equality on propositional constants for 
   * the proof construction *)
  Variable v_eq : eq_type V.


  (**************************************************************************)
  (** ***  ax rule oracle  *)
  (**************************************************************************)

  Definition rule_oracle_G_property(s : sequent V L)
                                   (r : sequent_rule V L) : Prop :=
    G_set V L r /\ r.(conclusion) = s.


  Definition is_negated_prop(v : V)(l : lambda_formula V L) : bool :=
    match l with
      | lf_neg (lf_prop vo) => if v_eq v vo then true else false
      | _ => false
    end.

  Fixpoint find_trivial(l orig : sequent V L)(count : nat) 
                                                   : option (nat * nat * V) :=
    match l with
      | [] => None
      | (lf_prop v) :: r =>
          match list_search orig 0 (is_negated_prop v) with
            | None => find_trivial r orig (1 + count)
            | Some nvo => Some(count, nvo, v)
          end
      | _ :: r => find_trivial r orig (1 + count)
    end.

  Lemma find_trivial_some_ind :
    forall(l1 l2 : sequent V L)(v : V)(vo nvo : nat),
      find_trivial l2 (l1 ++ l2) (length l1) = Some(vo, nvo, v) ->
        simple_tautology_witness (l1 ++ l2) vo nvo v.
  Proof.
    intros l1 l2.
    revert l1.
    induction l2.
      intros l1 v vo nvo H.
      simpl in *.
      discriminate.
    intros l1 v vo nvo H.
    assert (l1 ++ a :: l2 = (l1 ++ [a]) ++ l2).
      rewrite <- app_assoc.
      trivial.
    assert (length (l1 ++ [a]) = S (length l1)).
      rewrite app_length.
      simpl.
      omega.
    destruct a.
          simpl in *.
          destruct (list_search (l1 ++ lf_prop v0 :: l2) 0 (is_negated_prop v0))
              eqn:?.
            assert(H2:=list_search_some _ _ _ Heqo).
            clear IHl2 Heqo H0 H1.
            destruct H2.
            inversion H.
            subst v0 vo n.
            unfold simple_tautology_witness in *.
            assert (length l1 < length (l1 ++ lf_prop v :: l2)).
              rewrite app_length.
              simpl.
              omega.
            constructor 1 with (a := H0).
            constructor 1 with (a := a).
            split.
              rewrite nth_append_right with (n_greater := ge_refl (length l1)).
              generalize
                (nth_append_right_tcc l1 
                  (lf_prop v :: l2) (length l1) H0 (ge_refl (length l1))).
              rewrite minus_diag.
              simpl.
              trivial.
            destruct (nth (l1 ++ lf_prop v :: l2) nvo a).
                  discriminate.
                destruct l.
                      simpl in *.
                      destruct (v_eq v v0).
                        subst v0.
                        trivial.
                      discriminate.
                    discriminate.
                  discriminate.
                discriminate.
              discriminate.
            discriminate.
          clear Heqo.
          rewrite H0 in *.
          apply IHl2.
          rewrite H1.
          trivial.
        rewrite H0 in *.
        apply IHl2.
        rewrite H1.
        trivial.
      rewrite H0 in *.
      apply IHl2.
      rewrite H1.
      trivial.
    rewrite H0 in *.
    apply IHl2.
    rewrite H1.
    trivial.
  Qed.
  
  Lemma find_trivial_some :
    forall(l : sequent V L)(v : V)(vo nvo : nat),
      find_trivial l l 0 = Some(vo, nvo, v) ->
        simple_tautology l.
  Proof.
    clear.
    intros l v vo nvo H.
    unfold simple_tautology in *.
    exists vo.
    exists nvo.
    exists v.
    apply find_trivial_some_ind with (l1 := []).
    simpl.
    trivial.
  Qed.

  Lemma find_trivial_none_ind :
    forall(s1 s2 : sequent V L)(v : V)(vo nvo : nat),
      vo >= length s1 ->
      find_trivial s2 (s1 ++ s2) (length s1) = None ->
        not (simple_tautology_witness (s1 ++ s2) vo nvo v).
  Proof.
    intros s1 s2.
    revert s1.
    induction s2.
      intros s1 v vo nvo H H0 H1.
      rewrite app_nil_r in H1.
      unfold simple_tautology_witness in *.
      destruct H1.
      omega.
    intros s1 v vo nvo H H0 H1.
    assert (vo = length s1 \/ vo >= S (length s1)).
      omega.
    destruct H2.
      clear IHs2.
      subst vo.
      unfold simple_tautology_witness in *.
      decompose [and dep_and] H1; clear H1.
      revert H2.
      rewrite nth_append_right with (n_greater := ge_refl _).
      generalize 
        (nth_append_right_tcc s1 
          (a :: s2) (length s1) a0 (ge_refl (length s1))).
      rewrite minus_diag.
      simpl.
      intros l H2.
      subst a.
      simpl in *.
      destruct (list_search (s1 ++ lf_prop v :: s2) 0 (is_negated_prop v))
          eqn:?.
        discriminate.
      assert (H5:=list_search_none _ _ _ nvo a1 Heqo).
      rewrite H3 in *.
      simpl in *.
      rewrite eq_equal in H5.
      discriminate.
    assert (s1 ++ a :: s2 = (s1 ++ [a]) ++ s2).
      rewrite <- app_assoc.
      trivial.
    rewrite H3 in *; clear H3.
    assert (length (s1 ++ [a]) = S (length s1)).
      rewrite app_length.
      simpl.
      omega.
    rewrite <- H3 in H2.
    eapply IHs2; clear IHs2.
        eexact H2.
      rewrite H3.
      destruct a.
            simpl in *.
            destruct (list_search ((s1 ++ [lf_prop v0]) ++ s2) 0 
                                                 (is_negated_prop v0)).
              discriminate.
            trivial.
          trivial.
        trivial.
      trivial.
    eexact H1.
  Qed.
  
  Lemma find_trivial_none :
    forall(s : sequent V L),
      find_trivial s s 0 = None ->
        not (simple_tautology s).
  Proof.
    intros s H H0.
    unfold simple_tautology in *.
    decompose [ex] H0; clear H0.
    revert H1.
    apply find_trivial_none_ind with (s1 := []).
      simpl.
      omega.
    trivial.
  Qed.


  Definition build_ax_rule(s : sequent V L) : sequent_rule V L :=
    {| assumptions := [];
       conclusion := s
    |}.

  Lemma well_founded_ax_rule : forall(s : sequent V L),
    well_founded_rule V L (build_ax_rule s) sequent_measure.
  Proof.
    clear. 
    intros s i i_less.
    simpl in i_less.
    exfalso.
    omega.
  Qed.


  Lemma ax_oracle_tcc : 
    forall(s : sequent V L)(v : V)(vo nvo : nat),
      find_trivial s s 0 = Some(vo, nvo, v) ->
        rule_oracle_G_property s (build_ax_rule s).
  Proof.
    intros s v vo nvo H.
    split.
      unfold G_set in *.
      left.
      split.
        trivial.
      simpl.
      eapply find_trivial_some.
      eexact H.
    simpl.
    trivial.
  Qed.


  Definition ax_oracle(s : sequent V L) : 
                            (rule_oracle_result V L (G_set V L) s) :=
    match find_trivial s s 0 as ft 
    return find_trivial s s 0 = ft -> 
                            (rule_oracle_result V L (G_set V L) s) 
    with
      | None => fun _ => None
      | Some(vo, nvo, v) => fun(H : find_trivial s s 0 = Some(vo, nvo,v)) =>
        Some(dep_conj (sequent_rule V L)
                      (fun(r : sequent_rule V L) => rule_oracle_G_property s r)
               (build_ax_rule s) 
               (ax_oracle_tcc s v vo nvo H))
    end (eq_refl (find_trivial s s 0)). 

  Lemma ax_oracle_some :
    forall(s : sequent V L)
          (d : r # sequent_rule V L /#\ (rule_oracle_G_property s r)),
      ax_oracle s = Some d ->
        exists(p : rule_oracle_G_property s (build_ax_rule s)),
            d = dep_conj (sequent_rule V L) (rule_oracle_G_property s)
                  (build_ax_rule s) p.
  Proof.
    intros s d.
    unfold ax_oracle in *.
    generalize (eq_refl (find_trivial s s 0)).
    pattern (find_trivial s s 0) at 2 3.
    destruct (find_trivial s s 0).
      destruct p.
      destruct p.
      intros e H.
      inversion H.
      exists (ax_oracle_tcc s v n n0 e).
      trivial.
    intros e H.
    discriminate.
  Qed.


  (**************************************************************************)
  (** ***  and rule oracle  *)
  (**************************************************************************)

  Definition find_and(l : sequent V L)
                : option((lambda_formula V L) * (lambda_formula V L) * nat) :=
    let ls := list_search l 0 is_and in
    match ls as ls0 
             return ls = ls0 -> 
                     option((lambda_formula V L) * (lambda_formula V L) * nat)
    with
      | None => fun _ => None
      | Some ao => fun(H : ls = Some ao) =>
          let and_f := nth l ao (list_search_some_less l is_and ao H) in
          let (conj_1, conj_2) := get_and_forms and_f 
                (list_search_some_test l is_and ao H)
          in
            Some(conj_1, conj_2, ao)
    end (eq_refl ls).
  
  Lemma find_and_some :
    forall(l : sequent V L)(f1 f2 : lambda_formula V L)(ao : nat),
      find_and l = Some(f1, f2, ao) ->
        ao_less # ao < length l /#\
          nth l ao ao_less = lf_and f1 f2.
  Proof.
    clear.
    intros l f1 f2 ao.
    unfold find_and in *.
    generalize (eq_refl (list_search l 0 is_and)).
    pattern (list_search l 0 is_and) at 2 3.
    destruct (list_search l 0 is_and).
      intros e H.
      destruct(get_and_forms 
                    (nth l n (list_search_some_less l is_and n e))
                    (list_search_some_test l is_and n e))
          eqn:?.
      inversion H; clear H.
      subst n l0 l1.
      constructor 1 with (a:= list_search_some_less l is_and ao e).
      revert Heqp.
      generalize (list_search_some_test l is_and ao e).
      assert(H0:= list_search_some_test l is_and ao e).
      destruct (nth l ao (list_search_some_less l is_and ao e)).
            discriminate.
          discriminate.
        intros e0 Heqp.
        simpl in *.
        inversion Heqp.
        trivial.
      discriminate.
    intros e H.
    discriminate.
  Qed.

  Lemma find_and_none :
    forall(s : sequent V L)(f1 f2 : lambda_formula V L)(ao : nat)
          (ao_less : ao < length s),
      find_and s = None ->
        nth s ao ao_less <> lf_and f1 f2.
  Proof.
    intros s f1 f2 ao ao_less.
    unfold find_and in *.
    generalize (eq_refl (list_search s 0 is_and)).
    pattern (list_search s 0 is_and) at 2 3.
    destruct (list_search s 0 is_and).
      intros e H.
      destruct
           (get_and_forms (nth s n (list_search_some_less s is_and n e))
             (list_search_some_test s is_and n e)).
      discriminate.
    intros e H H0.
    assert (H1:=list_search_none _ _ _ _ ao_less e).
    rewrite H0 in *.
    simpl in *.
    discriminate.
  Qed.


  Definition build_and_rule(s : sequent V L)(n : nat)
                           (f1 f2 : lambda_formula V L) : sequent_rule V L :=
    {| assumptions := [(firstn n s) ++ f1 :: (skipn (1 + n) s);
                       (firstn n s) ++ f2 :: (skipn (1 + n) s)];
       conclusion := s
    |}.


  Lemma well_founded_and_rule : 
    forall(s : sequent V L)(f1 f2 : lambda_formula V L)(ao : nat)
          (ao_less : ao < length s),
      nth s ao ao_less = lf_and f1 f2 ->
        well_founded_rule V L (build_and_rule s ao f1 f2) sequent_measure.
  Proof.
    clear. 
    unfold well_founded_rule in *.
    intros s f1 f2 ao ao_less H i i_less.
    simpl in i_less.
    assert (i = 0 \/ i = 1).
      omega.
    destruct H0.
      subst i.
      unfold build_and_rule in *.
      remember (1 + ao) as ao_p_1.
      simpl.
      subst ao_p_1.
      apply sequent_measure_simple_context_lt with (n_less := ao_less).
      rewrite H.
      rewrite (formula_measure_char (lf_and _ _)).
      omega.
    subst i.
    unfold build_and_rule in *.
    remember (1 + ao) as ao_p_1.
    simpl.
    subst ao_p_1.
    apply sequent_measure_simple_context_lt with (n_less := ao_less).
    rewrite H.
    rewrite (formula_measure_char (lf_and _ _)).
    omega.
  Qed.


  Lemma and_oracle_tcc : 
    forall(s : sequent V L)(f1 f2 : lambda_formula V L)(ao : nat),
      find_and s = Some(f1, f2, ao) ->
        rule_oracle_G_property s (build_and_rule s ao f1 f2).
  Proof.
    clear. 
    intros s f1 f2 ao H.
    split.
      unfold G_set in *.
      right.
      left.
      unfold is_and_rule in *.
      exists (firstn ao s).
      exists (skipn (1 + ao) s).
      exists f1.
      exists f2.
      split.
        trivial.
      simpl conclusion.
      assert (H0 := find_and_some _ _ _ _ H).
      decompose [ex and or dep_and] H0; clear H0.
      rewrite <- b.
      rewrite <- list_split_at_n.
      trivial.
    trivial.
  Qed.

  Definition and_oracle(s : sequent V L) : 
                            rule_oracle_result V L (G_set V L) s :=
    match find_and s as fa 
    return find_and s = fa -> rule_oracle_result V L (G_set V L) s
    with
      | None => fun _ => None
      | Some(f1, f2, ao) => fun(H : find_and s = Some(f1, f2, ao)) =>
        Some(dep_conj (sequent_rule V L)
                      (fun(r : sequent_rule V L) => rule_oracle_G_property s r)
               (build_and_rule s ao f1 f2) 
               (and_oracle_tcc s f1 f2 ao H))
    end (eq_refl (find_and s)).

  Lemma and_oracle_some :
    forall(s : sequent V L)
          (d : r # sequent_rule V L /#\ (rule_oracle_G_property s r)),
      and_oracle s = Some d ->
        exists(f1 f2 : lambda_formula V L)(ao : nat)
              (p : rule_oracle_G_property s (build_and_rule s ao f1 f2)),
          d = dep_conj (sequent_rule V L) (rule_oracle_G_property s)
                  (build_and_rule s ao f1 f2) p
          /\ ao_less # ao < length s /#\ nth s ao ao_less = lf_and f1 f2.
  Proof.
    clear. 
    intros s d.
    unfold and_oracle in *.
    generalize (eq_refl (find_and s)).
    pattern (find_and s) at 2 3.
    destruct (find_and s).
      destruct p.
      destruct p.
      intros e H.
      exists l.
      exists l0.
      exists n.
      inversion H.
      exists (and_oracle_tcc s l l0 n e).
      split.
        trivial.
      apply find_and_some.
      trivial.
    intros e H.
    discriminate.
  Qed.


  (**************************************************************************)
  (** ***  neg-and rule oracle  *)
  (**************************************************************************)

  Definition is_neg_and(f : lambda_formula V L) : bool :=
    match f with
      | lf_neg (lf_and _ _) => true
      | _ => false
    end.
  
  Lemma find_neg_and_tcc_is_neg :
    forall(l : sequent V L)(nao : nat)
          (ls_res : list_search l 0 is_neg_and = Some nao),
      let neg_f := nth l nao (list_search_some_less l is_neg_and nao ls_res) 
      in
        is_neg neg_f = true.
  Proof.
    intros l nao ls_res.
    simpl.
    assert(H:=list_search_some_test l is_neg_and nao ls_res).
    destruct(nth l nao (list_search_some_less l is_neg_and nao ls_res)).
          discriminate.
        destruct l0.
              discriminate.
            discriminate.
          trivial.
        discriminate.
      discriminate.
    discriminate.
  Qed.
  
  Lemma find_neg_and_tcc_is_and :
    forall(l : sequent V L)(nao : nat)
          (ls_res : list_search l 0 is_neg_and = Some nao),
      let neg_f := nth l nao (list_search_some_less l is_neg_and nao ls_res) in
      let and_f := get_neg_form neg_f (find_neg_and_tcc_is_neg l nao ls_res) 
      in
        is_and and_f = true.
  Proof.
    intros l nao ls_res.
    simpl.
    assert(H:=list_search_some_test l is_neg_and nao ls_res).
    generalize(find_neg_and_tcc_is_neg l nao ls_res).
    simpl.
    destruct(nth l nao (list_search_some_less l is_neg_and nao ls_res)).
          discriminate.
        destruct l0.
              discriminate.
            discriminate.
          intros e.
          trivial.
        discriminate.
      discriminate.
    discriminate.
  Qed.
  
  Definition find_neg_and(l : sequent V L)
                : option((lambda_formula V L) * (lambda_formula V L) * nat) :=
    let ls := list_search l 0 is_neg_and in
    match ls as ls0 
             return ls = ls0 -> 
                    option((lambda_formula V L) * (lambda_formula V L) * nat)
    with
      | None => fun _ => None
      | Some nao => fun(H : ls = Some nao) =>
          let neg_f := nth l nao (list_search_some_less l is_neg_and nao H) in
          let and_f := get_neg_form neg_f (find_neg_and_tcc_is_neg l nao H) in
          let (conj_1, conj_2) := get_and_forms and_f
                 (find_neg_and_tcc_is_and l nao H) 
          in
             Some(conj_1, conj_2, nao)
    end (eq_refl ls).
  
  Lemma find_neg_and_some :
    forall(l : sequent V L)(f1 f2 : lambda_formula V L)(nao : nat),
      find_neg_and l = Some(f1, f2, nao) ->
        nao_less # nao < length l /#\
          nth l nao nao_less = lf_neg (lf_and f1 f2).
  Proof.
    intros l f1 f2 nao.
    unfold find_neg_and in *.
    generalize (eq_refl (list_search l 0 is_neg_and)).
    pattern (list_search l 0 is_neg_and) at 2 3.
    destruct (list_search l 0 is_neg_and).
      intros e H.
      destruct
          (get_and_forms
             (get_neg_form (nth l n (list_search_some_less l is_neg_and n e))
                (find_neg_and_tcc_is_neg l n e))
             (find_neg_and_tcc_is_and l n e))
          eqn:?.
      inversion H; clear H.
      subst n l0 l1.
      constructor 1 with (a:= list_search_some_less l is_neg_and nao e).
      revert Heqp.
      generalize (find_neg_and_tcc_is_and l nao e).
      generalize (find_neg_and_tcc_is_neg l nao e).
      simpl.
      assert(H0:= list_search_some_test l is_neg_and nao e).
      destruct (nth l nao (list_search_some_less l is_neg_and nao e)).
            discriminate.
          destruct l0.
                discriminate.
              discriminate.
            intros e0 e1 Heqp.
            simpl in *.
            inversion Heqp.
            trivial.
          discriminate.
        discriminate.
      discriminate.
    intros e H.
    discriminate.
  Qed.

  Lemma find_neg_and_none :
    forall(s : sequent V L)(f1 f2 : lambda_formula V L)(nao : nat)
          (nao_less : nao < length s),
      find_neg_and s = None ->
        nth s nao nao_less <> lf_neg (lf_and f1 f2).
  Proof.
    intros s f1 f2 nao nao_less.
    unfold find_neg_and.
    generalize (eq_refl (list_search s 0 is_neg_and)).
    pattern (list_search s 0 is_neg_and) at 2 3.
    destruct (list_search s 0 is_neg_and).
      intros e H.
      destruct
           (get_and_forms
             (get_neg_form (nth s n (list_search_some_less s is_neg_and n e))
                (find_neg_and_tcc_is_neg s n e))
             (find_neg_and_tcc_is_and s n e)).
      discriminate.
    intros e H H0.
    assert (H1:=list_search_none _ _ _ _ nao_less e).
    rewrite H0 in *.
    simpl in *.
    discriminate.
  Qed.


  Definition build_neg_and_rule(s : sequent V L)(n : nat)
                           (f1 f2 : lambda_formula V L) : sequent_rule V L :=
    {| assumptions := [(firstn n s) ++ 
                       (lf_neg f1) :: (lf_neg f2) :: (skipn (1 + n) s)];
       conclusion := s
    |}.

  Lemma well_founded_neg_and_rule : 
    forall(s : sequent V L)(f1 f2 : lambda_formula V L)(nao : nat)
          (nao_less : nao < length s),
      nth s nao nao_less = lf_neg (lf_and f1 f2) ->
        well_founded_rule V L 
          (build_neg_and_rule s nao f1 f2) sequent_measure.
  Proof.
    clear. 
    unfold well_founded_rule in *.
    intros s f1 f2 nao nao_less H i i_less.
    simpl in i_less.
    assert (i = 0).
      omega.
    subst i.
    unfold build_neg_and_rule in *.
    remember (1 + nao) as ao_p_1.
    simpl.
    subst ao_p_1.
    assert (lf_neg f1 :: lf_neg f2 :: skipn (1 + nao) s = 
              [lf_neg f1; lf_neg f2] ++ skipn (1 + nao) s).
      trivial.
    rewrite H0.
    apply sequent_measure_context_lt with (n_less := nao_less).
    rewrite H.
    unfold sequent_measure in *.
    simpl map.
    rewrite formula_measure_char.
    rewrite (formula_measure_char (lf_neg _)).
    rewrite (formula_measure_char (lf_neg _)).
    rewrite (formula_measure_char (lf_and _ _)).
    simpl.
    omega.
  Qed.

  Lemma neg_and_oracle_tcc :
    forall(s : sequent V L)(f1 f2 : lambda_formula V L)(nao : nat),
      find_neg_and s = Some(f1, f2, nao) ->
        rule_oracle_G_property s (build_neg_and_rule s nao f1 f2).
  Proof.
    clear. 
    intros s f1 f2 nao H.
    split.
      unfold G_set in *.
      right.
      right.
      left.
      unfold is_neg_and_rule in *.
      exists (firstn nao s).
      exists (skipn (1 + nao) s).
      exists f1.
      exists f2.
      split.
        trivial.
      simpl conclusion.
      assert (H0 := find_neg_and_some _ _ _ _ H).
      decompose [ex and or dep_and] H0; clear H0.
      rewrite <- b.
      rewrite <- list_split_at_n.
      trivial.
    trivial.
  Qed.


  Definition neg_and_oracle(s : sequent V L) : 
                            rule_oracle_result V L (G_set V L) s :=
    match find_neg_and s as fna 
    return find_neg_and s = fna -> rule_oracle_result V L (G_set V L) s
    with
      | None => fun _ => None
      | Some(f1, f2, nao) => fun H =>
        Some(dep_conj (sequent_rule V L)
                      (fun(r : sequent_rule V L) => rule_oracle_G_property s r)
               (build_neg_and_rule s nao f1 f2)
               (neg_and_oracle_tcc s f1 f2 nao H))
    end (eq_refl (find_neg_and s)).  

  Lemma neg_and_oracle_some :
    forall(s : sequent V L)
          (d : r # sequent_rule V L /#\ (rule_oracle_G_property s r)),
      neg_and_oracle s = Some d ->
        exists(f1 f2 : lambda_formula V L)(nao : nat)
              (p : rule_oracle_G_property s (build_neg_and_rule s nao f1 f2)),
          d = dep_conj (sequent_rule V L) (rule_oracle_G_property s)
                  (build_neg_and_rule s nao f1 f2) p
          /\ nao_less # nao < length s 
             /#\ nth s nao nao_less = lf_neg (lf_and f1 f2).
  Proof.
    clear. 
    intros s d.
    unfold neg_and_oracle in *.
    generalize (eq_refl (find_neg_and s)).
    pattern (find_neg_and s) at 2 3.
    destruct (find_neg_and s).
      destruct p.
      destruct p.
      intros e H.
      exists l.
      exists l0.
      exists n.
      inversion H.
      exists (neg_and_oracle_tcc s l l0 n e).
      split.
        trivial.
      apply find_neg_and_some.
      trivial.
    intros e H.
    discriminate.
  Qed.


  (**************************************************************************)
  (** ***  neg-neg rule oracle  *)
  (**************************************************************************)

  Definition is_neg_neg(f : lambda_formula V L) : bool :=
    match f with
      | lf_neg (lf_neg _) => true
      | _ => false
    end.
  
  Lemma find_neg_neg_tcc_neg_neg : 
    forall(l : sequent V L)(nno : nat)
          (H : list_search l 0 is_neg_neg = Some nno),
        let neg_neg_f := nth l nno (list_search_some_less l is_neg_neg nno H) 
        in
          is_neg neg_neg_f = true.
  Proof.
    intros l nno H.
    simpl.
    assert(H0:=list_search_some_test l is_neg_neg nno H).
    destruct (nth l nno (list_search_some_less l is_neg_neg nno H)).
          discriminate.
        trivial.
      discriminate.
    discriminate.
  Qed.
  
  Lemma find_neg_neg_tcc_neg : 
    forall(l : sequent V L)(nno : nat)
          (H : list_search l 0 is_neg_neg = Some nno),
        let neg_neg_f := nth l nno (list_search_some_less l is_neg_neg nno H) in
        let neg_f := get_neg_form neg_neg_f (find_neg_neg_tcc_neg_neg l nno H) 
        in
          is_neg neg_f = true.
  Proof.
    intros l nno H.
    simpl.
    assert(H0:=list_search_some_test l is_neg_neg nno H).
    generalize (find_neg_neg_tcc_neg_neg l nno H).
    simpl.
    destruct (nth l nno (list_search_some_less l is_neg_neg nno H)).
          discriminate.
        destruct l0.
              discriminate.
            intros e.
            trivial.
          discriminate.
        discriminate.
      discriminate.
    discriminate.
  Qed.
  
  Definition find_neg_neg(l : sequent V L)
                                       : option((lambda_formula V L) * nat) :=
    let ls := list_search l 0 is_neg_neg in
    match ls as ls0 return ls = ls0 -> option((lambda_formula V L) * nat)
    with
      | None => fun _ => None
      | Some nno => fun(H : ls = Some nno) =>
          let neg_neg_f := 
                    nth l nno (list_search_some_less l is_neg_neg nno H) in
          let neg_f := get_neg_form neg_neg_f 
                            (find_neg_neg_tcc_neg_neg l nno H) in
          let f := get_neg_form neg_f (find_neg_neg_tcc_neg l nno H) 
          in
            Some(f, nno)
    end (eq_refl ls).
  
  Lemma find_neg_neg_some :
    forall(l : sequent V L)(f : lambda_formula V L)(nno : nat),
      find_neg_neg l = Some(f, nno) ->
        nno_less # nno < length l /#\
          nth l nno nno_less = lf_neg (lf_neg f).
  Proof.
    intros l f nno.
    unfold find_neg_neg in *.
    generalize (eq_refl (list_search l 0 is_neg_neg)).
    pattern (list_search l 0 is_neg_neg) at 2 3.
    destruct (list_search l 0 is_neg_neg).
      intros e H.
      inversion H; clear H.
      subst n.
      rewrite H1 in *.
      constructor 1 with (a:= list_search_some_less l is_neg_neg nno e).
      revert H1.
      generalize (find_neg_neg_tcc_neg l nno e).
      generalize (find_neg_neg_tcc_neg_neg l nno e).
      simpl.
      assert(H0:= list_search_some_test l is_neg_neg nno e).
      destruct (nth l nno (list_search_some_less l is_neg_neg nno e)).
            discriminate.
          destruct l0.
                discriminate.
              intros e0 e1 H1.
              simpl in *.
              subst l0.
              trivial.
            discriminate.
          discriminate.
        discriminate.
      discriminate.
    intros e H.
    discriminate.
  Qed.

  Lemma find_neg_neg_none :
    forall(s : sequent V L)(f : lambda_formula V L)(nno : nat)
          (nno_less : nno < length s),
      find_neg_neg s = None ->
        nth s nno nno_less <> lf_neg (lf_neg f).
  Proof.
    intros s f nno nno_less.
    unfold find_neg_neg.
    generalize (eq_refl (list_search s 0 is_neg_neg)).
    pattern (list_search s 0 is_neg_neg) at 2 3.
    destruct (list_search s 0 is_neg_neg).
      intros e H.
      discriminate.
    intros e H H0.
    assert (H1:=list_search_none _ _ _ _ nno_less e).
    rewrite H0 in *.
    simpl in *.
    discriminate.
  Qed.


  Definition build_neg_neg_rule(s : sequent V L)(n : nat)
                               (f : lambda_formula V L) : sequent_rule V L :=
    {| assumptions := [(firstn n s) ++ f :: (skipn (1 + n) s)];
       conclusion := s
    |}.

  Lemma well_founded_neg_neg_rule : 
    forall(s : sequent V L)(f : lambda_formula V L)(nno : nat)
          (nno_less : nno < length s),
      nth s nno nno_less = lf_neg (lf_neg f) ->
        well_founded_rule V L (build_neg_neg_rule s nno f) sequent_measure.
  Proof.
    clear. 
    unfold well_founded_rule in *.
    intros s f nno nno_less H i i_less.
    simpl in i_less.
    assert (i = 0).
      omega.
    subst i.
    unfold build_neg_neg_rule in *.
    remember (1 + nno) as ao_p_1.
    simpl.
    subst ao_p_1.
    apply sequent_measure_simple_context_lt with (n_less := nno_less).
    rewrite H.
    rewrite (formula_measure_char (lf_neg _)).
    rewrite (formula_measure_char (lf_neg _)).
    omega.
  Qed.

  Lemma neg_neg_oracle_tcc :
    forall(s : sequent V L)(f : lambda_formula V L)(nno : nat),
      find_neg_neg s = Some(f, nno) ->
        rule_oracle_G_property s (build_neg_neg_rule s nno f).
  Proof.
    clear. 
    intros s f nno H.
    split.
      unfold G_set in *.
      right.
      right.
      right.
      unfold is_neg_neg_rule in *.
      exists (firstn nno s).
      exists (skipn (1 + nno) s).
      exists f.
      split.
        trivial.
      simpl conclusion.
      assert (H0 := find_neg_neg_some _ _ _ H).
      decompose [ex and or dep_and] H0; clear H0.
      rewrite <- b.
      rewrite <- list_split_at_n.
      trivial.
    trivial.
  Qed.

  Definition neg_neg_oracle(s : sequent V L) : 
                            rule_oracle_result V L (G_set V L) s :=
    match find_neg_neg s as fnn
    return find_neg_neg s = fnn -> rule_oracle_result V L (G_set V L) s
    with
      | None => fun _ => None
      | Some(f, nno) => fun H =>
        Some(dep_conj (sequent_rule V L)
                      (fun(r : sequent_rule V L) => rule_oracle_G_property s r)
               (build_neg_neg_rule s nno f)
               (neg_neg_oracle_tcc s f nno H))
    end (eq_refl (find_neg_neg s)).

  Lemma neg_neg_oracle_some :
    forall(s : sequent V L)
          (d : r # sequent_rule V L /#\ (rule_oracle_G_property s r)),
      neg_neg_oracle s = Some d ->
        exists(f : lambda_formula V L)(nno : nat)
              (p : rule_oracle_G_property s (build_neg_neg_rule s nno f)),
          d = dep_conj (sequent_rule V L) (rule_oracle_G_property s)
                  (build_neg_neg_rule s nno f) p
          /\ nno_less # nno < length s 
             /#\ nth s nno nno_less = lf_neg (lf_neg f).
  Proof.
    clear. 
    intros s d.
    unfold neg_neg_oracle in *.
    generalize (eq_refl (find_neg_neg s)).
    pattern (find_neg_neg s) at 2 3.
    destruct (find_neg_neg s).
      destruct p.
      intros e H.
      exists l.
      exists n.
      inversion H.
      exists (neg_neg_oracle_tcc s l n e).
      split.
        trivial.
      apply find_neg_neg_some.
      trivial.
    intros e H.
    discriminate.
  Qed.


  (**************************************************************************)
  (** ***  build_proof oracles  *)
  (**************************************************************************)

  Definition prop_G_oracle : rule_oracle_type V L (G_set V L) :=
    fun(s : sequent V L) => 
      match ax_oracle s with
        | Some r => Some r
        | None =>
          match and_oracle s with
            | Some r => Some r
            | None =>
              match neg_and_oracle s with
                | Some r => Some r
                | None => neg_neg_oracle s
              end
          end
      end.  

  Lemma well_founded_G_oracle : 
    well_founded_rule_oracle prop_G_oracle sequent_measure.
  Proof.
    unfold well_founded_rule_oracle, prop_G_oracle in *.
    intros s.
    destruct (ax_oracle s) eqn:?.
      assert (H := ax_oracle_some _ _ Heqr).
      decompose [ex] H; clear H.
      subst d.
      apply well_founded_ax_rule.
    clear Heqr.
    destruct (and_oracle s) eqn:?.
      assert (H := and_oracle_some _ _ Heqr).
      decompose [ex and dep_and] H; clear H.
      subst d.
      eapply well_founded_and_rule.
      eexact b.
    clear Heqr.
    destruct (neg_and_oracle s) eqn:?.
      assert (H := neg_and_oracle_some _ _ Heqr).
      decompose [ex and or dep_and] H; clear H.
      subst d.
      eapply well_founded_neg_and_rule.
      eexact b.
    clear Heqr.
    destruct (neg_neg_oracle s) eqn:?.
      assert (H := neg_neg_oracle_some _ _ Heqr).
      decompose [ex and or dep_and] H; clear H.
      subst d.
      eapply well_founded_neg_neg_rule.
      eexact b.
    trivial.
  Qed.

  Lemma prop_G_oracle_None : forall(s : sequent V L),
    prop_G_oracle s = None ->
      find_trivial s s 0 = None /\
      find_and s = None /\
      find_neg_and s = None /\
      find_neg_neg s = None.
  Proof.
    intros s H.
    unfold prop_G_oracle in *.
    destruct (ax_oracle s) eqn:?.
      discriminate.
    split.
      clear H.
      revert Heqr.
      unfold ax_oracle in *.
      generalize (eq_refl (find_trivial s s 0)).
      pattern (find_trivial s s 0) at 2 3.
      destruct (find_trivial s s 0). 
        destruct p.
        destruct p.
        discriminate.
      trivial.
    clear Heqr.
    destruct (and_oracle s) eqn:?.
      discriminate.
    split.
      clear H.
      revert Heqr.
      unfold and_oracle in *.
      generalize (eq_refl (find_and s)).
      pattern (find_and s) at 2 3.
      destruct (find_and s).
        destruct p.
        destruct p.
        discriminate.
      trivial.
    clear Heqr.
    destruct (neg_and_oracle s) eqn:?.
      discriminate.
    split.
      clear H.
      revert Heqr.
      unfold neg_and_oracle in *.
      generalize (eq_refl (find_neg_and s)).
      pattern (find_neg_and s) at 2 3.
      destruct (find_neg_and s).
        destruct p.
        destruct p.
        discriminate.
      trivial.
    clear Heqr.
    revert H.
    unfold neg_neg_oracle in *.
    generalize (eq_refl (find_neg_neg s)).
    pattern (find_neg_neg s) at 2 3.
    destruct (find_neg_neg s).
      destruct p.
      discriminate.
    trivial.
  Qed.

  
  Lemma prop_G_oracle_None_simple : forall(s : sequent V L),
    propositional_sequent s ->
    prop_G_oracle s = None ->
      prop_sequent s.
  Proof.
    intros s H H0.
    assert (H1 := prop_G_oracle_None _ H0).
    decompose [and] H1; clear H1.
    clear H0.
    unfold prop_sequent, every_nth in *.
    intros n n_less.
    unfold propositional_sequent, every_nth in *.
    specialize (H n n_less).
    destruct (nth s n n_less) eqn:?.
          simpl.
          trivial.
        destruct l.
              simpl.
              trivial.
            exfalso.
            eapply find_neg_neg_none; eauto.
          exfalso.
          eapply find_neg_and_none; eauto.
        eapply propositional_tcc_modal.
        apply propositional_neg.
        eexact H.
      exfalso.
      eapply find_and_none; eauto.
    eapply propositional_tcc_modal.
    eexact H.
  Qed.
  
  Lemma prop_G_oracle_None_tautology : forall(s : sequent V L),
    prop_G_oracle s = None ->
      not (simple_tautology s).
  Proof.
    intros s H.
    apply find_trivial_none.
    apply prop_G_oracle_None.
    trivial.
  Qed.


  (**************************************************************************)
  (** ***  G rank inductiveness  *)
  (**************************************************************************)

  Lemma rank_G_n_inductive : forall(n : nat),
    rule_inductive (G_n_set V L n) (rank_sequent n).
  Proof.
    unfold rule_inductive in *.
    intros n r H H0.
    apply const_rank_G_set.
      apply H.
    trivial.
  Qed.

End Build_prop_proof.

Implicit Arguments find_trivial [V L].
Implicit Arguments find_trivial_none [V L].
Implicit Arguments prop_G_oracle [[V] [L]].
