(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: classic.v,v 1.8 2013/04/10 11:17:14 tews Exp $
 *)

(** ** Classical reasoning
   
   There are a few results in the formalization that are only valid in
   classical logic. This module defines the necessary property
   [classical_logic] that I use instead of an axiom, because the
   property is easier to track in the source code. This module also
   derives a few classical standard results that are not valid in Coq.

   There are also some lemmas that prove that certain classical
   tautologis are not provable in Coq. Technically this is done by
   proving that [classical_logic] follows from these classical
   tautologies. I use these lemmas to convince myself that certain
   results really depend on classical reasoning.
*)


Require Export sets.

Definition classical_logic : Prop := forall(P : Prop), ~~P -> P.

  (**************************************************************************)
  (** *** standard results *)
  (**************************************************************************)

Lemma neg_or_and_neg : forall(P Q : Prop),
   ~ (P \/ Q) <-> ((~ P) /\ (~ Q)).
Proof.
  intros P Q.
  split.
    intros H.
    split.
      intros H0.
      apply H.
      left.
      trivial.
    intros H0.
    apply H.
    right.
    trivial.
  intros H H0.
  destruct H.
  destruct H0.
    contradiction.
  contradiction.
Qed.

Lemma de_morgan_neg_and : forall(P Q : Prop),
  classical_logic ->
    ~ (P /\ Q) -> ~ P \/ ~ Q.
Proof.
  intros P Q H H0.
  apply H.
  intros H1.
  apply H0.
  apply neg_or_and_neg in H1.
  destruct H1.
  apply H in H1.
  apply H in H2.
  auto.
Qed.

Lemma double_neg_or : forall(P Q : Prop),
  classical_logic ->
    ((P \/ Q) <-> ~(~P /\ ~Q)).
Proof.
  intros P Q H.
  split.
    intros H0 H1.
    tauto.
  intros H0.
  apply H.
  tauto.
Qed.

Lemma excluded_middle : forall(P : Prop), 
  classical_logic -> P \/ ~ P.
Proof.
  intros P H.
  apply H.
  intros H0.
  apply neg_or_and_neg in H0.
  destruct H0.
  contradiction.
Qed.


  (*************************************************************************)
  (** *** Unprovability in intuitionistic logic *)
  (*************************************************************************)

Lemma classic_axiom_neg_and_or : 
  (forall(P Q : Prop), ~(~P /\ ~Q) -> P \/ Q) -> classical_logic.
Proof.
  intros H P H0.
  lapply (H P (~ P)).
    intros H1.
    destruct H1.
      trivial.
    contradiction.
  intros H1.
  destruct H1.
  contradiction.
Qed.

Lemma classic_axiom_neg_disjunction_other_case :
  (forall(P Q : Prop), ~(~P /\ ~Q) -> ~P -> Q) -> classical_logic.
Proof.
  unfold classical_logic in *.
  intros H P H0.
  apply H with (P := ~P).
    intros H1.
    destruct H1.
    contradiction.
  trivial.
Qed.

Lemma classic_axiom_sound_cut_left : 
  (forall(P Q : Prop), (P /\ ~(~~P /\ ~Q)) -> Q) -> classical_logic.
Proof.
  intros H P H0.
  apply H with (P := ~~P).
  tauto.
Qed.

Lemma classic_axiom_sound_cut_right : 
  (forall(P Q : Prop), (~P /\ ~(~P /\ ~Q)) -> Q) -> classical_logic.
Proof.
  intros H P H0.
  apply H with (P := ~P).
  tauto.
Qed.

Lemma classic_axiom_sound_contraction :
  (forall(P : Prop), ~(~P /\ ~P) -> P) -> classical_logic.
Proof.
  intros H P H0.
  apply H.
  tauto.
Qed.


  (*************************************************************************)
  (** *** Some classical results for sets *)
  (*************************************************************************)

Section Classical_sets.

  Variable A : Type.

  Lemma union_complement : forall(P : set A)(a : A),
    classical_logic ->
      union P (set_inverse P) a.
  Proof.
    intros P a H.
    apply excluded_middle.
    trivial.
  Qed.

  Lemma union_set_inverse : forall(P Q : set A)(a : A),
    classical_logic ->
    set_inverse (intersection P Q) a -> 
      union (set_inverse P) (set_inverse Q) a.
  Proof.
    intros P Q a H.
    unfold union, set_inverse, intersection in *.
    apply de_morgan_neg_and.
    trivial.
  Qed.

  Lemma double_set_inverse_rev : forall(P : set A)(a : A),
    classical_logic ->
    set_inverse (set_inverse P) a ->
      P a.
  Proof.
    intros P a H H0.
    apply H.
    trivial.
  Qed.

  Lemma union_double_neg_intersection : forall(P Q : set A),
    classical_logic ->
    set_equal
      (union P Q)
      (set_inverse (intersection (set_inverse P) (set_inverse Q))).
  Proof.
    intros P Q H a.
    apply double_neg_or.
    trivial.
  Qed.

End Classical_sets.
