(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: cut_properties.v,v 1.18 2013/04/10 12:06:16 tews Exp $
 *)


(** ** GRC Properties 3.14 - 3.15  

      The name of this module could be better. It is not about
      properties of cut about the two lemmas 3.14 and 3.15 on GRC_n.

      The substitution lemma in the paper is wrong, see my paper
      "Formalizing Cut Elimination of Coalgebraic Logics in Coq" for a
      counter example.

      Apart from fixing the substitution lemma, I generalize it to
      arbitrary substitution closed rule sets. From this generalized
      version I can simply derive two substitution lemmas for GRC_n
      and GR_n. The latter permits to prove admissibility of
      non-atomic axioms in GR_n (5.6.1) independently from cut
      elimination in GR_n.
*)

Require Export admissibility rule_sets.

Section Cut_properties.

  Variable V : Type.
  Variable L : modal_operators.


  (***************************************************************************)
  (** *** Towards the substitution Lemma 3.14 *)
  (***************************************************************************)

  (** The substitution in the paper is restricted to GRC_n. Moreover, 
      it uses all propositional axioms, where actually only substitution 
      instances of [p, not p] are needed. I prove first a generic version,
      that lets me derive substitution lemmas for GRC_n and GR_n later.
      The restriction to substitution instances of axioms will simplify 
      the proof of 5.6.1 (non-atomic axioms in GR_n).
     
      (Moreover, the version in the paper is wrong, one really needs 
      non-atomic axioms of rank n + k!)
   *)

  Lemma generic_substitution_lemma :
    forall(rules : set (sequent_rule V L))(hypothesis : set (sequent V L))
          (sigma : lambda_subst V L)(n k : nat)(gamma : sequent V L),
      subst_closed_rule_set rules ->
      rank_sequent_set n hypothesis ->
      rank_subst (S k) sigma ->
      provable (rank_rules n (union is_ax_rule rules)) hypothesis gamma ->
        provable (rank_rules (n + k) (union is_ax_rule rules))
                 (union (subst_Ax_n_set sigma (n + k))
                        (subst_sequent_set sigma hypothesis))
          (subst_sequent sigma gamma).
  Proof.
    intros rules hypothesis sigma n k gamma H H0 H1 H2.
    destruct H2.
    clear H2.
    induction x using proof_sequent_ind.
      apply provable_with_assumption.
      right.
      exists gamma.
      auto.
    apply rank_rules_distribute_union in H2.
    destruct H2.
      apply provable_with_assumption.
      left.
      apply ax_rule_subst; trivial.
    lapply (every_nth_exists 
             (fun(s : sequent V L)
                 (p : proof (rank_rules (n + k) (union is_ax_rule rules))
                            (union (subst_Ax_n_set sigma (n + k))
                                   (subst_sequent_set sigma hypothesis))
                             s)
                => True)
             (assumptions (subst_sequent_rule sigma r))).
      clear H3.
      intros H3.
      destruct H3.
      clear H3.
      assert (rank_rules (n + k) (union is_ax_rule rules)
                (subst_sequent_rule sigma r)).
        clear x.
        apply rank_rules_distribute_union.
        right.
        split.
          apply H.
          apply H2.
        apply rule_has_rank_subst_rule.
          apply H2.
        trivial.
      exists (rule _
                   (union (subst_Ax_n_set sigma (n + k))
                          (subst_sequent_set sigma hypothesis))
                   (subst_sequent_rule sigma r)
                   H3 x).
      trivial.
    simpl.
    rewrite every_nth_map.
    trivial.
  Qed.

  (** **** Substitution lemma 3.14 for GRC_n *)
  Lemma GRC_n_substitution_lemma :
    forall(rules : set (sequent_rule V L))(hypothesis : set (sequent V L))
          (sigma : lambda_subst V L)(n k : nat)(gamma : sequent V L),
      one_step_rule_set rules ->
      rank_sequent_set n hypothesis ->
      rank_subst (S k) sigma ->
      provable (GRC_n_set rules n) hypothesis gamma ->
        provable (GRC_n_set rules (n + k))
                 (union (subst_Ax_n_set sigma (n + k)) 
                        (subst_sequent_set sigma hypothesis))
          (subst_sequent sigma gamma).
  Proof.
    intros rules hypothesis sigma n k gamma H H0 H1 H2.
    eapply proof_mono_rules.
      apply set_equal_implies_subset.
      apply set_equal_symm.
      apply GRC_n_set_struct_union.
    apply generic_substitution_lemma; trivial.
      apply subst_closed_GRC_set_wo_ax.
    eapply proof_mono_rules.
      apply set_equal_implies_subset.
      apply GRC_n_set_struct_union.
    trivial.
  Qed.

  (** **** Substitution lemma for GR_n *)
  Lemma GR_n_substitution_lemma :
    forall(rules : set (sequent_rule V L))(hypothesis : set (sequent V L))
          (sigma : lambda_subst V L)(n k : nat)(gamma : sequent V L),
      one_step_rule_set rules ->
      rank_sequent_set n hypothesis ->
      rank_subst (S k) sigma ->
      provable (GR_n_set rules n) hypothesis gamma ->
        provable (GR_n_set rules (n + k))
                 (union (subst_Ax_n_set sigma (n + k))
                        (subst_sequent_set sigma hypothesis))
          (subst_sequent sigma gamma).
  Proof.
    intros rules hypothesis sigma n k gamma H H0 H1 H2.
    eapply proof_mono_rules.
      apply set_equal_implies_subset.
      apply set_equal_symm.
      apply GR_n_set_struct_union.
    apply generic_substitution_lemma; trivial.
      apply subst_closed_GR_set_wo_ax.
    eapply proof_mono_rules.
      apply set_equal_implies_subset.
      apply GR_n_set_struct_union.
    trivial.
  Qed.

  (** And the variant for empty hypothesis *)
  Lemma GR_n_substitution_lemma_empty_hyp :
    forall(rules : set (sequent_rule V L))
          (sigma : lambda_subst V L)(n k : nat)(gamma : sequent V L),
      one_step_rule_set rules ->
      rank_subst (S k) sigma ->
      provable (GR_n_set rules n) (empty_sequent_set V L) gamma ->
        provable (GR_n_set rules (n + k))
                 (subst_Ax_n_set sigma (n + k))
          (subst_sequent sigma gamma).
  Proof.
    intros rules sigma n k gamma H H0 H1.
    eapply proof_mono_hyp.
      apply set_equal_implies_subset.
      apply set_equal_union_empty_right.
      change (empty_set (sequent V L)) with (empty_sequent_set V L).
      apply subst_sequent_set_empty with (sigma := sigma).
    apply GR_n_substitution_lemma; trivial.
    apply rank_sequent_set_empty.
  Qed.


  (***************************************************************************)
  (** *** Towards stratified GRC provability, 3.15  *)
  (***************************************************************************)

  Lemma proof_minimal_proof_rank : 
    forall(rules : set (sequent_rule V L))(s : sequent V L)(n : nat),
      forall(p : proof rules (empty_sequent_set V L) s),
        minimal_proof_rank p <= n ->
          provable (rank_rules n rules) (empty_sequent_set V L) s.
  Proof.
    induction p.
      intros H.
      contradiction.
    intros H0.
    rewrite minimal_proof_rank_char in H0.
    assert (rank_rules n rules r).
      apply rank_rules_minimal_rule_rank.
        trivial.
      eapply Max.max_lub_l.
      eexact H0.
    lapply (every_nth_exists 
             (fun(s : sequent V L)
                 (_ : proof (rank_rules n rules) (empty_sequent_set V L) s) =>
                True)
             (assumptions r)).
      intros H2.
      destruct H2.
      clear H2.
      exists (rule (rank_rules n rules) (empty_sequent_set V L) r H1 x).
      trivial.
    intros i i_less.
    apply H.
    clear H.
    assert (every_dep_nth (fun s p => minimal_proof_rank p <= n) 
                          (assumptions r) pl).
      apply every_dep_nth_dep_map_const with (P := fun m => m <= n).
      apply nat_list_max_le_inv.
      eapply Max.max_lub_r.
      eexact H0.
    apply H.
  Qed.


  (** **** Proposition 3.15, stratified GRC provability *)
  Lemma rank_proof_GRC : 
    forall(rules : set (sequent_rule V L))(s : sequent V L),
      provable (GRC_set rules) (empty_sequent_set V L) s 
         <->
      exists(n : nat),
        provable (GRC_n_set rules n) (empty_sequent_set V L) s.
  Proof.
    intros rules s.
    split.
      intros H.
      destruct H.
      clear H.
      eexists.
      eapply proof_minimal_proof_rank with (p := x).
      apply le_refl.
    intros H.
    destruct H.
    eapply proof_mono_rules.
      apply subset_rank_rules.
    eexact H.
  Qed.

  (** **** Stratified GR provability

          See also 
          #<A HREF="weakening.html##rank_proof_GR_fixed_rank"><spanclass="inlinecode">rank_proof_GR_fixed_rank</span></A>#
           in module 
       #<A HREF="weakening.html"><spanclass="inlinecode">weakening</span></A>#.
          This lemma does not depend on the set of one-step rules. On
          the other hand, the rank [n] can be greater than that of
          [s].
   *)
  Lemma rank_proof_GR : 
    forall(rules : set (sequent_rule V L))(s : sequent V L),
      provable (GR_set rules) (empty_sequent_set V L) s 
         <->
      exists(n : nat),
        provable (GR_n_set rules n) (empty_sequent_set V L) s.
  Proof.
    intros rules s.
    split.
      intros H.
      destruct H.
      clear H.
      eexists.
      eapply proof_minimal_proof_rank with (p := x).
      apply le_refl.
    intros H.
    destruct H.
    eapply proof_mono_rules.
      apply subset_rank_rules.
    eexact H.
  Qed.

End Cut_properties.
