(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: factor_two_subst.v,v 1.7 2013/04/10 12:06:16 tews Exp $
 *)

(** ** Factorize two substitutions 

      The syntactic cut elimination proof (5.6 (3) case a 2) needs to
      factorize two substitutions into one injetive substitution and
      two renamings. To stay constructive, I do something weaker.
      Given two pairs of sequents and substitutions [s1, sigma1] and
      [s2, sigma2], I construct an injective [sigma_i] and a two
      renamings [sigma_r1] and sigma_r2, such that
      [[
  subst_sequent sigma1 s1 = subst_sequent (subst_compose sigma_i sigma_r1) s1
  subst_sequent sigma2 s2 = subst_sequent (subst_compose sigma_i sigma_r2) s2
      ]]

      The basic idea is to split the substitutions [sigma1] and
      [sigma2] separately with the machinery of module 
      #<A HREF="factor_subst.html"><spanclass="inlinecode">factor_subst</span></A>#
      and combine the two resulting substitutions. There are, however,
      two slight complications.

      The first complication is that the join of the two substitutions
      works only if the two mappings to be joined have disjoint
      domains. Therefore, I rename all clashing variables in s2 before
      splitting sigma2. Of course, this renaming has to be taken into
      account at various places.

      The second complication is that the two injective substitutions
      that I get from the factorization may map different variables to
      the same formula. In order to deliver an injective substitution
      in the end, the join collects these duplications into a separate
      renaming. This renaming must be inserted into the substitution
      chain of [s2] at a suitable place.
 *)

Require Export factor_subst.

Section Factor_two_subst.

  Variable V : Type.
  Variable L : modal_operators.

  (** Need decidable equalitities for the factoring *)
  Variable op_eq : eq_type (operator L).
  Variable v_eq : eq_type V.


  (***************************************************************************)
  (** ***  Functions definitions  *)
  (***************************************************************************)

  (** This function uses the enumeration [f] to find a fresh variable
      that does not occur in [used]. For the initial call [bound] must
      be greater or equal then the length of [used]. We check [bound +
      1] variables from [f], always incrementing the [f]-argument. At
      least one of those cannot occur in [used]. The argument [n] is
      the starting index for [f]. The next index to be used with [f]
      is returned as second result.
   *)
  Fixpoint next_unused_var(f : nat -> V)(used : list V)(n bound : nat) :
                                                                    V * nat :=
    match bound with
      | 0 => (f n, S n)
      | S bound =>
        if member v_eq (f n) used
        then next_unused_var f used (S n) bound
        else (f n, S n)
    end.

  (** This function constructs the mapping for a renaming of the
      variables in the intersection of [pv1] and [pv2]. The renaming
      maps those duplicates in [pv2] to fresh variables that do
      neither occur in [pv1] nor in [pv2_orig]. Initially [pv2] and
      [pv2_orig] are identically. But then [pv2] is reduced in the
      recursion, while [pv2_orig] keeps the original [pv2] value.
   *)
  Fixpoint rename_disjoint(f : nat -> V)(n : nat)(pv1 pv2_orig pv2 : list V) :
                                                               list (V * V) :=
    match pv2 with
      | [] => []
      | v :: pv2 => 
        if member v_eq v pv1
        then 
          let (v', n') := next_unused_var f (pv1 ++ pv2_orig) n 
                                          (length (pv1 ++ pv2_orig)) in
          (v, v') :: (rename_disjoint f n' pv1 pv2_orig pv2)
        else rename_disjoint f n pv1 pv2_orig pv2
    end.

  (** This function joins the two injective mappings [smap1] and
      [smap2] into one injective mapping and a renaming. This function
      works only correct if the domains of [smap1] and [smap2] are
      disjoint. Elements [(v,f)] of [smap2] where [f] is not in
      [smap1] are simply added to [smap1]. Otherwiese, [v] is added to
      the renaming that is returned as second result.
   *)
  Fixpoint join_subst_maps(smap1 smap2 : list (V * lambda_formula V L)) :
                               list (V * lambda_formula V L) * list (V * V) :=
    match smap2 with
      | [] => (smap1, [])
      | (v, f) :: smap2 =>
        let (join_tl, r_tl) := join_subst_maps smap1 smap2 
        in
          match rassoc (lambda_formula_eq op_eq v_eq) join_tl f with
            | Some v' => (join_tl, (v, v') :: r_tl)
            | None => ((v, f) :: join_tl, r_tl)
          end
    end.

  (** This is the main function for factorizing two substitutions. It
      first constructs the mapping [ru_for] of a renaming to make the
      variables of [s1] and [s2] disjoint. ([ru_for] stands for
      rename-unique-forward). This renaming can easily be inverted by
      swapping the pairs in the mapping. Next, both substitutions are
      split using the functions from module 
 #<A HREF="factor_subst.html"><spanclass="inlinecode">factor_subst</span></A>#. 
      Next, the
      two injective substitution mappings obtained this way are
      joined. Finally, the renaming for [s2] must be assembled from
      various pieces and the variable-chain end points in the joined
      mapping must be fixed.
   *)
  Definition factor_two_subst(f : nat -> V)(subst1 subst2 : lambda_subst V L)
                             (s1 s2 : sequent V L) :
               (lambda_subst V L) * (lambda_subst V L) * (lambda_subst V L) :=
    let rumap := 
            rename_disjoint f 0 (prop_var_sequent s1)
                            (list_support v_eq (prop_var_sequent s2))
                            (list_support v_eq (prop_var_sequent s2)) in
    let ru_for := rename_of_map v_eq rumap in
    let s2_ru := subst_sequent ru_for s2 in
    let subst2_ru := 
              subst_compose subst2 (rename_of_map v_eq (swap_pairs rumap)) in
    let (smap1, rmap1) := 
                      divide_subst op_eq v_eq subst1 (prop_var_sequent s1) in
    let (smap2, rmap2) := 
                divide_subst op_eq v_eq subst2_ru (prop_var_sequent s2_ru) in
    let (sjoin, rjoin) := join_subst_maps smap1 smap2
    in (rename_of_map v_eq rmap1,
        subst_compose 
          (subst_compose (rename_of_map v_eq rjoin) (rename_of_map v_eq rmap2))
          ru_for,
        subst_of_map v_eq (fix_var_chain_ends v_eq sjoin)).


  (***************************************************************************)
  (** ***  rename_of_map substitution Properties  *)
  (***************************************************************************)

  Lemma prop_var_formula_subst_rename_of_map :
    forall(rmap : list (V * V))(f : lambda_formula V L),
      prop_var_formula (subst_form (rename_of_map v_eq rmap) f) =
        apply_assoc_map v_eq rmap (prop_var_formula f).
  Proof.
    clear op_eq.
    induction f.
          (* case lf_prop *)
          rewrite subst_form_char.
          rewrite (prop_var_formula_char (lf_prop _)).
          decompose [ex and or] (rename_of_map_prop _ L v_eq rmap v).
            rewrite H1.
            simpl.
            rewrite H.
            apply prop_var_formula_char.
          rewrite H1.
          simpl.
          rewrite H0.
          apply prop_var_formula_char.
        (* case lf_neg *)
        rewrite subst_form_char.
        rewrite prop_var_formula_char.
        rewrite (prop_var_formula_char (lf_neg _)).
        apply IHf.
      (* case lf_and *)
      rewrite subst_form_char.
      rewrite prop_var_formula_char.
      rewrite (prop_var_formula_char (lf_and _ _)).
      rewrite apply_assoc_map_append.
      f_equal.
        apply IHf1.
      apply IHf2.
    (* case lf_modal *)
    rewrite subst_form_char.
    rewrite prop_var_formula_char.
    rewrite (prop_var_formula_char (lf_modal _ _)).
    unfold prop_var_modal_args in *.
    rewrite list_of_counted_list_map.
    rewrite map_map.
    rewrite apply_assoc_map_flatten.
    rewrite map_map.
    f_equal.
    apply restricted_map_ext.
    trivial.
  Qed.

  Lemma prop_var_sequent_subst_rename_of_map :
    forall(rmap : list (V * V))(s : sequent V L),
      prop_var_sequent (subst_sequent (rename_of_map v_eq rmap) s) =
        apply_assoc_map v_eq rmap (prop_var_sequent s).
  Proof.
    clear op_eq.
    induction s.
      trivial.
    unfold prop_var_sequent.
    simpl.
    rewrite apply_assoc_map_append.
    f_equal.
      apply prop_var_formula_subst_rename_of_map.
    apply IHs.
  Qed.


  Lemma subst_eq_on_vars_compose_rename_change :
    forall(subst1 subst2 : lambda_subst V L)
          (rmap : list (V * V))(pv : list V),
      subst_eq_on_vars subst1 subst2 (apply_assoc_map v_eq rmap pv) ->
        subst_eq_on_vars 
          (subst_compose subst1 (rename_of_map v_eq rmap))
          (subst_compose subst2 (rename_of_map v_eq rmap))
          pv.
  Proof.
    unfold subst_eq_on_vars, subst_compose in *.
    intros subst1 subst2 rmap pv H v H0.
    decompose [ex and or] (rename_of_map_prop _ L v_eq rmap v).
      rename x into v'.
      rewrite H3.
      rewrite subst_form_char.
      rewrite subst_form_char.
      apply H.
      eapply In_apply_assoc_map_member.
        eexact H0.
      trivial.
    rewrite H3.
    rewrite subst_form_char in *.
    rewrite subst_form_char in *.
    apply H.
    apply In_apply_assoc_map_non_member.
      trivial.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  next_unused_var Properties  *)
  (***************************************************************************)

  Lemma unused_next_unused_var_ind :
    forall(f : nat -> V)(bound n n' : nat)(used unmet : list V)(v : V),
      injective f ->
      length unmet <= bound ->
      next_unused_var f used n bound = (v, n') ->
      (forall(v : V)(m : nat), In v used -> f m = v -> n <= m -> In v unmet) ->
        ~ In v used.
  Proof.
    induction bound.
      intros n n' used unmet v H H0 H1 H2 H3.
      simpl in *.
      inversion H1; clear H1.
      subst n' v.
      eapply H2 in H3; eauto.
      destruct unmet.
        contradiction.
      simpl in H0.
      omega.
    intros n n' used unmet v H H0 H1 H2.
    simpl in *.
    destruct (member v_eq (f n) used) eqn:H4.
      apply member_In in H4; trivial.
      eapply H2 in H4; eauto.
      apply IHbound with (unmet := remove v_eq (f n) unmet)(3 := H1).
          trivial.
        apply length_remove_In with (a_eq := v_eq) in H4.
        omega.
      clear - H H2.
      intros v m H0 H1 H3.
      apply In_remove_other.
        eapply H2.
            trivial.
          eexact H1.
        omega.
      intros H4.
      subst v.
      apply H in H4.
      omega.
    inversion H1; clear H1.
    subst v n'.
    eapply member_In_false.
    eexact H4.
  Qed.

  Lemma unused_next_unused_var :
    forall(f : nat -> V)(used : list V)(n n' : nat)(v : V),
      injective f ->
      next_unused_var f used n (length used) = (v, n') ->
        ~ In v used.
  Proof.
    intros f used n n' v H H0.
    apply unused_next_unused_var_ind with (unmet := used) in H0; auto.
  Qed.

  Lemma unused_next_unused_var_left :
    forall(f : nat -> V)(pv1 pv2 : list V)(n n' : nat)(v : V),
      injective f ->
      next_unused_var f (pv1 ++ pv2) n (length (pv1 ++ pv2)) = (v, n') ->
        ~ In v pv1.
  Proof.
    intros f pv1 pv2 n n' v H H0 H1.
    apply unused_next_unused_var in H0.
      apply H0.
      apply in_or_app.
      auto.
    trivial.
  Qed.

  Lemma next_unused_var_fresh :
    forall(f : nat -> V)(pv : list V)(bound n n' : nat)(v : V),
      next_unused_var f pv n bound = (v, n') ->
        n < n' /\ v = f (pred n').
  Proof.
    induction bound.
      intros n n' v H.
      simpl in *.
      inversion H; clear H.
      subst v n'.
      auto.
    intros n n' v H.
    simpl in *.
    destruct (member v_eq (f n) pv).
      apply IHbound in H.
      destruct H.
      split.
        omega.
      trivial.
    inversion H; clear H.
    subst v n'.
    auto.
  Qed.


  (***************************************************************************)
  (** ***  rename_disjoint Properties  *)
  (***************************************************************************)

  Lemma rename_disjoint_disjoint_ind :
    forall(f : nat -> V)(pv1 pv2 pv3 : list V)(n : nat)(rmap : list (V * V)),
      injective f ->
      incl pv3 pv2 ->
      (forall(v1 v2 : V), assoc v_eq rmap v1 = Some v2 -> ~ In v2 pv1) ->
        lists_disjoint 
          pv1
          (apply_assoc_map v_eq (rmap ++ rename_disjoint f n pv1 pv2 pv3) pv3)
        /\
        (forall(v1 v2 : V), 
           assoc v_eq (rename_disjoint f n pv1 pv2 pv3) v1 = Some v2 ->
             ~ In v2 pv1 /\ ~ In v2 pv2).
  Proof.
    induction pv3.
      intros n rmap H H0 H1.
      simpl.
      split.
        apply lists_disjoint_right_empty.
      intros v1 v2 H2.
      discriminate.
    rename a into v1.
    intros n rmap H H0 H1.
    unfold rename_disjoint.
    fold rename_disjoint.
    destruct (member v_eq v1 pv1) eqn:H2.
      destruct (next_unused_var f (pv1 ++ pv2) n (length (pv1 ++ pv2)))
               as [v2 n'] eqn:H3.
      lapply (IHpv3 n' (rmap ++ [(v1, v2)]) H); clear IHpv3.
        intros H4; lapply H4; clear H4.
          intros H4.
          destruct H4.
          rewrite app_assoc_reverse in H4.
          simpl in H4.
          split.
            clear H5.
            unfold apply_assoc_map.
            fold (apply_assoc_map (A:=V)).
            apply assoc_append_split.
                trivial.
              intros H5.
              specialize (H1 v1).
              destruct (assoc v_eq rmap v1) as [v3 |].
                apply lists_disjoint_right_cons.
                  trivial.
                apply H1.
                trivial.
              contradiction.
            intros H5; clear H5.
            rewrite assoc_cons_first; trivial.
            apply lists_disjoint_right_cons.
              trivial.
            apply unused_next_unused_var_left in H3; trivial.
          clear - v_eq H H3 H5.
          intros v0 v3.
          apply assoc_cons_split.
            intros H0 H1.
            inversion H1; clear H1.
            subst v3 v0.
            apply unused_next_unused_var in H3; trivial.
            split.
              intros H6.
              apply H3.
              apply in_or_app.
              auto.
            intros H6.
            apply H3.
            apply in_or_app.
            auto.
          intros H0.
          apply H5.
        clear - v_eq H H1 H3.
        intros v3 v4.
        apply assoc_append_split.
          intros H0 H2.
          eapply H1; eauto.
        intros H0 H2.
        simpl in H2.
        destruct (v_eq v3 v1).
          inversion H2; clear H2.
          subst v4.
          apply unused_next_unused_var_left in H3; trivial.
        discriminate.
      apply incl_left_tail in H0.
      trivial.
    lapply (IHpv3 n rmap H); clear IHpv3.
      intros H4; lapply H4; clear H4; trivial.
      intros H4.
      destruct H4.
      split.
        simpl.
        apply assoc_append_split.
            trivial.
          clear H4.
          intros H4.
          specialize (H1 v1).
          destruct (assoc v_eq rmap v1) as [v3 |].
            apply lists_disjoint_right_cons.
              trivial.
            apply H1.
            trivial.
          contradiction.
        intros H5; clear H5.
        specialize (H4 v1).
        destruct (assoc v_eq (rename_disjoint f n pv1 pv2 pv3) v1) as [v3 |].
          apply lists_disjoint_right_cons.
            trivial.
          apply H4.
          trivial.
        apply lists_disjoint_right_cons.
          trivial.
        eapply member_In_false; eauto.
      trivial.
    apply incl_left_tail in H0.
    trivial.
  Qed.

  Lemma rename_disjoint_disjoint :
    forall(f : nat -> V)(pv1 pv2 : list V)(n : nat),
      injective f ->
        lists_disjoint 
          pv1
          (apply_assoc_map v_eq (rename_disjoint f n pv1 pv2 pv2) pv2).
  Proof.
    intros f pv1 pv2 n H.
    apply rename_disjoint_disjoint_ind with (rmap := []).
        trivial.
      apply incl_refl.
    intros v1 v2 H0.
    simpl in *.
    discriminate.
  Qed.

  Lemma rename_disjoint_values_not_in_source :
    forall(f : nat -> V)(pv1 pv2 : list V)(v1 v2 : V),
      injective f ->
      assoc v_eq (rename_disjoint f 0 pv1 pv2 pv2) v1 = Some v2 ->
        ~ In v2 pv2.
  Proof.
    intros f pv1 pv2 v1 v2 H H0.
    eapply rename_disjoint_disjoint_ind with (rmap := []) in H0.
          destruct H0.
          trivial.
        trivial.
      apply incl_refl.
    intros v0 v3 H1.
    discriminate.
  Qed.


  Lemma assoc_mapping_rename_disjoint_ind :
    forall(f : nat -> V)(pv1 pv2 pv3 : list V)(n : nat),
      let rumap := rename_disjoint f n pv1 pv2 pv3
      in
        NoDup pv3 ->
          assoc_mapping v_eq rumap /\
          (forall(v1 v2 : V), assoc v_eq rumap v1 = Some v2 ->
             In v1 pv3).
  Proof.
    induction pv3.
      intros n rumap H.
      simpl in *.
      split.
        apply assoc_mapping_nil.
      intros v1 v2 H0.
      discriminate.
    rename a into v1.
    intros n rumap H.
    assert (H0 := NoDup_tail _ _ H).
    unfold rename_disjoint in rumap.
    fold rename_disjoint in rumap.
    destruct (member v_eq v1 pv1) eqn:H1.
      destruct (next_unused_var f (pv1 ++ pv2) n (length (pv1 ++ pv2)))
               as [v2 n'] eqn:H2.
      split.
        apply assoc_mapping_cons.
          apply IHpv3; trivial.
        destruct (assoc v_eq (rename_disjoint f n' pv1 pv2 pv3) v1) eqn:H3.
          apply IHpv3 in H3; trivial.
          apply NoDup_head in H.
          contradiction.
        simpl.
        trivial.
      intros v0 v3.
      apply assoc_cons_split.
        intros H3 H4.
        subst v0.
        left.
        trivial.
      intros H3 H4.
      apply IHpv3 in H4; trivial.
      right.
      trivial.
    split.
      apply IHpv3; trivial.
    intros v0 v2 H2.
    right.
    apply IHpv3 in H2; trivial.
  Qed.

  Lemma assoc_mapping_rename_disjoint :
    forall(f : nat -> V)(pv1 pv2 : list V),
      NoDup pv2 ->
        assoc_mapping v_eq (rename_disjoint f 0 pv1 pv2 pv2).
  Proof.
    intros f pv1 pv2 H.
    apply assoc_mapping_rename_disjoint_ind.
    trivial.
  Qed.

  Lemma injective_assoc_rename_disjoint_ind :
    forall(f : nat -> V)(pv1 pv2 pv3 : list V)(n : nat),
      let rumap := rename_disjoint f n pv1 pv2 pv3
      in
        injective f ->
          injective_assoc v_eq rumap /\
          (forall(v : V)(m : nat), assoc v_eq rumap v = Some (f m) ->
             n <= m).
  Proof.
    induction pv3.
      intros n rumap H.
      simpl in *.
      split.
        apply injective_assoc_empty.
      discriminate.
    rename a into v1.
    intros n rumap H.
    unfold rename_disjoint in rumap.
    fold rename_disjoint in rumap.
    destruct (member v_eq v1 pv1) eqn:H0.
      destruct (next_unused_var f (pv1 ++ pv2) n (length (pv1 ++ pv2)))
               as [v2 n'] eqn:H1.
      apply next_unused_var_fresh in H1.
      destruct H1.
      split.
        apply injective_assoc_cons_different.
          apply IHpv3; trivial.
        clear rumap.
        intros v3 H3.
        subst v2.
        apply IHpv3 in H3; trivial.
        omega.
      intros v m.
      apply assoc_cons_split.
        intros H3 H4.
        inversion H4; clear H4.
        subst v1 v2.
        apply H in H6.
        omega.
      clear v2 H2 rumap.
      intros H2 H3.
      apply IHpv3 in H3; trivial.
      omega.
    apply IHpv3.
    trivial.
  Qed.

  Lemma injective_assoc_rename_disjoint :
    forall(f : nat -> V)(pv1 pv2 : list V),
      injective f ->
        injective_assoc v_eq (rename_disjoint f 0 pv1 pv2 pv2).
  Proof.
    intros f pv1 pv2 H.
    apply injective_assoc_rename_disjoint_ind.
    trivial.
  Qed.

  Lemma rename_disjoint_compose_identity :
    forall(f : nat -> V)(pv1 pv2 : list V),
      let rumap := rename_disjoint f 0 pv1 pv2 pv2
      in
        injective f ->
        NoDup pv2 ->
          subst_eq_on_vars
            (subst_compose (rename_of_map v_eq (swap_pairs rumap))
                           (rename_of_map v_eq rumap))
            (@id_subst V L)
            pv2.
  Proof.
    intros f pv1 pv2 rumap H H0 v H1.
    unfold subst_compose, id_subst in *.
    decompose [ex and or] (rename_of_map_prop _ L v_eq rumap v).
      rename x into v'.
      rewrite H4.
      rewrite subst_form_char.
      decompose [ex and or] 
                (rename_of_map_prop _ L v_eq (swap_pairs rumap) v').
        rewrite assoc_swap_pairs in H3.
        apply assoc_rassoc_inj_some with (beq := v_eq) in H2.
            rewrite H2 in *.
            inversion H3; clear H3.
            subst x.
            trivial.
          apply assoc_mapping_rename_disjoint; trivial.
        apply injective_assoc_rename_disjoint; trivial.
      rewrite assoc_swap_pairs in H5.
      apply rassoc_assoc_none with (aeq := v_eq)(a := v) in H5.
      contradiction.
    rewrite H4.
    rewrite subst_form_char.
    decompose [ex and or] 
              (rename_of_map_prop _ L v_eq (swap_pairs rumap) v).
      rename x into v'.
      rewrite assoc_swap_pairs in H2.
      apply rassoc_assoc_some with (aeq := v_eq) in H2.
        apply rename_disjoint_values_not_in_source in H2; trivial.
        contradiction.
      apply assoc_mapping_rename_disjoint; trivial.
    trivial.
  Qed.


  Lemma map_subst_correct_disjoint_keys_half :
    forall(subst1 subst2 : lambda_subst V L)
          (smap1 smap2 : list (V * lambda_formula V L))
          (rmap1 rmap2 : list (V * V))(pv1 pv2 : list V)(v : V),
      map_subst_correct v_eq subst1 smap1 rmap1 pv1 ->
      map_subst_correct v_eq subst2 smap2 rmap2 pv2 ->
      lists_disjoint pv1 pv2 ->
      is_some (assoc v_eq smap1 v) -> 
        is_none (assoc v_eq smap2 v).
  Proof.
    intros subst1 subst2 smap1 smap2 rmap1 rmap2 pv1 pv2 v H H0 H1 H2.
    apply neg_is_none.
    intros H3.
    unfold map_subst_correct in *.
    decompose [and] H; clear H.
    decompose [and] H0; clear H0.
    clear H4 H7 H H9.
    lapply (iff_left (H6 v)); clear H6.
      lapply (iff_left (H8 v)); clear H8.
        intros H H0.
        apply H1 in H.
        contradiction.
      auto.
    auto.
  Qed.

  Lemma map_subst_correct_disjoint_keys :
    forall(subst1 subst2 : lambda_subst V L)
          (smap1 smap2 : list (V * lambda_formula V L))
          (rmap1 rmap2 : list (V * V))(pv1 pv2 : list V),
      map_subst_correct v_eq subst1 smap1 rmap1 pv1 ->
      map_subst_correct v_eq subst2 smap2 rmap2 pv2 ->
      lists_disjoint pv1 pv2 ->
        assoc_disjoint_keys v_eq smap1 smap2.
  Proof.
    intros subst1 subst2 smap1 smap2 rmap1 rmap2 pv1 pv2 H H0 H1 v.
    split.
      eapply map_subst_correct_disjoint_keys_half; eauto.
    eapply map_subst_correct_disjoint_keys_half; eauto.
    apply lists_disjoint_symm.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  join_subst_maps Properties  *)
  (***************************************************************************)

  Lemma injective_assoc_join_subst_maps :
    forall(smap1 smap2 sjoin : list (V * lambda_formula V L))
          (rmap1 rmap2 rjoin : list (V * V)),
      join_subst_maps smap1 smap2 = (sjoin, rjoin) ->
      injective_assoc v_eq smap1 ->
      injective_assoc v_eq smap2 ->
      assoc_mapping v_eq smap2 ->
        injective_assoc v_eq sjoin.
  Proof.
    induction smap2.
      intros sjoin rmap1 rmap2 rjoin H H0 H1 H2.
      simpl in *.
      inversion H; clear H.
      subst sjoin.
      trivial.
    intros sjoin rmap1 rmap2 rjoin H H0 H1 H2.
    simpl in *.
    destruct a as [v f].
    destruct (join_subst_maps smap1 smap2) as [join_tl r_tl] eqn:H3.
    apply injective_assoc_tail in H1; trivial.
    apply assoc_mapping_tail in H2.
    destruct (rassoc (lambda_formula_eq op_eq v_eq) join_tl f) 
             as [v'|] eqn:H4.
      inversion H; clear H.
      subst sjoin rjoin.
      eapply IHsmap2; eauto.
    inversion H; clear H.
    subst sjoin rjoin.
    apply injective_assoc_cons_rassoc 
          with (beq := lambda_formula_eq op_eq v_eq); trivial.
    eapply IHsmap2; eauto.
  Qed.

  Lemma assoc_mapping_join_subst_maps :
    forall(smap1 smap2 sjoin : list (V * lambda_formula V L))
          (rjoin : list (V * V)),
      join_subst_maps smap1 smap2 = (sjoin, rjoin) ->
      assoc_mapping v_eq smap1 ->
      assoc_mapping v_eq smap2 ->
      assoc_disjoint_keys v_eq smap1 smap2 ->
        assoc_mapping v_eq sjoin /\
        (forall(v : V), is_some (assoc v_eq sjoin v) ->
           is_some (assoc v_eq smap1 v) \/ is_some (assoc v_eq smap2 v)).
  Proof.
    induction smap2.
      intros sjoin rjoin H H0 H1 H2.
      simpl in *.
      inversion H; clear H.
      subst sjoin.
      auto.
    intros sjoin rjoin H H0 H1 H2.
    simpl in H.
    destruct a as [v f].
    destruct (join_subst_maps smap1 smap2) as [join_tl r_tl] eqn:H3.
    lapply (IHsmap2 _ _ eq_refl H0); clear IHsmap2.
      intros H4; lapply H4; clear H4.
        intros H4.
        destruct H4.
        destruct (rassoc (lambda_formula_eq op_eq v_eq) join_tl f) 
                 as [v'|] eqn:H6.
          inversion H; clear H.
          subst sjoin rjoin.
          split.
            trivial.
          intros v0 H.
          apply assoc_cons_split.
              trivial.
            intros H7.
            simpl.
            auto.
          intros H7.
          apply H5.
          trivial.
        inversion H; clear H.
        subst sjoin rjoin.
        split.
          apply assoc_mapping_cons.
            trivial.
          apply neg_is_none.
          intros H7.
          apply H5 in H7; clear H5.
          destruct H7.
            apply H2 in H.
            rewrite assoc_cons_first in H.
              contradiction.
            trivial.
          apply assoc_mapping_cons_first in H1.
          eapply option_contradiction.
            eexact H.
          trivial.
        intros v0 H.
        apply assoc_cons_split.
            trivial.
          intros H7.
          simpl.
          auto.
        intros H7.
        apply H5.
        rewrite assoc_cons_tail in H; trivial.
      apply assoc_disjoint_keys_right_tail in H2; trivial.
    apply assoc_mapping_tail in H1.
    trivial.
  Qed.

  Lemma join_subst_maps_subst_char :
    forall(smap1 smap2 sjoin : list (V * lambda_formula V L))
          (rjoin : list (V * V)),
      join_subst_maps smap1 smap2 = (sjoin, rjoin) ->
      assoc_mapping v_eq sjoin ->
        exists(smap2_part : list (V * lambda_formula V L)),
          sjoin = smap2_part ++ smap1 /\
          incl smap2_part smap2 /\
          assoc_mapping v_eq smap2_part.
  Proof.
    induction smap2.
      intros sjoin rjoin H H0.
      simpl in *.
      inversion H; clear H.
      subst sjoin rjoin.
      exists [].
      repeat split.
        apply incl_empty.
      apply assoc_mapping_nil.
    intros sjoin rjoin H H0.
    simpl in *.
    destruct a as [v f].
    destruct (join_subst_maps smap1 smap2) as [join_tl r_tl] eqn:H1.
    destruct (rassoc (lambda_formula_eq op_eq v_eq) join_tl f) as [v'|].
      inversion H; clear H.
      subst sjoin rjoin.
      specialize (IHsmap2 _ _ eq_refl H0).
      decompose [ex and] IHsmap2; clear IHsmap2.
      rename x into smap2_tl_part.
      exists smap2_tl_part.
      repeat split; trivial.
      apply incl_tl.
      trivial.
    inversion H; clear H.
    subst sjoin rjoin.
    assert (H2 := assoc_mapping_cons_first _ _ _ _ H0).
    apply assoc_mapping_tail in H0.
    specialize (IHsmap2 _ _ eq_refl H0).
    decompose [ex and] IHsmap2; clear IHsmap2.
    rename x into smap2_tl_part.
    exists ((v, f) :: smap2_tl_part).
    repeat split.
        simpl.
        f_equal.
        trivial.
      apply incl_cons.
        left.
        trivial.
      apply incl_tl.
      trivial.
    apply assoc_mapping_cons.
      trivial.
    subst join_tl.
    revert H2.
    apply assoc_append_split.
      intros H2 H3.
      exfalso.
      eapply option_contradiction.
        eexact H2.
      trivial.
    trivial.
  Qed.

  Lemma rank_join_subst_maps :
    forall(smap1 smap2 sjoin : list (V * lambda_formula V L))
          (n : nat)(rjoin : list (V * V)),
      join_subst_maps smap1 smap2 = (sjoin, rjoin) ->
      assoc_mapping v_eq smap2 ->
      assoc_mapping v_eq sjoin ->
      map_rank v_eq n smap1 ->
      map_rank v_eq n smap2 ->
        map_rank v_eq n sjoin.
  Proof.
    intros smap1 smap2 sjoin n rjoin H H0 H1 H2 H3.
    apply join_subst_maps_subst_char in H; trivial.
    decompose [ex and] H; clear H.
    rename x into smap2_part.
    subst sjoin.
    apply map_rank_append; trivial.
    eapply map_rank_incl; eauto.
  Qed.

  Lemma join_subst_maps_contains_first_vars :
    forall(subst1 : lambda_subst V L)(pv1 : list V)
          (smap1 smap2 sjoin : list (V * lambda_formula V L))
          (rmap1 rjoin : list (V * V))(v1 v2 : V),
      join_subst_maps smap1 smap2 = (sjoin, rjoin) ->
      assoc_mapping v_eq sjoin ->
      map_subst_correct v_eq subst1 smap1 rmap1 pv1 ->
      In v1 pv1 ->
      rename_of_map v_eq rmap1 v1 = lf_prop (L:=L) v2 -> 
        is_some (assoc v_eq sjoin v2).
  Proof.
    intros subst1 pv1 smap1 smap2 sjoin rmap1 rjoin v1 v2 H H0 H1 H2 H3.
    apply join_subst_maps_subst_char in H; trivial.
    decompose [ex and] H; clear H.
    rename x into smap2_part.
    subst sjoin.
    apply assoc_append_split.
      auto.
    clear - H1 H2 H3.
    intros H.
    unfold map_subst_correct in *.
    decompose [and] H1; clear H1.
    clear H0.
    decompose [ex and or] (rename_of_map_prop _ L v_eq rmap1 v1).
      rename x into v3.
      rewrite H3 in *.
      inversion H4; clear H4.
      subst v3.
      apply H6 in H0.
      trivial.
    rewrite H3 in *.
    inversion H4; clear H4.
    subst v2.
    apply (iff_right (H5 _)) in H2.
    rewrite H1 in *.
    destruct H2.
      trivial.
    contradiction.
  Qed.

  Lemma join_subst_maps_equal_first_map :
    forall(subst1 : lambda_subst V L)(pv1 : list V)
          (smap1 smap2 sjoin : list (V * lambda_formula V L))
          (rmap1 rjoin : list (V * V)),
      join_subst_maps smap1 smap2 = (sjoin, rjoin) ->
      assoc_disjoint_keys v_eq smap1 smap2 ->
      assoc_mapping v_eq smap2 ->
      assoc_mapping v_eq sjoin ->
      map_subst_correct v_eq subst1 smap1 rmap1 pv1 ->
        subst_eq_on_vars (subst_of_map v_eq smap1) (subst_of_map v_eq sjoin)
                         (apply_assoc_map v_eq rmap1 pv1).
  Proof.
    intros subst1 pv1 smap1 smap2 sjoin rmap1 rjoin H H0 H1 H2 H3.
    apply join_subst_maps_subst_char in H; trivial.
    decompose [ex and] H; clear H.
    rename x into smap2_part.
    subst sjoin.
    clear H7 H2.
    intros v H5.
    unfold map_subst_correct in *.
    decompose [and] H3; clear H3.
    clear H.
    apply In_apply_assoc_map_destruct in H5.
    decompose [ex and or] H5; clear H5.
      rename x into v'.
      apply H7 in H.
      unfold subst_of_map in *.
      apply assoc_append_split.
        intros H2.
        exfalso.
        apply incl_assoc_some with (2 := H4) in H2.
        eapply option_contradiction.
            eexact H2.
          apply H0.
          trivial.
        trivial.
      trivial.
    apply (iff_right (H6 _)) in H3.
    clear H6 H7.
    rewrite H2 in *.
    destruct H3.
      unfold subst_of_map in *.
      apply assoc_append_split.
        intros H3.
        exfalso.
        apply incl_assoc_some with (2 := H4) in H3.
        eapply option_contradiction.
            eexact H3.
          apply H0.
          trivial.
        trivial.
      trivial.
    contradiction.
  Qed.


  Lemma join_subst_maps_correct_second_map :
    forall(smap1 smap2 sjoin : list (V * lambda_formula V L))
          (rjoin : list (V * V)),
      join_subst_maps smap1 smap2 = (sjoin, rjoin) ->
      assoc_mapping v_eq smap2 ->
      assoc_mapping v_eq sjoin ->
        (forall(v1 v2 : V),
          assoc v_eq rjoin v1 = Some v2 ->
            exists(f : lambda_formula V L),
              assoc v_eq smap2 v1 = Some f /\ assoc v_eq sjoin v2 = Some f
        ) /\
        (forall(v1 : V)(f : lambda_formula V L),
          assoc v_eq rjoin v1 = None ->
          assoc v_eq smap2 v1 = Some f ->
            assoc v_eq sjoin v1 = Some f).
  Proof.
    induction smap2.
      intros sjoin rjoin H H0 H1.
      simpl in *.
      inversion H; clear H.
      subst sjoin rjoin.
      split.
        intros v1 v2 H.
        discriminate.
      intros v1 f H H3.
      discriminate.
    intros sjoin rjoin H H0 H1.
    destruct a as [v1 f1].
    assert (H2 := assoc_mapping_tail _ _ _ _ H0).
    simpl in H.
    destruct (join_subst_maps smap1 smap2) as [join_tl r_tl] eqn:H3.
    destruct (rassoc (lambda_formula_eq op_eq v_eq) join_tl f1) 
             as [v2|] eqn:H4.
      inversion H; clear H.
      subst sjoin rjoin.
      split.
        intros v3 v4.
        apply assoc_cons_split.
          intros H H5.
          inversion H5; clear H5.
          subst v3 v4.
          exists f1.
          split.
            rewrite assoc_cons_first.
            trivial.
          eapply rassoc_assoc_some; eauto.
        intros H.
        rewrite assoc_cons_tail; trivial.
        apply IHsmap2; trivial.
      intros v3 f2.
      apply assoc_cons_split.
        discriminate.
      intros H.
      rewrite assoc_cons_tail; trivial.
      apply IHsmap2; trivial.
    inversion H; clear H.
    subst sjoin rjoin.
    assert (H5 := assoc_mapping_tail _ _ _ _ H1).
    split.
      intros v2 v3 H.
      apply (IHsmap2 _ _ eq_refl) in H; clear IHsmap2; trivial.
      decompose [ex and] H; clear H.
      rename x into f'.
      exists f'.
      apply assoc_cons_split.
        intros H.
        subst v2.
        apply assoc_mapping_cons_first in H0.
        rewrite H7 in *.
        contradiction.
      intros H.
      split.
        trivial.
      apply assoc_cons_split.
        intros H6.
        subst v3.
        apply assoc_mapping_cons_first in H1.
        rewrite H8 in *.
        contradiction.
      trivial.
    intros v2 f2.
    apply assoc_cons_split.
      intros H H6 H7.
      inversion H7; clear H7.
      subst v2 f2.
      rewrite assoc_cons_first.
      trivial.
    intros H.
    rewrite assoc_cons_tail; trivial.
    apply IHsmap2; trivial.
  Qed.


  Lemma join_subst_maps_contains_second_vars :
    forall(subst2 : lambda_subst V L)(pv2 : list V)
          (smap1 smap2 sjoin : list (V * lambda_formula V L))
          (rmap2 rjoin : list (V * V))(v1 v2 : V),
      join_subst_maps smap1 smap2 = (sjoin, rjoin) ->
      map_subst_correct v_eq subst2 smap2 rmap2 pv2 ->
      assoc_mapping v_eq smap2 ->
      assoc_mapping v_eq sjoin ->
      In v1 pv2 ->
      subst_compose (rename_of_map v_eq rjoin) 
                    (rename_of_map v_eq rmap2) v1 = lf_prop (L:=L) v2 -> 
        is_some (assoc v_eq sjoin v2).
  Proof.
    intros subst2 pv2 smap1 smap2 sjoin rmap2 rjoin v1 v2 H H0 H1 H2 H3 H4.
    unfold subst_compose in *.
    apply join_subst_maps_correct_second_map in H; trivial.
    clear H1 H2.
    destruct H.
    decompose [ex and or] (rename_of_map_prop _ L v_eq rmap2 v1).
      rename x into v3.
      rewrite H6 in *; clear H6.
      rewrite subst_form_char in *.
      decompose [ex and or] (rename_of_map_prop _ L v_eq rjoin v3).
        rename x into v4.
        rewrite H7 in *; clear H7.
        inversion H4; clear H4.
        subst v4.
        apply H in H5.
        decompose [ex and] H5; clear H5.
        rewrite H7.
        simpl.
        trivial.
      rewrite H7 in *; clear H7.
      inversion H4; clear H4.
      subst v3.
      destruct (assoc v_eq smap2 v2) as [f |] eqn:H7.
        apply H1 in H7; trivial.
        rewrite H7.
        simpl.
        trivial.
      apply H0 in H2.
      rewrite H7 in *.
      contradiction.
    rewrite H6 in *; clear H6.
    rewrite subst_form_char in *.
    decompose [ex and or] (rename_of_map_prop _ L v_eq rjoin v1).
      rename x into v3.
      rewrite H7 in *; clear H7.
      inversion H4; clear H4.
      subst v3.
      apply H in H2.
      decompose [ex and] H2; clear H2.
      rewrite H7.
      simpl.
      trivial.
    rewrite H7 in *; clear H7.
    inversion H4; clear H4.
    subst v2.
    destruct (assoc v_eq smap2 v1) as [f |] eqn:H7.
      apply H1 in H7; trivial.
      rewrite H7.
      simpl.
      trivial.
    apply H0 in H3.
    rewrite H5 in *.
    rewrite H7 in *.
    destruct H3.
      contradiction.
    contradiction.
  Qed.

  Lemma join_subst_maps_equal_second_map :
    forall(subst2 : lambda_subst V L)(pv2 : list V)
          (smap1 smap2 sjoin : list (V * lambda_formula V L))
          (rmap2 rjoin : list (V * V)),
      join_subst_maps smap1 smap2 = (sjoin, rjoin) ->
      map_subst_correct v_eq subst2 smap2 rmap2 pv2 ->
      assoc_mapping v_eq smap2 ->
      assoc_mapping v_eq sjoin ->
        subst_eq_on_vars 
          (subst_compose (subst_of_map v_eq sjoin) (rename_of_map v_eq rjoin))
          (subst_of_map v_eq smap2)
          (apply_assoc_map v_eq rmap2 pv2).
  Proof.
    intros subst2 pv2 smap1 smap2 sjoin rmap2 rjoin H H0 H1 H2.
    apply join_subst_maps_correct_second_map in H; trivial.
    clear H1 H2.
    destruct H.
    intros v2 H2.
    unfold subst_compose in *.
    apply In_apply_assoc_map_destruct in H2.
    decompose [ex and or] H2; clear H2.
      rename x into v1.
      decompose [ex and or] (rename_of_map_prop _ L v_eq rjoin v2).
        rename x into v3.
        rewrite H6 in *; clear H6.
        rewrite subst_form_char.
        unfold subst_of_map in *.
        apply H in H2.
        decompose [ex and] H2; clear H2.
        rewrite H6.
        rewrite H7.
        trivial.
      rewrite H6; clear H6.
      rewrite subst_form_char.
      unfold subst_of_map in *.
      apply H0 in H3.
      destruct (assoc v_eq smap2 v2) as [f |] eqn:H6.
        apply H1 in H6.
          rewrite H6.
          trivial.
        trivial.
      contradiction.
    decompose [ex and or] (rename_of_map_prop _ L v_eq rjoin v2).
      rename x into v3.
      rewrite H6 in *; clear H6.
      rewrite subst_form_char.
      apply H in H2.
      decompose [ex and] H2; clear H2.
      unfold subst_of_map in *.
      rewrite H6.
      rewrite H7.
      trivial.
    rewrite H6; clear H6.
    rewrite subst_form_char.
    apply H0 in H5.
    rewrite H4 in *.
    destruct H5.
      destruct (assoc v_eq smap2 v2) as [f |] eqn:H6.
        unfold subst_of_map in *.
        rewrite H6.
        apply H1 in H6.
          rewrite H6.
          trivial.
        trivial.
      contradiction.
    contradiction.
  Qed.


  (***************************************************************************)
  (** ***  Final Properties  *)
  (***************************************************************************)

  Lemma factor_two_subst_equal_2 :
    forall(f : nat -> V)(subst2 : lambda_subst V L)
          (smap1 smap2 sjoin : list (V * lambda_formula V L))
          (rmap2 rjoin : list (V * V))
          (pv1 : list V)(s2 : sequent V L),
      let rumap := rename_disjoint f 0 pv1 
                       (list_support v_eq (prop_var_sequent s2))
                       (list_support v_eq (prop_var_sequent s2)) in
      let ru_for := rename_of_map v_eq rumap in
      let s2_ru := subst_sequent ru_for s2 in
      let subst2_ru := subst_compose subst2 
                                     (rename_of_map v_eq (swap_pairs rumap))
      in
        injective f ->
        join_subst_maps smap1 smap2 = (sjoin, rjoin) ->
        map_subst_correct v_eq subst2_ru smap2 rmap2 
                          (prop_var_sequent s2_ru) ->
        assoc_mapping v_eq smap2 ->
        assoc_mapping v_eq sjoin ->
        (* 
         * injective_assoc v_eq smap2
         * map_rank v_eq n smap2
         *)
          subst_eq_on_vars subst2
            (subst_compose (subst_of_map v_eq (fix_var_chain_ends v_eq sjoin))
               (subst_compose
                  (subst_compose (rename_of_map v_eq rjoin)
                     (rename_of_map v_eq rmap2)) ru_for)) 
            (prop_var_sequent s2).
  Proof.
    intros f subst2 smap1 smap2 sjoin rmap2 rjoin pv1 s2 rumap ru_for 
           s2_ru subst2_ru H H0 H1 H2 H3.
    apply subst_eq_on_vars_symm.
    eapply subst_eq_on_vars_trans.
      eapply subst_eq_on_vars_seq.
      apply feq_symmetric.
      apply subst_compose_assoc.
    eapply subst_eq_on_vars_trans.
      apply subst_eq_on_vars_compose_rename_change.
      rewrite <- prop_var_sequent_subst_rename_of_map.
      fold ru_for.
      fold s2_ru.
      apply subst_eq_on_vars_symm with (sigma1 := subst2_ru).
      apply weak_map_subst_correct_fix_var_chain_ends with (1 := op_eq).
          eapply subst_eq_on_vars_trans_rev.
            apply subst_eq_on_vars_seq.
            apply feq_symmetric.
            apply subst_compose_assoc.
          eapply subst_eq_on_vars_trans_rev.
            apply subst_eq_on_vars_compose_rename_change.
            eapply join_subst_maps_equal_second_map; eauto.
          apply H1.
        apply renaming_subst_compose.
          apply renaming_rename_of_map.
        apply renaming_rename_of_map.
      intros v1 v2.
      eapply join_subst_maps_contains_second_vars; eauto.
    eapply subst_eq_on_vars_trans.
      apply subst_eq_on_vars_seq.
      apply subst_compose_assoc.
    eapply subst_eq_on_vars_trans.
      apply subst_eq_on_vars_compose_right_change.
      eapply subst_eq_on_vars_support.
      apply rename_disjoint_compose_identity; trivial.
      apply list_support_correct_no_dup.
    apply subst_eq_on_vars_seq.
    apply subst_compose_id_right.
  Qed.


  Lemma factor_two_subst_property :
    forall(subst1 subst2 : lambda_subst V L)(s1 s2 : sequent V L)(n : nat),
      countably_infinite V ->
      rank_subst (S n) subst1 ->
      rank_subst (S n) subst2 ->
        exists(rsubst1 rsubst2 inj_subst : lambda_subst V L),
          renaming rsubst1 /\
          renaming rsubst2 /\
          injective inj_subst /\
          rank_subst (S n) inj_subst /\
          subst_eq_on_vars subst1 (subst_compose inj_subst rsubst1)
                           (prop_var_sequent s1) /\
          subst_eq_on_vars subst2 (subst_compose inj_subst rsubst2)
                           (prop_var_sequent s2).
  Proof.
    intros subst1 subst2 s1 s2 n H H0 H1.
    unfold countably_infinite, enumerator in *.
    decompose [ex sig] H; clear H.
    clear H2.
    rename x0 into f, p into H.
    destruct (factor_two_subst f subst1 subst2 s1 s2)
             as [r12 inj_subst] eqn:H2.
    destruct r12 as [rsubst1 rsubst2].
    unfold factor_two_subst in *.
    set (rumap := rename_disjoint f 0 (prop_var_sequent s1) 
                      (list_support v_eq (prop_var_sequent s2))
                      (list_support v_eq (prop_var_sequent s2))) in *.
    set (ru_for := rename_of_map v_eq rumap) in *.
    set (s2_ru := subst_sequent ru_for s2) in *.
    set (subst2_ru := subst_compose subst2 
                      (rename_of_map v_eq (swap_pairs rumap))) in *.
    destruct (divide_subst op_eq v_eq subst1 (prop_var_sequent s1))
             as [smap1 rmap1] eqn:H3.
    destruct (divide_subst op_eq v_eq subst2_ru (prop_var_sequent s2_ru))
             as [smap2 rmap2] eqn:H4.
    destruct (join_subst_maps smap1 smap2) as [sjoin rjoin] eqn:H5.
    exists rsubst1, rsubst2, inj_subst.
    inversion_clear H2.
    clear rsubst1 rsubst2.
    assert (H6 := divide_subst_prop _ _ _ n _ _ _ H3 H0).
    decompose [and] H6; clear H6.
    lapply (divide_subst_prop _ _ _ n _ _ _ H4).
      intros H6.
      decompose [and] H6; clear H6.
      assert (assoc_disjoint_keys v_eq smap1 smap2).
        eapply map_subst_correct_disjoint_keys; eauto.
        unfold s2_ru, ru_for.
        rewrite prop_var_sequent_subst_rename_of_map.
        eapply lists_disjoint_subset_right.
          apply apply_assoc_map_support.
        apply rename_disjoint_disjoint.
        trivial.
      assert (assoc_mapping v_eq sjoin).
        eapply assoc_mapping_join_subst_maps; eauto.
      repeat split.
                apply renaming_rename_of_map; trivial.
              apply renaming_subst_compose.
                apply renaming_subst_compose.
                  apply renaming_rename_of_map; trivial.
                apply renaming_rename_of_map; trivial.
              apply renaming_rename_of_map; trivial.
            eapply injective_subst_of_fixed_map; eauto.
            eapply injective_assoc_join_subst_maps; eauto.
          apply rank_subst_subst_of_fixed_map.
            trivial.
          eapply rank_join_subst_maps; eauto.
        eapply weak_map_subst_correct_fix_var_chain_ends with (1 := op_eq).
            eapply subst_eq_on_vars_trans_rev.
              apply subst_eq_on_vars_compose_rename_change.
              apply subst_eq_on_vars_symm.
              eapply join_subst_maps_equal_first_map; eauto.
            apply H8.
          apply renaming_rename_of_map.
        intros v1 v2.
        eapply join_subst_maps_contains_first_vars; eauto.
      eapply factor_two_subst_equal_2; eauto.
    eapply rank_subst_subst_compose.
        eexact H1.
      apply rank_renaming.
      apply renaming_rename_of_map.
      trivial.
    omega.
  Qed.

End Factor_two_subst.

Implicit Arguments factor_two_subst_property [V L].
