(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: functions.v,v 1.7 2013/04/10 11:17:14 tews Exp $
 *)

(** ** Functions

      This module defines the notation for function composition [∘]
      and extentional equality [≡]. It further defines pairing, the
      product of functions and the final map.

      Extentional equality on functions is very simple here: It uses
      Coq equality on the results. This is sufficient. The reasoning
      about extentional equality is always explicit, because I am not
      (yet) familiar with setoid rewriting.
*)


(***************************************************************************)
(** ***  composition  *)
(***************************************************************************)

Definition fcompose{X Y Z : Type}(f : Y -> Z)(g : X -> Y)(x : X) : Z :=
  f(g(x)).

Hint Unfold fcompose.

Notation " f ∘ g " := (fcompose f g)
  (at level 40, left associativity) : feq_scope.


(***************************************************************************)
(** ***  extentional equality  *)
(***************************************************************************)

Definition function_equal {X Y : Type}(f g : X -> Y) := 
  forall x : X, f x = g x.

Notation " g ≡ f " := (function_equal g f)
  (at level 70) : feq_scope.

Open Scope feq_scope.


Lemma feq_reflexive : forall (X Y : Type)(f : X -> Y), f ≡ f.
Proof.
  unfold "≡"; trivial.
Qed.

Lemma feq_transitive : 
  forall (X Y : Type)(f g h : X -> Y), f ≡ g -> g ≡ h -> f ≡ h.
Proof.
  unfold "≡"; intros.
  rewrite H, H0.
  trivial.
Qed.

Lemma feq_symmetric : forall (X Y : Type)(f g : X -> Y), f ≡ g -> g ≡ f.
Proof.
  unfold "≡"; intros.
  auto.
Qed.      

Lemma feq_eq : forall(X Y : Type)(f g : X -> Y), f = g -> f ≡ g.
Proof.
  unfold "≡" in *.
  intros X Y f g H x.
  rewrite H.
  trivial.
Qed.

(* Lemma feq_rw_l : use feq_transitive ! *)

Lemma feq_rw_r : forall (X Y : Type)(f g h : X -> Y), 
  g ≡ h -> f ≡ h -> f ≡ g.
Proof.
  intros X Y f g h H H0.
  eapply feq_transitive.
    eexact H0.
  apply feq_symmetric.
  trivial.
Qed.


(***************************************************************************)
(** ***  identity  *)
(***************************************************************************)

Definition id {X : Type} : X -> X := fun x => x.

Lemma feq_id_left : forall (X Y : Type)(f : X -> Y), (id ∘ f) ≡ f.
Proof.
  unfold "≡", "∘", id; intros.
  trivial.
Qed.

Lemma feq_id_right : forall (X Y : Type)(f : X -> Y), (f ∘ id) ≡ f.
Proof.
  unfold "≡", "∘", id; intros.
  trivial.
Qed.


(***************************************************************************)
(** ***  composition laws  *)
(***************************************************************************)

Lemma feq_compose_associative : 
  forall (U V X Y : Type)(f : X -> Y)(g : V -> X)(h : U -> V),
    (f ∘ (g ∘ h)) ≡ (f ∘ g ∘ h).
Proof.
  unfold "≡"; intros.
  auto.
Qed.

Lemma feq_compose_associative_reverse : 
  forall (U V X Y : Type)(f : X -> Y)(g : V -> X)(h : U -> V),
    (f ∘ g ∘ h) ≡ (f ∘ (g ∘ h)).
Proof.
  intros.
  apply feq_symmetric.
  apply feq_compose_associative.
Qed.


Lemma feq_compose_both :
  forall (X Y Z : Type)(f1 f2: Y -> Z)(g1 g2 : X -> Y),
    f1 ≡ f2 -> g1 ≡ g2 -> (f1 ∘ g1) ≡ (f2 ∘ g2).
Proof.
  intros X Y Z f1 f2 g1 g2 H H0 x.
  unfold "∘", "≡" in *.
  rewrite H0.
  rewrite H.
  trivial.
Qed.

Lemma feq_compose_left_both :
  forall (X Y Z : Type)(f : Y -> Z)(g1 g2 : X -> Y),
    g1 ≡ g2 -> (f ∘ g1) ≡ (f ∘ g2).
Proof.
  intros X Y Z f g1 g2 H.
  apply feq_compose_both.
    apply feq_reflexive.
  trivial.
Qed.

Lemma feq_compose_right_both : 
  forall (X Y Z : Type)(f1 f2 : Y -> Z)(g : X -> Y),
    f1 ≡ f2 -> (f1 ∘ g) ≡ (f2 ∘ g).
Proof.
  intros X Y Z f1 f2 g H.
  apply feq_compose_both.
    trivial.
  apply feq_reflexive.
Qed.

Lemma feq_left_compose_left : 
  forall(X Y Z : Type)(f1 f2 : Y -> Z)(g : X -> Y)(h : X -> Z),
    f1 ≡ f2 -> f2 ∘ g ≡ h -> f1 ∘ g ≡ h.
Proof.
  intros X Y Z f1 f2 g h H H0.
  eapply feq_transitive.
    eapply feq_compose_right_both.
    eexact H.
  trivial.
Qed.

Lemma feq_left_compose_right : 
  forall(X Y Z : Type)(f1 f2 : Y -> Z)(g : X -> Z)(h : X -> Y),
    f1 ≡ f2 -> g ≡ f2 ∘ h -> g ≡ f1 ∘ h.
Proof.
  intros X Y Z f1 f2 g h H H0.
  apply feq_symmetric.
  eapply feq_left_compose_left.
    eexact H.
  apply feq_symmetric.
  trivial.
Qed.

Lemma feq_right_compose_left : 
  forall(X Y Z : Type)(f : Y -> Z)(g1 g2 : X -> Y)(h : X -> Z),
    g1 ≡ g2  ->  f ∘ g2 ≡ h  ->  f ∘ g1 ≡ h.
Proof.
  intros X Y Z f g1 g2 h H H0.
  eapply feq_transitive.
    eapply feq_compose_left_both.
    eexact H.
  trivial.
Qed.

Lemma feq_right_compose_right : 
  forall(X Y Z : Type)(f1 f2 : X -> Y)(g : X -> Z)(h : Y -> Z),
    f1 ≡ f2  ->  g ≡ h ∘ f2  ->  g ≡ h ∘ f1.
Proof.
  intros X Y Z f1 f2 g h H H0.
  apply feq_symmetric.
  eapply feq_right_compose_left.
    eexact H.
  apply feq_symmetric.
  trivial.
Qed.


(***************************************************************************)
(** ***  final object  *)
(***************************************************************************)


(* 
 * unit is defined in Datatypes.v as 
 * 
 * Inductive unit : Set := tt : unit.
 *)

Definition final_map(X : Type) : X -> unit := fun _ => tt.

Lemma final_map_prop : forall(X : Type)(f : X -> unit),
  f ≡ final_map X.
Proof.
  intros X f x.
  destruct (f x).
  trivial.
Qed.


(***************************************************************************)
(** ***  product, pairing  *)
(***************************************************************************)

Definition pair{U X Y : Type}(f : U -> X)(g : U -> Y) : U -> X * Y :=
  fun(u : U) => (f u, g u).

Lemma pair_proj_left :
  forall(U X Y : Type)(f : U -> X)(g : U -> Y),
    (@fst X Y) ∘ (pair f g) = f.
Proof.
  intros U X Y f g.
  trivial.
Qed.

Lemma pair_proj_right :
  forall(U X Y : Type)(f : U -> X)(g : U -> Y),
    (@snd X Y) ∘ (pair f g) = g.
Proof.
  intros U X Y f g.
  trivial.
Qed.

Lemma pair_compose_right : 
  forall(U V X Y : Type)(h : U -> V)(f : V -> X)(g : V -> Y),
    (pair f g) ∘ h = pair (f ∘ h) (g ∘ h).
Proof.
  intros U V X Y h f g.
  trivial.
Qed.

Lemma feq_pair_proj : forall(U X Y : Type)(f : U -> X * Y),
  f ≡ pair ((fst (B := Y)) ∘ f) ((snd (B := Y)) ∘ f).
Proof.
  intros U X Y f u.
  unfold "∘", pair, fst, snd in *.
  apply injective_projections.
    trivial.
  trivial.
Qed.

Lemma feq_pair_both :
  forall(X Y Z : Type)(f1 f2 : X -> Y)(g1 g2 : X -> Z),
    f1 ≡ f2 -> g1 ≡ g2 -> pair f1 g1 ≡ pair f2 g2.
Proof.
  intros X Y Z f1 f2 g1 g2 H H0 x.
  unfold pair.
  rewrite H.
  rewrite H0.
  trivial.
Qed.

Lemma feq_pair_left :
  forall(X Y Z : Type)(f1 f2 : X -> Y)(g : X -> Z),
    f1 ≡ f2 -> pair f1 g ≡ pair f2 g.
Proof.
  intros X Y Z f1 f2 g H.
  apply feq_pair_both.
    trivial.
  apply feq_reflexive.
Qed.

Lemma feq_pair_right :
  forall(X Y Z : Type)(f : X -> Y)(g1 g2 : X -> Z),
    g1 ≡ g2 -> pair f g1 ≡ pair f g2.
Proof.
  intros X Y Z f g1 g2 H.
  apply feq_pair_both.
    apply feq_reflexive.
  trivial.
Qed.


(***************************************************************************)
(** ***  product map  *)
(***************************************************************************)

Definition ftimes{U V X Y : Type}(f : U -> X)(g : V -> Y) : U * V -> X * Y :=
  fun(uv : U * V) => (f (fst uv), g (snd uv)).

Lemma ftimes_def :
  forall(U V X Y : Type)(f : U -> X)(g : V -> Y),
    pair (f ∘ (fst (B := V))) (g ∘ (snd (B := V))) = ftimes f g.
Proof.
  intros U V X Y f g.
  trivial.
Qed.

Lemma feq_ftimes_id : forall(X Y : Type), 
  ftimes (@id X) (@id Y) ≡ @id (X * Y).
Proof.
  intros X Y xy.
  unfold ftimes, id in *.
  apply injective_projections.
    trivial.
  trivial.
Qed.

Lemma feq_ftimes_compose :
  forall(XL XR YL YR ZL ZR : Type)
        (fl : XL -> YL)(fr : XR -> YR)(gl : YL -> ZL)(gr : YR -> ZR),
    (ftimes gl gr) ∘ (ftimes fl fr)  ≡  ftimes (gl ∘ fl) (gr ∘ fr).
Proof.
  intros XL XR YL YR ZL ZR fl fr gl gr xlr.
  trivial.
Qed.

Lemma feq_ftimes_compose_pair :
  forall(X YL YR ZL ZR : Type)
        (fl : X -> YL)(fr : X -> YR)(gl : YL -> ZL)(gr : YR -> ZR),
    (ftimes gl gr) ∘ (pair fl fr)  ≡  pair (gl ∘ fl) (gr ∘ fr).
Proof.
  intros X YL YR ZL ZR fl fr gl gr x.
  trivial.
Qed.

Lemma feq_ftimes_both :
  forall(U V X Y : Type)(f1 f2 : U -> X)(g1 g2 : V -> Y),
    f1 ≡ f2 -> g1 ≡ g2 -> ftimes f1 g1 ≡ ftimes f2 g2.
Proof.
  intros U V X Y f1 f2 g1 g2 H H0 x.
  unfold ftimes, "≡" in *.
  rewrite H.
  rewrite H0.
  trivial.
Qed.

Lemma feq_ftimes_left :
  forall(U V X Y : Type)(f1 f2 : U -> X)(g : V -> Y),
    f1 ≡ f2 -> ftimes f1 g ≡ ftimes f2 g.
Proof.
  intros U V X Y f1 f2 g H.
  apply feq_ftimes_both.
    trivial.
  apply feq_reflexive.
Qed.

Lemma feq_ftimes_right :
  forall(U V X Y : Type)(f : U -> X)(g1 g2 : V -> Y),
    g1 ≡ g2 -> ftimes f g1 ≡ ftimes f g2.
Proof.
  intros U V X Y f g1 g2 H.
  apply feq_ftimes_both.
    apply feq_reflexive.
  trivial.
Qed.

