(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: functor.v,v 1.8 2013/04/10 11:17:15 tews Exp $
 *)


(** * Category theory *)

(** ** Functors

      This module defines the functors to be used in the semantics of
      coalgebraic logics.

      In addition to the usual laws of preservation of identity
      morphisms and preservation of composition, the missing function
      extentionality in Coq requires a third law, namely preservation
      of extentional equality.
*)

Require Export functions misc.

Record functor : Type := {
  obj : Type -> Type;
  fmap : forall{X Y : Type}, (X -> Y) -> ((obj X) -> (obj Y));
  id_law : forall(X : Type), fmap (@id X) ≡ id;
  comp_law : forall(X Y Z : Type)(f : X -> Y)(g : Y -> Z),
               fmap (g ∘ f) ≡ (fmap g) ∘ (fmap f);
  fmap_feq_law : forall(X Y : Type)(f1 f2 : X -> Y),
                   f1 ≡ f2 -> fmap f1 ≡ fmap f2
}.


(** With respect to non-trivial functors, the paper says
    [[
    TX <> \emptyset for some set X; 
         it follows that TY = \emptyset only if Y = \emptyset
    ]]
  Firstly, I don't know how the second point follows from the first. 
  Moreover, a definition like
  [[
    Definition non_trivial_functor(T : functor) : Prop :=
      forall(Y : Type), empty_type (obj T Y) -> empty_type Y.
  ]]
  is not useful, because from it I cannot derive the existence of an
  element in [obj T Y] from an element in [Y]. I therefore use the
  contrapositive of this definition.
*)

Definition non_trivial_functor(T : functor) : Prop :=
  forall(Y : Type), non_empty_type Y -> non_empty_type (obj T Y).

Lemma non_empty_coalg : forall(T : functor)(X : Type),
  non_trivial_functor T ->
  non_empty_type X ->
    exists(f : X -> obj T X), True.
Proof.
  intros T X H H0.
  unfold non_trivial_functor in *.
  apply H in H0.
  destruct H0.
  clear H0.
  exists (fun _ => x).
  trivial.
Qed.


(** This is a simple example to see if the defintions can be used at
    all 
*)
Definition x_nat_functor : functor.
Proof.
  refine {| obj := fun(X : Type) => prod X nat;
            fmap := fun{X Y : Type}(f : X -> Y)(xn : X * nat) =>
                     let (x,n) := xn 
                     in  (f x, n);
            id_law := _ ;
            comp_law := _;
            fmap_feq_law := _
         |}.
      intros X x.
      destruct x.
      trivial.
    intros X Y Z f g x.
    destruct x.
    trivial.
  intros X Y f1 f2 H x.
  destruct x.
  rewrite H.
  trivial.
Qed.

