(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: inversion.v,v 1.31 2013/04/10 11:17:15 tews Exp $
 *)


(** ** Inversion, 3.12 - 3.13

      This module proves the inversion lemmas 3.12 and 3.13. The first
      one, 3.12, states depth preserving inversion in G_n + H. This is
      the result which is needed inside the syntactic cut-elimination
      proof. It is therefore formalized completely here. The other,
      3.13, states depth preserving inversion for GR_n. The paper does
      not proof the depth-preserving part and one would need a
      completely different proof for that. Because 3.13 is never used,
      I only formalize inversion without depth preservation.
 *)


Require Export weakening.

Section Inversion.

  Variable V : Type.
  Variable L : modal_operators.

  (** Need decidable equality on propositional constants 
      for the unused Lemma inversion_admissible_GR_n (3.13)
   *)
  Variable v_eq : eq_type V.


  (***************************************************************************)
  (** *** Towards Lemma 3.12, Inversion in G_n + H
   *)


  (***************************************************************************)
  (** **** Closure under inversion for hypothesis'
   *)

  Definition conj_head_inversion_closed(ss : set (sequent V L)) : Prop :=
    forall(sr : sequent V L)(f1 f2 : lambda_formula V L),
      ss ((lf_and f1 f2) :: sr) ->
        ss (f1 :: sr) /\ ss (f2 :: sr).

  Definition disj_head_inversion_closed(ss : set (sequent V L)) : Prop :=
    forall(sr : sequent V L)(f1 f2 : lambda_formula V L),
      ss ((lf_neg (lf_and f1 f2)) :: sr) ->
        ss ((lf_neg f1) :: (lf_neg f2) :: sr).

  Definition neg_head_inversion_closed(ss : set (sequent V L)) : Prop :=
    forall(sr : sequent V L)(f : lambda_formula V L),
      ss ((lf_neg (lf_neg f)) :: sr) ->
        ss (f :: sr).

  Definition head_inversion_closed(ss : set (sequent V L)) : Prop :=
    conj_head_inversion_closed ss /\
    disj_head_inversion_closed ss /\
    neg_head_inversion_closed ss.

  Lemma head_inversion_closed_empty :
    head_inversion_closed (empty_sequent_set V L).
  Proof.
    clear. 
    split.
      intros sr f1 f2 H.
      contradiction.
    split.
      intros sr f1 f2 H.
      contradiction.
    intros sr f H.
    contradiction.
  Qed.


  (***************************************************************************)
  (** **** Prove first that inversion is admissible at head position
   *)

  Lemma sequent_other_axiom_G_n_set :
    forall(n : nat)(f1 : lambda_formula V L)(s1 s2 : sequent V L),
      simple_tautology (f1 :: s1) ->
      rank_sequent n (f1 :: s1) ->
      (rank_formula n f1 -> rank_sequent n s2) ->
      (not (neg_form_maybe prop_form f1)) ->
        G_n_set V L n {| assumptions := []; conclusion := s2 ++ s1 |}.
  Proof.
    clear. 
    intros n f1 s1 s2 H H0 H1 H2.
    repeat split.
        left.
        split.
          trivial.
        simpl.
        apply simple_tautology_append_left.
        eapply simple_tautology_tail.
          eexact H.
        trivial.
      simpl.
      apply every_nth_empty.
    simpl.
    eapply rank_sequent_different_head; eauto.
  Qed.

  Lemma other_axiom_G_n_set :
    forall(n : nat)(f1 f2 : lambda_formula V L)(s : sequent V L),
      simple_tautology (f1 :: s) ->
      rank_sequent n (f1 :: s) ->
      (rank_formula n f1 -> rank_formula n f2) ->
      (not (neg_form_maybe prop_form f1)) ->
        G_n_set V L n {| assumptions := []; conclusion := f2 :: s |}.
  Proof.
    clear. 
    intros n f1 f2 s H H0 H1 H2.
    eapply (sequent_other_axiom_G_n_set _ _ _ [f2]); eauto.
    intros H3.
    apply rank_sequent_cons.
      auto.
    apply rank_sequent_empty.
  Qed.

  Lemma list_subproofs_head_admissible :
    forall(hyp : set (sequent V L))(n d : nat)
          (f : lambda_formula V L)(sl sr new_ass : sequent V L)
          (ass : list (sequent V L))
          (pl : dep_list (sequent V L) (proof (G_n_set V L n) hyp)
                  (map (add_context (f :: sl) sr) ass)),
      every_dep_nth
        (fun(s : sequent V L)(p : proof (G_n_set V L n) hyp s) => 
            proof_depth p <= d)
        (map (add_context (f :: sl) sr) ass) pl ->
      every_dep_nth
        (fun(a : sequent V L)(p_a : proof (G_n_set V L n) hyp a) =>
           forall(d : nat)(a_tl : sequent V L),
             a = f :: a_tl ->
             proof_depth p_a <= d ->
             exists(op : proof (G_n_set V L n) hyp (new_ass ++ a_tl)),
               proof_depth op <= d)
        (map (add_context (f :: sl) sr) ass)
        pl
      ->
        exists(opl : dep_list (sequent V L) (proof (G_n_set V L n) hyp)
                              (map (add_context (new_ass ++ sl) sr) ass)),
          every_dep_nth
            (fun(s : sequent V L)(p : proof (G_n_set V L n) hyp s) =>
                proof_depth p <= d)
            (map (add_context (new_ass ++ sl) sr) ass) opl.
  Proof.
    clear. 
    intros hyp n d f sl sr new_ass ass pl H H0.
    apply every_nth_exists.
    intros i i_less.
    assert (i < length (map (add_context (f :: sl) sr) ass)).
      rewrite map_length in *.
      trivial.
    assert (i < length ass).
      rewrite map_length in *.
      trivial.
    specialize (H0 i H1 d (sl ++ (nth ass i H2) ++ sr)).
    destruct H0.
        clear. 
        rewrite nth_map.
        unfold add_context in *.
        simpl.
        erewrite nth_tcc_irr.
        trivial.
      apply H.
    rewrite nth_map.
    unfold add_context.
    rewrite <- app_assoc.
    erewrite nth_tcc_irr.
    exists x.
    trivial.
  Qed.


  Lemma subproofs_head_admissible :
    forall(hyp : set (sequent V L))(n d : nat)
          (f1 f2 : lambda_formula V L)(sl sr : sequent V L)
          (ass : list (sequent V L))
          (pl : dep_list (sequent V L) (proof (G_n_set V L n) hyp)
                  (map (add_context (f1 :: sl) sr) ass)),
      every_dep_nth
        (fun(s : sequent V L)(p : proof (G_n_set V L n) hyp s) => 
            proof_depth p <= d)
        (map (add_context (f1 :: sl) sr) ass) pl ->
      every_dep_nth
        (fun(a : sequent V L)(p_a : proof (G_n_set V L n) hyp a) =>
           forall(d : nat)(a_tl : sequent V L),
             a = f1 :: a_tl ->
             proof_depth p_a <= d ->
             exists(op : proof (G_n_set V L n) hyp (f2 :: a_tl)),
               proof_depth op <= d)
        (map (add_context (f1 :: sl) sr) ass)
        pl
      ->
        exists(opl : dep_list (sequent V L) (proof (G_n_set V L n) hyp)
                              (map (add_context (f2 :: sl) sr) ass)),
          every_dep_nth
            (fun(s : sequent V L)(p : proof (G_n_set V L n) hyp s) =>
                proof_depth p <= d)
            (map (add_context (f2 :: sl) sr) ass) opl.
  Proof.
    clear. 
    intros hyp n d f1 f2 sl sr ass pl H H0.
    eapply (list_subproofs_head_admissible _ _ _ _ _ _ [f2]).
      eexact H.
    trivial.
  Qed.

  Lemma conj_left_head_Gn_admissible :
    forall(n : nat)(hyp : set (sequent V L))(gamma : sequent V L)
          (f1 f2 : lambda_formula V L),
      conj_head_inversion_closed hyp ->
        forall(p_and : proof (G_n_set V L n) hyp gamma)
              (d : nat)(gamma_tl : sequent V L),
          gamma = (lf_and f1 f2) :: gamma_tl ->
          proof_depth p_and <= d ->
            exists(p : proof (G_n_set V L n) hyp (f1 :: gamma_tl)),
              proof_depth p <= d.
  Proof.
    clear. 
    induction p_and. 
      intros d gamma_tl H0 H1.
      subst gamma.
      specialize (H gamma_tl f1 f2 in_hypotheses).
      destruct H.
      exists (assume (G_n_set V L n) hyp (f1 :: gamma_tl) H).
      rewrite proof_depth_assume in *.
      trivial.
    intros d gamma_tl H1 H2.
    destruct d.
      eapply proof_depth_0.
      eexact H2.
    assert (H3 := rank_formula_and_left n f1 f2).
    destruct (decompose_G_n_set_coarsly n r in_rules).
      rewrite H1 in *.
      clear - H3 H4.
      decompose [and] H4; clear H4.
      lapply (other_axiom_G_n_set _ _ f1 _ H1 H2 H3).
        intros H0.
        exists (rule (G_n_set V L n) hyp 
                     {| assumptions := []; conclusion := f1 :: gamma_tl |}
                     H0 dep_nil).
        apply proof_depth_rule_le with 
                   (rules := (G_n_set V L n)) (r_rules := H0).
        apply every_dep_nth_empty.
      auto.
    apply proof_depth_rule_le_inv in H2.
    clear in_rules.
    decompose [ex and] H4; clear H4.
    clear x2 H5 H10.
    rename x into rbase, x0 into sl, x1 into sr.
    subst r.
    simpl in *.
    unfold add_context in H1.
    destruct sl.
      simpl in *.
      clear H3 H0.
      decompose [ex and or dep_and] H8; clear H8.
          subst rbase.
          simpl in *.
          inversion H1.
          subst x x0 sr; clear H1.
          unfold add_context in *.
          simpl in *.
          remember (@cons (sequent V L) (f1 :: gamma_tl) 
                      (@cons (sequent V L) (f2 :: gamma_tl) [])) in *.
          destruct pl.
            discriminate.
          inversion Heql.
          subst a al.
          exists p.
          apply le_S.
          apply every_dep_nth_head in H2.
          trivial.
        clear - H1 H3.
        exfalso.
        subst rbase.
        discriminate.
      clear - H1 H0.
      exfalso.
      subst rbase.
      discriminate.
    simpl in H1.
    inversion H1; clear H1.
    subst l gamma_tl.
    assert (H10 := other_context_G_n_set _ _ f1 _ _ _ H8 H6 H3).
    assert (H11 := subproofs_head_admissible _ _ _ _ _ _ _ _ _ H2 H0).
    destruct H11.
    exists (rule (G_n_set V L n) hyp 
                   (rule_add_context (f1 :: sl) sr rbase) H10 x).
    clear - H1.
    apply proof_depth_rule_le with (rules := (G_n_set V L n)) (r_rules := H10).
    trivial.
  Qed.


  Lemma conj_right_head_Gn_admissible :
    forall(n : nat)(hyp : set (sequent V L))(gamma : sequent V L)
          (f1 f2 : lambda_formula V L),
      conj_head_inversion_closed hyp ->
        forall(p_and : proof (G_n_set V L n) hyp gamma)
              (d : nat)(gamma_tl : sequent V L),
          gamma = (lf_and f1 f2) :: gamma_tl ->
          proof_depth p_and <= d ->
            exists(p : proof (G_n_set V L n) hyp (f2 :: gamma_tl)),
              proof_depth p <= d.
  Proof.
    clear. 
    induction p_and. 
      intros d gamma_tl H0 H1.
      subst gamma.
      specialize (H gamma_tl f1 f2 in_hypotheses).
      simpl in H.
      destruct H.
      exists (assume (G_n_set V L n) hyp (f2 :: gamma_tl) H0).
      rewrite proof_depth_assume in *.
      trivial.
    intros d gamma_tl H1 H2.
    destruct d.
      eapply proof_depth_0.
      eexact H2.
    assert (H3 := rank_formula_and_right n f1 f2).
    destruct (decompose_G_n_set_coarsly n r in_rules).
      rewrite H1 in *.
      clear - H3 H4.
      decompose [and] H4; clear H4.
      lapply (other_axiom_G_n_set _ _ f2 _ H1 H2 H3).
        intros H0.
        exists (rule (G_n_set V L n) hyp 
                     {| assumptions := []; conclusion := f2 :: gamma_tl |}
                     H0 dep_nil).
        apply proof_depth_rule_le with 
                   (rules := (G_n_set V L n)) (r_rules := H0).
        apply every_dep_nth_empty.
      auto.
    apply proof_depth_rule_le_inv in H2.
    clear in_rules.
    decompose [ex and] H4; clear H4.
    clear x2 H5 H10.
    rename x into rbase, x0 into sl, x1 into sr.
    subst r.
    simpl in *.
    unfold add_context in H1.
    destruct sl.
      simpl in *.
      clear H3 H0.
      decompose [ex and or dep_and] H8; clear H8.
          subst rbase.
          simpl in *.
          inversion H1.
          subst x x0 sr; clear H1.
          unfold add_context in *.
          simpl in *.
          remember (@cons (sequent V L) (f1 :: gamma_tl) 
                      (@cons (sequent V L) (f2 :: gamma_tl) [])) in *.
          destruct pl.
            discriminate.
          destruct pl.
            discriminate.
          inversion Heql.
          subst a a0 al.
          exists p0.
          apply le_S.
          apply every_dep_nth_tail in H2.
          apply every_dep_nth_head in H2.
          trivial.
        clear - H1 H3.
        exfalso.
        subst rbase.
        discriminate.
      clear - H1 H0.
      exfalso.
      subst rbase.
      discriminate.
    simpl in H1.
    inversion H1; clear H1.
    subst l gamma_tl.
    assert (H10 := other_context_G_n_set _ _ f2 _ _ _ H8 H6 H3).
    assert (H11 := subproofs_head_admissible _ _ _ _ _ _ _ _ _ H2 H0).
    destruct H11.
    exists (rule (G_n_set V L n) hyp 
                   (rule_add_context (f2 :: sl) sr rbase) H10 x).
    clear - H1.
    apply proof_depth_rule_le with (rules := (G_n_set V L n)) (r_rules := H10).
    trivial.
  Qed.

  Lemma neg_and_inv_head_Gn_admissible_ind :
    forall(n : nat)(hyp : set (sequent V L))(gamma : sequent V L)
          (f1 f2 : lambda_formula V L),
      disj_head_inversion_closed hyp ->
        forall(p_or : proof (G_n_set V L n) hyp gamma)
              (d : nat)(gamma_tl : sequent V L),
          gamma = (lf_neg (lf_and f1 f2)) :: gamma_tl ->
          proof_depth p_or <= d ->
            exists(p : proof (G_n_set V L n) hyp 
                             ((lf_neg f1) :: (lf_neg f2) :: gamma_tl)),
              proof_depth p <= d.
  Proof.
    clear. 
    induction p_or. 
      intros d gamma_tl H0 H1.
      subst gamma.
      specialize (H gamma_tl f1 f2 in_hypotheses).
      simpl in H.
      exists (assume (G_n_set V L n) hyp
                     ((lf_neg f1) :: (lf_neg f2) :: gamma_tl) H).
      rewrite proof_depth_assume in *.
      trivial.
    intros d gamma_tl H1 H2.
    destruct d.
      eapply proof_depth_0.
      eexact H2.
    assert (rank_sequent n (conclusion r)).
      apply in_rules.
    assert (rank_formula n (lf_neg f1) /\ rank_formula n (lf_neg f2)).
      rewrite H1 in *.
      clear - H3.
      apply rank_sequent_head in H3.
      rewrite rank_formula_lf_neg in *.
      split.
        eapply rank_formula_and_left; eauto.
      eapply rank_formula_and_right; eauto.
    clear H3.
    destruct H4.
    destruct (decompose_G_n_set_coarsly n r in_rules).
      rewrite H1 in *.
      clear - H3 H4 H5.
      decompose [and] H5; clear H5.
      lapply (sequent_other_axiom_G_n_set _ _ _ [lf_neg f1; lf_neg f2] H1 H2).
        intros H0; lapply H0; clear H0.
          intros H0.
          exists (rule (G_n_set V L n) hyp {| assumptions := []; 
                     conclusion := (lf_neg f1) :: (lf_neg f2) :: gamma_tl |}
                     H0 dep_nil).
          apply proof_depth_rule_le with 
                   (rules := (G_n_set V L n)) (r_rules := H0).
          apply every_dep_nth_empty.
        auto.
      intros H0.
      apply rank_sequent_cons; trivial.
      apply rank_sequent_cons; trivial.
      apply rank_sequent_empty.
    apply proof_depth_rule_le_inv in H2.
    decompose [ex and] H5; clear H5.
    clear in_rules x2 H6 H11.
    rename x into rbase, x0 into sl, x1 into sr.
    subst r.
    simpl in *.
    unfold add_context in H1.
    destruct sl.
      simpl in *.
      clear H3 H4 H0.
      decompose [ex and or dep_and] H9; clear H9.
          exfalso.
          clear - H1 H0.
          subst rbase.
          discriminate.
        subst rbase.
        simpl in *.
        inversion H1.
        subst x x0 sr; clear H1.
        unfold add_context in *.
        simpl in *.
        remember (@cons (sequent V L) ((lf_neg f1) :: (lf_neg f2) :: gamma_tl)
                        []) in *.
        destruct pl.
          discriminate.
        inversion Heql.
        subst a al.
        exists p.
        apply le_S.
        apply every_dep_nth_head in H2.
        trivial.
      exfalso.
      clear - H1 H0.
      subst rbase.
      discriminate.
    simpl in H1.
    inversion H1; clear H1.
    subst l gamma_tl.
    assert (G_n_set V L n (rule_add_context 
                         ((lf_neg f1) :: (lf_neg f2) :: sl) sr rbase)).
      clear pl H0 H2.
      apply sequent_other_context_G_n_set with (4 := H7); trivial.
      intros H0.
      apply rank_sequent_cons; trivial.
      apply rank_sequent_cons; trivial.
      apply rank_sequent_tail in H0.
      trivial.
    assert (H11 := list_subproofs_head_admissible _ _ _ _ _ _ 
                      [lf_neg f1; lf_neg f2] _ _ H2 H0).
    destruct H11.
    exists (rule (G_n_set V L n) hyp 
                   (rule_add_context 
                         ((lf_neg f1) :: (lf_neg f2) :: sl) sr rbase)
                    H1 x).
    apply proof_depth_rule_le with (rules := (G_n_set V L n)) (r_rules := H1).
    trivial.
  Qed.

  Lemma neg_and_inv_head_Gn_depth_admissible :
    forall(hyp : set (sequent V L))(n d : nat)
          (f1 f2 : lambda_formula V L)(s : sequent V L),
      disj_head_inversion_closed hyp ->
      provable_at_depth (G_n_set V L n) hyp d ((lf_neg (lf_and f1 f2)) :: s) ->
        provable_at_depth (G_n_set V L n) hyp d 
                          ((lf_neg f1) :: (lf_neg f2) :: s).
  Proof.
    intros hyp n d f1 f2 s H H0.
    destruct H0 as [p].
    apply neg_and_inv_head_Gn_admissible_ind with (p_or := p); trivial.
  Qed.

  Lemma neg_and_inv_head_G_n_hyp_admissible :
    forall(hyp : set (sequent V L))(n : nat)
          (f1 f2 : lambda_formula V L)(s : sequent V L),
      disj_head_inversion_closed hyp ->
      provable (G_n_set V L n) hyp ((lf_neg (lf_and f1 f2)) :: s) ->
        provable (G_n_set V L n) hyp ((lf_neg f1) :: (lf_neg f2) :: s).
  Proof.
    intros hyp n f1 f2 s H H0.
    destruct H0 as [p].
    clear H0.
    eapply provable_at_depth_provable.
    apply neg_and_inv_head_Gn_depth_admissible.
      trivial.
    apply provable_at_proof_depth with (p := p).
  Qed.


  Lemma neg_inv_head_Gn_depth_admissible_ind :
    forall(n : nat)(hyp : set (sequent V L))(gamma : sequent V L)
          (f : lambda_formula V L),
          neg_head_inversion_closed hyp ->
        forall(p_neg : proof (G_n_set V L n) hyp gamma)
              (d : nat)(gamma_tl : sequent V L),
          gamma = (lf_neg (lf_neg f)) :: gamma_tl ->
          proof_depth p_neg <= d ->
            exists(p : proof (G_n_set V L n) hyp (f :: gamma_tl)),
              proof_depth p <= d.
  Proof.
    clear. 
    induction p_neg.
      intros d gamma_tl H0 H1.
      subst gamma.
      specialize (H gamma_tl f in_hypotheses).
      simpl in H.
      exists (assume (G_n_set V L n) hyp (f :: gamma_tl) H).
      rewrite proof_depth_assume in *.
      trivial.
    intros d gamma_tl H1 H2.
    destruct d.
      eapply proof_depth_0.
      eexact H2.
    assert (rank_formula n (lf_neg (lf_neg f)) -> rank_formula n f).
      rewrite rank_formula_lf_neg.
      rewrite rank_formula_lf_neg.
      trivial.
    destruct (decompose_G_n_set_coarsly n r in_rules).
      rewrite H1 in *.
      clear - H3 H4.
      decompose [and] H4; clear H4.
      lapply (other_axiom_G_n_set _ _ f _ H1 H2 H3).
        intros H0.
        exists (rule (G_n_set V L n) hyp 
                     {| assumptions := []; conclusion := f :: gamma_tl |}
                     H0 dep_nil).
        apply proof_depth_rule_le with 
                   (rules := (G_n_set V L n)) (r_rules := H0).
        apply every_dep_nth_empty.
      auto.
    apply proof_depth_rule_le_inv in H2.
    clear in_rules.
    decompose [ex and] H4; clear H4.
    clear x2 H5 H10.
    rename x into rbase, x0 into sl, x1 into sr.
    subst r.
    simpl in *.
    unfold add_context in H1.
    destruct sl.
      simpl in *.
      clear H0.
      decompose [ex and or dep_and] H8; clear H8.
          exfalso.
          clear - H0 H1.
          subst rbase.
          discriminate.
        exfalso.
        clear - H1 H4.
        subst rbase.
        discriminate.
      subst rbase.
      simpl in *.
      inversion H1.
      subst x sr; clear H1.
      unfold add_context in *.
      simpl in *.
      remember (@cons (sequent V L) (f :: gamma_tl) []) in *.
      destruct pl.
        discriminate.
      inversion Heql.
      subst a al.
      exists p.
      apply le_S.
      apply every_dep_nth_head in H2.
      trivial.
    simpl in H1.
    inversion H1; clear H1.
    subst l gamma_tl.
    assert (H10 := other_context_G_n_set _ _ f _ _ _ H8 H6 H3).
    assert (H11 := subproofs_head_admissible _ _ _ _ _ _ _ _ _ H2 H0).
    destruct H11.
    exists (rule (G_n_set V L n) hyp 
                   (rule_add_context (f :: sl) sr rbase) H10 x).
    clear - H1.
    apply proof_depth_rule_le with (rules := (G_n_set V L n)) (r_rules := H10).
    trivial.
  Qed.

  Lemma neg_inv_head_Gn_depth_admissible :
    forall(hyp : set (sequent V L))(n d : nat)
          (f : lambda_formula V L)(s : sequent V L),
      neg_head_inversion_closed hyp ->
      provable_at_depth (G_n_set V L n) hyp d ((lf_neg (lf_neg f)) :: s) ->
        provable_at_depth (G_n_set V L n) hyp d (f :: s).
  Proof.
    intros hyp n d f s H H0.
    destruct H0 as [p_neg].
    apply neg_inv_head_Gn_depth_admissible_ind with (p_neg := p_neg); trivial.
  Qed.

  Lemma neg_inv_head_Gn_hyp_admissible :
    forall(hyp : set (sequent V L))(n : nat)
          (f : lambda_formula V L)(s : sequent V L),
      neg_head_inversion_closed hyp ->
      provable (G_n_set V L n) hyp ((lf_neg (lf_neg f)) :: s) ->
        provable (G_n_set V L n) hyp (f :: s).
  Proof.
    intros hyp n f s H H0.
    destruct H0 as [p].
    clear H0.
    eapply provable_at_depth_provable.
    apply neg_inv_head_Gn_depth_admissible.
      trivial.
    apply provable_at_proof_depth with (p := p).
  Qed.


  (***************************************************************************
   ** **** Lemma 3.12, Inversion is admissible in G_n + H
   *)

  Lemma inversion_depth_preserving_admissible_Gn_H :
    forall(n : nat)(hyp : set (sequent V L)),
      sequent_multiset hyp ->
      head_inversion_closed hyp ->
        depth_preserving_admissible_rule_set (G_n_set V L n) hyp 
          (inversion_rules V L).
  Proof.
    clear. 
    unfold depth_preserving_admissible_rule_set, 
                  depth_preserving_admissible in *.
    intros n hyp H H0 r H1 d H2.
    assert (H3 := G_n_multiset V L n).
    unfold inversion_rules, inverted_and_left_rule, inverted_and_right_rule,
           inverted_or_rule, inverted_neg_rule in *.
    decompose [ex and or dep_and] H1; clear H1.
            apply multiset_depth_provability with (s := x1 :: x ++ x0); trivial.
              rewrite H6.
              apply list_reorder_move_append.
            rewrite H5 in *.
            assert (provable_at_depth (G_n_set V L n) hyp d 
                    ((lf_and x1 x2) :: x ++ x0)).
              apply multiset_depth_provability 
                        with (s := (x ++ lf_and x1 x2 :: x0)); trivial.
                apply list_reorder_symm.
                apply list_reorder_move_append.
              eapply every_nth_head.
              eexact H2.
            destruct H1.
            eapply conj_left_head_Gn_admissible with (p_and := x3); trivial.
            unfold head_inversion_closed in *.
            tauto.
          apply multiset_depth_provability with (s := x2 :: x ++ x0); trivial.
            rewrite H6.
            apply list_reorder_move_append.
          rewrite H4 in *.
          assert (provable_at_depth (G_n_set V L n) hyp d 
                    ((lf_and x1 x2) :: x ++ x0)).
            apply multiset_depth_provability 
                        with (s := (x ++ lf_and x1 x2 :: x0)); trivial.
              apply list_reorder_symm.
              apply list_reorder_move_append.
            eapply every_nth_head.
            eexact H2.
          destruct H1.
          eapply conj_right_head_Gn_admissible with (p_and := x3); trivial.
          unfold head_inversion_closed in *.
          tauto.
        apply multiset_depth_provability 
              with (s := (lf_neg x2) :: (lf_neg x3) :: x ++ x0 ++ x1); trivial.
          rewrite H5.
          apply list_reorder_cons_parts.
          rewrite app_assoc.
          rewrite app_assoc.
          apply list_reorder_cons_parts.
          apply list_reorder_refl.
        rewrite H4 in *.
        assert (provable_at_depth (G_n_set V L n) hyp d 
                    ((lf_neg (lf_and x2 x3)) :: x ++ x0 ++ x1)).
          apply multiset_depth_provability 
                    with (s := (x ++ (lf_neg (lf_and x2 x3)) :: x0 ++ x1)); 
                    trivial.
            apply list_reorder_symm.
            apply list_reorder_move_append.
          eapply every_nth_head.
          eexact H2.
        destruct H1.
        eapply neg_and_inv_head_Gn_admissible_ind with (p_or := x4); trivial.
        unfold head_inversion_closed in *.
        tauto.
      apply multiset_depth_provability 
            with (s := (lf_neg x2) :: (lf_neg x3) :: x ++ x0 ++ x1); trivial.
        rewrite H5.
        rewrite (app_comm_cons x0).
        rewrite (app_assoc _ (lf_neg x3 :: _)).
        apply list_reorder_cons_parts.
        rewrite <- app_assoc.
        rewrite <- app_comm_cons.
        apply list_reorder_cons_parts.
        apply list_reorder_refl.
      rewrite H4 in *.
      assert (provable_at_depth (G_n_set V L n) hyp d 
                  ((lf_neg (lf_and x2 x3)) :: x ++ x0 ++ x1)).
        apply multiset_depth_provability 
                  with (s := (x ++ (lf_neg (lf_and x2 x3)) :: x0 ++ x1)); 
                  trivial.
          apply list_reorder_symm.
          apply list_reorder_move_append.
        eapply every_nth_head.
        eexact H2.
      destruct H1.
      eapply neg_and_inv_head_Gn_admissible_ind with (p_or := x4); trivial.
      unfold head_inversion_closed in *.
      tauto.
    apply multiset_depth_provability with (s := x1 :: x ++ x0); trivial.
      rewrite H6.
      apply list_reorder_move_append.
    rewrite H4 in *.
    assert (provable_at_depth (G_n_set V L n) hyp d 
                    ((lf_neg (lf_neg x1)) :: x ++ x0)).
      apply multiset_depth_provability 
                        with (s := (x ++ (lf_neg (lf_neg x1)) :: x0)); trivial.
        apply list_reorder_symm.
        apply list_reorder_move_append.
      eapply every_nth_head.
      eexact H2.
    destruct H1.
    eapply neg_inv_head_Gn_depth_admissible_ind with (p_neg := x2); trivial.
    unfold head_inversion_closed in *.
    tauto.
  Qed.


  Lemma inversion_admissible_Gn_H :
    forall(n : nat)(hyp : set (sequent V L)),
      sequent_multiset hyp ->
      head_inversion_closed hyp ->
        admissible_rule_set (G_n_set V L n) hyp (inversion_rules V L).
  Proof.
    intros n hyp H H0.
    apply admissible_depth_preserving_admissible.
    apply inversion_depth_preserving_admissible_Gn_H.
      trivial.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Towards Inversion Lemma 3.13 for GR_n

         I prove only simple admissibility here, because the
         depth-preserving part is not needed and the proof for depth
         preservation is missing in the paper.
    *)

  (***************************************************************************)
  (** prove first that the set of hypothesis in 3.9,
      provable_subst_n_conclusions, is closed under inversion at head
      position. This is slightly more difficult than indicated in the
      paper, because invertable formulas might occur in the the delta
      part.
   *)

  Lemma non_modal_weaken_subst_formula :
    forall(sigma : lambda_subst V L)(r : sequent_rule V L)
          (f : lambda_formula V L)(s delta : sequent V L),
      one_step_rule r ->
      not (top_modal_form f) ->
      list_reorder (f :: s) (subst_sequent sigma (conclusion r) ++ delta) ->
        exists(delta_l delta_r : sequent V L),
          delta = delta_l ++ f :: delta_r.
  Proof.
    intros sigma r f s delta H H0 H1.
    assert (H2 := list_reorder_occurence (f :: s) _ 0 (lt_0_Sn _) H1).
    decompose [ex and or dep_and] H2; clear H2.
    simpl in b.
    assert (x < length (subst_sequent sigma (conclusion r)) \/
            x >= length (subst_sequent sigma (conclusion r))).
      omega.
    destruct H2.
      exfalso.
      rewrite nth_append_left with (n_less_l1 := H2) in b.
      clear H1 a.
      unfold subst_sequent in *.
      rewrite nth_map in b.
      unfold one_step_rule in *.
      decompose [and] H; clear H.
      clear H1 H3 H6.
      specialize (H4 x (nth_map_tcc _ _ _ H2)).
      destruct (nth (conclusion r) x 
                      (nth_map_tcc (subst_form sigma) (conclusion r) x H2)).
            contradiction.
          simpl in *.
          rewrite subst_form_char in b.
          destruct l.
                contradiction.
              contradiction.
            contradiction.
          subst f.
          apply H0.
          rewrite subst_form_char.
          simpl.
          trivial.
        contradiction.
      subst f.
      apply H0.
      simpl.
      trivial.
    rewrite nth_append_right with (n_greater := H2) in b.
    eexists.
    eexists.
    subst f.
    apply list_split_at_n.
  Qed.

  Lemma change_sequent_provable_subst_n_conclusion :
    forall(rules : set (sequent_rule V L))(n : nat)(npos : 0 < n)
          (f : lambda_formula V L)(s1 s2 : sequent V L),
      one_step_rule_set rules ->
      not (top_modal_form f) ->
      provable_subst_n_conclusions rules n npos (f :: s1) ->
      rank_sequent n s2 ->
        provable_subst_n_conclusions rules n npos (s2 ++ s1).
  Proof.
    intros rules n npos f s1 s2 H H0 H1 H2.
    unfold provable_subst_n_conclusions in *.
    decompose [ex and or dep_and] H1; clear H1.
    exists {| assumptions := assumptions x; conclusion := s2 ++ s1 |}.
    split.
      clear H6.
      unfold rank_weaken_subst_rule in *.
      decompose [ex and or dep_and] H4; clear H4.
      rename x into rsubst, x0 into rbase, x1 into sigma, x2 into delta.
      simpl in *.
      rewrite <- H3 in *.
      assert (H10 := non_modal_weaken_subst_formula _ _ _ _ _ (H _ H5) H0 H9).
      decompose [ex and or dep_and] H10; clear H10.
      rename x into delta_l, x0 into delta_r.
      exists rbase, sigma, (delta_l ++ s2 ++ delta_r).
      repeat split; trivial.
        clear - H2 H6 H8.
        rewrite H8 in H6.
        clear H8.
        apply rank_sequent_append.
          eapply rank_sequent_append_left.
          eexact H6.
        apply rank_sequent_append.
          trivial.
        eapply rank_sequent_tail.
        eapply rank_sequent_append_right.
        eexact H6.
      clear - H9 H8.
      rewrite H8 in H9.
      clear H8.
      rewrite app_assoc in *.
      apply (list_reorder_insert_list [] _ _ _ s2 s2).
        simpl.
        eapply list_reorder_tail.
        eexact H9.
      apply list_reorder_refl.
    split.
      trivial.
    trivial.
  Qed.

  Lemma change_form_provable_subst_n_conclusion :
    forall(rules : set (sequent_rule V L))(n : nat)(npos : 0 < n)
          (f1 f2 : lambda_formula V L)(s : sequent V L),
      one_step_rule_set rules ->
      not (top_modal_form f1) ->
      provable_subst_n_conclusions rules n npos (f1 :: s) ->
      rank_formula n f2 ->
        provable_subst_n_conclusions rules n npos (f2 :: s).
  Proof.
    intros rules n npos f1 f2 s H H0 H1 H2.
    eapply change_sequent_provable_subst_n_conclusion with (s2 := [f2]); eauto.
    apply rank_sequent_cons.
      trivial.
    apply rank_sequent_empty.
  Qed.

  Lemma provable_subst_n_conclusions_rank_non_modal :
    forall(rules : set (sequent_rule V L))(n : nat)(npos : 0 < n)
          (f : lambda_formula V L)(s : sequent V L),
      one_step_rule_set rules ->
      not (top_modal_form f) ->
      provable_subst_n_conclusions rules n npos (f :: s) ->
        rank_formula n f.
  Proof.
    intros rules n npos f s H H0 H1.
    unfold provable_subst_n_conclusions in *.
    decompose [ex and or dep_and] H1; clear H1.
    clear H5.
    unfold rank_weaken_subst_rule in *.
    decompose [ex and or dep_and] H3; clear H3.
    rename x into rsubst, x0 into rbase, x1 into sigma, x2 into delta.
    rewrite <- H2 in *.
    assert (H9 := non_modal_weaken_subst_formula _ _ _ _ _ (H _ H4) H0 H8).
    clear - H5 H9.
    decompose [ex and or dep_and] H9; clear H9.
    subst delta.
    eapply rank_sequent_head.
    eapply rank_sequent_append_right.
    eexact H5.
  Qed.


  Lemma head_inversion_provable_subst_n_conclusion :
    forall(rules : set (sequent_rule V L))(n : nat)(npos : 0 < n),
      one_step_rule_set rules ->
        head_inversion_closed (provable_subst_n_conclusions rules n npos).
  Proof.
    intros rules n npos H.
    unfold head_inversion_closed in *.
    repeat split.
          eapply change_form_provable_subst_n_conclusion; eauto.
            auto.
          eapply rank_formula_and_left.
          eapply provable_subst_n_conclusions_rank_non_modal; eauto.
        eapply change_form_provable_subst_n_conclusion; eauto.
          auto.
        eapply rank_formula_and_right.
        eapply provable_subst_n_conclusions_rank_non_modal; eauto.
      unfold disj_head_inversion_closed in *.
      intros sr f1 f2 H0.
      eapply change_sequent_provable_subst_n_conclusion 
         with (s2 := [lf_neg f1; lf_neg f2]); eauto.
        auto.
      apply rank_sequent_cons.
        rewrite rank_formula_lf_neg.
        eapply rank_formula_and_left.
        rewrite <- rank_formula_lf_neg.
        eapply provable_subst_n_conclusions_rank_non_modal; eauto.
      apply rank_sequent_cons.
        rewrite rank_formula_lf_neg.
        eapply rank_formula_and_right.
        rewrite <- rank_formula_lf_neg.
        eapply provable_subst_n_conclusions_rank_non_modal; eauto.
      apply rank_sequent_empty.
    unfold neg_head_inversion_closed in *.
    intros sr f H0.
    eapply change_form_provable_subst_n_conclusion; eauto.
      auto.
    rewrite <- rank_formula_lf_neg.
    rewrite <- rank_formula_lf_neg.
    eapply provable_subst_n_conclusions_rank_non_modal; eauto.
  Qed.


  (***************************************************************************
   ** **** Inversion Lemma 3.13 for GR_n
          See notes before on the missing depth preservation.
   *)

  Lemma inversion_admissible_GR_n :
    forall(nonempty_V : V)(rules : set (sequent_rule V L))(n : nat),
      0 < n ->
      one_step_rule_set rules ->
        admissible_rule_set 
          (GR_n_set rules n) (empty_sequent_set V L)
          (rank_rules n (inversion_rules V L)).
  Proof.
    unfold admissible_rule_set, admissible.
    intros nonempty_V rules n H H0 r H1 H2.
    rewrite GR_n_provable_with_premises with (npos := H)(1 := v_eq); trivial.
    apply inversion_admissible_Gn_H.
          apply multiset_provable_subst_n_conclusions.
        apply head_inversion_provable_subst_n_conclusion.
        trivial.
      eapply subset_rank_rules.
      eexact H1.
    intros i i_less.
    rewrite <- GR_n_provable_with_premises with (npos := H)(1 := v_eq); 
                   trivial.
  Qed.


  (** **** Inversion Lemma for GR
          comment on page 15 
    *)
  Theorem inversion_admissible_GR :
    forall(nonempty_V : V)(rules : set (sequent_rule V L)),
      one_step_rule_set rules ->
        admissible_rule_set 
          (GR_set rules) (empty_sequent_set V L)
          (inversion_rules V L).
  Proof.
    unfold admissible_rule_set, admissible in *.
    intros nonempty_V rules H r H0 H1.
    rewrite rank_proof_GR_fixed_rank with (1 := v_eq).
        apply inversion_admissible_GR_n.
                trivial.
              apply minimal_rule_rank_gt_0.
              apply inversion_rules_nonempty_conclusion.
              eexact H0.
            trivial.
          apply rank_rules_minimal_rule_rank.
            trivial.
          trivial.
        intros i i_less.
        rewrite <- rank_proof_GR_fixed_rank with (1 := v_eq).
            apply H1.
          trivial.
        apply minimal_rule_rank_assumptions.
      trivial.
    apply minimal_rule_rank_conclusion.
  Qed.


End Inversion.

Implicit Arguments conj_head_inversion_closed [V L].
Implicit Arguments disj_head_inversion_closed [V L].
Implicit Arguments neg_head_inversion_closed [V L].
Implicit Arguments head_inversion_closed [V L].
Implicit Arguments conj_left_head_Gn_admissible [V L].
Implicit Arguments conj_right_head_Gn_admissible [V L].
