(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: k_semantics.v,v 1.6 2013/04/10 11:17:15 tews Exp $
 *)


(** ** K example, semantics, 4.6 

      This module defines the semantics of K by providing the functor
      and the lambda structure. It also proves characterizations for
      the assumption and the conclusion of the rules.
*)

Require Export some_nth k_syntax semantics.

Section K_sementics.

  Hypothesis pred_ext : forall(A : Type)(P Q : A -> Prop),
   (forall(a : A), P a <-> Q a) -> P = Q.


  (***************************************************************************)
  (** ***  Semantics  *)
  (***************************************************************************)

  (** This is the covariant powerset functor *)
  Definition k_functor : functor.
  Proof.
    refine {| obj := fun(X : Type) => set X;
              fmap := fun(X Y : Type)(f : X -> Y)(xs : set X) =>
                         direct_img f xs;
              id_law := _;
              comp_law := _;
              fmap_feq_law := _
           |}.
        intros X P.
        apply pred_ext.
        intros x.
        unfold direct_img, id in *.
        split.
          intros H.
          decompose [ex and] H; clear H.
          subst x0.
          trivial.
        intros H.
        exists x.
        auto.
      intros X Y Z f g P.
      apply pred_ext.
      intros z.
      unfold direct_img, "∘" in *.
      split.
        intros H.
        decompose [ex and] H; clear H.
        exists (f x).
        split.
          trivial.
        exists x.
        auto.
      intros H.
      decompose [ex and] H; clear H.
      exists x0.
      rewrite H2.
      auto.
    intros X Y f1 f2 H P.
    apply pred_ext.
    intros y.
    unfold direct_img.
    split.
      intros H0.
      decompose [ex and] H0; clear H0.
      exists x.
    rewrite H in H2.
      auto.
    intros H0.
    decompose [ex and] H0; clear H0.
    exists x.
    rewrite H.
    auto.
  Defined.

  Lemma non_trivial_k_functor : non_trivial_functor k_functor.
  Proof.
    intros X H.
    simpl.
    unfold non_empty_type in *.
    exists (empty_set _).
    trivial.
  Qed.

  Definition k_lifting : lambda_structure_type KL k_functor :=
    fun(op : k_operator)(X : Type)(arg : counted_list (set X) 1)
                        (P : set X) =>
      subset P (counted_head arg).
  
  Definition k_lambda : lambda_structure KL k_functor.
  Proof.
    refine {| modal_semantics := k_lifting;
              set_equal_modal_semantics := _;
              fibred_semantics := _
           |}.
      intros X op preds_x_1 preds_x_2 H P.
      unfold k_functor, k_lifting in *.
      simpl in *.
      decompose [ex] (counted_list_eta preds_x_1).
      rename x into P1, x0 into Ptl1.
      decompose [ex] (counted_list_eta preds_x_2).
      rename x into P2, x0 into Ptl2.
      subst preds_x_1 preds_x_2.
      specialize (H 0 (lt_0_Sn _)).
      simpl in *.
      apply set_equal_subset_char in H.
      destruct H.
      split.
        intros H1.
        eapply subset_trans.
          eexact H1.
        trivial.
      intros H1.
      eapply subset_trans.
        eexact H1.
      trivial.
    intros op X Y f preds_y P.
    unfold k_functor, k_lifting in *.
    simpl in *.
    decompose [ex] (counted_list_eta preds_y).
    rename x into Q, x0 into Qtl.
    subst preds_y.
    simpl.
    unfold inv_img at 2.
    apply direct_image_left_adjoint.
  Defined.

  (** Prove the semantics of box, as test for the semantics *)
  Lemma box_semantics : 
    forall(f : lambda_formula KV KL)(m : model KV k_functor)(x : state m),
      form_semantics k_lambda m (box f) x <->
        (forall(x' : state m), coalg m x x' ->
           form_semantics k_lambda m f x').
    intros f m x.
    unfold box in *.
    rewrite form_semantics_char.
    simpl.
    unfold k_lifting, inv_img in *.
    simpl.
    apply iff_refl.
  Qed.


  (***************************************************************************)
  (** *** Characterization of assumption semantics  *)
  (***************************************************************************)

  Lemma simple_prop_seq_val_k_assumption_head :
    forall(X : Type)(coval : X -> set KV)(x : X)(n : nat),
      coval x 0 ->
        simple_prop_seq_val coval (k_assumption n)
                (propositional_sequent_k_assumption n) x.
  Proof.
    clear. 
    intros X coval x n H.
    apply simple_prop_seq_val_head.
    simpl.
    trivial.
  Qed.

  Lemma prop_seq_valid_k_assumption_0 : 
    forall(X : Type)(coval : X -> set KV),
      prop_seq_val_valid 0 coval (k_assumption 0) 
                         (propositional_sequent_k_assumption 0)
      <->
        forall(x : X), coval x 0.
  Proof.
    clear. 
    intros X coval.
    apply iff_refl.
  Qed.

  Lemma prop_seq_valid_renamed_k_assumption_0 : 
    forall(X : Type)(coval : X -> set KV)(f : KV -> KV)
          (prop_subst_ass : propositional_sequent 
                      (subst_sequent (rename_of_fun f) (k_assumption 0))),
      prop_seq_val_valid 0 coval 
                         (subst_sequent (rename_of_fun f) (k_assumption 0))
                         prop_subst_ass
      <->
        forall(x : X), coval x (f 0).
  Proof.
    clear. 
    intros X coval f prop_subst_ass.
    apply iff_refl.
  Qed.

  Lemma subst_sequent_renamed_k_assumption :
    forall(f : KV -> KV)(n : nat),
      subst_sequent (rename_of_fun f) (k_assumption n) =
        (lf_prop (f 0)) :: map neg_v (map f (seq 1 n)).
  Proof.
    clear. 
    intros f n.
    unfold k_assumption in *.
    remember (k_assumption_tail n) as kat.
    simpl.
    f_equal.
    subst kat.
    rewrite map_map.
    unfold subst_sequent, k_assumption_tail in *.
    rewrite map_map.
    apply map_ext.
    intros a.
    unfold neg_v, rename_of_fun in *.
    trivial.
  Qed.

  Lemma neg_v_sequent_double_neg_semantics :
    forall(X : Type)(coval : X -> set KV)(vl : list KV)(x : X)
          (prop_subst_ass : propositional_sequent (map neg_v vl)),
      ~every_nth (coval x) vl ->
        ~~simple_prop_seq_val coval (map neg_v vl) prop_subst_ass x.
  Proof.
    clear. 
    induction vl.
      intros x prop_subst_ass H H0.
      apply H.
      apply every_nth_empty.
    intros x prop_subst_ass H H0.
    simpl in *.
    apply H0.
    apply simple_prop_seq_val_head.
    simpl.
    intros H1.
    eapply IHvl.
      intros H2.
      apply H.
      apply every_nth_cons.
        trivial.
      eexact H2.
    intros H2.
    apply H0.
    apply simple_prop_seq_val_tail.
    eexact H2.
  Qed.

  Lemma prop_seq_valid_renamed_k_assumption_Sn : 
    forall(X : Type)(coval : X -> set KV)(f : KV -> KV)(n : nat)
          (prop_subst_ass : propositional_sequent
                     (subst_sequent (rename_of_fun f) (k_assumption (S n)))),
      prop_seq_val_valid 0 coval
                   (subst_sequent (rename_of_fun f) (k_assumption (S n)))
                   prop_subst_ass
      <->
        forall(x : X),
          ~~(every_nth (coval x) (map f (seq 1 (S n))) -> coval x (f 0)).
  Proof.
    clear. 
    intros X coval f n prop_subst_ass.
    split.
      intros H x H0.
      revert prop_subst_ass H.
      rewrite subst_sequent_renamed_k_assumption.
      intros prop_subst_ass H.
      rewrite simple_prop_seq_val_valid_correct in H.
      specialize (H x).
      unfold simple_prop_seq_val, some_neg_dep in *.
      decompose [ex and or dep_and] H; clear H.
        clear b.
        discriminate.
      apply H3; clear H3; intros H3.
      decompose [ex and or dep_and] H3; clear H3.
      rename x0 into i, a into i_less, b into H3.
      destruct i.
        simpl in *.
        auto.
      revert H3.
      generalize (prop_subst_ass (S i) i_less).
      unfold KV in *.
      rewrite nth_tail.
      rewrite nth_map.
      intros r H3.
      remember (S n) as sn.
      simpl in *.
      clear - i_less H0 H3.
      unfold set_inverse in *.
      apply H0.
      intros H.
      assert (H4 := i_less).
      rewrite map_length in H4.
      apply lt_S_n in H4.
      specialize (H i H4).
      exfalso.
      apply H3.
      erewrite nth_tcc_irr.
      eexact H.
    intros H.
    revert prop_subst_ass.
    rewrite subst_sequent_renamed_k_assumption.
    intros prop_subst_ass.
    rewrite simple_prop_seq_val_valid_correct.
    intros x.
    specialize (H x).
    apply simple_prop_seq_val_length_case_intro.
      discriminate.
    intros H1.
    apply (iff_right (double_neg_impl_neg_or _ _)) in H.
    apply H; clear H; intros H.
    destruct H.
      apply (contrapositive (simple_prop_seq_val_tail _ _ _ _ _)) in H1.
      apply neg_v_sequent_double_neg_semantics in H1; trivial.
    apply H1; clear H1.
    apply simple_prop_seq_val_head.
    trivial.
  Qed.

  Lemma prop_seq_valid_k_assumption_Sn : 
    forall(X : Type)(coval : X -> set KV)(n : nat),
      prop_seq_val_valid 0 coval (k_assumption (S n)) 
                         (propositional_sequent_k_assumption (S n))
      <->
        forall(x : X),
          ~~(every_nth (coval x) (seq 1 (S n)) -> coval x 0).
  Proof.
    clear. 
    intros X coval n.
    assert (H := prop_seq_valid_renamed_k_assumption_Sn X coval id n).
    rewrite rename_of_id in *.
    rewrite subst_sequent_id in *.
    rewrite map_id in *.
    apply H.
  Qed.

  (***************************************************************************)
  (** *** Characterization of conclusion semantics  *)
  (***************************************************************************)

  Lemma neg_box_v_sequent_semantics_char :
    forall(X : Type)(coval : X -> set KV)(P : set X)(vl : list KV)
          (vl_prop : prop_modal_prop_sequent (map neg_box_v vl)),
      ~every_nth (fun v => subset P (fun x => coval x v)) vl ->
        ~~simple_mod_seq_val k_lambda coval (map neg_box_v vl) vl_prop P.
  Proof.
    induction vl.
      intros vl_prop H H0.
      apply H.
      apply every_nth_empty.
    intros i H H0.
    simpl in *.
    apply H0.
    apply simple_mod_seq_val_head.
    simpl.
    unfold set_inverse, k_lifting in *.
    simpl.
    intros H1.
    eapply IHvl.
      intros H2.
      apply H.
      apply every_nth_cons.
        trivial.
      eexact H2.
    intros H2.
    apply H0.
    apply simple_mod_seq_val_tail.
    eexact H2.
  Qed.

  Lemma box_v_sequent_semantics_char :
    forall(X : Type)(coval : X -> set KV)(P : set X)(vl : list KV)
          (vl_prop : prop_modal_prop_sequent (map box_v vl)),
      some_nth (fun v => subset P (fun x => coval x v)) vl ->
        simple_mod_seq_val k_lambda coval (map box_v vl) vl_prop P.
  Proof.
    intros X coval P vl vl_prop H.
    unfold some_nth in *.
    destruct H as [n].
    destruct H as [n_less].
    assert (n < length (map box_v vl)).
      rewrite map_length.
      trivial.
    apply simple_mod_seq_val_nth_intro with (n_less := H0).
    generalize (vl_prop n H0).
    rewrite nth_map.
    intros vl_n_prop.
    simpl.
    unfold k_lifting in *.
    simpl.
    erewrite nth_tcc_irr.
    eexact H.
  Qed.

  Lemma short_mod_seq_valid_char :
    forall(X : Type)(coval : X -> set KV)(mods negs : sequent KV KL)
          (mv nv : list KV)(mod_neg_nonempty : mods ++ negs <> [])
          (mod_neg_prop : prop_modal_prop_sequent (mods ++ negs)),
      length (mods ++ negs) = 1 ->
      mods = map box_v mv ->
      negs = map neg_box_v nv ->
        (mod_seq_val_valid k_lambda coval (mods ++ negs) 
                           mod_neg_nonempty mod_neg_prop
         <->
           nv = [] /\
           exists(v : KV), mv = [v] /\
           forall(P : set X), subset P (fun x => coval x v)).
  Proof.
    intros X coval mods negs mv nv mod_neg_nonempty mod_neg_prop H H0 H1.
    subst mods negs.
    rewrite app_length in H.
    rewrite map_length in H.
    rewrite map_length in H.
    split.
      intros H0.
      destruct nv.
        destruct mv.
          discriminate.
        destruct mv.
          split.
            trivial.
          exists k.
          split.
            trivial.
          simpl in *.
          trivial.
        discriminate.
      destruct mv.
        destruct nv.
          exfalso.
          specialize (H0 (empty_set _)).
          simpl in *.
          unfold mod_seq_val in *.
          simpl in *.
          unfold set_inverse, k_lifting in *.
          apply H0.
          apply subset_empty_set.
        discriminate.
      simpl in H.
      omega.
    intros H0.
    decompose [ex and] H0; clear H0.
    rename x into v.
    subst nv mv.
    simpl in *.
    trivial.
  Qed.

  (* this is an instance of the previous lemma, but ... *)
  Lemma mod_seq_valid_k_conclusion_0 : 
    forall(X : Type)(coval : X -> set KV),
      mod_seq_val_valid k_lambda coval (k_conclusion 0) 
                         (k_conclusion_nonempty 0)
                         (prop_modal_prop_sequent_k_conclusion 0)
      <->
        forall(P : set X), subset P (fun x => coval x 0).
  Proof.
    intros X coval.
    apply iff_refl.
  Qed.

  Lemma long_mod_seq_valid_char :
    forall(X : Type)(coval : X -> set KV)(mods negs : sequent KV KL)
          (mv nv : list KV)(mod_neg_nonempty : mods ++ negs <> [])
          (mod_neg_prop : prop_modal_prop_sequent (mods ++ negs)),
      length (mods ++ negs) <> 1 ->
      mods = map box_v mv ->
      negs = map neg_box_v nv ->
        (mod_seq_val_valid k_lambda coval (mods ++ negs) 
                           mod_neg_nonempty mod_neg_prop
         <->
           forall(P : set X),
             ~~(every_nth (fun v => subset P (fun x => coval x v)) nv 
                  -> some_nth (fun v => subset P (fun x => coval x v)) mv)).
  Proof.
    intros X coval mods negs mv nv mod_neg_nonempty mod_neg_prop H H0 H1.
    subst mods negs.
    split.
      intros H0 P H1.
      rewrite simple_mod_seq_val_valid_correct in H0.
      specialize (H0 P).
      unfold simple_mod_seq_val, some_neg_dep in *.
      decompose [ex and or dep_and] H0; clear H0.
        auto.
      apply H4; clear H4; intros H4.
      decompose [ex and or dep_and] H4; clear H4.
      rename x into i, a into i_less, b into H4.
      apply (split_nat_case_lt i (length (map box_v mv))).
        intros H0.
        revert H4.
        generalize (mod_neg_prop i i_less).
        rewrite nth_append_left with (n_less_l1 := H0).
        rewrite nth_map.
        generalize (nth_map_tcc box_v mv i H0).
        clear i_less H0.
        intros i_less mod_prop H4.
        simpl in *.
        unfold k_lifting in *.
        simpl in *.
        apply H1.
        intros H2.
        eapply some_nth_some.
        eexact H4.
      intros H0.
      revert H4.
      generalize (mod_neg_prop i i_less).
      rewrite nth_append_right with (n_greater := H0).
      rewrite nth_map.
      generalize (nth_map_tcc neg_box_v nv (i - length (map box_v mv))
                    (nth_append_right_tcc (map box_v mv) 
                       (map neg_box_v nv) i i_less H0)).
      clear i_less H0.
      intros i_less neg_prop H4.
      simpl in *.
      unfold set_inverse, k_lifting in *.
      simpl in *.
      apply H1.
      intros H0.
      specialize (H0 (i - length (map box_v mv)) i_less).
      contradiction.
    intros H0.
    rewrite simple_mod_seq_val_valid_correct.
    intros P.
    specialize (H0 P).
    apply simple_mod_seq_val_length_case_intro.
      trivial.
    intros H1.
    apply (iff_right (double_neg_impl_neg_or _ _)) in H0.
    apply H0; clear H0; intros H0.
    destruct H0.
      apply (contrapositive (simple_mod_seq_val_append_left _ _ _ _ _ _))
            in H1.
      apply neg_box_v_sequent_semantics_char in H1; trivial.
    apply H1; clear H1.
    apply simple_mod_seq_val_append_right.
    apply box_v_sequent_semantics_char.
    trivial.
  Qed.

  Lemma mod_seq_valid_k_conclusion_Sn : 
    forall(X : Type)(coval : X -> set KV)(n : nat),
      mod_seq_val_valid k_lambda coval (k_conclusion (S n)) 
                         (k_conclusion_nonempty (S n))
                         (prop_modal_prop_sequent_k_conclusion (S n))
      <->
        forall(P : set X),
          ~~(every_nth (fun v => subset P (fun x => coval x v))
                       (seq 1 (S n))
               -> subset P (fun x => coval x 0)).
  Proof.
    intros X coval n.
    generalize (k_conclusion_nonempty (S n)).
    generalize (prop_modal_prop_sequent_k_conclusion (S n)).
    unfold k_conclusion in *.
    rewrite append_single_rev.
    change [box_v 0] with  (map box_v [0]).
    intros p n0.
    rewrite long_mod_seq_valid_char with (mv := [0])(nv := seq 1 (S n)); 
            trivial.
      clear.
      split.
        intros H P.
        specialize (H P).
        rewrite <- some_nth_singleton in H.
        trivial.
      intros H P.
      rewrite <- some_nth_singleton.
      apply H.
    discriminate.
  Qed.

End K_sementics.
