(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: k_syntax.v,v 1.8 2013/04/10 11:17:15 tews Exp $
 *)


(** * K example *)

(** ** K example, syntax, 4.6 

      This module defines syntax for K and the rule set.
*)

Require Export rule_sets renaming.

Section K_syntax.

  (***************************************************************************)
  (** ***  Syntax  *)
  (***************************************************************************)

  Definition KV : Set := nat.

  Definition kv_enum : enumerator KV.
  Proof.
    clear. 
    apply exist with (x := fun n => n).
    intros n1 n2 H.
    trivial.
  Defined.

  Definition kv_eq : eq_type KV := eq_nat_dec.

  Inductive k_operator : Set := box_op : k_operator.

  Definition KL : modal_operators :=
    {| operator := k_operator;
       arity := fun _ => 1
    |}.

  Definition kop_eq : eq_type (operator KL).
  Proof.
    unfold eq_type in *.
    refine (fun _ _ => left _).
    destruct x.
    destruct y.
    trivial.
  Qed.

  Definition k_formulas : Type := lambda_formula KV KL.

  Definition k_sequent : Type := sequent KV KL.


  Definition box(f : lambda_formula KV KL) : lambda_formula KV KL :=
    lf_modal (box_op : operator KL) (counted_cons f counted_nil).

  Definition neg_v(v : KV) : lambda_formula KV KL := lf_neg (lf_prop v).

  Definition box_v(v : KV) : lambda_formula KV KL := box (lf_prop v). 

  Definition neg_box_v(v : KV) : lambda_formula KV KL := 
    lf_neg (box (lf_prop v)).

  Lemma propositional_neg_v : forall(v : KV), propositional (neg_v v).
  Proof.
    clear. 
    intros v.
    apply propositional_neg_inv.
    apply propositional_lf_prop.
  Qed.

  Lemma simple_modal_box_v : 
    forall(v : KV), neg_form_maybe simple_modal_form (box_v v).
  Proof.
    clear. 
    intros v.
    simpl.
    apply every_nth_cons.
      simpl.
      trivial.
    apply every_nth_empty.
  Qed.

  Lemma simple_modal_neg_box_v : 
    forall(v : KV), neg_form_maybe simple_modal_form (neg_box_v v).
  Proof.
    clear. 
    intros v.
    simpl.
    apply every_nth_cons.
      simpl.
      trivial.
    apply every_nth_empty.
  Qed.

  Lemma injective_neg_box_v : injective neg_box_v.
  Proof.
    intros v1 v2 H.
    unfold neg_box_v in *.
    inversion H; clear H.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Some Properties *)
  (***************************************************************************)

  Lemma destruct_neg_sequent : forall(s : sequent KV KL),
    prop_sequent s ->
    every_nth neg_form s ->
      s = map neg_v (prop_var_sequent s).
  Proof.
    induction s.
      trivial.
    rename a into f.
    intros H H0.
    specialize (IHs (prop_sequent_tail _ _ H) (every_nth_tail _ _ _ H0)).
    apply prop_sequent_head in H.
    apply every_nth_head in H0.
    rewrite prop_var_sequent_cons.
    destruct f; try contradiction.
    simpl in *.
    destruct f; try contradiction.
    simpl.
    f_equal.
    trivial.
  Qed.

  Lemma destruct_mod_form : forall(f : lambda_formula KV KL),
    simple_modal_form f ->
      exists(v : KV),
        prop_var_formula f = [v] /\ f = box_v v.
  Proof.
    clear. 
    intros f H.
    destruct f; try contradiction.
    destruct op.
    simpl in *.
    assert (H0 := counted_list_eta c).
    simpl in *.
    decompose [ex] H0; clear H0.
    rename x into f, x0 into arg_tl.
    assert (H3 := counted_list_eta arg_tl).
    simpl in *.
    subst arg_tl c.
    simpl in *.
    apply every_nth_head in H.
    destruct f; try contradiction.
    exists k.
    auto.
  Qed.

  Lemma destruct_mod_sequent : forall(s : sequent KV KL),
    every_nth simple_modal_form s ->
      s = map box_v (prop_var_sequent s).
  Proof.
    clear. 
    induction s.
      trivial.
    rename a into f.
    intros H.
    specialize (IHs (every_nth_tail _ _ _ H)).
    apply every_nth_head in H.
    apply destruct_mod_form in H.
    destruct H as [v].
    destruct H.
    rewrite prop_var_sequent_cons.
    rewrite H.
    simpl in *.
    f_equal.
      trivial.
    trivial.
  Qed.

  Lemma destruct_neg_mod_form : forall(f : lambda_formula KV KL),
    neg_form_maybe simple_modal_form f ->
    neg_form f ->
      exists(v : KV),
        prop_var_formula f = [v] /\ f = neg_box_v v.
  Proof.
    intros f H H0.
    destruct f; try contradiction.
    simpl in *.
    apply destruct_mod_form in H.
    decompose [ex and or dep_and] H; clear H.
    exists x.
    rewrite prop_var_formula_char.
    split.
      trivial.
    subst f.
    trivial.
  Qed.

  Lemma destruct_neg_mod_sequent : forall(s : sequent KV KL),
    simple_modal_sequent s ->
    every_nth neg_form s ->
      s = map neg_box_v (prop_var_sequent s).
  Proof.
    clear. 
    induction s.
      trivial.
    rename a into f.
    intros H H0.
    specialize (IHs (simple_modal_sequent_tail _ _ H) 
                    (every_nth_tail _ _ _ H0)).
    apply simple_modal_sequent_head in H.
    apply every_nth_head in H0.
    apply destruct_neg_mod_form in H.
      destruct H as [v].
      destruct H.
      rewrite prop_var_sequent_cons.
      rewrite H.
      simpl in *.
      f_equal.
        trivial.
      trivial.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  Rule assumptions  *)
  (***************************************************************************)

  Definition k_assumption_tail(n : nat) : sequent KV KL := map neg_v (seq 1 n).

  Definition k_assumption(n : nat) : sequent KV KL :=
    (lf_prop 0) :: (k_assumption_tail n).

  Lemma length_k_assumption_tail : forall(n : nat),
    length (k_assumption_tail n) = n.
  Proof.
    clear. 
    intros n.
    unfold k_assumption_tail in *.
    rewrite map_length.
    apply seq_length.
  Qed.

  Lemma length_k_assumption : forall(n : nat),
    length (k_assumption n) = S n.
  Proof.
    intros n.
    simpl.
    rewrite length_k_assumption_tail.
    trivial.
  Qed.

  Lemma nth_k_assumption_tail : 
    forall(i n : nat)(i_less : i < length (k_assumption_tail n)),
      nth (k_assumption_tail n) i i_less = neg_v (S i).
  Proof.
    intros i n i_less.
    unfold k_assumption_tail in *.
    rewrite nth_map.
    rewrite nth_seq.
    trivial.
  Qed.

  Lemma nth_subst_k_assumption_tail :
    forall(sigma : lambda_subst KV KL)(i n : nat)
          (i_less : i < length (subst_sequent sigma (k_assumption_tail n))),
      nth (subst_sequent sigma (k_assumption_tail n)) i i_less = 
        lf_neg (sigma (S i)).
  Proof.
    intros sigma i n i_less.
    unfold subst_sequent in *.
    rewrite nth_map.
    rewrite nth_k_assumption_tail.
    unfold neg_v in *.
    repeat rewrite subst_form_char.
    trivial.
  Qed.

  Lemma prop_sequent_k_assumption_tail : forall(n : nat),
    prop_sequent (k_assumption_tail n).
  Proof.
    clear. 
    unfold prop_sequent in *.
    intros n i i_less.
    rewrite nth_k_assumption_tail.
    simpl.
    trivial.
  Qed.

  Lemma prop_sequent_k_assumption : forall(n : nat),
    prop_sequent (k_assumption n).
  Proof.
    clear. 
    intros n.
    apply prop_sequent_cons.
      simpl.
      trivial.
    apply prop_sequent_k_assumption_tail.
  Qed.

  Lemma prop_sequent_subst_k_assumption_tail :
    forall(sigma : lambda_subst KV KL)(n : nat),
      renaming sigma ->
        prop_sequent (subst_sequent sigma (k_assumption_tail n)).
  Proof.
    intros sigma n H.
    apply prop_sequent_renaming; trivial.
    apply prop_sequent_k_assumption_tail.
  Qed.

  Lemma prop_sequent_subst_k_assumption :
    forall(sigma : lambda_subst KV KL)(n : nat),
      renaming sigma ->
        prop_sequent (subst_sequent sigma (k_assumption n)).
  Proof.
    intros sigma n H.
    apply prop_sequent_renaming; trivial.
    apply prop_sequent_k_assumption.
  Qed.

  Lemma propositional_sequent_k_assumption_tail : forall(n : nat),
    propositional_sequent (k_assumption_tail n).
  Proof.
    intros n.
    apply prop_sequent_is_propositional.
    apply prop_sequent_k_assumption_tail.
  Qed.

  Lemma propositional_sequent_k_assumption : forall(n : nat),
    propositional_sequent (k_assumption n).
  Proof.
    intros n.
    apply prop_sequent_is_propositional.
    apply prop_sequent_k_assumption.
  Qed.

  Lemma neg_sequent_subst_k_assumption_tail :
    forall(sigma : lambda_subst KV KL)(n : nat),
      every_nth neg_form (subst_sequent sigma (k_assumption_tail n)).
  Proof.
    intros sigma n.
    apply neg_sequent_subst_sequent.
    apply every_nth_map.
    intros i i_less.
    simpl.
    trivial.
  Qed.

  Lemma prop_var_sequent_k_assumption : forall(n : nat),
    prop_var_sequent (k_assumption n) = seq 0 (S n).
  Proof.
    clear. 
    unfold prop_var_sequent, k_assumption, k_assumption_tail in *.
    intros n.
    simpl.
    f_equal.
    rewrite map_map.
    rewrite map_ext with (g := fun(v : nat) => [v]).
      apply flatten_map_singleton_id.
    clear. 
    intros n.
    cbv.
    trivial.
  Qed.

  (***************************************************************************)
  (** ***  Rule conclusions  *)
  (***************************************************************************)

  Definition k_conclusion_tail(n : nat) : sequent KV KL := 
    map neg_box_v (seq 1 n).

  Definition k_conclusion(n : nat) : sequent KV KL :=
    (box_v 0) :: (k_conclusion_tail n).

  Lemma k_conclusion_nonempty : forall(n : nat),
    k_conclusion n <> [].
  Proof.
    discriminate.
  Qed.

  Lemma length_k_conclusion_tail : forall(n : nat),
    length (k_conclusion_tail n) = n.
  Proof.
    intros n.
    unfold k_conclusion_tail in *.
    rewrite map_length.
    rewrite seq_length.
    trivial.
  Qed.

  Lemma length_k_conclusion : forall(n : nat),
    length (k_conclusion n) = S n.
  Proof.
    intros n.
    simpl.
    rewrite length_k_conclusion_tail.
    trivial.
  Qed.

  Lemma nth_k_conclusion_tail : 
    forall(i n : nat)(i_less : i < length (k_conclusion_tail n)),
      nth (k_conclusion_tail n) i i_less = lf_neg (box (lf_prop (S i))).
  Proof.
    intros i n i_less.
    unfold k_conclusion_tail in *.
    rewrite nth_map.
    rewrite nth_seq.
    trivial.
  Qed.

  Lemma nth_subst_k_conclusion_tail : 
    forall(sigma : lambda_subst KV KL)(i n : nat)
          (i_less : i < length (subst_sequent sigma (k_conclusion_tail n))),
      nth (subst_sequent sigma (k_conclusion_tail n)) i i_less =
        lf_neg (box (sigma (S i))).
  Proof.
    intros sigma i n i_less.
    unfold subst_sequent in *.
    rewrite nth_map.
    rewrite nth_k_conclusion_tail.
    trivial.
  Qed.

  Lemma simple_modal_sequent_k_conclusion_tail : forall(n : nat),
    simple_modal_sequent (k_conclusion_tail n).
  Proof.
    intros n i i_less.
    rewrite nth_k_conclusion_tail.
    apply simple_modal_neg_box_v.
  Qed.

  Lemma simple_modal_sequent_k_conclusion : forall(n : nat),
    simple_modal_sequent (k_conclusion n).
  Proof.
    unfold k_conclusion in *.
    intros n.
    apply simple_modal_sequent_cons.
      apply simple_modal_box_v.
    apply simple_modal_sequent_k_conclusion_tail.
  Qed.

  Lemma simple_modal_sequent_subst_k_conclusion_tail :
    forall(sigma : lambda_subst KV KL)(n : nat),
      renaming sigma ->
        simple_modal_sequent (subst_sequent sigma (k_conclusion_tail n)).
  Proof.
    intros sigma n H.
    apply simple_modal_sequent_renaming; trivial.
    apply simple_modal_sequent_k_conclusion_tail.
  Qed.

  Lemma prop_modal_prop_sequent_k_conclusion : forall(n : nat),
    prop_modal_prop_sequent (k_conclusion n).
  Proof.
    clear. 
    intros n.
    apply simple_modal_sequent_is_prop_modal_prop.
    apply simple_modal_sequent_k_conclusion.
  Qed.

  Lemma neg_sequent_subst_k_conclusion_tail :
    forall(sigma : lambda_subst KV KL)(n : nat),
      every_nth neg_form (subst_sequent sigma (k_conclusion_tail n)).
  Proof.
    intros sigma n.
    apply neg_sequent_subst_sequent.
    apply every_nth_map.
    intros i i_less.
    simpl.
    trivial.
  Qed.

  Lemma prop_var_sequent_k_conclusion : forall(n : nat),
    prop_var_sequent (k_conclusion n) = seq 0 (S n).
  Proof.
    unfold prop_var_sequent, k_conclusion, k_conclusion_tail in *.
    intros n.
    simpl.
    f_equal.
    rewrite map_map.
    rewrite map_ext with (g := fun(v : nat) => [v]).
      apply flatten_map_singleton_id.
    clear. 
    intros n.
    cbv.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  Rules  *)
  (***************************************************************************)

  Definition k_rule(n : nat) : sequent_rule KV KL :=
    {| assumptions := [k_assumption n]; conclusion := k_conclusion n |}.

  Definition k_rules(r : sequent_rule KV KL) : Prop :=
    exists(n : nat)(sa : sequent KV KL),
      assumptions r = [sa] /\ list_reorder sa (k_assumption n) /\
        list_reorder (conclusion r) (k_conclusion n).

  Lemma k_rules_rule : forall(n : nat), k_rules (k_rule n).
  Proof.
    clear. 
    intros n.
    exists n, (k_assumption n).
    repeat split; trivial.
      apply list_reorder_refl.
    apply list_reorder_refl.
  Qed.

  Lemma one_step_rule_set_k_rules : one_step_rule_set k_rules.
  Proof.
    clear. 
    unfold one_step_rule_set, one_step_rule, k_rules in *.
    intros r H.
    decompose [ex and] H; clear H.
    rename x into n, x0 into sa.
    rewrite H0 in *; clear H0.
    repeat split.
          apply every_nth_cons.
            eapply prop_sequent_list_reorder.
              apply list_reorder_symm.
              eexact H1.
            apply prop_sequent_k_assumption.
          apply every_nth_empty.
        eapply simple_modal_sequent_list_reorder.
          apply list_reorder_symm.
          eexact H3.
        apply simple_modal_sequent_k_conclusion.
      apply every_nth_cons.
        eapply list_reorder_incl_both.
            apply prop_var_sequent_list_reorder.
            eexact H1.
          apply prop_var_sequent_list_reorder.
          apply list_reorder_symm.
          eexact H3.
        rewrite prop_var_sequent_k_assumption.
        rewrite prop_var_sequent_k_conclusion.
        apply incl_refl.
      apply every_nth_empty.
    intros H4.
    rewrite H4 in *; clear H4.
    apply list_reorder_nil_is_nil in H3.
    unfold k_conclusion in *.
    discriminate.
  Qed.


  (***************************************************************************)
  (** *** Renaming *)
  (***************************************************************************)

  Definition k_rename_fun(vl : list KV) : KV -> KV := fun(v : KV) =>
    match lt_dec v (length vl) with
      | left v_less => nth vl v v_less
      | right _ => v
    end.

  Lemma k_rename_fun_less : 
    forall(vl : list KV)(n : nat)(n_less : n < length vl),
      k_rename_fun vl n = nth vl n n_less.
  Proof.
    clear. 
    intros vl n n_less.
    unfold k_rename_fun in *.
    destruct (lt_dec n (length vl)).
      apply nth_tcc_irr.
    contradiction.
  Qed.

  Lemma subst_k_rename_k_assumption :
    forall(v : KV)(vl : list KV),
      subst_sequent (rename_of_fun (k_rename_fun (v :: vl)))
                    (k_assumption (length vl))
        = (lf_prop v) :: (map neg_v vl).
  Proof.
    intros v vl.
    simpl.
    f_equal.
    apply list_equal_nth_char.
      unfold subst_sequent in *.
      repeat rewrite map_length.
      apply length_k_assumption_tail.
    intros n n1_less n2_less.
    unfold subst_sequent, k_assumption_tail in *.
    repeat rewrite nth_map.
    rewrite nth_seq.
    unfold neg_v.
    repeat rewrite subst_form_char.
    repeat f_equal.
    unfold rename_of_fun in *.
    assert (H := n2_less).
    rewrite map_length in H.
    rewrite k_rename_fun_less with (n_less := lt_n_S _ _ H).
    f_equal.
    rewrite nth_tail.
    apply nth_tcc_irr.
  Qed.

  Lemma subst_k_rename_k_conclusion :
    forall(v : KV)(vl : list KV),
      subst_sequent (rename_of_fun (k_rename_fun (v :: vl)))
                    (k_conclusion (length vl))
        = (box_v v) :: (map neg_box_v vl).
  Proof.
    intros v vl.
    simpl.
    f_equal.
    unfold k_conclusion_tail in *.
    apply list_equal_nth_char.
      unfold subst_sequent in *.
      repeat rewrite map_length.
      apply seq_length.
    intros n n1_less n2_less.
    unfold subst_sequent in *.
    repeat rewrite nth_map.
    rewrite nth_seq.
    unfold neg_box_v, box.
    repeat rewrite subst_form_char.
    simpl.
    repeat f_equal.
    rewrite subst_form_char.
    unfold rename_of_fun in *.
    assert (H := n2_less).
    rewrite map_length in H.
    rewrite k_rename_fun_less with (n_less := lt_n_S _ _ H).
    f_equal.
    rewrite nth_tail.
    apply nth_tcc_irr.
  Qed.

  Lemma subst_form_box_v :
    forall(sigma : lambda_subst KV KL)(v : KV),
      renaming sigma ->
        exists(v' : KV), 
          sigma v = lf_prop v' /\
          subst_form sigma (box_v v) = box_v v'.
  Proof.
    intros sigma v H.
    specialize (H v).
    unfold box_v, box in *.
    rewrite subst_form_char.
    simpl.
    rewrite subst_form_char.
    destruct (sigma v); try contradiction.
    eauto.
  Qed.

  Lemma subst_form_neg_box_v :
    forall(sigma : lambda_subst KV KL)(v : KV),
      renaming sigma ->
        exists(v' : KV), 
          sigma v = lf_prop v' /\
          subst_form sigma (neg_box_v v) = neg_box_v v'.
  Proof.
    intros sigma v H.
    unfold neg_box_v in *.
    rewrite subst_form_char.
    assert (H0 := subst_form_box_v _ v H).
    decompose [ex and or dep_and] H0; clear H0.
    rename x into v'.
    exists v'.
    unfold box_v in *.
    rewrite H3.
    auto.
  Qed.

  Lemma prop_var_sequent_subst_k_conclusion_tail_ind :
    forall(sigma : lambda_subst KV KL)(vl : list KV),
      renaming sigma ->
        prop_var_sequent (subst_sequent sigma (map neg_box_v vl)) =
        prop_var_sequent (subst_sequent sigma (map neg_v vl)).
  Proof.
    induction vl.
      intros H.
      simpl.
      trivial.
    rename a into v.
    intros H.
    specialize (IHvl H).
    simpl.
    rewrite prop_var_sequent_cons.
    rewrite prop_var_sequent_cons.
    f_equal.
      clear IHvl.
      unfold neg_box_v, box, neg_v in *.
      repeat rewrite subst_form_char.
      simpl.
      repeat rewrite subst_form_char.
      rewrite prop_var_formula_char.
      rewrite prop_var_formula_char.
      rewrite prop_var_formula_char.
      unfold prop_var_modal_args in *.
      simpl.
      apply app_nil_r.
    trivial.
  Qed.

  Lemma prop_var_sequent_subst_k_conclusion_tail :
    forall(sigma : lambda_subst KV KL)(n : nat),
      renaming sigma ->
        prop_var_sequent (subst_sequent sigma (k_conclusion_tail n)) =
        prop_var_sequent (subst_sequent sigma (k_assumption_tail n)).
  Proof.
    intros sigma n.
    apply prop_var_sequent_subst_k_conclusion_tail_ind.
  Qed.

End K_syntax.
