(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: list_set.v,v 1.9 2013/04/10 11:17:15 tews Exp $
 *)

(** ** Lists as simple sets

      This module contains additional results on [incl] and [lists_disjoint].
*)


Require Export reorder.

Section ListSubset.

  Variable A : Type.

  Lemma incl_empty : forall(l : list A), incl [] l.
  Proof.
    unfold incl in *.
    intros l a H.
    contradiction.
  Qed.

  Lemma incl_left_tail : forall(l1 l2 : list A)(a : A),
    incl (a :: l1) l2 -> incl l1 l2.
  Proof.
    intros l1 l2 a H a' H0.
    apply H.
    right.
    trivial.
  Qed.

  Lemma incl_lappl : forall(l1 l2 l3 : list A),
    incl (l1 ++ l2) l3 -> incl l1 l3.
  Proof.
    induction l1.
      intros l2 l3 H.
      apply incl_empty.
    intros l2 l3 H a' H0.
    simpl in H.
    destruct H0.
      subst a'.
      apply H.
      left.
      trivial.
    eapply IHl1.
      eapply incl_left_tail.
      eexact H.
    trivial.
  Qed.
  
  Lemma incl_lappr : forall(l1 l2 l3 : list A),
    incl (l1 ++ l2) l3 -> incl l2 l3.
  Proof.
    induction l1.
      simpl.
      trivial.
    intros l2 l3 H a' H0.
    eapply IHl1.
      eapply incl_left_tail.
      eexact H.
    trivial.
  Qed.
  
  Lemma incl_list_reorder : forall(l1 l2 : list A),
    list_reorder l1 l2 -> incl l1 l2.
  Proof.
    induction 1.
      apply incl_refl.
    intros b H0.
    simpl in H0.
    destruct H0.
      subst b.
      apply in_or_app.
      right.
      simpl.
      auto.
    specialize (IHlist_reorder b H0).
    rewrite <- (firstn_skipn n l2) in IHlist_reorder.
    rewrite in_app_iff in IHlist_reorder.
    apply in_or_app.
    destruct IHlist_reorder.
      auto.
    right.
    simpl.
    right.
    trivial.
  Qed.
  
  Lemma list_reorder_incl_both : forall(l1 l2 l3 l4 : list A),
    list_reorder l1 l2 ->
    list_reorder l3 l4 ->
    incl l2 l3 ->
      incl l1 l4.
  Proof.
    intros l1 l2 l3 l4 H H0 H1.
    eapply incl_tran.
      apply incl_list_reorder.
      eexact H.
    eapply incl_tran.
      eexact H1.
    apply incl_list_reorder.
    trivial.
  Qed.

  Lemma incl_rev : forall(l : list A), incl (rev l) l.
  Proof.
    induction l.
      simpl.
      apply incl_refl.
    simpl.
    apply incl_app.
      apply incl_tl.
      trivial.
    intros b H.
    destruct H.
      left.
      trivial.
    contradiction.
  Qed.

  Lemma incl_flatten_every : 
    forall(ll : list (list A))(l : list A),
      every_nth (fun(lx : list A) => incl lx l) ll ->
        incl (flatten ll) l.
  Proof.
    induction ll.
      intros l H.
      simpl.
      apply incl_empty.
    intros l H.
    simpl.
    apply incl_app.
      apply (every_nth_head _ _ _ H).
    apply IHll.
    eapply every_nth_tail.
    eexact H.
  Qed.


  (***************************************************************************)
  (** *** Disjoint lists *)
  (***************************************************************************)

  Definition lists_disjoint(l1 l2 : list A) : Prop :=
    forall(a : A), (In a l1 -> ~ In a l2) /\ (In a l2 -> ~ In a l1).

  Lemma lists_disjoint_right_empty : forall(l : list A),
    lists_disjoint l [].
  Proof.
    intros l a.
    simpl.
    auto.
  Qed.

  Lemma lists_disjoint_symm : forall(l1 l2 : list A),
    lists_disjoint l1 l2 -> lists_disjoint l2 l1.
  Proof.
    intros l1 l2 H a.
    split.
      apply H.
    apply H.
  Qed.

  Lemma lists_disjoint_subset_right : 
    forall(l1 l2 l3 : list A),
      incl l3 l2 ->
      lists_disjoint l1 l2 ->
        lists_disjoint l1 l3.
  Proof.
    intros l1 l2 l3 H H0 a.
    split.
      intros H1 H2.
      apply H in H2.
      apply H0 in H1.
        trivial.
      trivial.
    intros H1 H2.
    apply H in H1.
    apply H0 in H1.
    contradiction.
  Qed.


  Lemma lists_disjoint_right_cons : forall(l1 l2 : list A)(a : A),
    lists_disjoint l1 l2 ->
    ~ In a l1 ->
      lists_disjoint l1 (a :: l2).
  Proof.
    intros l1 l2 a H H0 a'.
    split.
      intros H1 H2.
      destruct H2.
        subst a'.
        contradiction.
      revert H2.
      apply H.
      trivial.
    intros H1.
    destruct H1.
      subst a'.
      trivial.
    apply H.
    trivial.
  Qed.

End ListSubset.

Implicit Arguments lists_disjoint [A].
