(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: modal_formulas.v,v 1.20 2013/04/10 11:17:15 tews Exp $
 *)

(** ** Different subsets of formulas and sequents

   This module defines
   - simple modal formulas/sequents, S(Lambda(V)), possibly negated 
     modal formulas with variables as arguments only
   - prop_modal_prop, Prop(Lambda(V)), propositional formulas over
     modalities with variables as arguments only
   - prop_modal formulas/sequents, Prop(Lambda(S)), propositional
     formulas over modalities with arbitrary arguments
   - top_modal formulas/sequents, S(Lambda(S)), possibly negated
     modalities with arbitrary arguments
 *)


Require Export substitution.

Section Modal_formulas.

  Variable V : Type.
  Variable L : modal_operators.

  (** need a decidable equality on propositional constants for some results *)
  Variable v_eq : eq_type V.


  (***************************************************************************)
  (** *** Simple modal sequents  *)
  (***************************************************************************)

  Definition simple_modal_form(f : lambda_formula V L) : Prop :=
    match f with
      | lf_modal op args => every_nth prop_form (list_of_counted_list args)
      | _ => False
    end.

  Definition simple_modal_sequent(s : sequent V L) : Prop :=
    every_nth (neg_form_maybe simple_modal_form) s.

  Lemma simple_modal_sequent_empty : simple_modal_sequent [].
  Proof.
    apply every_nth_empty.
  Qed.
    
  Lemma simple_modal_sequent_append :
    forall(s1 s2 : sequent V L),
      simple_modal_sequent s1 -> 
      simple_modal_sequent s2 ->
        simple_modal_sequent (s1 ++ s2).
  Proof.
    intros s1 s2.
    apply every_nth_append.
  Qed.

  Lemma simple_modal_sequent_append_left :
    forall(s1 s2 : sequent V L),
      simple_modal_sequent (s1 ++ s2) -> simple_modal_sequent s1.
  Proof.
    intros s1 s2.
    apply every_nth_append_left.
  Qed.

  Lemma simple_modal_sequent_append_right :
    forall(s1 s2 : sequent V L),
      simple_modal_sequent (s1 ++ s2) -> simple_modal_sequent s2.
  Proof.
    intros s1 s2.
    apply every_nth_append_right.
  Qed.

  Lemma simple_modal_sequent_cons :
    forall(f : lambda_formula V L)(s : sequent V L),
      neg_form_maybe simple_modal_form f ->
      simple_modal_sequent s ->
        simple_modal_sequent (f :: s).
  Proof.
    intros f s H H0.
    apply every_nth_cons.
      trivial.
    trivial.
  Qed.

  Lemma simple_modal_sequent_head :
    forall(f : lambda_formula V L)(s : sequent V L),
      simple_modal_sequent (f :: s) -> neg_form_maybe simple_modal_form f.
  Proof.
    intros f s H.
    eapply every_nth_head.
    eexact H.
  Qed.

  Lemma simple_modal_sequent_tail :
    forall(f : lambda_formula V L)(s : sequent V L),
      simple_modal_sequent (f :: s) -> simple_modal_sequent s.
  Proof.
    intros f s H.
    eapply every_nth_tail.
    eexact H.
  Qed.

  Lemma simple_modal_sequent_list_reorder : forall(s1 s2 : sequent V L),
    list_reorder s1 s2 ->
    simple_modal_sequent s1 ->
      simple_modal_sequent s2.
  Proof.
    clear. 
    intros s1 s2 H H0.
    unfold simple_modal_sequent in *.
    rewrite <- every_nth_list_reorder.
      eexact H0.
    trivial.
  Qed.

  Lemma simple_modal_sequent_cutout_nth : forall(s : sequent V L)(n : nat),
    simple_modal_sequent s -> simple_modal_sequent (cutout_nth s n).
  Proof.
    intros s n.
    apply every_nth_cutout_nth.
  Qed.


  (***************************************************************************)
  (** **** Rank of simple modal formulas/sequents *)
  (***************************************************************************)

  Lemma rank_formula_simple_modal_form :
    forall(f : lambda_formula V L),
      simple_modal_form f ->
        rank_formula 2 f.
  Proof.
    intros f H.
    destruct f; try contradiction.
    apply rank_formula_modal.
      apply le_n.
    simpl in *.
    intros i i_less.
    apply rank_formula_prop_form.
    apply H.
  Qed.

  Lemma rank_simple_modal_sequent : forall(s : sequent V L),
    simple_modal_sequent s -> rank_sequent 2 s.
  Proof.
    intros s H i i_less.
    specialize (H i i_less).
    apply rank_formula_neg_form_maybe with (fs := simple_modal_form).
      clear. 
      intros f H.
      apply rank_formula_simple_modal_form.
      trivial.
    trivial.
  Qed.

  Lemma minimal_rank_simple_modal_sequent :
    forall(n : nat)(s : sequent V L),
      s <> [] -> 
      simple_modal_sequent s -> 
      rank_sequent n s ->
        1 < n.
  Proof.
    intros n s H H0 H1.
    unfold simple_modal_sequent, rank_sequent in *.
    destruct s.
      exfalso.
      apply H.
      trivial.
    assert (H2 := every_nth_head _ _ _ H0).
    assert (H3 := every_nth_head _ _ _ H1).
    clear H H0 H1.
    unfold rank_formula in *.
    destruct l.
          contradiction.
        simpl in *.
        destruct l.
              contradiction.
            contradiction.
          contradiction.
        simpl in *.
        rewrite modal_rank_char in H3.
        eapply rank_formula_modal_ge_2; eauto.
      contradiction.
    simpl in *.
    eapply rank_formula_modal_ge_2; eauto.
  Qed.

  Lemma rank_sequent_subst_simple_modal_sequent :
    forall(s : sequent V L)(sigma : lambda_subst V L)(n : nat),
      simple_modal_sequent s ->
      1 < n ->
      rank_subst (pred n) sigma ->
        rank_sequent n (subst_sequent sigma s).
  Proof.
    clear.
    intros s sigma n H H0 H1.
    assert (n = (2 + (pred n) - 1)).
      omega.
    rewrite H2.
    clear H2.
    eapply rank_sequent_subst_add.
          apply rank_simple_modal_sequent.
          trivial.
        eexact H1.
      omega.
    trivial.
  Qed.

  Lemma rank_subst_limit_simple_modal_form :
    forall(n : nat)(v : V)(sigma : lambda_subst V L)(f : lambda_formula V L),
      neg_form_maybe simple_modal_form f ->
      In v (prop_var_formula f) ->
      rank_formula n (subst_form sigma f) ->
        rank_formula (pred n) (sigma v).
  Proof.
    clear. 
    intros n v sigma f H H0 H1.
    destruct f; try contradiction.
      simpl in *.
      rewrite prop_var_formula_char in H0.
      rewrite subst_form_char in H1.
      destruct f; try contradiction.
      rewrite rank_formula_lf_neg in H1.
      eapply rank_subst_limit_lf_modal; eauto.
    eapply rank_subst_limit_lf_modal; eauto.
  Qed.

  Lemma rank_subst_limit_subst_rule :
    forall(n : nat)(s : sequent V L)(sigma : lambda_subst V L),
      s <> [] ->
      simple_modal_sequent s ->
      rank_sequent n (subst_sequent sigma s) ->
        rank_subst (pred n) (limit_subst v_eq (prop_var_sequent s) sigma).
  Proof.
    unfold rank_subst in *.
    intros n s sigma H H0 H1 v.
    unfold limit_subst in *.
    destruct (member v_eq v (prop_var_sequent s)) eqn:H2.
      apply member_In in H2.
      apply In_prop_var_sequent in H2.
      destruct H2 as [f].
      destruct H2.
      unfold rank_sequent, subst_sequent in *.
      rewrite every_nth_map in H1.
      apply every_nth_In with (a := f) in H1; trivial.
      apply rank_subst_limit_simple_modal_form with (f := f); trivial.
      eapply every_nth_In.
        eexact H0.
      trivial.
    apply rank_formula_id_subst.
    apply lt_pred.
    apply minimal_rank_simple_modal_sequent with (s := s); auto.
    eapply subst_sequent_rank_increase.
    eexact H1.
  Qed.


  (***************************************************************************)
  (** **** Lift injective substitutions to simple modal formulas  *)
  (***************************************************************************)

  Lemma injective_subst_prop_form :
    forall(sigma : lambda_subst V L)(f1 f2 : lambda_formula V L),
      injective sigma ->
      prop_form f1 ->
      prop_form f2 ->
      subst_form sigma f1 = subst_form sigma f2 ->
        f1 = f2.
  Proof.
    clear. 
    destruct f1; try contradiction.
    intros f2 H H0 H1 H2.
    destruct f2; try contradiction.
    rewrite subst_form_char in H2.
    rewrite subst_form_char in H2.
    apply H in H2.
    subst v0.
    trivial.
  Qed.

  Lemma injective_subst_simple_modal : 
    forall(sigma : lambda_subst V L)(f1 f2 : lambda_formula V L),
      injective sigma ->
      simple_modal_form f1 ->
      simple_modal_form f2 ->
      subst_form sigma f1 = subst_form sigma f2 ->
        f1 = f2.
  Proof.
    clear. 
    destruct f1 as [ | | | op args1]; try contradiction.
    destruct f2 as [ | | | op' args2]; try contradiction.
    intros H H0 H1 H2.
    simpl in *.
    rewrite subst_form_char in H2.
    rewrite subst_form_char in H2.
    assert (H3 := lf_modal_inversion_op _ _ _ _ H2).
    subst op'.
    apply lf_modal_inversion_args in H2.
    assert (list_of_counted_list (counted_map (subst_form sigma) args1) =
            list_of_counted_list (counted_map (subst_form sigma) args2)).
      rewrite H2.
      trivial.
    rewrite list_of_counted_list_map in H3.
    rewrite list_of_counted_list_map in H3.
    f_equal.
    apply counted_list_equal.
    revert H0 H1 H3.
    generalize (list_of_counted_list args1) as l1, 
               (list_of_counted_list args2) as l2.
    clear - H. 
    induction l1.
      intros l2 H0 H1 H3.
      destruct l2.
        trivial.
      discriminate.
    intros l2 H0 H1 H3.
    destruct l2.
      discriminate.
    simpl in H3.
    inversion H3; clear H3.
    apply injective_subst_prop_form in H4; trivial.
        subst l.
        rewrite IHl1 with (l2 := l2); trivial.
          eapply every_nth_tail; eauto.
        eapply every_nth_tail; eauto.
      eapply every_nth_head; eauto.
    eapply every_nth_head; eauto.
  Qed.

  Lemma injective_subst_neg_simple_modal : 
    forall(sigma : lambda_subst V L)(f1 f2 : lambda_formula V L),
      injective sigma ->
      neg_form_maybe simple_modal_form f1 ->
      neg_form_maybe simple_modal_form f2 ->
      subst_form sigma f1 = subst_form sigma f2 ->
        f1 = f2.
  Proof.
    intros sigma f1 f2 H H0 H1 H2.
    destruct f1 as [ | f1 | | op1 args1]; try contradiction.
      destruct f2 as [ | f2 | | op2 args2]; try contradiction.
        f_equal.
        rewrite subst_form_char in H2.
        rewrite (subst_form_char _ (lf_neg _)) in H2.
        inversion H2; clear H2.
        simpl in H0, H1.
        eapply injective_subst_simple_modal; eauto.
      exfalso.
      rewrite subst_form_char in H2.
      rewrite (subst_form_char _ (lf_modal _ _)) in H2.
      discriminate.
    destruct f2 as [ | f2 | | op2 args2]; try contradiction.
      exfalso.
      rewrite subst_form_char in H2.
      rewrite (subst_form_char _ (lf_neg _)) in H2.
      discriminate.
    simpl in H0, H1.
    eapply injective_subst_simple_modal; eauto.
  Qed.


  (***************************************************************************)
  (** **** Separete negated and unnegated formulas in simple modal sequents

          This is needed for one-step-completeness of K.
   *)
  (***************************************************************************)

  Lemma simple_modal_sequent_partition : forall(s : sequent V L),
    simple_modal_sequent s ->
    exists(ms ns : sequent V L),
      every_nth neg_form ns /\
      every_nth simple_modal_form ms /\
      list_reorder s (ms ++ ns).
  Proof.
    clear. 
    intros s H.
    assert (H0 := partition_result is_neg s).
    destruct H0.
    exists (snd (partition is_neg s)), (fst (partition is_neg s)).
    assert (list_reorder s 
                 (snd (partition is_neg s) ++ fst (partition is_neg s))).
      eapply list_reorder_trans.
        apply list_reorder_partition.
      apply list_reorder_append_swap.
    assert (H3 := H2).
    apply simple_modal_sequent_list_reorder in H3; trivial.
    repeat split.
        clear - H0.
        apply forallb_every_nth in H0.
        apply every_nth_mono with (Q := neg_form) in H0.
          trivial.
        clear. 
        intros f H.
        apply neg_form_is_neg in H.
        trivial.
      clear - H1 H3.
      apply simple_modal_sequent_append_left in H3.
      apply forallb_every_nth in H1.
      intros i i_less.
      specialize (H1 i i_less).
      specialize (H3 i i_less).
      rewrite negb_true_iff in H1.
      destruct (nth (snd (partition is_neg s)) i i_less); try contradiction.
        discriminate.
      trivial.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Prop(Lambda(V))  *)
  (***************************************************************************)

  (* Prop(Lambda(V)), page 7 *)
  Definition prop_modal_prop(f : lambda_formula V L) : Prop :=
    neg_and_over simple_modal_form f.

  Lemma prop_modal_prop_lambda_or :
    forall(f1 f2 : lambda_formula V L),
      prop_modal_prop f1 ->
      prop_modal_prop f2 ->
        prop_modal_prop (lambda_or f1 f2).
  Proof.
    intros f1 f2 H H0.
    unfold lambda_or, prop_modal_prop in *.
    simpl.
    auto.
  Qed.

  Lemma prop_modal_prop_tcc_prop : forall{X : Type}(v : V), 
    prop_modal_prop (lf_prop v) -> X.
  Proof.
    intros X v H.
    contradiction.
  Qed.

  Lemma prop_modal_prop_tcc_neg : forall(f : lambda_formula V L),
    prop_modal_prop (lf_neg f) -> prop_modal_prop f.
  Proof.
    intros f H.
    auto.
  Qed.
  
  Lemma prop_modal_prop_tcc_and_1 : forall(f1 f2 : lambda_formula V L),
    prop_modal_prop (lf_and f1 f2) -> prop_modal_prop f1.
  Proof.
    clear. 
    intros f1 f2 H.
    unfold prop_modal_prop in *.
    simpl in *.
    apply H.
  Qed.
  
  Lemma prop_modal_prop_tcc_and_2 : forall(f1 f2 : lambda_formula V L),
    prop_modal_prop (lf_and f1 f2) -> prop_modal_prop f2.
  Proof.
    clear. 
    intros f1 f2 H.
    unfold prop_modal_prop in *.
    apply H.
  Qed.
  
  Lemma prop_modal_prop_tcc_modal : 
    forall(op : operator L)
          (args : counted_list (lambda_formula V L) (arity L op)),
      prop_modal_prop (lf_modal op args) -> 
        every_nth prop_form (list_of_counted_list args).
  Proof.
    intros op args H.
    unfold prop_modal_prop in *.
    trivial.
  Qed.

  Lemma rank_formula_prop_modal_prop : forall(f : lambda_formula V L),
    prop_modal_prop f ->
      rank_formula 2 f.
  Proof.
    intros f H.
    apply rank_formula_neg_and_over with (fs := simple_modal_form).
      clear. 
      intros f H.
      apply rank_formula_simple_modal_form.
      trivial.
    trivial.
  Qed.

  Lemma simple_modal_form_is_prop_modal_prop :
    forall(f : lambda_formula V L),
      neg_form_maybe simple_modal_form f -> prop_modal_prop f.
  Proof.
    intros f H.
    destruct f.
          contradiction.
        destruct f.
              contradiction.
            contradiction.
          contradiction.
        simpl in *.
        trivial.
      contradiction.
    simpl in *.
    trivial.
  Qed.


  Definition prop_modal_prop_sequent(s : sequent V L) : Prop :=
    every_nth prop_modal_prop s. 

  Lemma prop_modal_prop_sequent_append_left :
    forall(s1 s2 : sequent V L),
      prop_modal_prop_sequent (s1 ++ s2) -> prop_modal_prop_sequent s1.
  Proof.
    intros s1 s2.
    apply every_nth_append_left.
  Qed.

  Lemma prop_modal_prop_sequent_append_right :
    forall(s1 s2 : sequent V L),
      prop_modal_prop_sequent (s1 ++ s2) -> prop_modal_prop_sequent s2.
  Proof.
    intros s1 s2.
    apply every_nth_append_right.
  Qed.

  Lemma prop_modal_prop_sequent_cons :
    forall(f : lambda_formula V L)(s : sequent V L),
      prop_modal_prop f ->
      prop_modal_prop_sequent s ->
        prop_modal_prop_sequent (f :: s).
  Proof.
    intros f s H H0.
    apply every_nth_cons.
      trivial.
    trivial.
  Qed.

  Lemma prop_modal_prop_sequent_head :
    forall(f : lambda_formula V L)(s : sequent V L),
      prop_modal_prop_sequent (f :: s) -> prop_modal_prop f.
  Proof.
    intros f s H.
    eapply every_nth_head.
    eexact H.
  Qed.

  Lemma prop_modal_prop_sequent_tail :
    forall(f : lambda_formula V L)(s : sequent V L),
      prop_modal_prop_sequent (f :: s) -> prop_modal_prop_sequent s.
  Proof.
    intros f s H.
    eapply every_nth_tail.
    eexact H.
  Qed.

  Lemma prop_modal_prop_or_formula_iter :
    forall(f : lambda_formula V L)(s : sequent V L),
      prop_modal_prop_sequent (f :: s) ->
        prop_modal_prop (or_formula_of_sequent_iter f s).
  Proof.
    intros f s.
    revert s f.
    induction s.
      intros f H.
      apply prop_modal_prop_sequent_head in H.
      trivial.
    rename a into f'.
    intros f H.
    simpl.
    apply IHs.
    apply prop_modal_prop_sequent_cons.
      apply prop_modal_prop_lambda_or.
        apply prop_modal_prop_sequent_head in H.
        trivial.
      apply prop_modal_prop_sequent_tail in H.
      apply prop_modal_prop_sequent_head in H.
      trivial.
    apply prop_modal_prop_sequent_tail in H.
    apply prop_modal_prop_sequent_tail in H.
    eexact H.
  Qed.

  Lemma prop_modal_prop_or_formula :
    forall(s : sequent V L)(nonempty_s : s <> []),
      prop_modal_prop_sequent s ->
        prop_modal_prop (or_formula_of_ne_sequent s nonempty_s).
  Proof.
    intros s nonempty_s H.
    destruct s.
      exfalso.
      auto.
    simpl.
    apply prop_modal_prop_or_formula_iter.
    trivial.
  Qed.

  Lemma simple_modal_sequent_is_prop_modal_prop :
    forall(s : sequent V L), 
      simple_modal_sequent s -> prop_modal_prop_sequent s.
  Proof.
    intros s.
    apply every_nth_mono.
    intros f.
    apply simple_modal_form_is_prop_modal_prop.
  Qed.

  Lemma rank_sequent_prop_modal_prop_sequent :
    forall(s : sequent V L),
      prop_modal_prop_sequent s ->
        rank_sequent 2 s.
  Proof.
    intros s H i i_less.
    apply rank_formula_prop_modal_prop.
    apply H.
  Qed.


  (***************************************************************************)
  (** *** Prop(Lambda(S))  *)
  (***************************************************************************)

  Definition modal_form(f : lambda_formula V L) : Prop :=
    match f with
      | lf_modal _ _ => True
      | _ => False
    end.
  
  Definition prop_modal(f : lambda_formula V L) : Prop :=
    neg_and_over modal_form f.

  Definition prop_modal_sequent(s : sequent V L) : Prop :=
    every_nth prop_modal s.

  Lemma prop_modal_sequent_cons :
    forall(f : lambda_formula V L)(s : sequent V L),
      prop_modal f ->
      prop_modal_sequent s ->
        prop_modal_sequent (f :: s).
  Proof.
    intros f s H H0.
    apply every_nth_cons.
      trivial.
    trivial.
  Qed.

  Lemma prop_modal_sequent_head :
    forall(f : lambda_formula V L)(s : sequent V L),
      prop_modal_sequent (f :: s) -> prop_modal f.
  Proof.
    intros f s H.
    eapply every_nth_head.
    eexact H.
  Qed.

  Lemma prop_modal_sequent_tail :
    forall(f : lambda_formula V L)(s : sequent V L),
      prop_modal_sequent (f :: s) -> prop_modal_sequent s.
  Proof.
    intros f s H.
    eapply every_nth_tail.
    eexact H.
  Qed.


  (***************************************************************************)
  (** *** Top modal sequents  *)
  (***************************************************************************)

  Definition top_modal_form(f : lambda_formula V L) : Prop :=
    neg_form_maybe modal_form f.

  Lemma top_modal_is_prop_modal : forall(f : lambda_formula V L),
    top_modal_form f -> prop_modal f.
  Proof.
    intros f H.
    unfold prop_modal in *.
    destruct f; try contradiction.
      destruct f; try contradiction.
      trivial.
    trivial.
  Qed.

  Lemma top_modal_form_lf_neg : forall(f : lambda_formula V L),
    top_modal_form (lf_neg f) -> top_modal_form f.
  Proof.
    intros f H.
    destruct f; try contradiction.
    trivial.
  Qed.

  Lemma top_modal_not_neg_prop : forall(f : lambda_formula V L),
    top_modal_form f -> ~ neg_form_maybe prop_form f.
  Proof.
    intros f H.
    destruct f; auto.
    destruct f; auto.
  Qed.

  Lemma top_modal_not_prop : forall(f : lambda_formula V L),
    top_modal_form f -> ~ prop_form f.
  Proof.
    intros f H.
    destruct f; auto.
  Qed.

  Definition top_modal_sequent(s : sequent V L) : Prop :=
    every_nth top_modal_form s.

  Lemma top_modal_sequent_list_reorder : forall(s1 s2 : sequent V L),
    list_reorder s1 s2 ->
    top_modal_sequent s2 ->
      top_modal_sequent s1.
  Proof.
    intros s1 s2 H H0.
    eapply every_nth_list_reorder.
      eexact H.
    trivial.
  Qed.

  Lemma top_modal_is_prop_modal_sequent : forall(s : sequent V L),
    top_modal_sequent s -> prop_modal_sequent s.
  Proof.
    intros s H i i_less.
    apply top_modal_is_prop_modal.
    apply H.
  Qed.

  Lemma top_modal_sequent_tail : 
    forall(f : lambda_formula V L)(s : sequent V L),
      top_modal_sequent (f :: s) -> top_modal_sequent s.
  Proof.
    intros f s.
    apply every_nth_tail.
  Qed.

  Lemma top_modal_sequent_head :
    forall(f : lambda_formula V L)(s : sequent V L),
      top_modal_sequent (f :: s) -> top_modal_form f.
  Proof.
    intros f s.
    apply every_nth_head.
  Qed.

End Modal_formulas.

Implicit Arguments prop_modal_prop [[V] [L]].
Implicit Arguments prop_modal_prop_lambda_or [V L].
Implicit Arguments prop_modal_prop_tcc_prop [V L X].
Implicit Arguments prop_modal_prop_tcc_neg [V L].
Implicit Arguments prop_modal_prop_tcc_and_1 [V L].
Implicit Arguments prop_modal_prop_tcc_and_2 [V L].
Implicit Arguments prop_modal_prop_tcc_modal [V L].
Implicit Arguments prop_modal_prop_sequent [V L].
Implicit Arguments prop_modal_prop_sequent_append_left [V L].
Implicit Arguments prop_modal_prop_sequent_append_right [V L].
Implicit Arguments prop_modal_prop_sequent_head [V L].
Implicit Arguments prop_modal_prop_sequent_tail [V L].
Implicit Arguments prop_modal_prop_or_formula [V L].
Implicit Arguments simple_modal_sequent_is_prop_modal_prop [V L].
Implicit Arguments modal_form [[V] [L]].
Implicit Arguments prop_modal_sequent [V L].
Implicit Arguments top_modal_form [V L].
Implicit Arguments top_modal_sequent [V L].
Implicit Arguments simple_modal_form [[V] [L]].
Implicit Arguments simple_modal_sequent [V L].
Implicit Arguments simple_modal_sequent_append_left [V L].
Implicit Arguments simple_modal_sequent_append_right [V L].
Implicit Arguments simple_modal_sequent_head [V L].
Implicit Arguments simple_modal_sequent_tail [V L].
Implicit Arguments simple_modal_sequent_list_reorder [V L].
Implicit Arguments injective_subst_neg_simple_modal [V L].
