(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: propositional_formulas.v,v 1.23 2013/04/10 11:17:16 tews Exp $
 *)

(** ** Propositional formulas and sequents *)

Require Export formulas.

Section Propositional_formulas.

  Variable V : Type.
  Variable L : modal_operators.


  Definition propositional(f : lambda_formula V L) : Prop :=
    rank_formula 1 f.

  Lemma propositional_lf_prop : forall(v : V),
    propositional (lf_prop v).
  Proof.
    intros v.
    apply rank_formula_lf_prop.
    trivial.
  Qed.

  Lemma propositional_neg : forall(f : lambda_formula V L),
    propositional (lf_neg f) -> propositional f.
  Proof.
    intros f.
    apply iff_right.
    apply rank_formula_lf_neg.
  Qed.
  
  Lemma propositional_neg_inv : forall(f : lambda_formula V L),
    propositional f -> propositional (lf_neg f).
  Proof.
    intros f.
    apply iff_left.
    apply rank_formula_lf_neg.
  Qed.
  
  Lemma propositional_lf_and : forall(f1 f2 : lambda_formula V L),
    propositional f1 ->
    propositional f2 ->
      propositional (lf_and f1 f2).
  Proof.
    intros f1 f2 H H0.
    apply rank_formula_lf_and; trivial.
  Qed.
  
  Lemma propositional_and_left : forall(f1 f2 : lambda_formula V L),
    propositional (lf_and f1 f2) -> propositional f1.
  Proof.
    intros f1 f2.
    apply rank_formula_and_left.
  Qed.
  
  Lemma propositional_and_right : forall(f1 f2 : lambda_formula V L),
    propositional (lf_and f1 f2) -> propositional f2.
  Proof.
    intros f1 f2.
    apply rank_formula_and_right.
  Qed.
  
  Lemma propositional_tcc_modal : 
    forall{X : Type}(op : operator L)
          (args : counted_list (lambda_formula V L) (arity L op)),
      propositional (lf_modal op args) -> X.
  Proof.
    intros X op args H.
    unfold propositional in *.
    assert (H0 := rank_formula_modal_ge_2 _ _ _ H).
    omega.
  Qed.

  Lemma propositional_lambda_or :
    forall(f1 f2 : lambda_formula V L),
      propositional f1 -> 
      propositional f2 -> 
        propositional (lambda_or f1 f2).
  Proof.
    intros f1 f2 H H0.
    apply rank_formula_lambda_or.
      trivial.
    trivial.
  Qed.

  (* 
   * Lemma propositional_false : forall(nonempty_v : V),
   *   propositional (lambda_false nonempty_v).
   * Proof.
   *   intros nonempty_v.
   *   apply rank_formula_false.
   *   trivial.
   * Qed.
   *)


  Definition propositional_sequent(s : sequent V L) : Prop :=
    rank_sequent 1 s.
  
  Lemma propositional_sequent_empty : propositional_sequent [].
  Proof.
    apply every_nth_empty.
  Qed.

  Lemma propositional_sequent_head : 
    forall(s : sequent V L)(f : lambda_formula V L),
      propositional_sequent (f :: s) -> propositional f.
  Proof.
    intros s f H.
    eapply every_nth_head.
    eexact H.
  Qed.

  Lemma propositional_sequent_tail : 
    forall(s : sequent V L)(f : lambda_formula V L),
      propositional_sequent (f :: s) -> propositional_sequent s.
  Proof.
    intros s f H n n_less.
    eapply every_nth_tail.
    eexact H.
  Qed.

  Lemma propositional_or_formula : 
    forall(nonempty_v : V)(s : sequent V L),
      propositional_sequent s ->
        propositional (or_formula_of_sequent s nonempty_v).
  Proof.
    intros s nonempty_v H.
    apply rank_formula_or_formula_of_sequent.
      trivial.
    trivial.
  Qed.

  Lemma propositional_sequent_cons : 
    forall(s : sequent V L)(f : lambda_formula V L),
      propositional f ->
      propositional_sequent s ->
        propositional_sequent (f :: s).
  Proof.
    unfold propositional_sequent in *.
    intros l f H H0.
    apply every_nth_cons.
      trivial.
    trivial.
  Qed.

  Lemma propositional_sequent_append :
    forall(sl sr : sequent V L),
      propositional_sequent sl -> 
      propositional_sequent sr ->
        propositional_sequent (sl ++ sr).
  Proof.
    unfold propositional_sequent.
    intros sl sr H H0.
    apply every_nth_append.
      trivial.
    trivial.
  Qed.

  Lemma propositional_sequent_append_left :
    forall(sl sr : sequent V L),
      propositional_sequent (sl ++ sr) ->
        propositional_sequent sl.
  Proof.
    unfold propositional_sequent in *.
    intros sl sr H.
    eapply every_nth_append_left.
    eexact H.
  Qed.

  Lemma propositional_sequent_append_right :
    forall(sl sr : sequent V L),
      propositional_sequent (sl ++ sr) ->
        propositional_sequent sr.
  Proof.
    unfold propositional_sequent in *.
    intros sl sr H.
    eapply every_nth_append_right.
    eexact H.
  Qed.

  Lemma propositional_sequent_cons_append :
    forall(l1 l2 : sequent V L)(f1 f2 : lambda_formula V L),
      (propositional f1 -> propositional f2) ->
      propositional_sequent (l1 ++ f1 :: l2) ->
        propositional_sequent (l1 ++ f2 :: l2).
  Proof.
    clear. 
    intros l1 l2 f1 f2 H H0.
    apply propositional_sequent_append.
      eapply propositional_sequent_append_left.
      eexact H0.
    assert (H1 := propositional_sequent_append_right _ _ H0).
    clear H0.
    apply propositional_sequent_cons.
      apply H.
      eapply propositional_sequent_head.
      eexact H1.
    eapply propositional_sequent_tail.
    eexact H1.
  Qed.


  Lemma prop_form_is_propositional : forall(f : lambda_formula V L),
    prop_form f -> propositional f.
  Proof.
    intros f H.
    apply rank_formula_prop_form.
    trivial.
  Qed.


  Lemma prop_sequent_is_propositional : forall(s : sequent V L),
    prop_sequent s -> propositional_sequent s.
  Proof.
    intros s.
    apply rank_prop_sequent.
  Qed.

End Propositional_formulas.

Implicit Arguments propositional [[V] [L]].
Implicit Arguments propositional_neg [V L].
Implicit Arguments propositional_and_left [V L].
Implicit Arguments propositional_and_right [V L].
Implicit Arguments propositional_tcc_modal [V L X].
Implicit Arguments propositional_sequent [[V] [L]].
Implicit Arguments propositional_sequent_tail [V L].
Implicit Arguments propositional_lambda_or [V L].
Implicit Arguments propositional_sequent_append_right [V L].
Implicit Arguments propositional_sequent_head [V L].
Implicit Arguments propositional_sequent_append [V L].
Implicit Arguments propositional_or_formula [V L].
Implicit Arguments prop_sequent_is_propositional [V L].
