(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: propositional_models.v,v 1.12 2013/04/10 11:17:16 tews Exp $
 *)


(** ** Models for the propositional fragment

      This module defines propositional models and proves utility
      results for the simplified [some_neg_dep] semantics of
      propositional sequents. This is the basis for soundness and
      completeness of the propositional fragment. 
 *)

Require Export some_neg_form propositional_rules.

Section Propositional_models.

  Variable V : Type.
  Variable L : modal_operators.


  (**************************************************************************)
  (** ***  Propositional models and validity  *)
  (**************************************************************************)

  Definition prop_model : Type := set V.

  Fixpoint is_prop_model(m : prop_model)(f : lambda_formula V L)
                              (prop_f : propositional f) : Prop :=
    match f return propositional f -> Prop with
      | lf_prop v => fun _ => m v
      | lf_neg f1 => fun(H : propositional (lf_neg f1)) =>
          not (is_prop_model m f1 (propositional_neg f1 H))
      | lf_and f1 f2 => fun(H : propositional (lf_and f1 f2)) =>
          is_prop_model m f1 (propositional_and_left f1 f2 H) /\
          is_prop_model m f2 (propositional_and_right f1 f2 H)
      | lf_modal op args => fun(H : propositional (lf_modal op args)) =>
          propositional_tcc_modal op args H
    end prop_f.

  Lemma is_prop_model_tcc_irr : 
    forall(m : prop_model)(f : lambda_formula V L)(p1 p2 : propositional f),
      is_prop_model m f p1 -> is_prop_model m f p2.
  Proof.
    intros m f p1 p2 H.
    induction f.
          trivial.
        simpl in *.
        intros H0.
        eapply IHf in H0.
        apply H.
        eexact H0.
      simpl in *.
      destruct H.
      split.
        eapply IHf1.
        eexact H.
      eapply IHf2.
      eexact H0.
    eapply propositional_tcc_modal.
    eexact p1.
  Qed.

  (** [nonempty_v] is in the wrong position here, but this way the
      lemma can be used directly in prop_model_sequent_interpretation.
   *)
  Lemma is_prop_model_false :
    forall(m : prop_model)(nonempty_v : V)
          (prop_false : propositional (lambda_false nonempty_v)),
      is_prop_model m (lambda_false nonempty_v) prop_false -> False.
  Proof.
    clear. 
    intros nonempty_v m prop_false H.
    unfold lambda_false in H.
    simpl in H.
    destruct H.
    contradiction.
  Qed.


  Definition prop_valid(f : lambda_formula V L)
                       (prop_f : propositional f) : Prop :=
    forall(m : prop_model), is_prop_model m f prop_f.

  Definition prop_valid_sequent(nonempty_v : V)(s : sequent V L)
                               (prop_s : propositional_sequent s) : Prop :=
    prop_valid (or_formula_of_sequent s nonempty_v)
               (propositional_or_formula nonempty_v s prop_s).


  (**************************************************************************)
  (** *** Simplified propositional sequent semantics  *)
  (**************************************************************************)

  Definition prop_sequent_interpretation(m : prop_model)(s : sequent V L)
                                 (prop_seq : propositional_sequent s) : Prop :=
    some_neg_dep propositional (is_prop_model m) s prop_seq.

  Lemma prop_sequent_interpretation_tcc_irr :
    forall(m : prop_model)(s : sequent V L)(ps1 ps2 : propositional_sequent s),
      prop_sequent_interpretation m s ps1 ->
        prop_sequent_interpretation m s ps2.
  Proof.
    intros m s ps1 ps2 H.
    eapply some_neg_dep_tcc_irr with (1 := is_prop_model_tcc_irr m).
    eexact H.
  Qed.

  Lemma prop_sequent_interpretation_empty :
    forall(m : prop_model)(prop : propositional_sequent []),
      prop_sequent_interpretation m [] prop -> False.
  Proof.
    intros m prop H.
    apply some_neg_dep_empty in H.
    trivial.
  Qed.

  Lemma prop_sequent_interpretation_singleton :
    forall(m : prop_model)(f : lambda_formula V L)
          (prop_s : propositional_sequent [f]),
      prop_sequent_interpretation m [f] prop_s <->
        is_prop_model m f (prop_s 0 (lt_0_Sn 0)).
  Proof.
    intros m f prop_s.
    split.
      intros H.
      apply some_neg_dep_singleton. 
        apply is_prop_model_tcc_irr.
      trivial.
    intros H.
    eapply some_neg_dep_singleton_rev.
      apply is_prop_model_tcc_irr.
    eexact H.
  Qed.

  Lemma prop_sequent_interpretation_nth_intro :
    forall(m : prop_model)(s : sequent V L)(n : nat)(n_less : n < length s)
          (prop_s : propositional_sequent s),
      is_prop_model m (nth s n n_less) (prop_s n n_less) ->
        prop_sequent_interpretation m s prop_s.
  Proof.
    intros m s n n_less prop_s H.
    eapply some_neg_dep_nth_intro.
      apply is_prop_model_tcc_irr.
    eexact H.
  Qed.

  Lemma prop_sequent_interpretation_cons_case_elim :
    forall(m : prop_model)(f : lambda_formula V L)(s : sequent V L)
          (prop_fs : propositional_sequent (f :: s)),
      prop_sequent_interpretation m (f :: s) prop_fs ->
        ((s = [] /\ is_prop_model m f 
                            (propositional_sequent_head _ _ prop_fs)) \/
         (s <> [] /\ 
            ~(~is_prop_model m f 
                            (propositional_sequent_head _ _ prop_fs) /\
              ~prop_sequent_interpretation m s 
                            (propositional_sequent_tail _ _ prop_fs)))).
  Proof.
    intros m f s prop_fs H.
    apply some_neg_dep_cons_case_elim in H.
      decompose [and or] H; clear H.
        left.
        split; trivial.
        eapply is_prop_model_tcc_irr.
        eexact H2.
      right.
      split; trivial.
      intros H3.
      destruct H3.
      apply H2; clear H2; split; intro H2.
        apply H; clear H H0.
        eapply is_prop_model_tcc_irr.
        eexact H2.
      apply H0; clear H H0.
      eapply prop_sequent_interpretation_tcc_irr.
      eexact H2.
    apply is_prop_model_tcc_irr.
  Qed.

  Lemma prop_sequent_interpretation_cons_cons_elim :
    forall(m : prop_model)(f1 f2 : lambda_formula V L)(s : sequent V L)
          (prop_s : propositional_sequent (f1 :: f2 :: s)),
      prop_sequent_interpretation m (f1 :: f2 :: s) prop_s ->
        ~(~is_prop_model m f1 (propositional_sequent_head _ _ prop_s) /\
          ~prop_sequent_interpretation m (f2 :: s)
                                    (propositional_sequent_tail _ _ prop_s)).
  Proof.
    intros m f1 f2 s prop_s H.
    apply prop_sequent_interpretation_cons_case_elim in H.
    decompose [and or] H; clear H.
      discriminate.
    trivial.
  Qed.

  Lemma prop_sequent_interpretation_length_case_intro :
    forall(m : prop_model)(s : sequent V L)(prop_s : propositional_sequent s),
      (length s = 1 -> prop_sequent_interpretation m s prop_s) ->
      (length s <> 1 -> ~~prop_sequent_interpretation m s prop_s) ->
         prop_sequent_interpretation m s prop_s.
  Proof.
    intros m s prop_s H H0.
    apply some_neg_dep_length_case_intro.
      clear H0.
      intros len.
      specialize (H len).
      generalize (prop_s 0 (nth_head_tcc s len)).
      destruct s.
        discriminate.
      destruct s.
        intros r.
        apply prop_sequent_interpretation_singleton in H.
        simpl.
        eapply is_prop_model_tcc_irr.
        eexact H.
      discriminate.
    trivial.
  Qed.

  Lemma prop_sequent_interpretation_cons_case_intro :
    forall(m : prop_model)(f : lambda_formula V L)(s : sequent V L)
          (prop_fs : propositional_sequent (f :: s)),
      (s = [] -> is_prop_model m f (propositional_sequent_head _ _ prop_fs))
        ->
      (s <> [] -> ~~prop_sequent_interpretation m (f :: s) prop_fs) -> 
        prop_sequent_interpretation m (f :: s) prop_fs.
  Proof.
    intros m f s prop_fs H H0.
    apply some_neg_dep_cons_case_intro.
        apply is_prop_model_tcc_irr.
      intros H1.
      eapply is_prop_model_tcc_irr.
      apply H.
      trivial.
    intros H1.
    apply H0.
    trivial.
  Qed.

  Lemma prop_sequent_interpretation_cons_cons_intro :
    forall(m : prop_model)(f1 f2 : lambda_formula V L)(s : sequent V L)
          (prop_fs : propositional_sequent (f1 :: f2 :: s)),
      ~~prop_sequent_interpretation m (f1 :: f2 :: s) prop_fs -> 
        prop_sequent_interpretation m (f1 :: f2 :: s) prop_fs.
  Proof.
    intros m f1 f2 s prop_fs H.
    apply prop_sequent_interpretation_cons_case_intro.
      discriminate.
    trivial.
  Qed.

  Lemma prop_sequent_interpretation_append :
    forall(m : prop_model)(sl sr : sequent V L)
          (prop_sl : propositional_sequent sl)
          (prop_sr : propositional_sequent sr),
      (prop_sequent_interpretation m sl prop_sl \/
         prop_sequent_interpretation m sr prop_sr) ->
      prop_sequent_interpretation m (sl ++ sr)
        (propositional_sequent_append sl sr prop_sl prop_sr).
  Proof.
    intros m sl sr prop_sl prop_sr H.
    destruct H.
      apply some_neg_dep_append_right.
        apply is_prop_model_tcc_irr.
      eapply prop_sequent_interpretation_tcc_irr.
      eexact H.
    apply some_neg_dep_append_left.
      apply is_prop_model_tcc_irr.
    eapply prop_sequent_interpretation_tcc_irr.
    eexact H.
  Qed.

  Lemma prop_sequent_interpretation_append_split :
    forall(m : prop_model)(sl sr : sequent V L)
          (prop_sl_sr : propositional_sequent (sl ++ sr)),
      prop_sequent_interpretation m (sl ++ sr) prop_sl_sr ->
        ~(~ (forall (prop_sl : propositional_sequent sl),
               prop_sequent_interpretation m sl prop_sl) /\
          ~ (forall (prop_sr : propositional_sequent sr),
               prop_sequent_interpretation m sr prop_sr)).
  Proof.
    intros m sl sr prop_sl_sr H H0.
    destruct H0.
    eapply some_neg_dep_append in H.
      apply H; clear H.
      split.
        intros H2.
        apply H0; clear H0 H1.
        intros prop_sl.
        eapply prop_sequent_interpretation_tcc_irr.
        eexact H2.
      intros H2.
      apply H1; clear H0 H1.
      intros prop_sr.
      eapply prop_sequent_interpretation_tcc_irr.
      eexact H2.
    apply is_prop_model_tcc_irr.
  Qed.


  (**************************************************************************)
  (** *** Simplified semantics for contexts  *)
  (**************************************************************************)

  Lemma prop_sequent_interpretation_add_context :
    forall(m : prop_model)(sl s sr : sequent V L)
          (prop_sl : propositional_sequent sl)
          (prop_s : propositional_sequent s)
          (prop_sr : propositional_sequent sr),
      (prop_sequent_interpretation m sl prop_sl \/
         prop_sequent_interpretation m s prop_s \/
         prop_sequent_interpretation m sr prop_sr) ->
      prop_sequent_interpretation m (add_context sl sr s)
        (propositional_add_context sl sr s prop_sl prop_sr prop_s).
  Proof.
    intros m sl s sr prop_sl prop_s prop_sr H.
    unfold add_context in *.
    eapply prop_sequent_interpretation_tcc_irr.
    apply prop_sequent_interpretation_append.
    decompose [or] H; clear H.
        left.
        eexact H0.
      right.
      apply prop_sequent_interpretation_append.
      left.
      eexact H1.
    right.
    apply prop_sequent_interpretation_append.
    right.
    eexact H1.
  Qed.

  Lemma prop_sequent_interpretation_add_context_split :
    forall(m : prop_model)(sl s sr : sequent V L)
          (prop_sl_s_sr : propositional_sequent (add_context sl sr s)),
      prop_sequent_interpretation m (add_context sl sr s) prop_sl_s_sr        
      ->
        ~(~(forall(prop_sl : propositional_sequent sl),
              prop_sequent_interpretation m sl prop_sl) /\
          ~(forall(prop_s : propositional_sequent s),
              prop_sequent_interpretation m s prop_s)   /\
          ~(forall(prop_sr : propositional_sequent sr),
              prop_sequent_interpretation m sr prop_sr)).
  Proof.
    intros m sl s sr prop_sl_s_sr H H0.
    decompose [and] H0; clear H0.
    unfold add_context in *.
    apply prop_sequent_interpretation_append_split in H.
    apply H; clear H; split; intro H.
      contradiction.
    specialize (H (propositional_sequent_append_right _ _ prop_sl_s_sr)).
    apply prop_sequent_interpretation_append_split in H.
    apply H; clear H; split; intro H; contradiction.
  Qed.


  (**************************************************************************)
  (** *** Equivalence for simplified semantics  *)
  (**************************************************************************)

  Lemma is_prop_model_or :
   forall(m : prop_model)(f1 f2 : lambda_formula V L)
         (prop_f1 : propositional f1)(prop_f2 : propositional f2),
     is_prop_model m (lambda_or f1 f2) 
                     (propositional_lambda_or _ _ prop_f1 prop_f2)
        <->
     ~(~is_prop_model m f1 prop_f1 /\ ~is_prop_model m f2 prop_f2).
  Proof.
    intros m f1 f2 prop_f1 prop_f2.
    simpl.
    split.
      intros H H0.
      destruct H0.
      apply H; clear H; split; intro H.
        eapply is_prop_model_tcc_irr in H.
        apply H0.
        eexact H.
      eapply is_prop_model_tcc_irr in H.
      apply H1.
      eexact H.
    intros H H0.
    destruct H0.
    apply H; clear H; split; intro H.
      eapply is_prop_model_tcc_irr in H.
      apply H0.
      eexact H.
    eapply is_prop_model_tcc_irr in H.
    apply H1.
    eexact H.
  Qed.

  Lemma prop_model_sequent_interpretation :
    forall(nonempty_v : V)(m : prop_model)(s : sequent V L)
          (prop_seq : propositional_sequent s),
      prop_sequent_interpretation m s prop_seq
        <->
      is_prop_model m (or_formula_of_sequent s nonempty_v) 
              (propositional_or_formula nonempty_v _ prop_seq).
  Proof.
    intros nonempty_v m s prop_seq.
    unfold prop_sequent_interpretation in *.
    eapply some_neg_dep_correct.
        apply is_prop_model_tcc_irr.
      apply is_prop_model_or.
    apply is_prop_model_false.
  Qed.

End Propositional_models.

Implicit Arguments is_prop_model [V L].
Implicit Arguments is_prop_model_tcc_irr [V L].
Implicit Arguments is_prop_model_false [V L].
Implicit Arguments prop_valid [V L].
Implicit Arguments prop_valid_sequent [V L].
Implicit Arguments prop_sequent_interpretation [V L].
Implicit Arguments prop_sequent_interpretation_tcc_irr [V L].
Implicit Arguments prop_sequent_interpretation_singleton [V L].
Implicit Arguments prop_sequent_interpretation_nth_intro [V L].
Implicit Arguments prop_model_sequent_interpretation [V L].
Implicit Arguments prop_sequent_interpretation_append [V L].
Implicit Arguments prop_sequent_interpretation_append_split [V L].
Implicit Arguments prop_sequent_interpretation_add_context [V L].
Implicit Arguments prop_sequent_interpretation_add_context_split [V L].
