(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: propositional_properties.v,v 1.23 2013/04/10 11:17:16 tews Exp $
 *)


(** ** Syntactic properties of propositional logic

      This module proves contraction, cut-elimination and
      admissibility of non-atomic axioms for the propositional part.
      These properties are needed in 5.6 for the base case. Finally,
      there is a proof for contraction/weakening under the cut rule. 
*)

Require Export propositional_rules generic_cut inversion.

Section Propositional_properties.

  Variable V : Type.
  Variable L : modal_operators.


  (***************************************************************************)
  (*** **  admissibility of non-atomic axioms  *)
  (***************************************************************************)

  Lemma G_non_atomic_axiom_head :
    forall(hyp : set (sequent V L))(f : lambda_formula V L)(s : sequent V L),
      sequent_multiset hyp ->
      propositional f ->
      propositional_sequent s ->
        provable (prop_G_set V L) hyp (f :: lf_neg f :: s).
  Proof.
    induction f.
          (* Case f = lf_prop v *)
          intros s H H0 H1.
          set (r := {| assumptions := []; 
                       conclusion := lf_prop v :: lf_neg (lf_prop v) :: s |}).
          assert (prop_G_set V L r).
            subst r.
            split.
              left.
              unfold is_ax_rule in *.
              simpl.
              split.
                trivial.
              exists 0, 1, v.
              apply dep_conj with (a := lt_0_Sn (S (length s))).
              assert (1 < length (lf_prop v :: lf_neg (lf_prop v) :: s)).
                simpl.
                apply lt_n_S.
                apply lt_0_Sn.
              apply dep_conj with (a := H2).
              auto.
            split.
              apply every_nth_empty.
            simpl.
            repeat apply rank_sequent_cons; trivial.
          exists (rule (prop_G_set V L) hyp r H2 dep_nil).
          trivial.
        (* Case f = lf_neg _ *)
        intros s H H0 H1.
        apply provable_with_neg_neg with (sl := [lf_neg f]).
            apply subset_refl.
          apply propositional_sequent_cons; trivial.
          apply propositional_sequent_cons; trivial.
        eapply provable_prop_G_hyp_list_reorder.
            trivial.
          apply list_reorder_move_append with (ll := [lf_neg f]).
        apply IHf; trivial.
      (* Case f = lf_and _ _ *)
      intros s H H0 H1.
      assert (H2 := propositional_and_left _ _ H0).
      assert (H3 := propositional_and_right _ _ H0).
      apply provable_with_neg_and with (sl := [lf_and f1 f2]).
          apply subset_refl.
        apply propositional_sequent_cons; trivial.
        apply propositional_sequent_cons; trivial.        
      apply provable_with_and with (sl := []).
            apply subset_refl.
          apply propositional_sequent_cons; trivial.
          apply propositional_sequent_cons; trivial.        
          apply propositional_sequent_cons; trivial.
        apply IHf1; trivial.
        apply rank_sequent_cons; trivial.
      eapply provable_prop_G_hyp_list_reorder; trivial.
        apply list_reorder_cons_head.
        apply list_reorder_move_append with (ll := [lf_neg f1]).
      apply IHf2; trivial.
      apply rank_sequent_cons; trivial.
    intros s H0 H1 H2.
    eapply propositional_tcc_modal.
    eexact H1.
  Qed.


  (***************************************************************************)
  (** ***  admissibility of contraction  *)
  (***************************************************************************)

  Definition head_contraction_closed(ss : set (sequent V L)) : Prop :=
    forall(f : lambda_formula V L)(s : sequent V L),
      ss (f :: f :: s) -> ss (f :: s).

  Lemma head_contraction_closed_empty :
    head_contraction_closed (empty_sequent_set V L).
  Proof.
    intros f s H.
    contradiction.
  Qed.


  Lemma G_n_contraction_head_ax :
    forall(hyp : set (sequent V L))
          (n d : nat)(f : lambda_formula V L)(s : sequent V L),
      simple_tautology (f :: f :: s) ->
      rank_sequent n (f :: f :: s) ->
        provable_at_depth (G_n_set V L n) hyp (S d) (f :: s).
  Proof.
    intros hyp n d f s H H0.
    apply provable_at_depth_with_rule with (assum := []).
      split.
        left.
        split.
          trivial.
        apply simple_tautology_contract_head.
        trivial.
      split.
        apply every_nth_empty.
      simpl.
      eapply rank_sequent_tail.
      eexact H0.
    apply every_nth_empty.
  Qed.

  Lemma G_n_contraction_and_assumptions_head :
    forall(hyp : set (sequent V L))(n d : nat)(f g1 g2 : lambda_formula V L)
          (s : sequent V L)(br : sequent_rule V L)
          (subproofs : dep_list (sequent V L) (proof (G_n_set V L n) hyp)
                                (map (add_context [f] s) (assumptions br))),
      conj_head_inversion_closed hyp ->
      (forall(s : sequent V L),
         provable_at_depth (G_n_set V L n) hyp d s ->
         forall(s_tl : sequent V L)(f : lambda_formula V L),
           s = f :: f :: s_tl ->
             provable_at_depth (G_n_set V L n) hyp d (f :: s_tl)) ->
      br = bare_and_rule g1 g2 ->
      [f] = conclusion br ->
      every_dep_nth
             (fun(s : sequent V L)(p : proof (G_n_set V L n) hyp s) => 
                 proof_depth p <= d)
             (map (add_context [f] s) (assumptions br))
             subproofs ->
        every_nth (provable_at_depth (G_n_set V L n) hyp d)
                  (map (add_context [] s) (assumptions br)).
  Proof.
    intros hyp n d f g1 g2 s br subproofs H H0 H1 H2 H3.
    subst br.
    unfold add_context in *.
    simpl in *.
    inversion H2; clear H2.
    subst f.
    remember [lf_and g1 g2 :: g1 :: s : sequent V L; 
              lf_and g1 g2 :: g2 :: s : sequent V L].
    destruct subproofs.
      discriminate.
    destruct subproofs.
      discriminate.
    inversion Heql.
    subst a a0.
    clear Heql H5.
    apply every_nth_cons.
      apply every_dep_nth_head in H3.
      assert (H4 := conj_left_head_Gn_admissible _ _ _ _ _ H _ d _ 
                         (eq_refl (lf_and g1 g2 :: g1 :: s)) H3).
      apply (H0 _ H4); trivial.
    apply every_nth_cons.
      apply every_dep_nth_tail in H3.
      apply every_dep_nth_head in H3.
      assert (H4 := conj_right_head_Gn_admissible _ _ _ _ _ H _ d _ 
                         (eq_refl (lf_and g1 g2 :: g2 :: s)) H3).
      apply (H0 _ H4); trivial.
    apply every_nth_empty.
  Qed.

  Lemma G_n_contraction_neg_and_assumptions_head :
    forall(hyp : set (sequent V L))(n d : nat)(f g1 g2 : lambda_formula V L)
          (s : sequent V L)(br : sequent_rule V L)
          (subproofs : dep_list (sequent V L) (proof (G_n_set V L n) hyp)
                                (map (add_context [f] s) (assumptions br))),
      sequent_multiset hyp ->
      disj_head_inversion_closed hyp ->
      (forall(s : sequent V L),
         provable_at_depth (G_n_set V L n) hyp d s ->
         forall(s_tl : sequent V L)(f : lambda_formula V L),
           s = f :: f :: s_tl ->
             provable_at_depth (G_n_set V L n) hyp d (f :: s_tl)) ->
      br = bare_neg_and_rule g1 g2 ->
      [f] = conclusion br ->
      every_dep_nth
            (fun(s : sequent V L)(p : proof (G_n_set V L n) hyp s) => 
                proof_depth p <= d)
            (map (add_context [f] s) (assumptions br)) subproofs ->
        every_nth (provable_at_depth (G_n_set V L n) hyp d)
          (map (add_context [] s) (assumptions br)).
  Proof.
    intros hyp n d f g1 g2 s br subproofs H H0 H1 H2 H3 H4.
    subst br.
    unfold add_context in *.
    simpl in *.
    inversion H3; clear H3.
    subst f.
    remember [lf_neg (lf_and g1 g2) :: (lf_neg g1) :: (lf_neg g2) :: s 
              : sequent V L].
    destruct subproofs.
      discriminate.
    inversion Heql.
    subst a.
    clear Heql H5.
    apply every_nth_cons.
      apply every_dep_nth_head in H4.
      apply H1 with (s := lf_neg g1 :: lf_neg g1 :: lf_neg g2 :: s).
        eapply provable_depth_G_n_hyp_list_reorder; trivial.
          apply list_reorder_rot_3.
        apply H1 with (s := lf_neg g2 :: lf_neg g2 :: 
                            lf_neg g1 :: lf_neg g1 :: s).
          eapply provable_depth_G_n_hyp_list_reorder; trivial.
            apply list_reorder_rot_3.
          apply neg_and_inv_head_Gn_depth_admissible; trivial.
          eapply provable_depth_G_n_hyp_list_reorder; trivial.
            apply list_reorder_rot_3.
          eapply provable_depth_G_n_hyp_list_reorder; trivial.
            eapply list_reorder_swap_head.
          exists p.
          trivial.
        trivial.
      trivial.
    apply every_nth_empty.
  Qed.

  Lemma G_n_contraction_neg_neg_assumptions_head :
    forall(hyp : set (sequent V L))(n d : nat)(f g : lambda_formula V L)
          (s : sequent V L)(br : sequent_rule V L)
          (subproofs : dep_list (sequent V L) (proof (G_n_set V L n) hyp)
                                (map (add_context [f] s) (assumptions br))),
      neg_head_inversion_closed hyp ->
      (forall(s : sequent V L),
         provable_at_depth (G_n_set V L n) hyp d s ->
         forall(s_tl : sequent V L)(f : lambda_formula V L),
           s = f :: f :: s_tl ->
             provable_at_depth (G_n_set V L n) hyp d (f :: s_tl)) ->
      br = bare_neg_neg_rule g ->
      [f] = conclusion br ->
      every_dep_nth
           (fun(s : sequent V L)(p : proof (G_n_set V L n) hyp s) => 
               proof_depth p <= d)
           (map (add_context [f] s) (assumptions br)) subproofs ->
        every_nth (provable_at_depth (G_n_set V L n) hyp d)
          (map (add_context [] s) (assumptions br)).
  Proof.
    intros hyp n d f g s br subproofs H H0 H1 H2 H3.
    subst br.
    unfold add_context in *.
    simpl in *.
    inversion H2; clear H2.
    subst f.
    remember [lf_neg (lf_neg g) :: g :: s : sequent V L].
    destruct subproofs.
      discriminate.
    inversion Heql.
    subst a.
    clear Heql H4.
    apply every_nth_cons.
      apply every_dep_nth_head in H3.
      apply H0 with (s := g :: g :: s).
        apply neg_inv_head_Gn_depth_admissible; trivial.
        exists p.
        trivial.
      trivial.
    apply every_nth_empty.
  Qed.

  Lemma G_n_contraction_head :
    forall(hyp : set (sequent V L))(n : nat)
          (f : lambda_formula V L)(s : sequent V L),
      sequent_multiset hyp ->
      head_inversion_closed hyp ->
      head_contraction_closed hyp ->
      provable (G_n_set V L n) hyp (f :: f :: s) ->
        provable (G_n_set V L n) hyp (f :: s).
  Proof.
    intros hyp n f s H H0 H1 H2.
    destruct H2.
    eapply provable_at_depth_provable.
    apply (proof_depth_sequent_ind (G_n_set V L n) hyp
            (fun d s => forall(s_tl : sequent V L)(f : lambda_formula V L),
               s = f :: f :: s_tl ->
                 provable_at_depth (G_n_set V L n) hyp
                    d (f :: s_tl))) with (p := x).
      clear - H H0 H1. 
      intros d H2 s H3 s_tl f H4.
      destruct H3 as [p].
      destruct p.
        subst gamma.
        specialize (H1 _ _ h).
        exists (assume (G_n_set V L n) hyp (f :: s_tl) H1).
        rewrite proof_depth_assume.
        omega.
      apply proof_depth_rule_le_inv in H3.
      destruct (decompose_G_n_set_coarsly _ _ g).
        rewrite H4 in *.
        clear - H5.
        decompose [and] H5; clear H5.
        apply G_n_contraction_head_ax; trivial.
      clear g.
      decompose [ex and] H5; clear H5.
      clear H6.
      rename x into br, x0 into sl, x1 into sr, x2 into cbr.
      subst r.
      simpl in *.
      unfold add_context in H4.
      destruct sl.
        rewrite <- H11 in H4.
        simpl in *.
        inversion H4; clear H4.
        subst cbr sr.
        change (f :: s_tl) with ([f] ++ s_tl).
        rewrite H11.
        apply provable_at_depth_with_rule with 
                 (assum := (map (add_context [] s_tl) (assumptions br))).
          apply smaller_context_G_n_set with (f := f); trivial.
          clear - H11 H7.
          rewrite <- H11 in *.
          unfold add_context in *.
          simpl in *.
          trivial.
        clear H7.
        assert (every_nth 
                  (provable_at_depth (G_n_set V L n) hyp d)
                  (map (add_context [f] s_tl) (assumptions br))).
          clear - H d H3.
          apply every_nth_map.
          intros i i_less.
          unfold add_context.
          simpl.
          eapply provable_depth_G_n_hyp_list_reorder.
              trivial.
            apply list_reorder_symm.
            apply list_reorder_move_append.
          assert (i < length (map (add_context [] (f :: s_tl)) 
                                  (assumptions br))).
            rewrite map_length in *.
            trivial.
          specialize (H3 i H0).
          simpl in *.
          revert H3.
          generalize (dep_nth (map (add_context [] (f :: s_tl)) 
                                   (assumptions br)) d0 i H0).
          clear. 
          rewrite nth_map.
          unfold add_context in *.
          simpl.
          erewrite nth_tcc_irr.
          intros p H3.
          exists p.
          trivial.
        clear d0 H3.
        apply every_nth_exists in H4.
        destruct H4.
        decompose [ex or] H9; clear H9.
            eapply G_n_contraction_and_assumptions_head; eauto.
            apply H0.
          eapply G_n_contraction_neg_and_assumptions_head; eauto.
          apply H0.
        eapply G_n_contraction_neg_neg_assumptions_head; eauto.
        apply H0.
      simpl in *.
      inversion H4.
      subst l.
      clear H4.
      apply provable_at_depth_with_rule with 
                 (assum := (map (add_context sl sr) (assumptions br))).
        eapply smaller_context_G_n_set; eauto.
      clear H7.
      destruct sl.
        rewrite <- H11 in *.
        simpl in *.
        inversion H8; clear H8.
        subst cbr sr.
        decompose [ex or] H9; clear H9.
            eapply G_n_contraction_and_assumptions_head; eauto.
            apply H0.
          eapply G_n_contraction_neg_and_assumptions_head; eauto.
          apply H0.
        eapply G_n_contraction_neg_neg_assumptions_head; eauto.
        apply H0.
      clear cbr H11.
      simpl in *.
      inversion H8; clear H8.
      subst l s_tl.
      apply every_nth_map.
      intros i i_less.
      assert (i < length (map (add_context (f :: f :: sl) sr)
                              (assumptions br))).
        rewrite map_length.
        trivial.
      specialize (H3 i H4).
      simpl in *.
      unfold add_context.
      simpl.
      apply H2 with (s := f :: f :: sl ++ nth (assumptions br) i i_less ++ sr).
        revert H3.
        generalize (dep_nth (map (add_context (f :: f :: sl) sr)
                                 (assumptions br)) d0 i H4).
        clear. 
        rewrite nth_map.
        unfold add_context in *.
        simpl.
        erewrite nth_tcc_irr.
        intros p H0.
        exists p.
        trivial.
      trivial.
    trivial.
  Qed.

  Lemma prop_contraction_head :
    forall(f : lambda_formula V L)(s : sequent V L),
      provable (prop_G_set V L) (empty_sequent_set V L) (f :: f :: s) ->
        provable (prop_G_set V L) (empty_sequent_set V L) (f :: s).
  Proof.
    intros f s H.
    apply G_n_contraction_head; trivial.
        apply sequent_multiset_empty.
      apply head_inversion_closed_empty.
    apply head_contraction_closed_empty.
  Qed.


  (***************************************************************************)
  (** ***  contraction on lists  *)
  (***************************************************************************)

  Lemma G_n_hyp_list_contraction_left :
    forall(n : nat)(hyp : set (sequent V L))(r s : sequent V L),
      sequent_multiset hyp ->
      head_inversion_closed hyp ->
      head_contraction_closed hyp ->
      provable (G_n_set V L n) hyp (r ++ r ++ s) ->
        provable (G_n_set V L n) hyp (r ++ s).
  Proof.
    intros n hyp r s H H0 H1 H2.
    revert s H2.
    induction r.
      trivial.
    simpl.
    intros s H2.
    eapply provable_G_n_hyp_list_reorder; trivial.
      apply list_reorder_symm.
      apply list_reorder_cons_parts.
      apply list_reorder_refl.
    apply IHr.
    eapply provable_G_n_hyp_list_reorder; trivial.
      rewrite app_assoc.
      apply (list_reorder_insert [] (r ++ r ++ s) (r ++ r) s a a).
        rewrite app_assoc_reverse.
        apply list_reorder_refl.
      trivial.
    simpl.
    apply G_n_contraction_head; trivial.
    eapply provable_G_n_hyp_list_reorder; trivial.
      apply (list_reorder_insert (a :: r) (r ++ s) [] (a :: r ++ r ++ s) a a).
        apply list_reorder_refl.
      trivial.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  admissibility of cut  *)
  (***************************************************************************)

  Lemma G_n_cut_elim_head_ax :
    forall(n : nat)(hyp : set (sequent V L))
          (f : lambda_formula V L)(q r : sequent V L),
      rank_sequent_set n hyp ->
      sequent_multiset hyp ->
      bounded_weakening_closed n hyp ->
      neg_head_inversion_closed hyp ->
      rank_sequent n q ->
      simple_tautology (f :: q) ->
      provable (G_n_set V L n) hyp ((lf_neg f) :: r) ->
        provable (G_n_set V L n) hyp (q ++ r).
  Proof.
    intros n hyp f q r H H0 H1 H2 H3 H4 H5.
    apply simple_tautology_cons_destruct in  H4.
    decompose [ex and or]  H4; clear  H4.
        apply provable_with_rule with (assum := []).
          split.
            left.
            split.
              trivial.
            simpl.
            apply simple_tautology_append_right.
            trivial.
          split.
            apply every_nth_empty.
          simpl.
          apply rank_sequent_append; trivial.
          apply provable_rank_rules_hyp_has_rank_n in H5.
            apply rank_sequent_tail in H5.
            trivial.
          trivial.
        apply every_nth_empty.
      rename x into qt.
      apply provable_G_n_hyp_list_reorder 
             with (s1 := ((lf_neg f) ::  r) ++ qt); trivial.
        simpl.
        eapply list_reorder_trans.
          apply list_reorder_move_append.
        eapply list_reorder_trans.
          apply list_reorder_append_swap.
        apply list_reorder_append_right.
        apply list_reorder_symm.
        trivial.
      apply list_weakening_admissible_G_n_hyp; trivial.
        apply provable_rank_rules_hyp_has_rank_n in H5; trivial.
      eapply rank_sequent_tail.
      eapply rank_sequent_list_reorder.
        apply list_reorder_symm.
        eexact H7.
      trivial.
    rename x into f', x0 into qt.
    subst f.
    apply provable_G_n_hyp_list_reorder with (s1 := (f' ::  r) ++ qt); trivial.
      simpl.
      eapply list_reorder_trans.
        apply list_reorder_move_append.
      eapply list_reorder_trans.
        apply list_reorder_append_swap.
      apply list_reorder_append_right.
      apply list_reorder_symm.
      trivial.
    apply list_weakening_admissible_G_n_hyp; trivial.
        apply provable_rank_rules_hyp_has_rank_n in H5; trivial.
        apply rank_sequent_cons.
          apply rank_sequent_head in H5.
          trivial.
        apply rank_sequent_tail in H5.
        trivial.
      eapply rank_sequent_tail.
      eapply rank_sequent_list_reorder.
        apply list_reorder_symm.
        eexact H7.
      trivial.
    apply neg_inv_head_Gn_hyp_admissible; trivial.
  Qed.

  Lemma G_n_cut_elim_head_and_inside :
    forall(n m d : nat)(hyp : set (sequent V L))
          (f g1 g2 : lambda_formula V L)(r q : sequent V L)
          (brule : sequent_rule V L),
      sequent_multiset hyp ->
      head_inversion_closed hyp ->
      head_contraction_closed hyp ->
      (forall(f : lambda_formula V L)(q r : sequent V L),
         formula_measure f < m ->
         provable (G_n_set V L n) hyp (f :: q) ->
         provable (G_n_set V L n) hyp (lf_neg f :: r) ->
           provable (G_n_set V L n) hyp (q ++ r)) ->
      formula_measure f < S m ->
      provable (G_n_set V L n) hyp (lf_neg f :: r) ->
      every_nth (provable (G_n_set V L n) hyp)
                (map (add_context [] q) (assumptions brule)) ->
      brule = bare_and_rule g1 g2 ->
      [f] = conclusion brule ->
        provable (G_n_set V L n) hyp (q ++ r).
  Proof.
    intros n m d hyp f g1 g2 r q brule H H0 H1 H2 H3 H4 H5 H6 H7.
    unfold bare_and_rule, add_context in *.
    subst brule.
    simpl in *.
    inversion H7; clear H7.
    subst f.
    apply neg_and_inv_head_G_n_hyp_admissible in H4.
      apply G_n_hyp_list_contraction_left; trivial.
      apply H2 with (f := g2); trivial.
          clear - H3.
          rewrite formula_measure_char in H3.
          omega.
        apply every_nth_tail in H5.
        apply every_nth_head in H5.
        trivial.
      eapply provable_G_n_hyp_list_reorder.
          trivial.
        apply list_reorder_symm.
        apply list_reorder_cons_parts.
        apply list_reorder_refl.
      apply H2 with (f := g1); trivial.
        clear - H3.
        rewrite formula_measure_char in H3.
        omega.
      apply every_nth_head in H5.
      trivial.
    apply H0.
  Qed.

  Lemma G_n_cut_elim_head_neg_and_inside :
    forall(n m : nat)(hyp : set (sequent V L))
          (f g1 g2 : lambda_formula V L)(r q : sequent V L)
          (brule : sequent_rule V L),
      sequent_multiset hyp ->
      head_inversion_closed hyp ->
      (forall(f : lambda_formula V L)(q r : sequent V L),
         formula_measure f < m ->
         provable (G_n_set V L n) hyp (f :: q) ->
         provable (G_n_set V L n) hyp (lf_neg f :: r) ->
           provable (G_n_set V L n) hyp (q ++ r)) ->
      formula_measure f < S m ->
      provable (G_n_set V L n) hyp (f :: q) ->
      provable (G_n_set V L n) hyp (lf_neg f :: r) ->
      brule = bare_neg_and_rule g1 g2 ->
      [f] = conclusion brule ->
        provable (G_n_set V L n) hyp (q ++ r).
  Proof.
    intros n m hyp f g1 g2 r q brule H H0 H1 H2 H3 H4 H5 H6.
    unfold bare_neg_and_rule, add_context in *.
    subst brule.
    simpl in *.
    inversion H6; clear H6.
    subst f.
    apply neg_inv_head_Gn_hyp_admissible in H4.
      eapply provable_G_n_hyp_list_reorder; trivial.
        apply list_reorder_append_swap.
      apply H1 with (f := lf_and g1 g2); trivial.
      clear - H2.
      rewrite formula_measure_char in H2.
      omega.
    apply H0.
  Qed.


  Lemma G_n_cut_elim_head_neg_neg_inside :
    forall(n m : nat)(hyp : set (sequent V L))
          (f : lambda_formula V L)(r q : sequent V L),
      sequent_multiset hyp ->
      head_inversion_closed hyp ->
      (forall(f : lambda_formula V L)(q r : sequent V L),
        formula_measure f < m ->
        provable (G_n_set V L n) hyp (f :: q) ->
        provable (G_n_set V L n) hyp (lf_neg f :: r) ->
          provable (G_n_set V L n) hyp (q ++ r)) ->
      formula_measure (lf_neg f) < S m ->
      provable (G_n_set V L n) hyp (lf_neg f :: q) ->
      provable (G_n_set V L n) hyp (lf_neg (lf_neg f) :: r) ->
        provable (G_n_set V L n) hyp (q ++ r).
  Proof.
    intros n m hyp f r q H H0 H1 H2 H3 H4.
    apply neg_inv_head_Gn_hyp_admissible in H4.
      eapply provable_G_n_hyp_list_reorder; trivial.
        apply list_reorder_append_swap.
      apply H1 with (f := f); trivial.
      rewrite formula_measure_char in H2.
      apply lt_S_n in H2.
      trivial.
    apply H0.
  Qed.


  Lemma G_n_cut_elim_head_and_outside :
    forall(n d : nat)(hyp : set (sequent V L))
          (f1 f2 g1 g2 : lambda_formula V L)(sl sr r : sequent V L)
          (brule : sequent_rule V L),
      (forall(q : sequent V L),
         provable_at_depth (G_n_set V L n) hyp d (f1 :: q) ->
           provable (G_n_set V L n) hyp (q ++ r)) ->
      provable (G_n_set V L n) hyp (f2 :: r) ->
      every_nth (provable_at_depth (G_n_set V L n) hyp d)
                (map (add_context (f1 :: sl) sr) (assumptions brule)) ->
      brule = bare_and_rule g1 g2 ->
        every_nth (provable (G_n_set V L n) hyp)
                  (map (add_context sl (sr ++ r)) (assumptions brule)).
  Proof.
    intros n d hyp f1 f2 g1 g2 sl sr r brule H H0 H1 H2.
    subst brule.
    unfold bare_and_rule, add_context in *.
    simpl in *.
    apply every_nth_cons.
      apply every_nth_head in H1.
      specialize (H _ H1).
      rewrite app_assoc_reverse in H.
      trivial.
    apply every_nth_cons.
      apply every_nth_tail in H1.
      apply every_nth_head in H1.
      specialize (H _ H1).
      rewrite app_assoc_reverse in H.
      trivial.
    apply every_nth_empty.
  Qed.

  Lemma G_n_cut_elim_head_neg_and_outside :
    forall(n d : nat)(hyp : set (sequent V L))
          (f1 f2 g1 g2 : lambda_formula V L)(sl sr r : sequent V L)
          (brule : sequent_rule V L),
      (forall(q : sequent V L),
         provable_at_depth (G_n_set V L n) hyp d (f1 :: q) ->
           provable (G_n_set V L n) hyp (q ++ r)) ->
      provable (G_n_set V L n) hyp (f2 :: r) ->
      every_nth (provable_at_depth (G_n_set V L n) hyp d)
                (map (add_context (f1 :: sl) sr) (assumptions brule)) ->
      brule = bare_neg_and_rule g1 g2 ->
        every_nth (provable (G_n_set V L n) hyp)
                  (map (add_context sl (sr ++ r)) (assumptions brule)).
  Proof.
    intros n d hyp f1 f2 g1 g2 sl sr r brule H H0 H1 H2.
    subst brule.
    unfold bare_neg_and_rule, add_context in *.
    simpl in *.
    apply every_nth_head in H1.
    specialize (H _ H1).
    rewrite app_assoc_reverse in H.
    simpl in H.
    apply every_nth_cons.
      trivial.
    apply every_nth_empty.
  Qed.

  Lemma G_n_cut_elim_head_neg_neg_outside :
    forall(n d : nat)(hyp : set (sequent V L))
          (f1 f2 g : lambda_formula V L)(sl sr r : sequent V L)
          (brule : sequent_rule V L),
      (forall(q : sequent V L),
         provable_at_depth (G_n_set V L n) hyp d (f1 :: q) ->
           provable (G_n_set V L n) hyp (q ++ r)) ->
      provable (G_n_set V L n) hyp (f2 :: r) ->
      every_nth (provable_at_depth (G_n_set V L n) hyp d)
                (map (add_context (f1 :: sl) sr) (assumptions brule)) ->
      brule = bare_neg_neg_rule g ->
        every_nth (provable (G_n_set V L n) hyp)
                  (map (add_context sl (sr ++ r)) (assumptions brule)).
  Proof.
    intros n d hyp f1 f2 g sl sr r brule H H0 H1 H2.
    subst brule.
    unfold bare_neg_neg_rule, add_context in *.
    simpl in *.
    apply every_nth_head in H1.
    specialize (H _ H1).
    rewrite app_assoc_reverse in H.
    simpl in H.
    apply every_nth_cons.
      trivial.
    apply every_nth_empty.
  Qed.


  Lemma prop_G_cut_elim_head_ind :
    forall(n : nat)(f : lambda_formula V L)(q r : sequent V L),
      formula_measure f < n ->
      provable (prop_G_set V L) (empty_sequent_set V L) (f :: q) ->
      provable (prop_G_set V L) (empty_sequent_set V L) ((lf_neg f) :: r) ->
        provable (prop_G_set V L) (empty_sequent_set V L) (q ++ r).
  Proof.
    induction n.
      intros f q r H.
      omega.
    intros f q r H H0 H1.
    destruct H0 as [pf].
    clear H0.
    apply (proof_depth_sequent_ind (prop_G_set V L) 
                                          (empty_sequent_set V L)
             (fun _ fq => forall(q r : sequent V L),
                 fq = f :: q ->
                 provable (prop_G_set V L) (empty_sequent_set V L) 
                          ((lf_neg f) :: r) ->
                   provable (prop_G_set V L) (empty_sequent_set V L) (q ++ r)))
           with (s := f :: q); 
           trivial.
    clear - IHn H.
    intros d H0 fq H1 q r H2 H3.
    subst fq.
    assert (forall(q : sequent V L),
              provable_at_depth (prop_G_set V L) (empty_sequent_set V L) d 
                                (f :: q) ->
                provable (prop_G_set V L) (empty_sequent_set V L) (q ++ r)).
      clear - H0 H3.
      eauto.
    clear H0.
    apply provable_at_depth_destruct in H1.
    decompose [ex and or] H1; clear H1.
      contradiction.
    rename x into rule.
    decompose [or] (decompose_G_n_set_coarsly _ _ H5).
      rewrite <- H0 in *.
      apply G_n_cut_elim_head_ax with (f := f); trivial.
                apply rank_sequent_set_empty.
              apply sequent_multiset_empty.
            apply bounded_weakening_closed_empty.
          apply head_inversion_closed_empty.
        apply provable_rank_rules_has_rank_n in H4.
        apply rank_sequent_tail in H4.
        trivial.
      apply H1.
    decompose [ex and] H1; clear H1.
    rename x into brule, x0 into sl, x1 into sr.
    clear H5 H6.
    subst rule.
    unfold rule_add_context in *.
    simpl in *.
    unfold add_context in H0, H8.
    destruct sl.
      clear H2 H8.
      rewrite <- H12 in H0.
      simpl in H0.
      inversion H0; clear H0.
      subst x2 sr.
      decompose [ex or] H10; clear H10.
          eapply G_n_cut_elim_head_and_inside; eauto.
                apply sequent_multiset_empty.
              apply head_inversion_closed_empty.
            apply head_contraction_closed_empty.
          eapply every_nth_mono in H7.
            eexact H7.
          clear. 
          intros s H.
          apply provable_at_depth_provable in H.
          trivial.
        eapply G_n_cut_elim_head_neg_and_inside; eauto.
          apply sequent_multiset_empty.
        apply head_inversion_closed_empty.
      clear H7.
      unfold bare_neg_neg_rule, add_context in *.
      subst brule.
      simpl in *.
      inversion H12; clear H12.
      subst f.
      eapply G_n_cut_elim_head_neg_neg_inside; eauto.
        apply sequent_multiset_empty.
      apply head_inversion_closed_empty.
    clear H4 x2 H12.
    simpl in H0, H8.
    inversion H0; clear H0.
    subst l.
    repeat rewrite app_assoc_reverse.
    apply provable_with_rule with 
             (s := add_context sl (sr ++ r) (conclusion brule))
             (assum := map (add_context sl (sr ++ r)) (assumptions brule)).
      apply sequent_other_context_G_n_set 
                          with (sl1 := f :: sl)(sr1 := sr); auto.
        clear. 
        intros H.
        apply rank_sequent_tail in H.
        trivial.
      clear - H3.
      intros H.
      apply rank_sequent_append.
        trivial.
      apply provable_rank_rules_has_rank_n in H3.
      apply rank_sequent_tail in H3.
      trivial.
    clear IHn q H5 H8.
    decompose [ex or] H10; clear H10.
        eapply G_n_cut_elim_head_and_outside; eauto.
      eapply G_n_cut_elim_head_neg_and_outside; eauto.
    eapply G_n_cut_elim_head_neg_neg_outside; eauto.
  Qed.

  Lemma prop_G_cut_elim_head :
    forall(f : lambda_formula V L)(q r : sequent V L),
      provable (prop_G_set V L) (empty_sequent_set V L) (f :: q) ->
      provable (prop_G_set V L) (empty_sequent_set V L) ((lf_neg f) :: r) ->
        provable (prop_G_set V L) (empty_sequent_set V L) (q ++ r).
  Proof.
    intros f q r H H0.
    eapply prop_G_cut_elim_head_ind; eauto.
  Qed.

  Lemma admissible_bounded_cut_prop_G :
    admissible_rule_set (prop_G_set V L) (empty_sequent_set V L)
                        (bounded_cut_rules V L 1).
  Proof.
    apply cut_admissibile_from_head_elim.
      apply G_multiset.
    apply prop_G_cut_elim_head.
  Qed.

  Lemma provable_GC_1_G_1 : forall(s : sequent V L),
    provable (GC_n_set V L 1) (empty_sequent_set V L) s ->
      provable (G_n_set V L 1) (empty_sequent_set V L) s.
  Proof.
    intros s H.
    rewrite admissible_prop with (rs := bounded_cut_rules V L 1).
      eapply proof_set_equal_rules.
        apply GC_n_as_G_C_union.
      trivial.
    apply admissible_bounded_cut_prop_G.
  Qed.

  Lemma provable_GRC_1_GR_1 : 
    forall(rules : set (sequent_rule V L))(s : sequent V L),
      one_step_rule_set rules ->
      provable (GRC_n_set rules 1) (empty_sequent_set V L) s ->
        provable (GR_n_set rules 1) (empty_sequent_set V L) s.
  Proof.
    intros rules s H H0.
    eapply proof_set_equal_rules.
      apply set_equal_symm.
      apply GR_1_is_G_prop.
      trivial.
    apply provable_GC_1_G_1.
    eapply proof_set_equal_rules.
      apply GRC_1_is_GC_1.
      eexact H.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Contraction / Weakening with cut *)
  (***************************************************************************)

  Lemma propositional_cut_contraction_weakening :
    forall(hyp : set (sequent V L))(os cs : sequent V L),
      sequent_multiset hyp ->
      rank_sequent_set 1 hyp ->
      incl os cs ->
      os <> [] ->
      propositional_sequent cs ->
      provable (prop_GC_set V L) hyp os ->
        provable (prop_GC_set V L) hyp cs.
  Proof.
    intros hyp os cs H H0 H1 H2 H3 H4.
    apply provable_with_cut with (gl := [])(gr := [])(dl := [])
                            (f := or_formula_of_ne_sequent os H2); trivial.
          apply subset_refl.
        apply rank_formula_or_formula_of_ne_sequent.
        eapply provable_rank_rules_hyp_has_rank_n; eauto.
      apply provable_or_formula_of_ne_sequent; trivial.
      apply G_n_subset_GC_n.
    apply provable_sequent_axiom; trivial.
          apply GC_multiset.
        apply G_n_subset_GC_n.
      eapply provable_rank_rules_hyp_has_rank_n.
        eexact H0.
      eexact H4.
    intros f s H5 H6.
    eapply proof_mono_rules.
      apply prop_G_subset_prop_GC.
    apply G_non_atomic_axiom_head; trivial.
  Qed.

End Propositional_properties.

Implicit Arguments head_contraction_closed [V L].
