(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: propositional_rules.v,v 1.22 2013/04/10 11:17:16 tews Exp $
 *)


(** ** Propositional rules and rule sets 

      This module proves some utility results for propositional rules
      and defines the propositional rule sets G and GC.
 *)

Require Export rule_sets.

Section Propositional_rules.

  Variable V : Type.
  Variable L : modal_operators.


  (***************************************************************************)
  (** *** Results for propositional rules *)
  (***************************************************************************)

  Lemma propositional_add_context :
    forall(sl sr s : sequent V L),
      propositional_sequent sl -> 
      propositional_sequent sr ->
      propositional_sequent s ->
        propositional_sequent (add_context sl sr s).
  Proof.
    clear. 
    unfold add_context.
    intros sl sr s H H0 H1.
    apply propositional_sequent_append; auto.
    apply propositional_sequent_append; auto.
  Qed.

  Lemma propositional_add_context_propositional :
    forall(sl sr s : sequent V L),
      propositional_sequent (add_context sl sr s) ->
        propositional_sequent s.
  Proof.
    clear. 
    unfold add_context in *.
    intros sl sr s H.
    eapply propositional_sequent_append_left.
    eapply propositional_sequent_append_right.
    eexact H.
  Qed.

  Lemma propositional_add_context_left :
    forall(sl sr s : sequent V L),
      propositional_sequent (add_context sl sr s) ->
        propositional_sequent sl.
  Proof.
    clear. 
    unfold add_context in *.
    intros sl sr s H.
    eapply propositional_sequent_append_left.
    eexact H.
  Qed.

  Lemma propositional_add_context_right :
    forall(sl sr s : sequent V L),
      propositional_sequent (add_context sl sr s) ->
        propositional_sequent sr.
  Proof.
    clear. 
    unfold add_context in *.
    intros sl sr s H.
    eapply propositional_sequent_append_right.
    eapply propositional_sequent_append_right.
    eexact H.
  Qed.

  Definition propositional_rule(r : sequent_rule V L) : Prop :=
    rule_has_rank 1 r.

  Lemma prop_sequent_rule_ass_tcc :
    forall(r : sequent_rule V L)(n : nat)(n_len : n < length (assumptions r)),
      propositional_rule r -> 
        propositional_sequent (nth (assumptions r) n n_len).
  Proof.
    intros r n n_len H.
    destruct H.
    apply H.
  Qed.

  Lemma prop_sequent_rule_concl_tcc : forall(r : sequent_rule V L),
    propositional_rule r -> propositional_sequent (conclusion r).
  Proof.
    intros r H.
    destruct H.
    trivial.
  Qed.

  Lemma propositional_rule_add_context_bare_rule :
    forall(r : sequent_rule V L)(sl sr : sequent V L),
      propositional_rule (rule_add_context sl sr r) ->
        propositional_rule r.
  Proof.
    unfold propositional_rule, rule_add_context, rule_has_rank in *.
    simpl.
    intros r sl sr H.
    destruct H.
    split.
      intros n n_len.
      assert (n < length (map (add_context sl sr) (assumptions r))).
        rewrite map_length.
        trivial.
      specialize (H n H1).
      rewrite nth_map in H.
      eapply propositional_add_context_propositional.
      erewrite nth_tcc_irr.
      eexact H.
    eapply propositional_add_context_propositional.
    eexact H0.
  Qed.
   

  Definition propositional_sequent_set(ss : sequent V L -> Prop) : Prop :=
    forall(s : sequent V L), ss s -> propositional_sequent s.

  Lemma propositional_empty_sequent_set : 
    propositional_sequent_set (empty_sequent_set V L).
  Proof.
    clear. 
    unfold propositional_sequent_set in *.
    intros s H.
    contradiction.
  Qed.


  Definition propositional_rule_set(rules : sequent_rule V L -> Prop) : Prop :=
    forall(r : sequent_rule V L), rules r -> propositional_rule r.


  (**************************************************************************)
  (** ***  Propositional G rule set  *)
  (**************************************************************************)

  Definition prop_G_set(r : sequent_rule V L) : Prop :=
    G_n_set V L 1 r.

  Lemma provable_depth_prop_list_reorder : 
    forall(d : nat)(s1 s2 : sequent V L),
      list_reorder s1 s2 ->
      provable_at_depth prop_G_set (empty_sequent_set V L) d s1 ->
        provable_at_depth prop_G_set (empty_sequent_set V L) d s2.
  Proof.
    intros d s1 s2 H H0.
    eapply provable_depth_G_n_hyp_list_reorder.
        apply sequent_multiset_empty.
      eexact H.
    trivial.
  Qed.

  Lemma provable_prop_G_hyp_list_reorder : 
    forall(hyp : set (sequent V L))(s1 s2 : sequent V L),
      sequent_multiset hyp ->
      list_reorder s1 s2 ->
      provable prop_G_set hyp s1 ->
        provable prop_G_set hyp s2.
  Proof.
    intros hyp s1 s2 H H0 H1.
    eapply provable_G_n_hyp_list_reorder.
        trivial.
      eexact H0.
    trivial.
  Qed.

  (* not needed
   * Lemma provable_prop_G_list_reorder : 
   *   forall(s1 s2 : sequent V L),
   *     list_reorder s1 s2 ->
   *     provable prop_G_set (empty_sequent_set V L) s1 ->
   *       provable prop_G_set (empty_sequent_set V L) s2.
   * Proof.
   *   intros s1 s2 H H0.
   *   eapply provable_prop_G_hyp_list_reorder.
   *       apply sequent_multiset_empty.
   *     eexact H.
   *   trivial.
   * Qed.
   *)


  (***************************************************************************)
  (** **** Subset lemmas for G *)
  (***************************************************************************)

  Lemma prop_G_subset_Gn : forall(n : nat),
    1 <= n -> subset prop_G_set (G_n_set V L n).
  Proof.
    intros n H.
    apply rank_rules_subset_rank.
    trivial.
  Qed.

  Lemma prop_G_subset_GR : 
    forall(rules : set (sequent_rule V L))(n : nat),
      1 <= n ->
        subset prop_G_set (GR_n_set rules n).
  Proof.
    intros rules n H.
    eapply subset_trans.
      apply prop_G_subset_Gn.
      eexact H.
    apply G_n_subset_GR_n.
  Qed.

  Lemma prop_G_subset_GRC : 
    forall(rules : set (sequent_rule V L))(n : nat),
      1 <= n ->
        subset prop_G_set (GRC_n_set rules n).
  Proof.
    intros rules n H.
    eapply subset_trans.
      apply rank_rules_subset_rank.
      eexact H.
    apply G_n_subset_GRC_n.
  Qed.

  Lemma propositional_prop_G_set : propositional_rule_set prop_G_set.
  Proof.
    intros r H.
    destruct H.
    trivial.
  Qed.

  Lemma G_set_prop_set : forall(r : sequent_rule V L), 
    G_set V L r -> propositional_sequent (conclusion r) ->
      prop_G_set r.
  Proof.
    intros r H H0.
    unfold prop_G_set in *.
    split; auto.
    split; auto.
    apply const_rank_G_set; auto.
  Qed.

  Lemma GR_1_is_G_prop :
    forall(rules : set (sequent_rule V L)),
      one_step_rule_set rules ->
        set_equal (GR_n_set rules 1) prop_G_set.
  Proof.
    intros rules H.
    unfold GR_n_set, GR_set in *.
    eapply set_equal_trans.
      apply rank_rules_distribute_union.
    apply set_equal_union_empty_right.
    apply R_set_1_empty.
    trivial.
  Qed.

  Lemma GRC_1_is_GC_1 :
    forall(rules : set (sequent_rule V L)),
      one_step_rule_set rules ->
        set_equal (GRC_n_set rules 1) (GC_n_set V L 1).
  Proof.
    intros rules H.
    unfold GRC_n_set, GRC_set in *.
    eapply set_equal_trans.
      apply rank_rules_distribute_union.
    eapply set_equal_trans.
      eapply set_equal_union.
        apply GR_1_is_G_prop.
        trivial.
      apply set_equal_refl.
    eapply set_equal_trans.
      apply set_equal_symm.
      apply rank_rules_distribute_union.
    apply set_equal_refl.
  Qed.


  (**************************************************************************)
  (** ***  Propositional GC rule set  *)
  (**************************************************************************)

  Definition prop_GC_set(r : sequent_rule V L) : Prop :=
    GC_n_set V L 1 r.

  Lemma prop_G_subset_prop_GC : subset prop_G_set prop_GC_set.
  Proof.
    eapply subset_trans.
      apply prop_G_subset_Gn.
      apply le_n.
    apply G_n_subset_GC_n.
  Qed.

End Propositional_rules.

Implicit Arguments propositional_add_context [V L].
Implicit Arguments propositional_rule [V L].
Implicit Arguments prop_sequent_rule_ass_tcc [V L].
Implicit Arguments prop_sequent_rule_concl_tcc [V L].
Implicit Arguments propositional_rule_set [V L].
Implicit Arguments propositional_sequent_set [V L].
Implicit Arguments propositional_rule_add_context_bare_rule [V L].
Implicit Arguments propositional_add_context_left [V L].
Implicit Arguments propositional_add_context_right [V L].
Implicit Arguments propositional_prop_G_set [[V] [L]].
Implicit Arguments G_set_prop_set [V L].
Implicit Arguments propositional_empty_sequent_set [[V] [L]].
Implicit Arguments propositional_add_context_propositional [V L].
