(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: propositional_sound.v,v 1.16 2013/04/10 11:17:16 tews Exp $
 *)

(** ** Soundness for propositional logic

      This module proves the soundness part of 3.2.1. This result is
      also needed as induction base case for the soundness of
      coalgebraic logics.
 *)

Require Export propositional_models.

Section Propositional_soundness.

  Variable V : Type.
  Variable L : modal_operators.


  (** Abbreviation for propositional G proofs *)
  Definition prop_proof_type : sequent V L -> Type :=
    proof (prop_G_set V L) (empty_sequent_set V L).


  (**************************************************************************)
  (** *** Downward correct rules *)
  (**************************************************************************)

  Definition prop_downward_correct_rule(r : sequent_rule V L)
                                     (prop_r : propositional_rule r) : Prop :=
    forall(m : prop_model V),
      (forall(n : nat)(n_len : n < length (assumptions r)),
         prop_sequent_interpretation m (nth (assumptions r) n n_len) 
           (prop_sequent_rule_ass_tcc r n n_len prop_r)) ->
      prop_sequent_interpretation m (conclusion r) 
          (prop_sequent_rule_concl_tcc r prop_r).

  (* not needed *)
  Lemma prop_downward_correct_rule_tcc_irr :
    forall(r : sequent_rule V L)(prop_r1 prop_r2 : propositional_rule r),
      prop_downward_correct_rule r prop_r1 -> 
        prop_downward_correct_rule r prop_r2.
  Proof.
    unfold prop_downward_correct_rule in *.
    intros r prop_r1 prop_r2 H m H0.
    eapply prop_sequent_interpretation_tcc_irr.
    apply H.
    intros n n_len.
    eapply prop_sequent_interpretation_tcc_irr.
    apply H0.
  Qed.


  Definition prop_downward_correct_rule_set(rules : sequent_rule V L -> Prop)
                          (prop_rules : propositional_rule_set rules) : Prop :=
    forall(r : sequent_rule V L)(r_rule : rules r),
      prop_downward_correct_rule r (prop_rules r r_rule).


  Lemma propositional_correct_derivation : 
    forall(m : prop_model V)
          (rules : sequent_rule V L -> Prop)
          (prop_rules : propositional_rule_set rules)
          (hypotheses : sequent V L -> Prop)
          (prop_hyp : propositional_sequent_set hypotheses)
          (l : sequent V L)(prop_l : propositional_sequent l),
      prop_downward_correct_rule_set rules prop_rules ->
      (forall(h : sequent V L)(in_hyp : hypotheses h),
                  prop_sequent_interpretation m h (prop_hyp h in_hyp)) ->
      proof rules hypotheses l ->
        prop_sequent_interpretation m l prop_l.
  Proof.
    intros m rules prop_rules hypotheses prop_hyp l prop_l H H0 p.
    revert prop_l.
    pattern l.
    eapply proof_sequent_ind.
        intros gamma H1 prop_l.
        eapply prop_sequent_interpretation_tcc_irr.
        apply (H0 gamma H1).
      intros r H1 H2 prop_l.
      eapply prop_sequent_interpretation_tcc_irr.
      apply (H r H1).
      intros n n_len.
      apply H2.
    trivial.
  Qed.


  (**************************************************************************)
  (** ***  Downward correctness of context rules  *)
  (**************************************************************************)

  Lemma assumption_add_context_interpretation :
    forall(m : prop_model V)(sl sr : sequent V L)
          (prop_sl : propositional_sequent sl)
          (prop_sr : propositional_sequent sr)
          (sml : list (sequent V L))
          (prop_sml : every_nth propositional_sequent sml)
          (prop_sml_context : 
             every_nth propositional_sequent (map (add_context sl sr) sml)),
   sml <> [] ->
   (forall(n : nat)(n_len : n < length (map (add_context sl sr) sml)),
      prop_sequent_interpretation m
        (nth (map (add_context sl sr) sml) n n_len)
        (prop_sml_context n n_len))
     ->
       ~(~(prop_sequent_interpretation m sl prop_sl) /\
         ~(prop_sequent_interpretation m sr prop_sr) /\
         ~(forall(n : nat)(n_less : n < length sml),
            prop_sequent_interpretation m
              (nth sml n n_less) (prop_sml n n_less))).
  Proof.
    induction sml.
      intros prop_sml prop_sml_context H.
      exfalso.
      auto.
    rename a into s1.
    destruct sml as [| s2 ].
      clear IHsml.
      intros prop_sml prop_sml_context H H0.
      clear H.
      assert (0 < length (map (add_context sl sr) [s1])).
        clear. 
        simpl.
        auto.
      specialize (H0 0 H).
      simpl in H0.
      apply prop_sequent_interpretation_add_context_split in H0.
      intros H1.
      decompose [and] H1; clear H1.
      apply H0; clear H0; repeat split; intro H0.
          apply H2; apply H0.
        apply H5; clear H2 H4 H5.
        intros n n_less.
        destruct n.
          simpl.
          auto.
        simpl in n_less.
        omega.
      apply H4; apply H0.
    intros prop_sml prop_sml_context H H0 H1.
    decompose [and] H1; clear H1.
    assert (H6 := every_nth_tail _ _ _ prop_sml).
    assert (every_nth propositional_sequent 
               (map (add_context sl sr) (s2 :: sml))).
      clear - prop_sml_context.
      rewrite every_nth_map in *.
      apply every_nth_tail in prop_sml_context.
      trivial.
    apply (IHsml H6 H1).
        discriminate.
        clear - H0.
      intros n n_len.
      generalize (H1 n n_len).
      clear H1.
      rewrite nth_map.
      specialize (H0 (S n) (lt_n_S _ _ n_len)).
      revert H0.
      generalize (prop_sml_context (S n) 
             (lt_n_S n (length (map (add_context sl sr) (s2 :: sml))) n_len)).
      clear prop_sml_context.
      rewrite nth_map.
      rewrite nth_tail.
      intros p H0.
      erewrite nth_tcc_irr.
      intros p0.
      eapply prop_sequent_interpretation_tcc_irr.
      eexact H0.
    clear IHsml.
    repeat split; trivial.
    intros H7.
    specialize (H0 0 (lt_0_Sn (length (map (add_context sl sr) (s2 :: sml))))).
    simpl in H0.
    apply prop_sequent_interpretation_add_context_split in H0.
    apply H0; clear H0; repeat split; auto.
    intros H8.
    apply H5; clear H5.
    intros n n_less.
    destruct n.
      apply H8.
    eapply prop_sequent_interpretation_tcc_irr.
    apply H7.
  Qed.

  Lemma prop_downward_correct_context :
    forall(r : sequent_rule V L)(sl sr : sequent V L)
          (prop_sl_r_sr : propositional_rule (rule_add_context sl sr r)),
      conclusion r <> [] ->
      prop_downward_correct_rule r 
        (propositional_rule_add_context_bare_rule r sl sr prop_sl_r_sr)
      ->
        prop_downward_correct_rule (rule_add_context sl sr r) prop_sl_r_sr.
  Proof.
    intros r sl sr prop_sl_r_sr H H0.
    assert (H1 := prop_sequent_rule_concl_tcc _ prop_sl_r_sr).
    simpl in H1.
    assert (H2 := propositional_add_context_left _ _ _ H1).
    assert (H3 := propositional_add_context_right _ _ _ H1).
    assert (H4 := propositional_add_context_propositional _ _ _ H1).
    clear H1.
    unfold prop_downward_correct_rule.
    simpl.
    intros m H1.
    assert (assumptions r = [] \/ assumptions r <> []).
      clear. 
      destruct (assumptions r).
        auto.
      right.
      discriminate.
    destruct H5.
      clear H1.
      unfold prop_downward_correct_rule in *.
      eapply prop_sequent_interpretation_tcc_irr.
      eapply (prop_sequent_interpretation_add_context _ _ _ _ H2 _ H3).
      right.
      left.
      apply H0.
      intros n n_len.
      exfalso.
      rewrite H5 in n_len.
      simpl in *.
      omega.
    apply prop_sequent_interpretation_length_case_intro.
      intros H6.
      assert (sl = [] /\ sr = []).
        clear - H H6.
        unfold add_context in *.
        rewrite app_length in H6.
        rewrite app_length in H6.
        destruct (conclusion r).
          exfalso.
          auto.
        destruct sl.
          destruct sr.
            auto.
          simpl in *.
          omega.
        simpl in *.
        omega.
      destruct H7.
      subst sl sr.
      generalize (prop_sequent_rule_concl_tcc (rule_add_context [] [] r) 
                   prop_sl_r_sr).
      revert H6.
      unfold rule_add_context, add_context.
      simpl.
      rewrite app_nil_r.
      intros H6 p.
      eapply prop_sequent_interpretation_tcc_irr.
      apply H0.
      intros n n_len.
      assert (n < length (map (add_context [] []) (assumptions r))).
        rewrite map_length.
        trivial.
      specialize (H1 n H7).
      revert H1.
      generalize (prop_sequent_rule_ass_tcc (rule_add_context [] [] r) 
                  n H7 prop_sl_r_sr).
      simpl.
      rewrite nth_map.
      generalize (nth_map_tcc (add_context [] []) (assumptions r) n H7).
      clear H7.
      unfold add_context in *.
      simpl.
      intros n_len'.
      rewrite app_nil_r.
      intros p0 H1.
      generalize (prop_sequent_rule_ass_tcc r n n_len
             (propositional_rule_add_context_bare_rule r [] [] prop_sl_r_sr)).
      erewrite nth_tcc_irr.
      intros p1.
      eapply prop_sequent_interpretation_tcc_irr.
      eexact H1.
    assert (H6 := assumption_add_context_interpretation m sl sr H2 H3
                    (assumptions r)
                    (fun (n : nat)(n_less : n < length (assumptions r)) =>
                       prop_sequent_rule_ass_tcc r n n_less 
                         (propositional_rule_add_context_bare_rule r sl sr 
                            prop_sl_r_sr))
                    (fun (n : nat)
                         (n_less : n < length 
                                  (map (add_context sl sr) (assumptions r))) =>
                       prop_sequent_rule_ass_tcc (rule_add_context sl sr r)
                         n n_less prop_sl_r_sr)
                    H5 H1).
    clear H1.
    simpl in H6.
    intros H1 H7.
    apply H6; clear H6; repeat split; intro H6; apply H7; clear H7.
        eapply prop_sequent_interpretation_tcc_irr.
        apply (prop_sequent_interpretation_add_context _ _ _ _ H2 H4 H3).
        auto.
      eapply prop_sequent_interpretation_tcc_irr.
      apply (prop_sequent_interpretation_add_context _ _ _ _ H2 H4 H3).
      auto.
    eapply prop_sequent_interpretation_tcc_irr.
    eapply (prop_sequent_interpretation_add_context _ _ _ _ H2 _ H3).
    right.
    left.
    apply H0.
    trivial.
  Qed.


  (**************************************************************************)
  (** ***  Downward correctness of G  *)
  (**************************************************************************)

  Lemma prop_downward_correct_ax : 
    forall(r : sequent_rule V L)(prop_r : propositional_rule r),
      is_ax_rule r -> 
        prop_downward_correct_rule r prop_r.
  Proof.
    unfold is_ax_rule, prop_downward_correct_rule in *.
    unfold simple_tautology, simple_tautology_witness in *.
    intros r prop_r H m H0.
    decompose [ex and or dep_and] H; clear H.
    rename x into n1, x0 into n2, x1 into v, a into n1_less, a0 into n2_less.
    clear H0 H1.
    apply prop_sequent_interpretation_length_case_intro.
      intros H.
      exfalso.
      assert (n1 <> n2).
        intros H4.
        subst n2.
        rewrite nth_tcc_irr with (inside_2 := n1_less) in H3.
        rewrite H2 in H3.
        discriminate.
      omega.
    intros H H4.
    clear H.
    assert (H5 := prop_sequent_rule_concl_tcc _ prop_r).
    assert (~ (~ is_prop_model m (nth (conclusion r) n1 n1_less) 
                                                 (H5 n1 n1_less) /\
               ~ is_prop_model m (nth (conclusion r) n2 n2_less) 
                                                 (H5 n2 n2_less))).
      generalize (H5 n1 n1_less).
      generalize (H5 n2 n2_less).
      rewrite H2.
      rewrite H3.
      intros pn1 pn2.
      simpl.
      tauto.
    apply H; clear H; split; intro H; apply H4; clear H4.
      eapply prop_sequent_interpretation_nth_intro.
      eapply is_prop_model_tcc_irr.
      eexact H.
    eapply prop_sequent_interpretation_nth_intro.
    eapply is_prop_model_tcc_irr.
    eexact H.
  Qed.

  Lemma prop_downward_correct_and : 
    forall(r : sequent_rule V L)(prop_r : propositional_rule r),
      is_and_rule r -> prop_downward_correct_rule r prop_r.
  Proof.
    intros r prop_r H.
    apply and_rule_context in H.
    decompose [ex] H; clear H.
    rename x into sl, x0 into sr, x1 into f1, x2 into f2.
    generalize prop_r; clear prop_r.
    rewrite H1.
    intros prop_r.
    apply prop_downward_correct_context.
      discriminate.
    generalize (propositional_rule_add_context_bare_rule 
                  (bare_and_rule f1 f2) sl sr prop_r).
    clear.
    unfold prop_downward_correct_rule in *.
    simpl (assumptions (bare_and_rule f1 f2)).
    simpl (conclusion (bare_and_rule f1 f2)).
    intros p m H.
    apply prop_sequent_interpretation_singleton.
    simpl.
    split.
      specialize (H 0 (lt_0_Sn 1)).
      simpl in *.
      apply prop_sequent_interpretation_singleton in H.
      eapply is_prop_model_tcc_irr.
      eexact H.
    specialize (H 1 (lt_n_S _ _ (lt_0_Sn 0))).
    simpl in *.
    apply prop_sequent_interpretation_singleton in H.
    eapply is_prop_model_tcc_irr.
    eexact H.
  Qed.


  Lemma prop_downward_correct_neg_and : 
    forall(r : sequent_rule V L)(prop_r : propositional_rule r),
      is_neg_and_rule r -> prop_downward_correct_rule r prop_r.
  Proof.
    intros r prop_r H.
    apply neg_and_rule_context in H.
    decompose [ex] H; clear H.
    rename x into sl, x0 into sr, x1 into f1, x2 into f2.
    generalize prop_r; clear prop_r.
    rewrite H1.
    intros prop_r.
    apply prop_downward_correct_context.
      discriminate.
    generalize (propositional_rule_add_context_bare_rule 
                  (bare_neg_and_rule f1 f2) sl sr prop_r).
    clear.
    unfold prop_downward_correct_rule in *.
    simpl (assumptions (bare_neg_and_rule f1 f2)).
    simpl (conclusion (bare_neg_and_rule f1 f2)).
    intros p m H.
    apply prop_sequent_interpretation_singleton.
    simpl.
    intros H0.
    destruct H0.
    specialize (H 0 (lt_0_Sn 0)).
    simpl in *.
    apply prop_sequent_interpretation_cons_cons_elim in H.
    rewrite prop_sequent_interpretation_singleton in H.
    simpl in H.
    apply H; clear H; split; intros H2; apply H2; clear H2.
      eapply is_prop_model_tcc_irr.
      eexact H0.
    eapply is_prop_model_tcc_irr.
    eexact H1.
  Qed.


  Lemma prop_downward_correct_neg_neg : 
    forall(r : sequent_rule V L)(prop_r : propositional_rule r),
      is_neg_neg_rule r -> prop_downward_correct_rule r prop_r.
  Proof.
    intros r prop_r H.
    apply neg_neg_rule_context in H.
    decompose [ex] H; clear H.
    rename x into sl, x0 into sr, x1 into f.
    generalize prop_r; clear prop_r.
    rewrite H0.
    intros prop_r.
    apply prop_downward_correct_context.
      discriminate.
    generalize (propositional_rule_add_context_bare_rule 
                  (bare_neg_neg_rule f) sl sr prop_r).
    clear.
    unfold prop_downward_correct_rule in *.
    simpl (assumptions (bare_neg_neg_rule f)).
    simpl (conclusion (bare_neg_neg_rule f)).
    intros p m H.
    apply prop_sequent_interpretation_singleton.
    specialize (H 0 (lt_0_Sn 0)).
    simpl in *.
    apply prop_sequent_interpretation_singleton in H.
    intros H0; apply H0; clear H0.
    eapply is_prop_model_tcc_irr.
    eexact H.
  Qed.


  Lemma prop_downward_correct_prop_G : 
    prop_downward_correct_rule_set (prop_G_set V L) propositional_prop_G_set.
  Proof.
    unfold prop_downward_correct_rule_set in *.
    intros r r_rule.
    unfold prop_G_set, G_n_set, rank_rules, G_set, union in *.
    decompose [and or] r_rule.
          apply prop_downward_correct_ax; trivial.
        apply prop_downward_correct_and.
        trivial.
      apply prop_downward_correct_neg_and.
      trivial.
    apply prop_downward_correct_neg_neg.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Soundness *)
  (***************************************************************************)

  Lemma propositional_correct_G_derivation : 
    forall(m : prop_model V)
          (l : sequent V L)(prop_l : propositional_sequent l),
      prop_proof_type l ->
        prop_sequent_interpretation m l prop_l.
  Proof.
    intros m l prop_l p.
    eapply (propositional_correct_derivation _ (prop_G_set V L) _ _
             propositional_empty_sequent_set).
        apply prop_downward_correct_prop_G.
      intros h in_hyp.
      contradiction.
    eexact p.
  Qed.

  Lemma propositional_sound_G :
    forall(nonempty_v : V)(s : sequent V L)(prop_s : propositional_sequent s),
      provable (prop_G_set V L) (empty_sequent_set V L) s ->
        prop_valid_sequent nonempty_v s prop_s.
  Proof.
    intros l l_nonempty prop_l H.
    destruct H.
    clear H.
    intros m.
    rewrite <- (prop_model_sequent_interpretation _ _ _ prop_l); trivial.
    apply propositional_correct_G_derivation; trivial.
  Qed.

End Propositional_soundness.

Implicit Arguments prop_proof_type [[V] [L]].
Implicit Arguments propositional_sound_G [V L].
