(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: semantics.v,v 1.40 2013/04/10 11:17:16 tews Exp $
 *)


(** * Semantics *)

(** ** Standard Semantics

      This module defines lambda structures, models and then three
      semantics for formulas and sequents. First the normative
      coalgebraic semantics. The other semantic definitions are only
      utility constructions. Second, the semantics [X,tau |= f] for
      propositional formulas [f], and, third, the semantics [TX,tau |=
      f] for modal formulas [f].

      Each semantics comes with a simplified sequent semantics (based
      on [some_neg]) and the necessary utility lemmas.
 *)

Require Export image functor propositional_formulas modal_formulas
               some_neg_form.

Section Semantics.

  Variable V : Type.
  Variable L : modal_operators.
  Variable T : functor.


  (***************************************************************************)
  (** *** Coalgebraic models *)
  (***************************************************************************)

  (** LAMBDA structures, page 5 *)
  Definition lambda_structure_type : Type :=
    forall(op : operator L){X : Type},
      counted_list (set X) (arity L op) -> set (obj T X).

  Record lambda_structure : Type := {
    modal_semantics : lambda_structure_type;
    set_equal_modal_semantics :
      forall(X : Type)(op : operator L)
            (preds_x_1 preds_x_2 : counted_list (set X) (arity L op)),
        (forall(i : nat)(i_less : i < arity L op),
           set_equal 
             (nth (list_of_counted_list preds_x_1) i 
                  (less_length_counted_list _ _ preds_x_1 i_less))
             (nth (list_of_counted_list preds_x_2) i 
                  (less_length_counted_list _ _ preds_x_2 i_less)))
          ->
            set_equal (modal_semantics op X preds_x_1) 
                       (modal_semantics op X preds_x_2);
    fibred_semantics :
      forall(op : operator L)(X Y : Type)(f : X -> Y)
            (preds_y : counted_list (set Y) (arity L op)),
        set_equal 
          (modal_semantics op X (counted_map (inv_img f) preds_y))
          (inv_img (fmap T f) (modal_semantics op Y preds_y))
    }.

  Implicit Arguments modal_semantics [X].

  (** T/P(V) coalgebras as models, page 4, 5 *)
  Record model : Type := {
    state : Type;
    coalg : state -> obj T state;
    coval : state -> set V
  }.


  (***************************************************************************)
  (** *** Normative semantics *)
  (***************************************************************************)

  Definition form_semantics(LS : lambda_structure)(m : model)
                           (f : lambda_formula V L) : set (state m) :=
    lambda_formula_rec
      (fun(v : V)(c : state m) => coval m c v)
      set_inverse
      intersection
      (fun(op : operator L)
          (sem_args : counted_list (set (state m)) (arity L op)) =>
         inv_img (coalg m) (modal_semantics LS op sem_args))
      f.

  Lemma form_semantics_char :
    forall(LS : lambda_structure)(m : model)(f : lambda_formula V L),
      form_semantics LS m f = match f with
        | lf_prop v => fun(c : state m) => coval m c v
        | lf_neg f => set_inverse (form_semantics LS m f)
        | lf_and f1 f2 => 
            intersection (form_semantics LS m f1) (form_semantics LS m f2)
        | lf_modal op args =>
            inv_img (coalg m) 
                    (modal_semantics LS op 
                       (counted_map (form_semantics LS m) args))
    end.
  Proof.
    intros LS m f.
    unfold form_semantics in *.
    rewrite lambda_formula_rec_char.
    trivial.
  Qed.

  Lemma form_semantics_false :
    forall(nonempty_v : V)(LS : lambda_structure)(m : model)(x : state m),
      not (form_semantics LS m (lambda_false nonempty_v) x).
  Proof.
    intros nonempty_v LS m x.
    unfold lambda_false in *.
    repeat rewrite form_semantics_char.
    apply intersection_complement.
  Qed.
  

  (***************************************************************************)
  (** **** Changing the model for substitutions *)
  (***************************************************************************)

  Definition subst_coval(LS : lambda_structure)(m : model)
                           (sigma : lambda_subst V L) : (state m) -> set V :=
    fun(x : state m)(v : V) =>
      form_semantics LS m (sigma v) x.
  
  Definition modal_subst_coval(LS : lambda_structure)(m : model)
                                         (sigma : lambda_subst V L) : model :=
    {| state := state m;
       coalg := coalg m;
       coval := subst_coval LS m sigma
    |}.
  
  Lemma set_equal_form_semantics_subst_change_coval :
    forall(LS : lambda_structure)(m : model)
          (sigma : lambda_subst V L)(f : lambda_formula V L),
      set_equal
        (form_semantics LS m (subst_form sigma f))
        (form_semantics LS (modal_subst_coval LS m sigma) f).
  Proof.
    induction f.
          intros x.
          split.
            intros H.
            trivial.
          intros H.
          trivial.
        rewrite subst_form_char.
        rewrite form_semantics_char.
        rewrite (form_semantics_char _ _ (lf_neg _)).
        apply set_equal_set_inverse.
        trivial.
      rewrite subst_form_char.
      rewrite form_semantics_char.
      rewrite (form_semantics_char _ _ (lf_and _ _)).
      apply set_equal_intersection; trivial.
    rewrite subst_form_char.
    rewrite form_semantics_char.
    rewrite (form_semantics_char _ _ (lf_modal _ _)).
    simpl.
    apply set_equal_inv_img_pred.
    apply (set_equal_modal_semantics LS).
    intros i i_less.
    generalize (less_length_counted_list i (arity L op)
                   (counted_map (form_semantics LS m)
                       (counted_map (subst_form sigma) args)) i_less).
    generalize (less_length_counted_list i (arity L op)
                  (counted_map (B:=set (state m)) 
                      (form_semantics LS (modal_subst_coval LS m sigma))
                       args) i_less).
    repeat rewrite list_of_counted_list_map.
    rewrite map_map.
    intros i_less_1 i_less_2.
    rewrite nth_map.
    rewrite nth_map.
    erewrite nth_tcc_irr.
    apply H.
  Qed.

  Lemma form_semantics_subst_change_coval :
    forall(LS : lambda_structure)(m : model)(x : state m)
          (sigma : lambda_subst V L)(f : lambda_formula V L),
      form_semantics LS m (subst_form sigma f) x <->
        form_semantics LS (modal_subst_coval LS m sigma) f x.
  Proof.
    intros LS m x sigma f.
    apply set_equal_form_semantics_subst_change_coval with (m := m).
  Qed.


  (***************************************************************************)
  (** **** Sequent semantics and validity *)
  (***************************************************************************)

  Definition seq_semantics(nonempty_v : V)(LS : lambda_structure)
                          (m : model)(s : sequent V L) : set (state m) :=
    form_semantics LS m (or_formula_of_sequent s nonempty_v).


  (** [M |= A], page 7 *)
  Definition valid_all_states(nonempty_v : V)(LS : lambda_structure)
                             (m : model)(s : sequent V L) : Prop :=
    is_full_set (seq_semantics nonempty_v LS m s).


  Lemma valid_all_states_subst_change_coval :
    forall(nonempty_v : V)(LS : lambda_structure)(m : model)
          (sigma : lambda_subst V L)(s : sequent V L),
      (valid_all_states nonempty_v LS m (subst_sequent sigma s) <->
        valid_all_states nonempty_v LS (modal_subst_coval LS m sigma) s).
  Proof.
    intros nonempty_v LS m sigma s.
    unfold valid_all_states, is_full_set in *.
    unfold seq_semantics in *.
    split.
      intros H0 a.
      destruct s.
        exfalso.
        simpl in *.
        specialize (H0 a).
        apply form_semantics_false in H0.
        trivial.
      assert (l :: s <> []).
        discriminate.
      rewrite <- form_semantics_subst_change_coval.
      rewrite or_formula_of_sequent_nonempty with (nonempty_s := H).
      rewrite <- or_formula_subst_sequent.
      rewrite <- or_formula_of_sequent_nonempty with (nonempty_v := nonempty_v).
      apply H0.
    intros H0 a.
    specialize (H0 a).
    destruct s.
      exfalso.
      simpl in *.
      apply form_semantics_false in H0.
      trivial.
    assert (l :: s <> []).
      discriminate.
    rewrite <- form_semantics_subst_change_coval in H0.
    rewrite or_formula_of_sequent_nonempty with (nonempty_s := H) in H0.
    rewrite <- or_formula_subst_sequent in H0.
    rewrite <- or_formula_of_sequent_nonempty 
                  with (nonempty_v := nonempty_v) in H0.
    trivial.
  Qed.


  (* Mod(T) |= A, page 7 *)
  Definition valid_all_models(nonempty_v : V)(LS : lambda_structure)
                             (s : sequent V L) : Prop :=
    forall(m : model), valid_all_states nonempty_v LS m s.


  (**************************************************************************)
  (** *** Simplified sequence semantics *)
  (**************************************************************************)
  
  Definition state_seq_semantics(LS : lambda_structure)
                                (m : model)(x : state m)
                                (s : sequent V L) : Prop :=
    some_neg (fun f => form_semantics LS m f x) s.

  Lemma state_seq_semantics_singleton :
    forall(LS : lambda_structure)(m : model)(x : state m)
          (f : lambda_formula V L),
      state_seq_semantics LS m x [f] <-> form_semantics LS m f x.
  Proof.
    intros LS m x f.
    apply some_neg_singleton.
  Qed.

  Lemma state_seq_semantics_nth_intro :
    forall(LS : lambda_structure)(m : model)(x : state m)
          (s : sequent V L)(n : nat)(n_less : n < length s),
      form_semantics LS m (nth s n n_less) x ->
        state_seq_semantics LS m x s.
  Proof.
    intros LS m x s n n_less H.
    eapply some_neg_nth_intro.
    eexact H.
  Qed.

  Lemma state_seq_semantics_cons_case_elim :
    forall(LS : lambda_structure)(m : model)(x : state m)
          (f : lambda_formula V L)(s : sequent V L),
      state_seq_semantics LS m x (f :: s) ->
        (s = [] /\ form_semantics LS m f x) \/
        (s <> [] /\ ~(~form_semantics LS m f x /\ 
                      ~state_seq_semantics LS m x s)).
  Proof.
    intros LS m x f s H.
    apply some_neg_cons_case_elim in H.
    trivial.
  Qed.

  Lemma state_seq_semantics_length_case_intro :
    forall(LS : lambda_structure)(m : model)(x : state m)(s : sequent V L),
      (forall(s_len : length s = 1),
         form_semantics LS m (nth s 0 (nth_head_tcc s s_len)) x) ->
      (length s <> 1 -> ~~state_seq_semantics LS m x s) ->
        state_seq_semantics LS m x s.
  Proof.
    intros LS m x s H H0.
    apply some_neg_length_case_intro.
      trivial.
    trivial.
  Qed.

  Lemma state_seq_semantics_append :
    forall(LS : lambda_structure)(m : model)(x : state m)
          (s1 s2 : sequent V L),
      state_seq_semantics LS m x (s1 ++ s2) ->
        ~ ( ~ (state_seq_semantics LS m x s1) /\ 
            ~ (state_seq_semantics LS m x s2)).
  Proof.
    intros LS m x s1 s2.
    apply some_neg_append.
  Qed.

  Lemma state_seq_semantics_append_right :
    forall(LS : lambda_structure)(m : model)(x : state m)
          (s1 s2 : sequent V L),
      state_seq_semantics LS m x s1 ->
        state_seq_semantics LS m x (s1 ++ s2).
  Proof.
    intros LS m x s1 s2.
    apply some_neg_append_right.
  Qed.

  Lemma state_seq_semantics_append_left :
    forall(LS : lambda_structure)(m : model)(x : state m)
          (s1 s2 : sequent V L),
      state_seq_semantics LS m x s2 ->
        state_seq_semantics LS m x (s1 ++ s2).
  Proof.
    intros LS m x s1 s2.
    apply some_neg_append_left.
  Qed.

  Lemma state_seq_semantics_reorder :
    forall(LS : lambda_structure)(m : model)(x : state m)
          (s1 s2 : sequent V L),
      list_reorder s1 s2 ->
      state_seq_semantics LS m x s1 ->
        state_seq_semantics LS m x s2.
  Proof.
    intros LS m x s1 s2.
    apply some_neg_reorder.
  Qed.


  (**************************************************************************)
  (** ***  Equivalence for simplified semantics  *)
  (**************************************************************************)

  Lemma state_seq_semantics_correct :
    forall(nonempty_v : V)(LS : lambda_structure)(m : model)(x : state m)
          (s : sequent V L),
      state_seq_semantics LS m x s <-> seq_semantics nonempty_v LS m s x.
  Proof.
    intros nonempty_v LS m x s.
    unfold state_seq_semantics, seq_semantics in *.
    apply some_neg_correct with (P := fun f => form_semantics LS m f x).
      intros nonempty_v0.
      apply form_semantics_false.
    intros f1 f2.
    apply iff_refl.
  Qed.

  Lemma state_seq_semantics_valid :
    forall(nonempty_v : V)(LS : lambda_structure)(m : model)(s : sequent V L),
      valid_all_states nonempty_v LS m s <->
        (forall(x : state m), state_seq_semantics LS m x s).
  Proof.
    intros nonempty_v LS m s.
    split.
      intros H x.
      rewrite state_seq_semantics_correct; trivial.
    intros H x.
    apply state_seq_semantics_correct; trivial.
  Qed.
  

  (**************************************************************************)
  (** *** Standard semantic lemma (derived via simplified semantics)  *)
  (**************************************************************************)

  Lemma valid_all_states_append_right :
    forall(nonempty_v : V)(LS : lambda_structure)(m : model)
          (s1 s2 : sequent V L),
      valid_all_states nonempty_v LS m s1 ->
        valid_all_states nonempty_v LS m (s1 ++ s2).
  Proof.
    intros nonempty_v LS m s1 s2 H.
    rewrite state_seq_semantics_valid in *; trivial.
    intros x.
    apply state_seq_semantics_append_right.
    apply H.
  Qed.

  Lemma valid_all_states_reorder :
    forall(nonempty_v : V)(LS : lambda_structure)(m : model)
          (s1 s2 : sequent V L),
      list_reorder s1 s2 ->
      valid_all_states nonempty_v LS m s1 ->
        valid_all_states nonempty_v LS m s2.
  Proof.
    intros nonempty_v LS m s1 s2 H H0.
    rewrite state_seq_semantics_valid in *; trivial.
    intros x.
    eapply state_seq_semantics_reorder.
      eexact H.
    apply H0.
  Qed.


  (**************************************************************************)
  (** *** Semantics [X,tau |= f] for propositional [f]  *)
  (**************************************************************************)

  Fixpoint propositional_valuation{X : Type}(coval : X -> set V)
                 (f : lambda_formula V L)(prop_f : propositional f) : set X :=
    match f return propositional f -> set X with
      | lf_prop v => fun _ (c : X) => coval c v
      | lf_neg f => fun(H : propositional (lf_neg f)) => 
          set_inverse (propositional_valuation coval f (propositional_neg _ H))
      | lf_and f1 f2 => fun(H : propositional (lf_and f1 f2)) => 
          intersection 
            (propositional_valuation coval f1 (propositional_and_left _ _ H))
            (propositional_valuation coval f2 (propositional_and_right _ _ H))
      | lf_modal op args => 
          fun(H : propositional (lf_modal op args)) =>
            propositional_tcc_modal op args H
    end prop_f.

  Lemma propositional_valuation_tcc_irr :
    forall(X : Type)(coval : X -> set V)(f : lambda_formula V L)
          (prop_f_1 prop_f_2 : propositional f),
      propositional_valuation coval f prop_f_1 = 
        propositional_valuation coval f prop_f_2.
  Proof.
    induction f.
          intros prop_f_1 prop_f_2.
          simpl.
          trivial.
        intros prop_f_1 prop_f_2.
        simpl.
        erewrite IHf.
        f_equal.
      intros prop_f_1 prop_f_2.
      simpl.
      erewrite IHf1.
      erewrite IHf2.
      f_equal.
    intros prop_f_1 prop_f_2.
    eapply propositional_tcc_modal.
    eexact prop_f_1.
  Qed.

  Lemma propositional_valuation_false :
    forall(X : Type)(coval : X -> set V)(v : V)
          (prop_false : propositional (lambda_false v))(x : X),
      not (propositional_valuation coval (lambda_false v) prop_false x).
  Proof.
    intros X coval0 v prop_false x.
    simpl.
    apply intersection_complement.
  Qed.


  Definition prop_seq_val{X : Type}(nonempty_v : V)(coval : X -> set V)
                (s : sequent V L)(prop_s : propositional_sequent s) : set X :=
    propositional_valuation coval 
      (or_formula_of_sequent s nonempty_v)
      (propositional_or_formula nonempty_v s prop_s).

  Lemma prop_seq_val_tcc_irr :
    forall(X : Type)(nonempty_v : V)(coval : X -> set V)(s : sequent V L)
          (prop_s_1 prop_s_2 : propositional_sequent s),
      prop_seq_val nonempty_v coval s prop_s_1 =
        prop_seq_val nonempty_v coval s prop_s_2.
  Proof.
    intros X nonempty_v coval0 s prop_s_1 prop_s_2.
    unfold prop_seq_val in *.
    apply propositional_valuation_tcc_irr.
  Qed.


  Definition prop_seq_val_valid{X : Type}(nonempty_v : V)(coval : X -> set V)
                  (s : sequent V L)(prop_s : propositional_sequent s) : Prop :=
    is_full_set (prop_seq_val nonempty_v coval s prop_s).

  Lemma prop_seq_val_valid_tcc_irr :
    forall(X : Type)(nonempty_v : V)(coval : X -> set V)(s : sequent V L)
          (prop_s_1 prop_s_2 : propositional_sequent s),
      prop_seq_val_valid nonempty_v coval s prop_s_1 <->
        prop_seq_val_valid nonempty_v coval s prop_s_2.
  Proof.
    intros X nonempty_v coval0 s prop_s_1 prop_s_2.
    unfold prop_seq_val_valid in *.
    erewrite prop_seq_val_tcc_irr.
    apply iff_refl.
  Qed.


  (**************************************************************************)
  (** *** Equivalence of standard semantics with [X,tau |= f]  *)
  (**************************************************************************)

  Lemma prop_val_semantics :
    forall(LS : lambda_structure)(m : model)
          (f : lambda_formula V L)(prop_f : propositional f),
      set_equal (form_semantics LS m f)
                (propositional_valuation (coval m) f prop_f).
  Proof.
    induction f.
          intros prop_f.
          simpl.
          rewrite form_semantics_char.
          apply set_equal_refl.
        intros prop_f.
        simpl.
        rewrite form_semantics_char.
        apply set_equal_set_inverse.
        apply IHf.
      intros prop_f.
      simpl.
      rewrite form_semantics_char.
      apply set_equal_intersection.
        apply IHf1.
      apply IHf2.
    intros prop_f.
    eapply propositional_tcc_modal.
    eexact prop_f.
  Qed.

  Lemma prop_val_semantics_valid :
    forall(nonempty_v : V)(LS : lambda_structure)(m : model)(s : sequent V L)
          (prop_s : propositional_sequent s),
      valid_all_states nonempty_v LS m s <-> 
        prop_seq_val_valid nonempty_v (coval m) s prop_s.
  Proof.
    intros nonempty_v LS m s prop_s.
    unfold valid_all_states, prop_seq_val_valid in *.
    unfold seq_semantics, prop_seq_val, is_full_set in *.
    split.
      intros H a.
      eapply prop_val_semantics.
      apply H.
    intros H a.
    eapply prop_val_semantics.
    apply H.
  Qed.


  (**************************************************************************)
  (** *** Simplified [X,tau |= f] sequent semantics  *)
  (**************************************************************************)

  Definition prop_val_pred{X : Type}(coval : X -> set V)(x : X)
                  (f : lambda_formula V L)(prop_f : propositional f) : Prop :=
    propositional_valuation coval f prop_f x.

  Lemma prop_val_pred_tcc_irr :
    forall(X : Type)(coval : X -> set V)(x : X)(f : lambda_formula V L)
          (prop_f_1 prop_f_2 : propositional f),
      prop_val_pred coval x f prop_f_1 ->
        prop_val_pred coval x f prop_f_2.
  Proof.
    intros X coval0 x f prop_f_1 prop_f_2 H.
    unfold prop_val_pred in *.
    erewrite propositional_valuation_tcc_irr.
    eexact H.
  Qed.

  Definition simple_prop_seq_val{X : Type}(coval : X -> set V)
          (s : sequent V L)(prop_s : propositional_sequent s)(x : X) : Prop :=
    some_neg_dep propositional (prop_val_pred coval x) s prop_s.

  Lemma simple_prop_seq_val_tcc_irr :
    forall(X : Type)(coval : X -> set V)(x : X)
          (s : sequent V L)(prop_s_1 prop_s_2 : propositional_sequent s),
      simple_prop_seq_val coval s prop_s_1 x ->
        simple_prop_seq_val coval s prop_s_2 x.
  Proof.
    intros X coval0 x.
    apply some_neg_dep_tcc_irr.
    apply prop_val_pred_tcc_irr.
  Qed.

  Lemma simple_prop_seq_val_head :
    forall(X : Type)(coval : X -> set V)(x : X)
          (f : lambda_formula V L)(s : sequent V L)
          (prop_fs : propositional_sequent (f :: s)),
      propositional_valuation coval f 
           (propositional_sequent_head _ _ prop_fs) x ->
        simple_prop_seq_val coval (f :: s) prop_fs x.
  Proof.
    intros X coval0 x f s prop_fs H.
    eapply some_neg_dep_head.
      apply prop_val_pred_tcc_irr.
    eexact H.
  Qed.

  Lemma simple_prop_seq_val_tail :
    forall(X : Type)(coval : X -> set V)(x : X)
          (f : lambda_formula V L)(s : sequent V L)
          (prop_fs : propositional_sequent (f :: s)),
      simple_prop_seq_val coval s (propositional_sequent_tail _ _ prop_fs) x ->
        simple_prop_seq_val coval (f :: s) prop_fs x.
  Proof.
    intros X coval0 x f s prop_fs H.
    eapply some_neg_dep_tail.
      apply prop_val_pred_tcc_irr.
    eapply simple_prop_seq_val_tcc_irr in H.
    eexact H.
  Qed.

  Lemma simple_prop_seq_val_length_case_intro :
    forall(X : Type)(coval : X -> set V)(x : X)(s : sequent V L)
          (prop_s : propositional_sequent s),
      length s <> 1 ->
      ~~simple_prop_seq_val coval s prop_s x ->
        simple_prop_seq_val coval s prop_s x.
  Proof.
    intros X coval0 x s prop_s H H0.
    apply some_neg_dep_length_case_intro.
      intros len.
      contradiction.
    trivial.
  Qed.


  Definition simple_prop_seq_val_valid{X : Type}(coval : X -> set V)
          (s : sequent V L)(prop_s : propositional_sequent s) : Prop :=
    is_full_set (simple_prop_seq_val coval s prop_s).


  (**************************************************************************)
  (** *** Equivalence for simplified [X,tau |= f] sequent semantics  *)
  (**************************************************************************)

  Lemma prop_val_pred_or :
   forall(X : Type)(coval : X -> set V)(x : X)(f1 f2 : lambda_formula V L)
         (prop_f1 : propositional f1)(prop_f2 : propositional f2),
     prop_val_pred coval x (lambda_or f1 f2) 
                     (propositional_lambda_or _ _ prop_f1 prop_f2)
        <->
     ~(~prop_val_pred coval x f1 prop_f1 /\ ~prop_val_pred coval x f2 prop_f2).
  Proof.
    simpl.
    unfold set_inverse, intersection in *.
    split.
      intros H H0.
      destruct H0.
      apply H; clear H; split; intro H.
        apply H0.
        eapply prop_val_pred_tcc_irr.
        eexact H.
      apply H1.
      eapply prop_val_pred_tcc_irr.
      eapply H.
    intros H H0.
    destruct H0.
    apply H; clear H; split; intro H.
      apply H0.
      eapply prop_val_pred_tcc_irr.
      eexact H.
    apply H1.
    eapply prop_val_pred_tcc_irr.
    eapply H.
  Qed.

  Lemma simple_prop_seq_val_valid_correct :
    forall(nonempty_v : V)(X : Type)(coval : X -> set V)(s : sequent V L)
          (prop_s : propositional_sequent s),
      prop_seq_val_valid nonempty_v coval s prop_s
        <-> simple_prop_seq_val_valid coval s prop_s.
  Proof.
    split.
      intros H x.
      unfold simple_prop_seq_val in *.
      rewrite some_neg_dep_correct.
            apply H.
          apply prop_val_pred_tcc_irr.
        apply prop_val_pred_or.
      intros nonempty_v0 t_false.
      apply propositional_valuation_false.
    intros H x.
    specialize (H x).
    unfold simple_prop_seq_val in *.
    rewrite some_neg_dep_correct in H.
          apply H.
        apply prop_val_pred_tcc_irr.
      apply prop_val_pred_or.
    intros nonempty_v0 t_false.
    apply propositional_valuation_false.
  Qed.

  Lemma prop_seq_val_valid_reorder :
    forall(nonempty_v : V)(X : Type)(coval : X -> set V)(s1 s2 : sequent V L)
          (prop_s1 : propositional_sequent s1)
          (prop_s2 : propositional_sequent s2),
      list_reorder s1 s2 ->
      prop_seq_val_valid nonempty_v coval s1 prop_s1 ->
        prop_seq_val_valid nonempty_v coval s2 prop_s2.
  Proof.
    intros nonempty_v X coval0 s1 s2 prop_s1 prop_s2 H H0.
    rewrite simple_prop_seq_val_valid_correct in *.
    intros x.
    eapply some_neg_dep_reorder.
        apply prop_val_pred_tcc_irr.
      eexact H.
    apply H0.
  Qed.


  (**************************************************************************)
  (** *** Semantics [TX,tau |= f] for counted modal arguments  *)
  (**************************************************************************)

  Fixpoint counted_prop_list_valuation{X : Type}(coval : X -> set V)(n : nat)
                   (fl : counted_list (lambda_formula V L) n)
                   (fl_prop : every_nth prop_form 
                                        (list_of_counted_list fl))
                    : counted_list (set X) n :=
    match fl in counted_list _ n
      return every_nth prop_form (list_of_counted_list fl) -> 
                                                       counted_list (set X) n
    with
      | counted_nil => fun _ => counted_nil
      | counted_cons n f fl => 
          fun(H : every_nth prop_form (f :: list_of_counted_list fl)) =>
            counted_cons
              (fun(x : X) => coval x (prop_form_acc f (every_nth_head _ _ _ H)))
              (counted_prop_list_valuation coval n fl (every_nth_tail _ _ _ H))
    end fl_prop.

  Lemma counted_prop_list_valuation_tcc_irr :
    forall(X : Type)(coval : X -> set V)(n : nat)
          (fl : counted_list (lambda_formula V L) n)
          (fl_prop_1 fl_prop_2 : every_nth prop_form (list_of_counted_list fl)),
      counted_prop_list_valuation coval n fl fl_prop_1 =
        counted_prop_list_valuation coval n fl fl_prop_2.
  Proof.
    induction fl.
      trivial.
    intros fl_prop_1 fl_prop_2.
    simpl.
    f_equal.
      erewrite prop_form_acc_tcc_irr.
      trivial.
    rewrite IHfl with (fl_prop_2 := every_nth_tail _ _ 
                                   (list_of_counted_list _) fl_prop_2).
    trivial.
  Qed.


  Lemma counted_prop_list_valuation_semantics :
    forall(LS : lambda_structure)(m : model)(n : nat)
          (fl : counted_list (lambda_formula V L) n)
          (fl_prop : every_nth prop_form (list_of_counted_list fl)),
      counted_map (form_semantics LS m) fl =
        counted_prop_list_valuation (coval m) n fl fl_prop.
  Proof.
    induction fl.
      intros fl_prop.
      simpl.
      trivial.
    simpl.
    intros fl_prop.
    f_equal.
      assert (H := every_nth_head _ _ _ fl_prop).
      destruct a.
            rewrite form_semantics_char.
            simpl.
            trivial.
          contradiction.
        contradiction.
      contradiction.
    apply IHfl.
  Qed.

  Lemma nth_counted_prop_list_valuation_tcc :
    forall(X : Type)(coval : X -> set V)(n : nat)
          (fl : counted_list (lambda_formula V L) n)
          (fl_prop : every_nth prop_form (list_of_counted_list fl))
          (i : nat),
      i < length (list_of_counted_list 
                    (counted_prop_list_valuation coval n fl fl_prop))
      -> i < length (list_of_counted_list fl).
  Proof.
    intros X coval0 n fl fl_prop i H.
    rewrite length_list_of_counted_list in *.
    trivial.
  Qed.

  Lemma nth_counted_prop_list_valuation :
    forall(X : Type)(coval : X -> set V)(n : nat)
          (fl : counted_list (lambda_formula V L) n)
          (fl_prop : every_nth prop_form (list_of_counted_list fl))
          (i : nat)
          (i_less : i < length (list_of_counted_list 
                    (counted_prop_list_valuation coval n fl fl_prop)))
          (x : X),
      nth (list_of_counted_list 
               (counted_prop_list_valuation coval n fl fl_prop))
        i i_less x
      = coval x 
          (prop_form_acc 
            (nth (list_of_counted_list fl) i
               (nth_counted_prop_list_valuation_tcc _ _ _ _ _ _ i_less))
            (fl_prop i
               (nth_counted_prop_list_valuation_tcc _ _ _ _ _ _ i_less))).
  Proof.
    induction fl.
      intros fl_prop i i_less x.
      exfalso.
      rewrite length_list_of_counted_list in i_less.
      omega.
    intros fl_prop i i_less x.
    destruct i.
      clear IHfl.
      simpl.
      f_equal.
      apply prop_form_acc_tcc_irr.
    simpl.
    generalize 
       (@nth_succ_tcc (set X) i
           (fun x0 : X =>
            coval0 x0
              (@prop_form_acc V L a
                 (@every_nth_head (lambda_formula V L) 
                    (@prop_form V L) a
                    (@list_of_counted_list (lambda_formula V L) n fl) fl_prop)))
           (@list_of_counted_list (set X) n
              (@counted_prop_list_valuation X coval0 n fl
                 (@every_nth_tail (lambda_formula V L) 
                    (@prop_form V L) a
                    (@list_of_counted_list (lambda_formula V L) n fl) fl_prop)))
           i_less).
    intros l.
    rewrite IHfl.
    clear IHfl.
    f_equal.
    generalize 
      (every_nth_tail prop_form a (list_of_counted_list fl) fl_prop i
        (nth_counted_prop_list_valuation_tcc X coval0 n fl
           (every_nth_tail prop_form a (list_of_counted_list fl) fl_prop) i l)).
    erewrite nth_tcc_irr.
    intros p.
    apply prop_form_acc_tcc_irr.
  Qed.


  (**************************************************************************)
  (** *** Semantics [TX,tau |= f] for modal formulas [f] *)
  (**************************************************************************)

  Fixpoint prop_modal_prop_valuation{X : Type}(LS : lambda_structure)
           (coval : X -> set V)
           (f : lambda_formula V L)(propm_f : prop_modal_prop f)
                                                           : set (obj T X) :=
    match f return prop_modal_prop f -> set (obj T X) with
      | lf_prop v => fun(H : prop_modal_prop (lf_prop v)) => 
          prop_modal_prop_tcc_prop v H
      | lf_neg f => fun(H : prop_modal_prop (lf_neg f)) => 
          set_inverse (prop_modal_prop_valuation LS coval f 
                                             (prop_modal_prop_tcc_neg f H))
      | lf_and f1 f2 => fun(H : prop_modal_prop (lf_and f1 f2)) => 
          intersection 
            (prop_modal_prop_valuation LS coval f1 
                  (prop_modal_prop_tcc_and_1 f1 f2 H))
            (prop_modal_prop_valuation LS coval f2 
                  (prop_modal_prop_tcc_and_2 f1 f2 H))
      | lf_modal op args => 
          fun(H : prop_modal_prop (lf_modal op args)) =>
            modal_semantics LS op 
              (counted_prop_list_valuation coval (arity L op) args 
                 (prop_modal_prop_tcc_modal op args H))
    end propm_f.

  Lemma prop_modal_prop_valuation_tcc_irr :
    forall(X : Type)(LS : lambda_structure)(coval : X -> set V)
          (f : lambda_formula V L)(propm_f_1 propm_f_2 : prop_modal_prop f),
      prop_modal_prop_valuation LS coval f propm_f_1 =
        prop_modal_prop_valuation LS coval f propm_f_2.
  Proof.
    induction f.
          intros propm_f_1 propm_f_2.
          contradiction.
        intros propm_f_1 propm_f_2.
        simpl.
        erewrite IHf.
        trivial.
      intros propm_f_1 propm_f_2.
      simpl.
      erewrite IHf1.
      erewrite IHf2.
      trivial.
    clear H.
    intros propm_f_1 propm_f_2.
    simpl.
    erewrite counted_prop_list_valuation_tcc_irr.
    trivial.
  Qed.


  Definition mod_seq_val{X : Type}(LS : lambda_structure)(coval : X -> set V)
                        (s : sequent V L)(nonempty_s : s <> [])
                      (propm_s : prop_modal_prop_sequent s) : set (obj T X) :=
    prop_modal_prop_valuation LS coval
      (or_formula_of_ne_sequent s nonempty_s)
      (prop_modal_prop_or_formula s nonempty_s propm_s).

  Lemma mod_seq_val_tcc_irr :
    forall(X : Type)(LS : lambda_structure)(coval : X -> set V)
          (s : sequent V L)(nonempty_s_1 nonempty_s_2 : s <> [])
          (propm_s_1 propm_s_2 : prop_modal_prop_sequent s),
      mod_seq_val LS coval s nonempty_s_1 propm_s_1 =
        mod_seq_val LS coval s nonempty_s_2 propm_s_2.
  Proof.
    intros X LS coval0 s nonempty_s_1 nonempty_s_2 propm_s_1 propm_s_2.
    unfold mod_seq_val in *.
    generalize (prop_modal_prop_or_formula s nonempty_s_1 propm_s_1).
    erewrite or_formula_of_ne_sequent_tcc_irr.
    intros p.
    erewrite prop_modal_prop_valuation_tcc_irr.
    trivial.
  Qed.


  Definition mod_seq_val_valid{X : Type}(LS : lambda_structure)
                              (coval : X -> set V)
                              (s : sequent V L)(nonempty_s : s <> [])
                              (propm_s : prop_modal_prop_sequent s) : Prop :=
    is_full_set (mod_seq_val LS coval s nonempty_s propm_s).

  Lemma mod_seq_val_valid_tcc_irr :
    forall(X : Type)(LS : lambda_structure)(coval : X -> set V)
          (s : sequent V L)(nonempty_s_1 nonempty_s_2 : s <> [])
          (propm_s_1 propm_s_2 : prop_modal_prop_sequent s),
      mod_seq_val_valid LS coval s nonempty_s_1 propm_s_1 <->
        mod_seq_val_valid LS coval s nonempty_s_2 propm_s_2.
  Proof.
    intros X LS coval0 s nonempty_s_1 nonempty_s_2 propm_s_1 propm_s_2.
    unfold mod_seq_val_valid in *.
    erewrite mod_seq_val_tcc_irr.
    split.
      eauto.
    eauto.
  Qed.


  (**************************************************************************)
  (** *** Equivalence of standard semantics with [TX,tau |= f]  *)
  (**************************************************************************)

  Lemma prop_modal_prop_semantics :
    forall(LS : lambda_structure)(m : model)
          (f : lambda_formula V L)(propm_f : prop_modal_prop f),
      set_equal (form_semantics LS m f)
                 (inv_img (coalg m) 
                           (prop_modal_prop_valuation LS (coval m) f propm_f)).
  Proof.
    induction f.
          intros propm_f.
          contradiction.
        intros propm_f.
        simpl.
        rewrite form_semantics_char.
        rewrite <- fibred_set_inverse.
        apply set_equal_set_inverse.
        apply IHf.
      intros propm_f.
      simpl.
      rewrite form_semantics_char.
      rewrite <- fibred_intersection.
      apply set_equal_intersection.
        apply IHf1.
      apply IHf2.
    intros propm_f.
    simpl.
    rewrite form_semantics_char.
    erewrite <- counted_prop_list_valuation_semantics.
    apply set_equal_refl.
  Qed.

  Lemma mod_val_semantics_valid :
    forall(nonempty_v : V)(LS : lambda_structure)(m : model)(s : sequent V L)
          (nonempty_s : s <> [])(propm_s : prop_modal_prop_sequent s),
      mod_seq_val_valid LS (coval m) s nonempty_s propm_s -> 
        valid_all_states nonempty_v LS m s.
  Proof.
    intros nonempty_v LS m s nonempty_s propm_s H.
    unfold valid_all_states, mod_seq_val_valid in *.
    unfold seq_semantics, mod_seq_val, is_full_set in *.
    destruct s.
      exfalso.
      auto.
    intros a.
    simpl in *.
    eapply prop_modal_prop_semantics.
    unfold inv_img in *.
    apply H.
  Qed.


  (**************************************************************************)
  (** *** Simplified [TX,tau |= f] sequent semantics  *)
  (**************************************************************************)

  Definition mod_val_pred{X : Type}(LS : lambda_structure)
             (coval : X -> set V)(x : obj T X)
             (f : lambda_formula V L)(propm_f : prop_modal_prop f) : Prop :=
    prop_modal_prop_valuation LS coval f propm_f x.

  Lemma mod_val_pred_tcc_irr :
    forall(X : Type)(LS : lambda_structure)(coval : X -> set V)
          (x : obj T X)(f : lambda_formula V L)
          (propm_f_1 propm_f_2 : prop_modal_prop f),
      mod_val_pred LS coval x f propm_f_1 ->
        mod_val_pred LS coval x f propm_f_2.
  Proof.
    intros X LS coval0 x f propm_f_1 propm_f_2 H.
    unfold mod_val_pred in *.
    erewrite prop_modal_prop_valuation_tcc_irr.
    eexact H.
  Qed.

  Definition simple_mod_seq_val{X : Type}(LS : lambda_structure)
             (coval : X -> set V)(s : sequent V L)
             (propm_s : prop_modal_prop_sequent s)(x : obj T X) : Prop :=
    some_neg_dep prop_modal_prop (mod_val_pred LS coval x) s propm_s.

  Lemma simple_mod_seq_val_tcc_irr :
    forall(X : Type)(LS : lambda_structure)(coval : X -> set V)(x : obj T X)
          (s : sequent V L)(propm_s_1 propm_s_2 : prop_modal_prop_sequent s),
      simple_mod_seq_val LS coval s propm_s_1 x ->
        simple_mod_seq_val LS coval s propm_s_2 x.
  Proof.
    intros X LS coval0 x.
    apply some_neg_dep_tcc_irr.
    apply mod_val_pred_tcc_irr.
  Qed.

  Lemma simple_mod_seq_val_nth_intro :
    forall(X : Type)(LS : lambda_structure)(coval : X -> set V)(x : obj T X)
          (s : sequent V L)(propm_s : prop_modal_prop_sequent s)
          (n : nat)(n_less : n < length s),
      prop_modal_prop_valuation LS coval 
                                (nth s n n_less) (propm_s n n_less) x ->
        simple_mod_seq_val LS coval s propm_s x.
  Proof.
    intros X LS coval0 x s propm_s n n_less H.
    eapply some_neg_dep_nth_intro.
      apply mod_val_pred_tcc_irr.
    eexact H.
  Qed.

  Lemma simple_mod_seq_val_head :
    forall(X : Type)(LS : lambda_structure)(coval : X -> set V)(x : obj T X)
          (f : lambda_formula V L)(s : sequent V L)
          (propm_fs : prop_modal_prop_sequent (f :: s)),
      prop_modal_prop_valuation LS coval f 
           (prop_modal_prop_sequent_head _ _ propm_fs) x ->
        simple_mod_seq_val LS coval (f :: s) propm_fs x.
  Proof.
    intros X LS coval0 x f s propm_fs H.
    eapply some_neg_dep_head.
      apply mod_val_pred_tcc_irr.
    eexact H.
  Qed.

  Lemma simple_mod_seq_val_tail :
    forall(X : Type)(LS : lambda_structure)(coval : X -> set V)(x : obj T X)
          (f : lambda_formula V L)(s : sequent V L)
          (propm_fs : prop_modal_prop_sequent (f :: s)),
      simple_mod_seq_val LS coval s 
                 (prop_modal_prop_sequent_tail _ _ propm_fs) x ->
        simple_mod_seq_val LS coval (f :: s) propm_fs x.
  Proof.
    intros X LS coval0 x f s propm_fs H.
    eapply some_neg_dep_tail.
      apply mod_val_pred_tcc_irr.
    eapply simple_mod_seq_val_tcc_irr in H.
    eexact H.
  Qed.

  Lemma simple_mod_seq_val_append_left :
    forall(X : Type)(LS : lambda_structure)(coval : X -> set V)(x : obj T X)
          (s1 s2 : sequent V L)
          (prop_s12 : prop_modal_prop_sequent (s1 ++ s2)),
      simple_mod_seq_val LS coval s2 
                 (prop_modal_prop_sequent_append_right _ _ prop_s12) x ->
        simple_mod_seq_val LS coval (s1 ++ s2) prop_s12 x.
  Proof.
    intros X LS coval0 x s1 s2 prop_s12 H.
    apply some_neg_dep_append_left.
      apply mod_val_pred_tcc_irr.
    eapply simple_mod_seq_val_tcc_irr in H.
    eexact H.
  Qed.

  Lemma simple_mod_seq_val_append_right :
    forall(X : Type)(LS : lambda_structure)(coval : X -> set V)(x : obj T X)
          (s1 s2 : sequent V L)
          (prop_s12 : prop_modal_prop_sequent (s1 ++ s2)),
      simple_mod_seq_val LS coval s1
                 (prop_modal_prop_sequent_append_left _ _ prop_s12) x ->
        simple_mod_seq_val LS coval (s1 ++ s2) prop_s12 x.
  Proof.
    intros X LS coval0 x s1 s2 prop_s12 H.
    apply some_neg_dep_append_right.
      apply mod_val_pred_tcc_irr.
    eapply simple_mod_seq_val_tcc_irr in H.
    eexact H.
  Qed.

  Lemma simple_mod_seq_val_length_case_intro :
    forall(X : Type)(LS : lambda_structure)(coval : X -> set V)
          (x : obj T X)(s : sequent V L)(propm_s : prop_modal_prop_sequent s),
      length s <> 1 ->
      ~~simple_mod_seq_val LS coval s propm_s x ->
        simple_mod_seq_val LS coval s propm_s x.
  Proof.
    intros X LS coval0 x s propm_s H H0.
    apply some_neg_dep_length_case_intro.
      intros len.
      contradiction.
    trivial.
  Qed.


  Definition simple_mod_seq_val_valid{X : Type}(LS : lambda_structure)
             (coval : X -> set V)(s : sequent V L)
             (propm_s : prop_modal_prop_sequent s) : Prop :=
    is_full_set (simple_mod_seq_val LS coval s propm_s).


  (**************************************************************************)
  (** *** Equivalence for simplified [TX,tau |= f] sequent semantics *)
  (**************************************************************************)

  Lemma mod_val_pred_or :
   forall(X : Type)(LS : lambda_structure)(coval : X -> set V)
         (x : obj T X)(f1 f2 : lambda_formula V L)
         (propm_f1 : prop_modal_prop f1)(propm_f2 : prop_modal_prop f2),
     mod_val_pred LS coval x (lambda_or f1 f2) 
                     (prop_modal_prop_lambda_or _ _ propm_f1 propm_f2)
        <->
     ~(~mod_val_pred LS coval x f1 propm_f1 /\ 
       ~mod_val_pred LS coval x f2 propm_f2).
  Proof.
    simpl.
    unfold set_inverse, intersection in *.
    split.
      intros H H0.
      destruct H0.
      apply H; clear H; split; intro H.
        apply H0.
        eapply mod_val_pred_tcc_irr.
        eexact H.
      apply H1.
      eapply mod_val_pred_tcc_irr.
      eapply H.
    intros H H0.
    destruct H0.
    apply H; clear H; split; intro H.
      apply H0.
      eapply mod_val_pred_tcc_irr.
      eexact H.
    apply H1.
    eapply mod_val_pred_tcc_irr.
    eapply H.
  Qed.

  Lemma simple_mod_seq_val_valid_correct :
    forall(X : Type)(LS : lambda_structure)
          (coval : X -> set V)(s : sequent V L)(nonempty_s : s <> [])
          (propm_s : prop_modal_prop_sequent s),
      mod_seq_val_valid LS coval s nonempty_s propm_s
        <-> simple_mod_seq_val_valid LS coval s propm_s.
  Proof.
    intros X LS coval0 s nonempty_s propm_s.
    destruct s as [| f].
      exfalso.
      auto.
    split.
      intros H x.
      unfold simple_mod_seq_val in *.
      rewrite some_neg_dep_cons_correct with 
                            (form_prop_or_formula_of_sequent_iter := 
                                  prop_modal_prop_or_formula_iter V L).
          eapply mod_val_pred_tcc_irr.
          apply H.
        apply mod_val_pred_tcc_irr.
      apply mod_val_pred_or.
    intros H x.
    specialize (H x).
    unfold simple_mod_seq_val in *.
    rewrite some_neg_dep_cons_correct with 
                            (form_prop_or_formula_of_sequent_iter := 
                                  prop_modal_prop_or_formula_iter V L) in H.
        eapply mod_val_pred_tcc_irr in H.
        apply H.
      apply mod_val_pred_tcc_irr.
    apply mod_val_pred_or.
  Qed.

  Lemma mod_seq_val_valid_reorder :
    forall(X : Type)(LS : lambda_structure)(coval : X -> set V)
          (s1 s2 : sequent V L)
          (nonempty_s1 : s1 <> [])(nonempty_s2 : s2 <> [])
          (propm_s1 : prop_modal_prop_sequent s1)
          (propm_s2 : prop_modal_prop_sequent s2),
      list_reorder s1 s2 ->
      mod_seq_val_valid LS coval s1 nonempty_s1 propm_s1 ->
        mod_seq_val_valid LS coval s2 nonempty_s2 propm_s2.
  Proof.
    intros X LS coval0 s1 s2 nonempty_s1 nonempty_s2 propm_s1 propm_s2 H H0.
    rewrite simple_mod_seq_val_valid_correct in *.
    intros x.
    eapply some_neg_dep_reorder.
        apply mod_val_pred_tcc_irr.
      eexact H.
    apply H0.
  Qed.


End Semantics.


Implicit Arguments modal_semantics [L T X].
Implicit Arguments set_equal_modal_semantics [L T].
Implicit Arguments fibred_semantics [L T].
Implicit Arguments state [V T].
Implicit Arguments coalg [V T].
Implicit Arguments coval [V T].
Implicit Arguments form_semantics [V L T].
Implicit Arguments valid_all_states [V L T].
Implicit Arguments valid_all_models [V L T].
Implicit Arguments state_seq_semantics [V L T].
Implicit Arguments state_seq_semantics_append [V L T].
Implicit Arguments propositional_valuation [V L X].
Implicit Arguments prop_seq_val_valid [V L X].
Implicit Arguments simple_prop_seq_val [V L X].
Implicit Arguments simple_prop_seq_val_tail [V L X].
Implicit Arguments prop_modal_prop_valuation [V L T X].
Implicit Arguments prop_modal_prop_semantics [V L T].
Implicit Arguments mod_seq_val_valid [V L T X].
Implicit Arguments simple_mod_seq_val [V L T X].
Implicit Arguments simple_mod_seq_val_valid [V L T X].
Implicit Arguments simple_mod_seq_val_append_left [V L T X].
Implicit Arguments simple_mod_seq_val_append_right [V L T X].
