(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: sequent_support.v,v 1.10 2013/04/10 11:17:16 tews Exp $
 *)


(** * Logic library *)

(** ** Sequent support

      This module defines a decidable equality on formulas and the
      support of a sequent.
*)

Require Export list_support modal_formulas.

Section Sequent_support.

  Variable V : Type.
  Variable L : modal_operators.

  Variable op_eq : eq_type (operator L).
  Variable v_eq : eq_type V.

  (***************************************************************************)
  (** ***  Equality over formulas, needed for the support in 5.3  *)
  (***************************************************************************)

  Fixpoint lambda_formula_bool(f1 f2 : lambda_formula V L) : bool :=
    match (f1, f2) with
      | (lf_prop v1, lf_prop v2) => if v_eq v1 v2 then true else false
      | (lf_neg f1, lf_neg f2) => lambda_formula_bool f1 f2
      | (lf_and f1 f2, lf_and f3 f4) =>
        lambda_formula_bool f1 f3 && 
        lambda_formula_bool f2 f4
      | (lf_modal op1 args1, lf_modal op2 args2) =>
        (if op_eq op1 op2 then true else false) &&
        ((fix eq_args(l1 l2 : nat)
                     (args1 : counted_list (lambda_formula V L) l1)
                     (args2 : counted_list (lambda_formula V L) l2) : bool :=
            match (args1, args2) with
              | (counted_nil, counted_nil) => true
              | (counted_cons n1 f1 args1, counted_cons n2 f2 args2) =>
                lambda_formula_bool f1 f2 &&
                eq_args n1 n2 args1 args2
              | _ => false
            end
        ) (arity L op1) (arity L op2) args1 args2)
      | _ => false
    end.

  Lemma lambda_formula_bool_char :
    forall (f1 f2 : lambda_formula V L), 
      f1 = f2 <-> lambda_formula_bool f1 f2 = true.
  Proof.
    induction f1.
          split.
            intros H.
            subst f2.
            simpl.
            rewrite eq_equal.
            trivial.
          destruct f2; try discriminate.
          intros H1.
          simpl in *.
          destruct (v_eq v v0).
            subst v0.
            trivial.
          discriminate.
        split.
          intros H.
          subst f2.
          apply IHf1.
          trivial.
        destruct f2; try discriminate.
        intros H1.
        simpl in *.
        f_equal.
        apply IHf1.
        trivial.
      split.
        intros H.
        subst f2.
        simpl.
        rewrite (iff_right (IHf1_1 _)); trivial.
        rewrite (iff_right (IHf1_2 _)); trivial.
      destruct f2; try discriminate.
      intros H1.
      simpl in *.
      rewrite andb_true_iff in H1.
      destruct H1.
      f_equal.
        apply IHf1_1; trivial.
      apply IHf1_2; trivial.
    split.
      intros H2.
      subst f2.
      simpl.
      rewrite eq_equal.
      simpl.
      induction args.
        trivial.
      apply andb_true_iff.
      split.
        apply (every_nth_head _ _ _ H); trivial.
      apply IHargs.
      apply (every_nth_tail _ _ _ H); trivial.
    destruct f2; try discriminate.
    rename op0 into op2, c into args2.
    intros H2.
    simpl in *.
    rewrite andb_true_iff in H2.
    destruct H2.
    destruct (op_eq op op2).
      subst op2.
      clear H0.
      f_equal.
      induction args.
        rewrite (counted_list_eta args2).
        trivial.
      assert (H2 := counted_list_eta args2).
      simpl in H2.
      decompose [ex] H2; clear H2.
      rename x into args2_hd, x0 into args2_tl.
      subst args2.
      rewrite andb_true_iff in H1.
      destruct H1.
      f_equal.
        apply (every_nth_head _ _ _ H); trivial.
      apply IHargs; trivial.
      apply (every_nth_tail _ _ _ H).
    discriminate.
  Qed.

  Definition lambda_formula_eq(f1 f2 : lambda_formula V L) 
                                                       : {f1 = f2}+{f1 <> f2}.
  Proof.
    destruct (lambda_formula_bool f1 f2) eqn:H.
      apply lambda_formula_bool_char in H.
      auto.
    assert (f1 <> f2).
      intros H0.
      eapply lambda_formula_bool_char in H0.
      rewrite H0 in H.
      discriminate.
    auto.
  Qed.


  (***************************************************************************)
  (** ***  Sequent support  *)
  (***************************************************************************)

  Definition sequent_support(s : sequent V L) : sequent V L :=
    list_support lambda_formula_eq s.

  Lemma sequent_support_correct_no_dup : forall(s : sequent V L),
    NoDup (sequent_support s).
  Proof.
    intros s.
    apply list_support_correct_no_dup.
  Qed.

  Lemma sequent_support_correct_content :
    forall(s : sequent V L),
      incl s (sequent_support s).
  Proof.
    apply list_support_correct_content.
  Qed.

  Lemma sequent_support_correct_subset : forall(s : sequent V L),
    multi_subset (sequent_support s) s.
  Proof.
    intros s.
    apply list_support_correct_subset.
  Qed.

  Lemma sequent_support_incl : 
    forall(s : sequent V L), incl (sequent_support s) s.
  Proof.
    apply list_support_incl.
  Qed.

  Lemma sequent_support_head_contract :
    forall(f : lambda_formula V L)(s : sequent V L),
      sequent_support (f :: f :: s) = 
        sequent_support (f :: s).
  Proof.
    intros f s.
    apply list_support_head_contract.
  Qed.

  Lemma sequent_support_reorder : forall(s1 s2 : sequent V L),
    list_reorder s1 s2 ->
      list_reorder (sequent_support s1) 
                   (sequent_support s2).
  Proof.
    intros s1 s2 H.
    apply list_support_reorder.
    trivial.
  Qed.

  Lemma sequent_support_destruct :
    forall(s1 : sequent V L), 
      exists(s2 : sequent V L),
        list_reorder s1 (s2 ++ sequent_support s1) /\
        incl s2 (sequent_support s1).
  Proof.
    intros s1.
    apply list_support_destruct.
  Qed.

  Lemma multi_subset_right_sequent_support :
    forall(s1 s2 : sequent V L),
      NoDup s1 ->
      incl s1 s2 ->
        multi_subset s1 (sequent_support s2).
  Proof.
    intros s1 s2.
    apply multi_subset_right_list_support.
  Qed.

  Lemma every_nth_sequent_support : 
    forall(P : lambda_formula V L -> Prop)(s : sequent V L),
      every_nth P s ->
        every_nth P (sequent_support s).
  Proof.
    apply every_nth_list_support.
  Qed.

  Lemma rank_sequent_sequent_support : 
    forall(s : sequent V L)(n : nat),
      rank_sequent n s ->
        rank_sequent n (sequent_support s).
  Proof.
    intros s n.
    apply every_nth_sequent_support.
  Qed.


  (***************************************************************************)
  (** *** Results for substitutions  *)
  (***************************************************************************)

  Lemma subst_eq_on_vars_support : 
    forall(sigma1 sigma2 : lambda_subst V L)(pv : list V),
      subst_eq_on_vars sigma1 sigma2 (list_support v_eq pv) ->
        subst_eq_on_vars sigma1 sigma2 pv.
  Proof.
    intros sigma1 sigma2 pv H v H0.
    apply H.
    apply list_support_correct_content.
    trivial.
  Qed.

  Lemma sequent_support_subst_sequent :
    forall(sigma : lambda_subst V L)(s : sequent V L),
      injective sigma ->
      simple_modal_sequent s ->
        sequent_support (subst_sequent sigma s) =
          subst_sequent sigma (sequent_support s).
  Proof.
    clear. 
    intros sigma s H H0.
    apply list_support_map.
    intros f1 f2 H1 H2 H3.
    apply injective_subst_neg_simple_modal in H3; trivial.
      apply (every_nth_In _ _ _ H0).
      trivial.
    apply (every_nth_In _ _ _ H0).
    trivial.
  Qed.


End Sequent_support.

Implicit Arguments lambda_formula_eq [V L].
Implicit Arguments sequent_support [V L].
Implicit Arguments sequent_support_correct_subset [V L].
Implicit Arguments sequent_support_reorder [V L].
Implicit Arguments sequent_support_destruct [V L].
