(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: sets.v,v 1.31 2013/04/10 11:17:16 tews Exp $
 *)

(** ** Sets

      I prefer to use my own set library. The collections in the Coq
      library are just too big and too complicated for me. Maybe in a
      few years.

      There is also a library for [dsets], decidable sets, for which
      member ship is decidable.
*)


Section Sets.

  Variable A : Type.


  (***************************************************************************)
  (** ***  definitions  *)
  (***************************************************************************)

  Definition set : Type := A -> Prop.

  Definition empty_set : set := fun _ => False.

  Definition full_set : set := fun _ => True.

  Definition is_full_set(P : set) : Prop := forall(a : A), P a.

  Definition add(a : A)(P : set) : set := fun(b : A) => a = b \/ P b.

  Definition subset(P Q : set) : Prop := forall(a : A), P a -> Q a.

  Definition union(P Q : set) : set := fun(a : A) => P a \/ Q a.

  Definition intersection(P Q : set) : set := fun(a : A) => P a /\ Q a.

  Definition set_inverse(P : set) : set := fun(a : A) => ~ (P a).


  (***************************************************************************)
  (** ***  subset properties  *)
  (***************************************************************************)

  Lemma subset_refl : forall(P : set), subset P P.
  Proof.
    unfold subset in *.
    trivial.
  Qed.

  Lemma subset_trans : forall(P1 P2 P3 : set),
    subset P1 P2 -> subset P2 P3 -> subset P1 P3.
  Proof.
    unfold subset in *.
    intros P1 P2 P3 H H0 a H1.
    auto.
  Qed.

  Lemma subset_add : forall(P : set)(a : A), subset P (add a P).
  Proof.
    unfold subset, add in *.
    intros P a a0 H.
    right.
    trivial.
  Qed.

  Lemma subset_union_left : forall(P Q : set), subset P (union P Q).
  Proof.
    intros P Q a H.
    left.
    trivial.
  Qed.

  Lemma subset_union_right : forall(P Q : set), subset Q (union P Q).
  Proof.
    intros P Q a H.
    right.
    trivial.
  Qed.

  Lemma subset_union_both : forall(P1 P2 Q1 Q2 : set),
    subset P1 Q1 -> subset P2 Q2 -> subset (union P1 P2) (union Q1 Q2).
  Proof.
    intros P1 P2 Q1 Q2 H H0 a H1.
    destruct H1.
      left.
      apply H.
      trivial.
    right.
    apply H0.
    trivial.
  Qed.

  Lemma subset_union_lub : forall(P1 P2 Q : set),
    subset P1 Q -> subset P2 Q -> subset (union P1 P2) Q.
  Proof.
    intros P1 P2 Q H H0 a H1.
    destruct H1.
      apply H.
      trivial.
    apply H0.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  set equality  *)
  (***************************************************************************)

  Definition set_equal(P Q : set) : Prop := 
    forall(a : A), P a <-> Q a.

  Lemma set_equal_refl : forall(P : set), set_equal P P.
  Proof.
    intros P a.
    apply iff_refl.
  Qed.

  Lemma set_equal_symm : forall(P Q : set),
    set_equal Q P -> set_equal P Q.
  Proof.
    intros P Q H a.
    specialize (H a).
    tauto.
  Qed.

  Lemma set_equal_trans : forall(P1 P2 P3 : set),
    set_equal P1 P2 -> set_equal P2 P3 -> set_equal P1 P3.
  Proof.
    unfold set_equal in *.
    intros P1 P2 P3 H H0 a.
    specialize (H a).
    specialize (H0 a).
    eapply iff_trans.
      eexact H.
    trivial.
  Qed.


  Lemma set_equal_subset_char : forall(P Q : set),
    set_equal P Q <-> (subset P Q /\ subset Q P).
  Proof.
    intros P Q.
    unfold set_equal, subset in *.
    split.
      intros H.
      split.
        intros a H0.
        apply H.
        trivial.
      intros a H0.
      apply H.
      trivial.
    intros H a.
    destruct H.
    split.
      apply H.
    apply H0.
  Qed.

  Lemma set_equal_subset : forall(P1 P2 : set)(Q1 Q2 : set),
    set_equal P1 P2 ->
    set_equal Q1 Q2 ->
    subset P1 Q1 ->
      subset P2 Q2.
  Proof.
    unfold set_equal, subset in *.
    intros P1 P2 Q1 Q2 H H0 H1 a H2.
    apply H0.
    apply H1.
    apply H.
    trivial.
  Qed.

  Lemma union_comm : forall(P Q : set),
    set_equal (union P Q) (union Q P).
  Proof.
    intros P Q a.
    unfold union in *.
    tauto.
  Qed.

  Lemma set_equal_union : forall(P1 P2 Q1 Q2 : set),
    set_equal P1 Q1 -> 
    set_equal P2 Q2 -> 
      set_equal (union P1 P2) (union Q1 Q2).
  Proof.
    intros P1 P2 Q1 Q2 H H0 a.
    specialize (H a).
    specialize (H0 a).
    unfold union in *.    
    tauto.
  Qed.

  Lemma set_equal_intersection : forall(P1 P2 Q1 Q2 : set),
    set_equal P1 Q1 -> 
    set_equal P2 Q2 -> 
      set_equal (intersection P1 P2) (intersection Q1 Q2).
  Proof.
    intros P1 P2 Q1 Q2 H H0 a.
    unfold intersection in *.
    specialize (H a).
    specialize (H0 a).
    tauto.
  Qed.

  Lemma set_equal_set_inverse : forall(P Q : set),
    set_equal P Q -> set_equal (set_inverse P) (set_inverse Q).
  Proof.
    intros P Q H a.
    unfold set_inverse in *.
    specialize (H a).
    tauto.
  Qed.

  Lemma set_equal_is_full : forall(P Q : set),
    set_equal P Q ->
    is_full_set P ->
      is_full_set Q.
  Proof.
    intros P Q H H0 a.
    apply H.
    apply H0.
  Qed.

  Lemma set_equal_union_subset_right : forall(P Q : set),
    subset Q P -> set_equal (union P Q) P.
  Proof.
    unfold subset, set_equal, union in *.
    intros P Q H a.
    specialize (H a).
    split.
      intros H0.
      tauto.
    auto.
  Qed.

  Lemma subset_empty_set : forall(P : set), subset empty_set P.
  Proof.
    unfold subset, empty_set in *.
    intros P a H.
    contradiction.
  Qed.

  Lemma set_equal_union_empty_right : forall(P Q : set),
    set_equal Q empty_set -> set_equal (union P Q) P.
  Proof.
    intros P Q H.
    apply set_equal_union_subset_right.
    eapply set_equal_subset.
        apply set_equal_symm.
        eexact H.
      apply set_equal_refl.
    apply subset_empty_set.
  Qed.

  Lemma set_equal_implies_subset : forall(P Q : set),
    set_equal P Q -> subset P Q.
  Proof.
    intros P Q H.
    apply (set_equal_subset_char P Q).
    trivial.
  Qed.

  Lemma set_equal_rw_r : forall(P1 P2 P3 : set),
    set_equal P2 P3 -> set_equal P1 P3 -> set_equal P1 P2.
  Proof.
    intros P1 P2 P3 H H0.
    eapply set_equal_trans.
      eexact H0.
    apply set_equal_symm.
    trivial.
  Qed.

  (***************************************************************************)
  (** ***  other  *)
  (***************************************************************************)

  Lemma intersection_complement : forall(P : set)(a : A),
    not (intersection P (set_inverse P) a).
  Proof.
    intros P a H.
    unfold intersection, set_inverse in *.
    destruct H.
    contradiction.
  Qed.

  Lemma set_inverse_intersection : forall(P Q : set)(a : A),
    union (set_inverse P) (set_inverse Q) a ->
      set_inverse (intersection P Q) a.
  Proof.
    intros P Q a H H0.
    destruct H0.
    destruct H.
      auto.
    auto.
  Qed.

  Lemma double_set_inverse : forall(P : set)(a : A),
    P a -> set_inverse (set_inverse P) a.
  Proof.
    intros P a H H0.
    apply H0.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  full set  *)
  (***************************************************************************)

  Lemma full_set_full : is_full_set full_set.
  Proof.
    intros a.
    cbv.
    trivial.
  Qed.

  Lemma full_subset_full : 
    forall(P : set), subset full_set P -> is_full_set P. 
  Proof.
    intros P H a.
    apply H.
    unfold full_set in *.
    trivial.
  Qed.

End Sets.

Implicit Arguments is_full_set [A].
Implicit Arguments add [A].
Implicit Arguments subset [A].
Implicit Arguments union [A].
Implicit Arguments intersection [[A]].
Implicit Arguments set_inverse [[A]].
Implicit Arguments set_equal [A].


Definition sets_containing{A : Type}(a : A) : set (set A) :=
  fun(s : set A) => s a.
