(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: step_semantics.v,v 1.29 2013/04/10 11:17:17 tews Exp $
 *)


(** ** N-step semantics 4.9 - 4.10, 4.12

      This module defines the n-step semantics (4.9) and proves its
      correctness statements 4.10 and 4.12. 

      There is also a lot of other material that is related to n-step
      semantics:
      - simplified n-step sequent semantics
      - upward correctness of G wrt. n-step semantics
      - relation to propositional models/validity
      - projections of the n-step semantics for modal and
        propositional sequents
      - relation to [X,tau |= f] and [TX,tau |= f] semantics via
        substitutions
*)

Require Export classic slice semantics build_proof propositional_models.

Section Step_semantics.

  Variable V : Type.
  Variable L : modal_operators.
  Variable T : functor.


  (***************************************************************************)
  (** *** Definition of n-step semantics, Def 4.9 *)
  (***************************************************************************)

  Fixpoint step_semantics(LS : lambda_structure L T)
                         (f : lambda_formula V L)
                         (n : nat)(rank : rank_formula (S n) f) :
                                         set (terminal_obj_sequence V T n) :=
    match f
      return rank_formula (S n) f -> set (terminal_obj_sequence V T n)
    with
      | lf_prop v => 
        fun(rank : rank_formula (S n) (lf_prop v)) =>
          inv_img (terminal_obj_sequence_pi_2 n) (sets_containing v)
      | lf_neg f => 
        fun(rank : rank_formula (S n) (lf_neg f)) =>
          set_inverse (step_semantics LS f n
                         (rank_formula_lf_neg_TCC (S n) f rank))
      | lf_and f1 f2 => 
        fun(rank : rank_formula (S n) (lf_and f1 f2)) =>
          intersection
            (step_semantics LS f1 n (rank_formula_and_left (S n) f1 f2 rank))
            (step_semantics LS f2 n (rank_formula_and_right (S n) f1 f2 rank))
      | lf_modal op args =>
        fun(rank : rank_formula (S n) (lf_modal op args)) =>
          match n 
            return rank_formula (S n) (lf_modal op args) -> 
                                            set (terminal_obj_sequence V T n)
          with
            | 0 => 
              fun(rank : rank_formula 1 (lf_modal op args)) => 
                rank_formula_modal_1_TCC _ _ _ rank
            | S n =>
              fun(rank : rank_formula (2 + n) (lf_modal op args)) =>
                inv_img 
                  (terminal_obj_sequence_pi_1 n)
                  (modal_semantics LS op
                     ((fix map_args(len : nat)
                               (args : counted_list (lambda_formula V L) len)
                               (rank : every_nth (rank_formula (S n))
                                                 (list_of_counted_list args))
                           : counted_list
                                   (set (terminal_obj_sequence V T n)) len :=
                       match args 
                         in counted_list _ len
                         return every_nth (rank_formula (S n))
                                          (list_of_counted_list args)
                             -> counted_list
                                  (set (terminal_obj_sequence V T n)) len
                       with
                         | counted_nil => fun _ => counted_nil
                         | counted_cons len f rargs =>
                           fun(rank : every_nth (rank_formula (S n))
                                      (f :: (list_of_counted_list rargs))) =>
                             counted_cons 
                               (step_semantics LS f n
                                                   (every_nth_head _ _ _ rank))
                               (map_args len rargs (every_nth_tail _ _ _ rank))
                       end rank
                      ) (arity L op) args 
                        (rank_formula_modal_args_TCC _ _ _ rank)
                     ))
          end rank
    end rank.

  (** Define the inner fixpoint on counted lists in the previous
      definition as separate function.
   *)
  Fixpoint step_semantics_args(LS : lambda_structure L T)(len : nat)
                              (args : counted_list (lambda_formula V L) len)
                              (n : nat)
                              (rank : every_nth (rank_formula (S n))
                                                (list_of_counted_list args))
                     : counted_list (set (terminal_obj_sequence V T n)) len :=
    match args in counted_list _ len
      return every_nth (rank_formula (S n)) (list_of_counted_list args) -> 
                          counted_list (set (terminal_obj_sequence V T n)) len
    with
      | counted_nil => fun _ => counted_nil
      | counted_cons len f rargs =>
        fun(rank : every_nth (rank_formula (S n)) 
                             (f :: (list_of_counted_list rargs))) =>
          counted_cons
            (step_semantics LS f n (every_nth_head _ _ _ rank))
            (step_semantics_args LS len rargs n (every_nth_tail _ _ _ rank))
    end rank.

  Lemma step_semantics_modal :
    forall(LS : lambda_structure L T)
          (op : operator L)
          (args : counted_list (lambda_formula V L) (arity L op))
          (n : nat)(rank : rank_formula (2 + n) (lf_modal op args)),
      step_semantics LS (lf_modal op args) (S n) rank =
      inv_img (terminal_obj_sequence_pi_1 n)
        (modal_semantics LS op
           (step_semantics_args LS (arity L op) args n
              (rank_formula_modal_args_TCC _ _ _ rank))).
  Proof.
    intros LS op args n rank.
    simpl.
    f_equal.
    f_equal.
    generalize (rank_formula_modal_args_TCC op args (S (S n)) rank).
    clear rank.
    induction args.
      intros e.
      trivial.
    intros e.
    simpl.
    rewrite IHargs.
    trivial.
  Qed.


  Lemma step_semantics_tcc_irr_eq :
    forall(LS : lambda_structure L T)(f : lambda_formula V L)
          (n : nat)(rank1 rank2 : rank_formula (S n) f),
      step_semantics LS f n rank1 = step_semantics LS f n rank2.
  Proof.
    induction f.
          intros n rank1 rank2.
          trivial.
        intros n rank1 rank2.
        simpl.
        erewrite IHf.
        trivial.
      intros n rank1 rank2.
      simpl.
      erewrite IHf1.
      erewrite IHf2.
      trivial.
    intros n rank1 rank2.
    destruct n.
      exfalso.
      eapply rank_formula_modal_1_TCC.
      eexact rank1.
    rewrite step_semantics_modal.
    rewrite step_semantics_modal.
    f_equal.
    f_equal.
    generalize (rank_formula_modal_args_TCC op args (2 + n) rank1).
    generalize (rank_formula_modal_args_TCC op args (2 + n) rank2).
    clear - H.
    revert args H.
    generalize (arity L op).
    clear. 
    induction args.
      intros H e e0.
      trivial.
    intros H e e0.
    simpl.
    f_equal.
      simpl in H.
      apply (every_nth_head _ _ _ H).
    apply IHargs.
    eapply every_nth_tail.
    eexact H.
  Qed.

  Lemma step_semantics_tcc_irr :
    forall(LS : lambda_structure L T)(n : nat)
          (x : terminal_obj_sequence V T n)(f : lambda_formula V L)
          (rank1 rank2 : rank_formula (S n) f),
      step_semantics LS f n rank1 x -> step_semantics LS f n rank2 x.
  Proof.
    intros LS n x f rank1 rank2 H.
    erewrite step_semantics_tcc_irr_eq.
    eexact H.
  Qed.

  Lemma nth_step_semantics_args :
    forall(LS : lambda_structure L T)
          (len : nat)(args : counted_list (lambda_formula V L) len)
          (n : nat)
          (rank : every_nth (rank_formula (S n)) (list_of_counted_list args))
          (i : nat)(i_less : i < len),
      nth (list_of_counted_list (step_semantics_args LS len args n rank)) 
        i (less_length_counted_list _ _ _ i_less) =
      step_semantics LS
        (nth (list_of_counted_list args) 
             i (less_length_counted_list _ _ _ i_less))
        n (rank i (less_length_counted_list _ _ _ i_less)).
  Proof.
    induction args.
      intros n rank i i_less.
      omega.
    intros n0 rank i i_less.
    destruct i.
      simpl.
      apply step_semantics_tcc_irr_eq.
    simpl.
    erewrite nth_tcc_irr.
    generalize (rank (S i) (less_length_counted_list (S i) (S n) 
                                 (counted_cons a args) i_less)).
    simpl.
    erewrite (nth_tcc_irr (list_of_counted_list args)).
    intros r.
    erewrite step_semantics_tcc_irr_eq. 
    apply IHargs with (i_less := lt_S_n _ _ i_less).
  Qed.


  Lemma step_semantics_lf_neg_rev :
    forall(LS : lambda_structure L T)(f : lambda_formula V L)(n : nat)
          (rank : rank_formula (S n) f)(x : terminal_obj_sequence V T n),
      not (step_semantics LS f n rank x) ->
        step_semantics LS (lf_neg f) n 
             (iff_right (rank_formula_lf_neg _ _) rank) x.
  Proof.
    intros LS f n rank x H.
    simpl.
    erewrite step_semantics_tcc_irr_eq.
    eexact H.
  Qed.

  Lemma step_semantics_false :
    forall(nonempty_v : V)(LS : lambda_structure L T)(n : nat)
          (rank : rank_formula (S n) (lambda_false nonempty_v))
          (x : terminal_obj_sequence V T n),
      not (step_semantics LS (lambda_false nonempty_v) n rank x).
  Proof.
    intros nonempty_v LS n rank x.
    simpl.
    apply intersection_complement.
  Qed.


  (***************************************************************************)
  (** **** N-step sequent semantics *)
  (***************************************************************************)

  Definition step_semantics_sequent
             (nonempty_v : V)(LS : lambda_structure L T)(s : sequent V L)
             (n : nat)(rank : rank_sequent (S n) s) :
                                         set (terminal_obj_sequence V T n) :=
    step_semantics LS (or_formula_of_sequent s nonempty_v) n
                (rank_formula_succ_or_formula_of_sequent n nonempty_v s rank).

  Lemma step_semantics_sequent_tcc_irr :
    forall(nonempty_v : V)(LS : lambda_structure L T)(s : sequent V L)
          (n : nat)(rank1 rank2 : rank_sequent (S n) s),
      step_semantics_sequent nonempty_v LS s n rank1 =
      step_semantics_sequent nonempty_v LS s n rank2.
  Proof.
    intros nonempty_v LS s n rank1 rank2.
    unfold step_semantics_sequent in *.
    apply step_semantics_tcc_irr_eq.
  Qed.


  Lemma step_semantics_sequent_empty :
    forall(nonempty_v : V)(LS : lambda_structure L T)
          (n : nat)(rank : rank_sequent (S n) []),
      set_equal (step_semantics_sequent nonempty_v LS [] n rank)
                 (empty_set (terminal_obj_sequence V T n)).
  Proof.
    intros nonempty_v LS n rank x.
    unfold step_semantics_sequent in *.
    simpl or_formula_of_sequent.
    split.
      intros H.
      exfalso.
      eapply step_semantics_false.
      eexact H.
    intros H.
    contradiction.
  Qed.


  (***************************************************************************)
  (** **** N-step validity *)
  (***************************************************************************)

  Definition step_semantics_valid
             (nonempty_v : V)(LS : lambda_structure L T)(s : sequent V L)
             (n : nat)(rank : rank_sequent (S n) s) : Prop :=
    is_full_set (step_semantics_sequent nonempty_v LS s n rank).

  Lemma step_semantics_valid_tcc_irr :
    forall(nonempty_v : V)(LS : lambda_structure L T)(s : sequent V L)
          (n : nat)(rank1 rank2 : rank_sequent (S n) s),
      step_semantics_valid nonempty_v LS s n rank1 ->
        step_semantics_valid nonempty_v LS s n rank2.
  Proof.
    unfold step_semantics_valid, is_full_set in *.
    intros nonempty_v LS s n rank1 rank2 H a.
    erewrite step_semantics_sequent_tcc_irr.
    apply H.
  Qed.

  Lemma step_semantics_valid_nonempty :
    forall(nonempty_v : V)(LS : lambda_structure L T)(s : sequent V L)
          (n : nat)(rank : rank_sequent (S n) s),
      step_semantics_valid nonempty_v LS s n rank ->
      (exists(x : terminal_obj_sequence V T n), True) ->
        s <> [].
  Proof.
    intros nonempty_v LS s n rank H H0 H1.
    subst s.
    destruct H0 as [x].
    specialize (H x).
    apply step_semantics_sequent_empty in H.
    contradiction.
  Qed.

  Lemma step_semantics_valid_taut :
    forall(nonempty_v : V)(LS : lambda_structure L T)
          (sigma : lambda_subst V L)(s : sequent V L)
          (k : nat)(rank : rank_sequent (S k) s),
      subst_Ax_set sigma s ->
        step_semantics_valid nonempty_v LS s k rank. 
  Proof.
    intros nonempty_v LS sigma s k rank H.
    unfold subst_Ax_set in *.
    destruct H as [v].
    revert rank.
    subst s.
    intros rank x.
    unfold step_semantics_sequent in *.
    simpl or_formula_of_sequent.
    assert (H := rank_sequent_head _ _ _ rank).
    assert (H0 := rank_sequent_tail _ _ _ rank).
    apply rank_sequent_head in H0.
    unfold lambda_or in *.
    simpl.
    erewrite step_semantics_tcc_irr_eq.
    apply intersection_complement.
  Qed.


  Definition step_semantics_valid_at_rank
             (nonempty_v : V)(LS : lambda_structure L T)(n : nat)
                                                        : set (sequent V L) :=
    fun(s : sequent V L) => 
      rank # rank_sequent (S n) s 
          /#\ step_semantics_valid nonempty_v LS s n rank.
    

  (***************************************************************************)
  (** *** Lemma 4.10, relate n-step semantics to standard semantics *)
  (***************************************************************************)

  Definition slice_model(m : model V T) 
                                   : (state m) -> slice_obj_T T (state m) V :=
    pair (coalg m) (coval m).


  Lemma semantics_step_semantics :
    forall(LS : lambda_structure L T)(m : model V T)
          (f : lambda_formula V L)(n : nat)(rank : rank_formula (S n) f),
      set_equal
        (form_semantics LS m f)
        (inv_img (terminal_seq_cone (slice_model m) n)
          (step_semantics LS f n rank)).
  Proof.
    induction f.
          intros n rank.
          rewrite form_semantics_char.
          simpl.
          clear rank.
          rewrite inv_img_compose.
          eapply set_equal_rw_r.
            eapply set_equal_inv_img_feq.
            destruct n.
              unfold slice_model in *.
              simpl.
              rewrite pair_proj_right.
              unfold slice_final_map, terminal_obj_sequence in *.
              simpl.
              unfold slice_final in *.
              rewrite pair_proj_right.
              apply feq_reflexive.
            unfold slice_model in *.
            simpl.
            unfold slice_map_T in *.
            eapply feq_right_compose_left.
              apply feq_ftimes_compose_pair.
            unfold terminal_obj_sequence in *.
            simpl.
            unfold slice_obj_T in *.
            rewrite pair_proj_right.
            apply feq_id_left.
          apply set_equal_refl.
        intros n rank.
        rewrite form_semantics_char.
        simpl.
        rewrite <- fibred_set_inverse.
        apply set_equal_set_inverse.
        apply IHf.
      intros n rank.
      rewrite form_semantics_char.
      simpl.
      rewrite <- fibred_intersection.
      apply set_equal_intersection.
        apply IHf1.
      apply IHf2.
    intros n rank.
    rewrite form_semantics_char.
    destruct n.
      exfalso.
      eapply rank_formula_modal_1_TCC.
      eexact rank.
    rewrite step_semantics_modal.
    rewrite inv_img_compose.
    eapply set_equal_rw_r with (P3 := inv_img 
                  (fmap T (terminal_seq_cone (slice_model m) n) ∘ coalg m) _).
      apply set_equal_inv_img_feq.
      simpl.
      unfold slice_model at 2.
      unfold slice_map_T in *.
      eapply feq_right_compose_left.
        apply feq_ftimes_compose_pair.
      destruct n.
        simpl.
        unfold terminal_obj_sequence.
        simpl iter_obj_T.
        unfold slice_obj_T in *.
        rewrite pair_proj_left.
        apply feq_reflexive.
      simpl terminal_obj_sequence_pi_1.
      unfold terminal_obj_sequence.
      simpl iter_obj_T.
      unfold slice_obj_T in *.
      rewrite pair_proj_left.
      apply feq_reflexive.
    rewrite <- inv_img_compose.
    apply set_equal_inv_img_pred.
    eapply set_equal_rw_r.
      apply set_equal_symm.
      apply (fibred_semantics LS).
    apply (set_equal_modal_semantics LS).
    intros i i_less.
    generalize (less_length_counted_list i (arity L op)
                     (counted_map (form_semantics LS m) args) i_less).
    generalize 
         (less_length_counted_list i (arity L op)
           (counted_map (inv_img (terminal_seq_cone (slice_model m) n))
              (step_semantics_args LS (arity L op) args n
                 (rank_formula_modal_args_TCC op args (2 + n) rank))) i_less).
    repeat rewrite list_of_counted_list_map.
    intros l l0.
    rewrite nth_map.
    rewrite nth_map.    
    eapply set_equal_trans.
      apply (H _ _ n (rank_formula_modal_args_TCC op args (2 + n) rank _ _)).
    clear H.
    apply set_equal_inv_img_pred.
    erewrite nth_tcc_irr.
    rewrite nth_step_semantics_args with (i_less := i_less).
    clear l.
    generalize (rank_formula_modal_args_TCC op args (2 + n) rank i
                 (nth_map_tcc (form_semantics LS m) 
                              (list_of_counted_list args) i l0)).
    erewrite nth_tcc_irr.
    intros r.
    erewrite step_semantics_tcc_irr_eq.
    apply set_equal_refl.
  Qed.


  (***************************************************************************)
  (** *** Towards corollary 4.12 *)
  (***************************************************************************)

  Definition nth_unit_model(c : slice_unit_coalg V T)(n : nat) : model V T :=
    {| state := terminal_obj_sequence V T n;
       coalg := terminal_obj_sequence_pi_1 n ∘ unit_coalg_sequence c n;
       coval := terminal_obj_sequence_pi_2 (S n) ∘ unit_coalg_sequence c n
    |}.

  Lemma slice_model_nth_unit :
    forall(c : slice_unit_coalg V T)(n : nat),
      slice_model (nth_unit_model c n) ≡ unit_coalg_sequence c n.
  Proof.
    intros c n.
    unfold slice_model, nth_unit_model in *.
    simpl.
    destruct n.
      simpl (terminal_obj_sequence_pi_1 0).
      apply feq_pair_proj.
    simpl (terminal_obj_sequence_pi_1 (S n)).
    apply feq_symmetric.
    apply feq_pair_proj.
  Qed.

  (** **** Corollary 4.12 *)
  Lemma step_semantics_validity :
    forall(nonempty_v : V)(LS : lambda_structure L T)(n : nat)
          (s : sequent V L)(rank : rank_sequent (S n) s),
      non_trivial_functor T ->
        (valid_all_models nonempty_v LS s <-> 
         step_semantics_valid nonempty_v LS s n rank).
  Proof.
    intros nonempty_v LS n s rank H.
    split.
      intros H0.
      assert (c := non_empty_coalg T (slice_final V) H (nonempty_slice_final V)).
      destruct c as [ c ].
      clear H H1.
      specialize (H0 (nth_unit_model c n)).
      unfold valid_all_states, step_semantics_valid, is_full_set in *.
      unfold seq_semantics, step_semantics_sequent in *.
      intros a.
      specialize (H0 a).
      eapply semantics_step_semantics in H0.
      eapply set_equal_inv_img_feq in H0.
        rewrite inv_img_id in H0.
        eexact H0.
      apply feq_symmetric.
      eapply feq_transitive.
        apply feq_terminal_seq_cone.
        apply slice_model_nth_unit.
      apply unit_coalg_seq_cone_identity.
    clear H.
    intros H.
    unfold valid_all_models, valid_all_states in *.
    unfold seq_semantics, step_semantics_sequent, is_full_set in *.
    intros m a.
    eapply semantics_step_semantics.
    unfold inv_img in *.
    apply H.
  Qed.


  (**************************************************************************)
  (** *** Simplified n-step sequent semantics *)
  (**************************************************************************)
  
  Definition state_seq_step_form_pred(LS : lambda_structure L T)(n : nat)
                                     (x : terminal_obj_sequence V T n)
                                     (f : lambda_formula V L)
                                     (rank_f : rank_formula (S n) f) : Prop :=
         step_semantics LS f n rank_f x.

  Definition state_seq_step_semantics(LS : lambda_structure L T)
                         (s : sequent V L)
                         (n : nat)(rank : rank_sequent (S n) s)
                         (x : terminal_obj_sequence V T n) : Prop :=
    some_neg_dep (rank_formula (S n)) 
                 (state_seq_step_form_pred LS n x) s rank.

  Lemma state_seq_step_semantics_tcc_irr :
    forall(LS : lambda_structure L T)(s : sequent V L)
          (n : nat)(rank1 rank2 : rank_sequent (S n) s)
          (x : terminal_obj_sequence V T n),
      state_seq_step_semantics LS s n rank1 x -> 
        state_seq_step_semantics LS s n rank2 x.
  Proof.
    intros LS s n rank1 rank2 x.
    apply some_neg_dep_tcc_irr.
    apply step_semantics_tcc_irr.
  Qed.

  Lemma state_seq_step_semantics_list_reorder :
    forall(LS : lambda_structure L T)(s1 s2 : sequent V L)
          (n : nat)(rank : rank_sequent (S n) s2)
          (x : terminal_obj_sequence V T n)
          (reorder : list_reorder s1 s2),
      state_seq_step_semantics LS s1 n 
            (iff_left (rank_sequent_list_reorder _ _ (S n) reorder) rank) x ->
        state_seq_step_semantics LS s2 n rank x.
  Proof.
    intros LS s1 s2 n rank x reorder.
    apply some_neg_dep_reorder.
      apply step_semantics_tcc_irr.
    trivial.
  Qed.

  Lemma state_seq_step_semantics_list_reorder_rev :
    forall(LS : lambda_structure L T)(s1 s2 : sequent V L)
          (n : nat)(rank : rank_sequent (S n) s1)
          (x : terminal_obj_sequence V T n)
          (reorder : list_reorder s1 s2),
      state_seq_step_semantics LS s1 n rank x ->
        state_seq_step_semantics LS s2 n 
          (iff_right (rank_sequent_list_reorder _ _ (S n) reorder) rank) x.
  Proof.
    intros LS s1 s2 n rank x reorder.
    apply some_neg_dep_reorder.
      apply step_semantics_tcc_irr.
    trivial.
  Qed.

  Lemma state_seq_step_semantics_append :
    forall(LS : lambda_structure L T)(s1 s2 : sequent V L)
          (n : nat)(rank : rank_sequent (S n) (s1 ++ s2))
          (x : terminal_obj_sequence V T n),
      state_seq_step_semantics LS (s1 ++ s2) n rank x ->
        ~(~state_seq_step_semantics LS s1 n
              (rank_sequent_append_left _ _ _ rank) x /\
          ~state_seq_step_semantics LS s2 n
              (rank_sequent_append_right _ _ _ rank) x).
  Proof.
    intros LS s1 s2 n rank x H H0.
    destruct H0.
    apply some_neg_dep_append in H.
      apply H; clear H; split; intro H.
        apply H0; clear H0 H1.
        eapply state_seq_step_semantics_tcc_irr.
        eexact H.
      apply H1; clear H0 H1.
      eapply state_seq_step_semantics_tcc_irr.
      eexact H.
    apply step_semantics_tcc_irr.
  Qed.

  Lemma state_seq_step_semantics_append_right :
    forall(LS : lambda_structure L T)(s1 s2 : sequent V L)
          (n : nat)(rank_s1 : rank_sequent (S n) s1)
          (rank_app : rank_sequent (S n) (s1 ++ s2))
          (x : terminal_obj_sequence V T n),
      state_seq_step_semantics LS s1 n rank_s1 x ->
        state_seq_step_semantics LS (s1 ++ s2) n rank_app x.
  Proof.
    intros LS s1 s2 n rank_s1 rank_app x H.
    apply some_neg_dep_append_right.
      apply step_semantics_tcc_irr.
    eapply state_seq_step_semantics_tcc_irr.
    eexact H.
  Qed.

  Lemma state_seq_step_semantics_append_left :
    forall(LS : lambda_structure L T)(s1 s2 : sequent V L)
          (n : nat)(rank_s2 : rank_sequent (S n) s2)
          (rank_app : rank_sequent (S n) (s1 ++ s2))
          (x : terminal_obj_sequence V T n),
      state_seq_step_semantics LS s2 n rank_s2 x ->
        state_seq_step_semantics LS (s1 ++ s2) n rank_app x.
  Proof.
    intros LS s1 s2 n rank_s2 rank_app x H.
    apply some_neg_dep_append_left.
      apply step_semantics_tcc_irr.
    eapply state_seq_step_semantics_tcc_irr.
    eexact H.
  Qed.

  Lemma state_seq_step_semantics_long_neg_intro :
    forall(LS : lambda_structure L T)(s : sequent V L)
          (n : nat)(rank : rank_sequent (S n) s)
          (x : terminal_obj_sequence V T n),
      length s <> 1 -> 
      ~~state_seq_step_semantics LS s n rank x ->
         state_seq_step_semantics LS s n rank x.
  Proof.
    intros LS s n rank x H H0.
    apply some_neg_dep_long_neg_intro.
      trivial.
    trivial.
  Qed.


  (**************************************************************************)
  (** *** Correctness of simplified n-step sequent semantics  *)
  (**************************************************************************)
  
  Lemma state_seq_step_form_pred_lambda_or :
    forall(LS : lambda_structure L T)(n : nat)
          (x : terminal_obj_sequence V T n)(f1 f2 : lambda_formula V L)
          (rf1 : rank_formula (S n) f1)(rf2 : rank_formula (S n) f2),
      state_seq_step_form_pred LS n x (lambda_or f1 f2)
                              (rank_formula_lambda_or (S n) f1 f2 rf1 rf2)
        <->
      ~(~state_seq_step_form_pred LS n x f1 rf1 /\
        ~ state_seq_step_form_pred LS n x f2 rf2).
  Proof.
    intros LS n x f1 f2 rf1 rf2.
    unfold state_seq_step_form_pred in *.
    simpl.
    unfold set_inverse, intersection in *.
    erewrite step_semantics_tcc_irr_eq.
    erewrite (step_semantics_tcc_irr_eq _ f2).
    apply iff_refl.
  Qed.

  Lemma state_seq_step_semantics_correct :
    forall(nonempty_v : V)(LS : lambda_structure L T)(s : sequent V L)
          (n : nat)(rank : rank_sequent (S n) s)
          (x : terminal_obj_sequence V T n),
      state_seq_step_semantics LS s n rank x <->
        step_semantics_sequent nonempty_v LS s n rank x.
  Proof.
    intros nonempty_v LS s n rank x.
    unfold step_semantics_sequent, state_seq_step_semantics in *.
    eapply some_neg_dep_correct with (P := state_seq_step_form_pred LS n x)
                              (form_prop_or := rank_formula_lambda_or (S n)).
        unfold state_seq_step_form_pred in *.
        apply step_semantics_tcc_irr.
      apply state_seq_step_form_pred_lambda_or.
    intros nonempty_v0 t_false.
    apply step_semantics_false.
  Qed.
  

  (***************************************************************************)
  (** *** Simplified n-step semantics validity and reorder lemmas  *)
  (***************************************************************************)

  Definition state_seq_step_semantics_valid
                   (LS : lambda_structure L T)(s : sequent V L)
                   (n : nat)(rank : rank_sequent (S n) s) : Prop :=
    forall(x : terminal_obj_sequence V T n), 
      state_seq_step_semantics LS s n rank x.

  Lemma state_seq_step_semantics_valid_correct :
    forall(nonempty_v : V)(LS : lambda_structure L T)(s : sequent V L)
          (n : nat)(rank : rank_sequent (S n) s),
      step_semantics_valid nonempty_v LS s n rank <->
        state_seq_step_semantics_valid LS s n rank.
  Proof.
    intros nonempty_v LS s n rank.
    unfold step_semantics_valid, state_seq_step_semantics_valid, 
           is_full_set in *.
    split.
      intros H x.
      eapply state_seq_step_semantics_correct; trivial.
    intros H a.
    eapply state_seq_step_semantics_correct; trivial.
  Qed.


  Lemma step_semantics_valid_list_reorder :
    forall(nonempty_v : V)(LS : lambda_structure L T)(s1 s2 : sequent V L)
          (n : nat)(rank_1 : rank_sequent (S n) s1)
          (rank_2 : rank_sequent (S n) s2),
      list_reorder s1 s2 ->
      step_semantics_valid nonempty_v LS s1 n rank_1 ->
        step_semantics_valid nonempty_v LS s2 n rank_2.
  Proof.
    intros nonempty_v LS s1 s2 n rank_1 rank_2 H H0.
    rewrite state_seq_step_semantics_valid_correct in *; trivial.
    intros x.
    eapply state_seq_step_semantics_tcc_irr.
    eapply state_seq_step_semantics_list_reorder_rev with (reorder := H).
    apply H0.
  Qed.

  Lemma step_semantics_valid_at_rank_list_reorder :
    forall(nonempty_v : V)(LS : lambda_structure L T)(n : nat)
          (s1 s2 : sequent V L),
      list_reorder s1 s2 ->
      step_semantics_valid_at_rank nonempty_v LS n s1 ->
        step_semantics_valid_at_rank nonempty_v LS n s2.
  Proof.
    intros nonempty_v LS n s1 s2 H H0.
    unfold step_semantics_valid_at_rank in *.
    decompose [ex and or dep_and] H0; clear H0.
    constructor 1 with (a := iff_right (rank_sequent_list_reorder _ _ _ H) a).
    eapply step_semantics_valid_list_reorder.
        trivial.
      eexact H.
    eexact b.
  Qed.

  Lemma step_semantics_sequent_append :
    forall(nonempty_v : V)(LS : lambda_structure L T)(n : nat)
          (s1 s2 : sequent V L)(rank : rank_sequent (S n) (s1 ++ s2)),
      subset
        (step_semantics_sequent nonempty_v LS (s1 ++ s2) n rank)
        (set_inverse 
          (intersection
            (set_inverse
               (step_semantics_sequent nonempty_v LS s1 n
                  (rank_sequent_append_left _ _ _ rank)))
            (set_inverse
               (step_semantics_sequent nonempty_v LS s2 n
                  (rank_sequent_append_right _ _ _ rank))))).
  Proof.
    intros nonempty_v LS n s1 s2 rank x.
    intros H.
    rewrite <- state_seq_step_semantics_correct in H; trivial.
    apply state_seq_step_semantics_append in H.
    unfold set_inverse, intersection.
    rewrite <- state_seq_step_semantics_correct.
    rewrite <- state_seq_step_semantics_correct.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Upward correctness of n-step semantics and context lemma *)
  (***************************************************************************)

  Definition upward_step_correct_rule(nonempty_v : V)(r : sequent_rule V L)
                             (n : nat)(rank : rule_has_rank (S n) r) : Prop :=
    forall(LS : lambda_structure L T)(x : terminal_obj_sequence V T n),
      step_semantics_sequent nonempty_v LS (conclusion r) n (proj2 rank) x ->
        forall(i : nat)(i_less : i < length (assumptions r)),
          step_semantics_sequent nonempty_v LS (nth (assumptions r) i i_less)
            n (proj1 rank i i_less) x.

  Lemma upward_step_correct_context :
    forall(nonempty_v : V)(r : sequent_rule V L)(sl sr : sequent V L)(n : nat)
          (rank_context : rule_has_rank (S n) (rule_add_context sl sr r)),
      every_nth (fun(a : sequent V L) => a <> []) (assumptions r) ->
      upward_step_correct_rule nonempty_v r n 
        (rank_rule_add_context_rev sl sr r (S n) rank_context)
      -> 
        upward_step_correct_rule nonempty_v (rule_add_context sl sr r)
          n rank_context.
  Proof.
    unfold upward_step_correct_rule in *.
    intros nonempty_v r sl sr n rank_context H H0 LS x H1 i i_less.
    revert H1.
    generalize (proj1 rank_context i i_less) as rank_a.
    generalize (proj2 rank_context) as rank_c.
    unfold rule_add_context, add_context in i_less |- *.
    simpl in *.
    rewrite nth_map in *.
    generalize (nth_map_tcc (B := sequent V L)
                   (fun s : sequent V L => sl ++ s ++ sr)
                       (assumptions r) i i_less).
    clear i_less.
    intros i_less rank_c rank_a H1.
    assert (length (sl ++ (nth (assumptions r) i i_less) ++ sr) = 1 \/
            length (sl ++ (nth (assumptions r) i i_less) ++ sr) <> 1).
      omega.
    destruct H2.
      assert (sl = [] /\ sr = []).
        clear - H H2.
        specialize (H i i_less).
        simpl in *.
        rewrite app_length in H2.
        rewrite app_length in H2.
        destruct sl.
          destruct sr.
            auto.
          destruct (nth (assumptions r) i i_less).
            exfalso.
            auto.
          exfalso.
          simpl in *.
          omega.
        destruct (nth (assumptions r) i i_less).
          exfalso.
          auto.
        exfalso.
        simpl in *.
        omega.
      clear H2.
      destruct H3.
      subst sl sr.
      simpl in *.
      revert rank_c rank_a H1.
      rewrite app_nil_r.
      rewrite app_nil_r.
      intros rank_c rank_a H1.
      erewrite step_semantics_sequent_tcc_irr.
      apply H0.
      erewrite step_semantics_sequent_tcc_irr.
      eexact H1.
    rewrite <- state_seq_step_semantics_correct in H1 |- *; trivial.
    assert (H3 := list_reorder_append_3_middle (conclusion r) sl sr).
    apply state_seq_step_semantics_list_reorder_rev 
           with (reorder := H3) in H1.
    assert (H4 := list_reorder_append_3_middle sl (nth _ i i_less) sr).
    apply state_seq_step_semantics_list_reorder with (reorder := H4).
    apply state_seq_step_semantics_long_neg_intro.
      clear - H2.
      rewrite app_length in *.
      rewrite app_length in *.
      omega.
    intros H5.
    apply state_seq_step_semantics_append in H1.
    apply H1; clear H1; split; intro H1; apply H5; clear H5.
      eapply state_seq_step_semantics_append_right.
      rewrite state_seq_step_semantics_correct in *.
      apply H0.
      erewrite step_semantics_sequent_tcc_irr.
      eexact H1.
    eapply state_seq_step_semantics_append_left.
    eexact H1.
  Qed.


  (***************************************************************************)
  (*** **  upward correctness of n-step semantics for G  *)
  (***************************************************************************)

  Lemma upward_step_correct_ax : 
    forall(nonempty_v : V)(r : sequent_rule V L)
          (n : nat)(rank : rule_has_rank (S n) r),
      is_ax_rule r -> upward_step_correct_rule nonempty_v r n rank.
  Proof.
    unfold upward_step_correct_rule in *.
    intros nonempty_v r n rank H LS x H0 i i_less.
    exfalso.
    clear - H i_less.
    unfold is_ax_rule in *.
    destruct H.
    rewrite H in *.
    simpl in *.
    omega.
  Qed.

  Lemma upward_step_correct_and : 
    forall(nonempty_v : V)(r : sequent_rule V L)
          (n : nat)(rank : rule_has_rank (S n) r),
      is_and_rule r -> 
        upward_step_correct_rule nonempty_v r n rank.
  Proof.
    intros nonempty_v r n rank H.
    apply and_rule_context in H.
    decompose [ex] H; clear H.
    rename x into sl, x0 into sr, x1 into f1, x2 into f2.
    subst r.
    apply upward_step_correct_context.
      simpl.
      apply every_nth_cons.
        discriminate.
      apply every_nth_cons.
        discriminate.
      apply every_nth_empty.
    generalize (rank_rule_add_context_rev sl sr 
                         (bare_and_rule f1 f2) (S n) rank).
    intros r.
    clear. 
    unfold upward_step_correct_rule, bare_and_rule.
    simpl assumptions.
    simpl conclusion.
    intros LS x H i i_less.
    unfold step_semantics_sequent in *.
    simpl in H.
    destruct H.
    destruct i.
      simpl.
      eapply step_semantics_tcc_irr.
      eexact H.
    destruct i.
      simpl.
      eapply step_semantics_tcc_irr.
      eexact H0.
    exfalso.
    simpl in i_less.
    omega.
  Qed.

  Lemma upward_step_correct_neg_and : 
    forall(nonempty_v : V)(r : sequent_rule V L)
          (n : nat)(rank : rule_has_rank (S n) r),
      is_neg_and_rule r -> 
        upward_step_correct_rule nonempty_v r n rank.
  Proof.
    intros nonempty_v r n rank H.
    apply neg_and_rule_context in H.
    decompose [ex] H; clear H.
    rename x into sl, x0 into sr, x1 into f1, x2 into f2.
    subst r.
    apply upward_step_correct_context.
      simpl.
      apply every_nth_cons.
        discriminate.
      apply every_nth_empty.
    generalize (rank_rule_add_context_rev sl sr 
                         (bare_neg_and_rule f1 f2) (S n) rank).
    intros r.
    clear.
    unfold upward_step_correct_rule, bare_neg_and_rule.
    simpl assumptions.
    simpl conclusion.
    intros LS x H i i_less.
    destruct i.
      unfold step_semantics_sequent in *.
      simpl in *.
      intros H1.
      destruct H1.
      apply H0; clear H0; intros H0.
      apply H1; clear H1; intros H1.
      apply H; clear H; split.
        eapply step_semantics_tcc_irr.
        eexact H0.
      eapply step_semantics_tcc_irr.
      eexact H1.
    exfalso.
    simpl in *.
    omega.
  Qed.

  Lemma upward_step_correct_neg_neg : 
    forall(nonempty_v : V)(r : sequent_rule V L)
          (n : nat)(rank : rule_has_rank (S n) r),
      classical_logic ->
      is_neg_neg_rule r -> 
        upward_step_correct_rule nonempty_v r n rank.
  Proof.
    intros nonempty_v r n rank classic H.
    apply neg_neg_rule_context in H.
    decompose [ex] H; clear H.
    rename x into sl, x0 into sr, x1 into f.
    subst r.
    apply upward_step_correct_context.
      apply every_nth_cons.
        discriminate.
      apply every_nth_empty.
    generalize (rank_rule_add_context_rev sl sr 
                         (bare_neg_neg_rule f) (S n) rank).
    intros r.
    clear - classic.
    unfold upward_step_correct_rule, bare_neg_neg_rule.
    simpl assumptions.
    simpl conclusion.
    intros LS x H i i_less.
    unfold step_semantics_sequent in *.
    destruct i.
      simpl in *.
      apply double_set_inverse_rev in H; trivial.
      eapply step_semantics_tcc_irr.
      eexact H.
    exfalso.
    simpl in i_less.
    omega.
  Qed.

  Lemma upward_step_correct_G : 
    forall(nonempty_v : V)(r : sequent_rule V L)
          (n : nat)(rank : rule_has_rank (S n) r),
      classical_logic ->
      G_set V L r -> 
        upward_step_correct_rule nonempty_v r n rank.
  Proof.
    intros nonempty_v r n rank classic H.
    unfold G_set, union in *.
    decompose [or] H; clear H.
          apply upward_step_correct_ax; trivial.
        apply upward_step_correct_and; trivial.
      apply upward_step_correct_neg_and; trivial.
    apply upward_step_correct_neg_neg; trivial.
  Qed.


  (***************************************************************************)
  (** *** Preservation of n-step validity in proof trees *)
  (***************************************************************************)

  Lemma step_semantics_valid_G_rule_inductive :
    forall(nonempty_v : V)(LS : lambda_structure L T)(n : nat),
      classical_logic ->
        rule_inductive (G_n_set V L (S n)) 
                       (step_semantics_valid_at_rank nonempty_v LS n).
  Proof.
    unfold rule_inductive, step_semantics_valid_at_rank in *.
    intros nonempty_v LS n classic r H H0 i i_less.
    decompose [dep_and] H0; clear H0.
    unfold G_n_set, rank_rules in *.
    destruct H.
    apply dep_conj with (a := const_rank_G_set (S n) r H a i i_less).
    unfold step_semantics_valid, is_full_set in *.
    intros a0.
    unfold step_semantics_sequent.
    eapply step_semantics_tcc_irr.
    apply upward_step_correct_G with (rank := H0); trivial.
    erewrite step_semantics_sequent_tcc_irr.
    apply b.
  Qed.


  (***************************************************************************)
  (** *** Relation to propositional models *)
  (***************************************************************************)

  Lemma one_step_semantics_propositional :
    forall(LS : lambda_structure L T)(n : nat)(f : lambda_formula V L)
          (prop_f : propositional f)(rank : rank_formula (S n) f)
          (x : terminal_obj_sequence V T n),
      step_semantics LS f n rank x <->
      is_prop_model (terminal_obj_sequence_pi_2 n x) f prop_f.
  Proof.
    induction f.
          intros prop_f rank x.
          simpl.
          apply iff_refl.
        intros prop_f rank x.
        simpl.
        erewrite <- IHf; trivial.
        unfold set_inverse in *.
        apply iff_refl.
      intros prop_f rank x.
      simpl.
      erewrite <- IHf1; trivial.
      erewrite <- IHf2; trivial.
      unfold intersection.
      apply iff_refl.
    intros prop_f rank x.
    eapply propositional_tcc_modal.
    eexact prop_f.
  Qed.

  Lemma one_step_semantics_valid_propositional :
    forall(nonempty_v : V)(LS : lambda_structure L T)
          (n : nat)(s : sequent V L)
          (prop_s : propositional_sequent s)(rank : rank_sequent (S n) s),
      non_trivial_functor T ->
      step_semantics_valid nonempty_v LS s n rank ->
        prop_valid_sequent nonempty_v s prop_s.
  Proof.
    unfold step_semantics_valid, step_semantics_sequent, prop_valid_sequent, 
           prop_valid, is_prop_model, is_full_set in *.
    intros nonempty_v LS n s prop_s rank H H0 m.
    assert (H1 := nonempty_terminal_obj_sequence n m H).
    destruct H1.
    specialize (H0 x).
    erewrite one_step_semantics_propositional in H0.
    rewrite H1 in H0.
    eexact H0.
  Qed.


  (***************************************************************************)
  (** *** First projection for n-step semantics of [prop_modal] sequents *)
  (***************************************************************************)

  Lemma step_mod_sequent_semantics :
    forall(nonempty_v : V)(LS : lambda_structure L T)(s : sequent V L)
          (n : nat)(rank : rank_sequent (2 + n) s),
      prop_modal_sequent s ->
        exists(mod_sem_s : set (obj T (terminal_obj_sequence V T n))),
          set_equal (step_semantics_sequent nonempty_v LS s (S n) rank)
                     (inv_img (terminal_obj_sequence_pi_1 n) mod_sem_s).
  Proof.
    destruct s.
      intros n rank H.
      exists (empty_set _).
      intros x.
      eapply iff_trans.
        apply step_semantics_sequent_empty.
      apply iff_refl.
    revert l.
    induction s.
      intros f n rank H.
      unfold step_semantics_sequent in *.
      simpl.
      apply prop_modal_sequent_head in H.
      revert f n rank H.
      induction f.
            intros n rank H.
            contradiction.
          intros n rank H.
          simpl in *.
          assert (rank_sequent (2 + n) [f]).
            apply rank_sequent_cons.
              apply rank_formula_lf_neg.
              eapply rank_sequent_head.
              eexact rank.
            apply rank_sequent_empty.
          lapply (IHf n H0); trivial.
          clear IHf H.
          intros H.
          destruct H.
          eexists.
          eapply set_equal_trans.
            apply set_equal_set_inverse.
            erewrite step_semantics_tcc_irr_eq.
            eexact H.
          rewrite fibred_set_inverse.
          apply set_equal_refl.
        intros n rank H.
        simpl in *.
        assert (rank_sequent (2 + n) [f1]).
          apply rank_sequent_cons.
            eapply rank_formula_and_left.
            eapply rank_sequent_head.
            eexact rank.
          apply rank_sequent_empty.
        assert (rank_sequent (2 + n) [f2]).
          apply rank_sequent_cons.
            eapply rank_formula_and_right.
            eapply rank_sequent_head.
            eexact rank.
          apply rank_sequent_empty.
        destruct H.
        lapply (IHf1 n H0); trivial.
        lapply (IHf2 n H1); trivial.
        clear IHf1 IHf2 H.
        intros H H3.
        destruct H.
        destruct H3.
        eexists.
        eapply set_equal_trans.
          apply set_equal_intersection.
            erewrite step_semantics_tcc_irr_eq.
            eexact H3.
          erewrite step_semantics_tcc_irr_eq.
          eexact H.
        rewrite fibred_intersection.
        apply set_equal_refl.
      clear H.
      intros n rank H.
      rewrite step_semantics_modal.
      eexists.
      apply set_equal_refl.
    rename a into f2.
    intros f1 n rank H.
    unfold step_semantics_sequent in *.
    simpl in *.
    assert (rank_sequent (2 + n) ((lambda_or f1 f2) :: s)).
      apply rank_sequent_cons.
        apply rank_formula_lambda_or.
          eapply rank_sequent_head.
          eexact rank.
        eapply rank_sequent_head.
        eapply rank_sequent_tail.
        eexact rank.
      eapply rank_sequent_tail.
      eapply rank_sequent_tail.
      eexact rank.
    erewrite step_semantics_tcc_irr_eq.
    apply IHs with (rank := H0).
    clear - H.
    apply prop_modal_sequent_cons.
      simpl.
      split.
        simpl.
        eapply prop_modal_sequent_head.
        eexact H.
      simpl.
      eapply prop_modal_sequent_head.
      eapply prop_modal_sequent_tail.
      eexact H.
    eapply prop_modal_sequent_tail.
    eapply prop_modal_sequent_tail.
    eexact H.
  Qed.


  (***************************************************************************)
  (** *** Second projection of n-step semantics for propositional sequents *)
  (***************************************************************************)

  Lemma step_prop_sequent_semantics :
    forall(nonempty_v : V)(LS : lambda_structure L T)(s : sequent V L)
          (n : nat)(rank : rank_sequent (S n) s),
      propositional_sequent s ->
        exists(prop_sem_s : set (set V)),
          set_equal (step_semantics_sequent nonempty_v LS s n rank)
                     (inv_img (terminal_obj_sequence_pi_2 n) prop_sem_s).
  Proof.
    destruct s.
      intros n rank H.
      exists (empty_set (set V)).
      intros x.
      eapply iff_trans.
        apply step_semantics_sequent_empty.
      apply iff_refl.
    revert l.
    induction s.
      intros f n rank H.
      unfold step_semantics_sequent in *.
      simpl.
      apply propositional_sequent_head in H.
      revert f n rank H.
      induction f.
            intros n rank H.
            simpl.
            eexists.
            apply set_equal_refl.
          intros n rank H.
          simpl in *.
          assert (rank_sequent (S n) [f]).
            apply rank_sequent_cons.
              apply rank_formula_lf_neg.
              eapply rank_sequent_head.
              eexact rank.
            apply rank_sequent_empty.
          lapply (IHf n H0).
            clear IHf H.
            intros H.
            destruct H.
            eexists.
            eapply set_equal_trans.
              apply set_equal_set_inverse.
              erewrite step_semantics_tcc_irr_eq.
              eexact H.
            rewrite fibred_set_inverse.
            apply set_equal_refl.
          apply propositional_neg.
          trivial.
        intros n rank H.
        simpl in *.
        assert (rank_sequent (S n) [f1]).
          apply rank_sequent_cons.
            eapply rank_formula_and_left.
            eapply rank_sequent_head.
            eexact rank.
          apply rank_sequent_empty.
        assert (rank_sequent (S n) [f2]).
          apply rank_sequent_cons.
            eapply rank_formula_and_right.
            eapply rank_sequent_head.
            eexact rank.
          apply rank_sequent_empty.
        lapply (IHf1 n H0).
          lapply (IHf2 n H1).
            clear IHf1 IHf2 H.
            intros H H2.
            destruct H.
            destruct H2.
            eexists.
            eapply set_equal_trans.
              apply set_equal_intersection.
                erewrite step_semantics_tcc_irr_eq.
                eexact H2.
              erewrite step_semantics_tcc_irr_eq.
              eexact H.
            rewrite fibred_intersection.
            apply set_equal_refl.
          eapply propositional_and_right.
          eexact H.
        eapply propositional_and_left.
        eexact H.
      intros n rank H0.
      eapply propositional_tcc_modal.
      eexact H0.
    rename a into f2.
    intros f1 n rank H.
    unfold step_semantics_sequent in *.
    simpl in *.
    assert (rank_sequent (S n) ((lambda_or f1 f2) :: s)).
      apply rank_sequent_cons.
        apply rank_formula_lambda_or.
          eapply rank_sequent_head.
          eexact rank.
        eapply rank_sequent_head.
        eapply rank_sequent_tail.
        eexact rank.
      eapply rank_sequent_tail.
      eapply rank_sequent_tail.
      eexact rank.
    erewrite step_semantics_tcc_irr_eq.
    apply IHs with (rank := H0).
    clear - H.
    apply propositional_sequent_cons.
      apply propositional_lambda_or.
        eapply propositional_sequent_head.
        eexact H.
      eapply propositional_sequent_head.
      eapply propositional_sequent_tail.
      eexact H.
    eapply propositional_sequent_tail.
    eapply propositional_sequent_tail.
    eexact H.
  Qed.


  (***************************************************************************)
  (** *** Relation to prop_modal_prop semantics [TX,tau |= f] via substitution 
   *)
  (***************************************************************************)

  Definition n_step_subst_coval(LS : lambda_structure L T)
                    (sigma : lambda_subst V L)
                    (n : nat)(rank_sigma : rank_subst (S n) sigma)
                                  : (terminal_obj_sequence V T n) -> set V :=
    fun(x : terminal_obj_sequence V T n)(v : V) =>
      step_semantics LS (sigma v) n (rank_sigma v) x.

  
  Lemma subst_coval_modal_step_semantics :
    forall(LS : lambda_structure L T)(f : lambda_formula V L)
          (propm_f : prop_modal_prop f)(sigma : lambda_subst V L)(n : nat)
          (rank_subst_f : rank_formula (2 + n) (subst_form sigma f))
          (rank_sigma : rank_subst (S n) sigma)
          (sx : obj T (terminal_obj_sequence V T n))(pv : set V),
      step_semantics LS (subst_form sigma f) (S n) rank_subst_f (sx, pv) <->
      prop_modal_prop_valuation LS (n_step_subst_coval LS sigma n rank_sigma)
        f propm_f sx.
  Proof.
    induction f.
          intros propm_f sigma n rank_subst_f rank_sigma sx pv.
          contradiction.
        intros propm_f sigma n rank_subst_f rank_sigma sx pv.
        revert rank_subst_f.
        rewrite subst_form_char.
        intros rank_subst_f.
        simpl.
        unfold set_inverse in *.
        apply iff_neg.
        apply IHf.
      intros propm_f sigma n rank_subst_f rank_sigma sx pv.
      revert rank_subst_f.
      rewrite subst_form_char.
      intros rank_subst_f.
      simpl.
      unfold intersection in *.
      apply iff_and.
        apply IHf1.
      apply IHf2.
    clear H.
    intros propm_f sigma n rank_subst_f rank_sigma sx pv.
    assert (H := prop_modal_prop_tcc_modal _ _ propm_f).
    revert rank_subst_f.
    rewrite subst_form_char.
    intros rank_subst_f.
    rewrite step_semantics_modal.
    unfold inv_img in *.
    simpl.
    rewrite terminal_obj_sequence_pi_1_char.
    apply (set_equal_modal_semantics LS _ op).
    clear. 
    intros i i_less x.
    rewrite nth_step_semantics_args.
    rewrite nth_counted_prop_list_valuation.
    generalize 
      (rank_formula_modal_args_TCC op (counted_map (subst_form sigma) args)
        (S (S n)) rank_subst_f i
        (less_length_counted_list i (arity L op)
           (counted_map (subst_form sigma) args) i_less)).
    generalize (less_length_counted_list i (arity L op)
                  (counted_map (subst_form sigma) args) i_less).
    rewrite list_of_counted_list_map.
    intros l.
    rewrite nth_map.
    generalize (nth_map_tcc (subst_form sigma) (list_of_counted_list args)
                     i l).
    clear l.
    intros l r.
    generalize
        (prop_modal_prop_tcc_modal op args propm_f i
           (nth_counted_prop_list_valuation_tcc V L
              (terminal_obj_sequence V T n)
              (n_step_subst_coval LS sigma n rank_sigma) 
              (arity L op) args (prop_modal_prop_tcc_modal op args propm_f) i
              (less_length_counted_list i (arity L op)
                 (counted_prop_list_valuation V L
                    (n_step_subst_coval LS sigma n rank_sigma) 
                    (arity L op) args
                    (prop_modal_prop_tcc_modal op args propm_f)) i_less))).
    generalize
           (nth_counted_prop_list_valuation_tcc V L
              (terminal_obj_sequence V T n)
              (n_step_subst_coval LS sigma n rank_sigma) 
              (arity L op) args (prop_modal_prop_tcc_modal op args propm_f) i
              (less_length_counted_list i (arity L op)
                 (counted_prop_list_valuation V L
                    (n_step_subst_coval LS sigma n rank_sigma) 
                    (arity L op) args
                    (prop_modal_prop_tcc_modal op args propm_f)) i_less)).
    intros l0.
    unfold n_step_subst_coval in *.
    rewrite nth_tcc_irr with (inside_2 := l).
    intros p.
    destruct (nth (list_of_counted_list args) i l); try contradiction.
    revert r.
    rewrite subst_form_char.
    intros r.
    simpl.
    erewrite step_semantics_tcc_irr_eq.
    apply iff_refl.
  Qed.

  Lemma subst_coval_modal_step_semantics_valid :
    forall(nonempty_v : V)(LS : lambda_structure L T)(s : sequent V L)
          (nonempty_s : s <> [])(propm_s : prop_modal_prop_sequent s)
          (sigma : lambda_subst V L)
          (n : nat)
          (rank_subst_s : rank_sequent (2 + n) (subst_sequent sigma s))
          (rank_sigma : rank_subst (S n) sigma),
      step_semantics_valid nonempty_v LS (subst_sequent sigma s) 
        (S n) rank_subst_s
      <->
      mod_seq_val_valid LS (n_step_subst_coval LS sigma n rank_sigma)
        s nonempty_s propm_s.
  Proof.
    intros nonempty_v LS s nonempty_s propm_s sigma n rank_subst_s rank_sigma.
    unfold step_semantics_valid, mod_seq_val_valid in *.
    unfold step_semantics_sequent, mod_seq_val, is_full_set in *.
    split.
      intros H a.
      assert (rank_formula (2 + n) 
                  (subst_form sigma (or_formula_of_ne_sequent s nonempty_s))).
        apply rank_formula_subst_formula_add with (n := 2) (k := S n).
            apply le_n_S.
            apply le_0_n.
          rewrite <- or_formula_of_sequent_nonempty
             with (nonempty_v := nonempty_v).
          apply rank_formula_succ_or_formula_of_sequent.
          apply rank_sequent_prop_modal_prop_sequent.
          trivial.
        trivial.
      rewrite <- subst_coval_modal_step_semantics 
         with (pv := empty_set V) (rank_subst_f := H0).
      revert H0.
      rewrite <- or_formula_subst_sequent.
      rewrite <- or_formula_of_sequent_nonempty
           with (nonempty_v := nonempty_v).
      intros H0.
      eapply step_semantics_tcc_irr.
      apply H.
    intros H a.
    destruct a as [tx pv].
    generalize(rank_formula_succ_or_formula_of_sequent (S n) nonempty_v
                                     (subst_sequent sigma s) rank_subst_s).
    rewrite or_formula_of_sequent_nonempty 
       with (nonempty_s := (or_formula_subst_sequent_tcc s sigma nonempty_s)).
    rewrite or_formula_subst_sequent.
    intros r.
    erewrite subst_coval_modal_step_semantics. 
    apply H.
  Qed.

  Lemma scssv_4_13_nonempty_tcc :
    forall(s simple_s : sequent V L)
          (sigma : lambda_subst V L),
      s <> [] ->
      subst_sequent sigma simple_s = s ->
        simple_s <> [].
  Proof.
    intros s simple_s sigma H H0 H1.
    apply H.
    rewrite <- H0.
    rewrite H1.
    trivial.
  Qed.


  Lemma subst_coval_modal_step_semantics_valid_for_4_13 :
    forall(nonempty_v : V)(LS : lambda_structure L T)
          (s simple_s : sequent V L)
          (nonempty_s : s <> [])
          (simple_simple_s : simple_modal_sequent simple_s)
          (n : nat)
          (rank : rank_sequent (2 + n) s)
          (sigma : lambda_subst V L)
          (rank_sigma : rank_subst (S n) sigma)
          (s_subst_eq : subst_sequent sigma simple_s = s),
      step_semantics_valid nonempty_v LS s (S n) rank ->
      mod_seq_val_valid LS (n_step_subst_coval LS sigma n rank_sigma)
        simple_s (scssv_4_13_nonempty_tcc _ _ _ nonempty_s s_subst_eq) 
        (simple_modal_sequent_is_prop_modal_prop _ simple_simple_s).
  Proof.
    intros nonempty_v LS s simple_s nonempty_s simple_simple_s n 
           rank sigma rank_sigma s_subst_eq H. 
    assert (rank_sequent (2 + n) (subst_sequent sigma simple_s)).
      rewrite s_subst_eq.
      trivial.
    rewrite <- subst_coval_modal_step_semantics_valid with (rank_subst_s := H0).
    generalize H0.
    rewrite s_subst_eq.
    intros H1.
    eapply step_semantics_valid_tcc_irr.
    eexact H.
  Qed.


  (***************************************************************************)
  (** *** Relation to propositional semantics [X,tau |= f] via substitution *)
  (***************************************************************************)

  Lemma subst_coval_prop_step_semantics :
    forall(LS : lambda_structure L T)(f : lambda_formula V L)
          (prop_f : propositional f)(sigma : lambda_subst V L)(n : nat)
          (rank_subst_f : rank_formula (S n) (subst_form sigma f))
          (rank_sigma : rank_subst (S n) sigma),
      set_equal
        (step_semantics LS (subst_form sigma f) n rank_subst_f)
        (propositional_valuation (n_step_subst_coval LS sigma n rank_sigma)
           f prop_f).
  Proof.
    induction f.
          intros prop_f sigma n rank_subst_f rank_sigma x.
          revert rank_subst_f.
          rewrite subst_form_char.
          intros rank_subst_f.
          unfold n_step_subst_coval in *.
          simpl.
          erewrite step_semantics_tcc_irr_eq.
          apply iff_refl.
        intros prop_f sigma n.
        rewrite subst_form_char.
        intros rank_subst_f rank_sigma.
        simpl.
        apply set_equal_set_inverse.
        apply IHf.
      intros prop_f sigma n.
      rewrite subst_form_char.
      intros rank_subst_f rank_sigma.
      simpl.
      apply set_equal_intersection.
        apply IHf1.
      apply IHf2.
    intros prop_f sigma n rank_subst_f rank_sigma.
    eapply propositional_tcc_modal.
    eexact prop_f.
  Qed.

  Lemma subst_coval_prop_step_semantics_valid :
    forall(nonempty_v : V)(LS : lambda_structure L T)(s : sequent V L)
          (prop_s : propositional_sequent s)
          (sigma : lambda_subst V L)
          (n : nat)
          (rank_subst_s : rank_sequent (S n) (subst_sequent sigma s))
          (rank_sigma : rank_subst (S n) sigma),
      prop_seq_val_valid nonempty_v 
                (n_step_subst_coval LS sigma n rank_sigma) s prop_s ->
        step_semantics_valid nonempty_v LS (subst_sequent sigma s) 
          n rank_subst_s.
  Proof.
    intros nonempty_v LS s prop_s sigma n rank_subst_s rank_sigma H.
    unfold prop_seq_val_valid, step_semantics_valid in *.
    unfold prop_seq_val, step_semantics_sequent, is_full_set in *.
    intros x.
    destruct s.
      exfalso.
      specialize (H x).
      unfold or_formula_of_sequent in H.
      apply propositional_valuation_false in H.
      trivial.
    rename l into f.
    assert (f :: s <> []).
      intros H0.
      discriminate.
    specialize (H x).
    assert (rank_formula (S n) (subst_form sigma 
                  (or_formula_of_sequent (f :: s) nonempty_v))).
      apply rank_formula_subst_formula_add with (n := 1) (k := S n).
          apply le_n_S.
          apply le_0_n.
        apply propositional_or_formula.
        trivial.
      trivial.
    apply subst_coval_prop_step_semantics with (rank_subst_f := H1) in H.
    revert H1 H.
    rewrite or_formula_of_sequent_nonempty with (nonempty_s := H0).
    rewrite <- or_formula_subst_sequent.
    intros H1 H.
    eapply step_semantics_tcc_irr.
    eexact H.
  Qed.

End Step_semantics.

Implicit Arguments step_semantics_valid [V L T].
Implicit Arguments slice_model [V T].
Implicit Arguments nth_unit_model [V T].
Implicit Arguments step_semantics_valid_nonempty [V L T].
Implicit Arguments step_semantics_valid_at_rank [V L T].
Implicit Arguments step_semantics_valid_G_rule_inductive [V L T].
Implicit Arguments step_mod_sequent_semantics [V L T].
Implicit Arguments step_prop_sequent_semantics [V L T].
Implicit Arguments n_step_subst_coval [V L T].
Implicit Arguments subst_coval_modal_step_semantics_valid_for_4_13 [V L T].
