(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: weakening.v,v 1.44 2013/04/10 12:06:16 tews Exp $
 *)


(** * Properties of coalgebraic logics, Section 3 *)

(** ** Weakening, 3.7 - 3.11

      This module formalizes the characterization of GR_n provability
      3.7 and 3.9, Lemma 3.10 and the weakening Lemma 3.11.
 *)


Require Export admissibility rule_sets.

Section Weakening.

  Variable V : Type.
  Variable L : modal_operators.

  (** need a decidable equality on propositional constants 
      for limit_subst in stratified_one_step_rules
   *)
  Variable v_eq : eq_type V.


  (***************************************************************************)
  (** *** Towards 3.7
         The characterization of ranked substitution instances of
         one-step rules.
    *)
  (***************************************************************************)

  (** Stratified weakend substitution rules, 
      right hand side in Lem 3.7, page 13. See also
      #<A HREF="one_step_conditions.html##simple_mod_weaken_rule"><spanclass="inlinecode">simple_mod_weaken_rule</span></A>#
       in module 
       #<A HREF="one_step_conditions.html"><spanclass="inlinecode">one_step_conditions</span></A>#.
   *)
  Definition rank_weaken_subst_rule(rules : set (sequent_rule V L))
                                   (n : nat)(posn : 0 < n)
                                                   : set (sequent_rule V L) :=
    fun(r_subst : sequent_rule V L) =>
      exists(r_base : sequent_rule V L)
            (sigma : lambda_subst V L)
            (delta : sequent V L),
        rules r_base /\
        rank_subst (pred n) sigma /\
        rank_sequent n delta /\
        assumptions r_subst = map (subst_sequent sigma) (assumptions r_base) /\
        list_reorder (conclusion r_subst) 
                     ((subst_sequent sigma (conclusion r_base)) ++ delta).

  Lemma rank_sequent_osr_subst_conclusion :
    forall(r : sequent_rule V L)(sigma : lambda_subst V L)
          (delta s : sequent V L)(n : nat),
      one_step_rule r ->
      rank_subst (S n) sigma ->
      rank_sequent (2 + n) delta ->
      list_reorder s (subst_sequent sigma (conclusion r) ++ delta) ->
        rank_sequent (2 + n) s.
  Proof.
    intros r sigma delta s n H H0 H1 H2.
    eapply rank_sequent_list_reorder.
      eexact H2.
    apply rank_sequent_append.
      apply rank_subst_conclusion; trivial.
      apply lt_n_S.
      apply lt_0_Sn.
    trivial.
  Qed.

  (* 
   * Lemma rank_weaken_subst_rule_rank_ge_2 :
   *   forall(rules : set (sequent_rule V L))(n : nat)(posn : 0 < n)
   *         (r : sequent_rule V L),
   *     one_step_rule_set v_eq rules ->
   *     rank_weaken_subst_rule rules n posn r ->
   *       1 < n.
   * Proof.
   *   intros rules n posn r H H0.
   *   unfold rank_weaken_subst_rule in *.
   *   decompose [ex and or dep_and] H0; clear H0.
   *   rename x into rbase, x0 into sigma, x1 into delta.
   *   eapply subst_conclusion_rank_ge_2.
   *     apply H.
   *     eexact H2.
   *)


  (** **** Lemma 3.7, page 13
          Stratified weakend one-step substitution rules.

          This result needs the nonempty [V] to rule out a border case
          for rank 0 ([n = 1]) in the inclusion from right to left 
    *)
  Lemma stratified_one_step_rules :
    forall(nonempty_v : V)(rules : set (sequent_rule V L))
          (n : nat)(npos : 0 < n),
      one_step_rule_set rules ->
      set_equal (rank_rules n (weaken_subst_rule rules))
        (rank_weaken_subst_rule rules n npos).
  Proof.
    intros v rules n H H0 r.
    split.
      intros H1.
      unfold rank_weaken_subst_rule, rank_rules, rule_has_rank, 
             weaken_subst_rule in *.
      decompose [ex and] H1; clear H1.
      rename x into r_base, x0 into sigma, x1 into delta.
      exists r_base.
      exists (limit_subst v_eq (prop_var_sequent (conclusion r_base)) sigma).
      exists delta.
      repeat split.
              trivial.
            specialize (H0 _ H2).
            unfold one_step_rule in *.
            apply rank_subst_limit_subst_rule; auto.
                tauto.
              tauto.
            eapply rank_sequent_append_left.
            erewrite <- rank_sequent_list_reorder.
              eexact H7.
            eexact H6.
          rewrite (rank_sequent_list_reorder _ _ _ H6) in H7.
          eapply rank_sequent_append_right.
          eexact H7.
        rewrite map_subst_sequent_limit_eq.
            trivial.
          trivial.
        apply incl_flatten_every.
        specialize (H0 _ H2).
        unfold one_step_rule in *.
        decompose [and] H0; clear H0.
        apply every_nth_map.
        trivial.
      rewrite subst_sequent_limit_eq.
          trivial.
        trivial.
      apply incl_refl.
    intros H1.
    unfold rank_weaken_subst_rule, rank_rules, weaken_subst_rule in *.
    decompose [ex and] H1; clear H1.
    rename x into r_base, x0 into sigma, x1 into delta.
    split.
      exists r_base, sigma, delta.
      auto.
    split.
      rewrite H5.
      eapply rank_subst_assumptions.
        apply H0.
        trivial.
      apply (rank_subst_ge (pred n)).
        omega.
      trivial.
    assert (H8 := nonempty_rank_subst v _ _ H2).
    destruct n.
      omega.
    destruct n.
      omega.
    apply rank_sequent_osr_subst_conclusion with (4 := H7); auto.
  Qed.


  (***************************************************************************)
  (** *** Towards 3.10, the equivalence of GR_n and GR
   *)
  (***************************************************************************)

  (** Backward application of rules in S(R) decreases the rank by one.
      Needed for 3.10, where the rank must not increase.
   *)
  Lemma decrease_rank_R :
    forall(n : nat)(r : sequent_rule V L)(rules : set (sequent_rule V L)),
      one_step_rule_set rules ->
      weaken_subst_rule rules r ->
      rank_sequent n (conclusion r) ->
        every_nth (rank_sequent (pred n)) (assumptions r).
  Proof.
    intros n r rules H H0 H1.
    unfold weaken_subst_rule in *.
    decompose [ex and or dep_and] H0; clear H0.
    rename x into r_base, x0 into sigma, x1 into delta.
    specialize (H _ H3).
    rewrite H2; clear H2.
    erewrite <- map_subst_sequent_limit_eq with (v_eq := v_eq).
      apply rank_subst_assumptions.
        trivial.
      unfold one_step_rule in *.
      decompose [and] H; clear H.
      apply rank_subst_limit_subst_rule.
          eexact H7.
        trivial.
      eapply rank_sequent_append_left.
      erewrite <- rank_sequent_list_reorder.
        eexact H1.
      eexact H5.
    unfold one_step_rule in *.
    decompose [and] H; clear H.
    apply incl_flatten_every.
    apply every_nth_map.
    trivial.
  Qed.

  Lemma const_rank_R :
    forall(n : nat)(r : sequent_rule V L)(rules : set (sequent_rule V L)),
      one_step_rule_set rules ->
      weaken_subst_rule rules r ->
      rank_sequent n (conclusion r) ->
        every_nth (rank_sequent n) (assumptions r).
  Proof.
    intros n r rules H H0 H1.
    eapply every_nth_mono with (P := rank_sequent (pred n)).
      apply rank_sequent_mono.
      omega.
    eapply decrease_rank_R; eauto.
  Qed.


  (** Backward application of rules in GR does not increase the rank.
      Implicit property in 3.10.
   *)
  Lemma const_rank_GR :
    forall(n : nat)(r : sequent_rule V L)(rules : set (sequent_rule V L)),
      one_step_rule_set rules ->
      GR_set rules r ->
      rank_sequent n (conclusion r) ->
        every_nth (rank_sequent n) (assumptions r).
  Proof.
    intros n r rules H H0 H1.
    destruct H0.
      apply const_rank_G_set; auto.
    eapply const_rank_R; eauto.
  Qed.


  (***************************************************************************)
  (** **** Lemma 3.10, equivalence of GR_n and GR
          Page 14. See also 
          #<A HREF="cut_properties.html##rank_proof_GR"><spanclass="inlinecode">rank_proof_GR</span></A>#
           in module 
           #<A HREF="cut_properties.html"><spanclass="inlinecode">cut_properties</span></A>#.
   *)
  Lemma rank_proof_GR_fixed_rank :
    forall(rules : set (sequent_rule V L))(n : nat)(s : sequent V L),
      one_step_rule_set rules ->
      rank_sequent n s ->
        (provable (GR_set rules) (empty_sequent_set V L) s 
           <->
         provable (GR_n_set rules n) (empty_sequent_set V L) s).
  Proof.
    intros rules n s H H0.
    split; intros H1.
      unfold provable in *.
      destruct H1.
      clear H1.
      induction x using proof_sequent_ind.
        unfold empty_sequent_set in *.
        contradiction.
      lapply (every_nth_exists_apply _ _ _ H2); clear H2.
        intros H2.
        destruct H2.
        clear H2.
        assert (GR_n_set rules n r).
          unfold rank_rules.
          repeat split.
              trivial.
            eapply const_rank_GR; eauto.
          trivial.
        exists (rule (GR_n_set rules n)
                     (empty_sequent_set V L)
                     r H2 x).
        trivial.
      eapply const_rank_GR; eauto.
    eapply proof_mono_rules.
      apply subset_rank_rules.
    eexact H1.
  Qed.


  (***************************************************************************)
  (** *** Towards Lemma 3.9, page 14

         GR_n provability coincides with G_n provability with 
         S(R) n conclusions as axioms
   *)

  (** Define first the set of S(R) n conclusions *)
  Definition provable_subst_n_conclusions
             (rules : set (sequent_rule V L))(n : nat)(npos : 0 < n) 
                                                        : set (sequent V L) :=
    fun(s : sequent V L) =>
      exists(r : sequent_rule V L),
        rank_weaken_subst_rule rules n npos r /\
        s = conclusion r /\
        every_nth
          (provable (GR_n_set rules (pred n)) (empty_sequent_set V L))
          (assumptions r).


  (***************************************************************************)
  (** **** Some properties that are needed later *)

  Lemma rank_sequent_set_provable_subst_n_conclusions :
    forall(rules : set (sequent_rule V L))(n : nat)(npos : 0 < n),
      1 < n ->
      one_step_rule_set rules ->
        rank_sequent_set n (provable_subst_n_conclusions rules n npos).
  Proof.
    unfold rank_sequent_set, provable_subst_n_conclusions, 
           rank_weaken_subst_rule in *.
    intros rules n npos H H0 s H1.
    decompose [ex and] H1; clear H1.
    rename x into r, x0 into rbase, x1 into sigma, x2 into delta.
    clear H9.
    subst s.
    eapply rank_sequent_list_reorder.
      eexact H8.
    apply rank_sequent_append.
      apply rank_subst_conclusion; trivial.
      apply H0.
      trivial.
    trivial.
  Qed.

  Lemma multiset_provable_subst_n_conclusions :
    forall(rules : set (sequent_rule V L))(n : nat)(npos : 0 < n),
      sequent_multiset (provable_subst_n_conclusions rules n npos).
  Proof.
    clear. 
    unfold sequent_multiset in *.
    intros rules n npos s r H H0.
    unfold provable_subst_n_conclusions in *.
    decompose [ex and or dep_and] H; clear H.
    exists {| assumptions := assumptions x; conclusion := r |}.
    simpl.
    split.
      clear - H0 H1 H2.
      unfold rank_weaken_subst_rule in *.
      decompose [ex and or dep_and] H2; clear H2.
      rename x0 into rbase, x1 into sigma, x2 into delta.
      exists rbase, sigma, delta.
      repeat split; auto.
      simpl.
      rewrite <- H1 in *.
      clear - H0 H7.
      eapply list_reorder_trans.
        apply list_reorder_symm.
        eexact H0.
      trivial.
    auto.
  Qed.

  Lemma bounded_weakening_provable_subst_n_conclusions :
    forall(rules : set (sequent_rule V L))(n : nat)(npos : 0 < n),
      bounded_weakening_closed n (provable_subst_n_conclusions rules n npos).
  Proof.
    clear. 
    unfold bounded_weakening_closed.
    intros rules n npos s r f H H0 H1.
    unfold provable_subst_n_conclusions in *.
    decompose [ex and or dep_and] H; clear H.
    exists {| assumptions := assumptions x; conclusion := r |}.
    simpl.
    split.
      clear - H0 H1 H2 H3.
      unfold rank_weaken_subst_rule in *.
      decompose [ex and or dep_and] H3; clear H3.
      rename x0 into rbase, x1 into sigma, x2 into delta.
      exists rbase, sigma, (f :: delta).
      repeat split; auto.
        apply rank_sequent_cons.
          trivial.
        trivial.
      rewrite <- H2 in *.
      simpl.
      eapply list_reorder_trans.
        eexact H1.
      apply list_reorder_cons_parts.
      trivial.
    auto.
  Qed.


  (** **** Lemma 3.9, provability of GR_n from S(R) n assumptions

          The nonempty V assumption is only needed for the implication from 
          right to left. The assumption about the rank of Gamma is not
          needed. 
   *)
  Lemma GR_n_provable_with_premises :
    forall(nonempty_V : V)(rules : set (sequent_rule V L))
          (n : nat)(npos : 0 < n)(gamma : sequent V L),
      one_step_rule_set rules ->
        (provable (GR_n_set rules n) (empty_sequent_set V L) gamma <->
         provable (G_n_set V L n)
                  (provable_subst_n_conclusions rules n npos)
                  gamma).
  Proof.
    intros nonempty_V rules n npos gamma H.
    split; intros H0.
      destruct H0.
      clear H0.
      induction x.
        contradiction.
      unfold GR_n_set, GR_set in in_rules.
      apply rank_rules_distribute_union in in_rules.
      destruct in_rules.
        lapply (every_nth_exists 
                 (fun(s : sequent V L)
                     (p : proof (G_n_set V L n) 
                                (provable_subst_n_conclusions rules n npos) s)
                     => True) (assumptions r)).
          clear pl H0.
          intros H0.
          destruct H0.
          clear H0.
          exists (rule (G_n_set V L n)
                       (provable_subst_n_conclusions rules n npos) r H1 x).
          trivial.
        intros i i_less.
        apply H0.
      clear H0.
      apply provable_with_assumption.
      lapply (decrease_rank_R n r _ H).
        intros H0; lapply H0; clear H0.
          intros H0.
          apply (stratified_one_step_rules nonempty_V _ n npos H) in H1.
          unfold provable_subst_n_conclusions.
          exists r.
          split; trivial.
          split; trivial.
          intros i i_less.
          rewrite <- rank_proof_GR_fixed_rank.
              rewrite rank_proof_GR_fixed_rank.
                  exists (dep_nth (assumptions r) pl i i_less).
                  trivial.
                trivial.
              eapply rank_sequent_mono with (n1 := pred n).
                apply le_pred_n.
              apply H0.
            trivial.
          apply H0.
        apply H1.
      eapply subset_rank_rules.
      eexact H1.
    destruct H0.
    clear H0.
    induction x using proof_sequent_ind.
      unfold provable_subst_n_conclusions in *.
      decompose [ex and or dep_and] H0; clear H0.
      rename x into rsubst.
      subst gamma.
      assert (GR_n_set rules n rsubst).
        clear H4.
        apply (stratified_one_step_rules nonempty_V _ n npos H) in H2.
        unfold GR_n_set.
        eapply rank_rules_mono with (rules1 := weaken_subst_rule rules).
          clear. 
          intros r H.
          unfold GR_set in *.
          right.
          trivial.
        trivial.
      assert (every_nth (provable (GR_n_set rules n) (empty_sequent_set V L))
                (assumptions rsubst)).
        eapply every_nth_mono.
          intros s H5.
          eapply proof_mono_rules.
            apply rank_rules_subset_rank with (n1 := (pred n)).
            apply le_pred_n.
          eexact H5.
        trivial.
      clear H4.
      assert (H3 := every_nth_exists _ _ H1).
      clear H1.
      destruct H3.
      clear H1.
      exists (rule (GR_n_set rules n) (empty_sequent_set V L) rsubst H0 x).
      trivial.
    assert (GR_n_set rules n r).
      apply G_n_subset_GR_n.
      trivial.
    apply every_nth_exists in H1.
    destruct H1.
    clear H1.
    exists (rule (GR_n_set rules n) (empty_sequent_set V L) r H2 x).
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Towards the weakening lemma 3.11 *)

  Lemma bounded_head_weakening_admissible_ax :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))(n : nat)
          (r : sequent_rule V L)(f : lambda_formula V L),
      rule_multiset rules ->
      subset (rank_rules n is_ax_rule) rules ->
      is_ax_rule r ->
      rank_formula n f ->
      every_nth (rank_sequent n) (assumptions r) ->
      rank_sequent n (conclusion r) ->
      every_nth (fun s : sequent V L => provable rules hyp (f :: s))
                (assumptions r) ->
        provable rules hyp (f :: conclusion r).
  Proof.
    intros rules hyp n r f H H0 H1 H2 H3 H4 H5.
    clear H5.
    unfold is_ax_rule in H1.
    destruct H1.
    clear H1.
    set (nr := {| assumptions := []; conclusion := f :: conclusion r |}).
    assert (rules nr).
      apply H0.
      split.
        split.
          trivial.
        apply simple_tautology_cons.
        trivial.
      split.
        apply every_nth_empty.
      apply rank_sequent_cons; trivial.
    exists (rule rules hyp nr H1 dep_nil).
    trivial.
  Qed.

  Lemma bounded_head_weakening_admissible_and :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))(n : nat)
          (r : sequent_rule V L)(f : lambda_formula V L),
      rule_multiset rules ->
      subset (rank_rules n is_and_rule) rules ->
      is_and_rule r ->
      rank_formula n f ->
      every_nth (rank_sequent n) (assumptions r) ->
      rank_sequent n (conclusion r) ->
      every_nth (fun s : sequent V L => provable rules hyp (f :: s))
                (assumptions r) ->
        provable rules hyp (f :: conclusion r).
  Proof.
    intros rules hyp n r f H H0 H1 H2 H3 H4 H5.
    unfold is_and_rule in H1.
    decompose [ex and] H1; clear H1.
    rename x into sl, x0 into sr, x1 into g1, x2 into g2.
    rewrite <- every_nth_map in H5.
    apply every_nth_exists in H5.
    destruct H5.
    clear H1.
    rewrite H6 in *.
    rewrite H8 in *.
    clear H6 H8.
    set (nr := rule_add_context (f :: sl) sr (bare_and_rule g1 g2)).
    assert (is_and_rule nr).
      apply context_and_rule.
    assert (rules nr).
      apply H0.
      split.
        trivial.
      assert (rank_sequent n (conclusion nr)).
        simpl.
        unfold add_context in *.
        simpl.
        apply rank_sequent_cons; trivial.
      split.
        apply const_rank_G_set.
          right.
          left.
          trivial.
        trivial.
      trivial.
    simpl in *.
    exists (rule rules hyp nr H5 x).
    trivial.
  Qed.

  Lemma bounded_head_weakening_admissible_neg_and :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))(n : nat)
          (r : sequent_rule V L)(f : lambda_formula V L),
      rule_multiset rules ->
      subset (rank_rules n is_neg_and_rule) rules ->
      is_neg_and_rule r ->
      rank_formula n f ->
      every_nth (rank_sequent n) (assumptions r) ->
      rank_sequent n (conclusion r) ->
      every_nth (fun s : sequent V L => provable rules hyp (f :: s))
                (assumptions r) ->
        provable rules hyp (f :: conclusion r).
  Proof.
    intros rules hyp n r f H H0 H1 H2 H3 H4 H5.
    unfold is_neg_and_rule in H1.
    decompose [ex and] H1; clear H1.
    rename x into sl, x0 into sr, x1 into g1, x2 into g2.
    rewrite <- every_nth_map in H5.
    apply every_nth_exists in H5.
    destruct H5.
    clear H1.
    rewrite H6 in *.
    rewrite H8 in *.
    clear H6 H8.
    set (nr := rule_add_context (f :: sl) sr (bare_neg_and_rule g1 g2)).
    assert (is_neg_and_rule nr).
      apply context_neg_and_rule.
    assert (rules nr).
      apply H0.
      split.
        trivial.
      assert (rank_sequent n (conclusion nr)).
        simpl.
        unfold add_context in *.
        simpl.
        apply rank_sequent_cons; trivial.
      split; trivial.
      apply const_rank_G_set.
        right.
        right.
        left.
        trivial.
      trivial.
    simpl in *.
    exists (rule rules hyp nr H5 x).
    trivial.
  Qed.

  Lemma bounded_head_weakening_admissible_neg_neg :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))(n : nat)
          (r : sequent_rule V L)(f : lambda_formula V L),
      rule_multiset rules ->
      subset (rank_rules n is_neg_neg_rule) rules ->
      is_neg_neg_rule r ->
      rank_formula n f ->
      every_nth (rank_sequent n) (assumptions r) ->
      rank_sequent n (conclusion r) ->
      every_nth (fun s : sequent V L => provable rules hyp (f :: s))
                (assumptions r) ->
        provable rules hyp (f :: conclusion r).
  Proof.
    intros rules hyp n r f H H0 H1 H2 H3 H4 H5.
    unfold is_neg_neg_rule in H1.
    decompose [ex and] H1; clear H1.
    rename x into sl, x0 into sr, x1 into g.
    rewrite <- every_nth_map in H5.
    apply every_nth_exists in H5.
    destruct H5.
    clear H1.
    rewrite H7 in *.
    rewrite H8 in *.
    clear H7 H8.
    set (nr := rule_add_context (f :: sl) sr (bare_neg_neg_rule g)).
    assert (is_neg_neg_rule nr).
      apply context_neg_neg_rule.
    assert (rules nr).
      apply H0.
      split.
        trivial.
      assert (rank_sequent n (conclusion nr)).
        simpl.
        unfold add_context in *.
        simpl.
        apply rank_sequent_cons; trivial.
      split; trivial.
      apply const_rank_G_set.
        right.
        right.
        right.
        trivial.
      trivial.
    simpl in *.
    exists (rule rules hyp nr H5 x).
    trivial.
  Qed.


  Lemma bounded_head_weakening_admissible_G_n_step :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))(n : nat)
          (r : sequent_rule V L)(f : lambda_formula V L),
      rule_multiset rules ->
      subset (G_n_set V L n) rules ->
      G_n_set V L n r ->
      rank_formula n f ->
      every_nth (rank_sequent n) (assumptions r) ->
      rank_sequent n (conclusion r) ->
      every_nth (fun s : sequent V L => provable rules hyp (f :: s))
                (assumptions r) ->
        provable rules hyp (f :: conclusion r).
  Proof.
    intros rules hyp n r f H H0 H1 H2 H3 H4 H5.
    unfold G_n_set, G_set, union, rank_rules in H1.
    decompose [and or] H1; clear H1.
          eapply bounded_head_weakening_admissible_ax; eauto.
          eapply subset_trans.
            apply ax_n_subset_G_n.
          trivial.
        eapply bounded_head_weakening_admissible_and; eauto.
        eapply subset_trans.
          apply and_n_subset_G_n.
        trivial.
      eapply bounded_head_weakening_admissible_neg_and; eauto.
      eapply subset_trans.
        apply neg_and_n_subset_G_n.
      trivial.
    eapply bounded_head_weakening_admissible_neg_neg; eauto.
    eapply subset_trans.
      apply neg_neg_n_subset_G_n.
    trivial.
  Qed.

  Lemma bounded_head_weakening_admissible_G_n_ind :
    forall(hyp : set (sequent V L))(s : sequent V L)(f : lambda_formula V L)
          (n : nat),
      bounded_weakening_closed n hyp ->
      sequent_multiset hyp ->
      rank_formula n f ->
      rank_sequent n s ->
      provable (G_n_set V L n) hyp s ->
        provable (G_n_set V L n) hyp (f :: s).
  Proof.
    intros hyp s f n H H0 H1 H2 H3.
    destruct H3.
    clear H3.
    induction x using proof_sequent_ind.
      assert (hyp (f :: gamma)).
      eapply H.
            eexact H3.
          eexact H1.
        apply list_reorder_refl.
      exists (assume (G_n_set V L n) hyp (f :: gamma) H4).
      trivial.
    lapply (const_rank_G_set n r).
      intros H5.
      specialize (H5 H2).
      apply bounded_head_weakening_admissible_G_n_step with (n := n); auto.
          apply G_n_multiset.
        apply subset_refl.
      intros i i_less.
      apply H4.
      apply H5.
    apply H3.
  Qed.

  (** Bounded weakening is admissible in G_n + H, 
      provided H is closed under bounded weakening used in 3.11
   *)
  Lemma weakening_admissible_Gn :
    forall(n : nat)(hyp : set (sequent V L)),
      bounded_weakening_closed n hyp ->
      sequent_multiset hyp ->
        admissible_rule_set 
           (G_n_set V L n) hyp 
           (bounded_weakening_rules V L n).
  Proof.
    intros n hyp H H0.
    unfold admissible_rule_set, admissible in *.
    intros r H1 H2.
    destruct r.
    unfold bounded_weakening_rules in *.
    simpl in *.
    decompose [ex and or dep_and] H1; clear H1.
    rename x into s, x0 into f.
    subst assumptions.
    apply every_nth_head in H2.
    eapply provable_G_n_hyp_list_reorder.
        trivial.
      apply list_reorder_symm.
      eexact H7.
    apply bounded_head_weakening_admissible_G_n_ind; auto.
  Qed.


  (** **** Weakening Lemma 3.11, page 14
          bounded weakening is admissible in GR_n 
    *)
  Lemma weakening_admissible_GR_n : 
    forall(nonempty_V : V)(rules : set (sequent_rule V L))(n : nat),
      0 < n ->
      one_step_rule_set rules ->
        admissible_rule_set 
          (GR_n_set rules n) (empty_sequent_set V L)
          (bounded_weakening_rules V L n).
  Proof.
    unfold admissible_rule_set, admissible in *.
    intros nonempty_V rules n H H0 r H1 H2.
    rewrite GR_n_provable_with_premises with (npos := H); auto.
    apply weakening_admissible_Gn.
          apply bounded_weakening_provable_subst_n_conclusions.
        apply multiset_provable_subst_n_conclusions.
      trivial.
    intros i i_less.
    rewrite <- GR_n_provable_with_premises with (npos := H); trivial.
  Qed.


  (** **** Another weakening lemma
          Weakening is admissible in GR, comment on page 15 
    *)
  Theorem weakening_admissible_GR : 
    forall(nonempty_V : V)(rules : set (sequent_rule V L)),
      one_step_rule_set rules ->
        admissible_rule_set 
          (GR_set rules) (empty_sequent_set V L)
          (full_weakening_rules V L).
  Proof.
    unfold admissible_rule_set, admissible in *.
    intros nonempty_V rules H r H0 H1.
    rewrite rank_proof_GR_fixed_rank.
        apply weakening_admissible_GR_n.
                trivial.
              apply minimal_rule_rank_gt_0.
              apply full_weakening_rules_nonempty_conclusion.
              eexact H0.
            trivial.
          apply bounded_full_weakening.
          trivial.
        intros i i_less.
        rewrite <- rank_proof_GR_fixed_rank.
            apply H1.
          trivial.
        apply minimal_rule_rank_assumptions.
      trivial.
    apply minimal_rule_rank_conclusion.
  Qed.


  (***************************************************************************)
  (** *** Towards weakening in GRC_n

         This is a missing lemma in the paper. It is needed in the
         proof of 4.13 to split the propositional part. In contrast to
         the previous weakening lemma, the GRC_n result cannot be
         proven via lemma 3.9. Here we have to do a full induction
         over the proof tree.
   *)
  (***************************************************************************)

  (** Induction step for one-step rules. *)
  Lemma bounded_head_weakening_admissible_R :
    forall(rules osr : set (sequent_rule V L))(n : nat)
          (r : sequent_rule V L)(f : lambda_formula V L),
      rule_multiset rules ->
      one_step_rule_set osr ->
      subset (rank_rules n (weaken_subst_rule osr)) rules ->
      weaken_subst_rule osr r ->
      rank_formula n f ->
      every_nth (rank_sequent n) (assumptions r) ->
      rank_sequent n (conclusion r) ->
      every_nth (fun s : sequent V L =>
                   provable rules (empty_sequent_set V L) (s) /\
                   provable rules (empty_sequent_set V L) (f :: s))
                (assumptions r) ->
        provable rules (empty_sequent_set V L) (f :: conclusion r).
  Proof.
    intros rules osr n r f H H0 H1 H2 H3 H4 H5 H6.
    eapply multiset_provability.
          trivial.
        apply sequent_multiset_empty.
      lapply (list_reorder_insert (conclusion r) [] [] (conclusion r) f f).
        intros H7; lapply H7; clear H7; trivial.
        intros H7.
        eexact H7.
      simpl.
      rewrite app_nil_r.
      apply list_reorder_refl.
    set (nr := {| assumptions := assumptions r;
                  conclusion := conclusion r ++ [f] |}).
    assert (rules nr).
      apply H1.
      split.
        clear - H2.
        unfold weaken_subst_rule in *.
        decompose [ex and or dep_and] H2; clear H2.
        rename x into r_base, x0 into sigma, x1 into delta.
        exists r_base, sigma, (delta ++ [f]).
        repeat split; trivial.
        simpl.
        rewrite app_assoc.
        apply list_reorder_append_right.
        trivial.
      split.
        trivial.
      apply rank_sequent_append.
        trivial.
      apply rank_sequent_cons.
        trivial.
      apply rank_sequent_empty.
    lapply (every_nth_exists 
              (fun(s : sequent V L)
                  (p2 : proof rules (empty_sequent_set V L) s) => True)
              (assumptions r)).
      intros H8.
      destruct H8.
      exists (rule rules (empty_sequent_set V L) nr H7 x).
      trivial.
    clear - H6.
    intros i i_less.
    apply H6.
  Qed.

  (** Induction step for cut *)
  Lemma bounded_head_weakening_admissible_cut :
    forall(rules : set (sequent_rule V L))(n : nat)
          (r : sequent_rule V L)(f : lambda_formula V L),
      rule_multiset rules ->
      subset (rank_rules n is_cut_rule) rules ->
      is_cut_rule r ->
      rank_formula n f ->
      every_nth (rank_sequent n) (assumptions r) ->
      rank_sequent n (conclusion r) ->
      every_nth (fun s : sequent V L =>
                   provable rules (empty_sequent_set V L) (s) /\
                   provable rules (empty_sequent_set V L) (f :: s))
                (assumptions r) ->
        provable rules (empty_sequent_set V L) (f :: conclusion r).
  Proof.
    intros rules n r f H H0 H1 H2 H3 H4 H5.
    unfold is_cut_rule in H1.
    decompose [ex and or dep_and] H1; clear H1.
    rename x into gl, x0 into gr, x1 into dl, x2 into dr, x3 into g.
    rewrite H7 in *.
    eapply multiset_provability.
          trivial.
        apply sequent_multiset_empty.
      apply list_reorder_symm.
      apply list_reorder_cons_head.
      eexact H8.
    simpl.
    set (nr := {| assumptions := [(f :: gl) ++ g :: gr; dl ++ lf_neg g :: dr];
                  conclusion := (f :: gl ++ gr ++ dl ++ dr) |}).
    assert (rules nr).
      apply H0.
      split.
        exists (f :: gl), gr, dl, dr, g.
        split.
          trivial.
        apply list_reorder_refl.
      split.
        apply every_nth_cons.
          simpl.
          apply rank_sequent_cons.
            trivial.
          apply every_nth_head in H3.
          trivial.
        apply every_nth_tail in H3.
        trivial.
      simpl.
      apply rank_sequent_cons.
        trivial.
      eapply rank_sequent_list_reorder.
        apply list_reorder_symm.
        eexact H8.
      trivial.
    lapply (every_nth_exists 
              (fun(s : sequent V L)
                  (p2 : proof rules (empty_sequent_set V L) s) => True)
                [(f :: gl) ++ g :: gr; dl ++ lf_neg g :: dr]).
      intros H6.
      destruct H6.
      clear H6.
      exists (rule rules (empty_sequent_set V L) nr H1 x).
      trivial.
    apply every_nth_cons.
      apply every_nth_head in H5.
      apply H5.
    apply every_nth_cons.
      apply every_nth_tail in H5.
      apply every_nth_head in H5.
      apply H5.
    apply every_nth_empty.
  Qed.


  (** Do the induction *)
  Lemma bounded_head_weakening_admissible_GRC_n :
    forall(rules : set (sequent_rule V L))(n : nat)
          (s : sequent V L)(f : lambda_formula V L),
      one_step_rule_set rules ->
      rank_formula n f ->
      proof (GRC_n_set rules n) (empty_sequent_set V L) s ->
        provable (GRC_n_set rules n) (empty_sequent_set V L) (s) /\
        provable (GRC_n_set rules n) (empty_sequent_set V L) (f :: s).
  Proof.
    induction 3 using proof_sequent_ind.
      contradiction.
    split.
      apply provable_with_rule with (assum := assumptions r).
        clear - H1.
        destruct r.
        trivial.
      intros i i_less.
      apply H2.
    unfold GRC_n_set, rank_rules in H1.
    destruct H1.
    assert (H4 := H3).
    unfold rule_has_rank in H4.
    decompose [and] H4; clear H4.
    unfold GRC_set, GR_set, union in H1.
    assert (H7 := GRC_n_multiset rules n).
    decompose [or] H1; clear H1.
        eapply bounded_head_weakening_admissible_G_n_step; eauto.
            apply G_n_subset_GRC_n.
          split; trivial.
        intros i i_less.
        apply H2.
      eapply bounded_head_weakening_admissible_R; eauto.
      apply R_n_subset_GRC_n.
    eapply bounded_head_weakening_admissible_cut; eauto.
    apply rank_rules_mono.
    apply subset_union_right.
  Qed.


  (** **** Weakening lemma for GRC_n *)
  Lemma weakening_admissible_GRC_n :
    forall(rules : set (sequent_rule V L))(n : nat),
      one_step_rule_set rules ->
        admissible_rule_set 
          (GRC_n_set rules n) (empty_sequent_set V L)
          (bounded_weakening_rules V L n).
  Proof.
    clear. 
    unfold admissible_rule_set, admissible in *.
    intros rules n H r H0 H1.
    unfold bounded_weakening_rules in *.
    decompose [ex and] H0; clear H0.
    rename x into s, x0 into f.
    eapply provable_GRC_n_list_reorder.
      apply list_reorder_symm.
      eexact H6.
    rewrite H4 in *.
    apply every_nth_head in H1.
    destruct H1.
    rename x into p.
    eapply bounded_head_weakening_admissible_GRC_n; trivial.
  Qed.


  (***************************************************************************)
  (** *** List weakening
         Lift weakening of a single formula to weakening with a sequent.
   *)
  (***************************************************************************)

  (** XXX Keep the following lemma for reproducing the print evar bug
      only. *)
  Lemma evar_bug_list_weakening_admissible_GRC_n :
    forall(rules : set (sequent_rule V L))(n : nat)
          (s1 s2 : sequent V L),
      one_step_rule_set rules ->
      rank_sequent n s1 ->
      rank_sequent n s2 ->
      provable (GRC_n_set rules n) (empty_sequent_set V L) s1 ->
        provable (GRC_n_set rules n) (empty_sequent_set V L) (s1 ++ s2).
  Proof.
    intros rules n s1 s2 H H0 H1 H2.
    eapply provable_GRC_n_list_reorder.
      apply list_reorder_append_swap.
    revert s1 H0 H2.
    induction s2.
      intros s1 H0 H2.
      simpl.
      trivial.
    rename a into f.
    intros s1 H0 H2.
    simpl.
    (*** print evar BUG: without H, this evar is missing in the printout *) 
    apply (weakening_admissible_GRC_n _ _ H
                {| assumptions := [s2 ++ s1]; conclusion := f :: s2 ++ s1 |}).
      (* Without H, there are three open existentials. Two goals and the 
         one_step_rule_set assumptions of weakening_admissible_GRC_n *)
      (* Show Existentials. *) 
      exists (s2 ++ s1), f.
      repeat split; trivial.
          eapply rank_sequent_head.
          eexact H1.
        apply rank_sequent_append.
          eapply rank_sequent_tail.
          eexact H1.
        trivial.
      simpl.
      apply list_reorder_refl.
    simpl.
    apply every_nth_cons.
      apply IHs2.
          eapply rank_sequent_tail.
          eexact H1.
        trivial.
      trivial.
    apply every_nth_empty.
  Qed.


  Lemma generic_bounded_list_weakening :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))(n : nat)
          (s1 s2 : sequent V L),
      rule_multiset rules ->
      admissible_rule_set rules hyp
        (bounded_weakening_rules V L n) ->
      sequent_multiset hyp ->
      rank_sequent n s1 ->
      rank_sequent n s2 ->
      provable rules hyp s1 ->
        provable rules hyp (s1 ++ s2).
  Proof.
    intros rules hyp n s1 s2 H H0 H1 H2 H3 H4.
    eapply multiset_provability; trivial.
      apply list_reorder_append_swap.
    revert s1 H2 H4.
    induction s2.
      intros s1 H2 H4.
      trivial.
    rename a into f.
    intros s1 H2 H4.
    simpl.
    apply (H0 {| assumptions := [s2 ++ s1]; conclusion := f :: s2 ++ s1 |}).
      exists (s2 ++ s1), f.
      repeat split; trivial.
          eapply rank_sequent_head.
          eexact H3.
        apply rank_sequent_append.
          eapply rank_sequent_tail.
          eexact H3.
        trivial.
      simpl.
      apply list_reorder_refl.
    simpl.
    apply every_nth_cons.
      apply IHs2; trivial.
      eapply rank_sequent_tail.
      eexact H3.
    apply every_nth_empty.
  Qed.

  Lemma list_weakening_admissible_G_n_hyp :
    forall(hyp : set (sequent V L))(n : nat)(s1 s2 : sequent V L),
      sequent_multiset hyp ->
      bounded_weakening_closed n hyp ->
      rank_sequent n s1 ->
      rank_sequent n s2 ->
      provable (G_n_set V L n) hyp s1 ->
        provable (G_n_set V L n) hyp (s1 ++ s2).
  Proof.
    intros hyp n s1 s2 H H0 H1 H2 H3.
    eapply generic_bounded_list_weakening; eauto.
      apply G_n_multiset.
    apply weakening_admissible_Gn; trivial.
  Qed.

  (* not needed
   * Lemma list_weakening_admissible_G_n_H :
   *   forall(rules : set (sequent_rule V L))(n : nat)(npos : 0 < n)
   *         (s1 s2 : sequent V L),
   *     1 < n ->
   *     one_step_rule_set rules ->
   *     rank_sequent n s2 ->
   *     provable (G_n_set V L n) 
   *              (provable_subst_n_conclusions rules n npos) s1 ->
   *       provable (G_n_set V L n) 
   *                (provable_subst_n_conclusions rules n npos) (s1 ++ s2).
   * Proof.
   *   intros rules n npos s1 s2 H H0 H1 H2.
   *   apply list_weakening_admissible_G_n_hyp; trivial.
   *       apply multiset_provable_subst_n_conclusions.
   *     apply bounded_weakening_provable_subst_n_conclusions.
   *   eapply provable_rank_rules_hyp_has_rank_n.
   *     apply rank_sequent_set_provable_subst_n_conclusions.
   *       trivial.
   *     eexact H0.
   *   eexact H2.
   * Qed.
   *)

  Lemma list_weakening_admissible_GR_n :
    forall(nonempty_v : V)(rules : set (sequent_rule V L))(n : nat)
          (s1 s2 : sequent V L),
      0 < n ->
      one_step_rule_set rules ->
      rank_sequent n s2 ->
      provable (GR_n_set rules n) (empty_sequent_set V L) s1 ->
        provable (GR_n_set rules n) (empty_sequent_set V L) (s1 ++ s2).
  Proof.
    intros nonempty_v rules n s1 s2 H H0 H1 H2.
    eapply generic_bounded_list_weakening; eauto.
          apply GR_n_multiset.
        apply weakening_admissible_GR_n; trivial.
      apply sequent_multiset_empty.
    eapply provable_rank_rules_hyp_has_rank_n.
      apply rank_sequent_set_empty.
    eexact H2.
  Qed.

  Lemma list_weakening_admissible_GRC_n :
    forall(rules : set (sequent_rule V L))(n : nat)
          (s1 s2 : sequent V L),
      one_step_rule_set rules ->
      rank_sequent n s1 ->
      rank_sequent n s2 ->
      provable (GRC_n_set rules n) (empty_sequent_set V L) s1 ->
        provable (GRC_n_set rules n) (empty_sequent_set V L) (s1 ++ s2).
  Proof.
    intros rules n s1 s2 H H0 H1 H2.
    eapply generic_bounded_list_weakening; eauto.
        apply GRC_n_multiset.
      apply weakening_admissible_GRC_n.
      trivial.
    apply sequent_multiset_empty.
  Qed.

End Weakening.

Implicit Arguments provable_subst_n_conclusions [V L].
