(* 
 * 
 *               Camlp4 quotations in original syntax
 * 
 *                 Copyright (C) 2005  Hendrik Tews
 * 
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 * 
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License in the file LICENCE in this or one
 *   of the parent directories for more details.
 * 
 *   Time-stamp: <Tuesday 14 June 05 0:08:56 tews@debian>
 * 
 *   $Id: pp_q.ml,v 1.15 2005/11/14 22:46:41 tews Exp $
 * 
 *)

open Pp_util
module IB = Indent_buffer

let id = "pp_q"

let rec pp_id_list pp ids =
  IB.indent pp;
  (match ids with
     | [] -> IB.endline pp "";
     | [s] -> IB.endline pp s;
     | s::sl -> IB.printf pp "%s." s; pp_id_list pp sl
  );
  IB.unindent pp


let rec pp_ctyp pp ast = 
  let astloc = MLast.loc_of_ctyp ast in
  let pl mesg = print_location pp mesg astloc in
  let pl_rec_1 s t =
      begin
	pl s;
	pp_ctyp pp t;
	if !terminate_structures then
	  IB.endline pp ("end " ^ s)
      end
  in
  let pl_rec_2 s t1 t2 =
      begin
	pl s;
	pp_ctyp pp t1;
	pp_ctyp pp t2;
	if !terminate_structures then
	  IB.endline pp ("end " ^ s)
      end
  in
  let _ = IB.indent pp in
  let _ = match ast with
    | <:ctyp< $t1$ == $t2$>> -> pl_rec_2 "type manifest" t1 t2
    | <:ctyp< private $t1$>> -> pl_rec_1 "private type" t1
    | <:ctyp< $t1$ as $t2$>> -> pl_rec_2 "type alias" t1 t2

    | <:ctyp< ! $list:tvl$ . $t$>> ->
	begin
	  IB.add_string pp "poly type ";
	  iterate_items pp (Some astloc) "tv list" "with %d tv" "type var %d: "
	    (fun name -> 
	       IB.indent pp;
	       IB.printf pp "tick %s\n" name;
	       IB.unindent pp)
	    tvl;
	  pp_ctyp pp t;
	  if !terminate_structures then
	    IB.endline pp "end poly type"
	end

    (* level "arrow" *)
    | <:ctyp< $t1$ -> $t2$>> -> pl_rec_2 "function type" t1 t2

    (* level "label" *)
        (* not present yet *)

    | <:ctyp< $t1$ $t2$ >> -> pl_rec_2 "type application" t1 t2
    | <:ctyp< $t1$ . $t2$ >> -> pl_rec_2 "type module access" t1 t2

    (* level "simple" *)
    | <:ctyp< ' $id$ >> -> pl ("tick " ^ id)
    | <:ctyp< _ >> -> pl "anon type"
    | <:ctyp< $lid:id$ >> -> pl ("typ lid " ^ id)
    | <:ctyp< $uid:id$ >> -> pl ("typ uid " ^ id)
    | <:ctyp< ( $list:types$ ) >> ->
	iterate_items pp (Some astloc) "tuple type" "with %d elements" "El %d: "
	(fun ctyp -> pp_ctyp pp ctyp)
	types

    | <:ctyp< [ $list:cdl$ ] >> -> 
	 begin
	   pl (Printf.sprintf "variant type with %d variants"
		 (List.length cdl));
	   pp_variant_decls pp cdl;
	   if !terminate_structures then
	     IB.endline pp "end variant type"
	 end
    | <:ctyp< { $list:ldl$ } >> ->
	 begin
	   pl (Printf.sprintf "record type with %d fields" (List.length ldl));
	   pp_label_decls pp ldl;
	   if !terminate_structures then
	     IB.endline pp "end record type"
	 end

      | <:ctyp< # $list:ids$ >> ->
	begin
	  pl "#-type";
	  pp_id_list pp ids;
	end

      | <:ctyp< < $list:fields$ $opt:b$ > >> -> 
	iterate_items pp (Some astloc) 
	  (if b then "object type (open)" else "object type (closed)")
	  "with %d methods" "method %d:" 
	  (fun (id, typ) ->
	     IB.indent pp;
	     IB.printf pp "name %s\n" id;
	     IB.unindent pp;
	     pp_ctyp pp typ;
	  )
	  fields

    | _ -> (pl "unknown ctyp"; abort_maybe())
  in
    IB.unindent pp



and pp_label_decls pp decls =
  iterate_items pp None "" "%d" "field %d: "
    (fun (loc, name, mut, ctyp) -> 
       IB.indent pp;
       print_location pp (Printf.sprintf "%s%s"
			     (if mut then "mutable " else "") name)
	 loc;

       (* the clean way would be to call pp_ctyp_without_indent *)
       IB.unindent pp;
       pp_ctyp pp ctyp)
    decls


and pp_variant_decls pp decls =
  iterate_items pp None "" "%d" "variant %d:"
    (fun (loc, name, ctyps) -> 
       IB.indent pp;
       print_location pp 
	 (Printf.sprintf "%s with %d args" name (List.length ctyps))
	 loc;
       iterate_items pp None "" "%d" "Arg %d: "
	 (fun ctyp -> pp_ctyp pp ctyp)
	 ctyps;
       IB.unindent pp
    )
    decls


let rec pp_class_sig_item pp ast =
  let astloc = MLast.loc_of_class_sig_item ast in
  let pl mesg = print_location pp mesg astloc in
  let _ = IB.indent pp in
  let _ = match ast with
    | <:class_sig_item< declare $list:clsitems$ end >> ->
      iterate_items pp (Some astloc) "class sig declare" 
	"with %d items" "Item %d: "
	(fun item -> pp_class_sig_item pp item)
	clsitems
    | <:class_sig_item< inherit $cltyp$ >> ->
      begin
	pl "inherit (class sig)";
	pp_class_type pp cltyp;
      end
    | <:class_sig_item< value $opt:mut$ $id$ : $ptyp$ >> ->
      begin
	pl (Printf.sprintf "object sig val %s (%s)" id
	      (if mut then "mutable" else "nonmutable"));
	pp_ctyp pp ptyp;
      end
    | <:class_sig_item< method virtual $opt:priv$ $id$ : $ptyp$ >> ->
      begin
	pl (Printf.sprintf "virtual method spec %s (%s)" id
	      (if priv then "private" else "public"));
	pp_ctyp pp ptyp;
      end
    | <:class_sig_item< method $opt:priv$ $id$ : $ptyp$ >> ->
      begin
	pl (Printf.sprintf "public method spec %s (%s)" id
	      (if priv then "private" else "public"));
	pp_ctyp pp ptyp;
      end
    | <:class_sig_item< type $t1$ = $t2$ >> ->
      begin
	pl "object constraint spec";
	pp_ctyp pp t1;
	pp_ctyp pp t2;
      end
    (* | _ -> (pl "unknown class sig item"; abort_maybe()) *)
  in
    IB.unindent pp



and pp_class_type pp ast =
  let astloc = MLast.loc_of_class_type ast in
  let pl mesg = print_location pp mesg astloc in
  let _ = IB.indent pp in
  let _ = match ast with
    | <:class_type< [ $t1$ ] -> $ct2$ >> -> 
      begin
	pl "class function";
	pp_ctyp pp t1;
	pp_class_type pp ct2;
	if !terminate_structures then
	  IB.endline pp "end class function"
      end
    | <:class_type< $list:ids$ [ $list:args$ ] >> ->
      begin
	pl (Printf.sprintf "%s class type longident" 
	      (if args = [] then "monomorphic" else "polymorphic"));
	IB.indent pp;
	if args <> [] then
	  begin
	    IB.printf pp "%d type args:\n" (List.length args);
	    iterate_items pp None "" "%d" "Arg %d:" (pp_ctyp pp) args;
	  end;
	IB.printf pp "%d ids:\n" (List.length ids);
	pp_id_list pp ids;
	IB.unindent pp;
	if !terminate_structures then
	  IB.endline pp "end class type longident";
      end

    | <:class_type< object $opt:selftype$ $list:sigitems$ end >> ->
      begin
	pl (Printf.sprintf "object type with %d items" (List.length sigitems));
	IB.indent pp;
	(match selftype with
	   | None -> IB.endline pp "no self type"
	   | Some ptyp -> 
	       begin
		 IB.add_string pp "self type:\n";
		 pp_ctyp pp ptyp;
	       end
	);
	IB.unindent pp;
	iterate_items pp None "" "%d" "item %d:"
	  (pp_class_sig_item pp)
	  sigitems;
	if !terminate_structures then
	  IB.endline pp "end object type"
      end

    (* | _ -> (pl "unknown class type"; abort_maybe()) *)
  in
    IB.unindent pp


let pp_type_params pp type_params =
  iterate_items pp None "" "%d" "param %d: "
    (fun (name, (leftvar, rightvar)) ->
       IB.indent pp;
       IB.printf pp "%s %s\n"
	 (match (leftvar, rightvar) with
	    | false,false -> "="
	    | true,false -> "+"
	    | false,true -> "-"
	    | _ -> assert false)
	 name;
       IB.unindent pp)
    type_params


let pp_type_constraints pp constr =
  iterate_items pp None "" "%d" "constraint %d:"
    (fun (left_type, right_type) ->
       IB.indent pp;
       pp_ctyp pp left_type;
       IB.endline pp "=";
       pp_ctyp pp right_type;
       IB.unindent pp;
    )
    constr


let end_of_type_decl typ cl =
  try
    snd (MLast.loc_of_ctyp (snd (List.nth cl ((List.length cl) -1))))
  with
    | Failure "nth" -> snd (MLast.loc_of_ctyp typ)


let pp_type_decl pp ((loc, name), tpl, typ, cl) =
  begin
    IB.indent pp;
    print_location pp (Printf.sprintf "type %s" name) 
      (fst loc, end_of_type_decl typ cl);
    IB.indent pp;
    IB.printf pp "%d type parameters:\n" (List.length tpl);
    pp_type_params pp tpl;
    IB.endline pp "rhs type:";
    pp_ctyp pp typ;
    IB.printf pp "%d constraints:\n" (List.length cl);
    pp_type_constraints pp cl;
    IB.unindent pp;
    if !terminate_structures then
      IB.endline pp "end type";
    IB.unindent pp;
  end


let pp_class_info pp name pp_ast ci =
  (* let module M = MLast in  there is a dynlink bug !! *)
  let pl mesg = print_location pp mesg ci.MLast.ciLoc 
  in
    IB.indent pp;
    pl (Printf.sprintf "%s%s %s"
	  (if ci.MLast.ciVir then "virtual " else "")
	  name ci.MLast.ciNam);
    IB.indent pp;
    iterate_items pp (Some (fst ci.MLast.ciPrm)) 
      "class type params" "(%d params)" "param %d: "
      (fun (name, (leftvar, rightvar)) ->
	 IB.indent pp;
	 IB.printf pp "%s %s\n"
	   (match (leftvar, rightvar) with
	      | false,false -> "="
	      | true,false -> "+"
	      | false,true -> "-"
	      | _ -> assert false)
	   name;
	 IB.unindent pp)
      (snd ci.MLast.ciPrm);
    IB.unindent pp;
    pp_ast pp ci.MLast.ciExp;
    if !terminate_structures then
      IB.endline pp ("end " ^ name);
    IB.unindent pp
	

let rec pp_expr pp ast =
  let astloc = MLast.loc_of_expr ast in
  let pl mesg = print_location pp mesg astloc in
  let pl_rec_1 s e =
      begin
	pl s;
	pp_expr pp e;
	if !terminate_structures then
	  IB.endline pp ("end " ^ s)
      end
  in
  let pl_rec_2 s e1 e2 =
      begin
	pl s;
	pp_expr pp e1;
	pp_expr pp e2;
	if !terminate_structures then
	  IB.endline pp ("end " ^ s)
      end
  in
  let pl_rec_3 s e1 e2 e3 =
      begin
	pl s;
	pp_expr pp e1;
	pp_expr pp e2;
	pp_expr pp e3;
	if !terminate_structures then
	  IB.endline pp ("end " ^ s)
      end 
  in
  let _ = IB.indent pp in
  let _ = match ast with

    (* level "top" *)
    | <:expr<let a b : $ctyp$ = 2 in 3>> when !test_more_quotations -> 
      begin
	pl "let a b : <type follows> = 2 in 3";
	pp_ctyp pp ctyp;
	if !terminate_structures then
	  IB.endline pp "end let a b"
      end
    | <:expr<let $opt:r$ $list:bindings$ in $e$>> ->
      begin
	pl (Printf.sprintf "let %sin (expr) with %d bindings" 
	      (if r then "rec " else "") (List.length bindings));
	IB.indent pp;
	iterate_items pp None "" "%d" "bind %d:"
	  (fun (p,e) ->
	     IB.indent pp;
	     pp_patt pp p;
	     IB.endline pp "to";
	     pp_expr pp e;
	     IB.unindent pp
	  )
	  bindings;
	IB.endline pp "in";
	pp_expr pp e;
	IB.unindent pp;
	if !terminate_structures then
	  IB.endline pp "end let"
      end

    | <:expr< let module $uid$ = $me$ in $e$ >> ->
	begin
	  pl (Printf.sprintf "let mod (expr) %s" uid);
	  IB.indent pp;
	  pp_module_expr pp me;
	  IB.endline pp "in";
	  pp_expr pp e;
	  IB.unindent pp;
	  if !terminate_structures then
	    IB.endline pp "end let mod";
	end

    | <:expr< fun [ $list:matches$ ] >> ->
      iterate_items pp (Some astloc) "fun" "with %d patterns" "case %d:"
	(fun (p,w,e) ->
	   IB.indent pp;
	   pp_patt pp p;
	   (match w with
	      | None -> IB.endline pp "when empty"
	      | Some e -> 
		  begin
		    IB.endline pp "when:";
		    pp_expr pp e
		  end
	   );			  
	   IB.endline pp "->";
	   pp_expr pp e;
	   IB.unindent pp
	)
	matches

    | <:expr< match $e$ with [ $list:matches$ ] >> ->
      begin
	(* check type *)
	ignore(matches : (MLast.patt * MLast.expr option * MLast.expr) list);

	pl (Printf.sprintf "match with %d cases" (List.length matches));
	pp_expr pp e;
	iterate_items pp None "" "%d" "case %d:" 
	  (fun (p,w,e) ->
	     IB.indent pp;
	     pp_patt pp p;
	     (match w with
		| None -> IB.endline pp "when empty"
		| Some e -> 
		    begin
		      IB.endline pp "when:";
		      pp_expr pp e
		    end
	     );			  
	     IB.endline pp "->";
	     pp_expr pp e;
	     IB.unindent pp
	  )
	  matches;
	if !terminate_structures then
	  IB.endline pp "end match"
      end

    | <:expr< try $e$ with [ $list:matches$ ] >> ->
      begin
	pl (Printf.sprintf "try with %d cases" (List.length matches));
	pp_expr pp e;
	iterate_items pp None "" "%d" "case %d:"
	  (fun (p,w,e) ->
	     IB.indent pp;
	     pp_patt pp p;
	     (match w with
		| None -> IB.endline pp "when empty"
		| Some e -> 
		    begin
		      IB.endline pp "when:";
		      pp_expr pp e
		    end
	     );			  
	     IB.endline pp "->";
	     pp_expr pp e;
	     IB.unindent pp
	  )
	  matches;
	if !terminate_structures then
	  IB.endline pp "end try";
      end

    | <:expr< if $e1$ then $e2$ else $e3$ >> -> 
      pl_rec_3 "if-then-else" e1 e2 e3
    | <:expr< do { $e1$; $e2$; }>> -> pl_rec_2 "two sequence" e1 e2
    | <:expr< do { $list:els$ }>> -> 
      iterate_items pp (Some astloc) "sequence" "- %d els" "El %d"
	(fun el -> pp_expr pp el)
	els

    | <:expr< for $i$ = $e1$ $to:updown$ $e2$ do { $list:els$ } >> ->
      begin
	pl (Printf.sprintf "for %s %s with %d statements"
	      i (if updown then "upto" else "downto") (List.length els));
	IB.indent pp;
	IB.endline pp "from:";
	pp_expr pp e1;
	IB.endline pp "to:";
	pp_expr pp e2;
	IB.endline pp "statements:";
	iterate_items pp None "" "%d" "El %d:" 
	  (fun el -> pp_expr pp el)
	  els;
	if !terminate_structures then
	  IB.endline pp "end for";
	IB.unindent pp;
      end
    | <:expr< while $e$ do { $list:els$ } >> ->
      begin
	pl (Printf.sprintf "while with %d statements" (List.length els));
	pp_expr pp e;
	iterate_items pp None "" "%d" "El %d:" 
	  (fun el -> pp_expr pp el)
	  els;
	if !terminate_structures then
	  IB.endline pp "end while";
      end

    (* 
     *   no quotations for immediate objects in q_MLast
     * | <:expr< object $opt:spopt$ $list:oitems$ end >> -> 
     *)
    | MLast.ExObj(loc, spopt, oitems) ->
      begin
	pl (Printf.sprintf "immediate object with %d items" 
	      (List.length oitems));
	IB.indent pp;
	(match spopt with
	   | None -> IB.endline pp "no self pattern"
	   | Some patt -> 
	       begin
		 IB.add_string pp "self pattern:\n";
		 pp_patt pp patt;
	       end
	);
	iterate_items pp None "" "%d" "item %d:"
	  (fun citem -> pp_class_str_item pp citem)
	  oitems;
	IB.unindent pp;
	if !terminate_structures then
	  IB.endline pp "end immediate object"
      end

    (* level "tuple" *)
    | <:expr< ($a$, $b$) >> -> pl_rec_2 "two tuple" a b
    | <:expr< ($list:els$) >> ->
      iterate_items pp (Some astloc) "tuple" "- %d els" "El %d: " 
      	(fun el -> pp_expr pp el)
      	els;

    (* level ":=" *)
    | <:expr< $e1$ := $e2$ >> -> pl_rec_2 "assignment" e1 e2

    (* level "||" *)
    (* | <:expr< $e1$ or $e2$ >> -> pl_rec_2 "ooor" e1 e2 *)
    | <:expr< $e1$ || $e2$ >> -> pl_rec_2 "||" e1 e2
	
    (* level "&&" *)
    (* | <:expr< $e1$ & $e2$ >> -> pl_rec_2 "" e1 e2 *)
    | <:expr< $e1$ && $e2$ >> -> pl_rec_2 "&&" e1 e2

    (* level "<" *)
    | <:expr< $e1$ < $e2$ >> -> pl_rec_2 "<" e1 e2
    | <:expr< $e1$ > $e2$ >> -> pl_rec_2 ">" e1 e2
    | <:expr< $e1$ <= $e2$ >> -> pl_rec_2 "<=" e1 e2
    | <:expr< $e1$ >= $e2$ >> -> pl_rec_2 ">=" e1 e2
    | <:expr< $e1$ = $e2$ >> -> pl_rec_2 "=" e1 e2
    | <:expr< $e1$ <> $e2$ >> -> pl_rec_2 "<>" e1 e2
    | <:expr< $e1$ == $e2$ >> -> pl_rec_2 "==" e1 e2
    | <:expr< $e1$ != $e2$ >> -> pl_rec_2 "!=" e1 e2
    (* 
     * $ is the antiquotation magic, don't know how to escape it 
     * | <:expr< $e1$ $ $e2$ >> -> pl_rec_2 "" e1 e2
     *)
    (* | <:expr< $e1$ === $e2$ >> -> pl_rec_2 "" e1 e2 *)

    (* level "^" *)
    | <:expr< $e1$ ^ $e2$ >> -> pl_rec_2 "^" e1 e2
    | <:expr< $e1$ @ $e2$ >> -> pl_rec_2 "@" e1 e2
    (* | <:expr< $e1$ ^@ $e2$ >> -> pl_rec_2 "" e1 e2 *)

    (* level "+" *)
    | <:expr< $e1$ + $e2$ >> -> pl_rec_2 "+" e1 e2
    | <:expr< $e1$ - $e2$ >> -> pl_rec_2 "-" e1 e2
    | <:expr< $e1$ +. $e2$ >> -> pl_rec_2 "+." e1 e2
    (* | <:expr< $e1$ +.+. $e2$ >> -> pl_rec_2 "+.+." e1 e2 *)

    (* level "*" *)
    | <:expr< $e1$ * $e2$ >> -> pl_rec_2 "*" e1 e2
    | <:expr< $e1$ / $e2$ >> -> pl_rec_2 "/" e1 e2
    | <:expr< $e1$ land $e2$ >> -> pl_rec_2 "land" e1 e2
    | <:expr< $e1$ lor $e2$ >> -> pl_rec_2 "lor" e1 e2
    | <:expr< $e1$ lxor $e2$ >> -> pl_rec_2 "lxor" e1 e2
    | <:expr< $e1$ mod $e2$ >> -> pl_rec_2 "mod" e1 e2
    | <:expr< $e1$ *. $e2$ >> -> pl_rec_2 "*." e1 e2
    (* | <:expr< $e1$ %%%% $e2$ >> -> pl_rec_2 "%%%%" e1 e2 *)

    (* level "**" *)
    | <:expr< $e1$ ** $e2$ >> ->  pl_rec_2 "**" e1 e2
    | <:expr< $e1$ asr $e2$ >> -> pl_rec_2 "asr" e1 e2
    | <:expr< $e1$ lsl $e2$ >> -> pl_rec_2 "lsl" e1 e2
    | <:expr< $e1$ lsr $e2$ >> -> pl_rec_2 "lsr" e1 e2

    (* level "unary minus" *)
    | <:expr< - - 5>> -> pl "minus minus 5"
    | <:expr< - 7L >> -> pl "minus 7L"
    | <:expr< - - 7.0 >> -> pl "minus minus 7.0"

    (* level "apply" *)
    | <:expr< Some(1,2,3) >> -> pl "Some(1, 2, 3)"
    | <:expr< Some($list:args$) >> -> 
      iterate_items pp (Some astloc) "Some" "%d tuple" "El %d " 
	(fun el -> pp_expr pp el)
	args;
    | <:expr< assert False>> -> pl "assert false"
    | <:expr< assert $e$>> -> pl_rec_1 "assert" e
    | <:expr< lazy $e$>> -> pl_rec_1 "lazy" e

    (* level "." *)

    | <:expr< $arr$ . ( $i$ ) >> -> pl_rec_2 "array get" arr i
    | <:expr< $s$ . [ $i$ ] >> -> pl_rec_2 "string get" s i
      
    (* no direct bigarray access in q_MLast *)

    (* level ~- *)

    | <:expr< ~- $e$ >> -> pl_rec_1 "int neg" e
    | <:expr< ~-. $e$ >> -> pl_rec_1 "float neg" e

    (* level simple *)

    | <:expr< $int:s$ >> -> pl ("int const " ^ s) 
    | <:expr< $int32:s$ >> -> pl ("int32 const " ^ s) 
    | <:expr< $int64:s$ >> -> pl ("int64 const " ^ s) 
    | <:expr< $nativeint:s$ >> -> pl ("nativeint const " ^ s) 
    | <:expr< $flo:s$ >> -> pl ("float const " ^ s) 
    | <:expr< $str:s$ >> -> pl ("string const \"" ^ s ^ "\"") 
    | <:expr< $chr:s$ >> -> pl ("char const " ^ s) 
    | <:expr< $lid:s$ >> -> pl (Printf.sprintf "lid \"%s\"" s)
      (* true and false are parsed into ExUid "True" / "False"
       * and True and False are parsed into uids " True" / " False"
       * so the next matches are partly redundant (but still useful)
       * make sure they appear before the <:expr< $uid$>> match
       *)
    | <:expr< True >> -> pl "true"
    | <:expr< False >> -> pl "false"
    | <:expr< $uid:" True"$ >> -> pl "True"
    | <:expr< $uid:" False"$ >> -> pl "False"

      (* Also [] and () are parsed as an uid *)
    | <:expr< [] >> -> pl "nil"
    | <:expr< () >> -> pl "unit"
    | <:expr< $uid:s$ >> -> pl (Printf.sprintf "uid \"%s\"" s)
      (*
       * the following is a bit redundant, 
       * because it really is just field selection
       *)
    | <:expr< $e$ . val >> -> pl_rec_1 "ref contents" e
    | <:expr< $e1$ . $e2$ >> -> 
      begin
	pl "record/module access";
	pp_expr pp e1;
	pp_expr pp e2;
	if !terminate_structures then
	  IB.endline pp "end record/module access"
      end
    | <:expr< [ $hd$ :: $tl$ ] >> -> pl_rec_2 "::" hd tl
    | <:expr< [| |] >> -> pl "empty array"
    | <:expr< [| $list:els$ |] >> -> 
      iterate_items pp (Some astloc) "array" "%d elements" "El %d "
	(fun el -> pp_expr pp el)
	els;
    | <:expr< { $id1$ = $v1$; $id2$ = $v2$ }>> ->
      begin
	pl "two field record";
	IB.indent pp;
	pl "id1:";
	pp_patt pp id1;
	pp_expr pp v1;
	pl "id2:";
	pp_patt pp id2;
	pp_expr pp v2;
	IB.unindent pp;
	if !terminate_structures then
	  IB.endline pp "two field record end";
      end
    | <:expr< { $list:fields$ } >> -> 
      iterate_items pp (Some astloc) "record" "with %d fields" "field %d: "
	(fun (l,ex) ->
	   pp_patt pp l;
	   pp_expr pp ex)
	fields;
    | <:expr< { ($ex$) with $list:fields$ } >> ->
      begin
	pl "record update";
	IB.indent pp;
	IB.endline pp "expr:";
	pp_expr pp ex;
	IB.printf pp "with %d updates:\n" (List.length fields);
	iterate_items pp None "" "%d" "update %d: "
	  (fun (l,ex) ->
	     pp_patt pp l;
	     pp_expr pp ex)
	  fields;
	IB.unindent pp;
	if !terminate_structures then
	  IB.endline pp "record update end";	  
      end
    | <:expr< ( $e$ : $t$ )>> -> 
      begin
      	pl "typed ex";
      	pp_expr pp e;
      	pp_ctyp pp t;
	if !terminate_structures then
      	  IB.endline pp "end typed ex"
      end

    (* belongs to level "apply" *)
    | <:expr< $f$ $e$>> -> pl_rec_2 "application" f e

    | <:expr< new $list:ids$ >> ->
      begin
	pl (Printf.sprintf "new with %d ids" (List.length ids));
	pp_id_list pp ids;
      end
    | <:expr< $expr$ # $lid$ >> ->
      begin
	pl (Printf.sprintf "method call %s" lid);
	pp_expr pp expr;
      end
    | <:expr< ( $expr$ : $t1$ :> $t2$ ) >> ->
      begin
	pl "complete object coercion";
	pp_expr pp expr;
	pp_ctyp pp t1;
	pp_ctyp pp t2;
      end
    | <:expr< ( $expr$ :> $t1$ ) >> ->
      begin
	pl "incomplete object coercion";
	pp_expr pp expr;
	pp_ctyp pp t1;
      end
    | <:expr< {< $list:fields$ >} >> ->
      iterate_items pp (Some astloc) "object duplication" "with %d updates"
      "update %d:" 
      (fun (id, expr) ->
	 IB.indent pp;
	 IB.printf pp "id %s\n" id;
	 IB.unindent pp;
	 pp_expr pp expr;
      )
      fields

    | _ -> (pl "unknown expression"; abort_maybe())
  in
    IB.unindent pp

and pp_patt pp ast =
  let astloc = MLast.loc_of_patt ast in
  let pl mesg = print_location pp mesg astloc in
  let pl_rec_2 s e1 e2 =
      begin
	pl s;
	pp_patt pp e1;
	pp_patt pp e2;
	if !terminate_structures then
	  IB.endline pp ("end " ^ s)
      end
  in
  let _ = IB.indent pp in
  let _ = match ast with
    | <:patt< $p1$ | $p2$ >> -> pl_rec_2 "or pattern" p1 p2

    | <:patt< [ $p1$; $p2$; $p3$ ] >> when !test_more_quotations ->
      begin
	pl "three element patt list";
	pp_patt pp p1;
	pp_patt pp p2;
	pp_patt pp p3;
	if !terminate_structures then
	  IB.endline pp "end three element list"
      end
    | <:patt< [0; 1 :: []] >> -> pl "patt [0; 1 :: []]"
    | <:patt< [$p1$ :: $p2$] >> -> pl_rec_2 "patt list ::" p1 p2

    | <:patt< $p1$ $p2$ >> -> pl_rec_2 "pattern application" p1 p2

    | <:patt< $p1$ .. $p2$ >> -> pl_rec_2 "pattern range" p1 p2

    | <:patt< $p1$ . $p2$ >> -> pl_rec_2 "patt rec/mod" p1 p2

    (* level "simple" *)

    | <:patt< () >> -> pl "unit pattern"
    | <:patt< True >> -> pl ("patt true")
    | <:patt< False >> -> pl ("patt false")
    | <:patt< $uid:" True"$ >> -> pl ("patt True")
    | <:patt< $uid:" False"$ >> -> pl ("patt False")
    | <:patt< $lid:i$ >> -> pl (Printf.sprintf "plid \"%s\"" i)
    | <:patt< $uid:i$ >> -> pl (Printf.sprintf "PUID \"%s\"" i)
    | <:patt< -5 >> -> pl ("int -5 pattern")
    | <:patt< $int:s$ >> -> pl ("int patt " ^ s) 
    | <:patt< $int32:s$ >> -> pl ("int32 patt " ^ s) 
    | <:patt< $int64:s$ >> -> pl ("int64 patt " ^ s) 
    | <:patt< $nativeint:s$ >> -> pl ("nativeint patt " ^ s) 
    | <:patt< $flo:s$ >> -> pl ("float patt " ^ s) 
    | <:patt< $str:s$ >> -> pl ("strint patt " ^ s) 
    | <:patt< $chr:s$ >> -> pl ("char patt " ^ s) 
    | <:patt< ($p1$ as $p2$) >> -> pl_rec_2 "alias pattern" p1 p2
    | <:patt< (0, 1) >> -> pl "tuple pattern 0,1"
    | <:patt< ($list:pls$) >> ->
      iterate_items pp (Some astloc) "tuple pattern" "- %d els" "El %d: "
	(fun p -> pp_patt pp p)
	pls;

    | <:patt< [||] >> -> pl "pattern empty array"
    | <:patt< [| $list:patts$ |] >> -> 
      iterate_items pp (Some astloc) "array pattern" "- %d els" "El %d: "
	(fun p -> pp_patt pp p)
	patts
    | <:patt< { $list:lblpatts$ } >> -> 
      iterate_items pp (Some astloc) "record pattern" "- %d els" "El %d: "
	(fun (l,p) -> 
	   IB.indent pp;
	   pp_patt pp l;
	   IB.endline pp "=";
	   pp_patt pp p;
	   IB.unindent pp)
	lblpatts
    | <:patt< ($p$ : $typ$) >> ->
      begin
	pl "pattern : type";
	pp_patt pp p;
	pp_ctyp pp typ;
	if !terminate_structures then
	  IB.endline pp "end pattern : typ"
      end
    | <:patt< _ >> -> pl "anon _"

    | _ -> (pl "unknown pattern"; abort_maybe())
  in
    IB.unindent pp


and pp_sig_item pp ast = 
  IB.indent pp;
  pp_sig_item_noindent pp ast;
  IB.unindent pp

and pp_sig_item_noindent pp ast =
  let astloc = MLast.loc_of_sig_item ast in
  let pl mesg = print_location pp mesg astloc
  in match ast with
    | <:sig_item< declare $list:items$ end >> ->
      begin
	pl (Printf.sprintf "sig declare with %d items" (List.length items));
	ignore(
	  List.fold_left
	    (fun n item ->
	       IB.printf pp "Item %d: " n;
	       pp_sig_item pp item;
	       n + 1;
	    )
	    1
	    items);
	if !terminate_structures then
	  IB.endline pp "end sig declare"
      end
		 
    | <:sig_item< exception $uid$ of $list:tl$ >> -> 
      iterate_items pp (Some astloc) 
	(Printf.sprintf "exception decl %s" uid) "with %d args" "Arg %d: "
	(fun ctyp -> pp_ctyp pp ctyp)
	tl

    | <:sig_item< external $lid$ : $typ$ = $list:strings$>> ->
      begin
	pl (Printf.sprintf "external decl %s with %d strings" 
	      lid (List.length strings));
	pp_ctyp pp typ;
	iterate_items pp None ""  "%d" "string %d: "
	  (fun s -> 
	     IB.indent pp;
	     IB.endline pp s;
	     IB.unindent pp)
	  strings;
	if !terminate_structures then
	  IB.endline pp "end external decl"
      end

    | <:sig_item< include $mt$ >> ->
      begin
	pl "include sig";
	pp_module_type pp mt;
	if !terminate_structures then
	  IB.endline pp "end include"
      end

    | <:sig_item< module A(B : C) : D >> -> pl "module A(B : C) : D"

    | <:sig_item< module rec $list:msigs$ >> ->
      iterate_items pp (Some astloc) "module rec decl" 
	"with %d modules" "mod rec %d: "
	(fun (id,mt) ->
	   IB.indent pp;
	   IB.printf pp "name %s\n" id;
	   IB.unindent pp;
	   pp_module_type pp mt;
	)
	msigs

    | <:sig_item< module $uid$ : $mt$ >> ->
      begin
	pl (Printf.sprintf "module decl %s" uid);
	pp_module_type pp mt;
	if !terminate_structures then
	  IB.endline pp "end module"
      end

    | <:sig_item< module type $uid$ = $mt$ >> ->
      begin
	pl (Printf.sprintf "module type decl %s" uid);
	pp_module_type pp mt;
	if !terminate_structures then
	  IB.endline pp "end module type"
      end

    | <:sig_item< open $uid:id1$ . $uid:id2$ >> ->
      pl (Printf.sprintf "sig open %s . %s" id1 id2)

    | <:sig_item< open $ids$ >> ->
      begin
	pl "sig open";
	pp_id_list pp ids;
	if !terminate_structures then
	  IB.endline pp "end open"
      end

    | <:sig_item< type $list:decls$ >> -> 
      iterate_items pp (Some astloc) "sig type decl" "with %d types" "type %d:"
	(fun decl -> pp_type_decl pp decl)
	decls

    | <:sig_item< value $lid$ : $ptyp$ >> -> 
      begin
	pl (Printf.sprintf "val decl %s : " lid);
	pp_ctyp pp ptyp;
	if !terminate_structures then
	  IB.endline pp "end val decl"
      end

    | <:sig_item< class type $list:cts$ >> ->
      iterate_items pp (Some astloc) "class type spec"
      "with %d class types" "class type %d:"
      (fun ct -> pp_class_info pp "class type spec" pp_class_type ct)
      (cts : MLast.class_type MLast.class_infos list)

    | <:sig_item< class $list:cts$ >> ->
      iterate_items pp (Some astloc) "class spec"
      "with %d classes" "class spec %d:"
      (fun ct -> pp_class_info pp "class spec" pp_class_type ct)
      (cts : MLast.class_type MLast.class_infos list)

    | _ -> (pl "unknown sig item"; abort_maybe())

and pp_module_type pp ast =
  let astloc = MLast.loc_of_module_type ast in
  let pl mesg = print_location pp mesg astloc in
  let pl_rec_2 s e1 e2 =
      begin
	pl s;
	pp_module_type pp e1;
	pp_module_type pp e2;
	if !terminate_structures then
	  IB.endline pp ("end " ^ s)
      end
  in 
  let _ = IB.indent pp in
  let _ = match ast with
    | <:module_type< functor($uid$ : $mt_arg$) -> $mt_res$ >> ->
	begin
	  pl "functor mod type";
	  IB.indent pp;
	  IB.printf pp "Argument %s:\n" uid;
	  pp_module_type pp mt_arg;
	  IB.endline pp "Result:";
	  pp_module_type pp mt_res;
	  IB.unindent pp;
	  if !terminate_structures then
	    IB.endline pp "end functor"
	end
    | <:module_type< $mt$ with $list:wcl$ >> ->
      begin
	pl (Printf.sprintf "module type with %d with constraints" 
	      (List.length wcl));
	pp_module_type pp mt;
	iterate_items pp None "" "%d" "constraint %d: "
	  (fun wc -> pp_with_constr pp wc)
	  wcl;
	if !terminate_structures then
	  IB.endline pp "end with constraint"
      end
    | <:module_type< sig $list:sis$ end >> ->
      iterate_items pp (Some astloc) "signature" "with %d fields" "item %d: "
	(fun sig_item -> pp_sig_item pp sig_item)
	sis
    | <:module_type< $mt1$ ( $mt2$ ) >> -> pl_rec_2 "mod type appl" mt1 mt2
    | <:module_type< $mt1$ . $mt2$ >> -> pl_rec_2 "mod type access" mt1 mt2
    | <:module_type< $uid:id$ >> -> pl (Printf.sprintf "mod type UID %s" id)
    | <:module_type< $lid:id$ >> -> pl (Printf.sprintf "mod type lid %s" id)
    | <:module_type< '$id$ >> -> pl (Printf.sprintf "abstract mod type '%s" id)
    (* | _ -> (pl "unknown module type"; abort_maybe()) *)
  in
    IB.unindent pp

and pp_with_constr pp ast =
  let pl mesg = print_location pp mesg (loc_of_with_constr ast) in
  let _ = IB.indent pp in
  let _ = match ast with
    | <:with_constr< type $mod_ident$ $list:tpl$ = $ptyp$ >> -> 
      begin
	pl "type with constraint";
	if tpl = [] 
	then
	  IB.endline pp "no type parameters"
	else
	  IB.printf pp "%d type params :" (List.length tpl);
	pp_type_params pp tpl;
	pp_id_list pp mod_ident;
	IB.endline pp "=";
	pp_ctyp pp ptyp;
	if !terminate_structures then
	  IB.endline pp "type constr"
      end
    | <:with_constr< module $mod_ident$ = $mexpr$ >> ->
      begin
	pl "module with constraint";
	pp_id_list pp mod_ident;
	IB.endline pp "=";
	pp_module_expr pp mexpr;
	if !terminate_structures then
	  IB.endline pp "module constr"
      end
    (* | _ -> (pl "unknown with constraint"; abort_maybe()) *)
  in
    IB.unindent pp


and pp_class_expr pp ast =
  let astloc = MLast.loc_of_class_expr ast in
  let pl mesg = print_location pp mesg astloc in
  let _ = IB.indent pp in
  let _ = match ast with

      (* level "top" *)
    | <:class_expr< fun a013489 b -> $ce$ >> when !test_more_quotations ->
	begin
	  pl "object fun a013489 b (special)";
	  pp_class_expr pp ce;
	end
	    
    | <:class_expr< fun $p$ -> $ce$ >> ->
	begin
	  pl "object fun";
	  pp_patt pp p;
	  pp_class_expr pp ce;
	  if !terminate_structures then
	    IB.endline pp "end obj fun"
	end

    | <:class_expr< let $opt:r$ $list:bindings$ in $cexpr$ >> ->
      begin
	pl (Printf.sprintf "let %sin (class_expr) with %d bindings" 
	      (if r then "rec " else "") (List.length bindings));
	IB.indent pp;
	iterate_items pp None "" "%d" "bind %d:"
	  (fun (p,e) ->
	     IB.indent pp;
	     pp_patt pp p;
	     IB.endline pp "to";
	     pp_expr pp e;
	     IB.unindent pp
	  )
	  bindings;
	IB.endline pp "in";
	pp_class_expr pp cexpr;
	IB.unindent pp;
	if !terminate_structures then
	  IB.endline pp "end let"
      end

      (* level "apply" *)

    | <:class_expr< $class_expr$ $expr$ >> ->
      begin
	pl "class application";
	pp_class_expr pp class_expr;
	pp_expr pp expr;
      end

      (* level "simple" *)

    | <:class_expr< $list:ids$ >> ->
      begin
	pl (Printf.sprintf "class longident (monomorphic) with %d ids" 
	      (List.length ids));
	pp_id_list pp ids;
      end	

    | <:class_expr< $list:ids$ [ $list:ptyps$ ] >> ->
      begin
	pl (Printf.sprintf "class longident (polymorphic)");
	IB.indent pp;
	IB.printf pp "%d type args:\n" (List.length ptyps);
	iterate_items pp None "" "%d" "Arg %d: "
	  (fun ctyp -> pp_ctyp pp ctyp)
	  ptyps;
	IB.printf pp "%d ids:\n" (List.length ids);
	pp_id_list pp ids;
	IB.unindent pp;
	if !terminate_structures then
	  IB.endline pp "end class longident";
      end	

    | <:class_expr< object $opt:spopt$ $list:citems$ end >> -> 
      begin
	pl (Printf.sprintf "object def with %d items" (List.length citems));
	IB.indent pp;
	(match spopt with
	   | None -> IB.endline pp "no self pattern"
	   | Some patt -> 
	       begin
		 IB.add_string pp "self pattern:\n";
		 pp_patt pp patt;
	       end
	);
	iterate_items pp None "" "%d" "item %d:"
	  (fun citem -> pp_class_str_item pp citem)
	  citems;
	IB.unindent pp;
	if !terminate_structures then
	  IB.endline pp "end object def"
      end

    | <:class_expr< ( $cexpr$ : $cl_typ$ ) >> ->
      begin
	pl "typed class expr";
	pp_class_expr pp cexpr;
	pp_class_type pp cl_typ;
	if !terminate_structures then
	  IB.endline pp "end typed class ex";
      end

    (* | _ -> (pl "unknown class expr"; abort_maybe()) *)
  in
    IB.unindent pp

and pp_class_str_item pp ast =
  let astloc = MLast.loc_of_class_str_item ast in
  let pl mesg = print_location pp mesg astloc in
  let _ = IB.indent pp in
  let _ = match ast with
    | <:class_str_item< declare value b12345 = $expr$; $cl_item$; end >> 
      when !test_more_quotations ->
	begin
	  pl "declare (cl str) val b12345 (special)";
	  pp_expr pp expr;
	  pp_class_str_item pp cl_item;
	end
    | <:class_str_item< declare $list:items$ end >> ->
	iterate_items pp (Some astloc) 
	  "declare (cl str)" "with %d items" "item %d:"
	  (pp_class_str_item pp)
	  items
    | <:class_str_item< inherit a455667 as $lid$ >> 
	when !test_more_quotations ->
      begin
	pl "inherit a455667 (special)";
	IB.indent pp;
	IB.printf pp "alias %s\n" lid;
	IB.unindent pp;
      end
    | <:class_str_item< inherit $cexpr$ $opt:olid$ >> ->
      begin
	pl "inherit";
	pp_class_expr pp cexpr;
	IB.indent pp;
	(match olid with
	   | None -> IB.endline pp "no alias"
	   | Some lid -> IB.printf pp "alias %s\n" lid
	);
	IB.unindent pp;
      end
    | <:class_str_item< value a878787 : $typ$ = $expr$ >> 
	when !test_more_quotations->
      begin
	pl "val a878787 nonmutable (special)";
	pp_ctyp pp typ;
	pp_expr pp expr;
      end
    | <:class_str_item< value $opt:mf$ $lid$ = $expr$ >> ->
      begin
	pl (Printf.sprintf "val %s (%s)" lid
	      (if mf then "mutable" else "nonmutable"));
	pp_expr pp expr;
      end
    | <:class_str_item< method virtual $opt:priv$ $id$ : $ptype$ >> ->
      begin
    	pl (Printf.sprintf "virtual %smethod %s" 
    	      (if priv then "private " else "") id);
    	pp_ctyp pp ptype;
	if !terminate_structures then
    	  IB.endline pp "end method"
      end
    | <:class_str_item< method $opt:priv$ $lid:id$ $opt:optype$ = $body$ >> ->
      begin
	pl (Printf.sprintf "%smethod %s" 
	      (if priv then "private " else "") id);
	IB.indent pp;
	(match optype with
	   | None -> IB.endline pp "no type annotation"
	   | Some ptype ->
	       begin
		 IB.endline pp "type annotation:";
		 pp_ctyp pp ptype;
	       end
	);
	IB.endline pp "method body:";
    	pp_expr pp body;
	IB.unindent pp;
	if !terminate_structures then
	  IB.endline pp "end method"
      end
    | <:class_str_item< type $t1$ = $t2$ >> ->
      begin
	pl "class constraint";
	IB.indent pp;
	pp_ctyp pp t1;
	IB.endline pp "=";
	pp_ctyp pp t2;
	IB.unindent pp
      end
    | <:class_str_item< initializer $expr$ >> ->
      begin
	pl "initializer";
	pp_expr pp expr;
      end

    (* | _ -> (pl "unknown class str item"; abort_maybe()) *)
  in
    IB.unindent pp


and pp_str_item pp ast =
  IB.indent pp;
  pp_str_item_noindent pp ast;
  IB.unindent pp;


and pp_str_item_noindent pp ast =
  let astloc = MLast.loc_of_str_item ast in
  let pl mesg = print_location pp mesg astloc 
  in match ast with

    | <:str_item< declare $list:items$ end >> ->
	begin
	  pl (Printf.sprintf "declare %d items" (List.length items));
	  ignore(
	    List.fold_left
	      (fun n item -> 
		 IB.printf pp "item %d: " n;
		 pp_str_item pp item;
		 n+1
	      )
	      1
	      items
	  );
	  if !terminate_structures then
	    IB.endline pp "end declare"
	end

    | <:str_item< exception $uid$ of $list:tl$ >> ->
      iterate_items pp (Some astloc) 
	(Printf.sprintf "exception %s" uid) "with %d args" "Arg %d: "
	(fun ctyp -> pp_ctyp pp ctyp)
	tl

    | <:str_item< exception $uid$ of $list:tl$ = $modid$ >> ->
      begin
	pl (Printf.sprintf "exception %s with %d args and rebind" 
	      uid (List.length tl));
	IB.indent pp;
	iterate_items pp None "" "%d" "Arg %d: "
	  (fun ctyp -> pp_ctyp pp ctyp)
	  tl;
	IB.add_string pp "rebind to ";
	pp_id_list pp modid;
	IB.unindent pp;
	if !terminate_structures then
	  IB.endline pp "end rebind exception"
      end

    | <:str_item< external $lid$ : $typ$ = $list:strings$>> ->
      begin
	pl (Printf.sprintf "external %s with %d strings" 
	      lid (List.length strings));
	pp_ctyp pp typ;
	iterate_items pp None "" "%d" "string %d: "
	  (fun s ->
	     IB.indent pp;
	     IB.endline pp s;
	     IB.unindent pp)
	  strings;
	if !terminate_structures then
	  IB.endline pp "end external"
      end

    | <:str_item< include $mexp$ >> ->
      begin
	pl "include";
	pp_module_expr pp mexp;
	if !terminate_structures then
	  IB.endline pp "end include"
      end
	
    | <:str_item< module M6(A : B) : C = D >> -> pl "module M6(A : B) : C = D"

    | <:str_item< module rec $list:mrecs$ >> ->
      iterate_items pp (Some astloc) "module rec" "with %d modules"
	"mod rec %d: "
	(fun (id,mt,me) ->
	   IB.indent pp;
	   IB.printf pp "name %s\nsig:\n" id;
	   pp_module_type pp mt;
	   IB.endline pp "module:";
	   pp_module_expr pp me;
	   IB.unindent pp)
	mrecs
    | <:str_item< module $uid$ = $mexp:mexpr$ >> ->
      begin
	pl (Printf.sprintf "module %s" uid);
	pp_module_expr pp mexpr;
	if !terminate_structures then
	  IB.endline pp "end module"
      end

    | <:str_item< module type $uid$ = $mt$ >> -> 
      begin
	pl (Printf.sprintf "module type %s" uid);
	pp_module_type pp mt;
	if !terminate_structures then
	  IB.endline pp "end module type"
      end

    | <:str_item< open $modid$ >> ->
      begin 
	pl "open";
	pp_id_list pp modid;
	if !terminate_structures then
	  IB.endline pp "end open";
      end

    | <:str_item< type $list:decls$ >> -> 
      iterate_items pp (Some astloc) "type decl" "with %d types" "decl %d:"
	(fun decl -> pp_type_decl pp decl)
	decls

    | <:str_item< let $opt:r$ $list:bindings$ in $e$>> ->
      begin
    	pl (Printf.sprintf "let %sin (str item) with %d bindings" 
    	      (if r then "rec " else "") (List.length bindings));
	IB.indent pp;
	iterate_items pp None "" "%d" "bind %d:"
    	  (fun (p,e) ->
	     IB.indent pp;
    	     pp_patt pp p;
    	     IB.endline pp "to";
    	     pp_expr pp e;
	     IB.unindent pp
	  )
    	  bindings;
    	IB.endline pp "in";
    	pp_expr pp e;
	IB.unindent pp;
	if !terminate_structures then
    	  IB.endline pp "end let"
      end

    | <:str_item< value $opt:r$ $list:bindings$>> ->
      iterate_items pp (Some astloc) 
	(if r then "let rec (str item)" else "let (str item)")
	"with %d bindings" "bind %d:"
    	(fun (p,e) ->
	   IB.indent pp;
    	   pp_patt pp p;
	   IB.endline pp "to";
    	   pp_expr pp e;
	   IB.unindent pp
	)
	bindings

    | <:str_item< let module $uid$ = $me$ in $e$ >> ->
      begin
	pl (Printf.sprintf "let mod (str item) %s" uid);
	IB.indent pp;
	pp_module_expr pp me;
	IB.endline pp "in";
	pp_expr pp e;
	IB.unindent pp;
	if !terminate_structures then
	  IB.endline pp "end let mod";
      end

    | <:str_item< $exp:e$ >> -> pl "top level expr"; pp_expr pp e

    (*************************************************************
     *      Objects and Classes
     *)

    | <:str_item< class a94837 $p1$ $p2$ = $cexpr$ >> 
    	when !test_more_quotations ->
      begin
    	pl "class def a94827 (special)";
    	pp_patt pp p1;
    	pp_patt pp p2;
    	pp_class_expr pp cexpr;
      end

    | <:str_item< class $list:cdefs$ >> -> 
      iterate_items pp (Some astloc) "class defs" 
      "with %d classes" "class %d:"
      (fun cdef -> pp_class_info pp "class def" pp_class_expr cdef)
      (cdefs : MLast.class_expr MLast.class_infos list)

    | <:str_item< class type $list:ctdefs$ >> ->
      iterate_items pp (Some astloc) "class type def"
      "with %d class types" "class type %d:"
      (fun ctdef -> pp_class_info pp "class type def" pp_class_type ctdef)
      (ctdefs : MLast.class_type MLast.class_infos list)

    | _ -> (pl "unknown str_item"; abort_maybe())


and pp_module_expr pp ast =
  let astloc = MLast.loc_of_module_expr ast in
  let pl mesg = print_location pp mesg astloc in
  let pl_rec_2 s e1 e2 =
      begin
	pl s;
	pp_module_expr pp e1;
	pp_module_expr pp e2;
	if !terminate_structures then
	  IB.endline pp ("end " ^ s)
      end
  in let _ = IB.indent pp in
  let _ = match ast with
    | <:module_expr< functor ( $id$ : $mt$ ) -> $me$ >> ->
	begin
	  pl "functor";
	  IB.printf pp "Argument %s: " id;
	  pp_module_type pp mt;
	  IB.add_string pp "Result: ";
	  pp_module_expr pp me;
	  if !terminate_structures then
	    IB.endline pp "end functor"
	end

    | <:module_expr< struct $list:strs$ end >> ->
      iterate_items pp (Some astloc) "struct" "with %d items" "item %d: "
	(fun str -> pp_str_item pp str)
	strs

    | <:module_expr< $me1$ $me2$ >> -> pl_rec_2 "mod appl" me1 me2
    | <:module_expr< $me1$ . $me2$ >> -> pl_rec_2 "mod access" me1 me2
    | <:module_expr< $uid:id$ >> -> pl (Printf.sprintf "mexpr uid %s" id)
    | <:module_expr< ( $me$ : $mt$ ) >> ->
      begin
	pl "module constraint";
	pp_module_expr pp me;
	pp_module_type pp mt;
	if !terminate_structures then
	  IB.endline pp "end mod constr"
      end
    (* | _ -> (pl "unknown module_expr"; abort_maybe()) *)
  in
    IB.unindent pp


let rec pp_implem pp = function
  | [] -> ()
  | (a,_)::l -> pp_str_item_noindent pp a; pp_implem pp l

let rec pp_interf pp = function
  | [] -> ()
  | (a,_)::l -> pp_sig_item_noindent pp a; pp_interf pp l

