(* 
 * 
 *               Camlp4 quotations in original syntax
 * 
 *                 Copyright (C) 2005  Hendrik Tews
 * 
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 * 
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License in the file LICENCE in this or one
 *   of the parent directories for more details.
 * 
 *   Time-stamp: <Tuesday 14 June 05 0:08:56 tews@debian>
 * 
 *   $Id: qocheck.ml,v 1.6 2005/11/07 16:19:35 tews Exp $
 * 
 *)

module L = Lexing
module Dl = Dynlink
module U = Unix
module B = Buffer
module IB = Pp_util.Indent_buffer

(*************** Error codes *****************)

(* 0-9 parse ok, equal *)
(* all ok  = 0 *)
let equal_pp_qo_failure = 1
let equal_pp_q_failure = 2
(* failure for both pp's = 3 *)

(* 10-19 parse ok, different *)
let differ_directive_error_code = 10
let differ_ast_len_error_code = 11
let differ_item_error_code = 12

(* 20-29 parse errors paqo_o
 * 120-129 parse errors pa_o *)
let paqo_error_level = 20
let paq_error_level = 120

let parse_error_code = 0
let parse_failure_error_code = 1 
let parse_assert_error_code = 2 
let parse_locex_error_code = 3
let parse_oex_error_code = 4
let quotation_error_code = 5
let lexing_error_code = 6

(* >=200 internal errors *)
let option_error_code = 200
let dynlink_error_code = 201
let read_pipe_error_code = 202
let no_input_error_code = 203


let debug = 0

let save_ast = ref false

let save_pp = ref false

let dump_diffs = ref false

let silent = ref false

let include_dir_list = ref [Check_config.camlp4libdir]

let add_include_dir dir =
  include_dir_list := dir :: !include_dir_list

let output_version () =
  Printf.printf "qocheck version %s\n" Pp_util.ocamlp4_version;
  exit 0


let options = 
  Arg.align
    [
      ( "-q",
	Arg.Set silent,
	" be silent"
      );
      ( "-pp",
	Arg.Set save_pp,
	if !save_pp 
	then
	  " save pretty printer output for different ast's [default]"
	else
	  " save pretty printer output for different ast's");
      ( "-nopp",
	Arg.Clear save_pp,
	if !save_pp 
	then
	  " don't save pretty printer output for different ast's"
	else
	  " don't save pretty printer output for different ast's [default]");
      ( "-dumpast",
	Arg.Set save_ast,
	if !save_ast 
	then
	  " dump ast's in case they differ [default]"
	else
	  " dump ast's in case they differ");
      ( "-nodumpast",
	Arg.Clear save_ast,
	if !save_ast 
	then
	  " don't dump ast's in case they differ"
	else
	  " don't dump ast's in case they differ [default]");
      ( "-dumpdiffs",
	Arg.Set dump_diffs,
	if !dump_diffs 
	then
	  " dump differences via toplevel [default]"
	else
	  " dump differences via toplevel");
      ( "-nodumpdiffs",
	Arg.Clear dump_diffs,
	if !dump_diffs 
	then
	  " don't dump differences via toplevel"
	else
	  " don't dump differences via toplevel [default]");
      ( "-I",
	Arg.String add_include_dir,
	"dir Add dir in the search path for object files");
      ( "-version",
	Arg.Unit output_version,
	" output version (" ^ Pp_util.ocamlp4_version ^ ") and exit");
    ]

let input_file = ref ""

let set_input_file = ref false

let short_usage = 
  "Usage: " ^ Sys.argv.(0) ^ " [options] file\n" ^
  "Recognized options are"

let usage = short_usage

let anon file = 
  if !set_input_file then
    begin
      prerr_endline short_usage;
      exit option_error_code
    end
  else
    begin
      input_file := file; 
      set_input_file := true
    end



let d dv s =
  if dv < debug then 
    begin
      Printf.fprintf stderr "%d: %s\n" (U.getpid()) s;
      flush stderr
    end
	
let msg s =
  if not !silent then
    print_string s



let string_of_loc (start,ende) =
  (Printf.sprintf "File \"%s\", line %d, "
     start.L.pos_fname start.L.pos_lnum
  ) ^
  (if start.L.pos_cnum = ende.L.pos_cnum
   then
     Printf.sprintf "character %d:" (start.L.pos_cnum - start.L.pos_bol)
   else
     Printf.sprintf "characters %d-%d:"
       (start.L.pos_cnum - start.L.pos_bol)
       (ende.L.pos_cnum - start.L.pos_bol)
  )


let string_of_err_ctx = function
  | Pcaml.Finding -> "Finding"
  | Pcaml.Expanding -> "Expanding"
  | Pcaml.ParsingResult(loc, name) -> 
      Printf.sprintf "ParsingResult %s %s" (string_of_loc loc) name
  | Pcaml.Locating -> "Locating"



let parse_input () = 
  let s = Stream.of_channel (open_in !input_file) in
  let ast = !Pcaml.parse_implem s
  in
    ast


let rec find_file_dir_list file = function
  | [] -> raise Not_found
  | dir :: dirs ->
      let p = Filename.concat dir file 
      in
	if Sys.file_exists p 
	then p
	else find_file_dir_list file dirs

let save_loadfile file =
  try
    Dl.loadfile (find_file_dir_list file !include_dir_list);
    true
  with
    | Dl.Error(error) -> 
	begin
	  Printf.fprintf stderr "dynlink %s failed: %s\n" 
	    file (Dl.error_message error);
	  false
	end
    | Not_found ->
	begin
	  Printf.fprintf stderr "library %s not found (consider option -I)\n"
	    file;
	  false
	end

type str_item = MLast.str_item * MLast.loc
type ast_type = str_item list * bool

type parse_result = 
  | Ast of ast_type
  | Parse_error of (L.position * L.position) * string
  | Parse_Failure of (L.position * L.position) * string
  | Assert_fail of  (L.position * L.position) * string * int * int
  | LocQerror of (L.position * L.position) * string * Pcaml.err_ctx * exn
  | LocTokenError of  (L.position * L.position) * string
  | LocEx of (L.position * L.position) * exn
  | Exn of exn
  | DynlinkEx
  | File_not_found

let write_ast wc (ast : parse_result) =
  Marshal.to_channel wc ast [Marshal.No_sharing]


let do_child setup_fun wc pa_mod =
  (* 
   * let _ = 
   *   if !silent then
   *     let null_c = open_out "/dev/null" 
   *     in
   * 	U.dup2 (U.descr_of_out_channel null_c) U.stdout;
   * 	U.dup2 (U.descr_of_out_channel null_c) U.stderr;
   * 	print_endline "hallo"
   * 	  
   * in
   *)
  let ast = 
    if not (save_loadfile pa_mod) 
    then DynlinkEx
    else
      try
	setup_fun ();
	Ast(parse_input ())
      with
	| Sys_error(m) when m = (!input_file ^ ": No such file or directory") ->
	    File_not_found
	| Stdpp.Exc_located(loc, Stream.Error msg) -> 
	    Parse_error(loc, msg)
	| Stdpp.Exc_located(loc, Failure msg) -> 
	    Parse_Failure(loc, msg)
	| Stdpp.Exc_located(loc, Assert_failure (file,line,col)) -> 
	    Assert_fail(loc, file, line, col)
	| Stdpp.Exc_located(loc, Pcaml.Qerror(name, qerr, ex)) ->
	    LocQerror(loc, name, qerr, ex)
	| Stdpp.Exc_located(loc, Token.Error(ms)) ->
	    LocTokenError(loc, ms)
	| Stdpp.Exc_located(loc, e) -> LocEx(loc, e)
	| e ->
	    Exn e
  in	
    write_ast wc ast;
    exit 0


let do_parent error_base rc parser_id =
  let pr = 
    try
      (Marshal.from_channel rc : parse_result)
    with
      | End_of_file -> 
	  begin
	    msg "reading ast from pipe failed - exiting\n";
	    exit read_pipe_error_code
	  end	      
  in
    close_in rc;
    match pr with
      | Ast ast -> 
	  begin
	    msg(Printf.sprintf
		  "%s successfully parsed %s\n" parser_id !input_file);
	    ast
	  end
      | File_not_found ->
	  begin
	    msg(Printf.sprintf "%s: No such file or directory\n" !input_file);
	    exit no_input_error_code;
	  end
      | Parse_error((start, ende) as loc, errmsg) ->
	  begin
	    let loc = 
	      if start.L.pos_fname = "" 
	      then ({start with L.pos_fname = !input_file}, ende)
	      else loc
	    in
	      msg(string_of_loc loc);
	      msg(Printf.sprintf 
		    "\n%s failed on %s with an parse error\n\t%s\n" 
		    parser_id !input_file errmsg);
	      exit (error_base + parse_error_code);
	  end
      | Parse_Failure(_loc, errmsg) ->
	  begin
	    msg(Printf.sprintf 
		  "%s failed on %s with an failure exception:\n\t %s\n" 
		  parser_id !input_file errmsg);
	    exit (error_base + parse_failure_error_code);
	  end
      | Assert_fail((start, ende) as loc, file, line, col) ->
	  begin
	    let loc = 
	      if start.L.pos_fname = "" 
	      then ({start with L.pos_fname = !input_file}, ende)
	      else loc
	    in
	      msg (string_of_loc loc);
	      msg (Printf.sprintf 
		     "\n%s failed on %s with an assertion\n\
                      in file %s line %d column %d.\n"
		     parser_id !input_file file line col);
	      exit (error_base + parse_assert_error_code);
	  end
      | LocQerror((start, ende) as loc, name, qerr, ex) ->
	  begin
	    let loc = 
	      if start.L.pos_fname = "" 
	      then ({start with L.pos_fname = !input_file}, ende)
	      else loc
	    in
	      msg (string_of_loc loc);
	      msg (Printf.sprintf
		     "\n%s failed on %s with an Quotation error\n\
                        name : %s, tag : %s, exception : %s\n"
		     parser_id !input_file name
		     (string_of_err_ctx qerr) (Printexc.to_string ex));
	      exit(error_base + quotation_error_code)
	  end
      | LocTokenError((start, ende) as loc, ms) ->
	  begin
	    let loc = 
	      if start.L.pos_fname = "" 
	      then ({start with L.pos_fname = !input_file}, ende)
	      else loc
	    in
	      msg (string_of_loc loc);
	      msg (Printf.sprintf
		     "\n%s failed on %s with an lexing error\n %s\n"
		     parser_id !input_file ms);
	      exit(error_base + lexing_error_code)
	  end
      | LocEx((start, ende) as loc, e) ->
	  begin
	    let loc = 
	      if start.L.pos_fname = "" 
	      then ({start with L.pos_fname = !input_file}, ende)
	      else loc
	    in
	      msg(string_of_loc loc);
	      msg(Printf.sprintf 
		    "\n%s failed on %s with an located exception\n\t%s\n" 
		    parser_id !input_file (Printexc.to_string e));
	      exit (error_base + parse_locex_error_code);
	  end
      | DynlinkEx ->
	  begin
	    msg "Dynlink error\n";
	    exit dynlink_error_code
	  end
		
      | Exn e ->
	  begin
	    msg(Printf.sprintf 
		  "%s failed on %s with an unexpected exception\n" 
		  parser_id !input_file);
	    msg (Printexc.to_string e);
	    msg "\n";
	    exit (error_base + parse_oex_error_code)
	  end
      

let parse_in_child setup_fun error_base pa_mod parser_id =
  let _ = d 5 (Printf.sprintf "parse in child %s" pa_mod) in
  let (rfd, wfd) = U.pipe() in
  let rc = U.in_channel_of_descr rfd in
  let wc = U.out_channel_of_descr wfd 
  in
    flush stderr;
    flush stdout;
    if U.fork() <> 0 
    then				(* parent *)
      begin
	close_out wc;
	do_parent error_base rc parser_id (* returns the ast from the child *)
      end
    else				(* child *)
      begin
	close_in rc;
	do_child setup_fun wc pa_mod	(* does not return *)
      end


let pp_file qo ext = 
  Printf.sprintf "tmp/%s_%s.%s" (Filename.basename !input_file) qo ext


let do_equal (ast, directive) =
  let exit_code = ref 0 in
  let do_pp buf pp_implem = 
    try
      pp_implem buf ast;
      true
    with
      | e -> 
	  IB.printf buf
	    "\nTerminated with exception:\n\t%s\n"
	    (Printexc.to_string e);
	  false
  in
  let buf_qo = IB.create 4096 in
  let buf_q = IB.create 4096 in
  let _ = Pp_util.abort_when_incomplete := true in
  let _ = Pp_util.test_more_quotations := true in
  let pp_qo = do_pp buf_qo Pp_qo.pp_implem in
  let pp_q =  do_pp buf_q Pp_q.pp_implem in
    msg "Ast's are identical";
    if pp_qo && pp_q then
      msg " and both pretty printer work.\n"
    else
      msg "\n";
    if not pp_qo then
      begin
	msg "original quotation pretty printer (pp_qo) failed.\n";
	exit_code := !exit_code + equal_pp_qo_failure;
	if !save_pp then
	  let oc = open_out (pp_file "qo" "pp")
	  in
	    IB.flush buf_qo oc;
	    close_out oc
      end;
    if not pp_q then
      begin
	msg "revised quotation pretty printer (pp_q) failed.\n";
	exit_code := !exit_code + equal_pp_q_failure;
	if !save_pp then
	  let oc = open_out (pp_file "q" "pp")
	  in
	    IB.flush buf_q oc;
	    close_out oc
      end;
    !exit_code
	
    

let dump_nth_item ast1 ast2 n = 
  let dump_el pipe ast =
    output_string pipe "(Marshal.from_string \"";
    output_string pipe
      (String.escaped 
	 (Marshal.to_string ((List.nth ast n) : str_item)
	    [Marshal.No_sharing]));
    output_string pipe "\" 0 : MLast.str_item * MLast.loc);;\n";
    ignore(U.close_process_out pipe)
  in
    msg(Printf.sprintf
	  "(***************************** qo Element %d \
           ******************************)\n"
	  n);
    flush stdout;
    dump_el (U.open_process_out "ocaml") ast1;
    msg(Printf.sprintf
	  "(******************************** End *****\
            ******************************)\n\n");
    msg(Printf.sprintf
	  "(***************************** q Element %d \
            ******************************)\n"
	  n);
    flush stdout;
    dump_el (U.open_process_out "ocaml") ast2;
    msg(Printf.sprintf
	  "(******************************** End *****\
           ******************************)\n\n")
      


let do_different (ast1, b1) (ast2, b2) =
  let exit_code = ref None in
  let set_exit_code c = match !exit_code with
    | None -> exit_code := Some c
    | _ -> ()
  in
  let len1 = List.length ast1 in
  let len2 = List.length ast2 in
  let rec list_cmp i res l1 l2 = match (l1,l2) with
    | ([], []) -> res
    | (s1::l3, s2::l4) -> 
	if s1 = s2 
	then
	  list_cmp (i+1) res l3 l4
	else
	  list_cmp (i+1) (i :: res) l3 l4
    | _ -> res
  in
  let diffs = List.rev (list_cmp 0 [] ast1 ast2)
  in
    msg "Ast's differ: ";
    if b1 <> b2 then
      begin
	msg "directive bool differs -- very odd!\n";
	set_exit_code differ_directive_error_code;
      end;
    if len1 = len2 then
      msg (Printf.sprintf "%d elements in both asts, " len1)
    else
      begin
	msg (Printf.sprintf 
	       "%d elements in the qo version, %d in the q version\n"
	       len1 len2);
	set_exit_code differ_ast_len_error_code
      end;
    if diffs = [] 
    then
      begin
	assert(b1 <> b2);
	msg "all str_items are equal\n"
      end
    else
      begin
	msg "differences at positions: ";
	msg(Printf.sprintf "%d" (List.hd diffs));
	List.iter
	  (fun i -> msg(Printf.sprintf ", %d" i))
	  (List.tl diffs);
	msg "\n";
	set_exit_code differ_item_error_code
      end;

    if !save_pp then
      begin
	let buf = IB.create 4096 in
	let name_1 = pp_file "qo" "pp" in
	let name_2 = pp_file "q" "pp" in
	let pp_1 = open_out name_1 in
	let pp_2 = open_out name_2
	in
	  Pp_util.abort_when_incomplete := false;
	  Pp_util.test_more_quotations := true;
	  Pp_qo.pp_implem buf ast1;
	  IB.flush buf pp_1;
	  IB.reset buf;
	  Pp_qo.pp_implem buf ast2;
	  IB.flush buf pp_2;
	  close_out pp_1;
	  close_out pp_2;
	  msg(Printf.sprintf "pretty printed to %s and %s\n" name_1 name_2)
      end;
    if !save_ast then
      begin 
	let name_1 = pp_file "qo" "ast" in
	let name_2 = pp_file "q" "ast" in
	let c_ast_1 = open_out name_1 in
	let c_ast_2 = open_out name_2
	in
	  Marshal.to_channel c_ast_1 
	    (ast1 : str_item list) [Marshal.No_sharing];
	  Marshal.to_channel c_ast_2 
	    (ast2 : str_item list) [Marshal.No_sharing];
	  close_out c_ast_1;
	  close_out c_ast_2;
	  msg(Printf.sprintf "dumped to %s and %s\n" name_1 name_2)
      end;
    msg "\n";
    if !dump_diffs then
      List.iter (dump_nth_item ast1 ast2) diffs;
    match !exit_code with
      | Some c -> c
      | None -> assert false


let invoke_pa_o_compat () =
  Arg.parse_argv ~current:(ref 0) [| "x"; "-compatible" |]
    (Pcaml.arg_spec_list()) (fun _ -> ()) "x"


let parse_and_compare () =
  let ast2 = 
    parse_in_child (fun _ -> ()) paq_error_level "pa_o.cmo" "pa_o" 
  in
  let ast1 = 
    parse_in_child invoke_pa_o_compat paqo_error_level "paqo_o.cmo" "paqo_o" 
  in
    if ast1 = ast2 
    then
      do_equal ast1 			(* return exit code *)
    else
      do_different ast1 ast2 		(* return exit code or exit self *)
  

let main() =
  Arg.parse options anon usage;
  if not !set_input_file then
    begin
      prerr_endline short_usage;
      exit option_error_code
    end;
  Dl.init ();
  exit (parse_and_compare ())
;;


Printexc.print main ()

